--------------------------------------------------------------------------------
-- Project:     MicroBlaze on TE-XC2Se
-- Name:        te_mem_core.vhd
-- Description: memory controller peripheral, core logic
--------------------------------------------------------------------------------
-- Note 1:      this core is optimized for 32-bit read access,
--              like issued from instruction fetches, 
--              to yield maximum processor performance.
--              for 8-bit, or 16-bit access, one memory
--              cycle could be saved- for the price of
--              a more complicated implementation
-- Note 2:      memory map
--                  0080_0000 \_ sram
--                  0087_ffff /
--                  00c0_0000 \_ flash (factory config)
--                  00c3_ffff /
--                  00c4_0000 \_ flash (user config)
--                  00c7_ffff /
--                  00c8_0000 \_ flash (application space)
--                  00cf_ffff /
--                  00e0_0000    push buttons
--                  00e0_0002    dip switch
--                  00e0_0004    LEDs
-- Note 3:      microblaze (big-endian) words:
--                  high:     mb[0:7]        be[0]
--                  mid-high: mb[8:15]       be[1]
--                  mid-low:  mb[16:23]      be[2]
--                  low:      mb[24:31]      be[3]
--              reversed (see rdd/wrd signals)
--                  high:     d[31:24]
--                  mid-high: d[23:16]
--                  mid-low:  d[15:8]
--                  low:      d[7:0]
--              ram/flash storage (little-endian) words:
--                  high:     (a+0)[7:0]     ble
--                  mid-high: (a+0)[15:0]    bhe
--                  mid-low:  (a+1)[7:0]     ble
--                  low:      (a+1)[15:0]    bhe
--------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity te_mem_core is
	generic (
		C_OPB_AWIDTH : integer := 32;
		C_OPB_DWIDTH : integer := 32;
		C_BASEADDR   : std_logic_vector(0 to 31) := x"0080_0000";
		C_HIGHADDR   : std_logic_vector(0 to 31) := x"00FF_FFFF"
		);
	port (
		-- Global signals
		OPB_Clk      : in    std_logic;
		OPB_Rst      : in    std_logic;
		-- OPB signals (mostly registered)
		OPB_ABus     : in    std_logic_vector(0 to C_OPB_AWIDTH-1);
		OPB_BE       : in    std_logic_vector(0 to C_OPB_DWIDTH/8-1);
		OPB_DBus     : in    std_logic_vector(0 to C_OPB_DWIDTH-1);
		OPB_RNW      : in    std_logic;
		-- simplified interface to OPB
		my_Select    : in    std_logic;
		my_Ack       : out   std_logic;
		my_DBus      : out   std_logic_vector(0 to C_OPB_DWIDTH-1);
		-- external memory bus
		mem_a        : out   std_logic_vector(21 downto 1); -- mem: address bus
		mem_d_i      : in    std_logic_vector(15 downto 0); -- mem: data bus
		mem_d_o      : out   std_logic_vector(15 downto 0); --      ...
		mem_d_t      : out   std_logic;                     --      ...
		mem_wen      : out   std_logic;                     -- mem: we#
		flash_cen    : out   std_logic;                     -- flash: ce#
		flash_oen    : out   std_logic;                     -- flash: oe#
		flash_rdy    : in    std_logic;                     -- flash: ready
		flash_8n     : out   std_logic;                     -- flash: byte#
		ram_csn      : out   std_logic;                     -- ram: cs#
		ram_oen      : out   std_logic;                     -- ram: oe#
		ram_blen     : out   std_logic;                     -- ram: ble#
		ram_bhen     : out   std_logic;                     -- ram: bhe#
		cpld_csn     : out   std_logic;                     -- cpld: cs#
		cpld_rwn     : out   std_logic);                    -- cpld: r/w#
end te_mem_core;

--------------------------------------------------------------------------------
architecture bhv of te_mem_core is
	-- reversed buses
	signal adx:       std_logic_vector(31 downto 0);
	signal rdd:       std_logic_vector(31 downto 0);
	signal wrd:       std_logic_vector(31 downto 0);
	-- timing #1: RAM read
	signal memsel1:   std_logic;
	signal membank1:  std_logic;
	signal memtim1:   std_logic_vector(7 downto 0);
	signal memalsb1:  std_logic;
	signal memrdd1:   std_logic_vector(31 downto 0);
	signal memce1:    std_logic;
	signal memoe1:    std_logic;
	signal membhe1:   std_logic;
	signal memble1:   std_logic;
	signal memdone1:  std_logic;
	--timing #2: RAM write
	signal memsel2:   std_logic;
	signal membank2:  std_logic;
	signal memtim2:   std_logic_vector(7 downto 0);
	signal memalsb2:  std_logic;
	signal memwrdd2:  std_logic_vector(15 downto 0);
	signal memce2:    std_logic;
	signal memwe2:    std_logic;
	signal memdt2:    std_logic;
	signal membhe2:   std_logic;
	signal memble2:   std_logic;
	signal memdone2:  std_logic;
	-- timing #3: flash read/write
	signal memsel3:   std_logic;
	signal membank3:  std_logic;
	signal memtim3:   std_logic_vector(7 downto 0);
	signal memalsb3:  std_logic;
	signal memrdd3:   std_logic_vector(31 downto 0);
	signal memwrdd3:  std_logic_vector(15 downto 0);
	signal memce3:    std_logic;
	signal memwe3:    std_logic;
	signal memoe3:    std_logic;
	signal memdt3:    std_logic;
	signal membhe3:   std_logic;
	signal memble3:   std_logic;
	signal memdone3:  std_logic;
	
begin
	
	--------------------------------
	-- reverse bit order so that bit zero is least-significant bit
	adx_gen: for i in 0 to 31 generate adx(i)<= OPB_ABus(31-i); end generate;
	wrd_gen: for i in 0 to 31 generate wrd(i)<= OPB_DBus(31-i); end generate;
	rdd_gen: for i in 0 to 31 generate my_DBus(31-i)<= rdd(i);  end generate;
	
	--------------------------------
	-- decode memory bank
	memsel1<= my_Select and not(adx(22)) and     OPB_RNW;
	memsel2<= my_Select and not(adx(22)) and not(OPB_RNW);
	memsel3<= my_Select and     adx(22);
	
	--------------------------------
	-- create mux control signals
	process(OPB_Clk, OPB_Rst)
	begin
		if OPB_Rst= '1' then
			membank1<= '0';
			membank2<= '0';
			membank3<= '0';
		elsif rising_edge(OPB_Clk) then
			if my_Select= '1' then
				membank1<= memsel1;
				membank2<= memsel2;
				membank3<= memsel3;
			end if;
			if memdone1= '1' then membank1<= '0'; end if;
			if memdone2= '1' then membank2<= '0'; end if;
			if memdone3= '1' then membank3<= '0'; end if;
		end if;
	end process;
	
	--------------------------------
	-- delay lines for each bank as timing base
	process(OPB_Clk, OPB_Rst)
	begin
		if OPB_Rst= '1' then
			memtim1(memtim1'high downto 1)<= (others=> '0');
			memtim2(memtim2'high downto 1)<= (others=> '0');
			memtim3(memtim3'high downto 1)<= (others=> '0');
		elsif rising_edge(OPB_Clk) then
			memtim1(memtim1'high downto 1)<= memtim1(memtim1'high-1 downto 1) & memsel1;
			memtim2(memtim2'high downto 1)<= memtim2(memtim2'high-1 downto 1) & memsel2;
			memtim3(memtim3'high downto 1)<= memtim3(memtim3'high-1 downto 1) & memsel3;
		end if;
	end process;
	memtim1(0)<= memsel1;
	memtim2(0)<= memsel2;
	memtim3(0)<= memsel3;
	
	--------------------------------
	-- timing #1: 32-bit RAM read
	--     total cycle time: 3 x 20.8ns = 63ns
	--
	--      |__    __    __    __   |
	-- clk  |  \__/  \__/  \__/  \__|
	--      |                       |
	--      |  0  /  1  /  2  / ... |
	--      |_____ _____ _____ _____|
	-- a,be |_____X__n__X_n+2_X_____|
	--      |           _     _     |
	-- di   |----------<_>---<_>----|
	--      |_____             _____|
	-- ce   |     \___________/     |
	--      |_____             _____|
	-- oe   |     \___________/     |
	process(OPB_Clk, OPB_Rst)
	begin
		if OPB_Rst= '1' then
			memalsb1<= '0';
			memce1  <= '1';
			memoe1  <= '1';
			membhe1 <= '0';
			memble1 <= '0';
		elsif rising_edge(OPB_Clk) then
			if memtim1(0)= '1' then 
				memce1  <= '0';
				memoe1  <= '0';
				memble1 <= not(OPB_BE(0)); -- high byte     <= ram[7:0]
				membhe1 <= not(OPB_BE(1)); -- mid-high byte <= ram[15:8]
			end if;
			
			if memtim1(1)= '1' then 
				memrdd1(31 downto 24)<= mem_d_i(7  downto 0); -- high byte     <= ram[7:0]
				memrdd1(23 downto 16)<= mem_d_i(15 downto 8); -- mid-high byte <= ram[15:8]
				memalsb1<= '1';
				memble1 <= not(OPB_BE(2)); -- mid-low byte <= ram[7:0]
				membhe1 <= not(OPB_BE(3)); -- low byte     <= ram[15:8]
			end if;
			
			if memtim1(2)= '1' then 
				-- memrdd1(15 downto 0)<= mem_d_i(7  downto 8); -- note: this is the "safe" solution
				-- memrddi(7  dwonto 0)<= mem_d_i(15 downto 8); --       ...
				memce1  <= '1';
				memoe1  <= '1';
				memalsb1<= '0';
			end if;
		end if;
	end process;
	-- memdone1<= memtiming(3); -- note: this is the "safe" solution
	
	-- note: this is the "faster" solution. 
	--       still ok, as data are registered in opb_memcon16
	memrdd1(15 downto 8)<= mem_d_i(7  downto 0); -- mid-low byte <= ram[7:0]
	memrdd1(7  downto 0)<= mem_d_i(15 downto 8); -- low byte     <= ram[15:8]
	memdone1<= memtim1(2);
	
	--------------------------------
	-- timing #2: 32-bit RAM write
	--     total cycle time: 7 x 20.8ns = 146ns
	--
	--      |__    __    __    __    __    __    __    __   |
	-- clk  |  \__/  \__/  \__/  \__/  \__/  \__/  \__/  \__|
	--      |                                               |
	--      |  0  /  1  /  2  /  3  /  4  /  5  /  6  / ... |
	--      |_____ _________________ _________________ _____|
	-- a,be |_____X_____n___________X_____n+2_________X_____|
	--      |_____ _________________ _________________ _____|
	-- do   |_____X_________________X_________________X_____|
	--      |_____                                     _____|
	-- dt   |     \___________________________________/     |
	--      |_____                                     _____|
	-- ce   |     \___________________________________/     |
	--      |___________       ___________       ___________|
	-- we   |           \_____/           \_____/           |
	process(OPB_Clk, OPB_Rst)
	begin
		if OPB_Rst= '1' then
			memalsb2<= '0';
			memce2  <= '1';
			memwe2  <= '1';
			memdt2  <= '1';
			membhe2 <= '0';
			memble2 <= '0';
		elsif rising_edge(OPB_Clk) then
			if memtim2(0)= '1' then 
				memwrdd2(7  downto 0)<= wrd(31 downto 24); -- high byte     => ram[7:0]
				memwrdd2(15 downto 8)<= wrd(23 downto 16); -- mid-high byte => ram[15:8]
				memce2  <= '0';
				memdt2  <= '0';
				memble2 <= not(OPB_BE(0)); -- high byte     => ram[7:0]
				membhe2 <= not(OPB_BE(1)); -- mid-high byte => ram[15:8]
			end if;
			
			if memtim2(1)= '1' then 
				memwe2  <= '0';
			end if;
			
			if memtim2(2)= '1' then
				memwe2  <= '1';
			end if;
			
			if memtim2(3)= '1' then 
				memalsb2<= '1';
				memwrdd2(7  downto 0)<= wrd(15 downto 8); -- mid-low byte => ram[7:0]
				memwrdd2(15 downto 8)<= wrd(7  downto 0); -- low byte     => ram[15:8]
				memble2 <= not(OPB_BE(2)); -- mid-low byte => ram[7:0]
				membhe2 <= not(OPB_BE(3)); -- low byte     => ram[15:8]
			end if;
			
			if memtim2(4)= '1' then
				memwe2  <= '0';
			end if;
			
			if memtim2(5)= '1' then
				memwe2  <= '1';
			end if;
			
			if memtim2(6)= '1' then
				memce2  <= '1';
				memdt2  <= '1';
				memalsb2<= '0';
			end if;
		end if;
	end process;
	--memdone2<= memtiming(7); -- the "safe" solution
	memdone2<= memtim2(6); -- the "faster" solution
	
	--------------------------------
	-- timing #3: 16-bit flash read/write
	--     total cycle time: 6 x 20.8ns = 125ns
	--
	--     |__    __    __    __    __    __    __   |
	-- clk |  \__/  \__/  \__/  \__/  \__/  \__/  \__|
	--     |                                         |
	--     |  0  /  1  /  2  /  3  /  4  /  5  / ... |
	--     |_____ _____________________________ _____|
	-- a   |_____X_____________________________X_____|
	--     |_____                               _____|
	-- ce  |     \_____________________________/     |
	--                                               |
	-- write access                                  |
	--     |_____ _____________________________ _____|
	-- do  |_____X_____________________________X_____|
	--     |_____                               _____|
	-- dt  |     \_____________________________/     |
	--     |___________                   ___________|
	-- we  |           \_________________/           |
	--                                               |
	-- read access                                   |
	--     |                             _           |
	-- di  |----------------------------<_>----------|
	--     |___________                   ___________|
	-- oe  |           \_________________/           |
	--                                               |
	-- CPLD read/write acess                         |
	--     |_____ _____________________________ _____|
	-- a   |_____X_____________________________X_____|
	--     |_____ _____________________________ _____|
	-- do  |_____X_____________________________X_____|
	--     |_____ _____________________________ _____|
	-- rw  |_____X_____________________________X_____|
	--     |___________                   ___________|
	-- cs  |           \_________________/           |
	
	process(OPB_Clk, OPB_Rst)
	begin
		if OPB_Rst= '1' then
			memalsb3<= '0';
			memce3  <= '1';
			memwe3  <= '1';
			memoe3  <= '1';
			memdt3  <= '1';
		elsif rising_edge(OPB_Clk) then
			if memtim3(0)= '1' then 
				memalsb3<= adx(1);
				memce3<= '0';
				memdt3<= OPB_RnW;
			end if;
			
			if memtim3(1)= '1' then 
				memoe3<= not(OPB_RnW);
				memwe3<= OPB_RnW;
			end if;
			
			if memtim3(4)= '1' then
				memwe3 <= '1';
				memrdd3(31 downto 24)<= mem_d_i(7  downto 0); -- high byte     <= flash[7:0]
				memrdd3(23 downto 16)<= mem_d_i(15 downto 8); -- mid-high byte <= flash[15:8]
				memrdd3(15 downto 8) <= mem_d_i(7  downto 0); -- mid-low byte  <= data mirroring
				memrdd3(7  downto 0) <= mem_d_i(15 downto 8); -- low byte      <= ...
				memoe3 <= '1';
			end if;
			
			if memtim3(5)= '1' then 
				memalsb3<= '0';
				memce3  <= '1';
				memdt3  <= '1';
			end if;
		end if;
	end process;
	memwrdd3(7  downto 0)<= wrd(15 downto 8); -- high byte => flash[7:0]
	memwrdd3(15 downto 8)<= wrd(7  downto 0); -- low byte  => flash[15:8]
	memdone3<= memtim3(6);
	
	--------------------------------
	-- combine bank signals
	
	my_Ack   <= (memdone1 and membank1)
	or          (memdone2 and membank2)
	or          (memdone3 and membank3);
	
	-- common bus signals
	mem_a(21 downto 2)<= adx(21 downto 2);
	mem_a(1) <= (memalsb1 and membank1)
	or          (memalsb2 and membank2)
	or          (memalsb3 and membank3);
	
	rdd      <= memrdd1 when membank1= '1' 
	else        memrdd3;
	
	mem_d_o  <= memwrdd2 when membank2= '1'
	else        memwrdd3;
	
	mem_d_t  <= (memdt2 or not(membank2))
	and         (memdt3 or not(membank3));
	
	mem_wen  <= (memwe2 or not(membank2))
	and         (memwe3 or not(membank3));
	
	-- flash-specific signals
	flash_cen<= memce3 or adx(21) or not(membank3);
	flash_oen<= memoe3 or adx(21) or not(membank3);
	flash_8n <= '1';
	
	-- ram-specific signals
	ram_csn  <= (memce1 or not(membank1))
	and         (memce2 or not(membank2));
	
	ram_oen  <= (memoe1 or not(membank1));
	
	ram_blen <= (memble1 or not(membank1))
	and         (memble2 or not(membank2));
	
	ram_bhen <= (membhe1 or not(membank1))
	and         (membhe2 or not(membank2));
	
	-- cpld-specific singals
	cpld_csn <= (memwe3 and memoe3) or not(adx(21)) or not(membank3);
	cpld_rwn <= OPB_RnW;
	
end bhv;

--------------------------------------------------------------------------------
-- end of file

