# 
# iPodder feeds module
#

import os
import stat
import logging
import contrib.urlnorm as urlnorm
import contrib.bloglines as bloglines

log = logging.getLogger('iPodder')

SUB_STATES = ('unsubscribed', 'newly-subscribed', 'subscribed', 'preview', 'disabled', 'force')

def mkfeedkey(feedorint): 
    "Make a feedkey out of a feed or an integer."
    assert feedorint is not None
    return 'feed#%d' % int(feedorint)

class Feed(object): 
    "Represent a podcast feed."

    def __init__(self, feeds, url, title='', sub_state='autopreview'): 
        "Initialise the feed."
        self.feeds = feeds
        self.id = feeds.nextfeedid()
        self.url = url
        self.normurl = urlnorm.normalize(str(url))          
        assert sub_state in SUB_STATES
        self.sub_state = sub_state
        self.title = title
        self.fix_state()
    
    def fix_state(self):
        "On unpickling, we might not have all the attributes we want."
        if not self.sub_state in SUB_STATES: 
            self.sub_state = 'subscribed'
            
        defaults = {
            # maintained exclusively by us
            'checked': None, # when did we last check? (time tuple)
            'error': '', # set to string to give feed error display
            'dirname': None,   # relative to config.download_dir

            # used when we start managing disk consumption: 
            'priority': 1,
            'mblimit': 0,
            
            # set from feedparser/HTTP
            'modified': '', # last-modified according to HTTP (time tuple)
            'etag': '', # etag last time we checked
            'status': -1, # http status
            
            # set from feedparser: 
            'version': 'unknown', 
            'title': '', 
            'tagline': '',
            'generator': '', 
            'copyright_detail': {}, 
            'copyright': '',
            'link': '',
            }
        for att, value in defaults.items(): 
            if not hasattr(self, att): 
                setattr(self, att, value)
        if hasattr(self, 'name'): 
            self.title = self.name
            del self.name
        if not self.url.strip(): 
            log.debug("Disabling Garth's empty feed from hell...")
            self.sub_state = 'disabled'
        if not hasattr(self, 'normurl'): 
            self.normurl = urlnorm.normalize(str(self.url))
        
    def __str__(self): 
        "Convert a Feed into a string: use its name."
        if self.title: 
            return self.title
        else: 
            return "Feed ID %d at %s" % (self.id, self.url)

    def __int__(self): 
        "Convert a Feed into an int: use its id."
        return self.id

    # TODO: add a feeds arg to __init__, and a flush method. 

    def half_flush(self): 
        """Flush this feed's information back to the state database."""
        self.feeds.write_feed_to_state(self)

    def flush(self): 
        """Flush all feeds everywhere."""
        self.feeds.flush()

    def mb_on_disk(self): 
        """Calculate how much disk space we're using. Doesn't scan the 
        playlist, so won't find anything outside of the feed's current 
        target directory."""
        if self.dirname is None: 
            return 0.0
        try: 
            path = os.path.join(self.feeds.config.download_dir, self.dirname)
            files = [os.path.join(path, f) for f in os.listdir(path)]
            files = [f for f in files if os.path.isfile(f)]
            bytes = sum([os.stat(f)[stat.ST_SIZE] for f in files])
            return float(bytes) / (1024.0*1024.0)
        except KeyError: 
            raise
        except WindowsError, ex: 
            errno, message = ex.args
            if errno == 3: # directory not found 
                return 0.0
            log.exception("Caught WindowsError (errno %d, \"%s\") "\
                          "scanning directory %s", errno, message, path)
            return 0.0
        except: # Oh, no. Another @(*&^! blind exception catcher. 
            log.exception("Can't calculate MB total for %s", path)
            return 0.0
        
class DuplicateFeedUrl(ValueError): 
    """Used to reject duplicate feeds."""
    pass
    
class Feeds(object): 
    "A list to keep track of feeds."
    
    def __init__(self, config, state): 
        "Initialize the feeds from `config` and `state`."
        object.__init__(self)
        self.config = config
        self.state = state
        self.feeds_by_normalized_url = {}
        self._members = []
        self.absorb()
        self.flush()
        self.clean_state_database()

    def nextfeedid(self): 
        "Return the next feed ID. Doesn't sync(); that's expected later."
        state = self.state
        feedid = state.get('lastfeedid', 1) # What's the last feedid we used?
        # Just in case, make sure we avoid collisions. 
        while True: 
            feedkey = mkfeedkey(feedid)
            if state.has_key(feedkey): 
                feedid = feedid + 1
            else: 
                break
        state['lastfeedid'] = feedid
        return feedid

    def get_feed_for_id(self, id)
        "Return a feed object by feed ID"
        for feed in self:
            if (feed.id == id)
                return feed
            else
                return None
    
    def addfeed(self, url, *a, **kw): 
        "Create and add a feed, taking care of all state."
        state = self.state
        feed = Feed(self, url, *a, **kw)
        assert feed is not None
        match = self.feeds_by_normalized_url.get(feed.normurl)
        if match is not None: 
            raise DuplicateFeedUrl, match
        else: 
            self.feeds_by_normalized_url[feed.normurl] = feed
        self.write_feed_to_state(feed)
        self._members.append(feed)
        return feed
        
    def absorb(self): 
        "Absorb feed definitions from everywhere."
        self.absorb_from_state()
        self.absorb_from_favorites_file()
        self.absorb_from_command_line()
        self.absorb_from_bloglines()
        if not len(self): 
            self.absorb_default_feed()

    def absorb_default_feed(self): 
        "Absorb the default feed if necessary."
        if not len(self): 
            log.info("No feeds defined! Adding the default feed.")
            url = 'http://radio.weblogs.com/0001014/'\
                  'categories/ipodderTestChannel/rss.xml'
            self.addfeed(url, title="Default Channel", sub_state='subscribed')

    def absorb_from_state(self): 
        "Absorb feeds from the state database."
        state = self.state
        feeds_by_normalized_url = self.feeds_by_normalized_url
        feedcount = 0
        for key in state: 
            if not key[:5] == 'feed#': 
                continue
            feed = state[key]
            feed.fix_state() # add new attributes, etc
            feed.feeds = self # and this one :)
            # feeds_by_url is used so we can avoid loading duplicate
            # feeds from all these different sources
            if feeds_by_normalized_url.has_key(feed.normurl): 
                log.warn("Feed \"%s\" (%s) has the same URL as feed \"%s\".", 
                         feeds_by_normalized_url[feed.normurl], feed.normurl)
            else: 
                feeds_by_normalized_url[feed.normurl] = feed
            self._members.append(feed)
            feedcount = feedcount + 1
        log.info("Loaded %d feeds from the state database.", feedcount)
            
    def absorb_from_favorites_file(self): 
        "Absorb feeds from the old favorites file."
        filename = self.config.favorites_file
        name, ext = os.path.splitext(filename)
        feedcount = 0
       
        # If it's a torrent, use the other method. 
        if ext == '.torrent': 
            return self.absorb_from_opml_file(filename)

        # Load from a flat file of URLs
        log.debug("Attempting to load favorites file %s", filename)
        try: 
            feeds = file(filename, 'rt')
            for line in feeds: 
                url = line.strip()
                if not url: 
                    continue # it's an empty line!
                if url[:1] == '#': 
                    continue # it's a comment!
                try: 
                    self.addfeed(url, sub_state='newly-subscribed')
                    log.info("Added from favorites file: %s", url)
                    feedcount = feedcount + 1
                except DuplicateFeedUrl, ex: 
                    log.debug("Skipping known feed %s", url)
            feeds.close()
        except (IOError, OSError), ex: 
            errno, message = ex.args
            if errno == 2: # ENOFILE
                log.debug("... but it doesn't exist. Oops.")
            else: 
                log.exception("Ran into some problem loading feeds "\
                              "from favorites file %s", filename)
        log.info("Loaded %d new feeds from %s", feedcount, filename)

    def absorb_from_command_line(self): 
        """Absorb favorites from the command line."""
        pass # not implemented yet, but let's not make it a show-stopper

    def absorb_from_opml_file(self, filename, default_sub_state='unknown'): 
        """Absorb favorites from an OPML file, defaulting their 
        subscription state."""
        raise NotImplementedError

    def absorb_from_bloglines(self): 
        """Absorb favorites from Bloglines."""
        if not self.config.bl_username:
            return
        log.debug("Attempting to load new feeds from Bloglines...")
        if not self.config.bl_password: 
            log.error("Can't access Bloglines; no password specified.")
            return
        if not self.config.bl_folder: 
            log.error("Can't access Bloglines; no blogroll folder specified.")
            return
        newfeeds = 0
        try: 
            for url in bloglines.extractsubs(config.bl_username, 
                    config.bl_password, config.bl_folder): 
                try: 
                    url = str(url) # strip Unicode
                    self.addfeed(url, sub_state='newly-subscribed')
                    log.info("Added from Bloglines: %s", url)
                    newfeeds = newfeeds + 1
                except DuplicateFeedUrl, ex: 
                    log.debug("Skipping known feed %s", url)
            else: 
                # Little-known Python trick! This happens if there were 
                # none at all. 
                log.error("Couldn't see anything in Bloglines. Either your "\
                          "folder is wrong, or you haven't subscribed to "\
                          "anything in it.")
        except KeyError: 
            log.error("Couldn't load feeds from Bloglines because blogroll "\
                      "folder %s doesn't exist.", self.config.bl_folder)
        except KeyboardInterrupt: 
            raise
        except bloglines.urllib2.HTTPError, ex: 
            log.debug("%s", repr(ex.__dict__))
            if ex.code == 401:
                log.error("Can't access Bloglines: authentication failure.")
            elif ex.code == 404:
                log.error("Bloglines service appears to no longer be "\
                          "available where we think it is (404).")
            elif ex.code == 503:
                log.error("Bloglines service unavailable (503).")
            else:
                log.error("Can't access Bloglines; HTTP return code %d", 
                        ex.code)
            return
        except: 
            log.exception("Experimental Bloglines support failed. "\
                          "Please report the following information:")
            return
        log.info("Loaded %d new feeds from Bloglines.", newfeeds)

    def flush(self): 
        """Flush feed definitions to our various places."""
        self.write_to_state()
        self.write_to_favorites_file()

    def write_feed_to_state(self, feed, sync=True): 
        """Write one feed's state to the state database."""
        # TODO: fix grotty hack by using pickle protocol properly
        state = self.state
        feedkey = mkfeedkey(feed)
        del feed.feeds
        state[feedkey] = feed
        if sync: 
            if hasattr(state, 'sync'): 
                state.sync()
        feed.feeds = self
        
    def write_to_state(self): 
        """Flush feed definitions to the state database."""
        state = self.state
        for feed in self._members: 
            self.write_feed_to_state(feed, sync=False)
        if hasattr(state, 'sync'): 
            state.sync()
        
    def write_to_favorites_file(self): 
        """Flush feed definitions to the favorites file."""
        filename = self.config.favorites_file
        name, ext = os.path.splitext(filename)
        
        # If it's a torrent, use the other method. 
        if ext == '.torrent': 
            return self.write_to_opml_file(filename)

        # Otherwise...
        try: 
            favorites = file(filename, 'wt')
            for feed in self._members: 
                if feed.sub_state in ('disabled',): 
                    continue
                print >> favorites, "# %s" % feed
                print >> favorites, feed.url
            favorites.close()
            log.info("Wrote %d entries to %s", len(self._members), filename)
        except (IOError, OSError): 
            log.exception("Unexpected problem writing favorites file %s", 
                          filename)
        
    def write_to_opml_file(self, filename): 
        """Flush feed definitions to an OPML file."""
        raise NotImplementedError

    def __len__(self): 
        "How long are we?"
        return len(self._members)

    def __iter__(self): 
        "Support iteration through our members."
        return iter(self._members)
        
    def clean_state_database(self): 
        "Delete now-obsolete state keys." 
        state = self.state
        first = True
        for key in state.iterkeys(): 
            if key[:5] == 'feed-': 
                if first: 
                    first = False
                    log.info("Cleaning up state database of stale feed "\
                             "status items.")
                    del state[key]

if __name__ == '__main__': 
    import shelve
    import types
    import pickle
    import conlogging
    import configuration
    logging.basicConfig()
    handler = logging.StreamHandler()
    handler.formatter = conlogging.ConsoleFormatter("%(message)s", wrap=False)
    log.addHandler(handler)
    log.propagate = 0
    log.setLevel(logging.DEBUG)
    parser = configuration.makeCommandLineParser()
    options, args = parser.parse_args()
    if args: 
        parser.error("only need options; no arguments.")
    config = configuration.Configuration(options)
    state = shelve.open(config.state_db_file, 'c', 
                        writeback=False, protocol=pickle.HIGHEST_PROTOCOL)
    feeds = Feeds(config, state)
    for feed in feeds: 
        print str(feed)
        atts = [att for att in dir(feed) 
                if att[:1] != '_' 
                and not att in ['feeds']
                and not isinstance(getattr(feed, att), types.MethodType)]
        atts.sort()
        for att in atts: 
            print "  %s = %s" % (att, repr(getattr(feed, att)))
