// --------------------------------------------------------------------
// Interface for Qt and Ipelib
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeq.h"
#include "ipestyle.h"

#include <QFile>

// --------------------------------------------------------------------

/*! \class IpeQStream
  \brief Stream writing into an open QIODevice.
*/

//! Constructor.
IpeQStream::IpeQStream(QIODevice *dev)
  : iDevice(dev)
{
  // nothing
}

void IpeQStream::PutChar(char ch)
{
  iDevice->putChar(ch);
}

void IpeQStream::PutString(IpeString s)
{
  for (int i = 0; i < s.size(); ++i)
    iDevice->putChar(s[i]);
}

void IpeQStream::PutCString(const char *s)
{
  while (*s)
    iDevice->putChar(*s++);
}

void IpeQStream::PutRaw(const char *data, int size)
{
  while (size-- > 0)
    iDevice->putChar(*data++);
}

int IpeQStream::Tell() const
{
  return iDevice->pos();
}

// --------------------------------------------------------------------

/*! \class XmlQStringSource
  \brief IpeXmlDataSource reading from a QString.
*/

//! Construct XML source.
XmlQStringSource::XmlQStringSource(QString str)
{
  iData = IpeQ(str);
  iPos = 0;
}

//! Get one character.
int XmlQStringSource::GetChar()
{
  if (iPos < iData.size())
    return iData[iPos++];
  else
    return EOF;
}

// --------------------------------------------------------------------

/*! \class XmlQSource
  \brief Data source for parsing XML from a QIODevice.
*/

XmlQSource::XmlQSource(QIODevice *dev, bool skipPercent)
  : iSkipPercent(skipPercent), iDev(dev)
{
  // nothing
}

int XmlQSource::GetChar()
{
  char ch;
  if (!iDev->getChar(&ch))
    return EOF;
  if (iSkipPercent && ch == '\n')
    iDev->getChar(0); // skip '%'
  return ch;
}

// --------------------------------------------------------------------
