// --------------------------------------------------------------------
// Output device for Xpdflib that renders text to a bitmap
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipefontpool.h"
#include "ipepdfoutputdev_p.h"
#include "ipefontpool_p.h"

#ifdef __BORLANDC__
#pragma warn -8060
#endif

/*! \class IpeOutputDev
  \brief An output device for Xpdflib that renders text to a bitmap.

  IpeLatex uses this output device to create bitmaps for the text
  objects.
*/

IpeOutputDev::IpeOutputDev(XRef *xref, IpeFontPool *pool,
			   IpeText::XForm *xform)
{
  iXRef = xref;
  iFontPool = pool;
  iXForm = xform;

  iGfxFont = 0;
  iFontObject = -1;
}

//! Destructor.
IpeOutputDev::~IpeOutputDev()
{
  // nothing
}

//! Update all graphics state elements in output device.
void IpeOutputDev::updateAll(GfxState *state)
{
  updateFont(state);
}

//! Update font in output device.
void IpeOutputDev::updateFont(GfxState *state)
{
  double m11, m12, m21, m22;

  if (!(iGfxFont = state->getFont())) {
    iFontObject = -1;
    return;
  }
  if (iGfxFont->getType() == fontType3) {
    iFontObject = -1;
    return;
  }
  state->getFontTransMat(&m11, &m12, &m21, &m22);
  m11 *= state->getHorizScaling();
  m12 *= state->getHorizScaling();

  iFontPool->iPriv->GetFace(iXRef, iGfxFont);
  iFontObject = iGfxFont->getID()->num;
  iFontMatrix[0] = m11;
  iFontMatrix[1] = m12;
  iFontMatrix[2] = m21;
  iFontMatrix[3] = m22;
}

//! Draw a single character.
void IpeOutputDev::drawChar(GfxState *state, double x, double y,
			    double /*dx*/, double /*dy*/,
			    double originX, double originY,
			    CharCode code, Unicode *u, int uLen)
{
  int render;
  double x1, y1; // , dx1, dy1;
  GfxRGB rgb;

  if (iFontObject < 0)
    return;

  // check for invisible text -- this is used by Acrobat Capture
  render = state->getRender();
  if ((render & 3) == 3)
    return;

  x -= originX;
  y -= originY;
  state->transform(x, y, &x1, &y1);
  // state->transformDelta(dx, dy, &dx1, &dy1);

  // fill
  if (!(render & 1)) {
    state->getFillRGB(&rgb);
    // iFont->drawChar(state, &rgb, x1, y1, dx1, dy1, code, u, uLen);
    IpeText::PdfChar ch;
    ch.iFontObject = iFontObject;
    for (int i = 0; i < 4; ++i)
      ch.iMatrix[i] = iFontMatrix[i];
    ch.iPos = IpeVector(x1, y1);
    ch.iCharCode = code;
    ch.iUnicode = uLen > 0 ? u[0] : 0;
    ch.iRgb =
      (int(rgb.r * 255) << 16) | (int(rgb.g * 255) << 8) | int(rgb.b * 255);
    iXForm->iRender.push_back(ch);
  }

  // stroking and clipping of character paths is not supported,
  // as it is not used by Pdflatex
}

void IpeOutputDev::stroke(GfxState *state)
{
  GfxRGB rgb;
  state->getStrokeRGB(&rgb);

  double pen = state->getTransformedLineWidth();
  GfxPath *path = state->getPath();

  int n = path->getNumSubpaths();
  double x, y;

  for (int i = 0; i < n; ++i) {
    GfxSubpath *subpath = path->getSubpath(i);
    int m = subpath->getNumPoints();
    state->transform(subpath->getX(0), subpath->getY(0), &x, &y);
    int j = 1;
    while (j < m) {
      if (subpath->getCurve(j)) {
	// not supported on screen
	j += 3;
      } else {
	IpeVector v(x, y); // begin point
	state->transform(subpath->getX(j), subpath->getY(j), &x, &y);
	IpeText::PdfSeg seg;
	seg.iRgb = (int(rgb.r * 255) << 16) |
	  (int(rgb.g * 255) << 8) | int(rgb.b * 255);
	seg.iSeg = IpeSegment(v, IpeVector(x,y));
	seg.iWidth = pen;
	iXForm->iLines.push_back(seg);
	++j;
      }
    }

    if (subpath->isClosed()) {
      // not supported on screen
    }
  }
}

// --------------------------------------------------------------------
