// --------------------------------------------------------------------
// Interface with Pdflatex
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipevisitor.h"
#include "ipestyle.h"
#include "ipegroup.h"
#include "iperef.h"

#include "ipeq.h"
#include "ipepdfdoc.h"
#include "ipefontpool.h"

#include "ipelatex_p.h"
#include "ipefontpool_p.h"
#include "ipepdflib_p.h"
#include "ipepdfoutputdev_p.h"

#include <qfile.h>
#include <qtextstream.h>

/*! \class IpeLatex
  \brief Object that converts latex source to PDF format.

  This object is responsible for creating the PDF representation of
  text objects.

*/

//! Create a converter object.
IpeLatex::IpeLatex()
{
  iDoc = 0;
  iFontPool = 0;
}

//! Destructor.
IpeLatex::~IpeLatex()
{
  for (XFormList::iterator it = iXForms.begin(); it != iXForms.end(); ++it)
    delete *it;
  delete iFontPool;
  delete iDoc;
}

// --------------------------------------------------------------------

//! Return the newly created font pool and pass ownership of pool to caller.
IpeFontPool *IpeLatex::TakeFontPool()
{
  IpeFontPool *pool = iFontPool;
  iFontPool = 0;
  return pool;
}

// --------------------------------------------------------------------

class TextCollectingVisitor : public IpeVisitor {
public:
  TextCollectingVisitor(IpeLatex::TextList *list);
  virtual void VisitText(const IpeText *obj);
  virtual void VisitGroup(const IpeGroup *obj);
  virtual void VisitReference(const IpeReference *obj);
public:
  bool iTextFound;
private:
  IpeLatex::TextList *iList;
  IpeAttributeSeq iSize;
};

TextCollectingVisitor::TextCollectingVisitor(IpeLatex::TextList *list)
  : iList(list)
{
  iSize.push_back(IpeAttribute()); // null text size
}

void TextCollectingVisitor::VisitText(const IpeText *obj)
{
  IpeLatex::SText s;
  s.iText = obj;
  if (iSize.back())
    s.iSize =  iSize.back();
  else
    s.iSize = obj->Size();
  iList->push_back(s);
  iTextFound = true;
}

void TextCollectingVisitor::VisitGroup(const IpeGroup *obj)
{
  if (iSize.back())
    iSize.push_back(iSize.back());
  else
    iSize.push_back(obj->TextSize());
  for (IpeGroup::const_iterator it = obj->begin(); it != obj->end(); ++it)
    (*it)->Accept(*this);
  iSize.pop_back();
}

void TextCollectingVisitor::VisitReference(const IpeReference *obj)
{
  const IpeObject *ref = obj->Object();
  if (ref)
    ref->Accept(*this);
}

// --------------------------------------------------------------------

/*! Scan a page and insert all text objects into IpeLatex's list.
  Returns total number of text objects found so far. */
int IpeLatex::ScanPage(IpePage *page)
{
  TextCollectingVisitor visitor(&iTextObjects);
  for (IpePage::const_iterator it = page->begin(); it != page->end(); ++it) {
    visitor.iTextFound = false;
    visitor(it->Object());
    if (visitor.iTextFound)
      it->InvalidateBBox();
  }
  return iTextObjects.size();
}

/*! Create a Latex source file with all the text objects collected
  before.  The client should have prepared a directory for the
  Pdflatex run, and pass the name of the Latex source file to be
  written by IpeLatex.

  Returns the number of text objects that did not yet have an XForm,
  or a negative error code.
*/
int IpeLatex::CreateLatexSource(QString preamble,
				const IpeStyleSheet *sheet,
				QString texFile)
{
  int count = 0;
  QFile file(texFile);
  if(!file.open(IO_WriteOnly))
    return -1;
  QTextStream stream(&file);
  stream << "\\pdfcompresslevel0\n"
	 << "\\nonstopmode\n"
	 << "\\documentclass{article}\n"
	 << "\\newcommand{\\Ipechar}[1]{\\unichar{#1}}\n"
	 << "\\usepackage{color}\n";
  IpeAttributeSeq colors;
  sheet->AllNames(IpeAttribute::EColor, colors);
  for (IpeAttributeSeq::const_iterator it = colors.begin();
       it != colors.end(); ++it) {
    // only symbolic names (not black, white, void)
    IpeString name = sheet->Repository()->String(*it);
    IpeColor value = sheet->Repository()->ToColor(sheet->Find(*it));
    if (value.IsGray())
      stream << "\\definecolor{" << QIpe(name) << "}{gray}{"
	     << value.iRed << "}\n";
    else
      stream << "\\definecolor{" << QIpe(name) << "}{rgb}{"
	     << value.iRed << "," << value.iGreen << ","
	     << value.iBlue << "}\n";
  }
  stream << QIpe(sheet->TotalPreamble()) << "\n"
	 << preamble << "\n"
	 << "\\pagestyle{empty}\n"
	 << "\\newcount\\bigpoint\\dimen0=0.01bp\\bigpoint=\\dimen0\n"
	 << "\\begin{document}\n"
	 << "\\begin{picture}(500,500)\n";
  for (TextList::iterator it = iTextObjects.begin();
       it != iTextObjects.end(); ++it) {
    const IpeText *text = it->iText;

    if (!text->GetXForm())
      count++;

    // skip objects with null or unknown symbolic size
    IpeAttribute fsAttr = sheet->Find(it->iSize);
    if (!fsAttr)
      continue;

    // compute x-stretch factor from textstretch
    IpeVector stretch(1.0, 1.0);
    if (it->iSize.IsSymbolic()) {
      IpeAttribute ts(IpeAttribute::ETextStretch, true, it->iSize.Index());
      IpeAttribute tsAttr = sheet->Find(ts);
      if (tsAttr)
	stretch = sheet->Repository()->ToVector(tsAttr);
    }

    IpeAttribute color = text->Stroke();
    IpeAttribute abs = sheet->Find(color);
    // invisible text objects: run with black color
    if (color.IsNullOrVoid() || abs.IsNull())
      abs = IpeAttribute::Black();
    IpeColor col = sheet->Repository()->ToColor(abs);

    stream << "\\setbox0=\\hbox{";
    char ipeid[10];
    sprintf(ipeid, "/%08x", uint(text));
    if (text->IsMiniPage()) {
      stream << "\\begin{minipage}{" << text->Width()/stretch.iX << "bp}";
    }

    if (fsAttr.IsNumeric()) {
      double fs = fsAttr.Number();
      stream << "\\fontsize{" << fs << "}"
	     << "{" << 1.2 * fs << "bp}\\selectfont\n";
    } else
      stream << QIpe(sheet->Repository()->String(fsAttr)) << "\n";
    stream << "\\color[rgb]{"
	   << col.iRed << "," << col.iGreen << ","
	   << col.iBlue << "}%\n";
    QString txt = QIpe(text->Text());
    for (uint i = 0; i < txt.length(); ++i) {
      int uc = txt[i].unicode();
      if (uc < 0x80)
	stream << char(uc);
      else
	stream << QString("\\Ipechar{%1}").arg(uc);
    }
    if (text->IsMiniPage())
      stream << "\\end{minipage}";
    stream << "}\n"
	   << "\\count0=\\dp0\\divide\\count0 by \\bigpoint\n"
	   << "\\pdfxform attr{/IpeId " << ipeid
	   << " /IpeStretch [ " << stretch.iX << " " << stretch.iY << " ] "
      	   << " /IpeDepth \\the\\count0}"
	   << "0\\put(0,0){\\pdfrefxform\\pdflastxform}\n";
  }
  stream << "\\end{picture}\n\\end{document}\n";
  file.close();
  return count;
}

bool IpeLatex::GetXForm(PdfObject &xform)
{
  if (!xform->isStream())
    return false;
  IpeText::XForm *xf = new IpeText::XForm;
  iXForms.push_back(xf);
  xform->streamReset();
  int ch;
  while ((ch = xform->streamGetChar()) != EOF)
    xf->iStream += char(ch);
  Dict *dict = xform->streamGetDict();
  /* Should we check /Matrix and /FormType?
     /Type /XObject
     /Subtype /Form
     /IpeId /abcd1234
     /IpeDepth 246
     /IpeStretch [ 3 3 ]
     /BBox [0 0 4.639 4.289]
     /FormType 1
     /Matrix [1 0 0 1 0 0]
     /Resources 11 0 R
  */
  // Get Ipe id
  PdfObject id;
  dict->lookup("IpeId", &id);
  if (id->isNull() || !id->isName())
    return false;
  IpeLex lex(IpeString(id->getName()).substr(1));
  xf->iRefCount = lex.GetHexNumber(); // abusing refcount field
  PdfObject depth;
  dict->lookup("IpeDepth", &depth);
  if (depth->isNull() || !depth->isNum())
    return false;
  xf->iDepth = int(depth->getNum());
  PdfObject stretch, xstretch, ystretch;
  dict->lookup("IpeStretch", &stretch);
  if (stretch->isNull() || !stretch->isArray())
    return false;
  stretch->arrayGet(0, &xstretch);
  stretch->arrayGet(1, &ystretch);
  if (xstretch->isNull() || !xstretch->isNum() ||
      ystretch->isNull() || !ystretch->isNum())
     return false;
  xf->iStretch = IpeVector(xstretch->getNum(), ystretch->getNum());
  // Get BBox
  PdfObject bbox;
  dict->lookup("BBox", &bbox);
  if (bbox->isNull() || !bbox->isArray())
    return false;
  PdfObject a[4];
  for (int i = 0; i < 4; i++) {
    bbox->arrayGet(i, &a[i]);
    if (a[i]->isNull() || !a[i]->isNum())
      return false;
  }
  IpeVector bl(a[0]->getNum(), a[1]->getNum());
  IpeVector sz(a[2]->getNum(), a[3]->getNum());
  xf->iBBox.AddPoint(bl);
  xf->iBBox.AddPoint(bl + sz);
  if (xf->iBBox.Min() != IpeVector::Zero) {
    ipeDebug("PDF bounding box is not zero-aligned: (%g, %g)",
	     xf->iBBox.Min().iX, xf->iBBox.Min().iY);
    return false;
  }
  PdfObject res;
  dict->lookup("Resources", &res);
  if (res->isNull() || !res->isDict()) {
    Warn("No /Resources in XForm.");
    return false;
  }
  /*
    /Font << /F8 9 0 R /F10 18 0 R >>
    /ProcSet [ /PDF /Text ]
  */
  PdfObject fontDict;
  res->dictLookup("Font", &fontDict);
  if (!fontDict->isNull()) {
    int numFonts = fontDict->dictGetLength();
    xf->iFonts.resize(numFonts);
    for (int i = 0; i < numFonts; i++) {
      IpeString fontName(fontDict->dictGetKey(i));
      if (fontName[0] != 'F') {
	Warn(IpeString("Weird font name: ") + fontName);
	return false;
      }
      int fontNumber = IpeLex(fontName.substr(1)).GetInt();
      xf->iFonts[i] = fontNumber;
      PdfObject fontRef;
      fontDict->dictGetValNF(i, &fontRef);
      if (!fontRef->isRef() || fontRef->getRef().gen != 0) {
	Warn("Font is not indirect");
	return false;
      }
      iFontObjects[fontNumber] = fontRef->getRef().num;
    }
  }

#ifdef IPE_DEBUG_PDFLATEX
  IPEDBG << "XForm found:\n"
      << " BBox: " << xf->iBBox << "\n"
      << " Stream: " << xf->iStream << "\n"
      << " Fonts: ";
  for (int i = 0; i < xf->iFonts.size(); ++i)
    IPEDBG << xf->iFonts[i] << " ";
  IPEDBG << "\n";
#endif
  IpeOutputDev *out = new IpeOutputDev(iDoc->getXRef(), iFontPool, xf);
  Page *page1 = iDoc->getCatalog()->getPage(1); // owned by catalog
  Dict *dict1 = page1->getResourceDict(); // owned by page attributes
  Gfx *gfx = new Gfx(iDoc->getXRef(), out, dict1, page1->getBox(), gFalse, 0);
  gfx->pushResources(res->getDict());
  gfx->display(&xform, gFalse);
  gfx->popResources();
  delete gfx;
  delete out;
  return true;
}

bool IpeLatex::GetEmbeddedFont(int fno, int objno)
{
  PdfObject fontObj;
  iDoc->getXRef()->fetch(objno, 0, &fontObj);
  /*
    /Type /Font
    /Subtype /Type1
    /Encoding 24 0 R
    /FirstChar 6
    /LastChar 49
    /Widths 25 0 R
    /BaseFont /YEHLEP+CMR10
    /FontDescriptor 7 0 R
  */
  if (!fontObj->isDict("Font"))
    return false;
  // Check whether the stream data is already there.
  IpeFontPoolPriv::FontSeq::iterator it = iFontPool->iPriv->iFonts.begin();
  while (it != iFontPool->iPriv->iFonts.end() &&
	 it->iObjectNumber != objno)
    ++it;
  if (it == iFontPool->iPriv->iFonts.end()) {
    iFontPool->iPriv->iFonts.push_back(IpeFontPoolPriv::EmbeddableFont());
    it = iFontPool->iPriv->iFonts.end() - 1;
  } else {
    assert(it->iLatexNumber == -1);
  }
  IpeFontPoolPriv::EmbeddableFont &font = *it;
  font.iLatexNumber = fno;
  // get font dictionary
  PdfObject fontDescriptor;
  for (int i = 0; i < fontObj->dictGetLength(); i++) {
    IpeString key(fontObj->dictGetKey(i));
    if (key == "FontDescriptor") {
      fontObj->dictGetVal(i, &fontDescriptor);
    } else {
      PdfObject data;
      fontObj->dictGetVal(i, &data);
      if (key == "Subtype") {
	IpeString d = data.String();
	if (d != "/Type1" && d != "/TrueType") {
	  Warn("Pdflatex has embedded a font of a type not supported by Ipe.");
	  return false;
	}
      }
      font.iFontDict += IpeString("/") + key + " " + data.String() + "\n";
    }
  }
#ifdef IPE_DEBUG_PDFLATEX
  IPEDBG << "Font resource for /F" << fno << "\n" << font.iFontDict << "\n";
#endif
  // get font descriptor
  /*
    /Ascent 694
    /CapHeight 683
    /Descent -194
    /FontName /YEHLEP+CMR10
    /ItalicAngle 0
    /StemV 69
    /XHeight 431
    /FontBBox [-251 -250 1009 969]
    /Flags 4
    /CharSet (/Sigma/one)
    /FontFile 8 0 R
  */
  if (fontDescriptor->isNone())
    return true;  // it's one of the 14 base fonts, no more data needed
  if (!fontDescriptor->isDict())
    return false;
  PdfObject fontFile;
  IpeString fontFileKey;
  for (int i = 0; i < fontDescriptor->dictGetLength(); i++) {
    IpeString key(fontDescriptor->dictGetKey(i));
    if (key.size() >= 8 && key.substr(0, 8) == "FontFile") {
      fontFileKey = key;
      fontDescriptor->dictGetVal(i, &fontFile);
    } else {
      PdfObject data;
      fontDescriptor->dictGetVal(i, &data);
      font.iFontDescriptor += IpeString("/") + key + " " +
	data.String() + "\n";
    }
  }
  font.iFontDescriptor += IpeString("/") + fontFileKey + " ";
#ifdef IPE_DEBUG_PDFLATEX
  IPEDBG << "Font descriptor for /F" << fno << "\n"
      << font.iFontDescriptor << "\n";
#endif
  // get embedded font file
  if (fontFile->isNull() || !fontFile->isStream())
    return false;
  // is stream already there?
  if (font.iStreamData.size() == 0) {
    // check length of stream
    int streamDataSize = 0;
    fontFile->streamReset();
    while (fontFile->streamGetChar() != EOF)
      streamDataSize++;
    font.iStreamData = IpeBuffer(streamDataSize);
    // copy stream to buffer
    fontFile->streamReset();
    int ch;
    int p = 0;
    while ((ch = fontFile->streamGetChar()) != EOF)
      font.iStreamData[p++] = char(ch);
  }
  Dict *dict = fontFile->streamGetDict();
  for (int i = 0; i < dict->getLength(); i++) {
    PdfObject data;
    dict->getVal(i, &data);
    font.iStreamDict += IpeString("/") + dict->getKey(i) + " " +
      data.String() + "\n";
  }
  return true;
}

//! Read the PDF file created by Pdflatex.
/*! Must have performed the call to Pdflatex, and pass the name of the
  resulting output file.
*/
bool IpeLatex::ReadPdf(QString pdfFile)
{
  InitXpdfLib();
  GString *name = new GString(QFile::encodeName(pdfFile));
  iDoc = new PDFDoc(name, 0); // takes ownership of name
  if (!iDoc->isOk())
    return false;

  Page *page1 = iDoc->getCatalog()->getPage(1); // owned by catalog
  Dict *dict1 = page1->getResourceDict(); // owned by page attributes
  iFontPool = new IpeFontPool;

  PdfObject obj;
  dict1->lookup("XObject", &obj);
  if (obj->isNull() || !obj->isDict()) {
    Warn("Page 1 has no XForms.");
    return false;
  }
  // collect list of XObject's and their fonts
  for (int i = 0; i < obj->dictGetLength(); i++) {
    PdfObject xform;
    obj->dictGetVal(i, &xform);
    if (!GetXForm(xform))
      return false;
  }
  // collect all fonts
  std::map<int, int>::iterator it;
  for (it = iFontObjects.begin(); it != iFontObjects.end(); ++it) {
    int fno = it->first;
    int objno = it->second;
    if (!GetEmbeddedFont(fno, objno))
      return false;
  }
  return true;
}

//! Notify all text objects about their updated PDF code.
/*! Returns true if successful. */
bool IpeLatex::UpdateTextObjects()
{
  for (TextList::iterator it = iTextObjects.begin();
       it != iTextObjects.end(); ++it) {
    XFormList::iterator xf = iXForms.begin();
    int ipeid = int(it->iText);
    while (xf != iXForms.end() && (*xf)->iRefCount != ipeid)
      ++xf;
    if (xf == iXForms.end())
      return false;
    IpeText::XForm *xform = *xf;
    iXForms.erase(xf);
    it->iText->SetXForm(xform);
  }
  return true;
}

/*! Messages about the (mis)behaviour of Pdflatex, probably
  incomprehensible to the user. */
void IpeLatex::Warn(IpeString msg)
{
  qDebug(msg.CString());
}

// --------------------------------------------------------------------
