// -*- C++ -*-
// --------------------------------------------------------------------
// A wrapper for the FreeType font rasterizer.
//
// Copyright 2001-2002 Glyph & Cog, LLC
// Modified by Otfried Cheong
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef FTFONT_H
#define FTFONT_H

#include "ipepdflib_p.h"

#include <qcolor.h>

#include <ft2build.h>
#include FT_FREETYPE_H

//------------------------------------------------------------------------

class QPixmap;

class FTFontEngine {
public:
  FTFontEngine(bool aa);
  bool IsOk() { return iOk; }
  virtual ~FTFontEngine();

private:
  FT_Library iLib;
  bool iAa;
  bool iOk;

  friend class FTFace;
  friend class FTFont;
};

//------------------------------------------------------------------------

enum FTFontIndexMode {
  ftFontModeUnicode,
  ftFontModeCharCode,
  ftFontModeCharCodeOffset,
  ftFontModeCodeMap,
  ftFontModeCodeMapDirect,
  ftFontModeCIDToGIDMap,
  ftFontModeCFFCharset
};

class FTFace {
public:
  FTFace(Ref *id, FTFontEngine *engine, char *fontFileName);
  FTFace(Ref *id, FTFontEngine *engine,
	 const char *buffer, int bufsize);
  ~FTFace();

  void SetMode(char **fontEnc, bool pdfFontHasEncoding, bool type1);
  void SetMode(Gushort *cidToGID, int cidToGIDLen);
  void SetMode();

  //! Check whether it was loaded okay.
  bool IsOk() { return iOk; }

  //! Does this font match the ID?
  bool IsID(Ref *id) const {return (iId.num == id->num &&
				     iId.gen == id->gen);}

private:
  Ref iId;
  FTFontEngine *iEngine;
  FT_Face iFace;
  FTFontIndexMode iMode;
  int iCharMapOffset;
  Guint *iCodeMap;
  Gushort *iCidToGID;
  int iCidToGIDLen;
  bool iOk;

  friend class FTFont;
};

//------------------------------------------------------------------------

struct FTFontCacheTag {
  Gushort code;
  Gushort mru;			// valid bit (0x8000) and MRU index
  int x, y, w, h;		// offset and size of glyph
};

class FTFont {
public:
  FTFont(const FTFace *face,
	 double m11, double m12,
	 double m21, double m22);
  ~FTFont();

  //! Was font created okay?
  bool IsOk() { return iOk; }

  //! Does this font match the ID and transform?
  bool Matches(Ref *id, double m11, double m12, double m21, double m22)
  { return iFace->IsID(id) && iM11 == m11 && iM12 == m12 &&
      iM21 == m21 && iM22 == m22; }

  bool DrawChar(QPixmap *pixmap, int x, int y,
		QRgb rgb, CharCode c, Unicode u);

private:
  Guchar *GetGlyphPixmap(CharCode c, Unicode u,
			 int *x, int *y, int *w, int *h);
  FT_UInt GetGlyphIndex(CharCode c, Unicode u);

 private:
  FTFace *iFace;
  double iM11, iM12, iM21, iM22;
  FT_Matrix iMatrix;
  int iGlyphW, iGlyphH;		// size of glyph pixmaps
  int iGlyphSize;		// size of glyph pixmaps, in bytes
  Guchar *iCache;		// glyph pixmap cache
  FTFontCacheTag *iCacheTags;	// cache tags, i.e., char codes
  int iCacheSets;		// number of sets in cache
  int iCacheAssoc;		// cache associativity (glyphs per set)
  bool iOk;
};

// --------------------------------------------------------------------
#endif
