// --------------------------------------------------------------------
// Convert image streams to QImage's
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipefilter.h"
#include "ipepdflib_p.h"

// --------------------------------------------------------------------

QImage IpeImageFilter::Decode(IpeBitmap bitmap)
{
  QImage qimg(bitmap.Width(), bitmap.Height(), 32);

  // need to convert image data to qimg
  if (bitmap.Filter() == IpeBitmap::EDirect) {
    const char *data = bitmap.Data();
    for (int row = 0; row < bitmap.Height(); ++row) {
      uint *p = (uint *) qimg.scanLine(row);
      for (int col = 0; col < bitmap.Width(); ++col) {
	if (bitmap.ColorSpace() == IpeBitmap::EDeviceRGB) {
	  char r = *data++;
	  char g = *data++;
	  char b = *data++;
	  *p++ = qRgb(r, g, b);
	} else {
	  char g = *data++;
	  *p++ = qRgb(g, g, g);
	}
      }
    }
  } else {
    // JPEG image
    GfxColorSpace *cs = 0;
    if (bitmap.ColorSpace() == IpeBitmap::EDeviceGray)
      cs = new GfxDeviceGrayColorSpace();
    else if (bitmap.ColorSpace() == IpeBitmap::EDeviceRGB)
      cs = new GfxDeviceRGBColorSpace();
    else if (bitmap.ColorSpace() == IpeBitmap::EDeviceCMYK)
      cs = new GfxDeviceCMYKColorSpace();

    PdfObject decode;
    decode->initNull();
    // colorMap takes ownership of cs
    GfxImageColorMap *colorMap =
      new GfxImageColorMap(bitmap.BitsPerComponent(), &decode, cs);

    Object dictObj;
    dictObj.initDict(0);
    MemStream *memStr = new MemStream(const_cast<char *>(bitmap.Data()),
				      bitmap.Size(), &dictObj);

    FilterStream *str = 0;
    if (bitmap.Filter() == IpeBitmap::EDCTDecode)
      str = new DCTStream(memStr);  // takes ownership of memStr
    else
      // 1 means without predictor---remaining args don't matter then
      str = new FlateStream(memStr, 1, 1, 1, 8);

    // imgStr does not take ownership of str
    ImageStream *imgStr = new ImageStream(str, bitmap.Width(),
					  bitmap.Components(),
					  bitmap.BitsPerComponent());
    imgStr->reset();

    GfxRGB rgb;
    // for each line...
    for (int y = 0; y < bitmap.Height(); ++y) {
      // write the line
      Guchar *p = imgStr->getLine();
      uint *dst = (uint *) qimg.scanLine(y);
      for (int x = 0; x < bitmap.Width(); ++x) {
	colorMap->getRGB(p, &rgb);
	*dst++ = qRgb(int(rgb.r * 255 + 0.5),
		      int(rgb.g * 255 + 0.5),
		      int(rgb.b * 255 + 0.5));
	p += colorMap->getNumPixelComps();
      }
    }
    delete imgStr;
    delete str;
    delete colorMap;
  }
  return qimg;
}

// --------------------------------------------------------------------
