// -*- C++ -*-
// --------------------------------------------------------------------
// Base header file --- must be included by all Ipe components.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPEBASE_H
#define IPEBASE_H

#include <cassert>
#include <cstdio>
#include <vector>
#include <map>
#include <list>
#include <algorithm>

// --------------------------------------------------------------------

#ifdef WIN32
typedef unsigned int uint;
typedef unsigned long int ulong;
#if defined(IPE_MAKEDLL)	/* create a DLL library */
#define IPE_EXPORT  __declspec(dllexport)
#else
#define IPE_EXPORT  __declspec(dllimport)
#endif
#else
#define IPE_EXPORT
#endif

// --------------------------------------------------------------------

//! Ipelib version.
const int IPELIB_VERSION = 60016;

extern IPE_EXPORT int IpelibVersion();

extern IPE_EXPORT void (*ipeDebugHandler)(const char *);

extern IPE_EXPORT void ipeDebug(const char *msg, ...);

template<class T>
class IpeAutoPtr {
public:
  IpeAutoPtr(T *ptr) : iPtr(ptr) { /* nothing */ }
  ~IpeAutoPtr() { delete iPtr; }
  T *operator->() { return iPtr; }
  T &operator*() { return *iPtr; }
  T *Ptr() { return iPtr; }
  T *Take() { T *p = iPtr; iPtr = 0; return p; }
private:
  T *iPtr;
};

template<class T>
inline bool operator!=(const T &lhs, const T &rhs)
{
  return !(lhs == rhs);
}

// --------------------------------------------------------------------

class IPE_EXPORT IpeString {
public:
  IpeString();
  IpeString(const char *str);
  IpeString(const IpeString &rhs);
  IpeString(const IpeString &rhs, int index, int len);
  IpeString &operator=(const IpeString &rhs);
  ~IpeString();
  //! Return character at index i.
  inline char operator[](int i) const { return iImp->iData[i]; }
  //! Is the string empty?
  inline bool empty() const { return (size() == 0); }
  //! Return read-only pointer to the data.
  const char *data() const { return iImp->iData; }
  //! Return number of bytes in the string.
  inline int size() const { return iImp->iSize; }
  //! Operator syntax for append.
  inline void operator+=(const IpeString &rhs) { append(rhs); }
  //! Operator syntax for append.
  void operator+=(char ch) { append(ch); }
  //! Create substring.
  inline IpeString substr(int i, int len = -1) const {
    return IpeString(*this, i, len); }
  //! Operator !=.
  inline bool operator!=(const IpeString &rhs) const {
    return !(*this == rhs); }

  void erase();
  void append(const IpeString &rhs);
  void append(char ch);
  bool operator==(const IpeString &rhs) const;
  bool operator<(const IpeString &rhs) const;
  IpeString operator+(const IpeString &rhs) const;
  const char *CString() const;
private:
  void Detach(int n);
private:
  struct Imp {
    int iRefCount;
    int iSize;
    int iCapacity;
    char *iData;
  };
  Imp *iImp;
};

// --------------------------------------------------------------------

class IPE_EXPORT IpeStream {
public:
  //! Virtual destructor, so you can delete using a pointer to IpeStream
  virtual ~IpeStream();
  //! Output character.
  virtual void PutChar(char ch) = 0;
  //! Output string.
  virtual void PutString(IpeString s) = 0;
  //! Output C string.
  virtual void PutCString(const char *s) = 0;
  //! Output raw character data.
  virtual void PutRaw(const char *data, int size) = 0;
  //! Output character.
  inline IpeStream &operator<<(char ch) { PutChar(ch); return *this; }
  //! Output string.
  inline IpeStream &operator<<(const IpeString &s) {
    PutString(s); return *this; }
  //! Output C string.
  inline IpeStream &operator<<(const char *s) { PutCString(s); return *this; }
  IpeStream &operator<<(int i);
  IpeStream &operator<<(double d);
  void PutHexByte(char b);
  void PutXmlString(IpeString s);
};

class IPE_EXPORT IpeStringStream : public IpeStream {
public:
  IpeStringStream(IpeString &string);
  virtual void PutChar(char ch);
  virtual void PutString(IpeString s);
  virtual void PutCString(const char *s);
  virtual void PutRaw(const char *data, int size);
private:
  IpeString &iString;
};

class IPE_EXPORT IpeFileStream : public IpeStream {
public:
  IpeFileStream(std::FILE *file);
  virtual void PutChar(char ch);
  virtual void PutString(IpeString s);
  virtual void PutCString(const char *s);
  virtual void PutRaw(const char *data, int size);
private:
  std::FILE *iFile;
};

class IPE_EXPORT IpeLex {
public:
  explicit IpeLex(IpeString str);

  IpeString Token();
  IpeString NextToken();
  int GetInt();
  int GetHexByte();
  int GetHexNumber();
  double GetDouble();
  //! Extract next character (not skipping anything).
  inline char GetChar() {
    return iString[iPos++]; }
  void SkipWhitespace();

  //! Operator syntax for GetInt().
  inline IpeLex &operator>>(int &i) {
    i = GetInt(); return *this; }
  //! Operator syntax for GetDouble().
  inline IpeLex &operator>>(double &d) {
    d = GetDouble(); return *this; }

  //! Mark the current position.
  inline void Mark() {
    iMark = iPos; }
  //! Reset reader to the marked position.
  inline void FromMark() {
    iPos = iMark; }

  //! Return true if at end of string (not even whitespace left).
  inline bool Eos() const {
    return (iPos == iString.size()); }

private:
  IpeString iString;
  int iPos;
  int iMark;
};

class IPE_EXPORT IpeBuffer {
public:
  IpeBuffer();
  ~IpeBuffer();
  IpeBuffer(const IpeBuffer &rhs);
  IpeBuffer &operator=(const IpeBuffer &rhs);

  explicit IpeBuffer(int size);
  explicit IpeBuffer(const char *data, int size);
  //! Character access.
  inline char &operator[](int index) { return iImp->iData[index]; }
  //! Character access (const version).
  inline const char &operator[](int index) const { return iImp->iData[index]; }
  //! Return size of buffer;
  inline int size() const { return iImp->iSize; }
  //! Return pointer to buffer data.
  inline char *data() { return iImp->iData; }
  //! Return pointer to buffer data (const version).
  inline const char *data() const { return iImp->iData; }

private:
  struct Imp {
    int iRefCount;
    char *iData;
    int iSize;
  };
  Imp *iImp;
};

// --------------------------------------------------------------------
#endif
