#!/usr/bin/perl -w

$sqlConnStr = "dbi:Pg:dbname=ipac;host=development.corpex.de;port=5432;";
$SQLuser = "root";
$SQLpass = "";

$SourceTable = "logs";
$TargetTable = "light_logs";

$TimestampFile = "/var/lib/ipac/ipacsum-light.stamps";

# This file is very important, since this script works incrementally.
# it must be present and must contain 2 lines:
#
# 1st line: The biggest timestamp, which we've read the 
#           last time we ran the script
# 2nd line: The actual timestamp the script was run last time...
#
# Use this command to create it the first time...
# echo -e "0\n0\n" > /var/lib/ipac/ipacsum-light.stamps




# SQL create for light_logs
############################
#
# CREATE TABLE "light_logs" (
#         "rule_name1" character varying(100) NOT NULL,
#         "rule_name2" character varying(100),
#         "rule_name3" character varying(100),
#         "timeframe" character varying(15) NOT NULL,
#         "bytes" bigint DEFAULT 0 NOT NULL
# );
# CREATE  INDEX "timeframe" on "light_logs" using hash ( "timeframe" "varchar_ops" );
# CREATE  INDEX "rule_name1" on "light_logs" using hash ( "rule_name1" "varchar_ops" );
# CREATE  INDEX "rule_name2" on "light_logs" using hash ( "rule_name2" "varchar_ops" );
# CREATE  INDEX "rule_name3" on "light_logs" using hash ( "rule_name3" "varchar_ops" );
#
# ...I think hash indices are faster than btrees in this case.... anyone knows better?
# ...and yes, the table above could be "better performing" as well ;)





#####################################
##  No need to edit below....      ##
#####################################



use DBI;
use POSIX qw(strftime);
use POSIX qw(ceil);
require "time_functions.pl";

use IO::Handle;
STDERR->autoflush(1);
STDOUT->autoflush(1);


$CurrentTime = time();
@pre_updates = ();
$verbose = 1;

## check the Timstampfile
## if we can't write to it... exit!
####################################

if(! -w $TimestampFile) {
	print "ERROR: Cannot stat/write to '$TimestampFile'. This file is essential!\n";
	print "ERROR: please correct this. exiting...";
	exit(99);
}

#read the timestamp file, that contains 2 unix timestamps
# 1st line: The biggest timestamp, that we've read the last time
# 2nd line: The last time this script was run...
open(F,"<$TimestampFile") || die "ERROR: Could not open TimestampFile '$TimestampFile'\n" . 
						"If you run this script for the first time, please create an empty file\n".
						"with two lines each containing a '0'\n";

$LastTimeStamp = <F>;
$LastTimeRun = <F>;
close(F);
chomp($LastTimeStamp);
chomp($LastTimeRun);



# lazy options reading...
if(scalar @ARGV > 0) {
	for($i=0; $i <= $#ARGV;$i++) {
	
		if($ARGV[$i] eq "--recreate") {
			push(@pre_updates,"DELETE FROM $TargetTable");
			$LastTimeStamp = 0;
			$LastTimeRun = 0;
		}
	
		elsif($ARGV[$i] eq "--quiet") {
				$verbose = 0;
		}
	
		else {
			&print_usage();
			exit(1);
		}
	}
}




## check integrity of $LastTimeStamp
####################################
if($LastTimeRun eq '') {
	$LastTimeRun = 0;
	print "+++ Couldn't read the timestamp when this script was run last.\n".
		"+++ Assuming you're running it for the first time or you want to \n".
		"+++ (re)create the whole database. If you want to recreate the db, make \n".
		"+++ sure that you have DELETED all former records!\n";
}

if($LastTimeStamp eq '') {
	$LastTimeStamp = 0;
	print "+++ Couldn't read the timestamp that identifies the last entry this script had read\n".
		"+++ Assuming you're running it for the first time or you want to \n".
		"+++ (re)create the whole database. If you want to recreate the db, make \n".
		"+++ sure that you have DELETED all former records!\n";
}

elsif ($LastTimeStamp > $CurrentTime || $LastTimeRun > $CurrentTime) {
	print "ERROR: Your LastTimeStamp or your LastTimeRun is greater than the current TimeStamp. CORRECT THIS!";
	print "+++ exiting....\n";
	exit(99);
}




##Check, whether it was at least yesterday, when this script was last run.
##if yes, delete timeframe='yesterday' and possibly UPDATE today to be yesterday
#########
if(strftime("%d",localtime($LastTimeRun)) != strftime("%d",localtime($CurrentTime))) {
	print "+++ It was at least yesterday that you run this script the last time.\n"
		if($verbose);
		
	push(@pre_updates,"DELETE FROM $TargetTable WHERE timeframe = 'yesterday'");
	
	if(strftime("%Y%m%d",localtime($LastTimeRun)) == strftime("%Y%m%d",localtime($CurrentTime)) - 1) {
		print "+++ It WAS yesterday that you run this script the last time.\n"
			if($verbose);
		push(@pre_updates,"UPDATE $TargetTable SET timeframe = 'yesterday' WHERE timeframe = 'today'");
	}
	else {
		print "+++ It was longer than yesterday that you run this script the last time.\n"
			if($verbose);
		push(@pre_updates,"DELETE FROM $TargetTable WHERE timeframe = 'today'");
	}
		
}


##Same as above but with weeks
#########
if(strftime("%Y%V",localtime($LastTimeRun)) != strftime("%Y%V",localtime($CurrentTime))) {
	print "+++ There was a week change...\n"
		if($verbose);
	push(@pre_updates,"DELETE FROM $TargetTable WHERE timeframe = 'last_week'");
	
	if(strftime("%Y%V",localtime($LastTimeRun)) == strftime("%Y%V",localtime($CurrentTime)) - 1) {
		print "+++ It WAS last week that you run this script the last time.\n"
			if($verbose);
		push(@pre_updates,"UPDATE $TargetTable SET timeframe = 'last_week' WHERE timeframe = 'this_week'");
	}
	else {
		print "+++ It was longer ago than last week, that you run this script the last time.\n"
			if($verbose);
		push(@pre_updates,"DELETE FROM $TargetTable WHERE timeframe = 'this_week'");
	}	
}


##set some variables to save time later...
#####
$today_start = &getTodaysFirstStamp();
$today_end = &getTodaysLastStamp();

$yesterday_start = &getYesterdaysFirstStamp();
$yesterday_end = &getYesterdaysLastStamp();

$this_week_start = &getThisWeeksFirstStamp();
$this_week_end = &getThisWeeksLastStamp();

$last_week_start = &getLastWeeksFirstStamp();
$last_week_end = &getLastWeeksLastStamp();

$this_month_start = &getThisMonthsFirstStamp();
$this_month_end = &getThisMonthsLastStamp();

$last_month_start = &getLastMonthsFirstStamp();
$last_month_end = &getLastMonthsLastStamp();


$this_month_ident = &getMonthFromStamp($this_month_start);
$last_month_ident = &getMonthFromStamp($last_month_start);



##init some vars
%UPDATE = ();
@available_rules = ();
$max_timestamp = $LastTimeStamp;
$i=0;
$count_updates = 0;
$percent = 0;

	
## hello database
$link = DBI->connect($sqlConnStr, $SQLuser,$SQLpass)
	or die "Can't make database connect: $DBI::errstr\n";


$SelectSQL = "SELECT * FROM " . $SourceTable . " WHERE that_time > ?";

print "+++ Selecting from database...\n"
	if($verbose);

# prepare the Select Statement and retrieve all rows, that are 'new'
$handler = $link->prepare($SelectSQL);
$handler->execute($LastTimeStamp);

$rows = $handler->rows;
$mod = ceil($rows / 100);

print "+++ Found $rows entries to process\n+++ now processing: "
	if($verbose);


printf("%03d%%",$percent)
	if($verbose);

### walk through the result set and update the associative array.....
while ( $row = $handler->fetchrow_hashref() ) {
	
	if((($i++%$mod) == 0) && $verbose == 1) {
		printf("\b\b\b\b%03d%%",$percent++);
	}
	
	# in case we there has been data inserted into the db after the script started
	next
		if($row->{"that_time"} > $CurrentTime);
		
	## at the end, we'll end up with the biggest timestamp, we've processed.
	## this will later be written to the file....
	$max_timestamp = $row->{"that_time"}
		if($max_timestamp < $row->{"that_time"});
	
	# skip unimportant lines....
	next
		if($row->{"bytes"} == 0);	
	
	## today?
	if(($row->{"that_time"} > $today_start) && ($row->{"that_time"} < $today_end)) {
		&update_array("today",$row->{"rule_name"},$row->{"bytes"});
		&update_array("this_week",$row->{"rule_name"},$row->{"bytes"});
		&update_array($this_month_ident,$row->{"rule_name"},$row->{"bytes"});
		next;	
	}
	
	## or yesterday?
	elsif(($row->{"that_time"} > $yesterday_start) && ($row->{"that_time"} < $yesterday_end)) {
		&update_array("yesterday",$row->{"rule_name"},$row->{"bytes"});
	}
	
	###############
	
	#this week?
	if(($row->{"that_time"} > $this_week_start) && ($row->{"that_time"} < $this_week_end)) {
		&update_array("this_week",$row->{"rule_name"},$row->{"bytes"});
	}
	#last week?
	elsif(($row->{"that_time"} > $last_week_start) && ($row->{"that_time"} < $last_week_end)) {
		&update_array("last_week",$row->{"rule_name"},$row->{"bytes"});
	}

	################
	
	#this month?
	if(($row->{"that_time"} > $this_month_start) && ($row->{"that_time"} < $this_month_end)) {
		&update_array($this_month_ident,$row->{"rule_name"},$row->{"bytes"});
	}
	
	#last month?
	elsif(($row->{"that_time"} > $last_month_start) && ($row->{"that_time"} < $last_month_end)) {
		&update_array($last_month_ident,$row->{"rule_name"},$row->{"bytes"});
	}
	
	#any other month!
	else {
		&update_array(&getMonthFromStamp($row->{"that_time"}),$row->{"rule_name"},$row->{"bytes"});
	}
		
}


printf("\b\b\b\b%03d%%\n",100)
	if($verbose);



###################################################
# data gathering done!!!!!!!!1
# now it's time to insert them into the database...
###################################################


# be transaction safe!
$link->begin_work;

# do we have to do any updates to the database beforehand?
if(scalar @pre_updates > 0) {
	print "+++ Shifting old database entries\n"
		if($verbose);
	
	foreach $cmd (@pre_updates) {	
		$link->do($cmd);
	}
}



#init some vars...
$mod = ceil($count_updates / 100);
$percent = 0;
$i=0;

print "+++ Updating database ($count_updates updates/inserts): "
	if($verbose);
printf("%03d%%",$percent)
	if($verbose);


# prepare the update and insert statements
######
$UpdateSQL = "UPDATE $TargetTable SET bytes = bytes + ? WHERE timeframe = ? AND rule_name1 = ? AND rule_name2 = ? AND rule_name3 = ?";
$InsertSQL = "INSERT INTO $TargetTable (bytes,timeframe,rule_name1,rule_name2,rule_name3) VALUES(?,?,?,?,?)";

$Uhandler = $link->prepare($UpdateSQL);
$Ihandler = $link->prepare($InsertSQL);


# walk through the associative array, that 
# contains all data necessary to create the
# updates/inserts
############################################

@keys = keys(%UPDATE);
#walk through array, 1st level
foreach $key (@keys) {
	#walk through array, 2nd level	
	foreach $rule (@available_rules) {
		
		#does the rule exist?
		if(exists($UPDATE{$key}->{$rule})) {
			if(($i++%$mod) == 0) {
				printf("\b\b\b\b%03d%%",$percent++)
					if($verbose);
			}
			
			# split the rule by whitespaces into 3 parts.
			# this will...
			# 1. make update queries faster
			# 2. allow us to easily display date for ie. ports
			#
			# intended for use with rule names such as:
			####   10.0.0.1 smtp in
			####   10.0.0.1 smtp out
			####   etc.
			
			my ($r1,$r2,$r3) = split(/\s+/,$rule,3);
			
			$Uhandler->execute($UPDATE{$key}->{$rule},$key,$r1,$r2,$r3);
		
			# if the update didn't succeed, we probably need to do an insert,
			# to lazy to do error checking here...
			if($Uhandler->rows == 0) {
				$Ihandler->execute($UPDATE{$key}->{$rule},$key,$r1,$r2,$r3);
			}
			
			# delete the key, in order to save memory
			# probably this will cost more time, than it actually saves us.
			# does anyone know?
			
			delete $UPDATE{$key}->{$rule};
		}
	}
}


printf("\b\b\b\b%03d%%\n+++ Update done\n",100)
	if($verbose);

# open the timestamp file, print the new timestamps into that
# file, commit the updates/inserts/deletes and ... EXIT, we're finally done ;)

if(! open(F,">$TimestampFile")) {
	$link->rollback;
	$link->disconnect;
	die "ERROR: Could not open TimestampFile '$TimestampFile' for writing\n";
}

print F "$max_timestamp\n";
print F "$CurrentTime\n";
close(F);

$link->commit;

print "+++ finished...\n"
	if($verbose);
exit(0);






sub update_array {
	## this function updates an associative array,
	## that will later be used to create update/insert statements
	
	my $tframe = shift(@_);
	my $rule = shift(@_);
	my $size = shift(@_);
	
	my $found = 0;
	my $r;
	
	if(! exists($UPDATE{$tframe})) {
		$UPDATE{$tframe} = {};
	}
	
	
	if(! exists($UPDATE{$tframe}->{$rule})) {
			$UPDATE{$tframe}->{$rule} = $size;
			$count_updates++;
	}
	
	else {
		$UPDATE{$tframe}->{$rule} += $size;
	}
	
	
	
	# ...I know, the following is bad ;)
	# I've tried to figure out a better for
	# hrr,... exactly 2 minutes, and I've given up ;)
	# it's late by now ;)
	
	foreach $r (@available_rules) {
		if($r eq "$rule") {
			$found = 1;
			last;
		}
	}
	if($found == 0) {
		push(@available_rules,$rule);
	}
}



sub print_usage {
	print "\nipacsum-light.pl - pg\@philipp.de.com - 2001-11-27\n";
	print "++++++++++++++++\n";
	print "Summarizes data gathered from ipac-ng in a different/faster way.\n";
	print "However, it will only produce an overview, no detailed data.\n";
	print "ipacsum-light.pl works with postgres only.\n\n";
	print "ipacsum-light.pl [--recreate] [--quiet]\n";
	print "--recreate\twill recreate the logs from scratch\n";
	print "--quiet\t\twill print no information about progressing\n\n";
}