/*********************************************************************
 * Filename:      utility.c
 * Version:       $Id: utility.c,v 1.20 2002/02/09 11:35:36 tfheen Exp $
 * Description:   
 * Author:        Tollef Fog Heen <tollef@add.no>
 * Created at:    Sun Oct 15 00:18:37 2000
 * Modified at:   Sat Feb  9 12:24:54 2002
 * Modified by:   Tollef Fog Heen <tollef@add.no>
 ********************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include <net/route.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <libnet.h>
#include <pcap.h>
#include <dirent.h>
#include <fcntl.h>

#include "utility.h"
#include "intuitively.h"

/*
 * Function parsenetmask (s,sockaddr_storage)
 *
 *    parses a string and returns the netmask.  Knows how to parse both netmasks
 *    of type 255.255.255.0 and 24, 28 etc.  Return value is zero if
 *    something went wrong, 1 if the netmask was of type 1 (255.255.255.
 *    0 notation), 2 if it was /24 notation.
 *
 */
int parsenetmask(const char *s,struct sockaddr_storage* netmask) {
  int ret;
  unsigned int ip1,ip2,ip3,ip4;
  switch (netmask->ss_family) {
  case AF_INET:
    ret =  net_aton(s, netmask);
    if ((ret == 1) && (sscanf(s,"%u.%u.%u.%u",&ip1,&ip2,&ip3,&ip4) == 4)) {
      return 1;
    } else if (sscanf(s,"%u", &ip1) == 1) {
      if (ip1 > 31) {  /* We don't do 32 bit and above. */
	return 0;
      }
      ((struct sockaddr_in *)netmask)->sin_addr.s_addr = htonl(0xffffffff << (32 - ip1));
      return 2;
    } else {
      return 0;
    }
    break;
  default:
    fprintf(stderr,"parsenetmask::Unknown protocol: %s\n", s);
    return 3;
  }
}

/*
 * Function configure_iface (entry, device)
 *
 *    Configures the interface according to the contents of the struct
 *    entry.
 * 
 */

int configure_iface(struct entry* entry, char* device) {
  int s;
  struct sockaddr_in *addrp;
  struct rtentry route;
  struct ifreq req;
  struct sockaddr_in addr;
  
  if (switches.verbose >= 0) 
    fprintf(stderr,"Found: \"%s\"\n",entry->description);

  if (switches.verbose >= 3) 
    fprintf(stderr,"configure_iface(entry,%s)\n",device);

  
  if (entry->dontsetip) {
    return 0;
  }
  switch (entry->myip.ss_family) {
  case AF_INET: {
    struct sockaddr_in *myip, *netmask, *broadcast, *gw;
    myip = ((struct sockaddr_in*)&entry->myip);
    netmask = ((struct sockaddr_in*)&entry->netmask);
    broadcast = ((struct sockaddr_in*)&entry->broadcast);
    gw = ((struct sockaddr_in*)&entry->gw);
    
    s = socket(AF_INET, SOCK_DGRAM, 0);
    
    /* we have to have basic information to get this far */
    addrp = (struct sockaddr_in *) &req.ifr_addr;
    addrp->sin_family = AF_INET;
    addrp->sin_port = 0;
    strcpy(req.ifr_name, device);
    
    addrp->sin_addr = myip->sin_addr;
    if (ioctl(s, SIOCSIFADDR, &req)) {
      perror("SIOCSIFADDR");
      return 2;
    }
    addrp->sin_addr = netmask->sin_addr;
    if (ioctl(s, SIOCSIFNETMASK, &req)) {
      perror("SIOCSIFNETMASK");
      return 2;
    }
    
    if (!broadcast->sin_addr.s_addr) {

      broadcast->sin_addr.s_addr = ((myip->sin_addr.s_addr & netmask->sin_addr.s_addr) + (0xffffffff- netmask->sin_addr.s_addr));
    }
    
    addrp->sin_addr = broadcast->sin_addr;
    if (ioctl(s, SIOCSIFBRDADDR, &req)) {
      perror("SIOCSIFBRDADDR");
      return 2;
    }
    
    /* Take the interface down - we don't want to carry old routes etc
       around */
    req.ifr_flags = 0;
    if (ioctl(s, SIOCSIFFLAGS, &req)) {
      perror("Taking down iface (SIOCSIFFLAGS)");
      return 2;
    }
    
    /* bring up the device, and specifically allow broadcasts through it */
    req.ifr_flags = IFF_UP | IFF_RUNNING | IFF_BROADCAST;
    if (ioctl(s, SIOCSIFFLAGS, &req)) {
      perror("Bringing up the iface (SIOCSIFFLAGS)");
      return 2;
    }
    
    /* Bring up the default gw - if there is one */
    
    if (gw->sin_addr.s_addr != 0) {

      addr.sin_family = AF_INET;
      addr.sin_port = 0;
      addr.sin_addr.s_addr = INADDR_ANY;
      memcpy(&route.rt_dst, &addr, sizeof(addr));
      memcpy(&route.rt_genmask, &addr, sizeof(addr));
      addr.sin_addr = gw->sin_addr;
      memcpy(&route.rt_gateway, &addr, sizeof(addr));
      
      route.rt_flags = RTF_UP | RTF_GATEWAY;
      route.rt_metric = 0;
      route.rt_dev = NULL;
      
      if (ioctl(s, SIOCADDRT, &route)) {
	fprintf(stderr,"%s \n",net_ntoa(entry->gw));
	perror("Setting default route (SIOCADDRT)");
	return 2;
      }
    }
    return 0;
  }
  break;

  default:
    fprintf(stderr,"configure_iface::Unknown protocol family\n");
    break;
    return 3;
  }
  return 3;

}

void dump_entry_as_text(struct entry *e) {
  int i;
  if (!e) 
    return;
  if (e->description != NULL) 
    fprintf(stderr, "[%s]\n",e->description);
  else
    fprintf(stderr,"[%s]\n",(e->dirroot != NULL ? e->dirroot : ""));
  fprintf(stderr, "myip = %s\n",net_ntoa(e->myip));
  fprintf(stderr, "default_gw = %s\n", net_ntoa(e->gw));
  fprintf(stderr, "netmask = %s\n", net_ntoa(e->netmask));
  if (net_ntoa(e->broadcast) != NULL) 
    fprintf(stderr, "broadcast = %s\n", net_ntoa(e->broadcast));
  fprintf(stderr, "script = %s\n",(e->script != NULL ? e->script : ""));
  fprintf(stderr, "description = %s\n",(e->description != NULL ? e->description : ""));
  for (i = 0; i < e->pingips; i++) {
    fprintf(stderr, "other_ip = %s\n", net_ntoa(e->pingip[i].ip));
    if (e->pingip[i].mac != NULL)
      fprintf(stderr, "other_mac = %s\n", e->pingip[i].mac);
  }
  fprintf(stderr,"\n");

}


/*
 * Function get_current_entry (device)
 *
 *    Populates a newly allocated struct entry according to the
 *    current settings.
 * 
 */

struct entry* get_current_entry(char* device) {
  int s;
  struct entry *retval;
  struct sockaddr_in *addrp;
  struct ifreq req;
  
  if (device == NULL)
    device = "eth0";

  retval = malloc(sizeof(struct entry));
  memset(retval,'\0',sizeof(struct entry));

  retval->dirroot = "CHANGEME";

  retval->myip.ss_family = AF_INET; /* FIXME - this is IPv4 only */
  retval->netmask.ss_family = AF_INET; /* FIXME - this is IPv4 only */
  retval->broadcast.ss_family = AF_INET; /* FIXME - this is IPv4 only */
  retval->gw.ss_family = AF_INET; /* FIXME - this is IPv4 only */

  switch (retval->myip.ss_family) {
  case AF_INET: {
    struct sockaddr_in *myip, *netmask, *broadcast, *gw;
    myip = ((struct sockaddr_in*)&retval->myip);
    netmask = ((struct sockaddr_in*)&retval->netmask);
    broadcast = ((struct sockaddr_in*)&retval->broadcast);
    gw = ((struct sockaddr_in*)&retval->gw);
    
    s = socket(AF_INET, SOCK_DGRAM, 0);
    
    /* we have to have basic information to get this far */
    addrp = (struct sockaddr_in *) &req.ifr_addr;
    addrp->sin_family = AF_INET;
    addrp->sin_port = 0;
    strcpy(req.ifr_name, device);
    
    if (ioctl(s, SIOCGIFADDR, &req)) {
      perror("Getting interface address (SIOCGIFADDR)");
      return NULL;
    }
    ((struct sockaddr_in*)&retval->myip)->sin_addr = addrp->sin_addr;

    if (ioctl(s, SIOCGIFNETMASK, &req)) {
      perror("Getting netmask (SIOCGIFNETMASK)");
      return NULL;
    }
    ((struct sockaddr_in*)&retval->netmask)->sin_addr = addrp->sin_addr;
    
    if (ioctl(s, SIOCGIFBRDADDR, &req)) {
      perror("Getting broadcast address (SIOCGIFBRDADDR)");
      return NULL;
    }
    ((struct sockaddr_in*)&retval->broadcast)->sin_addr = addrp->sin_addr;
    broadcast->sin_addr = addrp->sin_addr;

    /* ok, now for the hard part - routing stuff */

    if (get_default_route(gw) != 0) {
      fprintf(stderr,"Couldn't get default route!\n");
    }
    /* The router _has_ to be on the local subnet and reachable via
       ARP so, let's add that one to pingips */

    memcpy(&retval->pingip[retval->pingips].ip,&retval->gw,sizeof(struct sockaddr_storage));
    retval->pingips++;

    return retval;
  }
  break;

  default:
    fprintf(stderr,"get_current_entry::Unknown protocol family\n");
    break;
    return NULL;
  }
  return NULL;

}

/*
 * Function get_default_route (struct sockaddr_in *gw)
 *
 *    puts the default gw into gw
 *
 */

int get_default_route(struct sockaddr_in *gw) {
  char *b;
  FILE* rfile;

  rfile = fopen("/proc/net/route","r");
  if (rfile == NULL) {
    perror("Can't open /proc/net/route");
    return -1;
  }
  b = malloc(512);  /* /proc/net/route shouldn't have longer lines
		       than this - if so, I flag it as an error.  
		       Please tell me if this causes any problems.*/
  
  while (!feof(rfile)) {
    char iface_s[10],  flags_s[10], refcnt_s[10], use_s[10], metric_s[10], 
      netmask_s[10], mtu_s[10], window_s[10], irtt_s[10];
    unsigned int gw_i = 0;
    unsigned int dest_i = 0;
    
    b = fgets(b, 512,rfile);
    if (strlen(b) == 0) 
      continue;
    if (strlen(b) == 512) /* Line too long! */
      fprintf(stderr,"/proc/net/route: line too long!: %s\n",b);
    if (strncmp(b,"Iface",5) == 0) /* First line - skip it */
      continue;
    
    sscanf(b,"%s %x %x %s %s %s %s %s %s %s %s", iface_s, &dest_i, &gw_i, flags_s, 
	   refcnt_s, use_s, metric_s, netmask_s, mtu_s, window_s, irtt_s);
    
    /* The numbers in /proc/net/route are already in network byte
       order - no need to convert them. */

    if (dest_i == 0) {
      /* Default route */
      /*      fprintf(stderr,"seeming dest: %u / %u\n",dest_i,gw_i); */
      gw->sin_addr.s_addr = gw_i;
      return 0;
    }
  }
  return -1;
}


/*
 * Function create_symlinks_recursive (directory, root)
 *
 *    creates the symlinks, traversing from root/directory.  If it
 *    finds a directory, it calls itself.  If it finds a file it links
 *    it with root stripped off.  It skips filenames beginning with a
 *    dot, and files ending with '~'
 * 
 */

void create_symlinks_recursive(char* directory,char* root) {
  DIR *d;
  struct dirent* dir;
  struct stat statbuf, sb2;
  int ret;
  char dirname[MAXPATHLEN];
  if (switches.verbose >= 3) 
    fprintf(stderr,"create_symlinks_recursive(%s,%s)\n",directory,root);
  sprintf(dirname,"%s/%s",root,directory);
  ret = chdir(dirname);
  if (ret == -1) {
    fprintf(stderr,"%s\n", dirname);
    perror("Can't chdir");
    return;
  }
  if (strcmp(directory,".") == 0) {
    directory[0] = '\0';
  }
  
  d = opendir(".");
  if (d == NULL) {
    perror("opendir failed");
    exit(1);
  }
  while ((dir = readdir(d))) {

    /* Skip files beginning with '.' and ending with '~' */

    if (dir->d_name[0] == '.') {
      continue;
    }
    if (dir->d_name[strlen(dir->d_name) - 1] == '~') {
      continue;
    }
    ret = stat(dir->d_name,&statbuf);
    if (ret == -1) {
      perror("stat failed");
      exit(1);
    }

    if (S_ISDIR(statbuf.st_mode)) {
      char rel_dname[MAXPATHLEN];
      snprintf(rel_dname,MAXPATHLEN ,"%s/%s",directory,dir->d_name);
      create_symlinks_recursive(rel_dname,root);
    } else {
      char linkfrom[MAXPATHLEN], linkto[MAXPATHLEN];
      sprintf(linkfrom,"%s%s/%s",root,directory,dir->d_name);
      sprintf(linkto,"%s/%s",directory,dir->d_name);
      if (switches.verbose >= 3) 
	fprintf(stderr,"linking %s -> %s\n",linkfrom, linkto);

      errno = 0;
      ret = lstat(linkto,&sb2);
      if (ret == -1 && errno != ENOENT) {
	perror("lstat failed");
	exit(1);
      }
      if (errno != ENOENT && !S_ISLNK(sb2.st_mode) && switches.force == 0) {
	/* This isn't a symlink - print a warning and skip it */
	if (switches.verbose >= 0) 
	  fprintf(stderr,"%s is not a symlink - not removing.\n", linkto);
      } else {
	char buf[25];
	/* 
	   Check that the old symlink points to somewhere in SYSCONFDIR
	   so we don't destroy a thing like /etc/localtime
	*/
	ret = readlink(linkto,buf,sizeof(buf));
	if (ret == -1 && errno != ENOENT) {
	  perror("readlink failed");
	  exit(1);
	} else if (errno == ENOENT) {
	  ;
	} else {
	  buf[24] = '\0'; /* Just in case */
	  if (strncmp(buf,SYSCONFDIR,strlen(SYSCONFDIR)) != 0) {
	    if (switches.verbose >= 0) {
	      fprintf(stderr,"%s points outside of " SYSCONFDIR ", not removing.\n", linkto);
	      continue;
	    }
	  }
	}
	ret = unlink(linkto);
	if (ret == -1 && errno != ENOENT) {
	  /* Silently ignore file not found errors */
	  perror("unlink failed");
	  exit(1);
	}
	ret = symlink(linkfrom,linkto);
	if (ret == -1) {
	  perror("symlink failed");
	  exit(1);
	}
      }
    }
  }
  chdir("..");
}


/*
 * Function setup_location (location)
 *
 *    Does all the dirty work for configuring a specific location -
 *    calls configure_iface etc).
 *
 */
void setup_location(struct entry *location) {
  int i;
  char *a,*b;
  
  if (switches.dry_run == 1 && location) {
    printf("%s\n",location->name);
    exit(0);
  }

  if (configure_iface(location,switches.device) != 0) {
    exit(2);
  }

  /* interface now configured - let's symlink some files */
  a = malloc(MAXPATHLEN);
  b = strdup(".");
  snprintf(a,MAXPATHLEN,"/etc/intuitively/%s",location->dirroot);
	
  create_symlinks_recursive(b,a);

  free (a);
  free (b);

  if (switches.locationfile != NULL) {
    FILE* lfile = fopen(switches.locationfile,"w");
    if (lfile == NULL) {
      perror("Can't open locationfile");
    }
    if (location->description != NULL)
      i = fwrite(location->description,strlen(location->description),1,lfile);
    else 
      i = fwrite(location->dirroot,strlen(location->dirroot),1,lfile);
    if (i == -1) {
      perror("Writing to locationfile");
    }
    i = fwrite("\n",1,1,lfile);
    if (i == -1) {
      perror("Writing to locationfile");
    }

    fclose(lfile);

  }

  if (switches.run_scripts != 0 && location->script) {
    execlp(location->script,location->script,location->dirroot,0);
    perror("execlp failed for script");
    _exit(127);
  } else {
    exit(0);
  }
}


/*
 * Function net_aton (cp, sa)
 *
 *    Calls inet_aton if applicable, else it tries to find out which
 *    protocol this is and convert if applicable.
 *
 */

int net_aton(const char *cp, struct sockaddr_storage *sa) {
#ifdef IPV6
  struct addrinfo hints;
  int error;
  struct addrinfo* res;
#endif
  if (switches.verbose >= 3) 
    fprintf(stderr,"net_aton:: %s\n",cp);
  switch (sa->ss_family) {
  case AF_INET:
  /*
   * inet_aton() case.
   * This cannot handle IPv6 addresses.  Also, it cannot return
   * multiple addresses.
   */
    if (!inet_aton(cp, &((struct sockaddr_in *)sa)->sin_addr)) {
      perror("inet_aton");
      return 0;
    } else {
      return 1;
    }
  break;

#ifdef IPV6
  case AF_INET6:
  
    /* getaddrinfo() case.  It can handle multiple addresses. */
    memset(&hints, 0, sizeof(hints));
    /* set-up hints structure */
    hints.ai_family = PF_UNSPEC;
    error = getaddrinfo(cp, NULL, &hints, &res);
    fprintf(stderr,"family: %d\n",res->ai_addr->sa_family);
    if (error)
      perror(gai_strerror(error));
    return 0;
    break;
#endif
  default:
    fprintf(stderr,"net_aton::unknown protocol: %d\n", sa->ss_family);
    break;
  }
  return 1;
}

char* net_ntoa(struct sockaddr_storage ss) {
  char *name;

#ifdef IPV6  
  int error;
  char namebuf[BUFSIZ];
#endif
  switch (ss.ss_family) {
  case 0:
    return NULL; /* This is not really a valid struct sockaddr */ 
    break;
  case AF_INET:
    /*
     * inet_ntoa() case. This cannot handle IPv6 addresses.
     * No way to pass the error status.
     */
    name = inet_ntoa(((struct sockaddr_in *)&ss)->sin_addr);
    /*
      fprintf(stderr,"net_ntoa(%u = %s)\n", 
	    ((struct sockaddr_in *)&ss)->sin_addr.s_addr, name);

    fprintf(stderr,"net_ntoa::return %s\n", name);
    */
    return name;
    break;
#ifdef IPV6
  case AF_INET6:
    /* getnameinfo() case. NI_NUMERICHOST avoids DNS lookup. */
    error = getnameinfo((struct sockaddr *)&ss, ss.ss_align,
			namebuf, sizeof(namebuf), NULL, 0, NI_NUMERICHOST);
    if (error)
      perror("getnameinfo");
    name = namebuf;
    return name;
    break;
#endif
  default:
    fprintf(stderr,"net_ntoa::unknown protocol: %d\n", ss.ss_family);
    break;
  }
  return NULL;
}

char* ether_to_ascii(unsigned char *bin) {
  char *ret = malloc(19); /* Max size of the ASCII representation of
			     an ethernet address. */
  snprintf(ret,18,"%.2x:%.2x:%.2x:%.2x:%.2x:%.2x", bin[0], bin[1], bin[2],
	   bin[3], bin[4], bin[5]);
  return ret;
  
}
