/*  Inti-GL: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gdk/gl/config.h
//! @brief A GdkGLConfig C++ wrapper interface.
//!
//! Provides Config, an OpenGL FrameBuffer configuration object.

#ifndef INTI_GDK_GL_CONFIG_H
#define INTI_GDK_GL_CONFIG_H

#ifndef INTI_G_OBJECT_H
#include <inti/glib/object.h>
#endif

#ifndef __GDK_GL_CONFIG_H__
#include <gdk/gdkglconfig.h>
#endif

namespace Inti {

namespace Gdk {

class Colormap;
class Screen;
class Visual;

namespace GL {

class ConfigClass;

enum ConfigMode
{
	MODE_RGB = GDK_GL_MODE_RGB, //!< RGB mode (default).
	MODE_RGBA = GDK_GL_MODE_RGBA, //!< same as RGB
	MODE_INDEX = GDK_GL_MODE_INDEX, //!< Index buffer.
	MODE_SINGLE = GDK_GL_MODE_SINGLE, //!< Single buffer (default).
	MODE_DOUBLE = GDK_GL_MODE_DOUBLE, //!< Double buffered.
	MODE_STEREO = GDK_GL_MODE_STEREO, //!< Stereo visual.
	MODE_ALPHA = GDK_GL_MODE_ALPHA, //!< Color buffer has alpha bits.
	MODE_DEPTH = GDK_GL_MODE_DEPTH, //!< Depth buffer.
	MODE_STENCIL = GDK_GL_MODE_STENCIL, //!< Stencil buffer.
	MODE_ACCUM = GDK_GL_MODE_ACCUM, //!< Accumulation buffer.
	MODE_MULTISAMPLE = GDK_GL_MODE_MULTISAMPLE //!< not supported yet
};

typedef unsigned int ConfigModeField;

//! @class Config config.h inti/gdk/gl/config.h
//! @brief A GdkGLConfig C++ wrapper class.

class Config : public G::Object
{
	friend class G::Object;
	friend class ConfigClass;

	Config(const Config&);
	Config& operator=(const Config&);

protected:
//! @name Constructors
//! @{

	explicit Config(GdkGLConfig *config, bool reference = true);
	//!< Construct a new Config from an existing GdkGLConfig.
	//!< @param config A pointer to a GdkGLConfig.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>config</EM> can be a newly created GdkGLConfig or an existing
	//!< GdkGLConfig. (see G::Object::Object).

//! @}

public:
//! @name Constructors
//! @{

	virtual ~Config();
	//!< Destructor.

//! @}

	static Pointer<Config> create(const int *attrib_list, const Gdk::Screen *screen = 0);
	//!< Create an OpenGL frame buffer configuration that matchs the specified attributes.
	//!< @param attrib_list An int array that contains the attribute/value pairs.
	//!< @param screen The target screen, or null for the default screen.
	//!< @return A smart pointer to a new Config object, or null if an error occurs.
	//!<
	//!< <BR>Available attributes are: GDK_GL_USE_GL, GDK_GL_BUFFER_SIZE, GDK_GL_LEVEL,
	//!< GDK_GL_RGBA, GDK_GL_DOUBLEBUFFER, GDK_GL_STEREO, GDK_GL_AUX_BUFFERS,
	//!< GDK_GL_RED_SIZE, GDK_GL_GREEN_SIZE, GDK_GL_BLUE_SIZE, GDK_GL_ALPHA_SIZE,
	//!< GDK_GL_DEPTH_SIZE, GDK_GL_STENCIL_SIZE, GDK_GL_ACCUM_RED_SIZE,
	//!< GDK_GL_ACCUM_GREEN_SIZE, GDK_GL_ACCUM_BLUE_SIZE, GDK_GL_ACCUM_ALPHA_SIZE. The
	//!< last attribute in <EM>attrib_list</EM> must be GDK_GL_ATTRIB_LIST_NONE.

	static Pointer<Config> create(ConfigModeField mode, const Gdk::Screen *screen = 0);
	//!< Create an OpenGL frame buffer configuration that matches the specified display mode.
	//!< @param mode The display mode bit mask.
	//!< @param screen The target screen, or null for the default screen.
	//!< @return A smart pointer to a new Config object, or null if an error occurs.

//! @name Accessors
//! @{

	GdkGLConfig* gdk_gl_config() const { return (GdkGLConfig*)instance; }
	//!< Get a pointer to the GdkGLConfig structure.

	GdkGLConfigClass* gdk_gl_config_class() const;
	//!< Get a pointer to the GdkGLConfigClass structure.
	
	operator GdkGLConfig* () const;
	//!< Conversion operator; safely converts a Config to a GdkGLConfig pointer.

	Gdk::Screen* get_screen() const;
	//!< Returns the Gdk::Screen.

	bool get_attrib(int attribute, int& value);
	//!< Gets information about an OpenGL frame buffer configuration.
	//!< @param attribute The attribute to be returned.
	//!< @param value A reference to the return location for the requested value.
	//!< @return <EM>true</EM> if successful, <EM>false</EM> otherwise.

	Gdk::Colormap* get_colormap() const;
	//!< Gets the Gdk::Colormap that is appropriate for the OpenGL frame buffer configuration.
	//!< @returns The appropriate Gdk::Colormap.

	Gdk::Visual* get_visual() const;
	//!< Gets the Gdk::Visual that is appropriate for the OpenGL frame buffer configuration.
	//!< @return The appropriate Gdk::Visual.

	int get_depth() const;
	//!< Gets the color depth of the OpenGL-capable visual.
	//!< @return The number of bits per pixel.

	int get_layer_plane() const;
	//!< Gets the layer plane (level) of the frame buffer.
	//!< @return The layer plane.
	//!<
	//!< <BR>Zero is the default frame buffer. Positive layer planes correspond to frame buffers
	//!< that overlay the default buffer, and negative layer planes correspond to frame buffers
	//!< that underlie the default frame buffer.

	int get_n_aux_buffers() const;
	//!< Gets the number of auxiliary color buffers.
	//!< @return The number of auxiliary color buffers.

	int get_n_sample_buffers() const;
	//!< Gets the number of multisample buffers.
	//!< @return The number of multisample buffers.

	bool is_rgba() const;
	//!< Determines whether the configured frame buffer is RGBA mode.
	//!< @return <EM>true</EM> if the configured frame buffer is RGBA mode, <EM>false</EM> otherwise.

	bool is_double_buffered() const;
	//!< Determines whether the configuration supports the double-buffered visual.
	//!< @return <EM>true</EM> if the double-buffered visual is supported, <EM>false</EM> otherwise.

	bool is_stereo() const;
	//!< Determines whether the configuration supports the stereo visual.
	//!< @return <EM>true</EM> if the stereo visual is supported, <EM>false</EM> otherwise.

	bool has_alpha() const;
	//!< Determines whether the configured color buffer has alpha bits.
	//!< @return <EM>true</EM> if the color buffer has alpha bits, <EM>false</EM> otherwise.

	bool has_depth_buffer() const;
	//!< Determines whether the configured frame buffer has depth buffer.
	//!< @return <EM>true</EM> if the frame buffer has depth buffer, <EM>false</EM> otherwise.

	bool has_stencil_buffer() const;
	//!< Determines whether the configured frame buffer has stencil buffer.
	//!< @return <EM>true</EM> if the frame buffer has stencil buffer, <EM>false</EM> otherwise.

	bool has_accum_buffer() const;
	//!< Determines whether the configured frame buffer has accumulation buffer.
	//!< @return <EM>true</EM> if the frame buffer has accumulation buffer, <EM>false</EM> otherwise.

//! @}
};

} // namespace GL

} // namespace Gdk

} // namespace Inti

#endif // INTI_GDK_GL_CONFIG_H

