/***************************************************************************
                         meshmodifier.cpp  -  description
                            -------------------
   begin                : Sat Apr 21 2001
   copyright            : (C) 2001 by Jon Anderson
   email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <cassert>
#include "meshmodifier.h"
#include <Entities/mesh.h>

#include <i3d.h>
#include <i3dworkspace.h>
#include "modetoolbar.h"

#include "pmeshdlg.h"

#include "uvmodifier.h"
#include <TextureMap/planarwrap.h>
#include <TextureMap/cubewrap.h>
#include <TextureMap/cylinderwrap.h>
#include <TextureMap/sphericalwrap.h>
#include <Views/viewUV.h>
#include <Controls/mateditor.h>
#include "primitivecreator.h"

/*
 * These are macros for operating on every mesh in a selection
 * in an undoable manner. They are used in the following manner:
 *
 * BEGIN( "Mirroring Objects..." );
 * MESH() -> mirror( x );
 * END();
 *
 * There is probably a neat nice way to do this with function objects
 * but I haven't worked it out yet.
 */

#define BEGIN(c,n) \
  if(assertSelected(n)){ \
    setStatus( c ); \
    TransactionCommand *tc = new TransactionCommand(); \
    SelectionIterator it = m_selection -> begin(); \
    while( it != m_selection -> end() ) {\
      CheckPointCmd *pc = new CheckPointCmd( static_cast<Entity *> (*it ) );

#define END() \
      tc -> addCommand( pc ); \
      ++it; \
    } \
    tc -> save(); \
    setStatusDone(); \
  } \
  updateViews();

#define MESH() static_cast<Mesh *>( *it )

int MeshModifier::TYPE = IControl::getUID();

MeshModifier::MeshModifier() : ObjectModifier( "Meshes", Mesh::TYPE )
{
   setName( "&Meshes" );

   QPopupMenu *mmenu = new QPopupMenu();
   mmenu -> insertItem( "X Axis", this, SLOT( slotMirrorX() ) );
   mmenu -> insertItem( "Y Axis", this, SLOT( slotMirrorY() ) );
   mmenu -> insertItem( "Z Axis", this, SLOT( slotMirrorZ() ) );

   QPopupMenu *pmenu = new QPopupMenu();
   pmenu -> insertItem( "Points", this, SLOT( slotAddPoints() ) );
	 pmenu -> insertItem( "Triangles", this, SLOT( slotAddTris() ) );
	 pmenu -> insertItem( "Quads", this, SLOT( slotAddQuads() ) );
	 pmenu -> insertItem( "TriStrip", this, SLOT( slotAddTriStrip() ) );
	 pmenu -> insertItem( "QuadStrip", this, SLOT( slotAddQuadStrip() ) );


   m_popup -> insertSeparator();
   m_popup -> insertItem( "Merge", this, SLOT( slotMerge() ) );
   m_popup -> insertItem( "Explode", this, SLOT( slotExplode() ) );
   m_popup -> insertSeparator();
	 m_popup -> insertItem( "Add...", pmenu );
   m_popup -> insertSeparator();
   m_popup -> insertItem( "Triangulate", this, SLOT( slotTriangulate() ) );
   m_popup -> insertItem( "Subdivide", this, SLOT( slotSubdivide() ) );
   m_popup -> insertItem( "Smooth", this, SLOT( slotSmooth() ) );
   m_popup -> insertItem( "Tighten", this, SLOT( slotTighten() ) );
   m_popup -> insertItem( "Reverse", this, SLOT( slotReverse() ) );
   m_popup -> insertItem( "Recenter", this, SLOT( slotCenter() ) );
   m_popup -> insertItem( "Mirror", mmenu );
   m_popup->insertSeparator();

   QPopupMenu *map = new QPopupMenu();
   QPopupMenu *pmap = new QPopupMenu();
   pmap -> insertItem( "Top", this, SLOT( slotXZPlaneGizmo() ) );
	 pmap -> insertItem( "Side", this, SLOT( slotYZPlaneGizmo() ) );
	 pmap -> insertItem( "Front", this, SLOT( slotXYPlaneGizmo() ) );
	
   map -> insertItem( "Plane Gizmo", pmap ); //this, SLOT( slotPlaneGizmo() ) );
   map -> insertItem( "Cylinder Gizmo", this, SLOT( slotCylinderGizmo() ) );
   map -> insertItem( "Sphere Gizmo", this, SLOT( slotSphereGizmo() ) );
   map -> insertSeparator();
   map -> insertItem( "Edit UVs", this, SLOT( slotEditUVs() ) );
   map -> insertItem( "Apply material", this, SLOT( slotMatEditor() ) );
   m_popup -> insertItem( "Texture Mapping...", map );

   m_popup->insertSeparator();


   menu_edge = m_popup -> insertItem( "Draw Edges", this, SLOT( slotToggleEdges() ) );
   menu_face = m_popup -> insertItem( "Cull Faces", this, SLOT( slotToggleCullFaces() ) );
	 menu_ticks = m_popup -> insertItem( "Draw Vertex Ticks", this, SLOT( slotToggleVertexTicks() ) );
   m_popup->insertSeparator();

   m_popup -> insertItem( "Reset position", this, SLOT( slotResetPosition() ) );
   m_popup -> insertItem( "Reset Orientation", this, SLOT( slotResetOrientation() ) );
   m_popup -> insertSeparator();

   m_popup -> insertItem( "Properties", this, SLOT( slotProperties() ) );

   m_popup -> setItemChecked( menu_edge, Mesh::draw_edges );
   m_popup -> setItemChecked( menu_face, Mesh::cull_faces );
   m_popup -> setItemChecked( menu_ticks, Mesh::vertex_ticks );


}

MeshModifier::~MeshModifier()
{
}


void MeshModifier::activate()
{
   clearSelection();
   I3D::getWorkspace() -> setCurrentControl( this );
   I3D::setModeMsg( " Mesh Mode " );
   SelectMode::set( Mesh::TYPE );
   //change to select mode immediately.
   ModeToolbar::setMode( ModeToolbar::MODE_SELECT );

   updateViews();
}

void MeshModifier::deactivate()
{
}

void MeshModifier::slotMirrorX()
{

   BEGIN( "Mirroring on X axis...", 1 );
   MESH() -> mirror( 0 );
   END();

}

void MeshModifier::slotMirrorY()
{

   BEGIN( "Mirroring on Y axis...", 1 );
   MESH() -> mirror( 1 );
   END();

}

void MeshModifier::slotMirrorZ()
{

   BEGIN( "Mirroring on Z axis...", 1 );
   MESH() -> mirror( 2 );
   END();

}

void MeshModifier::slotMerge()
{
   if ( assertSelected( 2 ) )
   {
      setStatus( "Merging meshes..." );

      Mesh *m;
      Mesh *tm;
      TransactionCommand *tc = new TransactionCommand();

      SelectionIterator it = m_selection -> begin();
      m = static_cast<Mesh *>( *it );
      ++it;

      CheckPointCmd *c = new CheckPointCmd( m );
      tc -> addCommand( c );

      vector<Object *> objs;
      objs.reserve( m_selection -> size() );

      while ( it != m_selection -> end() )
      {

         tm = static_cast<Mesh *>( *it );
         objs.push_back( tm );
         Command *cc = new DBCommand( tm, REMOVE );
         cc->execute();
         tc->addCommand( c );
         ++it;

      }

      m -> merge( objs );

      tc->save();
      setStatusDone();

   }

   updateViews();

}

void MeshModifier::slotExplode()
{

   if ( assertSelected( 1 ) )
   {
      setStatus( "Exploding meshes..." );

      Mesh *m;
      TransactionCommand *tc = new TransactionCommand();

      SelectionIterator it = m_selection -> begin();


      vector<Mesh *> meshes;

      /** Capture all the exploded objects
        * into the meshes vector.
        */

      while ( it != m_selection -> end() )
      {
         m = static_cast<Mesh *>( *it );
         Command *c = new DBCommand( m, REMOVE );
         c -> execute();
         tc -> addCommand( c );

         m -> explode( meshes );
         ++it;

      }

      /** Insert all the new meshes into
        * the ObjectDB
        */
      for ( int i = 0; i < ( int ) meshes.size(); i++ )
      {
         Command *cc = new DBCommand( meshes[ i ], ADD );
         cc -> execute();
         tc -> addCommand( cc );
      }

      tc->save();
      setStatusDone();

   }

   updateViews();

}

void MeshModifier::slotTriangulate()
{

   BEGIN( "Triangulating mesh(s)...", 1 );
   MESH() -> triangulate( );
   END();

}

void MeshModifier::slotSubdivide()
{

   BEGIN( "Subdividing mesh(s)...", 1 );
   MESH() -> subdivide( );
   END();

}

void MeshModifier::slotSmooth()
{

   BEGIN( "Smoothing mesh(s)...", 1 );
   MESH() -> smooth( 1 );
   END();

}

void MeshModifier::slotTighten()
{

   BEGIN( "Tightening mesh(s)...", 1 );
   MESH() -> tighten( 1 );
   END();

}

void MeshModifier::slotReverse()
{

   BEGIN( "Reversing mesh(s)...", 1 );
   MESH() -> reverse( );
   END();

}

void MeshModifier::slotCenter()
{
   BEGIN( "Recentering mesh(s)...", 1 );
   MESH() -> center();
   END();
}

void MeshModifier::slotResetPosition()
{
   if ( assertSelected( 1 ) )
   {
      Vector4 p;
      setStatus( "Reseting Position..." );
      TransactionCommand *tc = new TransactionCommand();
      SelectionIterator it = m_selection -> begin();

      while ( it != m_selection -> end() )
      {

         Mesh * m = static_cast<Mesh *>( *it );
         TransformCmd *pc = new TransformCmd( m );
         pc -> begin();
         m -> setPosition( p );
         pc -> end();
         tc -> addCommand( pc );
         ++it;

      }

      tc -> save();
      setStatusDone();

   }

   updateViews();

}

void MeshModifier::slotResetOrientation()
{
   if ( assertSelected( 1 ) )
   {
      Quat q;
      setStatus( "Reseting Orientation..." );
      TransactionCommand *tc = new TransactionCommand();
      SelectionIterator it = m_selection -> begin();

      while ( it != m_selection -> end() )
      {

         Mesh * m = static_cast<Mesh *>( *it );
         TransformCmd *pc = new TransformCmd( m );
         pc -> begin();
         m -> setOrientation( q );
         pc -> end();
         tc -> addCommand( pc );
         ++it;

      }

      tc -> save();
      setStatusDone();

   }

   updateViews();

}

void MeshModifier::slotToggleEdges()
{
   Mesh::draw_edges = ! Mesh::draw_edges;
   getPopupMenu() -> setItemChecked( menu_edge, Mesh::draw_edges );
}

void MeshModifier::slotToggleCullFaces()
{
   Mesh::cull_faces = ! Mesh::cull_faces;
   getPopupMenu() -> setItemChecked( menu_face, Mesh::cull_faces );
}

void MeshModifier::slotToggleVertexTicks()
{
   Mesh::vertex_ticks = ! Mesh::vertex_ticks;
   getPopupMenu() -> setItemChecked( menu_ticks, Mesh::vertex_ticks );
}

void MeshModifier::slotProperties()
{
   setStatus( "Editing properties..." );

   /* Popup custom dialog here.
    */
   PMeshDlg *dlg = new PMeshDlg( 0, "Mesh Properties", true );

   SelectionIterator it = m_selection -> begin();

   if ( m_selection -> size() != 1 )
   {
      dlg -> setEnableAll( false );

      int f = 0;
      int v = 0;

      while ( it != m_selection -> end() )
      {

         Mesh * m = static_cast<Mesh *>( *it );
         f += m -> numFaces();
         v += m -> numVerts();
         ++it;

      }

      dlg -> setFaces( f );
      dlg -> setVertices( v );

   }

   else
   {

      dlg -> setEnableAll( true );
      dlg -> setMesh( static_cast<Mesh *>( *it ) );
   }

   if ( dlg -> exec() )
   {
      cerr << "Done with properties " << endl;
      Mesh *m = static_cast<Mesh *>( *it );
      CheckPointCmd *pc = new CheckPointCmd( m );
      pc -> save();
      Vector4 pos = dlg -> getPosition();
      Quat ori = dlg -> getOrientation();

      m -> setName( dlg -> getName() );
      m -> setNotes( dlg -> getNotes() );
      m -> setPosition( pos );
      m -> setOrientation( ori );

      updateViews();
      setStatusDone();
   }

   else
   {
      updateViews();
      setStatusDone();
   }
}


///////////////////////////////////////////
void MeshModifier::slotYZPlaneGizmo()
{
   m_using_gizmo = true;

	if ( m_gizmo != 0 )
   {
			delete m_gizmo;
			m_gizmo = 0;
   }

   m_gizmo = new PlanarWrap();

   m_gizmo -> rotate( 90, 1, 0, 0, 0, 0, 0 );
   m_gizmo -> rotate( -90, 0, 1, 0, 0, 0, 0 );
}

void MeshModifier::slotXZPlaneGizmo()
{
   m_using_gizmo = true;

   if ( m_gizmo != 0 )
   {
			delete m_gizmo;
			m_gizmo = 0;
   }

   m_gizmo = new PlanarWrap();
}

void MeshModifier::slotXYPlaneGizmo()
{
   m_using_gizmo = true;

	if ( m_gizmo != 0 )
   {
			delete m_gizmo;
			m_gizmo = 0;
   }

   m_gizmo = new PlanarWrap();
   m_gizmo -> rotate( 90, 1, 0, 0, 0, 0, 0 );
}


void MeshModifier::slotCylinderGizmo()
{
   m_using_gizmo = true;

		if ( m_gizmo != 0 )
   {
			delete m_gizmo;
			m_gizmo = 0;
   }

   m_gizmo = new CylinderWrap();

}

void MeshModifier::slotSphereGizmo()
{
   m_using_gizmo = true;

	if ( m_gizmo != 0 )
   {
			delete m_gizmo;
			m_gizmo = 0;
   }


   m_gizmo = new SphericalWrap();

}

void MeshModifier::slotCubeGizmo()
{
   m_using_gizmo = true;

   if ( m_gizmo != 0 )
   {
			delete m_gizmo;
			m_gizmo = 0;
   }

   m_gizmo = new CubeWrap();
}

void MeshModifier::slotEditUVs()
{
   if ( assertSelected( 1 ) )
   {

      Mesh * m = static_cast< Mesh *> ( getFirst() );

      vector<int> flist;

      int n = m -> numFaces();

      for ( int i = 0; i < n; i++ )
      {
         flist.push_back( i );
      }

      ViewUV *v = static_cast<ViewUV *>( I3D::getWorkspace() -> addView( VUV, true ) );

      v -> setMesh( m, flist );
      v -> setControl( new UVModifier() );



   }

   updateViews();
}

void MeshModifier::draw()
{
   if ( m_using_gizmo &&
         ( m_gizmo != 0 ) )
   {
      m_gizmo -> draw();
   }
}

void MeshModifier::transform ( Vector4 &diff, Vector4 &center )
{

   if ( ! m_using_gizmo )
   {
      Modifier::transform( diff, center );
      return ;
   }

   int mode = ModeToolbar::getMode();

   switch ( mode )
   {
      case ModeToolbar::MODE_ROTATE:
         m_gizmo -> rotate( diff, m_selection_center );
         break;
      case ModeToolbar::MODE_MOVE:
         m_gizmo -> move( diff );
         break;
      default:
         static_cast<Entity *>( m_gizmo ) -> scale( diff, m_selection_center );
         break;
   }

}

void MeshModifier::startTransform()
{
   //if we aren'nt using a gizmo, let the parent class handle things
   if ( !m_using_gizmo )
   {
      ObjectModifier::startTransform();
      return ;
   }

   //otherwise, modify the gizmo...
	cerr << "Starting gizmo transform"<<endl;
   m_cmds.clear();

   CheckPointCmd *c = new CheckPointCmd( m_gizmo );

   c -> save();

}


void MeshModifier::endTransform()
{

   //if we aren'nt using a gizmo, let the parent class handle things
   if ( !m_using_gizmo )
   {
      ObjectModifier::endTransform();
      return ;
   }
	cerr << "Ending gizmo transform"<<endl;
   //otherwise, we can just ignore the end cuz it's a
   //checkpnt.

}

void MeshModifier::keyEvent( QKeyEvent *e )
{

   switch ( e->key() )
   {
      case Key_Escape:
         slotQuitGizmo();
         break;
      case Key_Space:
         slotApplyGizmo();
         break;
      default:
         ObjectModifier::keyEvent( e );
         break;
   }

}

void MeshModifier::slotApplyGizmo()
{
   if ( m_using_gizmo &&
         ( m_gizmo != 0 ) &&
         assertSelected( 1 ) )
   {
      setStatus( "Applying UV wrapper..." );
      Mesh *m = static_cast< Mesh * >( getFirst() );
      CheckPointCmd *chc = new CheckPointCmd( m );

      //create a vector of the faces...
      vector<SubObject *> faces;

      faces = * ( m -> getFaces() ) ;

      m_gizmo -> createMapping( & faces );

      chc -> save();
      setStatusDone();
      updateViews();

   }

}

void MeshModifier::slotQuitGizmo()
{

   m_using_gizmo = false;

	 if ( m_gizmo != 0 )
   {
			delete m_gizmo;
			m_gizmo = 0;
   }

   updateViews();

}

void MeshModifier::slotMatEditor()
{
   I3D::getMatEditor() -> show();
}
void MeshModifier::addPrimitive( int i )
{
	if( assertSelectedExactly(1) )
	{
		Mesh * m = static_cast<Mesh *>( getFirst() );
		assert( m );
		PrimitiveCreator *pc = dynamic_cast<PrimitiveCreator*>( I3D::activateControl( PrimitiveCreator::TYPE ) );
		assert( pc );
		assert( m );
		pc -> addToMesh( i, m );
	}
}
void MeshModifier::slotAddPoints()
{
	addPrimitive( 0 );
}
void MeshModifier::slotAddTris()
{
	addPrimitive( 1 );
}
void MeshModifier::slotAddQuads()
{
	addPrimitive( 2 );
}
void MeshModifier::slotAddTriStrip()
{
	addPrimitive( 3 );
}
void MeshModifier::slotAddQuadStrip()
{
	addPrimitive( 4 );
}




