/***************************************************************************
                         meshcreator.cpp  -  description
                            -------------------
   begin                : Wed Apr 18 2001
   copyright            : (C) 2001 by Jon Anderson
   email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "meshcreator.h"
#include <Entities/cube.h>
#include <Entities/cylinder.h>
#include <Entities/heightmap.h>
#include <Entities/sphere.h>
#include <Entities/plane.h>
#include <i3d.h>
#include <i3dworkspace.h>
#include <qfiledialog.h>
#include <qimage.h>
#include <config.h>
#include "cmeshdlg.h"
#include "meshmodifier.h"

int MeshCreator::TYPE = IControl::getUID();

MeshCreator::MeshCreator() : Creator()
{
   setName( "&Mesh..." );

   QPopupMenu *plane = new QPopupMenu();

   //plane -> insertItem( "XY", this, SLOT( slotCreateXYPlane() ) );
   plane -> insertItem( "X&Z", this, SLOT( slotCreateXZPlane() ) );
   // plane -> insertItem( "YZ", this, SLOT( slotCreateYZPlane() ) );


   m_popup -> insertItem( "&Cube", this, SLOT( slotCreateCube() ) );
   m_popup -> insertItem( "&Sphere", this, SLOT( slotCreateSphere() ) );
   m_popup -> insertItem( "C&ylinder", this, SLOT( slotCreateCylinder() ) );
   m_popup -> insertItem( "&Plane", plane );
   m_popup -> insertItem( "&Height Map", this, SLOT( slotCreateHMap() ) );
   m_popup -> insertItem( "C&ustom Mesh", this, SLOT( slotCreateCustom() ) );


   m_mesh = 0;

}

MeshCreator::~MeshCreator()
{
}

void MeshCreator::mousePress( Vector4 &plane, Vector4 &pt, int i )
{
   if ( m_mesh != 0 )
   {
      cerr << "Inserting mesh" << endl;
      //insert the mesh and move to the given point.
      Command *c = new DBCommand( m_mesh, ADD );
      c->execute();
      c->save();

      m_mesh -> move( pt.x, pt.y, pt.z );

      //reset
      m_mesh = 0;
      setStatusDone();
      I3D::getWorkspace() -> revertControl();
      updateViews();
   }

   else
   {
      cerr << "No mesh to insert" << endl;
   }
}


void MeshCreator::keyEvent( QKeyEvent * c )
{
   //ignore any key presses for now.
   setStatusDone();
   I3D::getWorkspace() -> revertControl();
}

/** Set this creator to be active.
  */
void MeshCreator::activate()
{
   I3D::activateControl( MeshModifier::TYPE );
   I3D::setModeMsg( " Create Mode " );
   I3D::getWorkspace() -> setCurrentControl( this );
}

/** Clean up as control leaves this creator.
  */

void MeshCreator::deactivate()
{
   /* User skipped out before actually adding the mesh
    */

   if ( m_mesh != 0 )
      delete m_mesh;

}

/** Creates the default cube
  */
void MeshCreator::slotCreateCube()
{
   activate();
   m_mesh = new Cube( 0, 0, 0, 1, 1, 1 );
   setStatus( "Creating cube: Click to add Mesh." );

}

/** Creates the default cube
  */
void MeshCreator::slotCreateSphere()
{
   activate();
   m_mesh = new Sphere( 0, 0, 0, 1, 2 );
   setStatus( "Creating sphere: Click to add Mesh." );
}

/** Creates the default height map.
  */
void MeshCreator::slotCreateHMap()
{
   activate();
   /* popup a browse dialog here..
    */
   static QString directory = I3D::getConfig() -> getString( "Directories", "Textures" ).c_str();
   QString name = QFileDialog::getOpenFileName( directory, "Images (*.png *.xpm *.jpg *.rgb)" );

   directory = QString::null;

   QImage image( name );

   if ( image.isNull() )
   {
      QImage dummy( 256, 256, 32 );
      QColor c( 0, 0, 0 );
      dummy.fill( c.pixel() );
      image = dummy;
   }

   m_mesh = new HeightMap( image, 0, 0, 0, 25, 2, 25 );
   setStatus( "Creating heightmap: Click to add Mesh." );

}

/** Creates the default cylinder
  */
void MeshCreator::slotCreateCylinder()
{
   activate();
   m_mesh = new Cylinder( 0, 0, 0, 16, 2, 1 );
   setStatus( "Creating cylinder: Click to add Mesh." );
}

/** Creates the default plane
  */
void MeshCreator::slotCreateXYPlane()
{
   activate();
   m_mesh = new Plane( 0, 0, 0, 2, 2 );
   setStatus( "Creating XY Plane: Click to add Mesh." );

}

/** Creates the default plane
  */
void MeshCreator::slotCreateXZPlane()
{
   activate();
   m_mesh = new Plane( 0, 0, 0, 2, 2 );
   setStatus( "Creating XZ Plane: Click to add Mesh." );
}

/** Creates the default plane
  */
void MeshCreator::slotCreateYZPlane()
{
   activate();
   m_mesh = new Plane( 0, 0, 0, 2, 2 );
   setStatus( "Creating YZ Plane: Click to add Mesh." );
}


/** Creates a custom mesh.
  */
void MeshCreator::slotCreateCustom()
{
   activate();
   setStatus( "Creating a mesh: Click to add Mesh." );

   /* Popup custom dialog here.
    */
   CMeshDlg *dlg = new CMeshDlg( 0, "Create Mesh", true );

   if ( dlg -> exec() )
   {
      switch ( dlg -> getType() )
      {
         case CMeshDlg::CUBE:
            m_mesh = new Cube( 0, 0, 0, dlg->getA(), dlg->getB(), dlg->getC() );
            break;
         case CMeshDlg::SPHERE:
            m_mesh = new Sphere( 0, 0, 0, dlg->getA(), dlg->getB() );
            break;
         case CMeshDlg::HEIGHTMAP:
            {
               QImage image( dlg-> getFileName() );

               if ( image.isNull() )
               {
                  QImage dummy( 256, 256, 32 );
                  QColor c( 0, 0, 0 );
                  dummy.fill( c.pixel() );
                  image = dummy;
               }

               m_mesh = new HeightMap( image, 0, 0, 0, dlg->getA(), dlg->getB(), dlg->getC() );
            }

            break;
         case CMeshDlg::CYLINDER:
            m_mesh = new Cylinder( 0, 0, 0, dlg->getA(), dlg->getB(), dlg->getC() );
            break;
         case CMeshDlg::PLANE:
            m_mesh = new Plane( 0, 0, 0, dlg->getA(), dlg->getB() );
            break;
         default:
            m_mesh = new Cube( 0, 0, 0, 2, 2, 2 );
            break;
      }

      m_mesh -> setName( dlg -> getName().ascii() );
      m_mesh -> setNotes( dlg -> getNotes().ascii() );
      m_mesh -> setSize( dlg -> getD(), dlg -> getE(), dlg -> getF() );

   }

   else
   {
      I3D::getWorkspace() -> revertControl();
      updateViews();
      setStatusDone();
   }

   delete dlg;
}
