//
// (c) Polar Pyramid
// suggestions to stolken@kabelfoon.nl and/or stolk@xs4all.nl
//

#define EPSILON 0.00000000001


#include <assert.h>
#include <math.h>
#include <iostream>


#include "vector3.h"
#include "quat.h"
#include "matrix44.h"


using namespace std;

bool IsZero(float f)
{
  return (f < EPSILON && f > -EPSILON);
}


/////////////////////////////////////////////////////////////////////////////
//
// Default Constructor
//

Quat::Quat()
  : v(Vector3()), w(1)
{
}

/////////////////////////////////////////////////////////////////////////////
//
// Copy Constructor
//

Quat::Quat(const Quat& q)
  : v(q.v), w(q.w)
{
}

Quat::Quat(const Vector4 &q)
	: v(q.x, q.y, q.z), w(q.w)
{

}
/////////////////////////////////////////////////////////////////////////////
//
// This constructor is used in operators and sets the member variables
// directly
//

 Quat::Quat(float qw, const Vector3 &qv)
  : v(qv), w(qw)
{
}

/////////////////////////////////////////////////////////////////////////////
//
// This constructor is useful for creating rotations by specifying the
// angle of rotation and the axis around which you want to rotate,
// this axis does not have to be normalized
//

 Quat::Quat(const Vector3 &axis, float angle)
{
  if (!IsZero(angle))
  {
    w = cos(0.5f*angle);
    v = ((float) -sin(0.5f*angle))*axis;
  }
  else
  {
    w = 1.0;
    v = Vector3(0.0f, 0.0f, 0.0f);
  }
  Normalize();

}

/////////////////////////////////////////////////////////////////////////////
//
// 4 Element Constructor
//

 Quat::Quat(float qx, float qy, float qz, float qw)
  : v(qx, qy, qz), w(qw)
{
}

/////////////////////////////////////////////////////////////////////////////
//
// Array Access
//

 float& Quat::operator [](int i)
{
  switch (i)	
  {
  case 0: return v.x;
  case 1: return v.y;
  case 2: return v.z;
  case 3: return w;
  }
  assert(false);
  return w;
}

 const float& Quat::operator [](int i) const
{
  switch (i)
  {
  case 0: return v.x;
  case 1: return v.y;
  case 2: return v.z;
  case 3: return w;
  }
  assert(false);
  return w;
}

/////////////////////////////////////////////////////////////////////////////
//
// Real
//

 float Quat::Real(void) const
{
  return w;
}

/////////////////////////////////////////////////////////////////////////////
//
// Imaginary
//

 Vector3 Quat::Imaginary(void) const
{
  return v;
}

/////////////////////////////////////////////////////////////////////////////
//
// X
//

 float Quat::X(void) const
{
  return v.x;
}

/////////////////////////////////////////////////////////////////////////////
//
// Y
//

 float Quat::Y(void) const
{
  return v.y;
}

/////////////////////////////////////////////////////////////////////////////
//
// Z
//

 float Quat::Z(void) const
{
  return v.z;
}

/////////////////////////////////////////////////////////////////////////////
//
// Abs
//

 float Quat::Abs(void) const
{
  return sqrt(Magnitude());
}

/////////////////////////////////////////////////////////////////////////////
//
// Conj
//

 void Quat::Conj(void)
{
  v = -v;
}

/////////////////////////////////////////////////////////////////////////////
//
// Inverse
//

 Quat Quat::Inverse(void) const
{
  Quat q = *this;
  q.Conj();
  q.Normalize();
  return q;
}

/////////////////////////////////////////////////////////////////////////////
//
// Normalize
//

 void Quat::Normalize(void)
{
  float c = Abs();
  assert(!IsZero(c));
  w /= c;
  v /= c;
}

/////////////////////////////////////////////////////////////////////////////
//
// magnitude
//

 float Quat::Magnitude(void) const
{
  return v.lengthSqr() + w*w;
}

/////////////////////////////////////////////////////////////////////////////
// Math Operators
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// -Quat
//

 Quat Quat::operator - () const
{
  return Quat(-w, -v);
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat += Quat
//

 Quat& Quat::operator += (const Quat& q)
{
  w += q.w;
  v += q.v;
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat -= Quat
//

 Quat& Quat::operator -= (const Quat& q)
{
  w -= q.w;
  v -= q.v;
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat *= Quat
//

 Quat& Quat::operator *= (const Quat& q)
{
  Vector3 mul(w*q.X() + q.w*X() + Y()*q.Z() - Z()*q.Y(),
	      w*q.Y() + q.w*Y() + Z()*q.X() - X()*q.Z(),
	      w*q.Z() + q.w*Z() + X()*q.Y() - Y()*q.X());
  w = w*q.w   - X()*q.X() - Y()*q.Y() - Z()*q.Z();
  v = mul; // do not change the order of these lines! (w needs old v)
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat /= Quat
//

 Quat& Quat::operator /=(const Quat& q)
{
  *this *= q.Inverse();
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat ^= Quat
//

 Quat& Quat::operator ^=(const Quat& q)
{
  *this = Exp(q*Log(*this));
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat *= float
//

 Quat& Quat::operator *=(float c)
{
  w *= c;
  v *= c;
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat /= float
//

 Quat& Quat::operator /=(float c)
{
  assert(!IsZero(c));
  *this = *this/c;
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat * float
//

 Quat Quat::operator *(float c) const
{
  return Quat(c*w, c*v);
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat / float
//

 Quat Quat::operator /(float c) const
{
  assert(!IsZero(c));
  return Quat(w/c, v/c);
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat == Quat
//

 bool Quat::operator == (const Quat& q) const
{
  return ((w == q.w) && (v == q.v));
}

/////////////////////////////////////////////////////////////////////////////
//
// Quat != Quat
//

 bool Quat::operator != (const Quat& q) const
{
  return (!(*this == q));
}

/////////////////////////////////////////////////////////////////////////////
// Friend functions
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// float * Quat
//

 Quat operator *(float c, const Quat& q)
{
  return Quat(c*q.w, c*q.v);
}

/////////////////////////////////////////////////////////////////////////////
//
// Square(Quat)
//

 Quat Square(const Quat& q)
{
  return Quat(q.w*q.w - q.v.lengthSqr(), 2*q.w*q.v);
}

/////////////////////////////////////////////////////////////////////////////
//
// Exp(Quat)
//

 Quat Exp(const Quat &q1)
{
  Quat q = q1;

  float s = q.w;
  float se = exp(s);
  q.w = 0.0f;
  float scale = se;
  float theta;
  if ((theta = q.Abs()) > 0.0001f)
    scale *= sin(theta)/theta;
  q.v *= scale;
  q.w = se*cos(theta);
  return q;
}

/////////////////////////////////////////////////////////////////////////////
//
// Log(Quat)
//

 Quat Log(const Quat &q1)
{
  Quat q = q1;

  float sl = q.Abs();
  q.Normalize();
  float s = q.w;
  q.w = 0.0f;
  float scale = q.Abs();
  float theta = atan2(scale, s);
  if (scale > 0.0f)
    scale = theta/scale;
  q.v *= scale;
  q.w = log(sl);
  return q;
}

/////////////////////////////////////////////////////////////////////////////
//
// Interpolate(Quat, Quat, t)
//
// spherical interpolation between 2 quaternions
//

 Quat Interpolate(const Quat &p,
			const Quat &q,
			float t)
{
  Quat qt, q1;
  float sp, sq;

  if ((p - q).Abs() > (p + q).Abs())
    q1 = -q;
  else
    q1 = q;

  float c = (float) (p.w*q1.w +
		       p.v[0]*q1.v[0] +
		       p.v[1]*q1.v[1] +
		       p.v[2]*q1.v[2]);

  if (c > -0.9999999999)
  {
    if (c < 0.9999999999)
    {
      float o = acos(c);
      float s = 1.0/sin(o);
      sp = sin((1.0-t)*o)*s;
      sq = sin(t*o)*s;
    }
    else
    {
      sp = 1.0-t;
      sq = t;
    }
    qt.w   = sp*p.w   + sq*q1.w;
    qt.v[0] = sp*p.v[0] + sq*q1.v[0];
    qt.v[1] = sp*p.v[1] + sq*q1.v[1];
    qt.v[2] = sp*p.v[2] + sq*q1.v[2];
  }
  else
  {
    qt.w   =  p.v[2];
    qt.v[0] = -p.v[1];
    qt.v[1] =  p.v[0];
    qt.v[2] = -p.w;
    sp = sin((0.5-t)*M_PI);
    sq = sin(t*M_PI);
    qt.v[0] = sp*p.v[0] + sq*qt.v[0];
    qt.v[1] = sp*p.v[1] + sq*qt.v[1];
    qt.v[2] = sp*p.v[2] + sq*qt.v[2];
  }
  return qt;
}

//
// InsertInMatrix
//

 void Quat::InsertInMatrix(float m[4][4]) const
{
  float x2 = X() + X();
  float y2 = Y() + Y();
  float z2 = Z() + Z();
  float wx = w*x2;
  float wy = w*y2;
  float wz = w*z2;
  float xx = X()*x2;
  float xy = X()*y2;
  float xz = X()*z2;
  float yy = Y()*y2;
  float yz = Y()*z2;
  float zz = Z()*z2;
  m[0][0] = 1.0f-yy-zz;
  m[1][0] = xy+wz;
  m[2][0] = xz-wy;
         
  m[0][1] = xy-wz;
  m[1][1] = 1.0f-xx-zz;
  m[2][1] = yz+wx;
         
  m[0][2] = xz+wy;
  m[1][2] = yz-wx;
  m[2][2] = 1.0f-xx-yy;
}

//
// GetXAxis
//  TODO: could be optimized to forget about non used axis values
//

 Vector3 Quat::GetXAxis(void) const
{
  Matrix44 mat;
  InsertInMatrix(mat.m);
  return mat.GetAxis(0);
}

//
// GetYAxis
//  TODO: could be optimized to forget about non used axis values
//

 Vector3 Quat::GetYAxis(void) const
{
  Matrix44 mat;
  InsertInMatrix(mat.m);
  return mat.GetAxis(1);
}

//
// GetZAxis
//  TODO: could be optimized to forget about non used axis values
//

 Vector3 Quat::GetZAxis(void) const
{
  Matrix44 mat;
  InsertInMatrix(mat.m);
  return mat.GetAxis(2);
}

 void Quat::setEuler(Vector4 &e)
{
	/// Local Variables //////////////////////////////////////
	float rx,ry,rz,tx,ty,tz,cx,cy,cz,sx,sy,sz,cc,cs,sc,ss;
	//////////////////////////////////////////////////////////
	// FIRST STEP, CONVERT ANGLES TO RADIANS
	rx =  (e.x * (float)M_PI) / (360 / 2);
	ry =  (e.y * (float)M_PI) / (360 / 2);
	rz =  (e.z * (float)M_PI) / (360 / 2);
	// GET THE HALF ANGLES
	tx = rx * (float)0.5;
	ty = ry * (float)0.5;
	tz = rz * (float)0.5;
	cx = (float)cos(tx);
	cy = (float)cos(ty);
	cz = (float)cos(tz);
	sx = (float)sin(tx);
	sy = (float)sin(ty);
	sz = (float)sin(tz);

	cc = cx * cz;
	cs = cx * sz;
	sc = sx * cz;
	ss = sx * sz;

	v.x = (cy * sc) - (sy * cs);
	v.y = (cy * ss) + (sy * cc);
	v.z = (cy * cs) - (sy * sc);
	w = (cy * cc) + (sy * ss);

	// INSURE THE QUATERNION IS NORMALIZED
	// PROBABLY NOT NECESSARY IN MOST CASES
	Normalize();
	
}

 Vector4 Quat::getEuler() const
{
	float cx,sx;
	float cy,sy,yr;
	float cz,sz;
	Vector4 euler;
	
	
	Matrix44 mat;
  InsertInMatrix(mat.m);
	
	sy = -mat.m[2][0];
	cy = sqrt(1 - (sy * sy));
	yr = (float)atan2(sy,cy);
	
	euler.y = (yr * 180.0f) / (float)M_PI;
  /* avoid divide by zero error only where y = +-90 or +-270
	 * not checking cy becuase of precision erros
	 */

	if (sy != 1.0f && sy != -1.0f)	
	{
		cx = mat.m[2][2] / cy;
		sx = mat.m[2][1] / cy;
		euler.x = ((float)atan2(sx,cx) * 180.0f) / (float)M_PI;	// RAD TO DEG

		cz = mat.m[0][0] / cy;
		sz = mat.m[1][0] / cy;
		euler.z = ((float)atan2(sz,cz) * 180.0f) / (float)M_PI;	// RAD TO DEG
	}
	else
	{
		// SINCE Cos(Y) IS 0, I AM SCREWED.  ADOPT THE STANDARD Z = 0
		// I THINK THERE IS A WAY TO FIX THIS BUT I AM NOT SURE.  EULERS SUCK
		// NEED SOME MORE OF THE MATRIX TERMS NOW
		cx = mat.m[1][1];
		sx = -mat.m[1][2];
		euler.x = ((float)atan2(sx,cx) * 180.0f) / (float)M_PI;	// RAD TO DEG

		cz = 1.0f;
		sz = 0.0f;
		euler.z = ((float)atan2(sz,cz) * 180.0f) / (float)M_PI;	// RAD TO DEG
	}
	
	return euler;

}

 void Quat::Dump(void) const
{
  cout << "QUAT with w = " << w << " and vector is: ";
  v.Dump();
}
 Vector4 Quat::getVector(void) const
{
	return Vector4(v.x, v.y, v.z, w);
}
 Vector4 Quat::getAxisAngle(void) const
{
	Vector4 axis_angle;

   float cos_angle  = w;
   float angle      = acos( cos_angle ) * 180/M_PI;
   float sin_angle  = sqrt( 1.0 - cos_angle * cos_angle );


    if ( fabs( sin_angle ) < 0.0005 )
      sin_angle = 1;

    axis_angle.x = v.x / sin_angle;
    axis_angle.y = v.y / sin_angle;
    axis_angle.z = v.z / sin_angle;
    axis_angle.w = angle;

    return axis_angle;
}







