//
// (c) Polar Pyramid
// suggestions to stolken@kabelfoon.nl and/or stolk@xs4all.nl
//
//

float const DEGTORAD =  0.01745329251994329547;
float const RADTODEG = 57.29577951308232286465;


#include "quat.h"
#include "matrix44.h"
#include <stdlib.h>
#include <iostream>
#include <assert.h>


using namespace std;

/////////////////////////////////////////////////////////////////////////////
//
// Default Constructor: construct a unitmatrix
//

 Matrix44::Matrix44()
{
  Unit();	
}


/////////////////////////////////////////////////////////////////////////////
//
// Conversion Constructor: construct a translation transformation from
// a given 3D Vector
//

 Matrix44::Matrix44(const Vector3 &a)
{
  m[1][0] = m[2][0] = m[0][3] =
  m[0][1] = m[2][1] = m[1][3] =
  m[0][2] = m[1][2] = m[2][3] = 0;
  m[3][0] = a.x;
  m[3][1] = a.y;
  m[3][2] = a.z;
  m[0][0] = m[1][1] = m[2][2] = m[3][3] = 1;
}

//
// Conversion Constructor: construct a rotation matrix from a quaternion
//

 Matrix44::Matrix44(const Quat &q)
{
  q.InsertInMatrix(m);
  m[3][0] = 0.0f;
  m[3][1] = 0.0f;
  m[3][2] = 0.0f;
  m[0][3] = 0.0f;
  m[1][3] = 0.0f;
  m[2][3] = 0.0f;
  m[3][3] = 1.0f;
}

/////////////////////////////////////////////////////////////////////////////
//
// Copy Constructor
//

 Matrix44::Matrix44(const Matrix44& rhs)
{
  m[0][0] = rhs.m[0][0];
  m[0][1] = rhs.m[0][1];
  m[0][2] = rhs.m[0][2];
  m[0][3] = rhs.m[0][3];
  m[1][0] = rhs.m[1][0];
  m[1][1] = rhs.m[1][1];
  m[1][2] = rhs.m[1][2];
  m[1][3] = rhs.m[1][3];
  m[2][0] = rhs.m[2][0];
  m[2][1] = rhs.m[2][1];
  m[2][2] = rhs.m[2][2];
  m[2][3] = rhs.m[2][3];
  m[3][0] = rhs.m[3][0];
  m[3][1] = rhs.m[3][1];
  m[3][2] = rhs.m[3][2];
  m[3][3] = rhs.m[3][3];
}

/////////////////////////////////////////////////////////////////////////////
//
// Elements Constructor
//

 Matrix44::Matrix44(float m00, float m01, float m02, float m03,
			  float m10, float m11, float m12, float m13,
			  float m20, float m21, float m22, float m23,
			  float m30, float m31, float m32, float m33)
{
  m[0][0] = m00;
  m[0][1] = m01;
  m[0][2] = m02;
  m[0][3] = m03;
  m[1][0] = m10;
  m[1][1] = m11;
  m[1][2] = m12;
  m[1][3] = m13;
  m[2][0] = m20;
  m[2][1] = m21;
  m[2][2] = m22;
  m[2][3] = m23;
  m[3][0] = m30;
  m[3][1] = m31;
  m[3][2] = m32;
  m[3][3] = m33;
}

/////////////////////////////////////////////////////////////////////////////
//
// Assignment
//
// normal conversion
//

 Matrix44& Matrix44::Assign(const float mat[4][4])
{
  m[0][0] = mat[0][0];
  m[0][1] = mat[0][1];
  m[0][2] = mat[0][2];
  m[0][3] = mat[0][3];
  m[1][0] = mat[1][0];
  m[1][1] = mat[1][1];
  m[1][2] = mat[1][2];
  m[1][3] = mat[1][3];
  m[2][0] = mat[2][0];
  m[2][1] = mat[2][1];
  m[2][2] = mat[2][2];
  m[2][3] = mat[2][3];
  m[3][0] = mat[3][0];
  m[3][1] = mat[3][1];
  m[3][2] = mat[3][2];
  m[3][3] = mat[3][3];
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Assignment
//
// transposed conversion
//

 Matrix44& Matrix44::TransposeAssign(const float mat[4][4])
{
  m[0][0] = mat[0][0];
  m[1][0] = mat[0][1];
  m[2][0] = mat[0][2];
  m[3][0] = mat[0][3];
  m[0][1] = mat[1][0];
  m[1][1] = mat[1][1];
  m[2][1] = mat[1][2];
  m[3][1] = mat[1][3];
  m[0][2] = mat[2][0];
  m[1][2] = mat[2][1];
  m[2][2] = mat[2][2];
  m[3][2] = mat[2][3];
  m[0][3] = mat[3][0];
  m[1][3] = mat[3][1];
  m[2][3] = mat[3][2];
  m[3][3] = mat[3][3];
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// determinant = Matrix44.Invert()
//
// returns determinant and inverts matrix44 (not if det == 0)
//

 float Matrix44::Invert(void)
{
  int indxc[4], indxr[4], ipiv[4];
  int i, icol, irow, j, k, l, ll;
  float big, dum, pivinv;

  icol = irow = 0;
  ipiv[0] = ipiv[1] = ipiv[2] = ipiv[3] = 0;

  for (i = 0; i < 4; i++) // compile with loop unrolling
  {
    big = 0;
    for (j = 0; j < 4; j++)
    {
      if (ipiv[j] != 1)
      {
        for (k = 0; k < 4; k++)
        {
          if (!ipiv[k])
          {
            if ((dum = fabs(m[j][k])) >= big)
            {
              big = dum;
              irow = j;
              icol = k;
            }
          }
          else if (ipiv[k] > 1)
            return 0;
        }
      }
    }
    ++ipiv[icol];
    if (irow != icol)
    {
      for (l = 0; l < 4; l++)
      {
        float tmp = m[irow][l];
        m[irow][l] = m[icol][l];
        m[icol][l] = tmp;
      }
    }
    indxr[i] = irow;
    indxc[i] = icol;
    if ((dum = m[icol][icol])==0)
      return 0;
    pivinv = 1 / dum;
    m[icol][icol] = 1;
    for (l = 0; l < 4; l++)
      m[icol][l] *= pivinv;
    for (ll = 0; ll < 4; ll++)
    {
      if (ll != icol)
      {
        dum = m[ll][icol];
        m[ll][icol] = 0;
        for (l = 0; l < 4; l++)
          m[ll][l] -= m[icol][l]*dum;
      }
    }
  }
  for (l = 3; l >= 0; l--)
  {
    if (indxr[l] != indxc[l])
    {
      for (k = 0; k < 4; k++)
      {
        float tmp = m[k][indxr[l]];
        m[k][indxr[l]] = m[k][indxc[l]];
        m[k][indxc[l]] = tmp;
      }
    }
  }
  return 1;
}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44.Transpose()
//

 Matrix44 Matrix44::Transpose(void)
{
  return Matrix44(m[0][0], m[1][0], m[2][0], m[3][0],
		  m[0][1], m[1][1], m[2][1], m[3][1],
		  m[0][2], m[1][2], m[2][2], m[3][2],
		  m[0][3], m[1][3], m[2][3], m[3][3]);
}

/////////////////////////////////////////////////////////////////////////////
//
// Unit: construct a unitmatrix
//

 void Matrix44::Unit(void)
{
  m[1][0] = m[2][0] = m[3][0] =
  m[0][1] =           m[2][1] = m[3][1] =
  m[0][2] = m[1][2] =           m[3][2] =
  m[0][3] = m[1][3] = m[2][3] =           0.0f;

  m[0][0] = m[1][1] = m[2][2] = m[3][3] = 1.0f;
}

/////////////////////////////////////////////////////////////////////////////
// Math Operators
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44 *= Matrix44
//

 Matrix44& Matrix44::operator *=(const Matrix44& rhs)
{
  *this = Matrix44(rhs.m[0][0]*m[0][0] +
		   rhs.m[0][1]*m[1][0] +
		   rhs.m[0][2]*m[2][0] +
		   rhs.m[0][3]*m[3][0],
		   rhs.m[0][0]*m[0][1] +
		   rhs.m[0][1]*m[1][1] +
		   rhs.m[0][2]*m[2][1] +
		   rhs.m[0][3]*m[3][1],
		   rhs.m[0][0]*m[0][2] +
		   rhs.m[0][1]*m[1][2] +
		   rhs.m[0][2]*m[2][2] +
		   rhs.m[0][3]*m[3][2],
		   rhs.m[0][0]*m[0][3] +
		   rhs.m[0][1]*m[1][3] +
		   rhs.m[0][2]*m[2][3] +
		   rhs.m[0][3]*m[3][3],
		   rhs.m[1][0]*m[0][0] +
		   rhs.m[1][1]*m[1][0] +
		   rhs.m[1][2]*m[2][0] +
		   rhs.m[1][3]*m[3][0],
		   rhs.m[1][0]*m[0][1] +
		   rhs.m[1][1]*m[1][1] +
		   rhs.m[1][2]*m[2][1] +
		   rhs.m[1][3]*m[3][1],
		   rhs.m[1][0]*m[0][2] +
		   rhs.m[1][1]*m[1][2] +
		   rhs.m[1][2]*m[2][2] +
		   rhs.m[1][3]*m[3][2],
		   rhs.m[1][0]*m[0][3] +
		   rhs.m[1][1]*m[1][3] +
		   rhs.m[1][2]*m[2][3] +
		   rhs.m[1][3]*m[3][3],
		   rhs.m[2][0]*m[0][0] +
		   rhs.m[2][1]*m[1][0] +
		   rhs.m[2][2]*m[2][0] +
		   rhs.m[2][3]*m[3][0],
		   rhs.m[2][0]*m[0][1] +
		   rhs.m[2][1]*m[1][1] +
		   rhs.m[2][2]*m[2][1] +
		   rhs.m[2][3]*m[3][1],
		   rhs.m[2][0]*m[0][2] +
		   rhs.m[2][1]*m[1][2] +
		   rhs.m[2][2]*m[2][2] +
		   rhs.m[2][3]*m[3][2],
		   rhs.m[2][0]*m[0][3] +
		   rhs.m[2][1]*m[1][3] +
		   rhs.m[2][2]*m[2][3] +
		   rhs.m[2][3]*m[3][3],
		   rhs.m[3][0]*m[0][0] +
		   rhs.m[3][1]*m[1][0] +
		   rhs.m[3][2]*m[2][0] +
		   rhs.m[3][3]*m[3][0],
		   rhs.m[3][0]*m[0][1] +
		   rhs.m[3][1]*m[1][1] +
		   rhs.m[3][2]*m[2][1] +
		   rhs.m[3][3]*m[3][1],
		   rhs.m[3][0]*m[0][2] +
		   rhs.m[3][1]*m[1][2] +
		   rhs.m[3][2]*m[2][2] +
		   rhs.m[3][3]*m[3][2],
		   rhs.m[3][0]*m[0][3] +
		   rhs.m[3][1]*m[1][3] +
		   rhs.m[3][2]*m[2][3] +
		   rhs.m[3][3]*m[3][3]);
  return *this;
}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44 * Vector4
//

 Vector4 Matrix44::operator*(const Vector4& a) const
{
  return Vector4
    (m[0][0]*a.x + m[1][0]*a.y + m[2][0]*a.z + m[3][0]*a.w,
     m[0][1]*a.x + m[1][1]*a.y + m[2][1]*a.z + m[3][1]*a.w,
     m[0][2]*a.x + m[1][2]*a.y + m[2][2]*a.z + m[3][2]*a.w,
     m[0][3]*a.x + m[1][3]*a.y + m[2][3]*a.z + m[3][3]*a.w);
}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44 * Vector3
//

 Vector3 Matrix44::operator*(const Vector3& a) const
{
  return Vector3(m[0][0]*a.x + m[1][0]*a.y + m[2][0]*a.z,
		 m[0][1]*a.x + m[1][1]*a.y + m[2][1]*a.z,
		 m[0][2]*a.x + m[1][2]*a.y + m[2][2]*a.z);
}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44 == Matrix44
//

 bool Matrix44::operator ==(const Matrix44& rhs) const
{
  return ((m[0][0] == rhs.m[0][0]) &&
	  (m[0][1] == rhs.m[0][1]) &&
	  (m[0][2] == rhs.m[0][2]) &&
	  (m[0][3] == rhs.m[0][3]) &&
	  (m[1][0] == rhs.m[1][0]) &&
	  (m[1][1] == rhs.m[1][1]) &&
	  (m[1][2] == rhs.m[1][2]) &&
	  (m[1][3] == rhs.m[1][3]) &&
	  (m[2][0] == rhs.m[2][0]) &&
	  (m[2][1] == rhs.m[2][1]) &&
	  (m[2][2] == rhs.m[2][2]) &&
	  (m[2][3] == rhs.m[2][3]) &&
	  (m[3][0] == rhs.m[3][0]) &&
	  (m[3][1] == rhs.m[3][1]) &&
	  (m[3][2] == rhs.m[3][2]) &&
	  (m[3][3] == rhs.m[3][3]));

}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44 != Matrix44
//

 bool Matrix44::operator !=(const Matrix44& m) const
{
  return (!(*this == m));
}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44.CopyTo(m)
//

 void Matrix44::CopyTo(float mat[4][4]) const
{
  mat[0][0] = m[0][0];
  mat[0][1] = m[0][1];
  mat[0][2] = m[0][2];
  mat[0][3] = m[0][3];
  mat[1][0] = m[1][0];
  mat[1][1] = m[1][1];
  mat[1][2] = m[1][2];
  mat[1][3] = m[1][3];
  mat[2][0] = m[2][0];
  mat[2][1] = m[2][1];
  mat[2][2] = m[2][2];
  mat[2][3] = m[2][3];
  mat[3][0] = m[3][0];
  mat[3][1] = m[3][1];
  mat[3][2] = m[3][2];
  mat[3][3] = m[3][3];
}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44.TransposeCopyTo(m)
//
// transposed copy
//

 void Matrix44::TransposeCopyTo(float mat[4][4]) const
{
  mat[0][0] = m[0][0];
  mat[1][0] = m[0][1];
  mat[2][0] = m[0][2];
  mat[3][0] = m[0][3];
  mat[0][1] = m[1][0];
  mat[1][1] = m[1][1];
  mat[2][1] = m[1][2];
  mat[3][1] = m[1][3];
  mat[0][2] = m[2][0];
  mat[1][2] = m[2][1];
  mat[2][2] = m[2][2];
  mat[3][2] = m[2][3];
  mat[0][3] = m[3][0];
  mat[1][3] = m[3][1];
  mat[2][3] = m[3][2];
  mat[3][3] = m[3][3];
}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44.SetPos
//  overrides translation part of matrix according to given position
//

 void Matrix44::SetPos(const Vector4 &pos)
{
  m[3][0] = pos.x;
  m[3][1] = pos.y;
  m[3][2] = pos.z;
}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44.GetPos
//  Get translation part of matrix
//

 Vector4 Matrix44::GetPos(void) const
{
  return Vector4(m[3][0], m[3][1], m[3][2], m[3][3]);
}

/////////////////////////////////////////////////////////////////////////////
//
// Matrix44.GetAxis
//  Get given axis of matrix as a 3D array
//

 Vector3 Matrix44::GetAxis(int nr) const
{
  assert(nr >= 0 && nr <= 3);
  return Vector3(m[nr][0], m[nr][1], m[nr][2]);
}

//
// Matrix44.Orbit
//  get transformation of rotation around local x and y axis of current matrix
//


Matrix44 Matrix44::Orbit(float horizontal,
			 float vertical) const
{
  Vector3 x_axis = GetAxis(0);
  Vector3 y_axis = GetAxis(1);

  Quat horizontalq(y_axis, horizontal*DEGTORAD);
  Quat verticalq(x_axis, vertical*DEGTORAD);

  Quat qmul = verticalq*horizontalq;
  qmul.Normalize();  // Was left out?

  return Matrix44(qmul);
}

//
// Matrix44.Ori
//  return orientation part of matrix
//

 Matrix44 Matrix44::Ori(void) const
{
  return Matrix44(m[0][0], m[0][1], m[0][2], 0.0,
		  m[1][0], m[1][1], m[1][2], 0.0,
		  m[2][0], m[2][1], m[2][2], 0.0,
		  0.0, 0.0, 0.0, 1.0);
}

//
// GetQuat
//  convert rotation matrix part to unit quaternion
//

 Quat Matrix44::GetQuat(void) const
{
  int next_idx[3] = {1, 2, 0};
  Quat q;
  float tr = m[0][0] + m[1][1] + m[2][2];
  if (tr > 0.0)
  {
    float s = sqrt(tr + 1.0);
    q.w = 0.5*s;
    s = 0.5/s;
    q.v.x = (m[2][1] - m[1][2])*s;
    q.v.y = (m[0][2] - m[2][0])*s;
    q.v.z = (m[1][0] - m[0][1])*s;
  }
  else
  {
    int i = 0;
    if (m[1][1] > m[0][0])
      i = 1;
    if (m[2][2] > m[i][i])
      i = 2;
    int j = next_idx[i];
    int k = next_idx[j];
    float s = sqrt((m[i][i] - (m[j][j] + m[k][k])) + 1.0);
    q[i] = 0.5*s;
    s = 0.5/s;
    q.w = (m[k][j] - m[j][k])*s;
    q[j] = (m[j][i] + m[i][j])*s;
    q[k] = (m[k][i] + m[i][k])*s;
  }
  q.Normalize();
  return q;
}

 void Matrix44::ResetOrientation(void)
{
  m[1][0] = m[2][0] = m[0][3] =
  m[0][1] = m[2][1] = m[1][3] =
  m[0][2] = m[1][2] = m[2][3] = 0;
  m[0][0] = m[1][1] = m[2][2] = m[3][3] = 1;
}

 bool FuzEQ(float x, float y, float tolerance=0.00001)
{
  return (fabs((x)-(y)) < 0.00001);
}

 bool Matrix44::IsValid(void) const
{
  if (!FuzEQ(m[3][3], 1.0, 0.0001))         
    return false;          
  if (!FuzEQ(m[0][3], 0.0, 0.0001))
    return false;                          
  if (!FuzEQ(m[1][3], 0.0, 0.0001))                
    return false;
  if (!FuzEQ(m[2][3], 0.0, 0.0001))        
    return false;                      
  return true;
}

 void Matrix44::Dump(void) const
{
  cerr << m[0][0] << "," << m[0][1] << "," << m[0][2] << "," << m[0][3] << "\n";
  cerr << m[1][0] << "," << m[1][1] << "," << m[1][2] << "," << m[1][3] << "\n";
  cerr << m[2][0] << "," << m[2][1] << "," << m[2][2] << "," << m[2][3] << "\n";
  cerr << m[3][0] << "," << m[3][1] << "," << m[3][2] << "," << m[3][3] << "\n";
}



