/***************************************************************************
                      vertex.cpp  -  description                
                         -------------------                   
begin                : Sun Apr 25 1999                    
copyright            : (C) 1999 by Jonathan E. Anderson            
email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "vertex.h"
#include "texturematerial.h"
#include <objectdb.h>
#include "mesh.h"
#include "controller.h"
#include "face.h"
#include "subobject.h"
#include "keyframe.h"

int Vertex::TYPE = Typed::getUID();

Vertex::Vertex( float x, float y, float z, Object *_parent )
    : SubObject( _parent ),
    normal( 0, 1, 0, 0 ),
    pos( x, y, z, 1 ),
    color( -1, -1, -1, -1 )
{
  m_colored = false;
  addType( TYPE );
}

Vertex::Vertex( Vector4 &p, Object *_parent )
    : SubObject( _parent ),
    normal( 0, 1, 0, 0 ),
    pos( p.x, p.y, p.z, 1 ),
    color( -1, -1, -1, -1 )
{

  m_colored = false;
  addType( TYPE );
}

void Vertex::getBoundingMin( Vector4 *v )
{
  *v = pos;
}

void Vertex::getBoundingMax( Vector4 *v )
{
  *v = pos;
}

void Vertex::setColor( float r, float g, float b, float a )
{
  color.assign( r, g, b, a );
  m_colored = true;
}

void Vertex::resetColor()
{
  color.assign( -1, -1, -1, -1 );
  m_colored = false;
}

Vertex::Vertex( Object *_parent )
    : SubObject( _parent ),
    normal( 0, 1, 0, 0 )
{

  addType( TYPE );
}

Vertex::~Vertex()
{}

int Vertex::draw( int i )
{
  //   float v[ 4 ];

  /* Set the color if the vertex is asked to, otherwise,
   * it uses what ever color has been currently set.
   */

  if ( i & VERTEX_COLOR )
  {
    if ( m_colored )
    {
      glColor4f( color.x, color.y, color.z, color.w );
    }

    else
    {
      TextureMaterial *tm = getParentObject() -> getTextureMaterial( 0 );

      if ( getParentObject() -> isSelected() )
        glColor4f( 1, 0, 0, 1 );
      else
        glColor4f( ( float ) tm->cDiffuse.r / 255,
                   ( float ) tm->cDiffuse.g / 255,
                   ( float ) tm->cDiffuse.b / 255,
                   ( float ) tm->alpha / 255 );

    }
  }

  if ( i & VERTEX_NORMAL )
    glNormal3f( normal.x, normal.y, normal.z );

  if ( i & VERTEX_COORD )
    glVertex3f( pos.x, pos.y, pos.z );

  return 0;
}


void Vertex::saveKeyframe( Keyframe * k )
{

  if ( isAnimatable() )
  {
    Keyframe nk( k->getTime() );
    VectorChannel *pc, *nc, *cc;
    pc = new VectorChannel( pos );
    nc = new VectorChannel( normal );
    cc = new VectorChannel( color );
    nk.addChannel( POSITION_CHANNEL, pc );
    nk.addChannel( NORMAL_CHANNEL, nc );
    nk.addChannel( COLOR_CHANNEL, cc );

    *k = nk;
  }
}

void Vertex::loadKeyframe( Keyframe * k )
{
  if ( isAnimatable() )
  {
    pos = dynamic_cast < VectorChannel * > ( k -> getChannel( POSITION_CHANNEL ) ) -> value;
    normal = dynamic_cast < VectorChannel * > ( k -> getChannel( NORMAL_CHANNEL ) ) -> value;
    color = dynamic_cast < VectorChannel * > ( k -> getChannel( COLOR_CHANNEL ) ) -> value;
  }

}

void Vertex::normalize()
{
  normal.assign( 0, 0, 0, 0 );

  int count = flist.size();
  Vector4 tnorm;

  for ( int i = 0; i < ( int ) flist.size(); i++ )
  {
    tnorm = FACE( flist[ i ] ) ->getNormal( index );
    normal += tnorm;
  }

  normal.x /= count;
  normal.y /= count;
  normal.z /= count;
  normal.w = 0;

  normal.normalize();

}

void Vertex::rotate( float amount, float x, float y, float z, float px, float py, float pz )
{
  //change amount to radians
  amount = M_PI / 180 * amount;
  Quat q( Vector3( x, y, z ), amount );
  Matrix44 m;
  q.InsertInMatrix( m.m );

  Vector4 newpos( m * pos );
  setPosition( newpos.x, newpos.y, newpos.z );

}

void Vertex::setRotation( Quat &q )
{

  Matrix44 m;
  q.InsertInMatrix( m.m );
  Vector4 newpos( m * pos );
  setPosition( newpos.x, newpos.y, newpos.z );

}


void Vertex::setMaterial( TextureMaterial *m )
{
  tm = m;
}

void Vertex::move( float x, float y, float z )
{
  //translate into world space...
  Matrix44 m;
  m_parent -> getCompleteMatrix( &m );

  pos = m * pos; //transform to world space
  pos.x += x;
  pos.y += y;
  pos.z += z;
  Matrix44 inv( m );
  inv.Invert();
  pos = inv * pos;  //transform back into object space

  m_parent->setDirtyBox( true );

}

void Vertex::scale( float x, float y, float z, float ox, float oy, float oz )
{

  Matrix44 m;
  Matrix44 mi;
  Vector4 pt;
  Vector4 st;

  m_parent -> getCompleteMatrix( &m );

  mi = m;
  mi.Invert();

  getPosition( &pt );
  pt = m * pt;

  st = pt;

  pt.x = ( ( pt.x - ox ) * x ) + ox;
  pt.y = ( ( pt.y - oy ) * y ) + oy;
  pt.z = ( ( pt.z - oz ) * z ) + oz;

  //figure out how much to move by...
  pt.x += ( pt.x - st.x );
  pt.y += ( pt.y - st.y );
  pt.z += ( pt.z - st.z );

  pt = mi * pt;
  pos = pt;

  m_parent -> setDirtyBox( true );

}

void Vertex::setNormal( float x, float y, float z )
{
  normal.assign( x, y, z, 1 );
}

Vertex & Vertex::operator=( Vertex &rhs )
{

  SubObject::operator=( rhs );

  normal = rhs.normal;

  pos = rhs.pos;

  m_colored = rhs.m_colored;
  color = rhs.color;

  return *this;

}

void Vertex::copyFrom( SubObject *s )
{
  *this = * static_cast < Vertex * > ( s );
}

/**Returns a pointer to a clone of this vertex
 */
Vertex * Vertex::clone()
{

  return 0;
}

/**Loops through this vertex's faces and sets the face UV coord appropriately
 */
void Vertex::setFaceUV( Vector4 &tuv )
{

  for ( int i = 0; i < ( int ) flist.size(); i++ )
  {
    Face *f = FACE( flist[ i ] );
    int findex = f->findVert( index );
    f->setUVCoord( findex, tuv );
  }
}

/**Returns a vector that would be the new position of the vertex after smoothing.
 */
Vector4 Vertex::getSmoothPosition( float tolerance )
{
  Vector4 np;

  vector < SubObject * > uverts;

  uverts.reserve( 25 );

  //put all neighboring faces verts into a vector
  for ( int i = 0; i < ( int ) flist.size(); i++ )
  {
    vector < int > *vlist = FACE( flist[ i ] ) ->getVerts();

    for ( int j = 0; j < ( int ) vlist->size(); j++ )
    {
      uverts.push_back( VERT( ( *vlist ) [ j ] ) );
    }
  }

  //remove all duplicate verts
  uverts.erase( unique( uverts.begin(), uverts.end() ), uverts.end() );

  int n = uverts.size();

  //find the average position.
  for ( int i = 0; i < ( int ) uverts.size(); i++ )
  {
    Vector4 p2;
    Vector4 dp;
    p2 = uverts[ i ] ->getPosition();

    dp = p2 - pos;
    np += dp;
  }

  np.x /= n;
  np.y /= n;
  np.z /= n;

  np += pos;

  return np;

}

/**Returns a vertex that is midway between this vertex, and the given. Note: This must
   be deleted by the caller! */
int Vertex::getMidpoint( int v1 )
{

  Vector4 & ep1 = getPosition();
  Vector4 &ep2 = VERT( v1 ) ->getPosition();

  Vertex *v = getParentObject() -> createVertex( ( ep1.x + ep2.x ) / 2,
              ( ep1.y + ep2.y ) / 2,
              ( ep1.z + ep2.z ) / 2 );

  return v->getParentIndex();
}

/**Checks the references to this vertex to determine if it's deletable*/
bool Vertex::isDeletable()
{
  if ( flist.size() == 0 )
    return true;
  else
    return false;
}

void Vertex::getTransformedPosition( Vector4 *v )
{
  Matrix44 m;
  v->assign( 0, 0, 0, 1 );
  m_parent -> getCompleteMatrix( &m );

  *v = m * pos;
}


























