/***************************************************************************
                        subobject.h  -  description
                           -------------------
  begin                : Sun Aug 13 2000
  copyright            : (C) 2000 by Jon Anderson
  email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SUBOBJECT_H
#define SUBOBJECT_H

#include <string>
#include "entity.h"

class Object;

//Short cut macros

#define VERT(x) getParentObject()->getVertex(x)
#define EDGE(x) getParentObject()->getEdge(x)
#define FACE(x) getParentObject()->getFace(x)
#define UV(x)  getParentObject()->getUVCoord(x)

#define FaceList vector<SubObject*>
#define VertexList vector<SubObject*>
#define EdgeList vector<SubObject*>
#define UVList vector<SubObject*>
#define SubObjectList vector<SubObject*>

class Face;

class Edge;

class Vertex;

class UVCoord;


/** Subobjects are the parts that make up an Object.  An object can
  * have any combination of the following: Vertices, Edges,
  * Faces.
  *
  * Each subobject has a list of subobjects that it is connected to. (references)
  * These are not pointers, but are indexes into the parents list of subobjects.
  * Because of this, subobjects can only be connected to other subobjects with the
  * same parent.
  *
  * In some sense, the subobject is the basis for a winged-edged data structure.
  *
  * The SubObject class is the base class for Vertices, Edges, and Faces.
  * As a general rule, all connectivity maintenance (adding, deleting, swapping, etc)
  * is implemented in the SubObject class.  Any specific modeling behaviour is implemented
  * in the subclasses.
  *
  * @author Jon Anderson
  */

class SubObject : public Animatable, public Selectable
{

public:
   SubObject( float x, float y, float z, Object *o );
   SubObject( Object *o );

   virtual ~SubObject();

   virtual void reindex( int oldi, int newi );
   virtual void setIndex( int );
   virtual void init();

   virtual int draw( int d_options = 0 ) = 0;

   bool visible;

   int getParentIndex()
   {
      return index;
   };

   /** Controls visibility of the sub object.
      * @param flag Whether or not it's visible.
      */
   void setVisible( bool flag )
   {
      visible = flag;
   };

   bool isVisible();

   void removeDuplicateSubObjects( int type );
   void removeDuplicateVerts( );
   void removeDuplicateFaces( );
   void removeDuplicateEdges( );
   void removeDuplicateUVs();

   /** Adds a reference subobject.  The reference so
     * is not modified.
     *
     * @param t The type of subobject.
     * @param x The parent index of the subobject.
     */
   void addSubObject( int t, int x );
   void addFace( int x );
   void addEdge( int x );
   void addVert( int x );


   /** Removes a referenced subobject.  The reference so
     * is not modified.
     *
     * @param t The type of subobject.
     * @param x The parent index of the subobject.
     */
   void removeSubObject( int t, int x );
   void removeFace( int x );
   void removeEdge( int x );
   void removeVert( int x );


   /** Returns the parent index of the referenced subobject.
     *
     * @param t The type of subobject.
     * @param x The index into the subobject's list.
     */
   int getSubObj( int t, int x );
   int getFace( int x );
   int getEdge( int x );
   int getVert( int x );
   int getUV( int x );

   /** Returns a pointer to the referenced subobject.
     *
     * @param t The type of subobject.
     * @param x The index into the subobject's list.
     */
   SubObject * getSubObjPtr( int t, int x );
   Face * getFacePtr( int x );
   Edge * getEdgePtr( int x );
   UVCoord * getUVPtr( int x );
   Vertex * getVertPtr( int x );



   /** Returns the number of subobjects referenced.
     *
     * @param t The type of subobject.
     */
   int getNumSubObjs( int t );
   int getNumFaces();
   int getNumEdges();
   int getNumVerts();


   /** Swaps an old reference with a new one. Neither
     * reference is modified.
     *
     * @param ox The parent index of the old reference.
     * @param nx The parent index of the new reference.

     */
   void swapSubObject( int type, int ox, int nx );
   void swapFace( int ox, int nx );
   void swapVert( int ox, int nx );
   void swapEdge( int ox, int nx );


   /** Replaces this subobject with the given reference.
    * This method will loop through all the connected
    * subobjects and swap references to this subobject
    * with the new reference. It also adds all the connected
    * subobjects to the new subobject.
    *
    * @param nx The parent index of the new reference.
    */
   void replace( int nx );


   /** Determines if two subojects share a 3rd suboject of the given type.
     *
     *
     * @param type The type of suboject to check with.
     * @param index The parent index of the seconf suboject.
     * @param sub   The parent index of the potentially shared subobject.
     */
   bool sharesSubObject( int type, int index, int sub );
   bool sharesVert( int v, int i );
   bool sharesEdge( int e, int i );
   bool sharesFace( int f, int i );


   /** Returns a list of subobjects that are common/shared between
     * two subojects.
     *
     * @param type The type of suboject to check for common subobjects.
     * @param sub The second subobject.
     */
   vector<int> getCommonSubObjects( int type, int sub );
   vector<int> getCommonVerts( int i );
   vector<int> getCommonEdges( int i );
   vector<int> getCommonFaces( int i );


   /** Determines if thwo subobject share <i>any</i> subojects.
    *
    * @param type The type of suboject to check for.
    * @param sub  The suboject to check with.
    */
   bool hasCommonSubObjects( int type, int sub );
   bool hasCommonVerts( int i );
   bool hasCommonEdges( int i );
   bool hasCommonFaces( int i );


   /** Determines if a subobjects is connected to
     * the given subobject
     *
     * @param type The type of the other subobject.
     * @param s  The parent index of the other subobject.
     */
   bool hasSubObject( int type, int s );
   bool hasFace( int f );
   bool hasEdge( int e );
   bool hasVert( int v );


   /** Given a subobject, returns the index into this subobjects references.
     *
     * @param type The type of the other subobject.
     * @so     The parent index of the other subobject.
     */
   int findSubObject( int type, int so );
   int findFace( int f );
   int findVert( int v );
   int findEdge( int e );

   /** Returns a pointer to a list of subobject references.
     *
     * @param t The type of subobject list to get.
     */
   vector<int> * getSubObjectList( int t );
   vector<int> * getFaces();
   vector<int> * getEdges();
   vector<int> * getVerts();
   vector<int> * getUVs();

   /** Returns a pointer to the parent object of this subobject.
     */
   Object * getParentObject()
   {
      return m_parent;
   };

   Vector4 getCenter();

   void remove();
   void detach( bool remove_me = true );

   bool lonely();
   bool adjacent( SubObject * );

   void dump();
   string getDescription( int type = 0 );

   SubObject & operator=( SubObject &rhs );

   virtual void copyFrom( SubObject * )
   {}

   ;

   static int TYPE;

   /* Transformable methods
    * default implement is to do nothing for most. (Only verts will do alot of it,
    * for others, the default is to apply the transform to all contained
    * vertices
    */ 
   //do nothing.
   virtual void getPosition( Vector4 * )
   {}

   ;

   virtual void getOrientation( Quat * )
   {}

   ;


   virtual void getCompleteMatrix( Matrix44 *m );
   virtual void getParentMatrix( Matrix44 *m );

   virtual void getBoundingMin( Vector4 *v );
   virtual void getBoundingMax( Vector4 *v );

   virtual void setOrientation( float d, float x, float y, float z )
   {}

   ;

   virtual void setOrientation( Quat &q )
   {}

   ;

   virtual Quat & getOrientation();

   virtual void setPosition( float x, float y, float z )
   {}

   ;

   virtual Vector4 & getPosition();
   virtual void getTransformedPosition( Vector4 * )
   {}

   ;

   virtual void move( float x, float y, float z );
   virtual void rotate( float amount, float x, float y, float z, float px, float py, float pz );
   virtual void scale( float x, float y, float z, float ox = 0, float oy = 0, float oz = 0 );

   /* Animatable traits
    */
   virtual void loadKeyframe( Keyframe * )
   {}

   ;

   virtual void saveKeyframe( Keyframe * )
   {}

   ;

protected:

   Object * m_parent;

   vector<int> vlist;
   vector<int> elist;
   vector<int> flist;
   vector<int> uvlist;

   int index;

};

#endif
