/***************************************************************************
                      light.cpp  -  description                
                         -------------------                   
begin                : Sun Apr 25 1999                    
copyright            : (C) 1999 by Jonathan E. Anderson            
email                : ande1514@tc.umn.edu               
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "object.h"
#include <Util/util.h>

#include "vertex.h"
#include "material.h"
#include "texture.h"
#include "edge.h"
#include "uvcoord.h"
#include "texturematerial.h"
#include "subobject.h"
#include "face.h"


bool Object::show_axis = false;
int Object::TYPE = Typed::getUID();
TextureMaterial * Object::m_default_material = new TextureMaterial();

Object::Object( float x, float y, float z, Entity *_parent ) :
    Entity( x, y, z, _parent ),
    verts(),
    segs(),
    uvs(),
    edges(), faces(), defunct()
{

  addType( TYPE );

  verts.reserve( 250 );
  tmlayer[ 0 ] = 0;
  m_vis_group = -1;
}

Object::Object( Vector4 &p, Entity *_parent )
    : Entity( p, _parent ),
    verts(),
    segs(),
    uvs(),
    edges(), faces(), defunct()
{
  addType( TYPE );
  verts.reserve( 250 );

  tmlayer[ 0 ] = 0;
  m_vis_group = -1;
}


Object::Object( Entity *_parent )
    : Entity( _parent ),
    verts(),
    segs(),
    uvs(),
    edges(), faces(), defunct()
{

  addType( TYPE );

  verts.reserve( 250 );

  tmlayer[ 0 ] = 0;
  m_vis_group = -1;

}

Object::~Object()
{
  vector < SubObject * > ::iterator it;

  it = verts.begin();

  while ( it != verts.end() )
  {
    SubObject * s = *it;
    delete s;
    ++it;
  }


  it = faces.begin();

  while ( it != faces.end() )
  {
    SubObject * s = *it;
    delete s;
    ++it;
  }

  it = edges.begin();

  while ( it != edges.end() )
  {
    SubObject * s = *it;
    delete s;
    ++it;
  }

  it = uvs.begin();

  while ( it != uvs.end() )
  {
    SubObject * s = *it;
    delete s;
    ++it;
  }

  verts.clear();
  edges.clear();
  faces.clear();
  uvs.clear();





}

/**Return the index in the list of the given subobject */
int Object::findSubObject( int type, SubObject *s )
{
  vector < SubObject * > *solist;
  solist = getSubObjectList( type );

  for ( int i = 0; i < ( int ) solist->size(); i++ )
  {
    if ( ( *solist ) [ i ] == s )
      return i;
  }

  return -1;

}

/**Returns a subobject specified by mode, with the index of index */
SubObject * Object::getSubObject( int type, int index )
{
  vector < SubObject * > * solist;

  if ( type == Vertex::TYPE )
    solist = & verts;
  else if ( type == UVCoord::TYPE )
    solist = & uvs;
  else if ( type == Edge::TYPE )
    solist = & edges;
  else if ( type == Face::TYPE )
    solist = & faces;
  else
  {
    cerr << "ILLEGAL SUBOBJECT TYPE REQUEST IN OBJECT." << endl;
    return 0;
  }

  if ( index > ( int ) solist->size() - 1 )
  {
    cerr << "REQUESTING SUBOBJECT OUT OF BOUNDS. ";
    cerr << "TYPE:" << type << "," << "Index:" << index << endl;
    return 0;
  }

  return ( *solist ) [ index ];

}

int Object::render( int d_options )
{
  //reset colors..
  // glColor4f( 1,1, 1, 1 );

  glPushMatrix();

  /* Setup the transformation matix */
  Matrix44 m;

  getCompleteMatrix( &m );
  glMultMatrixf( ( GLfloat * ) ( m.m ) );

  /* Draw the pivot point and bounding box if we are selected */

  if ( selected )
  {
    drawBoundingBox();
    //temporarily force the red color
    getTextureMaterial( 0 ) -> setSelected( true );
  }



  if ( selected || Object::show_axis )
  {
    drawPivotPoint();
  }


  glPushName( getID() );
  //set up the material
  getTextureMaterial( 0 ) -> beginMaterial();

  draw( d_options );

  glPopName();
  glPopMatrix();

  getTextureMaterial( 0 ) -> setSelected( false );
  getTextureMaterial( 0 ) -> endMaterial();


  return 0; //end-start;


}


int Object::draw( int d_options )
{
  return 0;
}

void Object::drawPivotPoint()
{
  glPushAttrib( GL_LIGHTING_BIT | GL_COLOR_BUFFER_BIT );
  glDisable( GL_LIGHTING );
  glBegin( GL_LINES );
  glColor4f( 0, 0, 1, 1 );
  glVertex3f( 0, 0, 0 );
  glVertex3f( 0, 0, .5 );
  glColor4f( 0, 1, 0, 1 );
  glVertex3f( 0, 0, 0 );
  glVertex3f( 0, .5, 0 );
  glColor4f( 1, 0, 0, 1 );
  glVertex3f( 0, 0, 0 );
  glVertex3f( .5, 0, 0 );
  glEnd();

  glPopAttrib();
}

/**
  * Draws the vertices in a selectable manner.
  * Since they are only points being drawn,
  * we disable the lighting.
  */
void Object::drawVerts()
{
  Vector4 p;
  float v[ 4 ];
  glPointSize( 4.0 );
  glPushAttrib( GL_LIGHTING_BIT );
  glDisable( GL_LIGHTING );
  bool sel = false;
  bool vis = false;

  for ( int i = 0; i < ( int ) verts.size(); i++ )
  {
    sel = verts[ i ] -> isSelected();
    vis = verts[ i ] -> isVisible();

    if ( sel )
      verts[ i ] ->drawBoundingBox();

    glPushName( verts[ i ] ->getParentIndex() );

    verts[ i ] -> getPosition( &p );

    glBegin( GL_POINTS );

    if ( vis )
    {
      if ( sel )
        glColor4f( 1, 0, 0, 1 );
      else
        glColor4f( 0, 1, 0, 1 );

      glVertex4fv( p.v( v ) );
    }

    glEnd();

    glPopName();
  }

  glPointSize( 1.0 );
  glPopAttrib();
}

void Object::drawEdges()
{
  glPushAttrib( GL_COLOR_BUFFER_BIT );

  for ( int i = 0; i < ( int ) edges.size(); i++ )
  {
    glPushName( edges[ i ] ->getParentIndex() );
    static_cast < Edge * > ( edges[ i ] ) -> draw();
    glPopName();
  }

  glPopAttrib();

}

void Object::normalize()
{
  int n = verts.size();

  for ( int i = 0; i < n; i++ )
    getVertex( i ) ->normalize();

}



void Object::scale( float x, float y, float z, float ox, float oy, float oz )
{
  int n = verts.size();

  Matrix44 m;
  Matrix44 mi;
  Vector4 pt;
  Vector4 st;
  Vector4 dt;

  getCompleteMatrix( &m );

  mi = m;
  mi.Invert();

  for ( int i = 0; i < n; i++ )
  {
    Vertex * v = getVertex( i );
    v -> getPosition( &pt );
    pt = m * pt;

    st = pt;

    pt.x = ( ( pt.x - ox ) * x ) + ox;
    pt.y = ( ( pt.y - oy ) * y ) + oy;
    pt.z = ( ( pt.z - oz ) * z ) + oz;

    //figure out how much to move by...
    pt.x += ( pt.x - st.x );
    pt.y += ( pt.y - st.y );
    pt.z += ( pt.z - st.z );

    pt = mi * pt;

    v -> setPosition( pt );

  }

  setDirtyBox( true );

  bounding_max.x *= x;
  bounding_max.y *= y;
  bounding_max.z *= z;
  bounding_min.x *= x;
  bounding_min.y *= y;
  bounding_min.z *= z;

}

void Object::rotate( float amount, float x, float y, float z, float px, float py, float pz )
{
  Entity::rotate( amount, x, y, z, px , py, pz );
  calcBoundingBox();
}


void Object::setSize( float w, float h, float d )
{
  //ensure we have a good bounding box.
  calcBoundingBox();

  float o_w, o_h, o_d;

  o_w = bounding_max.x - bounding_min.x;
  o_h = bounding_max.y - bounding_min.y;
  o_d = bounding_max.z - bounding_min.z;


  float wp, hp, dp;  //find the amounts to scale by
  wp = w / o_w;
  hp = h / o_h;
  dp = d / o_d;

  //scale each individual vertex;
  int n = verts.size();

  for ( int i = 0; i < n; i++ )
    verts[ i ] ->scale( wp, hp, dp );

  bounding_min.assign( -w / 2, -h / 2, -d / 2, 1 );

  bounding_max.assign( w / 2, h / 2, d / 2, 1 );


}

void Object::getBoundingMin( Vector4 *v )
{
  calcBoundingBox();
  *v = bounding_min;
}

void Object::getBoundingMax( Vector4 *v )
{
  calcBoundingBox();
  *v = bounding_max;
}

void Object::calcBoundingBox()
{
  int n = verts.size();

  float max_x = -100000;
  float max_y = -100000;
  float max_z = -100000;
  float min_x = 100000;
  float min_y = 100000;
  float min_z = 100000;

  Vector4 p;

  //Matrix44 inv;

  // getCompleteMatrix(&inv);

  //inv.Invert();

  for ( int i = 0; i < n; i++ )
  {
    verts[ i ] ->getBoundingMax( &p );   //get the global vert position
    // p = inv * p;                //transform into object space.

    if ( p.x > max_x )
      max_x = p.x;

    if ( p.y > max_y )
      max_y = p.y;

    if ( p.z > max_z )
      max_z = p.z;

    if ( p.x < min_x )
      min_x = p.x;

    if ( p.y < min_y )
      min_y = p.y;

    if ( p.z < min_z )
      min_z = p.z;
  }

  bounding_max.assign( max_x, max_y, max_z, 1 );
  bounding_min.assign( min_x, min_y, min_z, 1 );

}

void Object::center()
{
  //cerr << "Centering" << endl;

  Vector4 min( 100000, 100000, 100000, 100000 );
  Vector4 max( -100000, -100000, -100000, -100000 );

  Vector4 p;

  //find the bounds of the object.
  for ( int i = 0; i < ( int ) verts.size(); i++ )
  {
    verts[ i ] ->getPosition( &p );

    min.min( p );
    max.max( p );

  }

  bounding_max.assign( max.x, max.y, max.z, 1 );
  bounding_min.assign( min.x, min.y, min.z, 1 );

  //move the center to the middle of the poly.
  float cx = ( max.x + min.x ) / 2;
  float cy = ( max.y + min.y ) / 2;
  float cz = ( max.z + min.z ) / 2;


  Vector4 m( cx, cy, cz, 1 );

  m.Dump();

  //current position is 0, as far as the vertices are concerned.
  move( m );
  //setPosition( cx, cy, cz );

  //shift the bounding box and vertices by (pos - cx)
  bounding_max.x -= cx;
  bounding_max.y -= cy;
  bounding_max.z -= cz;

  bounding_min.x -= cx;
  bounding_min.y -= cy;
  bounding_min.z -= cz;

  for ( int i = 0; i < ( int ) verts.size(); i++ )
  {
    verts[ i ] -> move( -cx, -cy, -cz );
  }

}

void Object::dumpVerts( vector < Vertex * > &list )
{
  Vector4 p;
  cerr << "Vertex List :" << list.size() << " verts" << endl;

  for ( int i = 0; i < ( int ) list.size(); i++ )
  {
    p = list[ i ] ->getPosition();
    cerr << " v: " << p.x << "," << p.y << "," << p.z << endl;
  }
}

void Object::dump()
{
  cerr << "Verts" << endl;

  for ( int i = 0; i < ( int ) verts.size(); i++ )
  {
    verts[ i ] -> dump();
  }

  cerr << "Faces" << endl;

  for ( int i = 0; i < ( int ) faces.size(); i++ )
  {
    faces[ i ] -> dump();
  }

  cerr << "Edges" << endl;

  for ( int i = 0; i < ( int ) edges.size(); i++ )
  {
    edges[ i ] -> dump();
  }


}

vector < SubObject * > * Object::getSubObjectList( int type )
{
  if ( type == Vertex::TYPE )
    return & verts;
  else if ( type == Face::TYPE )
    return & faces;
  else if ( type == Edge::TYPE )
    return & edges;
  else if ( type == UVCoord::TYPE )
    return & uvs;
  else
  {
    cerr << "ILLEGAL SUBOBJECT LIST REQUEST IN OBJECT, TYPE:" << type << endl;
    cerr.flush();
    return 0;
  }
}

void Object::addSubObject( SubObject *so )
{
  vector < SubObject * > *solist;
  solist = getSubObjectList( so -> getType() );
  solist->push_back( so );

}

void Object::removeSubObject( SubObject *s )
{

  vector < SubObject * > *solist;

  solist = getSubObjectList( s->getType() );
  int ind = s->getParentIndex();

  //erase the subobject
  vector < SubObject * > ::iterator er = find( solist->begin(), solist->end(), s );

  if ( er == solist->end() )
  {
    cerr << "Didn't find subobject to remove" << endl;
  }

  solist->erase( er );


  //reindex all subobjects afterwards.
  for ( int i = ind; i < ( int ) solist->size(); i++ )
  {
    SubObject *ts = ( *solist ) [ i ];
    ts->setIndex( i );
  }

}

/**Returns an edge connecting the two indexed vertices.
  *If no edge exists, a new one is created.
  */
int Object::getEdge( int v1, int v2 )
{
  vector < int > *elist = verts[ v1 ] -> getEdges();

  for ( int i = 0; i < ( int ) elist->size(); i++ )
  {
    Edge *e = getEdge( ( *elist ) [ i ] );

    if ( e->hasVert( v2 ) )
      return ( *elist ) [ i ];
  }

  //no previous edge was found, so create a new one.
  Edge *e = createEdge( v1, v2 );

  return e->getParentIndex();

}

void Object::copyFrom( Entity *rhs )
{
  *this = * static_cast < Object * > ( rhs );
}


Object & Object::operator=( Object &rhs )
{
  Entity::operator=( rhs );


  rhs.copyVertsTo( this );
  rhs.copyEdgesTo( this );
  rhs.copyFacesTo( this );
  rhs.copyUVsTo( this );

  //copy layers
  for ( int i = 0; i < 5; i++ )
  {
    tmlayer[ i ] = rhs.tmlayer[ i ];
  }

  m_vis_group = rhs.m_vis_group;

  return *this;

}

/**
  * Copies only the vertex positional data over to the new object.
  * tries to reuse Vertex objects as necessary.
  */
void Object::copyVertsPosTo( Object * o )
{

  int numVs = numVerts();
  int o_numVerts = o -> numVerts();

  vector < SubObject * > *o_verts = o -> getVerts();

  //make room for more verts if necessary
  if ( numVs > o_numVerts )
  {

    o_verts -> reserve( numVs );

  }

  else if ( o_numVerts > numVs )
  {

    for ( int i = numVs; i < o_numVerts; i++ )
    {
      SubObject * s = ( *o_verts ) [ i ];
      delete s;
    }

    o_verts -> resize( numVs );

  }

  for ( int i = 0; i < numVs; i++ )
  {
    Vertex * v = static_cast < Vertex * > ( verts[ i ] );

    if ( i >= o_numVerts )
    {
      o -> createVertex( v -> getPosition() );
    }

    else
    {
      Vertex *o_v = static_cast < Vertex * > ( ( *o_verts ) [ i ] );
      o_v -> setPosition( v -> getPosition() );
    }
  }


}

void Object::copyVertsTo( Object * o )
{
  copySubObjectsTo( Vertex::TYPE, o );
}

void Object::copyUVsTo( Object * o )
{
  copySubObjectsTo( UVCoord::TYPE, o );
}

void Object::copyFacesTo( Object * o )
{
  copySubObjectsTo( Face::TYPE, o );
}

void Object::copyEdgesTo( Object * o )
{
  copySubObjectsTo( Edge::TYPE, o );
}

void Object::copyMaterialsTo( Object * o )
{
  // not needed for now.
}

void Object::copySubObjectsTo( int type, Object * o )
{
  int numSos = getSubObjectList( type ) -> size();
  int o_numSos = o -> getSubObjectList( type ) -> size();

  vector < SubObject * > *o_sos = o -> getSubObjectList( type );

  //make room for more verts if necessary
  if ( numSos > o_numSos )
  {

    o_sos -> reserve( numSos );

  }

  else if ( o_numSos > numSos )
  {

    for ( int i = numSos; i < o_numSos; i++ )
    {
      SubObject * s = ( *o_sos ) [ i ];
      delete s;
    }

    o_sos -> resize( numSos );

  }

  for ( int i = 0; i < numSos; i++ )
  {

    SubObject * s = getSubObject( type, i );

    if ( i >= o_numSos )
    {

      SubObject * o_s = o -> createSubObject( type );
      o_s -> copyFrom( s );

    }

    else
    {
      SubObject * s = getSubObject( type, i );
      SubObject *o_s = ( *o_sos ) [ i ] ;
      o_s -> copyFrom( s );
    }
  }

}

SubObject * Object::createSubObject( int type )
{

  if ( type == Vertex::TYPE )
    return createVertex();
  else if ( type == Edge::TYPE )
    return createEdge();
  else if ( type == Face::TYPE )
    return createFace();
  else if ( type == UVCoord::TYPE )
    return createUVCoord();
  else
    return 0;

}



int Object::getMidPoint( int v1, int v2 )
{
  for ( int i = 0; i < ( int ) segs.size(); i++ )
  {
    if ( segs[ i ] ->v1 == v1 &&
         segs[ i ] ->v2 == v2 )
      return segs[ i ] ->mid->getParentIndex();

    if ( segs[ i ] ->v2 == v1 &&
         segs[ i ] ->v1 == v2 )
      return segs[ i ] ->mid->getParentIndex();
  }

  Vector4 p1 = verts[ v1 ] ->getPosition();
  Vector4 p2 = verts[ v2 ] ->getPosition();

  p1 += p2;
  p1 /= 2;

  Vertex *t = createVertex( p1 );
  VertSeg *vs = new VertSeg();
  vs->v1 = v1;
  vs->v2 = v2;
  vs->mid = t;
  segs.push_back( vs );

  return t->getParentIndex();
}


//gets the orientation of the given triangle
float Object::getFaceOrientation( vector < int > & tri )
{
  Vector4 a, b;

  a = getVertex( tri[ 1 ] ) -> getPosition() - getVertex( tri[ 0 ] ) -> getPosition();
  b = getVertex( tri[ 2 ] ) -> getPosition() - getVertex( tri[ 0 ] ) -> getPosition();

  float o = a.dot( b );
  return o;

}

/**Returns the list of subobjects that are connected to the given subobjects
  */
vector < int > Object::getSubObjectSubObjects( int type, int stype, vector < int > &solist )
{
  vector < int > connected_subobjects;

  for ( int i = 0; i < ( int ) solist.size(); i++ )
  {

    vector < int > *types;

    if ( type == Vertex::TYPE )
      types = getSubObject( stype, solist[ i ] ) ->getVerts();
    else if ( type == Face::TYPE )
      types = getSubObject( stype, solist[ i ] ) ->getFaces();
    else if ( type == Edge::TYPE )
      types = getSubObject( stype, solist[ i ] ) ->getEdges();
    else
      types = getSubObject( stype, solist[ i ] ) ->getVerts();

    for ( int j = 0; j < ( int ) types->size(); i++ )
    {
      connected_subobjects.push_back( ( *types ) [ j ] );
    }
  }

  return connected_subobjects;



}

void Object::markForDeletion( SubObject *s )
{
  defunct.insert( defunct.end(), s );
}

/**Remove all the subobjects that are marked for deletion
  */
void Object::cleanSubObjects()
{
  // list<SubObject *>::iterator it;
  /*
    typedef map< SubObject *, int, less<SubObject *> > SubObjectMap;
    typedef SubObjectMap::value_type SubObjectPair;
   
   
    SubObjectMap sobjs;
    list<SubObject*>::iterator sit = defunct.begin();
    while( sit != defunct.end() ){
    SubObject *s = *sit;
    vector<SubObject *> *solist;
   
     solist = getSubObjectList( s->getType() );
     int ind = s->getParentIndex();
     
     //add up all the reindexing todo.
     for(int i=ind+1; i<(int)solist->size(); i++){
       SubObject *ts = (*solist)[i];
          sobjs[ts]++;
  //     ts->setIndex( i );
     }
     ++sit;
    }
   
    cerr << "Cached reindexing..."<<endl;
   
    SubObjectMap::iterator it = sobjs . begin();
   
    while( it != sobjs . end() )
    {
   
      SubObject *ts =  it -> first;
      ts -> setIndex( ts -> getParentIndex() - (it -> second) );
      ++it;
    }
   
    cerr << "Applied reindexing, removing..."<<endl;
   
    while( !defunct.empty() ){
    SubObject *s = defunct.front();
    defunct.pop_front();
   
    vector<SubObject *> *solist;
   
     solist = getSubObjectList( s->getType() );
   
    //erase the subobject
     vector<SubObject *>::iterator er = find(solist->begin(), solist->end(), s );
     if(er == solist->end()){
      cerr<<"Didn't find subobject to remove"<<endl;
     }
     solist->erase(er);
    
   }
   
  */

  //sort so the duplicates are removed.
  defunct.sort();

  //remove any duplicates in the defunct queue..
  list < SubObject * > ::iterator dups = std::unique( defunct.begin(), defunct.end() );

  defunct.erase( dups, defunct.end() );


  // for(it=defunct.begin(); it != defunct.end(); ++it) {
  //  SubObject *s = *it;
  //  cerr << s->getDescription() << " : " << s->getParentIndex() << ":"<<s<<endl;
  //}


  while ( !defunct.empty() )
  {
    SubObject * s = defunct.front();
    defunct.pop_front();
    removeSubObject( s );
  }

}



inline void Object::addVert( Vertex *v )
{
  verts.push_back( v );
}

inline void Object::addFace( Face *f )
{
  faces.push_back( f );
}

inline void Object::addEdge( Edge *e )
{
  edges.push_back( e );
}

/** Returns the list of edges order such that a single face separates each.
 */
vector < int > Object::getFaceOrderedEdges( vector < int > &e )
{
  vector < int > ordered_edges;
  vector < int > elist;

  ordered_edges.reserve( elist.size() );
  elist = e;

  int index = 0;
  ordered_edges.push_back( elist[ 0 ] );
  elist.erase( find( elist.begin(), elist.end(), elist[ 0 ] ) );

  while ( ! elist.empty() )
  {
    int i = 0;

    while ( ! ( getEdge( ordered_edges[ index ] ) -> hasCommonFaces( elist[ i ] ) ) )
      i++;

    ordered_edges.push_back( elist[ i ] );

    elist.erase( find( elist.begin(), elist.end(), elist[ i ] ) );

    index++;
  }

  return ordered_edges;
}

void Object::removeVert( int i )
{
  // removeSubObject(ENTITY_VERTEX, i);
}

void Object::removeFace( int i )
{
  // removeSubObject(ENTITY_FACE, i);
}

void Object::removeEdge( int i )
{
  // removeSubObject(ENTITY_EDGE, i);
}

int Object::findFace( Face *f )
{
  return findSubObject( Face::TYPE, f );
}

int Object::findEdge( Edge *f )
{
  return findSubObject( Edge::TYPE, f );
}

int Object::findVertex( Vertex *f )
{
  return findSubObject( Vertex::TYPE, f );
}

int Object::findUV( UVCoord *f )
{
  return findSubObject( UVCoord::TYPE, f );
}

void Object::dumpVector( vector < int > &l )
{
  for ( int i = 0; i < ( int ) l.size(); i++ )
  {
    cerr << l[ i ] << " ";
  }

  cerr << endl;

}

/* removes any duplicate or coincident edges.
 */
void Object::cleanEdges()
{

  int n = edges.size();

  for ( int i = 0; i < ( int ) n; i++ )
  {
    for ( int j = i + 1; j < ( int ) n; j++ )
    {
      Edge *e1 = getEdge( i );
      Edge *e2 = getEdge( j );
      //      vector<int> *vlist = e1 -> getVerts();
      if ( *e1 == *e2 )
      {
        cerr << "Removing dup. edges.\n";
        e1 -> dump();
        e2 -> dump();
        e2 -> replace( i );
        e1 -> removeDuplicateVerts();
        e1 -> removeDuplicateFaces();

      }
    }
  }

  cleanSubObjects();

}


/* removes any duplicate or coincident edges.
 * Takes in a list of possible dupes inorder
 * to process faster.
 */
void Object::cleanEdges( vector < int > &elist )
{

  int n = elist.size();

  for ( int i = 0; i < ( int ) n; i++ )
  {
    for ( int j = i + 1; j < ( int ) n; j++ )
    {
      Edge *e1 = getEdge( elist[ i ] );
      Edge *e2 = getEdge( elist[ j ] );

      if ( *e1 == *e2 )
      {
        e2 -> replace( i );
        e1 -> removeDuplicateVerts();
        e1 -> removeDuplicateFaces();

      }
    }
  }

  cleanSubObjects();
}

void Object::cleanVerts()
{}

/** Deletes a subobject of the given type.
  */
void Object::deleteSubObject( int type, int i )
{}

TextureMaterial * Object::getTextureMaterial( int i )
{
  if ( tmlayer[ i ] == 0 )
  {
    return m_default_material;
  }

  else
  {
    return tmlayer[ i ];
  }
}

Face * Object::getFace( int i )
{
  return ( Face * ) getSubObject( Face::TYPE, i );
};

inline Vertex * Object::getVertex( int i )
{
  return ( Vertex * ) getSubObject( Vertex::TYPE, i );
};

inline Edge * Object::getEdge( int i )
{
  return ( Edge * ) getSubObject( Edge::TYPE, i );
};

UVCoord * Object::getUVCoord( int i )
{
  return ( UVCoord * ) getSubObject( UVCoord::TYPE, i );
};

inline vector < int > Object::getSubObjectFaces( int stype, vector < int > &solist )
{
  return getSubObjectSubObjects( Face::TYPE, stype, solist );
};

inline vector < int > Object::getSubObjectEdges( int stype, vector < int > &solist )
{
  return getSubObjectSubObjects( Edge::TYPE, stype, solist );
};

inline vector < int > Object::getSubObjectVerts( int stype, vector < int > &solist )
{
  return getSubObjectSubObjects( Vertex::TYPE, stype, solist );
};

inline void Object::deleteVert( int i )
{
  deleteSubObject( Vertex::TYPE, i );
};

inline void Object::deleteEdge( int i )
{
  deleteSubObject( Edge::TYPE, i );
};

inline void Object::deleteFace( int i )
{
  deleteSubObject( Face::TYPE, i );
};



inline Vertex * Object::createVertex( )
{
  Vertex * v = new Vertex( this );
  addVert( v );

  v -> setIndex( verts.size() - 1 );
  return v;
}

inline Vertex * Object::createVertex( Vector4 &p )
{
  Vertex * v = new Vertex( p, this );
  addVert( v );

  v -> setIndex( verts.size() - 1 );
  return v;
}

Vertex * Object::createVertex( float x, float y, float z )
{
  Vertex * v = new Vertex( x, y, z, this );
  addVert( v );
  v -> setIndex( verts.size() - 1 );
  return v;
}

inline Edge * Object::createEdge( )
{
  Edge * e = new Edge( this );
  addEdge( e );
  e -> setIndex( edges.size() - 1 );
  return e;
}

inline Edge * Object::createEdge( int v1, int v2 )
{
  Edge * e = new Edge( v1, v2, this );
  addEdge( e );
  e -> setIndex( edges.size() - 1 );

  e -> init();

  return e;
}

Face * Object::createFace ( int v1, int v2, int v3 )
{
  Face * f = new Face( this );

  addFace( f );

  f-> setIndex( faces.size() - 1 );

  vector < int > vs;
  vs.reserve( 3 );
  vs.push_back( v1 );
  vs.push_back( v2 );
  vs.push_back( v3 );

  f -> init( vs );
  return f;
}

Face * Object::createFace ( int v1, int v2, int v3, int v4 )
{
  Face * f = new Face( this );
  addFace( f );
  f-> setIndex( faces.size() - 1 );
  vector < int > vs;
  vs.reserve( 4 );
  vs.push_back( v1 );
  vs.push_back( v2 );
  vs.push_back( v3 );
  vs.push_back( v4 );

  f -> init( vs );
  return f;
}

Face * Object::createFace ( vector < int > &v )
{

  Face * f = new Face( this );
  addFace( f );
  f-> setIndex( faces.size() - 1 );

  f -> init( v );
  return f;

}


Face * Object::createFace ( vector < int > &v, vector < int > &uv )
{

  Face * f = new Face( this );
  addFace( f );
  f-> setIndex( faces.size() - 1 );

  f -> init( v, uv );
  return f;

}

inline Face * Object::createFace ( )
{

  Face * f = new Face( this );
  addFace( f );
  f-> setIndex( faces.size() - 1 );

  return f;

}

UVCoord* Object::createUVCoord( float x, float y )
{
  Vector4 p( x, y, 0, 1 );

  UVCoord * uv = new UVCoord( p, this );
  addSubObject( uv );
  uv -> setIndex( uvs.size() - 1 );
  return uv;
}

inline UVCoord* Object::createUVCoord( )
{
  UVCoord * uv = new UVCoord( this );
  addSubObject( uv );
  uv -> setIndex( uvs.size() - 1 );
  return uv;
}






































