/***************************************************************************
                       nurbscurve.cpp  -  description
                          -------------------                                         
 begin                : Thu Nov 18 1999                                           
 copyright            : (C) 1999 by Jon Anderson                         
 email                : janderson@onelink.com                                     
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/



#include "nurbscurve.h"
#include "ctrlpt.h"
#include "vertex.h"
#include <vector.h>
#include "texturematerial.h"
#include "polyskin.h"
#include "../objectdb.h"
#include "nurbssurface.h"
#include <GL/glut.h>
#include "nurbscurveparser.h"

int NurbsCurve::TYPE = Typed::getUID();
int NurbsCurveParser::TYPE = Parser::setParser( NurbsCurve::TYPE, "curve", new NurbsCurveParser() );

NurbsCurve::NurbsCurve( Entity *p, int ctype ) : Spline( p ), create_pts()
{
  init();
  create_type = ctype;
}

NurbsCurve::NurbsCurve( Vector4 &pt, int ctype ) : Spline(), create_pts()
{
  init();

  create_pts.push_back( pt );
  create_type = ctype;

}

NurbsCurve::~NurbsCurve()
{
  gluDeleteNurbsRenderer( theNurbs );
}

void NurbsCurve::init()
{

  cerr << "Creating Curve" << endl;
  addType( TYPE );

  theNurbs = gluNewNurbsRenderer();

  usegs = 35;

  gluNurbsProperty( theNurbs, ( GLenum ) GLU_SAMPLING_METHOD, GLU_DOMAIN_DISTANCE );
  gluNurbsProperty( theNurbs, ( GLenum ) GLU_SAMPLING_TOLERANCE, 75.0 );
  gluNurbsProperty( theNurbs, ( GLenum ) GLU_U_STEP, 4 );

}

int NurbsCurve::numVerts()
{
  if( verts.size() > 0 )
    return verts.size();

  return create_pts.size();
}

void NurbsCurve::extend( Vector4 &pt )
{

  switch ( create_type )
  {

    case CONTROL_CURVE:
      create_pts.push_back( pt );
      controlPoints( create_pts );
      break;

    case INTERPOLATE_CURVE:
      create_pts.push_back( pt );
      interpolate( create_pts );
      break;

    case CIRCLE_CURVE:

      if ( create_pts.size() < 2 )
      {
        create_pts.push_back( pt );

        if ( create_pts.size() == 2 )
          makeCircle( create_pts[ 0 ], create_pts[ 1 ] );
      }

      else
      {
        create_pts[ 1 ] = pt;
        makeCircle( create_pts[ 0 ], create_pts[ 1 ] );
      }

      break;

    default:
      break;
  }

}


void NurbsCurve::makeCircle( Vector4 & pt1, Vector4 & pt2 )
{

  //first point is the center, second is used to determine axis and radius.

  Vector4 axis, x, y;
  float radius;

  axis = pt2 - pt1;
  radius = pt1.distance3d( pt2 );

  if ( radius == 0 )
    radius = 1;

  x.assign( 1, 0, 0, 0 );

  y.assign( 0, 1, 0, 0 );

  //assume that one element of axis is zero.
  if ( axis.x == 0 )   //YZ Plane
    x.assign( 0, 0, 1, 0 );

  if ( axis.y == 0 )   //XZ plane
    y.assign( 0, 0, 1, 0 );

  PLib::Point3Df x_axis( x.x, x.y, x.z );

  PLib::Point3Df y_axis( y.x, y.y, y.z );

  PLib::Point3Df origin( 0, 0, 0 );

  curve.makeCircle( origin, x_axis, y_axis, radius, 0, M_PI * 2 );

  setUpControlPoints();

  setPosition( pt1.x, pt1.y, pt1.z );

}

void NurbsCurve::interpolate( vector < Vector4 > &_verts )
{
  int degree = 1;

  int count = _verts.size();

  if ( count < 2 )
    return ;

  if ( count == 2 )
    degree = 1;

  if ( count == 3 )
    degree = 2;

  if ( count > 3 )
    degree = 3;


  points = Vector_Point3Df( count );

  tangents = Vector_Point3Df( count );

  for ( int i = 0; i < count; i++ )
  {
    Vector4 &p = _verts[ i ];
    points[ i ] = PLib::Point3Df( p.x, p.y, p.z );
    tangents[ i ] = PLib::Point3Df( 1, 0, 0 );
  }

  //curve.makeCircle(PLib::Point3Df(0, 0, 0), 1, 0, 2*M_PI);

  curve.globalInterp( points, degree );

  //get the control points and add them to this nurbs list.
  setUpControlPoints();

}

void NurbsCurve::interpolateClosed( vector < Vector4 > &_verts )
{
  int degree = 1;

  int count = _verts.size();

  if ( count == 2 )
    degree = 1;

  if ( count == 3 )
    degree = 2;

  if ( count > 3 )
    degree = 3;


  points = Vector_Point3Df( count );

  tangents = Vector_Point3Df( count );

  for ( int i = 0; i < count; i++ )
  {
    Vector4 &p = _verts[ i ];
    points[ i ] = PLib::Point3Df( p.x, p.y, p.z );
    tangents[ i ] = PLib::Point3Df( 1, 0, 0 );
  }

  //curve.makeCircle(PLib::Point3Df(0, 0, 0), 1, 0, 2*M_PI);

  curve.globalInterpClosed( points, degree );

  //get the control points and add them to this nurbs list.
  setUpControlPoints();

}


void NurbsCurve::controlPoints( vector < Vector4 > &_verts )
{
  int degree = 1;

  int count = _verts.size();

  if ( count == 2 )
    degree = 1;

  if ( count == 3 )
    degree = 2;

  if ( count > 3 )
    degree = 3;


  Vector_HPoint3Df pts( count );

  PLib::Vector < float > knot( count + degree + 1 );

  PLib::Vector < float > weights( count );

  for ( int i = 0; i < count; i++ )
  {
    Vector4 &p = _verts[ i ];
    pts[ i ] = PLib::HPoint3Df( p.x, p.y, p.z, 1 );
  }

  for ( int i = 0;i < degree + 1;++i )
    knot[ i ] = 0 ;

  for ( int i = degree + 1;i < pts.n();++i )
    knot[ i ] = float( i - degree ) / float( pts.n() - degree ) ;

  for ( int i = pts.n();i < knot.n();++i )
    knot[ i ] = 1.0 ;

  //  for(int i=1; i<count-1; i++)
  //   weights[i] = 100;

  //  weights[0] = 1;
  //  weights[count-1] = 1;




  curve = PLib::NurbsCurvef( pts, knot, degree );

  //  curve = PLib::NurbsCurvef(pts, weights, knot, degree);

  //get the control points and add them to this nurbs list.
  setUpControlPoints();
}

int NurbsCurve::draw( int d_options )
{
  if ( create_pts.size() == 1 )
  {
    glTranslatef( create_pts[ 0 ].x, create_pts[ 0 ].y, create_pts[ 0 ].z );
    glutSolidSphere( .25, 4, 4 );
    return 0;
  }

  glPushAttrib( GL_LIGHTING_BIT );
  glDisable( GL_LIGHTING );

  gluBeginCurve( theNurbs );
  gluNurbsCurve( theNurbs, curve.knot().n(), curve.knot().memory(),
                 4,
                 curve.ctrlPnts().memory() ->data,
                 curve.degree() + 1,
                 GL_MAP1_VERTEX_4 );

  gluEndCurve( theNurbs );


  glPopAttrib();

  if ( SelectMode::is( Vertex::TYPE ) )
  {     //need to draw verts too?
    drawControlPolygon();
    drawVerts();
  }

  return 0;

}


void NurbsCurve::setUpControlPoints()
{
  if ( create_pts.size() < 2 )
    return ;

  int x;

  Vector_HPoint3Df v = curve.ctrlPnts();

  PLib::HPoint3Df temp;

  CtrlPt *c;

  verts.clear();

  x = v.n();

  for ( int i = 0; i < x; ++i )
  {
    temp = v[ i ];
    c = createCtrlPt( temp.x() / temp.w(), temp.y() / temp.w(), temp.z() / temp.w() );
    c->setMapping( i, 0 );
  }

  degree = curve.degree();

}

void NurbsCurve::moveControlPoint( CtrlPt *c, float x, float y, float z )
{
  //ignore the j
  curve.modCPby( c->i, PLib::HPoint3Df( x, y, z, 0 ) );

  PLib::HPoint3Df temp;
  temp = curve.ctrlPnts( c->i );
  c->setPosition( temp.x() / temp.w(), temp.y() / temp.w(), temp.z() / temp.w() );

}


void NurbsCurve::setDirtyBox( bool flag )
{
  //reconstruct the curve;
  /*  Vector4 *p;
    
  int count = verts->count();
  points = Vector_Point3Df(count);

  for(int i=0; i<count; i++){
   p =verts->at(i)->getPosition();
   points[i] = PLib::Point3Df(p->x, p->y, p->z);
}
  //curve.makeCircle(PLib::Point3Df(0, 0, 0), 1, 0, 2*M_PI);
  curve.globalInterp(points, 3);
   */
  Entity::setDirtyBox( flag );

}

void NurbsCurve::drawControlPolygon()
{
  glPushAttrib( GL_COLOR_BUFFER_BIT | GL_LIGHTING_BIT );
  glDisable( GL_LIGHTING );

  glColor4f( 0, 0, 1, 1 );
  glLineWidth( 2 );
  glBegin( GL_LINE_STRIP );

  for ( int i = 0;i < curve.ctrlPnts().n();++i )
    glVertex4fv( curve.ctrlPnts() [ i ].data ) ;

  glEnd() ;

  glLineWidth( 1 );

  glPopAttrib();
}

void NurbsCurve::reverse()
{}

void NurbsCurve::closeSpline()
{}

void NurbsCurve::breakSpline()
{}

PLib::NurbsCurvef NurbsCurve::getWorldSpaceCurve()
{
  PLib::NurbsCurvef world_curve;

  int degree = 1;


  int count = verts.size();

  if ( count == 2 )
    degree = 1;

  if ( count == 3 )
    degree = 2;

  if ( count > 3 )
    degree = 3;

  //points = Vector_HPoint3Df(count);

  Vector_HPoint3Df pts( count );

  tangents = Vector_Point3Df( count );

  PLib::Vector < float > knot( count + degree + 1 );

  for ( int i = 0; i < count; i++ )
  {
    Vector4 p;
    verts[ i ] ->getTransformedPosition( &p );
    pts[ i ] = PLib::HPoint3Df( p.x, p.y, p.z, 1 );
  }

  for ( int i = 0;i < degree + 1;++i )
    knot[ i ] = 0 ;

  for ( int i = degree + 1;i < pts.n();++i )
    knot[ i ] = float( i - degree ) / float( pts.n() - degree ) ;

  for ( int i = pts.n();i < knot.n();++i )
    knot[ i ] = 1.0 ;

  // PLib::NurbsCurvef c = PLib::NurbsCurvef(points, knot, 3);
  world_curve = PLib::NurbsCurvef( pts, knot, degree );

  return world_curve;

}

void NurbsCurve::copyFrom( Entity *rhs )
{
  *this = * static_cast < NurbsCurve * > ( rhs );
}

NurbsCurve & NurbsCurve::operator=( NurbsCurve &rhs )
{

  Spline::operator=( rhs );

  create_pts = rhs.create_pts;
  create_type = rhs.create_type;

  if ( create_pts.size() > 1 )
  {
    curve = rhs.curve;
    setUpControlPoints();
  }


  return *this;

}

Entity * NurbsCurve::clone()
{
  cerr << "Cloning" << endl;
  NurbsCurve *c = new NurbsCurve();
  *c = *this;
  return c;
}

vector < float > NurbsCurve::getUKnots()
{
  vector < float > uknots;

  PLib::Vector < float > knts;
  knts = curve.knot();

  for ( int i = 0; i < knts.n(); i++ )
  {
    uknots.push_back( knts[ i ] );
  }

  return uknots;

}

void NurbsCurve::createCurve( vector < Vector4 > &p, vector < float > &knots, int degree )
{
  //hack to getaround the create_pts...
  Vector4 v( 1, 1, 1, 1 );

  create_pts.push_back( v );
  create_pts.push_back( v );


  Vector_HPoint3Df pts( p.size() );

  PLib::Vector < float > knot( knots.size() );

  for ( int i = 0; i < ( int ) p.size(); i++ )
    pts[ i ] = PLib::HPoint3Df( p[ i ].x, p[ i ].y, p[ i ].z, 1 );


  for ( int i = 0; i < knot.n(); i++ )
    knot[ i ] = knots[ i ];



  curve = PLib::NurbsCurvef( pts, knot, degree );

  //get the control points and add them to this nurbs list.
  setUpControlPoints();

}

void NurbsCurve::createCurve( vector < float > &knots, int degree )
{
  //hack to getaround the create_pts...
  Vector4 v( 1, 1, 1, 1 );

  create_pts.push_back( v );
  create_pts.push_back( v );


  Vector_HPoint3Df pts( numVerts() );
  Vector4 pt;

  PLib::Vector < float > knot( knots.size() );

  for ( int i = 0; i < ( int ) numVerts(); i++ )
  {
    verts[i] -> getPosition( &pt );
    pts[ i ] = PLib::HPoint3Df( pt.x, pt.y, pt.z, 1 );
  }


  for ( int i = 0; i < knot.n(); i++ )
    knot[ i ] = knots[ i ];


  for ( int i = 0; i < ( int ) numVerts(); i++ )
  {
    delete verts[i];
  }


  verts.clear();

  curve = PLib::NurbsCurvef( pts, knot, degree );

  //get the control points and add them to this nurbs list.
  setUpControlPoints();

}





/**Revolves a line along another spline by creating new lines, and skinning them.
  */

Entity * NurbsCurve::revolve( Spline &s, int degrees, int segments )
{
  //find the vector corresponding to the first segment in the spline.
  VertexList * vlist = s.getVerts();

  Vector4 axis;
  axis = ( *vlist ) [ 1 ] ->getPosition() - ( *vlist ) [ 0 ] ->getPosition();
  axis.normalize();

  /*special case for 360 degrees...use the nurbs++ function
   *to guarantee roundness.
   */
  /*
  if(degrees == 360){
   Vector4 c;
   
   NurbsSurface *sf = new NurbsSurface();
   sf->revolve(*this, axis, c, 360);
   return sf; 
}
  */
  vector < Spline * > slist;

  slist.reserve( segments );


  //duplicate the line x segments
  for ( int i = 0; i < segments; i++ )
  {
    NurbsCurve *c = new NurbsCurve();
    *c = *this;
    slist.push_back( c );
  }

  //rotate the line segments;
  float segment_theta = ( float ) degrees / segments;

  for ( int i = 0; i < segments; i++ )
  {
    slist[ i ] ->rotate( segment_theta * i, axis.x, axis.y, axis.z, 0, 0, 0 );
  }

  if ( degrees == 360 )
  {
    slist.push_back( slist[ 0 ] );
  }


  //skin the
  PolySkin ps;

  NurbsSurface *sf = ps.getSurface( slist );

  return sf;


}


/**Sweeps a line along another spline by creating new Lines, and
  *skinning them.
  */
Entity * NurbsCurve::sweep( Spline &s )
{
  vector < Spline * > slist;

  VertexList *vlist = s.getVerts();

  slist.reserve( vlist->size() );
  int n = vlist->size();

  float u_seg = ( float ) 1 / n;


  NurbsCurve *c;

  //align the first line with the vector from the 1st to second
  c = getProfile( ( NurbsCurve * ) & s, 0 );
  slist.push_back( c );


  for ( int i = 1; i < ( int ) n - 1; i++ )
  {
    c = getProfile( ( NurbsCurve * ) & s, ( float ) i * u_seg );
    slist.push_back( c );
  }

  //align the last line with the vector from last to second to last
  c = getProfile( ( NurbsCurve * ) & s, 1 );

  slist.push_back( c );

  PolySkin ps;

  NurbsSurface *sf = ps.getSurface( slist );

  return sf;

}

/**Returns a profile curve along the path 'c' at the point 'u'.
  * 'u' is a value between 0 and 1.
  */
NurbsCurve * NurbsCurve::getProfile( NurbsCurve *c, float u )
{
  NurbsCurve * nc = new NurbsCurve();
  *nc = *this;

  nc->center();

  Vector4 crossResult;
  Vector4 tangent;


  //if line has less than 2 points, use Y as base normalVector, otherwise
  //use the first three points to get a normal vector.
  Vector4 normalVector( 0, 1, 0, 1 );

  VertexList *vlist = nc->getVerts();


  if ( vlist->size() > 2 )
  {
    Vector4 l, r;
    Vector3 norm;

    l = ( *vlist ) [ 0 ] ->getPosition() - ( *vlist ) [ 1 ] ->getPosition();
    r = ( *vlist ) [ 2 ] ->getPosition() - ( *vlist ) [ 1 ] ->getPosition();

    l.normalize();
    r.normalize();

    norm = l.ToVector3() * r.ToVector3();
    normalVector.assign( norm.x, norm.y, norm.z, 1 );
    normalVector.normalize();

  }

  PLib::Point3Df tmp = c->getCurve().derive3D( u, 1 );

  tangent.assign( tmp.x(), tmp.y(), tmp.z(), 1 );
  tangent.normalize();


  float cosAngle = tangent.dot3d( normalVector );

  if ( cosAngle < 0.9999 )
  {
    //use the cross product to determine the vector to rotate about.
    Vector3 r = tangent.ToVector3() * normalVector.ToVector3();
    crossResult.x = r.x;
    crossResult.y = r.y;
    crossResult.z = r.z;
    crossResult.normalize();

    float turnAngle = -acos( ( float ) cosAngle );
    float turnDeg = turnAngle * 180 / M_PI;

    nc->rotate( turnDeg, crossResult.x, crossResult.y, crossResult.z, 0, 0, 0 );
  }

  tmp = c->getCurve().pointAt( u );

  nc->setPosition( tmp.x(), tmp.y(), tmp.z() );

  //ObjectDB::getInstance()->addEntity(nc);

  return nc;

}

CtrlPt * NurbsCurve::createCtrlPt( float x, float y, float z )
{
  CtrlPt * c = new CtrlPt( x, y, z, this );
  addVert( c );

  c -> setIndex( verts.size() - 1 );
  return c;
}












