/***************************************************************************
                       line.cpp  -  description                
                          -------------------                   
 begin                : Sun Apr 25 1999                    
 copyright            : (C) 1999 by Jonathan E. Anderson            
 email                : ande1514@tc.umn.edu               
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "line.h"
#include "vertex.h"
#include "material.h"
#include "texture.h"
#include "texturematerial.h"
#include "mesh.h"
#include "polyskin.h"
#include "poly.h"
#include "lineparser.h"

#include <GL/glut.h>
#include "../objectdb.h"

int Line::TYPE = Typed::getUID();
int LineParser::TYPE = Parser::setParser( Line::TYPE, "line", new LineParser() );

Line::Line( std::vector < Vector4 > &_verts,
            Entity *_parent ) : Spline( _verts, _parent )
{
  addType( TYPE );

  //center();

}

Line::Line( Entity *_parent ) : Spline( _parent )
{
  addType( TYPE );
}

Line::Line( Vector4 &p, Entity*parent ) : Spline( parent )
{

  addType( TYPE );
  createVertex( p );

}

Line::~Line()
{}

void Line::reverse()
{
  Spline::reverse();

}

void Line::breakSpline()
{
  if ( closed )
  {
    verts.erase( verts.end(), verts.end() );
  }

  closed = false;
}

void Line::closeSpline()
{
  if ( !closed )
  {
    verts.push_back( verts[ 0 ] );
  }

  closed = true;
}

int Line::draw( int d_options )
{


  if ( numVerts() == 1 )
  {
    Vector4 p = verts[ 0 ] ->getPosition();
    glTranslatef( p.x, p.y, p.z );
    glutSolidSphere( .25, 4, 4 );
    return 0;
  }

  glPushAttrib( GL_LIGHTING_BIT );
  glDisable( GL_LIGHTING );

  glBegin( GL_LINE_STRIP );

  for ( int i = 0; i < ( int ) verts.size(); i++ )
    getVertex( i ) ->draw( VERTEX_NORMAL | VERTEX_COORD );

  glEnd();


  glPopAttrib();


  if ( SelectMode::is( Vertex::TYPE ) )
  {     //need to draw verts too?

    drawVerts();
  }

  return 0;

}

void Line::copyFrom( Entity *rhs )
{
  *this = * static_cast < Line * > ( rhs );
}


Line & Line::operator=( Line &rhs )
{

  Spline::operator=( rhs );

  return *this;

}

Line & Line::operator=( Poly&rhs )
{

  Spline::operator=( rhs );

  return *this;

}


Entity * Line::clone()
{
  Line * l = new Line();
  *l = *this;
  return l;
}

/** Extends the line by adding a point
  */
void Line::extend( Vector4 &p )
{
  createVertex( p );
}


/**Sweeps a line along another spline by creating new Lines, and
  *skinning them.
  */
Entity * Line::sweep( Spline &s )
{
  std::vector < Spline * > slist;

  VertexList *vlist = s.getVerts();

  slist.reserve( vlist->size() );
  int n = vlist->size();


  //save the current position of the line, if it's away from
  Vector4 old_pos = getPosition();

  Line *l;
  Vector4 curPos;
  Vector4 transPos;
  Vector4 tarPos;
  Vector4 prevPos;

  //align the first line with the vector from the 1st to second
  ( *vlist ) [ 0 ] ->getTransformedPosition( &curPos );
  ( *vlist ) [ 1 ] ->getTransformedPosition( &tarPos );
  l = getProfile( curPos, curPos, tarPos );
  transPos = curPos - getPosition();
  l->move( transPos.x, transPos.y, transPos.z );
  slist.push_back( l );


  for ( int i = 1; i < ( int ) n - 1; i++ )
  {
    ( *vlist ) [ i ] ->getTransformedPosition( &curPos );
    ( *vlist ) [ i + 1 ] ->getTransformedPosition( &tarPos );
    ( *vlist ) [ i - 1 ] ->getTransformedPosition( &prevPos );
    l = getProfile( prevPos, curPos, tarPos );
    transPos = curPos - getPosition();
    l->move( transPos.x, transPos.y, transPos.z );
    slist.push_back( l );
  }

  //align the last line with the vector from last to second to last
  ( *vlist ) [ n - 2 ] ->getTransformedPosition( &curPos );

  ( *vlist ) [ n - 1 ] ->getTransformedPosition( &tarPos );

  l = getProfile( curPos, curPos, tarPos );

  transPos = tarPos - getPosition();  //target is the end vector.

  l->move( transPos.x, transPos.y, transPos.z );

  slist.push_back( l );

  PolySkin ps;

  ps.setUseQuads( true );

  Mesh *m = ps.getMesh( slist );

  return m;

}

/**Revolve the line around the first segment of the given spline.
  *Do this by duplicating the splines, rotating them, and then skinning them.
  */
Entity * Line::revolve( Spline &s, int degrees, int segments )
{

  //find the vector corresponding to the first segment in the spline.
  VertexList * vlist = s.getVerts();

  Vector4 axis;
  Vector4 sPos;
  axis = ( *vlist ) [ 1 ] ->getPosition() - ( *vlist ) [ 0 ] ->getPosition();
  axis.normalize();

  sPos = ( *vlist ) [ 0 ] ->getPosition();



  std::vector < Spline * > slist;

  slist.reserve( segments );


  //duplicate the line x segments
  for ( int i = 0; i < segments; i++ )
  {
    Line *l = new Line();
    *l = *this;
    slist.push_back( l );
  }


  //rotate the line segments;
  float segment_theta = ( float ) degrees / segments;

  for ( int i = 0; i < segments; i++ )
  {
    slist[ i ] ->rotate( segment_theta * i, axis.x, axis.y, axis.z, 0, 0, 0 );
    slist[ i ] ->move( sPos.x, sPos.y, sPos.z );
  }

  //close it for 360 degree revoles...
  if ( degrees == 360 )
    slist.push_back( slist[ 0 ] );


  //skin the
  PolySkin ps;

  ps.setUseQuads( true );

  Mesh *m = ps.getMesh( slist );

  return m;


}


Line * Line::getProfile( Vector4 &prevPos, Vector4 &curPos, Vector4 &tarPos )
{
  Line * l = new Line();
  *l = *this;

  l->center();

  Vector4 crossResult;
  Vector4 tangent;


  //if line has less than 2 points, use Y as base normalVector, otherwise
  //use the first three points to get a normal vector.
  Vector4 normalVector( 0, 1, 0, 1 );

  VertexList *vlist = getVerts();


  if ( vlist->size() > 2 )
  {
    Vector4 l, r;
    Vector3 norm;

    l = ( *vlist ) [ 0 ] ->getPosition() - ( *vlist ) [ 1 ] ->getPosition();
    r = ( *vlist ) [ 2 ] ->getPosition() - ( *vlist ) [ 1 ] ->getPosition();

    l.normalize();
    r.normalize();

    norm = l.ToVector3() * r.ToVector3();
    normalVector.assign( norm.x, norm.y, norm.z, 1 );

  }


  // cerr<<"Normal ";
  //  normalVector.Dump();

  if ( prevPos == curPos )
  {
    tangent = curPos - tarPos;
    tangent.normalize();
  }

  else
  {
    Vector4 pv, nv;
    nv = curPos - tarPos;
    pv = prevPos - curPos;
    nv.normalize();
    pv.normalize();
    tangent = nv + pv;
    tangent /= 2;
    tangent.normalize();
  }


  float cosAngle = tangent.dot3d( normalVector );

  if ( cosAngle < 0.9999 )
  {
    //use the cross product to determine the vector to rotate about.
    Vector3 r = tangent.ToVector3() * normalVector.ToVector3();
    crossResult.x = r.x;
    crossResult.y = r.y;
    crossResult.z = r.z;
    crossResult.normalize();

    float turnAngle = -acos( ( float ) cosAngle );
    float turnDeg = turnAngle * 180 / M_PI;
    //cerr<<"About:";
    //crossResult.Dump();

    l->rotate( turnDeg, crossResult.x, crossResult.y, crossResult.z, 0, 0, 0 );
  }

  //ObjectDB::getInstance()->addEntity(l);

  return l;

}











