/***************************************************************************
                       entity.cpp  -  description                
                          -------------------                   
 begin                : Sun Apr 25 1999                    
 copyright            : (C) 1999 by Jonathan E. Anderson            
 email                : ande1514@tc.umn.edu               
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "entity.h"
#include "controller.h"
#include "boundingbox.h"
#include <assert.h>
#include <stdio.h>
#include <keyframe.h>

int Typed::m_uid = 0;
int IDed::ID = 0;
int Entity::TYPE = Typed::getUID();

Entity::Entity( float x, float y, float z, Entity *_parent ) : Animatable(), Selectable(), Renderable(), m_data(), pos(), quat()
{
  init( x, y, z, _parent );
}

Entity::Entity( Vector4 &p, Entity *_parent ) : Animatable(), Selectable(), Renderable(), m_data(), pos(), quat()
{
  init( p.x, p.y, p.z, _parent );
}

Entity::Entity( Entity *_parent ) : Animatable(), Selectable(), Renderable(), m_data(), pos(), quat()
{
  init( 0, 0, 0, _parent );
}


Entity::Entity() : Animatable(), Selectable(), Renderable(), m_data(), pos(), quat()
{
  init( 0, 0, 0, 0 );
}

void Entity::init( float x, float y, float z, Entity *_parent )
{

  pos.assign( x, y, z, 1 );

  selected = false;

  char buffer[ 16 ];

  parent = _parent;
  dirty = true;
  visible = true;
  m_vis_group = -1;

  dirty_box = true;
  drawable = true;

  sprintf( buffer, "Entity_%d", m_id );
  name = buffer;
  notes = "none";

}


Entity::~Entity()
{

  IDataMap::iterator it = m_data.begin();

  while ( it != m_data.end() )
  {
    IData * d = it -> second;
    delete d;
    ++it;
  }

  m_data.clear();

}


SubObject * Entity::getSubObject( int mode, int index )
{
  return 0;
}

void Entity::move( float x, float y, float z )
{
  pos.x += x;
  pos.y += y;
  pos.z += z;

}

void Entity::rotate( float amount, float x, float y, float z, float ox, float oy, float oz )
{
  Vector3 v( x, y, z );
  float rad_amount = amount * M_PI / 180;
  Quat qn = Quat( v, rad_amount );
  qn.Normalize();
  quat = quat * qn;

  quat.Normalize();

}

/**
 * Scales an  relative to a world point
 *
 */
void Entity::scale( float x, float y, float z, float ox, float oy, float oz )
{

  Vector4 pt;
  Vector4 st;

  getTransformedPosition( &pt );

  st = pt;

  pt.x = ( ( pt.x - ox ) * x ) + ox;
  pt.y = ( ( pt.y - oy ) * y ) + oy;
  pt.z = ( ( pt.z - oz ) * z ) + oz;

  //figure out how much to move by...
  pos.x += pt.x - st.x;
  pos.y += pt.y - st.y;
  pos.z += pt.z - st.z;

}

void Entity::setPosition( float x, float y, float z )
{
  pos.assign( x, y, z, 1 );

}

void Entity::setOrientation( Quat &q )
{
  quat = q;
}

void Entity::setOrientation( float d, float x, float y, float z )
{
  quat = Quat( x, y, z, d );
}

int Entity::draw( int d_options )
{
  return 0;
}

int Entity::render( int d_options )
{
  return 0;
}



void Entity::saveKeyframe( Keyframe *k )
{

  if ( isAnimatable() )
  {
    Keyframe nk( k->getTime() );
    //Keyframe will delete all it's channels.
    VectorChannel *pc = new VectorChannel( pos );
    QuatChannel *qc = new QuatChannel( quat );

    nk.addChannel( POSITION_CHANNEL, pc );
    nk.addChannel( ORIENTATION_CHANNEL, qc );

    *k = nk;
  }

}

void Entity::loadKeyframe( Keyframe *k )
{

  if ( isAnimatable() )
  {
    pos = dynamic_cast < VectorChannel* > ( k -> getChannel( POSITION_CHANNEL ) ) -> value;
    quat = dynamic_cast < QuatChannel* > ( k -> getChannel( ORIENTATION_CHANNEL ) ) -> value;
  }

}

void Entity::getPosition( Vector4 * v )
{
  assert( v );

  *v = pos;

}

void Entity::getOrientation( Quat * q )
{
  assert( q );
  *q = quat;
}

void Entity::getParentMatrix( Matrix44 *m )
{
  m->Unit();

  if ( parent != 0 )
  {
    parent->getCompleteMatrix( m );
  }
}

void Entity::getCompleteMatrix( Matrix44 *m )
{
  //calculate the complete matrix...
  m->Unit();

  quat.InsertInMatrix( m->m );
  m->SetPos( pos );


  if ( parent != 0 )
  {
    Matrix44 p;
    parent->getCompleteMatrix( &p );
    *m = p * *m;  //*p *= *m;
  }

}

void Entity::getLocalMatrix( Matrix44 *m )
{
  //calculate the local matrix...
  m->Unit();
  quat.InsertInMatrix( m->m );
  m->SetPos( pos );

}


void Entity::setParent( Entity *_parent )
{
  parent = _parent;
}

void Entity::getBoundingMin( Vector4 *v )
{

  /* Matrix44 inv;
   parent->getCompleteMatrix(&inv);
   inv.Invert();
   
   getTransformedPosition(v);
   
   *v = inv * *v; 
   */
  *v = pos;
}



void Entity::getBoundingMax( Vector4 *v )
{

  /* Matrix44 inv;
   parent->getCompleteMatrix(&inv);
   inv.Invert();
   
   getTransformedPosition(v);
   
   *v = inv * *v;
   */
  *v = pos;
}

void Entity::getTransformedPosition( Vector4 *v )
{
  Matrix44 m;
  v->assign( 0, 0, 0, 1 );
  getCompleteMatrix( &m );
  *v = m * *v;
}


Entity & Entity::operator=( Entity &rhs )
{

  // controller = rhs.controller;

  dirty = rhs.dirty;
  name = rhs.name;
  notes = rhs.notes;
  quat = rhs.quat;

  visible = rhs.visible;
  m_vis_group = rhs.m_vis_group;


  for ( int i = 0; i < num_types; i++ )
  {
    rhs.type_list[ i ] = type_list[ i ];
  }

  rhs.num_types = num_types;

  pos = rhs.pos;

  return *this;
}

void Entity::copyFrom( Entity *rhs )
{
  *this = *rhs;
}

Entity * Entity::clone()
{
  return new Entity( pos.x, pos.y, pos.z, getParent() );
}

string Entity::getBaseDescription()
{
  /*
   int type = getBaseType();
   
   if(type == ENTITY_SPLINE)
    return "spline";
   if(type == ENTITY_MESH)
    return "mesh";
   if(type == ENTITY_SURFACE)
    return "surface";
   if(type == ENTITY_MISC)
    return "misc";
   if(type == ENTITY_BONE)
    return "bone";
    
   return "unknown";
  */ 
  return "entity";
}

IData * Entity::getData( string tag )
{
  IDataMap::iterator pos = m_data.find( tag );

  if ( pos != m_data.end() )
    return pos->second;

  return 0;
}

void Entity::putData( string tag, IData *d )
{
  m_data[ tag ] = d;
}

bool Entity::isVisible()
{
  return visible;
}


Transformable::Transformable()
{}

inline void Transformable::setPosition( Vector4 &p )
{
  setPosition( p.x, p.y, p.z );
};

inline void Transformable::scale( Vector4 &amt, Vector4 &pt )
{
  scale( amt.x, amt.y, amt.z, pt.x, pt.y, pt.z );
};

inline void Transformable::move( Vector4 &p )
{
  move( p.x, p.y, p.z );
};

inline void Transformable::rotate( Vector4 &amt, Vector4 &pt )
{
  rotate( amt.w, amt.x, amt.y, amt.z, pt.x, pt.y, pt.z );
};

int Transformable::drawBoundingBox()
{
  BoundingBox bb;
  Vector4 slop( .05, .05, .05, 0 );

  Vector4 bmin;
  Vector4 bmax;
  getBoundingMax( &bmax );
  getBoundingMin( &bmin );

  bmax += slop;
  bmin -= slop;

  bb.setBounds( bmin, bmax );
  bb.draw();

  return 0;

}

bool Typed::isA( int p_type )
{
  for ( int i = 0; i < num_types; i++ )
  {
    if ( type_list[ i ] == p_type )
      return true;
  }

  return false;
}

Typed::Typed()
{
  type_list = new int[ 8 ];
  num_types = 0;
};

Typed::~Typed()
{
  delete type_list;
};

int Typed::getType() const
{
  return type_list[ num_types -1 ];
};

void Typed::setType( int _type )
{
  type_list[ num_types - 1 ] = _type;
};

int Typed::getBaseType()
{
  return type_list[ 1 ];
};

int Typed::getRootType()
{
  return type_list[ 0 ];
};

void Typed::addType( int b )
{
  type_list[ num_types ] = b;
  num_types++;
};

int Typed::getUID()
{
  return m_uid++;
};

Animatable::Animatable()
{
  m_controller = 0;
  m_animatable = false;
}

Animatable::~Animatable()
{
  if ( m_controller != 0 )
    delete m_controller;
}

bool Animatable::isAnimatable()
{
  return m_animatable;
}

void Animatable::setAnimatable( bool b )
{
  m_animatable = true;

  if ( !b )
  {
    delete m_controller;
  }

  if ( b && m_controller == 0 )
  {
    m_controller = new Controller();
    m_controller -> setAnimatable( this );
    //save the default keyframe at time 1.
    m_controller -> saveTime( 1 );
  }

}

Controller * Animatable::getController()
{
  return m_controller;
}

void Animatable::setController( Controller * c )
{
  m_controller = c;
}


Selectable::Selectable() : Typed(), IDed(), Transformable()
{
  selected = false;
}

bool Selectable::isSelected()
{
  return selected;
};

void Selectable::setSelected( bool flag )
{
  selected = flag;
};

inline void SelectMode::clear()
{
  for ( int i = 0; i < m_size; i++ )
  {
    m_mode[ i ] = -1;
  }

  m_size = 0;

}

bool SelectMode::is( int m )
{
  for ( int i = 0; i < m_size; i++ )
  {
    if ( m == m_mode[ i ] )
      return true;
  }

  return false;
}

inline void SelectMode::add( int m )
{

  assert( m_size < 7 );
  m_mode[ m_size ] = m;
  m_size++;

}

void SelectMode::set
( int m )
{

  clear();
  add( m );
}

int SelectMode::m_mode[ 8 ] = { -1, -1, -1, -1, -1, -1, -1, 1};
int SelectMode::m_size = 0;




























