/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <string.h>

#include "connman-manager.h"
#include "connman-service.h"

#define HIDDEN_SSID "hiddenssid"

static GMainLoop *loop;
static ConnmanManager *connman;

static const gchar *connect_path;

static enum {
  TEST_LIST_SERVICES,
  TEST_CONNECT_SERVICE,
  TEST_CONNECT,
} test;

static void usage(const gchar *name)
{
  g_print("%s:\n", name);
  g_print("connect-service\n");
  g_print("list-services\n");
  g_print("connect <path>\n");
}

static void connect_hidden_ssid_cb(GObject *object, GAsyncResult *res,
				   gpointer user_data)
{
  GError *error = NULL;

  connman_manager_connect_service_finish(connman, res, &error);

  if (error != NULL) {
    g_print("connect hidden ssid failed: %s\n", error->message);
    g_error_free(error);
    return;
  }

  g_print("hidden ssid connected\n");
}

static void disconnect_cb(GObject *object, GAsyncResult *res,
			  gpointer user_data)
{
  ConnmanService *service = CONNMAN_SERVICE(object);
  GError *error = NULL;

  connman_service_disconnect_finish(service, res, &error);

  if (error != NULL) {
    g_print("disconnect failed: %s\n", error->message);
    g_error_free(error);
    return;
  }

  g_print("service %s disconnected\n", connman_service_get_path(service));
}

static void connect_cb(GObject *object, GAsyncResult *res,
		       gpointer user_data)
{
  ConnmanService *service = CONNMAN_SERVICE(object);
  GError *error = NULL;

  connman_service_connect_finish(service, res, &error);

  if (error != NULL) {
    g_print("connect failed: %s\n", error->message);
    g_error_free(error);
    return;
  }

  g_print("service %s connected\n", connman_service_get_path(service));

  connman_service_disconnect(service, disconnect_cb, NULL);
}

static void state_updated(ConnmanService *service, GParamSpec *pspec,
			  gpointer user_data)
{
  g_print("service %s state %d\n", connman_service_get_path(service),
	  connman_service_get_state(service));
}

static void error_updated(ConnmanService *service, GParamSpec *pspec,
			  gpointer user_data)
{
  g_print("service %s error %s\n", connman_service_get_path(service),
	  connman_service_get_error(service));
}

static void name_updated(ConnmanService *service, GParamSpec *pspec,
			 gpointer user_data)
{
  g_print("service %s name %s\n", connman_service_get_path(service),
	  connman_service_get_name(service));
}

static void type_updated(ConnmanService *service, GParamSpec *pspec,
			  gpointer user_data)
{
  g_print("service %s type %d\n", connman_service_get_path(service),
	  connman_service_get_service_type(service));
}

static void security_updated(ConnmanService *service, GParamSpec *pspec,
			     gpointer user_data)
{
  g_print("service %s security %d\n", connman_service_get_path(service),
	  connman_service_get_security(service));
}

static void strength_updated(ConnmanService *service, GParamSpec *pspec,
			     gpointer user_data)
{
  g_print("service %s strength %d\n", connman_service_get_path(service),
	  connman_service_get_strength(service));
}

static void setup_required_updated(ConnmanService *service, GParamSpec *pspec,
				   gpointer user_data)
{
  g_print("service %s setup-required %d\n", connman_service_get_path(service),
	  connman_service_get_setup_required(service));
}

static void new_service(ConnmanService *service)
{
  g_print("new service %s\n", connman_service_get_path(service));
  g_print("  state %d\n", connman_service_get_state(service));
  g_print("  error %s\n", connman_service_get_error(service));
  g_print("  name %s\n", connman_service_get_name(service));
  g_print("  type %d\n", connman_service_get_service_type(service));
  g_print("  security %d\n", connman_service_get_security(service));
  g_print("  strength %d\n", connman_service_get_strength(service));
  g_print("  setup_required %d\n", connman_service_get_setup_required(service));

  g_signal_connect(service, "notify::state", G_CALLBACK(state_updated),
		   NULL);
  g_signal_connect(service, "notify::error", G_CALLBACK(error_updated),
		   NULL);
  g_signal_connect(service, "notify::name", G_CALLBACK(name_updated),
		   NULL);
  g_signal_connect(service, "notify::type", G_CALLBACK(type_updated),
		   NULL);
  g_signal_connect(service, "notify::security", G_CALLBACK(security_updated),
		   NULL);
  g_signal_connect(service, "notify::strength", G_CALLBACK(strength_updated),
		   NULL);
  g_signal_connect(service, "notify::setup-required",
		   G_CALLBACK(setup_required_updated), NULL);

  if (test == TEST_CONNECT &&
      g_strcmp0(connman_service_get_name(service), connect_path) == 0)
    connman_service_connect(service, connect_cb, NULL);
}

static void connected_cb(ConnmanManager *proxy, GParamSpec *pspec,
			 gpointer user_data)
{
  ConnmanService **services, *service;
  gboolean connected;
  guint count;

  g_object_get(proxy, "connected", &connected, NULL);

  g_print("%s\n", connected ? "connected" : "disconnected");

  if (!connected)
    return;

 if (test == TEST_CONNECT_SERVICE) {
    connman_manager_connect_service(connman, CONNMAN_SERVICE_TYPE_WIFI,
				    CONNMAN_SERVICE_MODE_MANAGED,
				    CONNMAN_SERVICE_SECURITY_NONE,
				    (guint8 *) HIDDEN_SSID,
				    strlen(HIDDEN_SSID),
				    NULL,
				    connect_hidden_ssid_cb,
				    NULL);
    return;
  }

  services = connman_manager_get_services(connman);

  if (services == NULL) {
    g_print("no services\n");
    return;
  }

  count = 0;

  while (*services != NULL) {
    new_service(*services);
    services++;
    count++;
  }

  g_print("%d services received\n", count);

  if (connman_manager_get_connected(connman)) {
    service = connman_manager_get_default_service(connman);
    g_print("connected, default service %s\n",
	    service ? connman_service_get_path(service) : "none");
  } else {
    g_print("not connected\n");
  }
}

static void default_service_notify(ConnmanManager *proxy, GParamSpec *pspec,
				   gpointer user_data)
{
  ConnmanService *service = connman_manager_get_default_service(connman);

  g_print("new default service %s\n",
	  service ? connman_service_get_path(service) : "none");
}

static void service_added(ConnmanManager *proxy, ConnmanService *service,
			  gpointer user_data)
{
  g_return_if_fail(CONNMAN_IS_MANAGER(proxy));
  g_return_if_fail(CONNMAN_IS_SERVICE(service));

  new_service(service);
}

static void service_removed(ConnmanManager *proxy, const gchar *path,
			    gpointer user_data)
{
  g_print("service-removed %s\n", path);
}


int main(int argc, char **argv)
{
  g_type_init();

  if (argc < 2) {
    usage(argv[0]);
    return 1;
  }

  loop = g_main_loop_new(NULL, FALSE);

  connman = connman_manager_new();

  if (g_strcmp0(argv[1], "list-services") == 0) {
    test = TEST_LIST_SERVICES;
    if (argc != 2) {
      usage(argv[0]);
      return 1;
    }
  }
  else if (g_strcmp0(argv[1], "connect-service") == 0) {
    test = TEST_CONNECT_SERVICE;
    if (argc != 2) {
      usage(argv[0]);
      return 1;
    }
  }
  else if (g_strcmp0(argv[1], "connect") == 0) {
    test = TEST_CONNECT;
    if (argc != 3) {
      usage(argv[0]);
      return 1;
    }
    connect_path = argv[2];
  }

  g_signal_connect(connman, "notify::connected", G_CALLBACK(connected_cb),
		   NULL);

  g_signal_connect(connman, "notify::default-service",
		   G_CALLBACK(default_service_notify),
		   NULL);

  if (test == TEST_LIST_SERVICES) {
    g_signal_connect(connman, "service-added", G_CALLBACK(service_added),
		     NULL);
    g_signal_connect(connman, "service-removed", G_CALLBACK(service_removed),
		     NULL);
  }

  g_main_loop_run(loop);

  return 0;
}
