/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef _CONNMAN_SERVICE_H_
#define _CONNMAN_SERVICE_H_

#include "connman-ipv4.h"
#include "connman-ipv6.h"

#include <glib-object.h>
#include <gio/gio.h>

G_BEGIN_DECLS

#define CONNMAN_TYPE_SERVICE connman_service_get_type()

#define CONNMAN_SERVICE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), CONNMAN_TYPE_SERVICE, \
			      ConnmanService))

#define CONNMAN_SERVICE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), CONNMAN_TYPE_SERVICE, \
			   ConnmanServiceClass))

#define CONNMAN_IS_SERVICE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), CONNMAN_TYPE_SERVICE))

#define CONNMAN_IS_SERVICE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), CONNMAN_TYPE_SERVICE))

#define CONNMAN_SERVICE_GET_CLASS(obj) \
  (G_TYPE_INSTANCE_GET_CLASS((obj), CONNMAN_TYPE_SERVICE, \
			     ConnmanServiceClass))

typedef struct {
  GObject parent;
} ConnmanService;

typedef struct {
  GObjectClass parent_class;
} ConnmanServiceClass;

typedef enum {
  CONNMAN_SERVICE_STATE_IDLE,
  CONNMAN_SERVICE_STATE_FAILURE,
  CONNMAN_SERVICE_STATE_ASSOCIATION,
  CONNMAN_SERVICE_STATE_CONFIGURATION,
  CONNMAN_SERVICE_STATE_READY,
  CONNMAN_SERVICE_STATE_LOGIN,
  CONNMAN_SERVICE_STATE_ONLINE,
  CONNMAN_SERVICE_STATE_DISCONNECT,
} ConnmanServiceState;

typedef enum {
  CONNMAN_SERVICE_SECURITY_UNKNOWN,
  CONNMAN_SERVICE_SECURITY_NONE,
  CONNMAN_SERVICE_SECURITY_WEP,
  CONNMAN_SERVICE_SECURITY_PSK,
  CONNMAN_SERVICE_SECURITY_IEEE8021X,
} ConnmanServiceSecurity;

typedef enum {
  CONNMAN_SERVICE_TYPE_ETHERNET,
  CONNMAN_SERVICE_TYPE_WIFI,
  CONNMAN_SERVICE_TYPE_BLUETOOTH,
  CONNMAN_SERVICE_TYPE_CELLULAR,
} ConnmanServiceType;

typedef enum {
  CONNMAN_SERVICE_MODE_MANAGED,
  CONNMAN_SERVICE_MODE_ADHOC,
  CONNMAN_SERVICE_MODE_GPRS,
  CONNMAN_SERVICE_MODE_EDGE,
  CONNMAN_SERVICE_MODE_UMTS,
} ConnmanServiceMode;

#define CONNECT_TIMEOUT 120000

GType connman_service_get_type(void);

/* property accessors */
ConnmanServiceState connman_service_get_state(ConnmanService *self);
const gchar *connman_service_get_error(ConnmanService *self);
const gchar *connman_service_get_name(ConnmanService *self);
ConnmanServiceType connman_service_get_service_type(ConnmanService *self);
ConnmanServiceSecurity connman_service_get_security(ConnmanService *self);
gboolean connman_service_get_login_required(ConnmanService *self);
const gchar *connman_service_get_passphrase(ConnmanService *self);
void connman_service_set_passphrase(ConnmanService *self,
				    const gchar *passphrase);
gboolean connman_service_get_passphrase_required(ConnmanService *self);
guint8 connman_service_get_strength(ConnmanService *self);
gboolean connman_service_get_favorite(ConnmanService *self);
gboolean connman_service_get_immutable(ConnmanService *self);
gboolean connman_service_get_autoconnect(ConnmanService *self);
void connman_service_set_autoconnect(ConnmanService *self,
				     gboolean autoconnect);
gboolean connman_service_get_setup_required(ConnmanService *self);
const gchar *connman_service_get_apn(ConnmanService *self);
void connman_service_set_apn(ConnmanService *self, const gchar *apn);
const gchar *connman_service_get_mcc(ConnmanService *self);
const gchar *connman_service_get_mnc(ConnmanService *self);
gboolean connman_service_get_roaming(ConnmanService *self);
gchar **connman_service_get_nameservers(ConnmanService *self);
gchar **connman_service_get_nameservers_configuration(ConnmanService *self);
void connman_service_set_nameservers_configuration(ConnmanService *self,
						   gchar **nameservers);
gchar **connman_service_get_domains(ConnmanService *self);
gchar **connman_service_get_domains_configuration(ConnmanService *self);
void connman_service_set_domains_configuration(ConnmanService *self,
					       gchar **domains);
ConnmanIPv4 *connman_service_get_ipv4(ConnmanService *self);
ConnmanIPv4 *connman_service_get_ipv4_configuration(ConnmanService *self);
void connman_service_set_ipv4_configuration(ConnmanService *self,
					    ConnmanIPv4 *ipv4);
ConnmanIPv6 *connman_service_get_ipv6(ConnmanService *self);
ConnmanIPv6 *connman_service_get_ipv6_configuration(ConnmanService *self);
void connman_service_set_ipv6_configuration(ConnmanService *self,
					    ConnmanIPv6 *ipv6);

/* methods */
void connman_service_connect(ConnmanService *self,
			     GAsyncReadyCallback callback,
			     gpointer user_data);
void connman_service_connect_finish(ConnmanService *self,
				    GAsyncResult *res,
				    GError **error);

void connman_service_disconnect(ConnmanService *self,
				GAsyncReadyCallback callback,
				gpointer user_data);
void connman_service_disconnect_finish(ConnmanService *self,
				       GAsyncResult *res,
				       GError **error);

const gchar *connman_service_get_path(ConnmanService *self);
gboolean connman_service_is_ready(ConnmanService *self);
ConnmanService *connman_service_new(const gchar *path);

/* helper functions */
ConnmanServiceState connman_service_str2state(const gchar *state);
const gchar *connman_service_state2str(ConnmanServiceState state);
ConnmanServiceType connman_service_str2type(const gchar *type);
const gchar *connman_service_type2str(ConnmanServiceType type);
ConnmanServiceSecurity connman_service_str2security(const gchar *security);
const gchar *connman_service_security2str(ConnmanServiceSecurity security);
ConnmanServiceMode connman_service_str2mode(const gchar *mode);
const gchar *connman_service_mode2str(ConnmanServiceMode mode);

#endif
