#ifndef IMAPFILTER_H
#define IMAPFILTER_H


#include <stdio.h>
#include <sys/types.h>

#include <lua.h>
#include <lualib.h>

#include "session.h"

#ifndef NO_SSLTLS
#include <openssl/ssl.h>
#endif


/* Fatal error exit codes. */
#define ERROR_SIGNAL		1
#define ERROR_CONFIG		2
#define ERROR_MEMALLOC		3

/* IMAP protocol supported by the server. */
#define PROTOCOL_NONE		0
#define PROTOCOL_IMAP4REV1	1
#define PROTOCOL_IMAP4		2

/* Capabilities of mail server. */
#define CAPABILITY_NONE		0x00
#define CAPABILITY_NAMESPACE	0x01
#define CAPABILITY_CRAMMD5	0x02
#define CAPABILITY_STARTTLS	0x04

/* Response codes. */
#define RESPONSE_NONE		0
#define RESPONSE_OK		1
#define RESPONSE_NO		2
#define RESPONSE_BAD		3
#define RESPONSE_BYE		4
#define RESPONSE_PREAUTH	5
#define RESPONSE_READONLY	6
#define RESPONSE_TRYCREATE	7
#define RESPONSE_CONTINUE	8

/* Initial size of server's response dynamically allocated buffer. */
#define RESPONSE_BUF		4096

/* Initial size of client's request dynamically allocated buffer. */
#define REQUEST_BUF		1024

/* Initial size of mailbox's name dynamically allocated buffer. */
#define NAMESPACE_BUF		128


/* Program's options. */
typedef struct options {
	int debug;		/* Debugging level (0..2). */
	int verbose;		/* Verbose mode. */
	char *log;		/* Log file for error messages. */
	char *config;		/* Configuration file. */
	char *home;		/* User's home directory. */
} options_t;


/*	auth.c		*/
#ifndef NO_CRAMMD5
int auth_cram_md5(session_t * ssn, const char *user, const char *pass);
#endif

/*	cert.c		*/
#ifndef NO_SSLTLS
int get_cert(session_t * ssn);
#endif

/*	core.c		*/
LUALIB_API int luaopen_ifcore(lua_State * L);

/*	file.c		*/
int create_homedir(void);
int exists_file(char *fname);
int exists_dir(char *fname);
int create_file(char *fname, mode_t mode);

/*	imap.c		*/
int imap_continuation(session_t * ssn, const char *cont);
int imap_noop(session_t * ssn);
int imap_capability(session_t * ssn);
int imap_namespace(session_t * ssn);
int imap_logout(session_t * ssn);
#ifndef NO_SSLTLS
int imap_starttls(session_t * ssn);
#endif
int imap_authenticate(session_t * ssn, const char *auth);
int imap_login(session_t * ssn, const char *user, const char *pass);
int imap_subscribe(session_t * ssn, const char *mbox);
int imap_examine(session_t * ssn, const char *mbox);
int imap_select(session_t * ssn, const char *mbox);
int imap_status(session_t * ssn, const char *mbox, const char *items);
int imap_create(session_t * ssn, const char *mbox);
int imap_search(session_t * ssn, int uid, const char *charset, const char *criteria);
int imap_fetch(session_t * ssn, int uid, const char *mesg, const char *items);
int imap_store(session_t * ssn, int uid, const char *mesg, const char *mode, const char *flags);
int imap_copy(session_t * ssn, int uid, const char *mesg, const char *mbox);
int imap_append(session_t * ssn, const char *mbox, const char *flags, const char *date, unsigned int size);
int imap_close(session_t * ssn);
int imap_expunge(session_t * ssn);

/*	log.c		*/
void verbose(const char *info,...);
void debug(const char *debug,...);
void error(const char *errmsg,...);
void fatal(unsigned int errnum, const char *fatal,...);

int open_debug(void);
int close_debug(void);

int open_log(void);
int close_log(void);

/*	lua.c	*/
void start_lua(void);
void stop_lua(void);

int get_option_boolean(const char *opt);
lua_Number get_option_number(const char *opt);
const char *get_option_string(const char *opt);

int get_table_type(const char *key);
lua_Number get_table_number(const char *key);
const char *get_table_string(const char *key);

int set_table_nil(const char *key);
int set_table_boolean(const char *key, int value);
int set_table_number(const char *key, lua_Number value);
int set_table_string(const char *key, const char *value);

/*	memory.c	*/
void *xmalloc(size_t size);
void *xrealloc(void *ptr, size_t size);
void xfree(void *ptr);
char *xstrdup(const char *s);

/*	misc.c		*/
char *xstrcasestr(const char *haystack, const char *needle);
#if 0
char *xstrncpy(char *dest, const char *src, size_t size);
#endif

/*	request.c	*/
int request_login(const char *server, const char *user, const char *pass, unsigned int port, const char *ssl);
int request_logout(const char *server, const char *user);
int request_status(const char *server, const char *user, const char *mbox, unsigned int *exist, unsigned int *recent, unsigned int *unseen);
int request_select(const char *server, const char *user, const char *mbox);
int request_close(const char *server, const char *user);
int request_search(const char *server, const char *user, const char *criteria, char **mesgs);
int request_fetch(const char *server, const char *user, const char *mesg, const char *items, char **fetch);
int request_store(const char *server, const char *user, const char *mesg, const char *mode, const char *flags);
int request_copy(const char *server, const char *user, const char *mesg, const char *mbox);
int request_append(const char *server, const char *user, const char *mbox, const char *mesg, const char *flags, const char *date);

/*	response.c	*/
int response_generic(session_t * ssn, int tag);
int response_greeting(session_t * ssn);
int response_capability(session_t * ssn, int tag);
int response_authenticate(session_t * ssn, int tag, unsigned char **cont);
int response_namespace(session_t * ssn, int tag);
int response_status(session_t * ssn, int tag, int *exist, int *recent, int *unseen);
int response_examine(session_t * ssn, int tag, int *exist, int *recent, int *unseen);
int response_select(session_t * ssn, int tag);
int response_search(session_t * ssn, int tag, char **mesgs);
int response_fetch(session_t * ssn, int tag, char **fetch);
int response_append(session_t * ssn, int tag);
int response_copy(session_t * ssn, int tag);

/*	signal.c	*/
void catch_signals(void);
void release_signals(void);

/*	socket.c	*/
int open_connection(session_t * ssn, const char *server, unsigned int port, const char *sprotocol);
int close_connection(session_t * ssn);
#ifndef NO_SSLTLS
int open_secure_connection(session_t * ssn, const char *protocol);
#if 0
char *get_cipher(session_t * ssn);
#endif
int close_secure_connection(session_t * ssn);
#endif
ssize_t socket_read(session_t * ssn, char *buf, size_t len);
ssize_t socket_write(session_t * ssn, const char *buf);

/*	system.c	*/
LUALIB_API int luaopen_ifsys(lua_State * L);


#endif				/* IMAPFILTER_H */
