#include "Histogram1DDataBlock.h"

#include "RasterDataBlock.h"
using namespace std;
using namespace UVFTables;

Histogram1DDataBlock::Histogram1DDataBlock() : DataBlock() {
  ulBlockSemantics = BS_1D_HISTOGRAM;
  strBlockID       = "1D Histogram";
}

Histogram1DDataBlock::Histogram1DDataBlock(const Histogram1DDataBlock &other) :
  DataBlock(other),
  m_vHistData(other.m_vHistData)
{
}

Histogram1DDataBlock& Histogram1DDataBlock::operator=(const Histogram1DDataBlock& other) {
  strBlockID = other.strBlockID;
  ulBlockSemantics = other.ulBlockSemantics;
  ulCompressionScheme = other.ulCompressionScheme;
  ulOffsetToNextDataBlock = other.ulOffsetToNextDataBlock;

  m_vHistData = other.m_vHistData;

  return *this;
}


Histogram1DDataBlock::Histogram1DDataBlock(LargeRAWFile* pStreamFile, UINT64 iOffset, bool bIsBigEndian) {
  GetHeaderFromFile(pStreamFile, iOffset, bIsBigEndian);
}

Histogram1DDataBlock::~Histogram1DDataBlock() 
{
}

DataBlock* Histogram1DDataBlock::Clone() const {
  return new Histogram1DDataBlock(*this);
}

UINT64 Histogram1DDataBlock::GetHeaderFromFile(LargeRAWFile* pStreamFile, UINT64 iOffset, bool bIsBigEndian) {
  UINT64 iStart = iOffset + DataBlock::GetHeaderFromFile(pStreamFile, iOffset, bIsBigEndian);
  pStreamFile->SeekPos(iStart);

  UINT64 ulElementCount;
  pStreamFile->ReadData(ulElementCount, bIsBigEndian);

  m_vHistData.resize(size_t(ulElementCount));
  pStreamFile->ReadRAW((unsigned char*)&m_vHistData[0], ulElementCount*sizeof(UINT64));
  return pStreamFile->GetPos() - iOffset;
}

bool Histogram1DDataBlock::Compute(const RasterDataBlock* source) {

  // TODO: right now we can only compute Histograms of scalar data this
  // should be changed to a more general approach
  if (source->ulElementDimension != 1 ||
      source->ulElementDimensionSize.size() != 1)
    return false;

  /// \todo right now compute Histogram assumes that at least the
  // lowest LOD level consists only of a single brick, this brick is
  // used for the hist.-computation this should be changed to a more
  // general approach
  vector<UINT64> vSmallestLOD = source->GetSmallestBrickIndex();
  const vector<UINT64>& vBricks = source->GetBrickCount(vSmallestLOD);
  for (size_t i = 0;i<vBricks.size();i++) if (vBricks[i] != 1) return false;
  
  // create temp histogram 
  size_t iValueRange = size_t(1<<(source->ulElementBitSize[0][0]));
  m_vHistData.resize(iValueRange);
  if (m_vHistData.size() != iValueRange) return false;
  std::fill(m_vHistData.begin(), m_vHistData.end(), 0);

  // LargestSingleBrickLODBrickIndex is well defined as we tested above
  // if we have a single brick LOD
  std::vector<unsigned char> vcSourceData;
  vector<UINT64> vLOD = source->LargestSingleBrickLODBrickIndex();
  vector<UINT64> vOneAndOnly;
  for (size_t i = 0;i<vBricks.size();i++) vOneAndOnly.push_back(0);
  if (!source->GetData(vcSourceData, vLOD, vOneAndOnly)) return false;

  vector<UINT64> vSize  = source->LargestSingleBrickLODBrickSize();

  UINT64 iDataSize = 1;
  for (size_t i = 0;i<vSize.size();i++) iDataSize*=vSize[i];

  /// @todo only 8 and 16 bit integer data are supported.  this should be
  ///       changed to use a more general approach.
  if (source->ulElementBitSize[0][0] == 8) {
    for (UINT64 i = 0;i<iDataSize;i++) {
       m_vHistData[vcSourceData[size_t(i)]]++;
    }
  } else {
    if (source->ulElementBitSize[0][0] == 16) {
      unsigned short *psSourceData = (unsigned short*)(&(vcSourceData.at(0)));
      for (UINT64 i = 0;i<iDataSize;i++) {
        m_vHistData[psSourceData[size_t(i)]]++;
      }
    } else {
      return false;
    }
  }

  // find maximum-index non zero entry
  size_t iSize = 0;
  for (size_t i = 0;i<iValueRange;i++) if (m_vHistData[i] != 0) iSize = i+1;
  m_vHistData.resize(iSize);
  
  // set data block information
  strBlockID = "1D Histogram for datablock " + source->strBlockID;

  return true;
}


void Histogram1DDataBlock::CopyHeaderToFile(LargeRAWFile* pStreamFile, UINT64 iOffset, bool bIsBigEndian, bool bIsLastBlock) {
  DataBlock::CopyHeaderToFile(pStreamFile, iOffset, bIsBigEndian, bIsLastBlock);
  UINT64 ulElementCount = UINT64(m_vHistData.size());
  pStreamFile->WriteData(ulElementCount, bIsBigEndian);
}

UINT64 Histogram1DDataBlock::CopyToFile(LargeRAWFile* pStreamFile, UINT64 iOffset, bool bIsBigEndian, bool bIsLastBlock) {
  CopyHeaderToFile(pStreamFile, iOffset, bIsBigEndian, bIsLastBlock);
  pStreamFile->WriteRAW((unsigned char*)&m_vHistData[0], m_vHistData.size()*sizeof(UINT64));
  return pStreamFile->GetPos() - iOffset;
}


UINT64 Histogram1DDataBlock::GetOffsetToNextBlock() const {
  return DataBlock::GetOffsetToNextBlock() + ComputeDataSize();
}

UINT64 Histogram1DDataBlock::ComputeDataSize() const {
  return sizeof(UINT64) +                  // length of the vector
       m_vHistData.size()*sizeof(UINT64);  // the vector itself
}
