/*
Copyright 1990-2001 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/


package sun.awt.im.iiimp;

import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.StringReader;
import java.util.StringTokenizer;
import java.util.Hashtable;
import java.util.Vector;
import java.text.AttributedString;
import java.text.AttributedCharacterIterator;
import java.awt.AWTEvent;
import java.awt.event.KeyEvent;
import java.awt.event.InputEvent;
import java.awt.font.TextAttribute;
import java.awt.im.InputMethodHighlight;
import com.sun.iiim.*;

public class LWESyntax implements Cloneable {

    /**
     * ccdef rule's keywords
     */
    static final String COMMENT = 		"#";
    static final String DEFMODE = 		"defmode";
    static final String INITIALMODE = 		"initialmode";
    static final String MODE = 			"mode";
    static final String STATUS_KEY =		"_STATUS_";
    static final String FALLTHROUGH =		"fallthrough";
    static final String FALLTHROUGH_KEY = 	"_FALLTHROUGH_";

    /**
     * ccdef actions
     */
    static final String MODE_SWITCH	= "goto";
    static final String CONVERT		= "convert";
    static final String BACKSPACE	= "backspace";
    static final String CONVERT_S	= "convert-s";
    static final String UNCONVERT	= "unconvert";
    static final String NEXT		= "next";
    static final String NEXT_S		= "next-s";
    static final String PREVIOUS	= "previous";
    static final String PREVIOUS_S	= "previous-s";
    static final String FORWARD		= "forward";
    static final String BACKWARD	= "backward";
    static final String MOVE_TOP	= "move-top";
    static final String MOVE_BOTTOM	= "move-bottom";
    static final String CLEAR		= "clear";
    static final String EXPAND		= "expand";
    static final String EXPAND_S	= "expand-s";
    static final String SHRINK		= "shrink";
    static final String SHRINK_S	= "shrink-s";
    static final String EXPAND_NOCONV	= "expand-noconv";
    static final String EXPAND_NOCONV_S = "expand-noconv-s";
    static final String SHRINK_NOCONV	= "shrink-noconv";
    static final String SHRINK_NOCONV_S	= "shrink-noconv-s";
    static final String FIX		= "fix";
    // .....

    /**
     * Constructor
     */
    public LWESyntax(String syntax)
			throws InvalidSyntaxException {
	BufferedReader bReader = new BufferedReader(new StringReader(syntax));
	readRule(bReader);
    }

    public LWESyntax(InputStreamReader reader)
			throws InvalidSyntaxException {
	BufferedReader bReader = new BufferedReader(reader);
	readRule(bReader);
    }

    private void readRule(BufferedReader bReader) throws InvalidSyntaxException {
	// Syntax is represented by kinput2's ccdef rule,
	// so it will come as character stream.
	// This constructer need to interpret ccdef syntax
	// and construct the Hashtable chain.
	modeTable = new Hashtable();

	String line;
	int lineNo = 0;
	try {
	    while ((line = bReader.readLine()) != null) {
		parseRule(line);
		lineNo++;
	    }
	} catch(Exception e) {
	    throw new InvalidSyntaxException(lineNo);
	}
    }

    LWESyntax getCopy() {
	try {
	    return (LWESyntax)clone();
	} catch(Exception e) {
	    if (Manager.DEBUG) {
		e.printStackTrace();
	    }
	    return null;
	}
    }
    
    private Hashtable modeTable;
    private Hashtable firstTable;
    private Hashtable currentTable;

    private void parseRule(String line) throws InvalidSyntaxException {
	String rule = line.trim();
	
	if (rule.startsWith(COMMENT)) {
	    // comment
	    return;
	}
	
	StringTokenizer st = new StringTokenizer(rule);
	if (st.hasMoreTokens() == false) {
	    // blank line
	    return;
	}
	
	String firstToken = st.nextToken();
	
	if (firstToken.equals(DEFMODE)) {
	    // mode kind declaration
	    while (st.hasMoreTokens()) {
		String mode = st.nextToken();
		modeTable.put(mode, new Hashtable());
	    }
	    return;
	}

	if (firstToken.equals(INITIALMODE)) {
	    // initial mode declaration
	    if (!st.hasMoreTokens()) {
		throw new InvalidSyntaxException();
	    }
	    String mode = st.nextToken();
	    firstTable = (Hashtable)modeTable.get(mode);
	    if (firstTable == null) {
		throw new InvalidSyntaxException();
	    }
	    return;
	}

	if (firstToken.equals(MODE)) {
	    // begin rule definition
	    try {
		String mode = st.nextToken();
		currentTable = (Hashtable)modeTable.get(mode);
		if (currentTable == null) {
		    throw new InvalidSyntaxException();
		}
		String status = st.nextToken();
		currentTable.put(STATUS_KEY, status);

		// fallthrough handling
		if (st.hasMoreTokens()) {
		    String fallthrough = st.nextToken();
		    if (fallthrough.equals(FALLTHROUGH)) {
			String target = st.nextToken();
			Hashtable ftTable = (Hashtable)modeTable.get(target);
			if (ftTable == null) {
			    throw new InvalidSyntaxException();
			}
			currentTable.put(FALLTHROUGH_KEY, ftTable);
		    }
		}
	    } catch(Exception e) {
		throw new InvalidSyntaxException();
	    }
	    return;
	}

	if (firstToken.equals("endmode")) {
	    return;
	}

	try {
	    String secondToken = st.nextToken();

	    Hashtable secondTable;
	    if (currentTable.containsKey(secondToken)) {
		secondTable = (Hashtable)currentTable.get(secondToken);
	    } else {
		secondTable = new Hashtable();
		currentTable.put(secondToken, secondTable);
	    }

	    ActionList actions = new ActionList();

	    while(st.hasMoreTokens()) {
		actions.addAction(st.nextToken());
	    }

	    secondTable.put(firstToken, actions);
	} catch(Exception e) {
	    throw new InvalidSyntaxException();
	}
    }   

    /**
     * Returns the IIIMEvent are which are generated
     * with incoming event. If no event will be produced
     * returns null.
     */
    public IIIMEvent[] getEventList(IIIMEvent e) throws InvalidSyntaxException {
	// Get event and search mapping tables maintaining
	// Mode and Context, and returns events
	// and/or move to the next mode.
	AWTEvent aev = e.getAWTEvent(); 
	if (aev instanceof KeyEvent) {
	    e.consume();
	    KeyEvent kev = (KeyEvent)aev;
	    int id = kev.getID();
	    if (id == KeyEvent.KEY_RELEASED) {
		return null;
	    }
	    if (id == KeyEvent.KEY_TYPED && unprintable(kev.getKeyChar())) {
		return null;
	    }

	    if (id == KeyEvent.KEY_PRESSED && printable(kev.getKeyCode())) {
		if (kev.getModifiers() == 0) {
		    return null;
		}
	    }

	    /*
	     * process only
	     *    - typed event with printable keyChar
	     *        ex: a, A, &, $, etc...
	     *    - pressed event with unprintalbe keyCode
	     *        ex: VK_TAB, VK_F1, VK_PAGE_DOWN etc...
	     */

	    String input = null;
	    String rawInput = null;

	    if (id == KeyEvent.KEY_TYPED) {
		rawInput = String.valueOf(kev.getKeyChar());
		input = "'" + rawInput + "'";
	    } else {
		// KEY_PRESSED event which does not produce
		// following KEY_TYPED event
		rawInput = input =
		    makeKeyString(KeyEvent.getKeyText(kev.getKeyCode()),
				  kev.getModifiers());
	    }

	    Hashtable secondTable = (Hashtable)firstTable.get(input);

	    if (secondTable == null) {
		// check fallthrough
		Hashtable ftTable = (Hashtable)firstTable.get(FALLTHROUGH_KEY);
		if (ftTable != null) { 
		    secondTable = (Hashtable)ftTable.get(input);
		    if (secondTable != null) {
			// check fallthroh table
			ActionList result =
			    searchResult(secondTable, context);
			if (result != null) {
			    return processActions(result, context);
			}
		    }
		}

		if (!Character.isISOControl(kev.getKeyChar())) {
		    if (input.startsWith("'")) { 
			context.append(rawInput);
		    }
		}
		    
		return makePreeditEvent(context.toString());
	    }

	    // Context maching
	    ActionList result = searchResult(secondTable, context);
	    if (result == null) {
		// check fallthrough
		Hashtable ftTable = (Hashtable)firstTable.get(FALLTHROUGH_KEY);
		if (ftTable != null) {
		    secondTable = (Hashtable)ftTable.get(input);
		    if (secondTable != null) {
			// check fallthrough table
			result = searchResult(secondTable, context);
			if (result != null) {
			    return processActions(result, context);
			}
		    }
		}

		if (!Character.isISOControl(kev.getKeyChar())) {
		    if (input.startsWith("'")) {
			context.append(rawInput);
		    }
		}

		return makePreeditEvent(context.toString());
	    }
	    return processActions(result, context);
	}
	return null;
    }

    private ActionList searchResult(Hashtable table, StringBuffer keyString) {
	String key = keyString.toString();
	/*
	 * if keyString is "XYZ", and result is null,
	 * then try "YZ", "Z", "" respectively.
	 */
	while(true) {
	    String skey = "\"" + key + "\"";
	    Object value = table.get(skey);
	    if (value != null) {
		int len = keyString.length();
		int keyLen = key.length();
		if (len > 0) {
		    keyString.delete(len - keyLen, len);
		}
		return (ActionList)value;
	    }
	    if (key.length() < 1) {
		return null;
	    }
	    key = key.substring(1, key.length());
	}
    }

    private IIIMEvent[] processActions(ActionList result, StringBuffer context)
	throws InvalidSyntaxException {
	    
	String[] actionArray = result.getActions();
	Vector retVector = new Vector();
	
	for (int i = 0; i < actionArray.length; i++) {
	    String action = actionArray[i];
	    if (action.equals("")) {
		continue;
	    }
	    if (action.startsWith("\"")) {
		String raw = action.substring(1, action.length() - 1);
		context.append(raw);
		IIIMEvent[] imea = makePreeditEvent(context.toString());
		if (imea != null)
		    retVector.add(imea[0]);
	    } else {
		// check mode switch
		if (action.equals(MODE_SWITCH)) {
		    if (actionArray.length <= i + 1) {
			throw new InvalidSyntaxException(0);
		    }
		    String newMode = actionArray[++i];
		    firstTable = (Hashtable)modeTable.get(newMode);
		    if (firstTable == null) {
			throw new InvalidSyntaxException(0);
		    }
		    retVector.add(new IIIMCommittedEvent
				  (context.toString()));
		    context.setLength(0);
		} else if (action.equals(CONVERT)) {
		    IIIMActionEvent e =
			new IIIMActionEvent
			(IIIMActionEvent.FORWARD_STRING,
			 new String[] {action, context.toString()});
System.out.println(" make a event = " + action);
		    retVector.add(e);
		    context.setLength(0);
		} else if (action.equals(BACKSPACE)){
		    if (context.length() > 0) {
			StringBuffer newSB =  context.delete(context.length() - 1,
							     context.length());
			IIIMEvent[] ie = makePreeditEvent(newSB.toString());
			retVector.add(ie[0]);
		    }
		} else {
		    debug(" action = " + action +
				       " is not supported yet.");
		}
	    }
	}
	// context.setLength(0);
	Object[] oa = retVector.toArray();
	IIIMEvent[] iiimea = new IIIMEvent[oa.length];
	for (int i = 0; i < oa.length; i++) {
	    iiimea[i] = (IIIMEvent)oa[i];
	}
	return iiimea;
    }

    private void debug(String str) {
	if (Manager.DEBUG) {
	    System.err.println(str);
	}
    }

    String getCurrentContext() {
	return context.toString();
    }

    private IIIMEvent[] makePreeditEvent(String str) {
	AttributedString attrstr = new AttributedString(str);
	if (str.length() > 0) {
	    attrstr.addAttribute(TextAttribute.INPUT_METHOD_HIGHLIGHT,
				 InputMethodHighlight.SELECTED_CONVERTED_TEXT_HIGHLIGHT,
				 0, str.length());
	}
	AttributedCharacterIterator iterator = attrstr.getIterator();
	IIIMPreeditEvent pe =
	    new IIIMPreeditEvent(IIIMPreeditEvent.DRAW, iterator, 0);
	IIIMEvent[] iiimea = new IIIMEvent[1];
	iiimea[0] = pe;
	return iiimea;
    }

    private static int[] masks = {
	InputEvent.SHIFT_MASK,
	InputEvent.CTRL_MASK,
	InputEvent.META_MASK,
	// below mask will be added if ccdef uses these
	// 
	// InputEvent.ALT_MASK,
	// InputEvent.ALT_GRAPH_MASK,
	// InputEvent.BUTTON1_MASK,
	// InputEvent.BUTTON2_MASK,
	// InputEvent.BUTTON3_MASK,
    };

    private static String[] maskStrings = {
	"shift",
	"control",
	"mod1",
    };
	
    private static String makeKeyString(String key, int mod) {
	String modString = "";
	for (int i = 0; i < masks.length; i++) {
	    if ((masks[i] & mod) != 0) { 
		modString += (maskStrings[i] + "-");
	    }
	}
	String ret = modString;
	if (key.equals("Backspace")) {
	    ret = "'^H'";
	} else {
	    if (modString.equals("control-") && key.length() == 1) {
		ret = "'^" + key + "'";
	    } else {
		ret += key;
	    }
	}
	return ret;
    }

    private StringBuffer context = new StringBuffer();

    private static boolean printable(int keyCode) {
	switch(keyCode) {
	  case KeyEvent.VK_0:
	  case KeyEvent.VK_1:
	  case KeyEvent.VK_2:
	  case KeyEvent.VK_3:
	  case KeyEvent.VK_4:
	  case KeyEvent.VK_5:
	  case KeyEvent.VK_6:
	  case KeyEvent.VK_7:
	  case KeyEvent.VK_8:
	  case KeyEvent.VK_9:
	  case KeyEvent.VK_A:
	  case KeyEvent.VK_B:
	  case KeyEvent.VK_C:
	  case KeyEvent.VK_D:
	  case KeyEvent.VK_E:
	  case KeyEvent.VK_F:
	  case KeyEvent.VK_G:
	  case KeyEvent.VK_H:
	  case KeyEvent.VK_I:
	  case KeyEvent.VK_J:
	  case KeyEvent.VK_K:
	  case KeyEvent.VK_L:
	  case KeyEvent.VK_M:
	  case KeyEvent.VK_N:
	  case KeyEvent.VK_O:
	  case KeyEvent.VK_P:
	  case KeyEvent.VK_Q:
	  case KeyEvent.VK_R:
	  case KeyEvent.VK_S:
	  case KeyEvent.VK_T:
	  case KeyEvent.VK_U:
	  case KeyEvent.VK_V:
	  case KeyEvent.VK_W:
	  case KeyEvent.VK_X:
	  case KeyEvent.VK_Y:
	  case KeyEvent.VK_Z:
	  case KeyEvent.VK_SPACE:
	  case KeyEvent.VK_COMMA:
	  case KeyEvent.VK_MINUS:
	  case KeyEvent.VK_PERIOD:
	  case KeyEvent.VK_SLASH:
	  case KeyEvent.VK_SEMICOLON:
	  case KeyEvent.VK_EQUALS:
	  case KeyEvent.VK_OPEN_BRACKET:
	  case KeyEvent.VK_BACK_SLASH:
	  case KeyEvent.VK_CLOSE_BRACKET:
	  case KeyEvent.VK_NUMPAD0:
	  case KeyEvent.VK_NUMPAD1:
	  case KeyEvent.VK_NUMPAD2:
	  case KeyEvent.VK_NUMPAD3:
	  case KeyEvent.VK_NUMPAD4:
	  case KeyEvent.VK_NUMPAD5:
	  case KeyEvent.VK_NUMPAD6:
	  case KeyEvent.VK_NUMPAD7:
	  case KeyEvent.VK_NUMPAD8:
	  case KeyEvent.VK_NUMPAD9:
	  case KeyEvent.VK_MULTIPLY:
	  case KeyEvent.VK_ADD:
	  case KeyEvent.VK_SEPARATER:
	  case KeyEvent.VK_SUBTRACT:
	  case KeyEvent.VK_DECIMAL:
	  case KeyEvent.VK_DIVIDE:
	  case KeyEvent.VK_BACK_QUOTE:
	  case KeyEvent.VK_QUOTE:
	  case KeyEvent.VK_DEAD_GRAVE:
	  case KeyEvent.VK_DEAD_ACUTE:
	  case KeyEvent.VK_DEAD_CIRCUMFLEX:
	  case KeyEvent.VK_DEAD_TILDE:
	  case KeyEvent.VK_DEAD_MACRON:
	  case KeyEvent.VK_DEAD_BREVE:
	  case KeyEvent.VK_DEAD_ABOVEDOT:
	  case KeyEvent.VK_DEAD_DIAERESIS:
	  case KeyEvent.VK_DEAD_ABOVERING:
	  case KeyEvent.VK_DEAD_DOUBLEACUTE:
	  case KeyEvent.VK_DEAD_CARON:
	  case KeyEvent.VK_DEAD_CEDILLA:
	  case KeyEvent.VK_DEAD_OGONEK:
	  case KeyEvent.VK_DEAD_IOTA:
	  case KeyEvent.VK_DEAD_VOICED_SOUND:
	  case KeyEvent.VK_DEAD_SEMIVOICED_SOUND:
	  case KeyEvent.VK_AMPERSAND:
	  case KeyEvent.VK_ASTERISK:
	  case KeyEvent.VK_QUOTEDBL:
	  case KeyEvent.VK_LESS:
	  case KeyEvent.VK_GREATER:
	  case KeyEvent.VK_BRACELEFT:
	  case KeyEvent.VK_BRACERIGHT:
	  case KeyEvent.VK_AT:
	  case KeyEvent.VK_COLON:
	  case KeyEvent.VK_CIRCUMFLEX:
	  case KeyEvent.VK_DOLLAR:
	  case KeyEvent.VK_EURO_SIGN:
	  case KeyEvent.VK_EXCLAMATION_MARK:
	  case KeyEvent.VK_INVERTED_EXCLAMATION_MARK:
	  case KeyEvent.VK_LEFT_PARENTHESIS:
	  case KeyEvent.VK_NUMBER_SIGN:
	  case KeyEvent.VK_PLUS:
	  case KeyEvent.VK_RIGHT_PARENTHESIS:
	  case KeyEvent.VK_UNDERSCORE:
	    return true;
	}
	return false;
    }

    private static boolean unprintable(char c) {
	if (c >= '\u0000' && c <= '\u0019' || c == '\u007f') {
	    return true;
	}
	return false;
    }

    class ActionList {

	Vector vector = new Vector();

	void addAction(String action) {
	    vector.add(action);
	}

	int size() {
	    return vector.size();
	}

	String[] getActions() {
	    Object[] oa = vector.toArray();
	    String[] sa = new String[oa.length];
	    for (int i = 0; i < sa.length; i++) {
		sa[i] = (String)oa[i];
	    }
	    return sa;
	}
    }
}
