#include <config.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifndef WIN32
#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#define DIRECTORY_SEP "/"
#define IS_DIRECTORY_SEP(c) ((c) == '/')
#else
#define stat _stat
#define chdir _chdir
#define S_ISDIR(m) ((m) & _S_IFDIR)
#define S_IRUSR _S_IREAD
#define S_IWUSR _S_IWRITE
#define DIRECTORY_SEP "\\"
#define IS_DIRECTORY_SEP(c) (((c) == '/') || ((c) == '\\'))
#endif

#ifndef WIN32
#include <unistd.h>
#include <dirent.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/resource.h>
#else
// TODO
#endif

#if defined(HAVE_POLL)
#include <sys/poll.h>
#elif defined(HAVE_SELECT)
#include <sys/select.h>
#endif

#include "IMUtil.hh"
#include "IMLog.hh"

/*******************************************************************************
                             DirectoryInfo
 *******************************************************************************/

bool
DirectoryInfo::addfile(
    const char *filename
)
{
    struct stat s;
    DirectoryEntry d;

    string full = filename;
    if (full.empty()) return false;

    char head = full.at(0);
    if (!IS_DIRECTORY_SEP(head)) {
	full = path;
	char tail = full.at(full.size() - 1);
	if (!IS_DIRECTORY_SEP(tail)) {
	    full = full.append(DIRECTORY_SEP);
	}
	full = full.append(filename);
    }

    if (stat(full.c_str(), &s) != 0) return false;
    if (!subdirp && S_ISDIR(s.st_mode)) return false;
    if (!unreadablep && (!(s.st_mode & S_IRUSR))) return false;

    d.fullname = full;

    {
	string::size_type fpos;
#ifdef WIN32
	fpos = d.fullname.find_last_of("\\/");
#else
	fpos = d.fullname.rfind('/');
#endif
	if (fpos == string::npos) {
	    d.filename = d.fullname;
	    d.dirname = "";
	} else {
	    d.filename = d.fullname.substr(fpos + 1);
	    d.dirname = d.fullname.substr(0, fpos + 1);
	}
    }

    d.readable_p = (s.st_mode & S_IRUSR);
    d.writable_p = (s.st_mode & S_IWUSR);
    d.directory_p = S_ISDIR(s.st_mode); 
    dirvec.push_back(d);

    return true;
}

bool
DirectoryInfo::refresh(
	const char *x_path
)
{
    if (x_path) path = x_path;
    if (path.empty()) return false;

    if (!dirvec.empty()) dirvec.erase(dirvec.begin(), dirvec.end());

#if defined(WIN32)
#error "TODO"
#elif defined(HAVE_READDIR_R)
    DIR *dir;
    struct dirent *entry;
    struct dirent *result;
    int ret;

    dir = opendir(path.c_str());
    if (!dir) return false;
    entry = (struct dirent*) malloc(sizeof(struct dirent) + NAME_MAX);
    if (!entry) return false;
    while (((ret = readdir_r(dir, entry, &result)) == 0)
	   && result) {
	addfile(result->d_name);
    }
    free(entry);
    closedir(dir);
#else /* normal readdir */
    DIR *dir;
    struct dirent *result;
    int ret;

    dir = opendir(path.c_str());
    if (!dir) return false;
    while (result = readdir(dir)) {
	addfile(result->d_name);
    }
    closedir(dir);
#endif

    return true;
}

DirectoryInfo::DirectoryInfo(
    bool include_subdir_p,
    bool include_unreable_p
)
{
    subdirp = include_subdir_p;
    unreadablep = include_unreable_p;
}

/*******************************************************************************
                             File manipulation.
 *******************************************************************************/

CARD8BIT*
IM_read_file(
    const char* filename,
    int size
)
{
    size_t rsize;
    FILE *fd;
    CARD8BIT *pr;

    fd = fopen(filename, "rb");
    if (!fd) {
	return NULL;
    }
    pr = (CARD8BIT*) malloc(sizeof(CARD8BIT) * size);
    if (!pr) {
	fclose(fd);
	return NULL;
    }
    rsize = fread(pr, sizeof(CARD8BIT), size, fd);
    fclose(fd);
    if (rsize != size) {
	free(pr);
	return NULL;
    }

    return pr;
}

// return -1 if error occurs.
int IM_file_size(
    const char* filename
)
{
    struct stat s;
    if (stat(filename, &s) != 0) return -1;
    return s.st_size;
}

/*******************************************************************************
			       IMSocket & IMSocketListen
 *******************************************************************************/

//
// Address object.
//

void*
IMSocketAddress::get_addrinfo()
{
    struct addrinfo hints, *pst;
    int st;

    if (addrinfo) return addrinfo;

    memset(&hints, 0, sizeof(hints));
    hints.ai_family = PF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    if (!address.c_str()) hints.ai_flags = AI_PASSIVE;
    st = getaddrinfo(address.c_str(), service.c_str(), &hints, &pst);
    if (st != 0) {
	LOG_ERROR("Fail to getaddrinfo:%s, %s",
		  address.c_str(), service.c_str());
	return NULL;
    }

    addrinfo = pst;

    return pst;
}

void
IMSocketAddress::bind_fail(
    int fd
)
{
    int e = errno;

    LOG_ERROR("Fail to bind socket: %s:%s", address.c_str(), service.c_str());
#ifdef EADDRINUSE
    const int err_addr_in_use = EADDRINUSE;
#else
    const int err_addr_in_use = EINVAL;
#endif
    if (e == err_addr_in_use) {
	LOG_ERROR("Another IIIMP server might be running(address already in use).");
    }

    close(fd);
 }

void
IMSocketAddress::configure_socket(
    int fd
)
{
    int reuse = 1;
    setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse));
}

int
IMSocketAddress::setup_inet(
    void *paddressinfo
)
{
    struct addrinfo *pai = (struct addrinfo*) paddressinfo;

    int fd = socket(pai->ai_family, pai->ai_socktype, pai->ai_protocol);

    if (fd < 0) {
	LOG_ERROR("Cannot open socket for %s:%s",
		  address.c_str(), service.c_str());
	return -1;
    }

    configure_socket(fd);

    if (bind(fd, pai->ai_addr, pai->ai_addrlen) < 0) {
	bind_fail(fd);
	return -1;
    }

    return fd;
}

int
IMSocketAddress::setup_unix()
{
#if 0
    struct sockaddr_un sin;
    static const char sockdir[] = "/tmp/.iiimp-unix"

    int old_umask;

    int fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if (fd == -1) {
	LOG_ERROR("Cannot open socket. (%s)", address.c_str());
	status = ERROR;
	return false;
    }

    configure_socket(fd);

    sin.sun_family = AF_UNIX;
    snprintf(sin.sun_path, sizeof(sin.sun_path), "%s/%d",
	     sockdir, address.c_str());

    old_umask = umask(0);
    mkdir(sockdir, 0777);
    chmod(sockdir, 0777);
    unlink(sin.sun_path);
    umask(old_umask);

    if (bind(fd, (struct sockaddr*) &sin, sizeof(sin)) < 0) {
	bind_fail(fd);
	return -1;
    }

    status = OPEN;

    return fd;
#else
    return -1;
#endif
}

int
IMSocketAddress::create_sockets(
    IMSocketVec &sockvec
)
{
    if (type == INET) {
	int i = 0;
	int fd;
	struct addrinfo *pai = (struct addrinfo*) get_addrinfo();
	IMSocket *ps;

	for (; pai; pai = pai->ai_next) {
	    fd = setup_inet(pai);
	    if (fd < 0) continue;
	    ps = new IMSocket(fd);
	    // it should be dealt with memory exception.
	    if (!ps) {
		close(fd);
		return -1;
	    }
	    sockvec.push_back(ps);
	    i++;
	}

	if (i == 0) {
	    LOG_ERROR("Failed to create any sockets for %s:%s ... exit.",
		      address.c_str(), service.c_str());
	    exit(255);
	}

	return i;
    } else if (type == UNIX_DOMAIN) {
	IMSocket *ps;
	int fd = setup_unix();

	if (fd >= 0) {
	    ps = new IMSocket(fd);
	    if (!ps) {
		close(fd);
		return -1;
	    }
	}
	if (fd < 0) {
	    LOG_ERROR("Failed to create a unix domain socket for %s:%s ... exit.",
		      address.c_str(), service.c_str());
	    exit(255);
	}

	sockvec.push_back(ps);
	return 1;
    }

    ERROR_INTERNAL("invalid IMSocketAddress type");

    return 0;
}

IMSocketAddress::~IMSocketAddress()
{
    if (addrinfo) {
	freeaddrinfo((struct addrinfo*) addrinfo);
    }
}

IMSocketAddress::
IMSocketAddress(
    ADDRESS_TYPE x_type,
    const string &x_address,
    const string &x_service
)
{
    type = x_type;
    address = x_address;
    service = x_service;

    addrinfo = NULL;
}

//
// Socket object.
//

bool
IMSocket::do_listen()
{
    if (::listen(fd, 5) < 0) {
	LOG_ERROR("Fail to listen:%d", fd);
	close(fd);
	status = ERROR;
	return false;
    }

    status = OPEN;

    return true;
}

bool
IMSocket::listen()
{
    if (status == ERROR) return false;
    if (status == OPEN) return true;

    return do_listen();
}

IMSocketTrans*
IMSocket::accept()
{
    int st;
    struct sockaddr_storage ss;
    struct sockaddr *psa = (struct sockaddr*) &ss;
    socklen_t fromlen;

    if (status == ERROR) return NULL;

    do {
	fromlen = sizeof(ss);
	st = ::accept(fd, psa, &fromlen);
	if (st != -1) return new IMSocketTrans(st);
    } while (errno == EINTR);

    status = ERROR;
    LOG_ERROR("Could not accept.\n");

    return NULL;
}

int
IMSocket::get_fd()
{
    if (status == OPEN) return fd;
    return -1;
}

IMSocket::IMSocket(
    int x_fd
)
{
    fd = x_fd;
    status = CLOSED;
}

IMSocket::~IMSocket()
{
    if (status == OPEN) {
	close(fd);
    }
}

//
// Trans object.
//

int
IMSocketTrans::send(
    const void *p,
    size_t n
)
{
    return ::send(fd, p, n, 0);
}

int
IMSocketTrans::recv(
    void *p,
    size_t n
)
{
    return ::recv(fd, p, n, 0);
}

IMSocketTrans::IMSocketTrans(
    int x_fd
)
{
    fd = x_fd;
}

IMSocketTrans::~IMSocketTrans()
{
    if (fd > 0) {
	shutdown(fd, SHUT_RDWR);
	close(fd);
    }
}


//
// Listen object.
//

bool
IMSocketListen::listen(
    IMSocketAddressVec& endvec
)
{
    if (status == IMSocket::ERROR) return false;
    if (status == IMSocket::OPEN) return true;

    IMSocketAddressVec::iterator ait;
    for (ait = endvec.begin(); ait!= endvec.end(); ait++) {
	if (ait->create_sockets(sockvec) < 0) return false;
    }

    IMSocketVec::iterator it;
    IMSocket *pims;
    for (it = sockvec.begin(); it != sockvec.end(); it++) {
	pims = *it;
	if (pims->get_status() == IMSocket::ERROR) return false;
	if (pims->get_status() == IMSocket::OPEN) continue;
	if (!pims->listen()) {
	    status = IMSocket::ERROR;
	    return false;
	}
    }
    status = IMSocket::OPEN;

    return true;
}

IMSocketTrans*
IMSocketListen::accept()
{
    IMSocketVec::iterator it;
    IMSocket *pims;

    if (status != IMSocket::OPEN) return NULL;

#if defined(HAVE_POLL)
    int fd, i, n;
    struct pollfd *fds;

    i = sockvec.size();
    if (i <= 0) return NULL;
    fds = new struct pollfd[i];
    if (!fds) return NULL;
    for (n = 0, it = sockvec.begin(); it != sockvec.end(); it++) {
	pims = *it;
	fd = pims->get_fd();
	if (fd < 0) continue;
	fds[n].fd = fd;
	fds[n].events = POLLIN;
	fds[n].revents = 0;
	n++;
    }
    if (n > 0) {
	i = poll(fds, n, -1);
	if (i > 0) {
	    for (i = 0; i < n; i++) {
		if (fds[i].revents) {
		    fd = fds[i].fd;
		    for (it = sockvec.begin(); it != sockvec.end(); it++) {
			pims = *it;
			if (fd == pims->get_fd()) {
			    free(fds);
			    return pims->accept();
			}
		    }
		}
	    }
	}
    }
    free(fds);

#elif defined(HAVE_SELECT)
    int fd, maxfd;
    fd_set fds;
    FD_ZERO(&fds);

    maxfd = -1;
    for (it = sockvec.begin(); it != sockvec.end(); it++) {
	pims = *it;
	fd = pims->get_fd();
	if (fd < 0) continue;
	if (fd > maxfd) maxfd = fd;
	FD_SET(fd, &fds);
    }
    if (maxfd < 0) return NULL;
    maxfd++;
    fd = select(maxfd, &fds, NULL, NULL, NULL);
    if (fd <= 0) return NULL;
    for (it = sockvec.begin(); it != sockvec.end(); it++) {
	pims = *it;
	fd = pims->get_fd();
	if (fd < 0) continue;
	if (FD_ISSET(fd, &fds)) {
	    fd = pims->accept();
	    if (fd < 0) return NULL;
	    return fd;
	}
    }
#else
#error "We need poll() or select() to multiplex socket handles"
#endif

    return NULL;
}

IMSocketListen::IMSocketListen()
{
    status = IMSocket::CLOSED;
}

IMSocketListen::~IMSocketListen()
{
    delete_all(sockvec);
}

/*******************************************************************************
                    IMDaemon (helper object to make a program a daemon)
 *******************************************************************************/

IMDaemon* IMDaemon::pimdaemon = NULL;

IMDaemon::IMDaemon()
{
    already_finished = false;
}

IMDaemon*
IMDaemon::get_instance()
{
    if (pimdaemon) return pimdaemon;
    pimdaemon = new IMDaemon();
    return pimdaemon;
}

void
IMDaemon::cleanup()
{
    if (pimdaemon) delete pimdaemon;
}

void
IMDaemon::closefd()
{
    int fd;
#ifdef HAVE_SYSCONF
    int fdmax = sysconf(_SC_OPEN_MAX);
#else
    int fdmax = 3;
#endif
    for (fd = 0; fd < fdmax; fd++) {
	close(fd);
    }

#ifndef WIN32
    // 0 : discard stdout
    open("/dev/null", O_RDWR);
    // 1 : discard stdin
    dup(0);
    // 2 : discard stderr
    dup(0);
#endif
}

bool
IMDaemon::discard_parent()
{
#ifdef HAVE_FORK
    pid_t pid = fork();
    if (pid != 0) {
	if (pid == -1) {
	    return false;
	} else {
	    _exit(0);
	}
    }
#endif
    return true;
}

bool
IMDaemon::daemonize()
{
    if (!discard_parent()) return false;
#ifdef HAVE_SETSID
    if (setsid() < 0)
	return false;
#endif
    if (!discard_parent()) return false;

    return true;
}

void
IMDaemon::setup(
    const char* todir
)
{
    if (!already_finished) {
	if (!daemonize()) {
	    LOG_ERROR("Fail to make a daemon.");
	}
	closefd();
    }
    if (todir) {
	chdir(todir);
    }
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
