/* GTK - The GIMP Toolkit
 * Copyright (C) 2000 Red Hat Software
 * Copyright (C) 2003 Motonobu Ichimura
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Motonobu Ichimura <famao@momonga-linux.org>
 *
 */

#include <locale.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <gtk/gtkimmodule.h>
#include <stdlib.h>
#include <string.h>
#include "iiimcf.h"
#include "gtkimcontextiiim.h"
#include "IIIMGdkEventKey.h"

typedef struct _StatusWindow StatusWindow;
typedef struct _CandidateWindow CandidateWindow;

struct _GtkIIIMInfo
{
#if (GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2)
  GdkScreen *screen;
#endif
  IIIMCF_handle iiim;
  IIIMCF_language lang;
  char *locale;
  char *le;
  GSList *ics;
};

/* A candiate window */
struct _CandidateWindow
{
  GtkWidget *tree;
  GtkWidget *toplevel;
  GtkWidget *frame;

  /* Toplevel window to which the candiddate window corresponds */
  GtkWidget *app_toplevel;
  GtkListStore *store;
  gint choices_per_window;
  gint number_of_rows;
  gint number_of_columns;
  gint direction;

  gulong destroy_handler_id;
  gulong configure_handler_id;
};

/* A context status window; these are kept in the status_windows list. */
struct _StatusWindow
{
  GtkWidget *window;
  
  /* Toplevel window to which the status window corresponds */
  GtkWidget *toplevel;
  
  /* Signal connection ids; we connect to the toplevel */
  gulong destroy_handler_id;
  gulong configure_handler_id;
};

static void im_context_iiim_class_init (GtkIMContextIIIMClass *class);
static void im_context_iiim_init (GtkIMContextIIIM *im_context);
static void im_context_iiim_finalize (GObject *obj);
static void im_context_iiim_set_client_window (GtkIMContext *context, 
					       GdkWindow *client_window);
static gboolean im_context_iiim_filter_keypress (GtkIMContext *context, 
						 GdkEventKey *key);
static void im_context_iiim_reset (GtkIMContext *context);
static void im_context_iiim_focus_in (GtkIMContext *context);
static void im_context_iiim_focus_out (GtkIMContext *context);

static void im_context_iiim_set_cursor_location (GtkIMContext *context, 
						 GdkRectangle *area);
static void im_context_iiim_set_use_preedit (GtkIMContext *context, 
					     gboolean use_preedit);
static void im_context_iiim_get_preedit_string (GtkIMContext *context, 
						gchar **str,
						PangoAttrList **attrs, 
						gint *cursor_pos);


/* Session Context */
static IIIMCF_context iiim_get_session_context (GtkIMContextIIIM *context_iiim);

/* Candidate Window */
static IIIMCF_lookup_choice iiim_get_lookup_choice (GtkIMContextIIIM *context_iiim);
static GtkWidget *iiim_get_candidate_window (GtkIMContextIIIM *context_iiim);
static void iiim_candidate_show (GtkIMContextIIIM *context_iiim);
static void iiim_destroy_candidate_window (GtkIMContextIIIM *context_iiim);
static gboolean iiim_candidate_window_configure (GtkWidget *toplevel, 
						 GdkEventConfigure *event, 
						 GtkIMContextIIIM *context_iiim);

/* Status */
static void iiim_status_update (GtkIMContextIIIM *context_iiim);

/* Event */
static gboolean iiim_event_dispatch (GtkIMContextIIIM *context_iiim);

static void get_lookup_choice (GtkIMContext *context);
static gchar *utf16_to_utf8 (IIIMCF_text text);

static void
set_sc_client_window (GtkIMContextIIIM *context_iiim,
		      GdkWindow *client_window,
		      gboolean send_signal);

/* status window */
static void status_window_show (GtkIMContextIIIM *context_iiim);
static void status_window_hide (GtkIMContextIIIM *context_iiim);
static void status_window_set_text (GtkIMContextIIIM *context_iiim,
		                    const gchar *text);
static GObjectClass *parent_class;

GType gtk_type_im_context_iiim = 0;

static GSList *open_iiims = NULL;

static gboolean iiim_is_initialized = FALSE;
static IIIMCF_handle iiim = NULL;

/* List of status windows for different toplevels */
static GSList *status_windows = NULL;


void
im_context_iiim_register_type (GTypeModule *type_module)
{
  static const GTypeInfo im_context_iiim_info =
    {
      sizeof (GtkIMContextIIIMClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) im_context_iiim_class_init,
      NULL,
      NULL,
      sizeof (GtkIMContextIIIM),
      0,
      (GInstanceInitFunc) im_context_iiim_init,
    };

  gtk_type_im_context_iiim =
    g_type_module_register_type (type_module,
				 GTK_TYPE_IM_CONTEXT,
				 "GtkIMContextIIIM",
				 &im_context_iiim_info, 0);
}


static void
im_context_iiim_class_init (GtkIMContextIIIMClass *class)
{
  GtkIMContextClass *im_context_class = GTK_IM_CONTEXT_CLASS(class);
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

                                                                                             
  im_context_class->set_client_window = im_context_iiim_set_client_window;
  im_context_class->filter_keypress = im_context_iiim_filter_keypress;
  im_context_class->reset = im_context_iiim_reset;
  im_context_class->get_preedit_string = im_context_iiim_get_preedit_string;
  im_context_class->focus_in = im_context_iiim_focus_in;
  im_context_class->focus_out = im_context_iiim_focus_out;
  im_context_class->set_cursor_location = im_context_iiim_set_cursor_location;
  im_context_class->set_use_preedit = im_context_iiim_set_use_preedit;
  gobject_class->finalize = im_context_iiim_finalize;
  return;
}

/* 
 * Update Status Window Message 
 */
static void
iiim_status_update (GtkIMContextIIIM *context_iiim)
{
  gchar *utf8;
  IIIMCF_context c = iiim_get_session_context (context_iiim);
  IIIMCF_text text;
  IIIMF_status st;
  st = iiimcf_get_status_text (c , &text);
  if (st != IIIMF_STATUS_SUCCESS) {
    status_window_set_text (context_iiim, NULL);
    return;
  }
  utf8 = utf16_to_utf8 (text);
  status_window_set_text (context_iiim, utf8);
  g_free (utf8);
}

/* get Lang List */
static void
iiim_get_languages (GtkIIIMInfo *info)
{
  IIIMF_status st;
  IIIMCF_handle handle = info->iiim;
  IIIMCF_language *lang;
  char *langid;
  int n_lang;
  int i;

  if (!handle)
    return;
  st = iiimcf_get_supported_languages (handle,
				       &n_lang, &lang);
  if (st != IIIMF_STATUS_SUCCESS) return;
  g_message ("locale %s", info->locale);
  for (i = 0; i < n_lang; i++)
    {
      st = iiimcf_get_language_id (lang[i],
				   (const char **)&langid);
      if (st != IIIMF_STATUS_SUCCESS) return;
      if (!strncmp (langid, info->locale, strlen(langid)))
	{
	  info->lang = lang[i];
	  break;
	}
      g_message ("lang id %s", langid);
    }

}

/*
 * Get IM List
 */
#if 0
static void
iiim_get_im_list (GtkIIIMInfo *info)
{
  char *langs;
  char *im_id;
  char *domain;
  IIIMP_card16 *im_hrn_string;
  int im_hrn_len;
  int input_methods_n;
  int i;
  IIIMCF_handle handle = info->iiim;
  IIIMCF_input_method *input_methods;
  IIIMF_status st;
  if (!handle)
    return;
  st = iiimcf_get_supported_input_methods (handle, 
					   &input_methods_n, &input_methods);
  if (st != IIIMF_STATUS_SUCCESS) return;

  for (i = 0; i < input_methods_n; i++)
    {
      st = iiimcf_get_input_method_desc (input_methods[i], &im_id,
					 &domain, &im_hrn_len, &im_hrn_string);
      if (st != IIIMF_STATUS_SUCCESS) return;
      g_message ("%s %s", im_id, domain);
    }
  return;
}
#endif

/*
 * Event Dispatch
 */
static gboolean
iiim_event_dispatch (GtkIMContextIIIM *context_iiim)
{
  IIIMCF_context c;
  IIIMF_status st;
  IIIMCF_event ev;
  IIIMCF_event_type et;
  gboolean result = TRUE;

  c = iiim_get_session_context (context_iiim);
  if (!c) return FALSE;

  while ((st = iiimcf_get_next_event (c, &ev)) == IIIMF_STATUS_SUCCESS)
    {
      st = iiimcf_get_event_type (ev, &et);
      if (st != IIIMF_STATUS_SUCCESS) continue;
      g_message ("event type %d", et);
      switch (et)
	{
	case IIIMCF_EVENT_TYPE_KEYEVENT:
	  {
	    /* FIXME */
	    /* when IIIMCF_EVENT_TYPE_KEYEVENT arrives,
	     * we need to send Event to a Client.
	     */
	    IIIMCF_keyevent kev;
	    XKeyEvent xev;
	    char ch[2];
	    st = iiimcf_get_keyevent_value (ev, &kev);
	    if (st != IIIMF_STATUS_SUCCESS)
	      {
		result = FALSE;
		break;
	      }
#if 0
	    VirtualKeyToXKeyEvent (kev.keycode,
				   kev.keychar, kev.modifier, &xev);
	    xev.time = kev.time_stamp;
	    xev.same_screen = True;
	    xev.display = GDK_DRAWABLE_XDISPLAY (context_iiim->client_window);
	    xev.window = GDK_DRAWABLE_XID (context_iiim->client_window);
	    xev.subwindow = xev.window;
	    xev.root = GDK_DRAWABLE_XID (gdk_screen_get_root_window (gdk_drawable_get_screen (context_iiim->client_window)));
	    xev.x = xev.x_root = 0;
	    xev.y = xev.y_root = 0;
	    xev.serial = 0;

	    XSendEvent (xev.display, xev.window, FALSE, KeyPressMask | KeyReleaseMask, 
			(XEvent *)&xev);
	    return TRUE;
#endif
	    if (kev.keychar > 0x20 && kev.keychar < 0x7f)
	      {
		ch[0] = kev.keychar;
		ch[1] = '\0';
		g_signal_emit_by_name (context_iiim,
				       "commit", ch);
	      }
	    else
	      {
		result = FALSE;
	      }
	  }
	  break;
	case IIIMCF_EVENT_TYPE_UI_LOOKUP_CHOICE_START:
	  g_message ("lookup_choice start");
	  context_iiim->candidate_start = TRUE;
	  iiim_candidate_show (context_iiim);
	  break;
	case IIIMCF_EVENT_TYPE_UI_LOOKUP_CHOICE_CHANGE:
	  iiim_candidate_show (context_iiim);
	  g_message ("lookup_choice change");
	  break;
	case IIIMCF_EVENT_TYPE_UI_LOOKUP_CHOICE_DONE:
	  context_iiim->candidate_start = FALSE;
	  iiim_destroy_candidate_window (context_iiim);
	  g_message ("lookup_choice_done");
	  break;
	case IIIMCF_EVENT_TYPE_UI_STATUS_START:
	  g_message ("ui_status_start");
	  status_window_show (context_iiim);
	  iiim_status_update (context_iiim);
	  break;
	case IIIMCF_EVENT_TYPE_UI_STATUS_CHANGE:
	  g_message ("ui_status_change");
	  iiim_status_update (context_iiim);
	  break;
	case IIIMCF_EVENT_TYPE_UI_STATUS_END:
	  g_message ("ui_status_end");
	  status_window_hide (context_iiim);
	  iiim_status_update (context_iiim);
	  break;
	default:
	  break;
	}
    }
  iiimcf_dispatch_event (c, ev);
  iiimcf_ignore_event (ev);
  
  return result;
}

#if (GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION ==2)
static void
iiim_info_display_closed (GdkDisplay *display,
			  gboolean is_error,
			  GtkIIIMInfo *info)
{
  GSList *ics, *tmp_list;

  open_iiims = g_slist_remove (open_iiims, info);

  ics = info->ics;
  info->ics = NULL;

  for (tmp_list = ics; tmp_list; tmp_list = g_slist_next (tmp_list))
    {
      set_sc_client_window (tmp_list->data, NULL, TRUE);
    }

  g_slist_free (tmp_list);

  g_free (info->locale);
  if (info->le)
    g_free (info->le);

  /* TODO */
  g_free (info);
}
#endif

static GtkIIIMInfo *
get_iiim (GdkWindow *client_window,
	  const char *locale)
{
  GSList *tmp_list;
  GtkIIIMInfo *info;
  IIIMCF_attr attr;
  IIIMF_status st;
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
  GdkScreen *screen = gdk_drawable_get_screen (client_window);
  GdkDisplay *display = gdk_screen_get_display (screen);

  tmp_list = open_iiims;
  while (tmp_list)
    {
      info = tmp_list->data;
      if (info->screen == screen && (strcmp (info->locale, locale) == 0))
	return info;
      tmp_list = tmp_list->next;
    }
#else
  tmp_list = open_iiims;
  while (tmp_list)
    {
      info = tmp_list->data;
      if ((strcmp (info->locale, locale) == 0))
	return info;
      tmp_list = tmp_list->next;
    }
#endif
  info = NULL;

  if (!iiim_is_initialized)
    {
      st = iiimcf_initialize (IIIMCF_ATTR_NULL);
      if (st != IIIMF_STATUS_SUCCESS)
	goto Error;
      st = iiimcf_create_attr (&attr);
      if (st != IIIMF_STATUS_SUCCESS)
	goto Error;
      st = iiimcf_attr_put_string_value (attr, 
					 IIIMCF_ATTR_CLIENT_TYPE,
					 "Gtk IIIMCF Module");
      if (st != IIIMF_STATUS_SUCCESS)
	goto Error;
      st = iiimcf_create_handle (attr, &iiim);
      if (st != IIIMF_STATUS_SUCCESS)
	goto Error;
      st = iiimcf_destroy_attr (attr);
      if (iiim)
	{
	  iiim_is_initialized = TRUE;
	}
    }

 Error:	
  info = g_new0 (GtkIIIMInfo, 1);
  open_iiims = g_slist_prepend (open_iiims, info);
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
  info->screen = screen;
#endif
  info->locale = g_strdup (locale);
  info->iiim = iiim;
  info->ics = NULL;
#if 0
  iiim_get_im_list (info);
#endif
  iiim_get_languages (info);

  if (!info->iiim)
    g_warning ("Unable to Connect IIIM input method, falling back to XLookupString()");

#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
  g_signal_connect (display, "closed",
		    G_CALLBACK (iiim_info_display_closed), info);
#endif
  return info;
}

/* 
 * Candidate Window Handling
 */

static void
iiim_candidate_show (GtkIMContextIIIM *context_iiim)
{
  GtkWidget *w = iiim_get_candidate_window (context_iiim);
  IIIMF_status st;
  IIIMCF_lookup_choice luc;
  CandidateWindow *cw;
  GtkTreeIter iter;
  GtkTreeIter riter;
  int size;
  int first_candidate;
  int last_candidate;
  int current_candidate;
  int i,j;
  IIIMCF_text candidate, label;
  int flag;
  gint x,y;

  g_message ("candidate show");
  if (!w)
    {
      g_message ("w not found");
      return;
    }
  if (!context_iiim->candidate_start)
    {
      g_message ("candidate not start");
      return;
    }

  luc = iiim_get_lookup_choice (context_iiim);
  if (!luc)
    {
      g_message ("fail to obtain IIIMCF_lookup_choice");
      return;
    }

  cw = g_object_get_data (G_OBJECT (w), "iiim-candidate-win");
  if (!cw)
    {
      g_message ("candidate window not found");
      return;
    }
  /* get candidates's amount from IIIMSF */
  st = iiimcf_get_lookup_choice_size (luc, &size, &first_candidate, 
				      &last_candidate, &current_candidate);
  g_message ("size %d first %d last %d current %d",
	     size, first_candidate, 
	     last_candidate, current_candidate);
  /* clear */
  gtk_list_store_clear (cw->store);
  /* set iter */
  gtk_list_store_append (cw->store, &iter);
  /* adding candidate to treeview */
  for (i = first_candidate, j = 0; 
       i < (last_candidate - first_candidate + 1) ;
       i++, j++) {
    gchar *candidate_u8, *label_u8 = NULL , *result = NULL;
    /* get candidates from IIIMSF */
    st = iiimcf_get_lookup_choice_item (luc, i, &candidate, &label, &flag);
    if (st != IIIMF_STATUS_SUCCESS) break;
    if (label)
      label_u8 = utf16_to_utf8  (label);
    candidate_u8 = utf16_to_utf8 (candidate);
    g_message ("candidate %s", candidate_u8);
    if (label_u8)
      {
	result = g_strconcat (label_u8, " ", candidate_u8, NULL);
      }
    /* max columns */
    if (j == cw->number_of_columns)
      {
	/* set next row */
	gtk_list_store_insert_after (cw->store, &riter, &iter);
	iter = riter;
	j = 0;
      }
    gtk_list_store_set (cw->store, &iter, j , 
			result ? result : candidate_u8, -1);
    /* current candidate */
    if (i == current_candidate)
      {
	GtkTreeSelection *selection = gtk_tree_view_get_selection (
								   GTK_TREE_VIEW (cw->tree));
	gtk_tree_selection_select_iter (selection,
					&iter);
      }
    if (result)
      {
	g_free (result);
	g_free (label_u8);
      }
    g_free (candidate_u8);
  }
  gdk_window_get_origin (context_iiim->client_window, &x, &y);
  gtk_window_move (GTK_WINDOW (w), x + context_iiim->cursor.x , y + context_iiim->cursor.y 
		   + context_iiim->cursor.height);
  gtk_widget_show_all (w);
}

static GtkListStore *
iiim_create_candidate_model (int number_of_columns)
{
  GtkListStore *ret;
  GType *types;
  int i;

  g_message ("create_candidate_model");
  types = g_new0 (GType, number_of_columns);
  for (i = 0; i < number_of_columns; i++)
    {
      types[i] = G_TYPE_STRING;
    }
  ret = gtk_list_store_newv (number_of_columns,
			     types);
  g_free (types);
  return ret;
}

static GtkWidget *
iiim_create_candidate_window (GtkIMContextIIIM *context_iiim)
{
  CandidateWindow *w;
  GtkWidget *box;
  GtkListStore *store;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  IIIMCF_lookup_choice luc = iiim_get_lookup_choice (context_iiim);
  IIIMF_status st;
  IIIMCF_text title;
  GdkWindow *toplevel_gdk;
  GtkWidget *toplevel;
  GtkWidget *window;
  gchar *title_u8 = NULL;
  int choices_per_window;
  int number_of_rows;
  int number_of_columns;
  int direction;
  int i;

#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
  GdkScreen *screen;
  GdkWindow *root_window;
#endif

  if (!context_iiim->client_window)
    return NULL;

  toplevel_gdk = context_iiim->client_window;

#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
  screen = gdk_drawable_get_screen (toplevel_gdk);
  root_window = gdk_screen_get_root_window (screen);
#endif
  
  while (TRUE)
    {
      GdkWindow *parent = gdk_window_get_parent (toplevel_gdk);
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
      if (parent == root_window)
#else
	if (parent == gdk_get_default_root_window ())
#endif	      
	  break;
	else
	  toplevel_gdk = parent;
    }

  gdk_window_get_user_data (toplevel_gdk, (gpointer *)&toplevel);
  if (!toplevel)
    return NULL;

  g_message ("create candidate window");

  if (!luc)
    {
      g_message ("lookup choice not found");
      return NULL;
    }

  st = iiimcf_get_lookup_choice_configuration (luc,
					       &choices_per_window,
					       &number_of_rows,
					       &number_of_columns,
					       &direction);
  if (st != IIIMF_STATUS_SUCCESS)
    {
      g_message ("config failed");
      return NULL;
    }

  st = iiimcf_get_lookup_choice_title (luc, &title);
  if (st != IIIMF_STATUS_SUCCESS)
    {
      g_message ("Failed to get lookup choice title");
      return NULL;
    }

  if (title)
    title_u8 = utf16_to_utf8 (title);

  if ((number_of_columns < 0)
      || (number_of_rows < 0))
    {
      g_message ("column %d %d", number_of_columns, number_of_rows);
      return NULL;
    }
  store = iiim_create_candidate_model (number_of_columns);
  if (!store) {
    g_message ("create model failed");
    return NULL;
  }
  w = g_new0 (CandidateWindow, 1);
  w->toplevel = gtk_window_new (GTK_WINDOW_POPUP);
  gtk_container_set_border_width (GTK_CONTAINER (w->toplevel), 2);

  w->frame = gtk_frame_new (title_u8);
  gtk_frame_set_shadow_type (GTK_FRAME (w->frame),
			     GTK_SHADOW_ETCHED_OUT);

  w->tree = gtk_tree_view_new ();
  gtk_tree_view_set_model (GTK_TREE_VIEW (w->tree),
			   GTK_TREE_MODEL (store));
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (w->tree), FALSE);
  gtk_tree_view_columns_autosize (GTK_TREE_VIEW (w->tree));

  for (i = 0; i < number_of_columns; i++)
    {
      renderer = gtk_cell_renderer_text_new ();
      column = gtk_tree_view_column_new_with_attributes ("",
							 renderer, "text", i, NULL);
      gtk_tree_view_column_set_resizable (column, TRUE);
      gtk_tree_view_column_set_sizing (column,
				       GTK_TREE_VIEW_COLUMN_AUTOSIZE);
      gtk_tree_view_append_column (GTK_TREE_VIEW (w->tree), column);
    }
  gtk_tree_selection_set_mode (gtk_tree_view_get_selection (GTK_TREE_VIEW (w->tree)),
			       GTK_SELECTION_SINGLE);
  w->store = store;
  w->app_toplevel = toplevel;
  w->choices_per_window = choices_per_window;
  w->number_of_rows = number_of_rows;
  w->number_of_columns = number_of_columns;
  w->direction = direction;

  gtk_container_add (GTK_CONTAINER (w->toplevel), w->frame);
  gtk_container_add (GTK_CONTAINER (w->frame), w->tree);

  g_object_set_data (G_OBJECT (w->toplevel), "iiim-candidate-win", (gpointer)w);
  g_message ("create_candidate_window");

  if (title_u8)
    g_free (title_u8);


  w->configure_handler_id = g_signal_connect (toplevel, "configure_event",
					      G_CALLBACK (iiim_candidate_window_configure),
					      context_iiim);
  w->destroy_handler_id = g_signal_connect_swapped (toplevel, "destroy",
						    G_CALLBACK (iiim_destroy_candidate_window),
						    context_iiim);
  return w->toplevel;
}

static void
iiim_destroy_candidate_window (GtkIMContextIIIM *context_iiim)
{
  GtkWidget *w = context_iiim->candidate;
  CandidateWindow *cw;
  if (!w)
    return;
  cw = gtk_object_get_data (GTK_OBJECT (w), "iiim-candidate-win");
  g_signal_handler_disconnect (cw->app_toplevel, cw->destroy_handler_id);
  g_signal_handler_disconnect (cw->app_toplevel, cw->configure_handler_id);
  gtk_widget_destroy (cw->toplevel);
  g_free (cw);
  context_iiim->candidate = NULL;
  return;
}

static gboolean
iiim_candidate_window_configure (GtkWidget *toplevel,
				 GdkEventConfigure *event,
				 GtkIMContextIIIM *context_iiim)
{
  GtkWidget *candwin = iiim_get_candidate_window (context_iiim);
  gint x, y;

  if (!candwin)
    return FALSE;
  gdk_window_get_origin (context_iiim->client_window, &x, &y);
  gtk_window_move (GTK_WINDOW (candwin), x + context_iiim->cursor.x , y + context_iiim->cursor.y 
		   + context_iiim->cursor.height);
  return FALSE;
}

static GtkWidget *
iiim_get_candidate_window (GtkIMContextIIIM *context_iiim)
{
  GtkWidget *candidate = NULL;

  if (!context_iiim->candidate_start)
    {
      g_message ("candidate not started yet");
      return NULL;
    }
  if (context_iiim->candidate)
    {
      return context_iiim->candidate;
    }
  else
    {
      candidate = iiim_create_candidate_window (context_iiim);
      g_message ("candidate %p", candidate);
      context_iiim->candidate = candidate;
    }
  return candidate;
}

static IIIMCF_lookup_choice
iiim_get_lookup_choice (GtkIMContextIIIM *context_iiim)
{
  IIIMCF_context context = iiim_get_session_context (context_iiim);
  IIIMCF_lookup_choice luc;
  IIIMF_status st;

  if (context && context_iiim->candidate_start)
    {
      if (context_iiim->lookup_choice)
	return context_iiim->lookup_choice;
      st = iiimcf_get_lookup_choice (context, &luc);
      if (st != IIIMF_STATUS_SUCCESS)
	return NULL;
      context_iiim->lookup_choice = luc;
      return context_iiim->lookup_choice;
    }

  return NULL;
}

static IIIMCF_context
iiim_get_session_context (GtkIMContextIIIM *context_iiim)
{
  IIIMF_status st;
  IIIMCF_attr attr;

  if (!context_iiim->iiim_info->iiim) return NULL;
  if (!context_iiim->context && context_iiim->iiim_info)
    {
      st = iiimcf_create_attr (&attr);
      if (st != IIIMF_STATUS_SUCCESS) return NULL;
      if (context_iiim->iiim_info->lang)
	{
	  iiimcf_attr_put_ptr_value (attr, IIIMCF_ATTR_INPUT_LANGUAGE,
				     context_iiim->iiim_info->lang);
	}
      st = iiimcf_create_context (context_iiim->iiim_info->iiim, 
				  attr, &(context_iiim->context));
      if (st != IIIMF_STATUS_SUCCESS) return NULL;
      iiimcf_destroy_attr (attr);
    }

  return context_iiim->context;
}

static void
im_context_iiim_init (GtkIMContextIIIM *im_context_iiim)
{
  im_context_iiim->context = NULL;
  im_context_iiim->locale = NULL;
  im_context_iiim->le_name = NULL;
  im_context_iiim->candidate = NULL;
  im_context_iiim->candidate_start = FALSE;
  im_context_iiim->use_preedit = FALSE;
  im_context_iiim->trigger_notified = FALSE;
  im_context_iiim->status_visible = FALSE;
}

static gchar *
utf16_to_utf8 (IIIMCF_text text)
{
  IIIMF_status st;
  IIIMP_card16 *u16str;
  st = iiimcf_get_text_utf16string (text, (const IIIMP_card16 **)&u16str);
  return g_utf16_to_utf8 ((const gunichar2 *)u16str, -1, NULL, NULL, NULL);
}

static void
get_lookup_choice (GtkIMContext *context)
{
  GtkIMContextIIIM *context_iiim = GTK_IM_CONTEXT_IIIM (context);
  IIIMF_status st;
  IIIMCF_lookup_choice ilc;

  st = iiimcf_get_lookup_choice (context_iiim->context, &ilc);
}

static gboolean
im_context_iiim_filter_keypress (GtkIMContext *context ,
				 GdkEventKey *event)
{
  GtkIMContextIIIM *context_iiim = GTK_IM_CONTEXT_IIIM (context);
  gboolean result = FALSE;
  IIIMCF_context c;
  IIIMF_status st;
  IIIMCF_keyevent kev;
  IIIMCF_event ev;
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
  GdkWindow *root_window = gdk_screen_get_root_window (gdk_drawable_get_screen (event->window));
#else
  GdkWindow *root_window = gdk_get_default_root_window ();
#endif

  XKeyPressedEvent xevent;

  /* IIIMF doesn't recognize */
  if (event->type == GDK_KEY_RELEASE)
    return FALSE;

  c = iiim_get_session_context (context_iiim);
  if (!c) return FALSE;

  st = convert_GdkEventKey_to_IIIMCF_keyevent(event, &kev);
  if (st != IIIMF_STATUS_SUCCESS) return FALSE;
  st = iiimcf_create_keyevent (&kev, &ev);
  if (st != IIIMF_STATUS_SUCCESS) return FALSE;

  /* Send Message to IIIMSF */
  st = iiimcf_forward_event (c, ev);
  if (st == IIIMF_STATUS_EVENT_NOT_FORWARDED)
    {
      /* Failed */
      if (kev.keychar > 0x20 && kev.keychar != 0x7f)
	{
	  char ch[2];
	  ch[0] = kev.keychar;
	  ch[1] = '\0';
	  g_signal_emit_by_name (context_iiim, "commit", ch);
	  return TRUE;
	} 
      return FALSE;
    }
  else
    {
      int ui_flag;

      /* Check UI */
      st = iiimcf_is_UIstate_changed (c, &ui_flag);
      if (st != IIIMF_STATUS_SUCCESS) return FALSE;
      /* Preedit String */
      if (ui_flag & IIIMCF_STATE_PREEDIT_CHANGED)
	{
	  g_signal_emit_by_name (context_iiim, "preedit_changed");
	}
      /* Status String */
      if (ui_flag & IIIMCF_STATE_STATUS_CHANGED)
	{
	  iiim_status_update (context_iiim);
	}
#if 0
      /* Lookup Choice */
      if (ui_flag & IIIMCF_STATE_LOOKUP_CHOICE_CHANGED)
	{
	  get_lookup_choice (context);
	  g_message ("choice changed");
	}
#endif
      /* Commit String */
      if (ui_flag & IIIMCF_STATE_COMMIT_REQUIRED)
	{
	  IIIMCF_text text;
	  gchar *utf8 = NULL;

	  st = iiimcf_get_committed_text (c,  &text);
	  utf8 = utf16_to_utf8 (text);
	  g_signal_emit_by_name (context_iiim, "commit", utf8);
	  g_free (utf8);
	}
      return TRUE;
    }

  return result;
}

static void
set_sc_client_window (GtkIMContextIIIM *context_iiim,
		      GdkWindow *client_window,
		      gboolean send_signal)
{
  g_message ("set_sc_client_window");
  if (context_iiim->client_window)
    {
      context_iiim->iiim_info->ics = g_slist_remove (context_iiim->iiim_info->ics, context_iiim);
      context_iiim->iiim_info = NULL;
    }
  context_iiim->client_window = client_window;

  if (context_iiim->client_window)
    {
      context_iiim->iiim_info = get_iiim (context_iiim->client_window, context_iiim->locale);
      context_iiim->iiim_info->ics = g_slist_prepend (context_iiim->iiim_info->ics, context_iiim);
    }
}

static void
im_context_iiim_set_client_window (GtkIMContext *context,
				   GdkWindow *client_window)
{
  GtkIMContextIIIM *context_iiim = GTK_IM_CONTEXT_IIIM (context);
  IIIMF_status st;
  IIIMCF_event ev;

  set_sc_client_window (context_iiim, client_window, TRUE);
  if (!context_iiim->context)
    return;
#if 1
  st = iiimcf_create_trigger_notify_event (TRUE , &ev);
  if (st != IIIMF_STATUS_SUCCESS) return;
  iiimcf_forward_event (context_iiim->context , ev);
#endif
  g_message ("set client window");
}

static void
im_context_iiim_finalize (GObject *obj)
{
  GtkIMContextIIIM *context_iiim = GTK_IM_CONTEXT_IIIM (obj);

  set_sc_client_window (context_iiim, NULL, FALSE);

  g_free (context_iiim->locale);
  g_message ("im_context_iiim_finalize");

}

static void
reinitialize_sc (GtkIMContextIIIM *context_iiim,
		 gboolean send_signal)
{
  IIIMF_status st;
  if (context_iiim->context)
    {
      st = iiimcf_destroy_context (context_iiim->context);
      status_window_hide (context_iiim);
      context_iiim->context = NULL;
    }
}

static void
im_context_iiim_focus_in (GtkIMContext *context)
{
  IIIMF_status st;
  IIIMCF_event ev;
  IIIMCF_context c;
  GtkIMContextIIIM *context_iiim = GTK_IM_CONTEXT_IIIM (context);

  g_message ("im_context_iiim_focus_in");

  c = iiim_get_session_context (context_iiim);
  if (!c) return;

  status_window_show (context_iiim);
  iiim_status_update (context_iiim);

  st = iiimcf_create_seticfocus_event (&ev);
  if (st != IIIMF_STATUS_SUCCESS) return;
  iiimcf_forward_event (c, ev);

  iiim_event_dispatch (context_iiim);
  /* this causes SIGSEGV
     iiim_event_dispatch (context_iiim);
  */
  return;
}

static void
im_context_iiim_focus_out (GtkIMContext *context)
{
  GtkIMContextIIIM *context_iiim = GTK_IM_CONTEXT_IIIM (context);
  IIIMF_status st;
  IIIMCF_event ev;
  IIIMCF_context c;

  g_message ("im_context_iiim_focus_out");

  c = iiim_get_session_context (context_iiim);
  if (!c) return;

  status_window_hide (context_iiim);
  iiim_status_update (context_iiim);

  iiimcf_create_unseticfocus_event (&ev);
  if (st != IIIMF_STATUS_SUCCESS) return;

  iiimcf_forward_event (c ,ev);
  iiim_event_dispatch (context_iiim);

  return;
}

static void
im_context_iiim_set_cursor_location (GtkIMContext *context,
				     GdkRectangle *area)
{
  GtkIMContextIIIM *context_iiim = GTK_IM_CONTEXT_IIIM (context);
  g_message ("im_context_iiim_set_cursor_location");
  if (!context_iiim)
    return;
  context_iiim->cursor.x = area->x;
  context_iiim->cursor.y = area->y;
  context_iiim->cursor.height = area->height;
  context_iiim->cursor.width = area->width;
  return;
}

static void
im_context_iiim_set_use_preedit (GtkIMContext *context,
				 gboolean use_preedit)
{
  GtkIMContextIIIM *context_iiim = GTK_IM_CONTEXT_IIIM (context);
  g_message ("im_context_iiim_set_use_preedit");

  use_preedit = use_preedit != FALSE;

  if (context_iiim->use_preedit != use_preedit)
    {
      context_iiim->use_preedit = use_preedit;
      reinitialize_sc (context_iiim, TRUE);
    }
  return;
}

static void
im_context_iiim_reset (GtkIMContext *context)
{
  /* TODO */
  g_message ("im_context_iiim_reset");
}

static void
add_feedback_attr (PangoAttrList *attrs,
		   const gchar *str,
		   const IIIMP_card32 *feedback,
		   gint nfb,
		   gint start_pos,
		   gint end_pos)
{
  PangoAttribute *attr;
  gint start_index = g_utf8_offset_to_pointer (str, start_pos) - str;
  gint end_index = g_utf8_offset_to_pointer (str, end_pos) - str;
  gint i;

  g_message ("start %d end %d nfb %d", start_index, end_index, nfb);

  for (i = 0; i < nfb; i++)
    {
      g_message ("feedback %d", feedback[i]);

      if (feedback[i] == 2 /* Underline */)
	{
	  attr = pango_attr_underline_new (PANGO_UNDERLINE_SINGLE);
	  attr->start_index = start_index;
	  attr->end_index = end_index;
	
	  pango_attr_list_change (attrs, attr);
	}

      if (feedback[i] == 1 /* Reverse */)
	{
	  attr = pango_attr_foreground_new (0xffff, 0xffff, 0xffff);
	  attr->start_index = start_index;
	  attr->end_index = end_index;
	
	  pango_attr_list_change (attrs, attr);
	
	  attr = pango_attr_background_new (0, 0, 0);
	  attr->start_index = start_index;
	  attr->end_index = end_index;

	  pango_attr_list_change (attrs, attr);
	}
    }
}

static gboolean
iiim_check_feedback (const IIIMP_card32 *feedback, gint nfb,
		     const IIIMP_card32 *feedback2, gint nfb2)
{
  gint i;
  gboolean result = TRUE;
  if (!feedback)
    return FALSE;
  if (nfb != nfb2)
    return FALSE;
  for (i = 0; i < nfb; i++)
    {
      if (feedback[i] != feedback2[i])
	result = FALSE;
    }
  return result;
}

static void
im_context_iiim_get_preedit_string (GtkIMContext *context,
				    gchar **str,
				    PangoAttrList **attrs,
				    gint *cursor_pos)
{
  GtkIMContextIIIM *im_context_iiim = GTK_IM_CONTEXT_IIIM (context);
  IIIMF_status st;
  IIIMCF_text text;
  gint caret_position;
  gint preedit_len;
  gchar *utf8;

  if (attrs)
    *attrs = pango_attr_list_new ();

  if (!im_context_iiim->context)
    {
      g_message ("preedit_string context is not initialized");
      goto Error;
    }
  st = iiimcf_get_preedit_text (im_context_iiim->context, &text, &caret_position);
  if (st != IIIMF_STATUS_SUCCESS)
    goto Error;
  utf8 = utf16_to_utf8 (text);

  if (attrs)
    {
      gint i;
      gint j;
      gint start = -1;
      gint last_nfb = 0;
      const IIIMP_card32 *last_feedback = NULL;

      st = iiimcf_get_text_length (text, &preedit_len);
      if (st != IIIMF_STATUS_SUCCESS) goto Error;
      for (i = 0; i < preedit_len; i++)
	{
	  IIIMP_card16 ch;
	  const IIIMP_card32 *feedback_ids, *feedbacks;
	  int nfb;
	  st = iiimcf_get_char_with_feedback (text, i, &ch, &nfb, &feedback_ids, 
					      &feedbacks);
	  if (st != IIIMF_STATUS_SUCCESS) goto Error;
	  if (!iiim_check_feedback (last_feedback, last_nfb, feedbacks, nfb))
	    {
	      if (start >= 0)
		add_feedback_attr (*attrs, utf8, last_feedback, last_nfb, start,i);
	      last_feedback = feedbacks;
	      last_nfb = nfb;
	      start = i;
	    }
	}
      if (start >= 0)
	add_feedback_attr (*attrs, utf8, last_feedback, last_nfb, start, i);
    }

  if (str)
    *str = utf8;
  else
    g_free (utf8);

  g_message ("cursor %d", caret_position);
  if (cursor_pos)
    *cursor_pos = preedit_len - caret_position;
  return;

 Error:
  if (str)
    {
      *str = g_strdup ("");
    }
  if (cursor_pos)
    {
      *cursor_pos = 0;
    }

}

GtkIMContext *
im_context_iiim_new (void)
{
  GtkIMContextIIIM *result;
  gchar *locale = NULL;
  gchar *le = NULL;

  result = GTK_IM_CONTEXT_IIIM (g_object_new (GTK_TYPE_IM_CONTEXT_IIIM, NULL));

  /* Dirty Hack for Test Purpose */

  locale = getenv ("IIIM_LOCALE");
  le = getenv ("IIIM_LE");
  if (locale && *locale)
    {
      result->locale = g_strdup (locale);
    }
  else
    {
      result->locale = g_strdup (setlocale (LC_CTYPE, NULL));
    }
  if (le && *le)
    result->le_name = g_strdup (le);
  return GTK_IM_CONTEXT (result);
}

/**************************
 *                        *
 * Status Window handling *
 *                        *
 **************************/

static gboolean
status_window_expose_event (GtkWidget      *widget,
			    GdkEventExpose *event)
{
  gdk_draw_rectangle (widget->window,
		      widget->style->base_gc [GTK_STATE_NORMAL],
		      TRUE,
		      0, 0,
		      widget->allocation.width, widget->allocation.height);
  gdk_draw_rectangle (widget->window,
		      widget->style->text_gc [GTK_STATE_NORMAL],
		      FALSE,
		      0, 0,
		      widget->allocation.width - 1, widget->allocation.height - 1);

  return FALSE;
}

static void
status_window_style_set (GtkWidget *toplevel,
			 GtkStyle  *previous_style,
			 GtkWidget *label)
{
  gint i;
  
  for (i = 0; i < 5; i++)
    gtk_widget_modify_fg (label, i, &toplevel->style->text[i]);
}

/* Frees a status window and removes its link from the status_windows list */
static void
status_window_free (StatusWindow *status_window)
{
  status_windows = g_slist_remove (status_windows, status_window);
 
  g_signal_handler_disconnect (status_window->toplevel, status_window->destroy_handler_id);
  g_signal_handler_disconnect (status_window->toplevel, status_window->configure_handler_id);
  gtk_widget_destroy (status_window->window);
  g_object_set_data (G_OBJECT (status_window->toplevel), "gtk-im-xim-status-window", NULL);
 
  g_free (status_window);
}

static gboolean
status_window_configure (GtkWidget         *toplevel,
			 GdkEventConfigure *event,
  			 StatusWindow      *status_window)
{
  GdkRectangle rect;
  GtkRequisition requisition;
  gint y;

#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
  gint height = gdk_screen_get_height (gtk_widget_get_screen (toplevel));
#else
  gint height = gdk_screen_height ();
#endif

  gdk_window_get_frame_extents (toplevel->window, &rect);
  gtk_widget_size_request (status_window->window, &requisition);

  if (rect.y + rect.height + requisition.height < height)
    y = rect.y + rect.height;
  else
    y = height - requisition.height;
  
  gtk_window_move (GTK_WINDOW (status_window->window), rect.x, y);

  return FALSE;
}

static GtkWidget *
status_window_get (GtkIMContextIIIM *context_iiim,
		   gboolean         create)
{
  GdkWindow *toplevel_gdk;
  GtkWidget *toplevel;
  GtkWidget *window;
  StatusWindow *status_window;
  GtkWidget *status_label;
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
  GdkScreen *screen;
  GdkWindow *root_window;
#endif

  if (!context_iiim->client_window)
    return NULL;

  toplevel_gdk = context_iiim->client_window;

#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
  screen = gdk_drawable_get_screen (toplevel_gdk);
  root_window = gdk_screen_get_root_window (screen);
#endif
  
  while (TRUE)
    {
      GdkWindow *parent = gdk_window_get_parent (toplevel_gdk);
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION == 2
      if (parent == root_window)
#else
	if (parent == gdk_get_default_root_window ())
#endif	      
	  break;
	else
	  toplevel_gdk = parent;
    }

  gdk_window_get_user_data (toplevel_gdk, (gpointer *)&toplevel);
  if (!toplevel)
    return NULL;

  status_window = g_object_get_data (G_OBJECT (toplevel), "gtk-im-xim-status-window");
  if (status_window)
    return status_window->window;
  else if (!create)
    return NULL;

  status_window = g_new (StatusWindow, 1);
  status_window->window = gtk_window_new (GTK_WINDOW_POPUP);
  status_window->toplevel = toplevel;

  status_windows = g_slist_prepend (status_windows, status_window);

  window = status_window->window;

  gtk_window_set_resizable (GTK_WINDOW (window), FALSE);
  gtk_widget_set_app_paintable (window, TRUE);

  status_label = gtk_label_new ("");
  gtk_misc_set_padding (GTK_MISC (status_label), 1, 1);
  gtk_widget_show (status_label);
  
  gtk_container_add (GTK_CONTAINER (window), status_label);

  status_window->destroy_handler_id = g_signal_connect_swapped (toplevel, "destroy",
								G_CALLBACK (status_window_free),
								status_window);
  status_window->configure_handler_id = g_signal_connect (toplevel, "configure_event",
							  G_CALLBACK (status_window_configure),
							  status_window);

  status_window_configure (toplevel, NULL, status_window);

  g_signal_connect (window, "style_set",
		    G_CALLBACK (status_window_style_set), status_label);
  g_signal_connect (window, "expose_event",
		    G_CALLBACK (status_window_expose_event), NULL);

  g_object_set_data (G_OBJECT (toplevel), "gtk-im-xim-status-window", status_window);

  return window;
}

static gboolean
status_window_has_text (GtkWidget *status_window)
{
  GtkWidget *label = GTK_BIN (status_window)->child;
  const gchar *text = gtk_label_get_text (GTK_LABEL (label));

  return text[0] != '\0';
}

static void
status_window_show (GtkIMContextIIIM *context_iiim)
{
  context_iiim->status_visible = TRUE;
}

static void
status_window_hide (GtkIMContextIIIM *context_iiim)
{
  GtkWidget *status_window = status_window_get (context_iiim, FALSE);

  context_iiim->status_visible = FALSE;

  if (status_window)
    status_window_set_text (context_iiim, "");
}

static void
status_window_set_text (GtkIMContextIIIM *context_iiim,
			const gchar     *text)
{
  GtkWidget *status_window = status_window_get (context_iiim, TRUE);

  if (status_window)
    {
      GtkWidget *label = GTK_BIN (status_window)->child;
      gtk_label_set_text (GTK_LABEL (label), text);
      
      if (context_iiim->status_visible && status_window_has_text (status_window))
	gtk_widget_show (status_window);
      else
	gtk_widget_hide (status_window);
    }
}

/**
 * im_context_iiim_shutdown:
 *
 * Destroys all the status windows that are kept by the IIIM contexts. This
 * function should only be called by the IIIM module exit routine.
 **/
void
im_context_iiim_shutdown (void)
{
  g_message ("shutdown\n");
  if (iiim)
    iiimcf_destroy_handle (iiim);
  iiimcf_finalize ();
  while (status_windows)
    status_window_free (status_windows->data);
}

/* Local Variables: */
/* c-file-style: "gnu" */
/* End: */
