/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ireducepolydatafilter2.h"


#include "vtkCellArray.h"
#include "vtkEdgeTable.h"
#include "vtkDoubleArray.h"
#include "vtkIdList.h"
#include "vtkMath.h"
#include "vtkPointData.h"
#include "vtkPriorityQueue.h"

//
//  Templates
//
#include "igenericfiltertemplate.h"


iReducePolyDataFilter2::iReducePolyDataFilter2(iViewSubject *vo) : iGenericPolyDataToPolyDataFilter<vtkQuadricDecimation>(vo,1,true,false)
{
}
	

void iReducePolyDataFilter2::ProduceOutput()
{
	vtkPolyData *input = vtkPolyData::SafeDownCast(this->GetInput());
	vtkPolyData *output = this->GetOutput();

	//
	//  The code between === lines is copied from vtkQuadricDecimation.cxx file with minor modifications, as marked behind /// comments
	//  Below is the original copyright notice.
	//
	//  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
	//  All rights reserved.
	//  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.
	//
	// =========================================================================
	//
	vtkIdType numPts = input->GetNumberOfPoints();
	vtkIdType numTris = input->GetNumberOfPolys();
	vtkIdType edgeId, i;
	int j;
	double cost;
	double *x;
	vtkCellArray *polys;
	vtkDataArray *attrib;
	vtkPoints *points;
	vtkPointData *pointData;
	vtkIdType endPtIds[2];
	vtkIdList *outputCellList;
	vtkIdType npts, *pts;
	vtkIdType numDeletedTris=0;

	// check some assuptiona about the data
	if (input->GetPolys() == NULL || input->GetPoints() == NULL || 
		input->GetPointData() == NULL  || input->GetFieldData() == NULL)
	{
		vtkErrorMacro("Nothing to decimate");
		return;
	}

	if (input->GetPolys()->GetMaxCellSize() > 3) 
	{
		vtkErrorMacro("Can only decimate triangles");
		return;
	}

	polys = vtkCellArray::New();

	/// ------ Removed ------------------------------------------
	/// points = vtkPoints::New();
	/// ---------------------------------------------------------

	/// ++++++ Added ++++++++++++++++++++++++++++++++++++++++++++
	points = vtkPoints::New(input->GetPoints()->GetDataType());
	/// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++

	pointData = vtkPointData::New();
	outputCellList = vtkIdList::New();

	// copy the input (only polys) to our working mesh
	this->Mesh = vtkPolyData::New();
	points->DeepCopy(input->GetPoints());
	this->Mesh->SetPoints(points);
	points->Delete();
	polys->DeepCopy(input->GetPolys());
	this->Mesh->SetPolys(polys);
	polys->Delete();
	if (this->AttributeErrorMetric) 
	{
		this->Mesh->GetPointData()->DeepCopy(input->GetPointData());
	}
	pointData->Delete();
	this->Mesh->GetFieldData()->PassData(input->GetFieldData());
	this->Mesh->BuildCells();
	this->Mesh->BuildLinks();

	this->ErrorQuadrics = 
		new vtkQuadricDecimation::ErrorQuadric[numPts];

	vtkDebugMacro(<<"Computing Edges");
	this->Edges->InitEdgeInsertion(numPts, 1); // storing edge id as attribute
	this->EdgeCosts->Allocate(this->Mesh->GetPolys()->GetNumberOfCells() * 3);
	for (i = 0; i <  this->Mesh->GetNumberOfCells(); i++) 
	{
		this->Mesh->GetCellPoints(i, npts, pts); 

		for (j = 0; j < 3; j++)
		{
			if (this->Edges->IsEdge(pts[j], pts[(j+1)%3]) == -1)
			{
				// If this edge has not been processed, get an id for it, add it to
				// the edge list (Edges), and add its endpoints to the EndPoint1List
				// and EndPoint2List (the 2 endpoints to different lists).
				edgeId = this->Edges->GetNumberOfEdges();
				this->Edges->InsertEdge(pts[j], pts[(j+1)%3], edgeId);
				this->EndPoint1List->InsertId(edgeId, pts[j]);
				this->EndPoint2List->InsertId(edgeId, pts[(j+1)%3]);
			}
		}
	}

	this->UpdateProgress(0.1);

	this->NumberOfComponents = 0;  
	if (this->AttributeErrorMetric) 
	{
		this->ComputeNumberOfComponents();
	}
	x = new double [3+this->NumberOfComponents];
	this->CollapseCellIds = vtkIdList::New();
	this->TempX = new double [3+this->NumberOfComponents];
	this->TempQuad = new double[11 + 4 * this->NumberOfComponents];

	this->TempB = new double [3 +  this->NumberOfComponents];
	this->TempA = new double*[3 +  this->NumberOfComponents];
	this->TempData = new double [(3 +  this->NumberOfComponents)*(3 +  this->NumberOfComponents)];
	for (i = 0; i < 3 +  this->NumberOfComponents; i++) 
	{
		this->TempA[i] = this->TempData+i*(3 +  this->NumberOfComponents);
	}
	this->TargetPoints->SetNumberOfComponents(3+this->NumberOfComponents);

	vtkDebugMacro(<<"Computing Quadrics");
	this->InitializeQuadrics(numPts);
	this->AddBoundaryConstraints();
	this->UpdateProgress(0.15);

	vtkDebugMacro(<<"Computing Costs");
	// Compute the cost of and target point for collapsing each edge.
	for (i = 0; i < this->Edges->GetNumberOfEdges(); i++)
	{
		if (this->AttributeErrorMetric) 
		{
			cost = this->ComputeCost2(i, x);
		}
		else
		{
			cost = this->ComputeCost(i, x);
		}
		this->EdgeCosts->Insert(cost, i);
		this->TargetPoints->InsertTuple(i, x);
	}
	this->UpdateProgress(0.20);

	// Okay collapse edges until desired reduction is reached
	this->ActualReduction = 0.00f;
	this->NumberOfEdgeCollapses = 0;
	edgeId = this->EdgeCosts->Pop(0,cost);

	int abort = 0;
	while ( !abort && edgeId >= 0 && cost < VTK_DOUBLE_MAX &&
		this->ActualReduction < this->TargetReduction ) 
	{
		if ( ! (this->NumberOfEdgeCollapses % 10000) ) 
		{
			vtkDebugMacro(<<"Collapsing edge#" << this->NumberOfEdgeCollapses);
			this->UpdateProgress (0.20 + 0.80*this->NumberOfEdgeCollapses/numPts);
			abort = this->GetAbortExecute();
		}

		endPtIds[0] = this->EndPoint1List->GetId(edgeId);
		endPtIds[1] = this->EndPoint2List->GetId(edgeId);
		this->TargetPoints->GetTuple(edgeId, x);

		// check for a poorly placed point
		if ( !this->IsGoodPlacement(endPtIds[0], endPtIds[1], x)) 
		{
			vtkDebugMacro(<<"Poor placement detected " << edgeId << " " <<  cost);
			// return the point to the queue but with the max cost so that 
			// when it is recomputed it will be reconsidered
			this->EdgeCosts->Insert(VTK_DOUBLE_MAX, edgeId);

			edgeId = this->EdgeCosts->Pop(0, cost);
			continue;
		}

		this->NumberOfEdgeCollapses++;

		// Set the new coordinates of point0.
		this->SetPointAttributeArray(endPtIds[0], x);
		vtkDebugMacro(<<"Cost: " << cost << " Edge: " 
			<< endPtIds[0] << " " << endPtIds[1]);

		// Merge the quadrics of the two points.
		this->AddQuadric(endPtIds[1], endPtIds[0]);

		this->UpdateEdgeData(endPtIds[0], endPtIds[1]);

		// Update the output triangles.
		numDeletedTris += this->CollapseEdge(endPtIds[0], endPtIds[1]);
		this->ActualReduction = (double) numDeletedTris / numTris;
		edgeId = this->EdgeCosts->Pop(0, cost);
	}

	vtkDebugMacro(<<"Number Of Edge Collapses: "
		<< this->NumberOfEdgeCollapses << " Cost: " << cost);

	// clean up working data
	for (i = 0; i < numPts; i++)
	{
		delete [] this->ErrorQuadrics[i].Quadric;
	}
	delete [] this->ErrorQuadrics;
	delete [] x;
	this->CollapseCellIds->Delete();
	delete [] this->TempX;
	delete [] this->TempQuad;
	delete [] this->TempB;
	delete [] this->TempA;
	delete [] this->TempData;

	// copy the simplified mesh from the working mesh to the output mesh
	for (i = 0; i < this->Mesh->GetNumberOfCells(); i++) 
	{
		if (this->Mesh->GetCell(i)->GetCellType() != VTK_EMPTY_CELL) 
		{
			outputCellList->InsertNextId(i);
		}
	} 

	output->Reset();

	/// ++++++ Added ++++++++++++++++++++++++++++++++++++++++++++
	output->SetPoints(points);
	/// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++

	output->Allocate(this->Mesh, outputCellList->GetNumberOfIds());
	output->GetPointData()->CopyAllocate(this->Mesh->GetPointData(),1);
	output->CopyCells(this->Mesh, outputCellList);

	this->Mesh->DeleteLinks();
	this->Mesh->Delete();
	outputCellList->Delete();

	// renormalize, clamp attributes
	if (this->AttributeErrorMetric) 
	{
		if (NULL != (attrib = output->GetPointData()->GetNormals())) 
		{
			for (i = 0; i < attrib->GetNumberOfTuples(); i++) 
			{
				vtkMath::Normalize(attrib->GetTuple3(i));
			}
		}
		// might want to add clamping texture coordinates??
	}
	//
	// =========================================================================
	//
}
