/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ipiecewisefunction.h"

#include "ierror.h"
#include "imath.h"
#include "ivtk.h"

#include <vtkPiecewiseFunction.h>

//
//  Templates
//
#include "iarraytemplate.h"


class iPiecewiseFunctionHelper : public vtkPiecewiseFunction
{

public:

	iPiecewiseFunctionHelper(iPiecewiseFunction *p)
	{
		mParent = p;
	}

	virtual void Update()
	{
		int i;

		this->RemoveAllPoints();
		for(i=0; i<mParent->mArr.Size(); i++) this->AddPoint(255*mParent->mArr[i].X,mParent->mArr[i].Y);

		vtkPiecewiseFunction::Update();
	}

	void ParentChanged()
	{
#ifdef IVTK_4
		this->Modified();
#else
		this->Update(); // Very inefficient, but needed to work around a VTK 5.0.0 bug
#endif
		mParent->Modified();
	}

private:

	iPiecewiseFunction *mParent;
};


//
// iPiecewiseFunction class
//
iPiecewiseFunction* iPiecewiseFunction::New(float ymin, float ymax)
{
	return new iPiecewiseFunction(ymin,ymax,10);
}


iPiecewiseFunction::iPiecewiseFunction(float ymin, float ymax, int inc) : mArr(inc)
{
	mYmin = ymin;
	mYmax = ymax;
	mArr.Add(Point(0.0,ymin));
	mArr.Add(Point(1.0,ymax));
	
	mHelper = new iPiecewiseFunctionHelper(this); IERROR_ASSERT(mHelper);
	mHelper->ParentChanged();
}


iPiecewiseFunction::~iPiecewiseFunction()
{
	mHelper->Delete();
}


vtkPiecewiseFunction* iPiecewiseFunction::GetVTKFunction() const
{
	return mHelper;
}


void iPiecewiseFunction::Copy(const iPiecewiseFunction *f)
{
	if(this==f || f==0) return;

	int i;
	for(i=0; i<mArr.Size() && i<f->mArr.Size(); i++) mArr[i] = f->mArr[i];

	while(mArr.Size() < f->mArr.Size()) mArr.Add(f->mArr[mArr.Size()]);
	while(mArr.Size() > f->mArr.Size()) mArr.RemoveLast();

	mHelper->ParentChanged();
}


void iPiecewiseFunction::SetMinMax(float ymin, float ymax)
{
	mYmin = ymin;
	mYmax = ymax;
	for(int i=0; i<mArr.Size(); i++) 
	{
		mArr[i].Y = (mArr[i].Y > mYmax) ? mYmax : mArr[i].Y;
		mArr[i].Y = (mArr[i].Y < mYmin) ? mYmin : mArr[i].Y;
	}

	mHelper->ParentChanged();
}


void iPiecewiseFunction::AddPoint(int n) 
{
	n++;
	if(n < 1) n = 1;
	if(n > mArr.MaxIndex()) n = mArr.MaxIndex();

	mArr.Add(Point(0.5*(mArr[n-1].X+mArr[n].X),0.5*(mArr[n-1].Y+mArr[n].Y)));
	mHelper->ParentChanged();
}


void iPiecewiseFunction::AddPoint(float x, float y) 
{
	if(x>mArr[0].X && x<mArr.Last().X) 
	{
		mArr.Add(Point(x,y));
		mHelper->ParentChanged();
	}
}


void iPiecewiseFunction::RemovePoint(int n) 
{
	if(n>0 && n<mArr.MaxIndex() && mArr.Size()>2) 
	{
		mArr.iArray<Point>::Remove(n);
		mHelper->ParentChanged();
	}
}


void iPiecewiseFunction::MovePoint(int n, float x, float y) 
{
	if(n>0 && n<mArr.MaxIndex()) 
	{
		float xl = mArr[n-1].X;
		float xr = mArr[n+1].X;
		x = (x < xl) ? xl : x;
		x = (x > xr) ? xr : x;
		y = (y < mYmin) ? mYmin : y;
		y = (y > mYmax) ? mYmax : y;
		mArr[n] = Point(x,y);
	} 
	else if(n==0 || n==mArr.MaxIndex()) 
	{
		y = (y < mYmin) ? mYmin : y;
		y = (y > mYmax) ? mYmax : y;
		mArr[n].Y = y;
	}

	mHelper->ParentChanged();
}


int iPiecewiseFunction::FindPoint(float x, float y, float &dmin) const
{
	int i, imin;
	float d, dx, dy;
	
	dmin = iMath::_LargeFloat;
	imin = -1;
	for(i=0; i<mArr.Size(); i++) 
	{
		dx = x - mArr[i].X;
		dy = y - mArr[i].Y;
		d = dx*dx + dy*dy;
		if(d < dmin) 
		{
			dmin = d;
			imin = i;
		}
	}
	
	dmin = sqrt(dmin);
	return imin;
}


float iPiecewiseFunction::GetValue(float x) const
{
	int i;
	
	if(x < mArr[0].X) return mArr[0].Y;
	if(x > mArr.Last().X) return mArr.Last().Y;

	for(i=1; i<mArr.MaxIndex() && mArr[i].X<x; i++);
	return mArr[i].Y + (mArr[i].Y-mArr[i-1].Y)/(mArr[i].X-mArr[i-1].X+1.0e-30)*(x-mArr[i].X);
}


void iPiecewiseFunction::Reset()
{
	while(mArr.Size() > 2) mArr.RemoveLast();
	mArr[0] = Point(0.0,mYmin);
	mArr[1] = Point(1.0,mYmax);

	mHelper->ParentChanged();
}

