/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iviewmodule.h"


#include "iactor.h"
#include "iactorcollection.h"
#include "ianimator.h"
#include "icamera.h"
#include "icaptioninteractorstyle.h"
#include "iclipplane.h"
#include "icolorbars.h"
#include "icontrolmodule.h"
#include "icrosssectionviewobject.h"
#include "icubeaxesactor.h"
#include "idata.h"
#include "idatahandler.h"
#include "idatareader.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ieventobserver.h"
#include "iextensionfactory.h"
#include "ifile.h"
#include "iimagecomposer.h"
#include "iinteractoreventrecorder.h"
#include "ikeyboardinteractorstyle.h"
#include "ilegend.h"
#include "ilightkit.h"
#include "imarkerviewobject.h"
#include "imath.h"
#include "imeasuringbox.h"
#include "imeasuringboxinteractorstyle.h"
#include "irendertool.h"
#include "ipalette.h"
#include "iparallelmanager.h"
#include "iparticlesviewobject.h"
#include "ipicker.h"
#include "ipointglyph.h"
#include "iposition.h"
#include "ishell.h"
#include "isurfaceviewobject.h"
#include "iruler.h"
#include "ishellfactory.h"
#include "itensorfieldviewobject.h"
#include "itextactor.h"
#include "iviewobjectfamily.h"
#include "ivectorfieldviewobject.h"
#include "ivolumeviewobject.h"
#include "iwriter.h"

#include <vtkActor2D.h>
#include <vtkArrowSource.h>
#include <vtkCamera.h>
#include <vtkCellArray.h>
#include <vtkCylinderSource.h>
#include <vtkCubeSource.h>
#include <vtkDiskSource.h>
#include <vtkFloatArray.h>
#include <vtkImageData.h>
#include <vtkInteractorStyleFlight.h>
#include <vtkInteractorStyleJoystickCamera.h>
#include <vtkInteractorStyleTrackballCamera.h>
#include <vtkInteractorStyleUnicam.h>
#include <vtkLight.h>
#include <vtkMapper.h>
#include <vtkMath.h>
#include <vtkMatrix4x4.h>
#include <vtkPointData.h>
#include <vtkPointWidget.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkPropAssembly.h>
#include <vtkProperty2D.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowCollection.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkTextProperty.h>
#include <vtkTimerLog.h>

//
//  Templates
//
#include "iarraytemplate.h"
#include "ifamilytemplate.h"
#include "iviewobjectfamilytemplate.h"


using namespace iParameter;


//
//  Define keys
//
IOBJECT_DEFINE_TYPE(iViewModule,ViewModule,vm,iObjectType::_Module);

IOBJECT_DEFINE_KEY(iViewModule,Antialiasing,a,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,AxesBoxLabels,abl,String,3);
IOBJECT_DEFINE_KEY(iViewModule,AxesBoxRanges,abr,Float,6);
IOBJECT_DEFINE_KEY(iViewModule,AnimationOutput,ao,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,BoundingBox,bb,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,BoundingBoxType,bbt,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,BackgroundColor,bg,Color,1);
IOBJECT_DEFINE_KEY(iViewModule,BoxSize,bs,Float,1);
IOBJECT_DEFINE_KEY(iViewModule,ColorBars,cb,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,CloneOfWindow,co,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,ClipPlane,cp,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,ClipPlaneDirection,cpd,Float,3);
IOBJECT_DEFINE_KEY(iViewModule,ClipPlaneDistance,cpl,Float,1);
IOBJECT_DEFINE_KEY(iViewModule,DumpImage,di,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,FontSize,fs,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,FontType,ft,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,GlassClipPlane,gcp,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,OpenGLCoordinates,glc,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,ImageFormat,if,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,InteractorStyle,is,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,ImageMagnification,ix,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,LightAngles,la,Float,2);
IOBJECT_DEFINE_KEY(iViewModule,Label,lb,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,LabelDigits,ld,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,LightIntensity,li,Float,3);
IOBJECT_DEFINE_KEY(iViewModule,LabelName,ln,String,1);
IOBJECT_DEFINE_KEY(iViewModule,LabelScale,ls,Float,1);
IOBJECT_DEFINE_KEY(iViewModule,LabelUnit,lu,String,1);
IOBJECT_DEFINE_KEY(iViewModule,MeasuringBox,mb,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,MarkerLegend,ml,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,MarkerLegendPosition,mlp,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,MoveMarkerCaption,mmc,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,NoClone,nc,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,MarkerCurrent,omc,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,MarkerMax,omm,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,ParticlesCurrent,opc,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,ParticlesMax,opm,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,SurfaceCurrent,osc,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,SurfaceMax,osm,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,CrossSectionCurrent,oxc,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,CrossSectionMax,oxm,OffsetInt,1);
IOBJECT_DEFINE_KEY(iViewModule,PlaceMarker,pm,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,Position,p,Int,2);
IOBJECT_DEFINE_KEY(iViewModule,PostScriptPaperFormat,psf,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,PostScriptOrientation,pso,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,Ruler,rl,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,RulerScale,rs,Float,1);
IOBJECT_DEFINE_KEY(iViewModule,Size,s,Int,2);
IOBJECT_DEFINE_KEY(iViewModule,UpdateRate,r,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,StereoAlignmentMarkers,sam,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,Stereo,ss,Bool,1);
IOBJECT_DEFINE_KEY(iViewModule,StereoType,st,Int,1);
IOBJECT_DEFINE_KEY(iViewModule,WindowNumber,wn,Int,1);

IOBJECT_DEFINE_KEY(iViewModule,JustifyLabelLeft,-lj,Bool,1);


//
//  Main class
//
iViewModule* iViewModule::New(iControlModule *cm)
{
	return iExtensionFactory::CreateViewModule(cm);
}


iViewModule::iViewModule(iControlModule *cm) : iControlModuleComponent(cm)
{
	int i;
	
	mFontSize = 16;
	mFontType = _TextTypeBitmap;
	mBoxType = -1; // needs to be reset
	mImageMagnification = 1;
	mSavedProjection = 0;

	mIsDebug = false;
	mAntialiasing = true;
	mInImageRender = false;

	mRate = 0;
	mBoxSize = -1.0;
	mCloneOfModule = 0;
	mWinNum = 0;

	mIsBoundingBoxOn = true;
	mIsLabelOn = true;
	mIsColorBarsOn = true;
	mIsMeasuringBoxOn = false;
	mIsRulerOn = false;
	mIsClipPlaneOn = false;
	mIsGlassClipPlaneOn = false;
	mIsStereoAlignmentMarkersOn = true;

	mLabelJustificationLeft = false;
	mStereoType = VTK_STEREO_RED_BLUE;

	mLabelName = "Record";
	mLabelUnit = "";
	mLabelScale = 1.0;
	mLabelDigits = 2;

	int bc = 255;
	mBgColor = iColor(bc,bc,bc);

	//
	//  Create render view
	//
	mRenderTool = 0; // must have this zeroing - who knows what happens inside the call to iExtensionFactory::CreateRenderTool
	mRenderTool = iExtensionFactory::CreateRenderTool(this); IERROR_ASSERT_NULL_POINTER(mRenderTool);

	//
	//  File reader, gateway, volume converter
	//
	mDataReader = iDataReader::New(this); IERROR_ASSERT_NULL_POINTER(mDataReader);
	mAnimator = iAnimator::New(this); IERROR_ASSERT_NULL_POINTER(mAnimator);
	
	//
	//  Image writer
	//
	mWriter = iWriter::New(this); IERROR_ASSERT_NULL_POINTER(mWriter); 

	//
	//  Bounding boxes
	//
	iActor *actor[12];

	//
	//  Bounding box 1
	//
	static float x[8][3]={{-1,-1,-1}, {1,-1,-1}, {1,1,-1}, {-1,1,-1}, {-1,-1,1}, {1,-1,1}, {1,1,1}, {-1,1,1}};
	static vtkIdType lns[12][2]={{0,1}, {1,2}, {2,3}, {3,0}, {0,4}, {1,5}, {2,6}, {3,7}, {4,5}, {5,6}, {6,7}, {7,4}};

	vtkPolyData *pd = vtkPolyData::New(); IERROR_ASSERT_NULL_POINTER(pd);
	vtkPoints *points = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT_NULL_POINTER(points);
	vtkCellArray *polys = vtkCellArray::New(); IERROR_ASSERT_NULL_POINTER(polys);
	vtkFloatArray *scalars = vtkFloatArray::New(); IERROR_ASSERT_NULL_POINTER(scalars);
	//
	// Load the point, cell, and data attributes.
	//
	for (i=0; i<8; i++) points->InsertPoint(i,x[i]);
	for (i=0; i<8; i++) scalars->InsertTuple1(i,(i<4));
	for (i=0; i<12; i++) polys->InsertNextCell(2,lns[i]);
	//
	// We now assign the pieces to the vtkPolyData.
	//
	pd->SetLines(polys);
	polys->Delete();
	pd->SetPoints(points);
	points->Delete();
	pd->GetPointData()->SetScalars(scalars);
	scalars->Delete();
	//
	//  Create the data and assign to the mapper
	//
	pd->Update();
	//
	//  Configure the actor
	//
	mBox1Actor = iActor::New(); IERROR_ASSERT_NULL_POINTER(mBox1Actor);
	mBox1Actor->PickableOff();
	mBox1Actor->VisibilityOff();
//	mBox1Actor->SetScalarRange(0,1);
	mBox1Actor->GetProperty()->SetAmbient(1.0);
	mBox1Actor->GetProperty()->SetDiffuse(0.0);
	mBox1Actor->GetProperty()->SetSpecular(0.0);
	mBox1Actor->GetProperty()->SetLineWidth(4.0);
	mBox1Actor->GetProperty()->SetColor(0.0,0.0,0.0);
	mBox1Actor->SetInput(pd);
	pd->Delete();

	mRenderTool->AddObject(mBox1Actor);

	//
	//  Bounding box 2
	//
	vtkCylinderSource *cs;
	float diam = 0.02;
	for(i=0; i<12; i++) 
	{
		actor[i] = iActor::New(); IERROR_ASSERT_NULL_POINTER(actor[i]);
		actor[i]->PickableOff();
		cs = vtkCylinderSource::New(); IERROR_ASSERT_NULL_POINTER(cs);
		cs->SetResolution(6);
		cs->SetHeight(2.0+2.0*diam);
		cs->SetRadius(diam);
		cs->Update();
		actor[i]->SetInput(cs->GetOutput());
		cs->Delete();
	}
	
	for(i=0; i<4; i++) 
	{
		actor[i]->GetProperty()->SetColor(0.0,0.0,1.0);
		actor[i]->RotateX(90.0);
	}
	for(i=4; i<8; i++) actor[i]->GetProperty()->SetColor(0.0,1.0,0.0);
	for(i=8; i<12; i++) 
	{
		actor[i]->GetProperty()->SetColor(1.0,0.0,0.0);
		actor[i]->RotateZ(90.0);
	}
	
	actor[ 0]->SetPosition(-1.0,-1.0,0.0);
	actor[ 1]->SetPosition(-1.0, 1.0,0.0);
	actor[ 2]->SetPosition( 1.0,-1.0,0.0);
	actor[ 3]->SetPosition( 1.0, 1.0,0.0);
	actor[ 4]->SetPosition(-1.0,0.0,-1.0);
	actor[ 5]->SetPosition(-1.0,0.0, 1.0);
	actor[ 6]->SetPosition( 1.0,0.0,-1.0);
	actor[ 7]->SetPosition( 1.0,0.0, 1.0);
	actor[ 8]->SetPosition(0.0,-1.0,-1.0);
	actor[ 9]->SetPosition(0.0,-1.0, 1.0);
	actor[10]->SetPosition(0.0, 1.0,-1.0);
	actor[11]->SetPosition(0.0, 1.0, 1.0);
	
	mBox2Actor = iActorCollection::New(); IERROR_ASSERT_NULL_POINTER(mBox2Actor);
	for(i=0; i<12; i++) 
	{
		actor[i]->GetProperty()->SetAmbient(0.2);
		actor[i]->GetProperty()->SetDiffuse(0.2);
		actor[i]->GetProperty()->SetSpecular(0.2);
		mBox2Actor->AddActor(actor[i]);
		mRenderTool->AddObject(actor[i]);
		actor[i]->Delete();
	}
	mBox2Actor->SetVisibility(false);

	//
	//  Bounding box 3
	//
	diam = 0.02;
	vtkArrowSource *as;
	for(i=0; i<3; i++) 
	{
		actor[i] = iActor::New(); IERROR_ASSERT_NULL_POINTER(actor[i]);
		actor[i]->PickableOff();
		as = vtkArrowSource::New(); IERROR_ASSERT_NULL_POINTER(as);
		as->SetShaftResolution(6);
		as->SetTipResolution(6);
		as->SetTipLength(0.15);
		as->SetShaftRadius(0.5*diam);
		as->SetTipRadius(1.5*diam);
		as->Update();
		actor[i]->SetInput(as->GetOutput());
		as->Delete();
		actor[i]->SetAxisScale(2.5,2.5,2.5);
	}
	
	diam *= 0.75;
	for(i=3; i<12; i++) 
	{
		actor[i] = iActor::New(); IERROR_ASSERT_NULL_POINTER(actor[i]);
		actor[i]->PickableOff();
		cs = vtkCylinderSource::New(); IERROR_ASSERT_NULL_POINTER(cs);
		cs->SetResolution(6);
		cs->SetHeight(2.0+2.0*diam);
		cs->SetRadius(diam);
		actor[i]->SetInput(cs->GetOutput());
		cs->Delete();
	}
	
	actor[0]->GetProperty()->SetColor(0.0,0.0,1.0);
	actor[0]->RotateY(-90.0);
	actor[1]->GetProperty()->SetColor(0.0,1.0,0.0);
	actor[1]->RotateZ(90.0);
	actor[2]->GetProperty()->SetColor(1.0,0.0,0.0);
	
	for(i=3; i<6; i++) 
	{
		actor[i]->GetProperty()->SetColor(0.0,0.0,1.0);
		actor[i]->RotateX(90.0);
	}
	for(i=6; i<9; i++) actor[i]->GetProperty()->SetColor(0.0,1.0,0.0);
	for(i=9; i<12; i++) 
	{
		actor[i]->GetProperty()->SetColor(1.0,0.0,0.0);
		actor[i]->RotateZ(90.0);
	}
	
	actor[ 0]->SetPosition(-1.0,-1.0,-1.0);
	actor[ 1]->SetPosition(-1.0,-1.0,-1.0);
	actor[ 2]->SetPosition(-1.0,-1.0,-1.0);
	
	actor[ 3]->SetPosition(-1.0, 1.0,0.0);
	actor[ 4]->SetPosition( 1.0,-1.0,0.0);
	actor[ 5]->SetPosition( 1.0, 1.0,0.0);
	
	actor[ 6]->SetPosition(-1.0,0.0, 1.0);
	actor[ 7]->SetPosition( 1.0,0.0,-1.0);
	actor[ 8]->SetPosition( 1.0,0.0, 1.0);
	
	actor[ 9]->SetPosition(0.0, 1.0,-1.0);
	actor[10]->SetPosition(0.0,-1.0, 1.0);
	actor[11]->SetPosition(0.0, 1.0, 1.0);
	
	mBox3Actor = iActorCollection::New(); IERROR_ASSERT_NULL_POINTER(mBox3Actor);
	for(i=0; i<12; i++) 
	{
		actor[i]->GetProperty()->SetAmbient(0.2);
		actor[i]->GetProperty()->SetDiffuse(0.2);
		actor[i]->GetProperty()->SetSpecular(0.2);
		mBox3Actor->AddActor(actor[i]);
		mRenderTool->AddObject(actor[i]);
		actor[i]->Delete();
	}
	mBox3Actor->SetVisibility(false);
	
	mBox3AxesLabels = iCubeAxesActor::New(mRenderTool); IERROR_ASSERT_NULL_POINTER(mBox3AxesLabels);
	mBox3AxesLabels->PickableOff();
	mBox3AxesLabels->VisibilityOff();
	mBox3AxesLabels->SetCamera(mRenderTool->GetCamera());
	this->SetAxesBox("X","Y","Z",-1.0,1.0,-1.0,1.0,-1.0,1.0);
	mRenderTool->AddObject(mBox3AxesLabels);
	mBox3AxesLabels->Delete();

	//
	//  Set bounding box
	//	
	this->SetBoundingBoxType(_BoundingBoxTypeDefault);

	//
	//  Ruler actor
	//
	mRulerActor = iRuler::New(this); IERROR_ASSERT_NULL_POINTER(mRulerActor);
	mRulerActor->VisibilityOff();
	mRulerActor->PickableOff();
	
	mRenderTool->AddObject(mRulerActor);

	//
	//  Text label actors
	//
	mLabelActor = iTextActor::New(mRenderTool); IERROR_ASSERT_NULL_POINTER(mLabelActor);
	mLabelActor->SetText("");
	mLabelActor->SetBold(true);
	mLabelActor->SetHJustification(_TextJustRight);
	mLabelActor->SetVJustification(_TextJustTop);
	mLabelActor->SetPosition(0.95,0.9);
	mLabelActor->VisibilityOff();
	mLabelActor->PickableOff();
	mRenderTool->AddObject(mLabelActor);
	mIsLabelOn = true;

	//
	//  Measuring box
	//
	mMeasuringBox = iMeasuringBox::New(this); IERROR_ASSERT_NULL_POINTER(mMeasuringBox);
	mMeasuringBox->VisibilityOff();
	mRenderTool->AddObject(mMeasuringBox);

	mInteractorStyleMeasuringBox = iMeasuringBoxInteractorStyle::New(); IERROR_ASSERT_NULL_POINTER(mInteractorStyleMeasuringBox);
	mInteractorStyleMeasuringBox->SetMeasuringBox(mMeasuringBox);

	//
	//  Color bars
	//
	mColorBars = iColorBars::New(this); IERROR_ASSERT_NULL_POINTER(mColorBars);
	mColorBars->PickableOff();
	mRenderTool->AddObject(mColorBars);

	//
	//  Clipping plane
	//
	mClipPlane = iClipPlane::New(this); IERROR_ASSERT_NULL_POINTER(mClipPlane);

	//
	//  Progress callback functions
	//
	mObsProgress = iProgressEventObserver::New(this); IERROR_ASSERT_NULL_POINTER(mObsProgress);
	
	mObsAbortRender = iAbortRenderEventObserver::New(this); IERROR_ASSERT_NULL_POINTER(mObsAbortRender);
	mRenderTool->AddObserver(vtkCommand::AbortCheckEvent,mObsAbortRender);
	mRenderTool->AddObserver(vtkCommand::StartEvent,mObsAbortRender);
	mRenderTool->AddObserver(vtkCommand::EndEvent,mObsAbortRender);

	mObsSlaveAbortRender = iSlaveAbortRenderEventObserver::New(mObsAbortRender); IERROR_ASSERT_NULL_POINTER(mObsSlaveAbortRender);

	//
	//  Interactor styles
	//
	mCurrentInteractorStyle = 0;

	vtkInteractorStyleFlight *isf = vtkInteractorStyleFlight::New();

	mInteractorStyle[_InteractorStyleTrackball] = vtkInteractorStyleTrackballCamera::New();
	mInteractorStyle[_InteractorStyleJoystick] = vtkInteractorStyleJoystickCamera::New();
	mInteractorStyle[_InteractorStyleFlight] = isf;
	mInteractorStyle[_InteractorStyleKeyboard] = iKeyboardInteractorStyle::New();
//	mInteractorStyle[_InteractorStyleKeyboard] = vtkInteractorStyleUnicam::New();

	for(i=0; i<__NumInteractorStyles; i++) IERROR_ASSERT_NULL_POINTER(mInteractorStyle[i]);

	isf->SetAngleStepSize(0.1);
	isf->SetMotionStepSize(0.001);

	//
	//  Work around for a VTK bug: undefined angles in vtkInteractorStyleFlight.
	//
	{
		mRenderTool->GetInteractor()->SetInteractorStyle(isf);
		isf->ForwardFly();
	}

	mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyle[mCurrentInteractorStyle]);

	//
	//  Debug props
	//
	diam = 0.2;
	for(i=0; i<3; i++) 
	{
		actor[i] = iActor::New(); IERROR_ASSERT_NULL_POINTER(actor[i]);
		as = vtkArrowSource::New(); IERROR_ASSERT_NULL_POINTER(as);
		as->SetShaftResolution(6);
		as->SetTipResolution(18);
		as->SetTipLength(0.5);
		as->SetShaftRadius(0.7*diam);
		as->SetTipRadius(1.3*diam);
		as->Update();
		actor[i]->SetInput(as->GetOutput());
		as->Delete();
		actor[i]->SetScale(0.5);
		actor[i]->GetProperty()->SetAmbient(0.25);
		actor[i]->GetProperty()->SetDiffuse(0.25);
		actor[i]->GetProperty()->SetSpecular(0.5);
		actor[i]->PickableOff();
	}
	
	actor[0]->GetProperty()->SetColor(0.0,0.0,1.0);
	actor[0]->RotateY(-90.0);
	actor[1]->GetProperty()->SetColor(0.0,1.0,0.0);
	actor[1]->RotateZ(90.0);
	actor[2]->GetProperty()->SetColor(1.0,0.0,0.0);
	
	actor[ 0]->SetPosition(0.0,0.0,0.0);
	actor[ 1]->SetPosition(0.0,0.0,0.0);
	actor[ 2]->SetPosition(0.0,0.0,0.0);

	mDebugActor = iActorCollection::New(); IERROR_ASSERT_NULL_POINTER(mDebugActor);
	for(i=0; i<3; i++) 
	{
		mDebugActor->AddActor(actor[i]);
		mRenderTool->AddObject(actor[i]);
		actor[i]->Delete();
	}
	mDebugActor->VisibilityOff();
	
	//
	// icon
	//
	mIconActor = vtkActor2D::New(); IERROR_ASSERT_NULL_POINTER(mIconActor);
	vtkPolyDataMapper2D *mapper2D = vtkPolyDataMapper2D::New(); IERROR_ASSERT_NULL_POINTER(mapper2D);
	vtkDiskSource *ds = vtkDiskSource::New(); IERROR_ASSERT_NULL_POINTER(ds);

	ds->SetInnerRadius(0.03);
	ds->SetOuterRadius(0.07);
	ds->SetRadialResolution(1);
	ds->SetCircumferentialResolution(8);

	ds->Update();
	int npoi = ds->GetOutput()->GetPoints()->GetNumberOfPoints();
	vtkFloatArray *mpoi;
	mpoi = vtkFloatArray::New(); IERROR_ASSERT_NULL_POINTER(mpoi);
	mpoi->SetNumberOfComponents(1);
	mpoi->SetNumberOfTuples(npoi);
	for(i=0; i<npoi; i++)
	{
		mpoi->SetTuple1(i,(i%4)/2);
	}
	ds->GetOutput()->GetPointData()->SetScalars(mpoi);
	mpoi->Delete();

	mapper2D->SetInput(ds->GetOutput());
	ds->Delete();
	vtkCoordinate *c = vtkCoordinate::New(); IERROR_ASSERT_NULL_POINTER(c);
	c->SetCoordinateSystemToView();
	mapper2D->SetTransformCoordinate(c);
	c->Delete();
	mapper2D->SetScalarRange(0,1);

	mIconActor->SetMapper(mapper2D);
	mapper2D->Delete();
	mIconActor->GetProperty()->SetColor(0.0,0.0,0.0);
	mIconActor->GetPositionCoordinate()->SetCoordinateSystemToNormalizedDisplay();
	mIconActor->SetPosition(-0.45,0.45);
	mIconActor->GetProperty()->SetOpacity(0.0);
	mIconActor->VisibilityOff();
	mIconActor->PickableOff();
	mIconActor->SetLayerNumber(10);

	mRenderTool->AddObject(mIconActor);

	//
	//  Picker
	//
	mPicker = iPicker::New(this); IERROR_ASSERT_NULL_POINTER(mPicker);
	mRenderTool->GetInteractor()->SetPicker(mPicker);

	//
	//  Event recorder & marker helpers
	//  
	mIsMarkerLegendOn = false;
	mMarkerLegendPosition = 0;

	mEventRecorder = iInteractorEventRecorder::New(); IERROR_ASSERT_NULL_POINTER(mEventRecorder);
	mEventRecorder->SetInteractor(mRenderTool->GetInteractor());
	mEventRecorder->SetEnabled(1);

	mObsRecord = iRecordEventObserver::New(this); IERROR_ASSERT_NULL_POINTER(mObsRecord);

	for(i=0; i<__NumInteractorStyles; i++) mInteractorStyle[i]->AddObserver(vtkCommand::InteractionEvent,mObsRecord);

	mMarkerPlacer = vtkPointWidget::New(); IERROR_ASSERT_NULL_POINTER(mMarkerPlacer);
	mMarkerPlacer->XShadowsOn();
	mMarkerPlacer->YShadowsOn();
	mMarkerPlacer->ZShadowsOn();
	mMarkerPlacer->OutlineOff();
	mMarkerPlacer->TranslationModeOff();
	mMarkerPlacer->SetInteractor(mRenderTool->GetInteractor());
	mMarkerPlacer->GetProperty()->SetColor(0.0,0.0,0.0);
	mMarkerPlacer->GetProperty()->SetLineWidth(4.0);
//	mMarkerPlacer->GetSelectedProperty()->SetColor(1.0,0.0,0.0);
	mMarkerPlacer->GetSelectedProperty()->SetLineWidth(2.0);
	mMarkerPlacer->PlaceWidget(-1.0,1.0,-1.0,1.0,-1.0,1.0);

	mObsMarker = iMarkerEventObserver::New(this); IERROR_ASSERT_NULL_POINTER(mObsMarker);

	mMarkerPlacer->AddObserver(vtkCommand::InteractionEvent,mObsMarker);

	//
	//  vtkLegendBoxActor is defined under earlier VTK versions, but
	//  markers cannot be positioned without the vtkPointWidgets, so the legend
	//  is useless then too.
	//
	mMarkerLegend = iLegend::New(mRenderTool); IERROR_ASSERT_NULL_POINTER(mMarkerLegend);
	this->SetMarkerLegendPosition(mMarkerLegendPosition);
	mRenderTool->AddObject(mMarkerLegend);
	mMarkerLegend->VisibilityOff();

	mInteractorStyleCaption = iCaptionInteractorStyle::New(this); IERROR_ASSERT_NULL_POINTER(mInteractorStyleCaption);

	//
	//  Image composer
	//
	mImageComposer = 0;

	//
	//  iViewObjects
	//
	mFamSurface = iViewObjectFamily<iSurfaceViewObject>::New(this); IERROR_ASSERT_NULL_POINTER(mFamSurface);
	mFamCrossSection = iViewObjectFamily<iCrossSectionViewObject>::New(this); IERROR_ASSERT_NULL_POINTER(mFamCrossSection);
	mFamVolume = iViewObjectFamily<iVolumeViewObject>::New(this); IERROR_ASSERT_NULL_POINTER(mFamVolume);
	mFamVectorField = iViewObjectFamily<iVectorFieldViewObject>::New(this); IERROR_ASSERT_NULL_POINTER(mFamVectorField);
	mFamTensorField = iViewObjectFamily<iTensorFieldViewObject>::New(this); IERROR_ASSERT_NULL_POINTER(mFamTensorField);
	mFamParticles = iViewObjectFamily<iParticlesViewObject>::New(this); IERROR_ASSERT_NULL_POINTER(mFamParticles);

	mFamMarker = iViewObjectFamily<iMarkerViewObject>::New(this); IERROR_ASSERT_NULL_POINTER(mFamMarker);

	//
	//  Timer
	//
	mUpdateTimer = vtkTimerLog::New(); IERROR_ASSERT_NULL_POINTER(mUpdateTimer);
	mUpdateTime = 0.0f;

	//
	//  Render the scene
	//
	mRenderTool->GetCamera()->Reset();

	mRulerActor->SetBaseScale(mRenderTool->GetCamera()->GetParallelScale());
	mMeasuringBox->SetBaseScale(mRenderTool->GetCamera()->GetParallelScale());
	
	mRenderTool->GetInteractor()->SetDesiredUpdateRate(mRate);
	mRenderTool->GetInteractor()->SetStillUpdateRate(0.001);

	vtkMapper::SetResolveCoincidentTopologyToPolygonOffset();

	//
	//  Set the font size
	//
	this->SetFontSize(mFontSize);
	this->SetFontType(mFontType);
	this->SetAntialiasing(mAntialiasing);
	this->SetBackgroundColor(mBgColor);

	mWriter->SetImageFormat(_ImageFormatPNG);
	mWriter->SetAnimationOutput(_AnimationOutputImage);

	//
	//  Initialize the interactor
	//
	mRenderTool->GetInteractor()->Initialize();

	//
	//  Debug area
	//

}


iViewModule::~iViewModule()
{
	// Clean up
	int i;

	if(this->IsClone())
	{
		//
		//  Unregister with the parent first
		//
		mCloneOfModule->mClones.Remove(this);
	}
	//
	//  Check that all clones are already deleted
	//
#ifdef I_CHECK1
	if(mClones.Size() > 0) IERROR_REPORT_ERROR("Deleting a window with undeleted clones.");
#endif

	//
	//  Delete picker first in case it holds onto an object
	//
	mRenderTool->GetInteractor()->SetPicker(0);
	mPicker->Delete();

//	if(mRenderTool->GetRenderWindowInteractor() != 0)
//	{
//		iPointerCast<vtkInteractorStyle,vtkInteractorObserver>(mRenderTool->GetRenderWindowInteractor()->GetInteractorStyle())->HighlightProp(0);
//	}
	mEventRecorder->SetInteractor(0);
	mEventRecorder->Delete();
	mObsRecord->Delete();
	mMarkerPlacer->Off();
	mMarkerPlacer->SetInteractor(0);
	mMarkerPlacer->Delete();
	mMarkerLegend->Delete();
	mInteractorStyleCaption->Delete();

	mFamSurface->Delete();
	mFamCrossSection->Delete();
	mFamVolume->Delete();
	mFamParticles->Delete();
	mFamVectorField->Delete();
	mFamTensorField->Delete();

	mFamMarker->Delete();
	
	mUpdateTimer->Delete();

	mAnimator->Delete();

	for(i=0; i<__NumInteractorStyles; i++) mInteractorStyle[i]->Delete();

	this->RemoveObject(mBox1Actor);
	mBox1Actor->Delete();
	mBox2Actor->Delete();
	mBox3Actor->Delete();
	mDebugActor->Delete();

	this->RemoveObject(mRulerActor);
	mRulerActor->Delete();

	this->RemoveObject(mLabelActor);
	mLabelActor->Delete();

	this->RemoveObject(mColorBars);
	mColorBars->Delete();

	this->RemoveObject(mMeasuringBox);
	mMeasuringBox->Delete();
	mInteractorStyleMeasuringBox->Delete();

	this->RemoveObject(mIconActor);
	mIconActor->Delete();

	mClipPlane->Delete();

	if(!this->IsClone()) 
	{
		mDataReader->Delete();
	}
	mWriter->Delete();

	mObsProgress->Delete();
	mObsSlaveAbortRender->Delete();
	mObsAbortRender->Delete();

	mRenderTool->Delete();

	//
	//  Check that all DataConsumers have been deleted too
	//
	if(mDataConsumers.Size() > 0)
	{
		IERROR_REPORT_FATAL_ERROR("Not all DataConsumers have been deleted.");
	}
}


void iViewModule::UpdateWindowNumber()
{
	int i;
	iString windowName = "IFrIT - Visualization Window";

	for(i=0; i<this->GetControlModule()->GetNumberOfViewModules(); i++)
	{
		if(this->GetControlModule()->GetViewModule(i) == this)
		{
			mWinNum = i;
			break;
		}
	}
	if(this->GetControlModule()->GetNumberOfViewModules() > 1) windowName += " #" + iString::FromNumber(mWinNum+1);

	mRenderTool->UpdateWindowName(windowName);
	this->ClearCache();
}


bool iViewModule::BecomeClone(iViewModule *v)
{
	int i;
	bool ret = true;

	if(v == 0)
	{
		//
		//	Unmake the clone
		//
		v = mDataReader->GetViewModule(); 
		if(v != this)
		{
#ifdef I_CHECK1
			if(v != mCloneOfModule)
			{
				IERROR_REPORT_ERROR("Clones configured incorrectly.");
			}
#endif
			mCloneOfModule = 0;
			v->mClones.Remove(this);
			//
			//  Re-create data reader
			//
			mDataReader = iDataReader::New(this); IERROR_ASSERT_NULL_POINTER(mDataReader);
			mDataReader->CopyState(v->mDataReader);

			mFamSurface->Reset();
			mFamCrossSection->Reset();
			mFamVolume->Reset();
			mFamParticles->Reset();
			mFamVectorField->Reset();
			mFamTensorField->Reset();

			for(i=0; i<=mFamParticles->GetMaxMemberIndex(); i++)
			{
				mFamParticles->GetMember(i)->BecomeClone(0);
			}
			//
			//  Un-show label
			//
			this->ShowLabel(mIsLabelOn); 
		}
	}
	else
	{
		//
		//	Make the clone
		//
		if(mDataReader->GetViewModule() == this)
		{
			mCloneOfModule = v;
			v->mClones.Add(this);
			//
			//  Link the Data Reader
			//
			mDataReader->Delete();
			mDataReader = v->mDataReader;
			//
			//  Copy iViewObject families
			//
			ret = ret & mFamSurface->Copy(v->mFamSurface);
			ret = ret & mFamCrossSection->Copy(v->mFamCrossSection);
			ret = ret & mFamVolume->Copy(v->mFamVolume);
			ret = ret & mFamVectorField->Copy(v->mFamVectorField);
			ret = ret & mFamTensorField->Copy(v->mFamTensorField);
			ret = ret & mFamParticles->Copy(v->mFamParticles);
			ret = ret & mFamMarker->Copy(v->mFamMarker);
			//
			//  ParticleSet family is never shown as a whole, so show individual members
			//
			for(i=0; i<=mFamParticles->GetMaxMemberIndex(); i++)
			{
				mFamParticles->GetMember(i)->BecomeClone(v->mFamParticles->GetMember(i));
				mFamParticles->GetMember(i)->Show(v->mFamParticles->GetMember(i)->IsVisible());
			}
			//
			//  Copy Animator state
			//
			mAnimator->CopyState(v->mAnimator);
			//
			//  Copy camera location
			//
			mRenderTool->GetCamera()->SetPosition(v->mRenderTool->GetCamera()->GetPosition());
			mRenderTool->GetCamera()->SetFocalPoint(v->mRenderTool->GetCamera()->GetFocalPoint());
			mRenderTool->GetCamera()->SetViewUp(v->mRenderTool->GetCamera()->GetViewUp());
			mRenderTool->GetCamera()->SetParallelScale(v->mRenderTool->GetCamera()->GetParallelScale());
			//
			//  Show label
			//
			this->ShowLabel(v->IsLabelVisible()); 
		}
	}

	this->ClearCache();
	return ret;
}


void iViewModule::SetDebugMode(bool s)
{
	int i;

	if(s != mIsDebug)
	{
		mIsDebug = s;
		if(s)
		{
			
			mVisibleProps.Clear();
			for(i=0; i<mProps.Size(); i++) if(mProps[i]->GetVisibility())
			{
				mVisibleProps.Add(mProps[i]);
				mProps[i]->VisibilityOff();
			}
			
			mDebugActor->VisibilityOn();
		}
		else
		{
			
			mAnimator->SetDebugFlag(0);
			mDebugActor->VisibilityOff();
			
			for(i=0; i<mVisibleProps.Size(); i++)
			{
				mVisibleProps[i]->VisibilityOn();
			}
		}
		this->Render();
	}
	this->ClearCache();
}


void iViewModule::AddObject(vtkProp* p)
{
	mRenderTool->AddObject(p);
}


void iViewModule::RemoveObject(vtkProp* p)
{
	mRenderTool->RemoveObject(p);
}


vtkRenderWindowCollection* iViewModule::GetRenderWindowCollection(bool inited) const
{
	vtkRenderWindowCollection *c = mRenderTool->GetRenderWindowCollection();
	if(inited) c->InitTraversal();
	return c;
}


iParallelManager* iViewModule::GetParallelManager() const
{
	return this->GetControlModule()->GetParallelManager();
}


void iViewModule::Render()  
{
	//
	//  Activate counting for parallel performance
	//
	this->GetParallelManager()->StartCounters();
	mUpdateTimer->StartTimer();

	mRenderTool->Render();

	mUpdateTimer->StopTimer();
	mUpdateTime = mUpdateTimer->GetElapsedTime();
	this->UpdatePerformance();
	this->GetParallelManager()->StopCounters();
}


void iViewModule::UpdatePerformance()
{
	mObsAbortRender->PostFinished(); // manually driven
}


void iViewModule::SetAxesBox(const iString &labelX, const iString &labelY, const iString &labelZ, float xMin, float xMax, float yMin, float yMax, float zMin, float zMax)
{
	mBox3AxesLabels->SetXLabel(labelX.ToCharPointer());
	mBox3AxesLabels->SetYLabel(labelY.ToCharPointer());
	mBox3AxesLabels->SetZLabel(labelZ.ToCharPointer());
	mBox3AxesLabels->SetRanges(xMin,xMax,yMin,yMax,zMin,zMax);
	this->ClearCache();
}


void iViewModule::SetFontSize(int s)
{
	mLabelActor->SetFontSize(s); 
	if(s == mLabelActor->GetFontSize())
	{
		mFontSize = s;
		mRulerActor->SetFontSize(s);
		mMeasuringBox->GetTextActor()->SetFontSize(s);
		mColorBars->SetFontSize(s);
		mBox3AxesLabels->SetFontSize(s);
		int i;
		for(i=0; i<=mFamMarker->GetMaxMemberIndex(); i++) mFamMarker->GetMember(i)->SetFontSize(s);
		mMarkerLegend->SetFontSize(s);
		this->ClearCache();
	}
}


void iViewModule::SetFontType(int s)
{
	mLabelActor->SetFontType(s); 
	if(s == mLabelActor->GetFontType())
	{
		mFontType = s; 
//		mRulerActor->SetFontType(s);
		mMeasuringBox->GetTextActor()->SetFontType(s);
		mColorBars->SetFontType(s);
		this->ClearCache();
	}
}


void iViewModule::ShowBoundingBox(bool s)
{
	this->ClearCache();

	mBox1Actor->VisibilityOff(); 
	mBox2Actor->VisibilityOff();
	mBox3Actor->VisibilityOff();
	mBox3AxesLabels->VisibilityOff();
	mIsBoundingBoxOn = s;

	if(!mIsBoundingBoxOn) return;
		
	switch(mBoxType)
	{
	case _BoundingBoxTypeDefault:
		{
			mBox1Actor->GetMapper()->ScalarVisibilityOn();
			mBox1Actor->GetProperty()->SetLineWidth(4.0);
			mBox1Actor->VisibilityOn();
			break;
		}
	case _BoundingBoxTypeClassic:
		{
			mBox2Actor->VisibilityOn();
			break;
		}
	case _BoundingBoxTypeHairThin:
		{
			mBox1Actor->GetMapper()->ScalarVisibilityOff();
			mBox1Actor->GetProperty()->SetLineWidth(1.0);
			mBox1Actor->VisibilityOn();
			break;
		}
	case _BoundingBoxTypeAxes:
		{
			mBox3Actor->VisibilityOn();
			mBox3AxesLabels->VisibilityOn();
			break;
		}
	}
}


void iViewModule::ShowRuler(bool s)
{
	mRulerActor->SetVisibility(s?1:0);
	mIsRulerOn = s;
	this->ClearCache();
}


void iViewModule::ShowColorBars(bool s)
{
	mColorBars->SetVisibility(s?1:0);
	mIsColorBarsOn = s;
	this->ClearCache();
}


void iViewModule::ShowMeasuringBox(bool s)
{
	if(s)
	{
		mMeasuringBox->VisibilityOn(); 
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyleMeasuringBox);
	} 
	else 
	{
		mMeasuringBox->VisibilityOff();
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyle[mCurrentInteractorStyle]);
	}
	mIsMeasuringBoxOn = s;
	this->ClearCache();
}


void iViewModule::ShowGlassClipPlane(bool s)
{
	mIsGlassClipPlaneOn = s;
	if(mIsClipPlaneOn) mClipPlane->SetGlassPlaneVisible(mIsGlassClipPlaneOn);
	this->ClearCache();
}


void iViewModule::ShowClipPlane(bool s)
{
	int i;
	
	for(i=0; i<=mFamSurface->GetMaxMemberIndex(); i++) mFamSurface->GetMember(i)->ShowClipPlane(s);
	for(i=0; i<=mFamVolume->GetMaxMemberIndex(); i++) mFamVolume->GetMember(i)->ShowClipPlane(s);
	for(i=0; i<=mFamParticles->GetMaxMemberIndex(); i++) mFamParticles->GetMember(i)->ShowClipPlane(s);
	for(i=0; i<=mFamVectorField->GetMaxMemberIndex(); i++) mFamVectorField->GetMember(i)->ShowClipPlane(s);
	for(i=0; i<=mFamTensorField->GetMaxMemberIndex(); i++) mFamTensorField->GetMember(i)->ShowClipPlane(s);
	mIsClipPlaneOn = s;
	if(mIsClipPlaneOn) mClipPlane->SetGlassPlaneVisible(mIsGlassClipPlaneOn); else mClipPlane->SetGlassPlaneVisible(false); 
	this->ClearCache();
}


void iViewModule::ShowStereoAlignmentMarkers(bool s)
{
	mIsStereoAlignmentMarkersOn = s;
	mRenderTool->ShowStereoAlignmentMarkers(s);
	this->ClearCache();
}


void iViewModule::UpdateLabel(float *v)
{
	this->ShowLabel(mIsLabelOn,v);
	//
	//  Update my clones as well
	//
	int i;
	for(i=0; i<mClones.Size(); i++) mClones[i]->UpdateLabel(v);
}


void iViewModule::ShowLabel(bool s, float *pv)
{
	mIsLabelOn = s;
	if(s && mDataReader->IsFileAnimatable())
	{
		iString s;
		int r = mDataReader->GetRecordNumber();
		float v;
		if(pv == 0)
		{
			if(fabs(mLabelScale) > 0.0) v = r*mLabelScale; else v = 1.0e4/((r>0)?r:1)-1.0;
		}
		else
		{
			v = *pv;
		}
		iString num = iString::FromNumber(v,"%g");
		int n = num.Find('.') + 1;
		if(n>0 && n+mLabelDigits<num.Length())
		{
			num = num.Part(0,n+mLabelDigits);
			if(num[num.Length()-1] == '.') num[num.Length()-1] = ' ';
		}
		s = mLabelName + "=" + num + mLabelUnit;
		mLabelActor->SetText(s);
		mLabelActor->VisibilityOn(); 	
	} 
	else mLabelActor->VisibilityOff();
	this->ClearCache();
}


void iViewModule::JustifyLabelLeft(bool s)
{
	if(s)
	{
		if(!mLabelJustificationLeft)
		{
			mLabelJustificationLeft = true;
			float s[2];
			const float *p;
			mLabelActor->GetSize(s);
			mLabelActor->SetHJustification(_TextJustLeft);
			p = mLabelActor->GetPosition();
			mLabelActor->SetPosition(p[0]-s[0],p[1]);
		}
	}
	else
	{
		if(mLabelJustificationLeft)
		{
			mLabelJustificationLeft = false;
			float s[2];
			const float *p;
			mLabelActor->GetSize(s);
			mLabelActor->SetHJustification(_TextJustRight);
			p = mLabelActor->GetPosition();
			mLabelActor->SetPosition(p[0]+s[0],p[1]);
		}
	}
}


void iViewModule::SetUpdateRate(int r)
{
	r = (r<0) ? 0 : r;
	r = (r>60) ? 60 : r;
	mRate = r;
	mRenderTool->GetInteractor()->SetDesiredUpdateRate(mRate);
	this->ClearCache();
}


void iViewModule::SetInteractorStyle(int n)
{
	if(n>=0 && n<__NumInteractorStyles)
	{
		//
		//  Specials
		//
		if(n == _InteractorStyleFlight)
		{
			mSavedProjection = mRenderTool->GetCamera()->GetParallelProjection();
			mRenderTool->GetCamera()->ParallelProjectionOff();
		} 
		else if(mCurrentInteractorStyle == _InteractorStyleFlight)
		{
			mRenderTool->GetCamera()->SetParallelProjection(mSavedProjection);
			mRenderTool->GetCamera()->Reset();
		}
		mCurrentInteractorStyle = n;
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyle[mCurrentInteractorStyle]);
		this->ClearCache();
	}
}


void iViewModule::SetBackgroundColor(const iColor &bg)
{
	mBgColor = bg;
	iColor fg = mBgColor.Reverse();

	mRenderTool->SetBackground(mBgColor);
	mRulerActor->SetColor(fg);
	mMeasuringBox->SetColor(fg);
	mColorBars->SetColor(fg);
	mLabelActor->SetColor(fg);
	mBox1Actor->GetProperty()->SetColor(fg.ToVTK());
	mIconActor->GetProperty()->SetColor(fg.ToVTK());
//	pickedPointActor->GetProperty()->SetColor(fg.ToVTK());
	mAnimator->SetCameraPathColor(fg);
	mMarkerLegend->GetProperty()->SetColor(fg.ToVTK());
	mMarkerLegend->GetEntryTextProperty()->SetColor(fg.ToVTK());
	mMarkerPlacer->GetProperty()->SetColor(fg.ToVTK());
	for(int i=0; i<=mFamMarker->GetMaxMemberIndex(); i++) if(mFamMarker->GetMember(i)->GetColorAutomatic()) mFamMarker->GetMember(i)->SetTextColor(fg);
	this->ClearCache();
}


void iViewModule::SetBoundingBoxType(int s)
{
	if(s!=mBoxType && s>=0 && s<__NumBoundingBoxTypes)
	{
		mBoxType = s;
		this->ShowBoundingBox(mIsBoundingBoxOn);
		this->ClearCache();
	}		
}


void iViewModule::SetAntialiasing(bool s)
{
	mAntialiasing = s;
	mRenderTool->SetAntialiasing(s);
	this->ClearCache();
}


void iViewModule::UpdateIcon()
{
	int i;

	if(this->GetControlModule()->GetNumberOfViewModules() == 1) 
	{
		mIconActor->VisibilityOff(); 
	}
	else 
	{
		if(mWinNum == this->GetControlModule()->GetCurrentViewModuleIndex())
		{
			//
			//  Current window
			//
			mIconActor->VisibilityOn(); 
			mIconActor->GetProperty()->SetOpacity(1.0);
			for(i=0; i<mClones.Size(); i++)
			{ 
				mClones[i]->mIconActor->VisibilityOn(); 
				mClones[i]->mIconActor->GetProperty()->SetOpacity(0.5);
				mClones[i]->Render();
			}
		}
		else
		{ 
			mIconActor->VisibilityOff();
			for(i=0; i<mClones.Size(); i++)
			{
				mClones[i]->mIconActor->VisibilityOff();
				mClones[i]->Render();
			}
		}
		this->Render();
	}
}


void iViewModule::SetImageMagnification(int m)
{
	if(m>0 && m<101)
	{
		mImageMagnification = m;
		this->ClearCache();
	}
}


int iViewModule::GetFullImageWidth()
{
	if(mImageComposer != 0)
	{
		return mImageComposer->GetImageWidth();
	}
	else return this->GetThisImageWidth();
}


int iViewModule::GetFullImageHeight()
{
	if(mImageComposer != 0)
	{
		return mImageComposer->GetImageHeight();
	}
	else return this->GetThisImageHeight();
}


int iViewModule::GetThisImageWidth()
{
	const int *s = mRenderTool->GetRenderWindowSize();
	return mImageMagnification*s[0];
}


int iViewModule::GetThisImageHeight()
{
	const int *s = mRenderTool->GetRenderWindowSize();
	return mImageMagnification*s[1];
}


void iViewModule::CreateImages(iStereoImageArray &images)
{
	images.Clear();
	if(mImageComposer != 0)
	{
		//
		//  ask composer to composer the image
		//
		this->GetErrorStatus()->Monitor(mImageComposer);
		mImageComposer->Compose(this,mImages);
	}
	else
	{
		this->RenderImages(mImages);
	}
	if(this->GetErrorStatus()->NoError())
	{
		int i;
		images.Clear();
		for(i=0; i<mImages.Size(); i++) images.Add(mImages[i]);
	}
}


void iViewModule::RenderImages(iStereoImageArray &images)
{
	//
	//  Prepare for render; do a few more things for better representation
	//
	int iconVis = mIconActor->GetVisibility();
	mIconActor->VisibilityOff();
	float box2LineWidth = mBox1Actor->GetProperty()->GetLineWidth();
	mBox1Actor->GetProperty()->SetLineWidth(box2LineWidth*mImageMagnification);
	
	//
	//  do the render
	//
	mInImageRender = true;		
	mRenderTool->RenderImages(mImageMagnification,images);
	mInImageRender = false;

	//
	//  Reset to the present state
	//
	mBox1Actor->GetProperty()->SetLineWidth(box2LineWidth);
	mIconActor->SetVisibility(iconVis);
		
	//
	// render again to restore the original view
	//
	mRenderTool->Render();
}


void iViewModule::RenderStereoImage(iStereoImage &image)
{
	//
	//  Prepare for render; do a few more things for better representation
	//
	int iconVis = mIconActor->GetVisibility();
	mIconActor->VisibilityOff();
	float box2LineWidth = mBox1Actor->GetProperty()->GetLineWidth();
	mBox1Actor->GetProperty()->SetLineWidth(box2LineWidth*mImageMagnification);
	
	//
	//  do the render
	//
	mInImageRender = true;		
	mRenderTool->RenderStereoImage(mImageMagnification,image);
	mInImageRender = false;

	//
	//  Reset to the present state
	//
	mBox1Actor->GetProperty()->SetLineWidth(box2LineWidth);
	mIconActor->SetVisibility(iconVis);
		
	//
	// render again to restore the original view
	//
	mRenderTool->Render();
}


void iViewModule::DumpImages(int m)
{
	static const iStereoImageArray dummy;
	this->DumpImages(dummy,m);
}


void iViewModule::DumpImages(const iStereoImageArray &images, int m)
{
	int i;
	iStereoImageArray tmp;

	if(images.Size() != mRenderTool->GetNumberOfActiveViews()) 
	{
		this->CreateImages(tmp);
		if(images.Size() != 0) IERROR_REPORT_ERROR("Incorrect number of images in ImageArray in DumpImages."); 
	}
	else
	{
		for(i=0; i<images.Size(); i++) tmp.Add(images[i]);
	}

	//
	// Create and write files
	//
	if(this->GetErrorStatus()->NoError())
	{
		mObsProgress->SetMode(iProgressEventObserver::_Writing);  // auto-driven
		mWriter->WriteFrame(tmp,m);
	}
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iViewModule::PackStateBody(iString &s) const
{
	int i; float fv[6]; iString sv[3];

	this->PackValue(s,KeyWindowNumber(),mWinNum);
	this->PackValue(s,KeyCloneOfWindow(),this->GetCloneOfWindow());
	this->PackValue(s,KeyNoClone(),!this->IsClone());

//	this->PackValue(s,KeyPosition(),win->GetPosition(),2);
	this->PackValue(s,KeySize(),mRenderTool->GetRenderWindowSize(),2);

	this->PackValue(s,KeyInteractorStyle(),mCurrentInteractorStyle);
	this->PackValue(s,KeyLabel(),mIsLabelOn);
	this->PackValue(s,KeyColorBars(),mIsColorBarsOn);
	this->PackValue(s,KeyBoundingBox(),mIsBoundingBoxOn);
	this->PackValue(s,KeyRuler(),mIsRulerOn);
	this->PackValue(s,KeyGlassClipPlane(),mIsGlassClipPlaneOn);
	this->PackValue(s,KeyClipPlane(),mIsClipPlaneOn);
	this->PackValue(s,KeyMeasuringBox(),mIsMeasuringBoxOn);
	this->PackValue(s,KeyAntialiasing(),mAntialiasing);
	this->PackValue(s,KeyJustifyLabelLeft(),mLabelJustificationLeft);
	this->PackValue(s,KeyMarkerLegend(),mIsMarkerLegendOn);
	this->PackValue(s,KeyStereo(),this->GetStereo());
	this->PackValue(s,KeyStereoAlignmentMarkers(),mIsStereoAlignmentMarkersOn);

	this->PackValue(s,KeyLabelName(),mLabelName);
	this->PackValue(s,KeyLabelUnit(),mLabelUnit);
	this->PackValue(s,KeyLabelScale(),mLabelScale);
	this->PackValue(s,KeyLabelDigits(),mLabelDigits);

	this->PackValue(s,KeyFontSize(),mFontSize);
	this->PackValue(s,KeyFontType(),mFontType);
	this->PackValue(s,KeyBoundingBoxType(),mBoxType);
	this->PackValue(s,KeyImageFormat(),mWriter->GetImageFormat());
	this->PackValue(s,KeyAnimationOutput(),mWriter->GetAnimationOutput());
	this->PackValue(s,KeyMarkerLegendPosition(),mMarkerLegendPosition);
	this->PackValue(s,KeyImageMagnification(),mImageMagnification);
	this->PackValue(s,KeyStereoType(),this->GetStereoType());

	this->PackValue(s,KeyBackgroundColor(),mBgColor);

	this->PackValue(s,KeyUpdateRate(),mRate);
	this->PackValue(s,KeyRulerScale(),this->GetRulerScale());
	this->PackValue(s,KeyOpenGLCoordinates(),this->GetOpenGLCoordinates());
	this->PackValue(s,KeyBoxSize(),this->GetBoxSize());

	this->PackValue(s,KeyClipPlaneDistance(),this->GetClipPlaneDistance());
	this->PackValue(s,KeyClipPlaneDirection(),this->GetClipPlaneDirection(),3);

	this->PackValue(s,KeyPostScriptPaperFormat(),mWriter->GetPostScriptPaperFormat());
	this->PackValue(s,KeyPostScriptOrientation(),mWriter->GetPostScriptOrientation());

	sv[0] = mBox3AxesLabels->GetXLabel();
	sv[1] = mBox3AxesLabels->GetYLabel();
	sv[2] = mBox3AxesLabels->GetZLabel();
	for(i=0; i<6; i++) fv[i] = mBox3AxesLabels->GetRanges()[i];
	this->PackValue(s,KeyAxesBoxLabels(),sv,3);
	this->PackValue(s,KeyAxesBoxRanges(),fv,6);

	//
	//  Light parameters
	//
	mRenderTool->GetLights()->GetAngles(fv);
	this->PackValue(s,KeyLightAngles(),fv,2);
	mRenderTool->GetLights()->GetIntensities(fv);
	this->PackValue(s,KeyLightIntensity(),fv,3);

	//
	//  Extra keys for data quering
	//
	this->PackValue(s,KeyMarkerCurrent(),mFamMarker->GetCurrentMemberIndex());
	this->PackValue(s,KeyParticlesCurrent(),mFamParticles->GetCurrentMemberIndex());
	this->PackValue(s,KeySurfaceCurrent(),mFamSurface->GetCurrentMemberIndex());
	this->PackValue(s,KeyCrossSectionCurrent(),mFamCrossSection->GetCurrentMemberIndex());

	i = mFamMarker->GetMaxMemberIndex();
	if(i==0 && !mFamMarker->GetMember(0)->IsVisible()) i = -1;
	this->PackValue(s,KeyMarkerMax(),i);
	this->PackValue(s,KeyParticlesMax(),mFamParticles->GetMaxMemberIndex());
	this->PackValue(s,KeySurfaceMax(),mFamSurface->GetMaxMemberIndex());
	this->PackValue(s,KeyCrossSectionMax(),mFamCrossSection->GetMaxMemberIndex());

	this->PackValue(s,KeyPlaceMarker(),mMarkerPlacer->GetEnabled()!=0);
	this->PackValue(s,KeyMoveMarkerCaption(),mRenderTool->GetInteractor()->GetInteractorStyle()==mInteractorStyleCaption);
}


void iViewModule::UnPackStateBody(const iString &s)
{
	int i, iv[2]; bool b; float f, fv[6]; iColor c; iString w, sv[3];

//	if(this->UnPackValue(s,KeyPosition(),iv,2)) win->SetPosition(iv);
	iv[0] = mRenderTool->GetRenderWindowSize()[0];
	iv[1] = mRenderTool->GetRenderWindowSize()[1];
	if(this->UnPackValue(s,KeySize(),iv,2))
	{
		mRenderTool->SetRenderWindowSize(iv[0],iv[1]);
		this->ClearCache();
	}

	if(this->UnPackValue(s,KeyInteractorStyle(),i)) this->SetInteractorStyle(i);
	if(this->UnPackValue(s,KeyLabel(),b)) this->ShowLabel(b);
	if(this->UnPackValue(s,KeyColorBars(),b)) this->ShowColorBars(b);
	if(this->UnPackValue(s,KeyBoundingBox(),b)) this->ShowBoundingBox(b);
	if(this->UnPackValue(s,KeyRuler(),b)) this->ShowRuler(b);
	if(this->UnPackValue(s,KeyGlassClipPlane(),b)) this->ShowGlassClipPlane(b);
	if(this->UnPackValue(s,KeyClipPlane(),b)) this->ShowClipPlane(b);
	if(this->UnPackValue(s,KeyMeasuringBox(),b)) this->ShowMeasuringBox(b);
	if(this->UnPackValue(s,KeyAntialiasing(),b)) this->SetAntialiasing(b);
//	if(this->UnPackValue(s,KeyJustifyLabelLeft(),b)) this->justifyLabelLeft(b);
	if(this->UnPackValue(s,KeyMarkerLegend(),b)) this->ShowMarkerLegend(b);
	if(this->UnPackValue(s,KeyStereo(),b)) this->SetStereo(b);
	if(this->UnPackValue(s,KeyStereoAlignmentMarkers(),b)) this->ShowStereoAlignmentMarkers(b);

	if(this->UnPackValue(s,KeyFontSize(),i)) this->SetFontSize(i);
	if(this->UnPackValue(s,KeyFontType(),i)) this->SetFontType(i);
	if(this->UnPackValue(s,KeyBoundingBoxType(),i)) this->SetBoundingBoxType(i);
	if(this->UnPackValue(s,KeyImageFormat(),i)) mWriter->SetImageFormat(i);
	if(this->UnPackValue(s,KeyAnimationOutput(),i)) mWriter->SetAnimationOutput(i);
	if(this->UnPackValue(s,KeyMarkerLegendPosition(),i)) this->SetMarkerLegendPosition(i);
	if(this->UnPackValue(s,KeyImageMagnification(),i)) this->SetImageMagnification(i);
	if(this->UnPackValue(s,KeyStereoType(),i)) this->SetStereoType(i);
	
	if(this->UnPackValue(s,KeyLabelName(),w)) this->SetLabelName(w);
	if(this->UnPackValue(s,KeyLabelUnit(),w)) this->SetLabelUnit(w);
	if(this->UnPackValue(s,KeyLabelScale(),f)) this->SetLabelScale(f);
	if(this->UnPackValue(s,KeyLabelDigits(),i)) this->SetLabelDigits(i);

	if(this->UnPackValue(s,KeyBackgroundColor(),c)) this->SetBackgroundColor(c);

	if(this->UnPackValue(s,KeyUpdateRate(),i)) this->SetUpdateRate(i);
	if(this->UnPackValue(s,KeyRulerScale(),f)) this->SetRulerScale(f);
	if(this->UnPackValue(s,KeyOpenGLCoordinates(),b)) this->SetOpenGLCoordinates(b);
	if(this->UnPackValue(s,KeyBoxSize(),f)) this->SetBoxSize(f);

	mClipPlane->GetDirection(fv);
	if(this->UnPackValue(s,KeyClipPlaneDirection(),fv,3)) this->SetClipPlaneDirection(fv);
	if(this->UnPackValue(s,KeyClipPlaneDistance(),f)) this->SetClipPlaneDistance(f);

	if(this->UnPackValue(s,KeyPostScriptPaperFormat(),i)) mWriter->SetPostScriptPaperFormat(i);
	if(this->UnPackValue(s,KeyPostScriptOrientation(),i)) mWriter->SetPostScriptOrientation(i);

	sv[0] = mBox3AxesLabels->GetXLabel();
	sv[1] = mBox3AxesLabels->GetYLabel();
	sv[2] = mBox3AxesLabels->GetZLabel();
	for(i=0; i<6; i++) fv[i] = mBox3AxesLabels->GetRanges()[i];
	if(this->UnPackValue(s,KeyAxesBoxLabels(),sv,3) || this->UnPackValue(s,KeyAxesBoxRanges(),fv,6))
	{
		this->SetAxesBox(sv[0],sv[1],sv[2],fv[0],fv[1],fv[2],fv[3],fv[4],fv[5]);
	}

	//
	//  Light parameters
	//
	mRenderTool->GetLights()->GetAngles(fv);
	if(this->UnPackValue(s,KeyLightAngles(),fv,2))
	{
		mRenderTool->GetLights()->SetAngles(fv[0],fv[1]);
		this->ClearCache();
	}
	mRenderTool->GetLights()->GetIntensities(fv);
	if(this->UnPackValue(s,KeyLightIntensity(),fv,3))
	{
		mRenderTool->GetLights()->SetIntensities(fv[0],fv[1],fv[2]);
		this->ClearCache();
	}

	//
	//  Clone (since clones are always created after the paret window, this should work always
	//
	if(this->UnPackValue(s,KeyCloneOfWindow(),i)) if(i >= 0) BecomeClone(this->GetControlModule()->GetViewModule(i));

	//
	//  Current instances
	//
	if(this->UnPackValue(s,KeyMarkerCurrent(),i))
	{
		mFamMarker->SetCurrentMemberIndex(i);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyParticlesCurrent(),i))
	{
		mFamParticles->SetCurrentMemberIndex(i);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeySurfaceCurrent(),i))
	{
		mFamSurface->SetCurrentMemberIndex(i);
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyCrossSectionCurrent(),i))
	{
		mFamCrossSection->SetCurrentMemberIndex(i);
		this->ClearCache();
	}

	//
	//  "Action" keys
	//
#ifdef I_CHECK1
	bool saveReportMissingKeys = iObject::ReportMissingKeys;
	iObject::ReportMissingKeys = false; //action keys are not part of the states
#endif

	if(this->UnPackValue(s,KeyPlaceMarker(),b)) this->PlaceMarker(b);
	if(this->UnPackValue(s,KeyMoveMarkerCaption(),b)) this->MoveMarkerCaption(b);
	if(this->UnPackValue(s,KeyDumpImage(),b) && b) this->DumpImages();

#ifdef I_CHECK1
	iObject::ReportMissingKeys = saveReportMissingKeys;
#endif
}


void iViewModule::PlaceMarker(bool s)
{
	if(s)
	{
		mMarkerPlacer->SetPosition(mFamMarker->GetCurrentMember()->GetPosition()[0],mFamMarker->GetCurrentMember()->GetPosition()[1],mFamMarker->GetCurrentMember()->GetPosition()[2]);
		mMarkerPlacer->On();
	}
	else
	{
		mMarkerPlacer->Off();
	}
}


void iViewModule::MoveMarkerCaption(bool s)
{
	if(s)
	{
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyleCaption);
	}
	else
	{
		mRenderTool->GetInteractor()->SetInteractorStyle(mInteractorStyle[mCurrentInteractorStyle]);
	}
}


void iViewModule::ShowMarkerLegend(bool s)
{
	mIsMarkerLegendOn = s;
	if(s) 
	{
		this->BuildMarkerLegend();
		mMarkerLegend->VisibilityOn(); 
	}
	else mMarkerLegend->VisibilityOff();
	this->ClearCache();
}


void iViewModule::BuildMarkerLegend(int n)
{
	int i;
	if(n<0 || n>mFamMarker->GetMaxMemberIndex())
	{
		mMarkerLegend->SetNumberOfEntries(1+mFamMarker->GetMaxMemberIndex());
		for(i=0; i<=mFamMarker->GetMaxMemberIndex(); i++)
		{
			//
			//  Fix a bug in vtkLegendBoxActor
			//
			if(mFamMarker->GetMember(i)->GetCaptionText().Length() > 0) mMarkerLegend->SetEntry(i,mFamMarker->GetMember(i)->GetMarkerObject()->GetOutput(),mFamMarker->GetMember(i)->GetCaptionText().ToCharPointer(),mFamMarker->GetMember(i)->GetColor(0).ToVTK()); else mMarkerLegend->SetEntry(i,mFamMarker->GetMember(i)->GetMarkerObject()->GetOutput()," ",mFamMarker->GetMember(i)->GetColor(0).ToVTK());
		}
	}
	else
	{
		//
		//  Fix a bug in vtkLegendBoxActor
		//
		if(mFamMarker->GetMember(n)->GetCaptionText().Length() > 0) mMarkerLegend->SetEntry(n,mFamMarker->GetMember(n)->GetMarkerObject()->GetOutput(),mFamMarker->GetMember(n)->GetCaptionText().ToCharPointer(),mFamMarker->GetMember(n)->GetColor(0).ToVTK()); else mMarkerLegend->SetEntry(n,mFamMarker->GetMember(n)->GetMarkerObject()->GetOutput()," ",mFamMarker->GetMember(n)->GetColor(0).ToVTK());
	}
}


void iViewModule::SetMarkerLegendPosition(int p)
{
	if(p>=0 && p<=1)
	{
		mMarkerLegendPosition = p;
		switch (p)
		{
		case 0:
			{
				mMarkerLegend->SetPosition(0.73,0.02);
				break;
			}
		case 1:
			{
				mMarkerLegend->SetPosition(0.02,0.02);
				break;
			}
		}
		this->ClearCache();
	}
}


float iViewModule::GetMemorySize() const
{
	float s = 0.0;
	s += mDataReader->GetMemorySize();
	s += mFamSurface->GetMemorySize();
	s += mFamCrossSection->GetMemorySize();
	s += mFamVolume->GetMemorySize();
	s += mFamParticles->GetMemorySize();
	s += mFamVectorField->GetMemorySize();
	s += mFamTensorField->GetMemorySize();
	return s;
}


//
//  Coordinate transformation functions
//
void iViewModule::SetOpenGLCoordinates(bool s)
{
	if(s == (mBoxSize<0.0)) return;
	if(s) 
	{
		mBoxSize = -fabs(mBoxSize); 
		mBox3AxesLabels->SetRanges(-1.0,1.0,-1.0,1.0,-1.0,1.0);
	}
	else 
	{
		mBoxSize = fabs(mBoxSize);
		mBox3AxesLabels->SetRanges(0.0,mBoxSize,0.0,mBoxSize,0.0,mBoxSize);
	}
	if(mBoxType == 3) this->Render();
	this->ClearCache();
}


void iViewModule::SetBoxSize(float s)
{ 
	if(s > 0.0)
	{
		if(mBoxSize < 0) mBoxSize = -s; else
		{
			mBoxSize = s;
			mBox3AxesLabels->SetRanges(0.0,s,0.0,s,0.0,s); 
			if(mBoxType == 3) this->Render();
		}
		this->ClearCache();
	}
}


void iViewModule::SetClipPlaneDistance(float l)
{
	mClipPlane->SetDistance(l);
	this->ClearCache();
}


float iViewModule::GetClipPlaneDistance() const
{
	return mClipPlane->GetDistance();
}


void iViewModule::SetClipPlaneDirection(const float *nd)
{
	mClipPlane->SetDirection(nd);
	this->ClearCache();
}


const float* iViewModule::GetClipPlaneDirection() const
{
	return mClipPlane->GetDirection();
}


void iViewModule::SetImageComposer(iImageComposer *ic)
{
	mImageComposer = ic;
}


//
//  DataConsumer registry functionality
//
void iViewModule::RegisterDataConsumer(iDataConsumer *c)
{
	mDataConsumers.AddUnique(c);
}


void iViewModule::UnRegisterDataConsumer(iDataConsumer *c)
{
	mDataConsumers.Remove(c);
}


void iViewModule::NotifyDataConsumers(const iDataType &type, int info, void *data)
{
	int i;
	for(i=0; i<mDataConsumers.Size(); i++) if(mDataConsumers[i]->IsUsingData(type,false))
	{
		mDataConsumers[i]->SyncWithData(info,data);
	}
}


void iViewModule::SetRulerScale(float s)
{
	mRulerActor->SetScale(s);
	this->ClearCache();
}


float iViewModule::GetRulerScale() const
{
	return mRulerActor->GetScale();
}


void iViewModule::SetStereo(bool s)
{
	mRenderTool->SetStereo(s);
	this->ClearCache();
}


bool iViewModule::GetStereo() const
{
	return 	mRenderTool->GetStereo();
}


void iViewModule::SetStereoType(int n)
{
	mRenderTool->SetStereoType(n);
	this->ClearCache();
}


int iViewModule::GetStereoType() const
{
	return mRenderTool->GetStereoType();
}


//
//  Access to RenderTool components
//
void iViewModule::SetRenderWindowSize(int w, int h)
{
	if(mRenderTool != 0) mRenderTool->SetRenderWindowSize(w,h);
	this->ClearCache();
}

 
const int* iViewModule::GetRenderWindowSize() const
{ 
	return mRenderTool->GetRenderWindowSize(); 
}


vtkRenderer* iViewModule::GetRenderer() const
{
	return mRenderTool->GetRenderer();
}


vtkRenderWindow* iViewModule::GetRenderWindow() const
{
	return mRenderTool->GetRenderWindow();
}


vtkRenderWindowInteractor* iViewModule::GetInteractor() const
{
	return mRenderTool->GetInteractor(); 
}


void iViewModule::SetLabelName(const iString &s)
{
	mLabelName = s;
	this->UpdateLabel();
	this->ClearCache();
}


void iViewModule::SetLabelUnit(const iString &s)
{ 
	mLabelUnit = s; 
	this->UpdateLabel();
	this->ClearCache();
}


void iViewModule::SetLabelScale(float v)
{
	mLabelScale = v;
	this->UpdateLabel();
	this->ClearCache();
}


void iViewModule::SetLabelDigits(int v)
{
	if(v>=0 && v<=7)
	{
		mLabelDigits = v;
		this->UpdateLabel();
		this->ClearCache();
	}
}
