/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "isurfaceviewobject.h"


#include "icolorbars.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatastretch.h"
#include "ierror.h"
#include "iextensionfactory.h"
#include "imarkerviewobject.h"
#include "ipicker.h"
#include "ireplicatedactor.h"
#include "isurfaceparallelpipeline.h"
#include "isurfaceserialpipeline.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"
#include "iviewobjectobserver.h"

#include <vtkStructuredPoints.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWOBJECT_DEFINE_TYPE(iSurfaceViewObject,Surface,s);

IOBJECT_DEFINE_KEY(iSurfaceViewObject,AttachProbeToMarker,am,OffsetInt,1);  //  for sphere and plane
IOBJECT_DEFINE_KEY(iSurfaceViewObject,IsoSurfaceLevel,l,Float,1);
IOBJECT_DEFINE_KEY(iSurfaceViewObject,Method,m,Int,1);
IOBJECT_DEFINE_KEY(iSurfaceViewObject,NormalsFlipped,nf,Bool,1);
IOBJECT_DEFINE_KEY(iSurfaceViewObject,IsoSurfaceOptimization,op,Bool,1);
IOBJECT_DEFINE_KEY(iSurfaceViewObject,ProbePositionToPickedPoint,p2p,Bool,1); // action key
IOBJECT_DEFINE_KEY(iSurfaceViewObject,PlaneDirection,pd,Float,3);
IOBJECT_DEFINE_KEY(iSurfaceViewObject,PaintVar,pv,OffsetInt,2);
IOBJECT_DEFINE_KEY(iSurfaceViewObject,IsoSurfaceReduction,rd,Int,1);
IOBJECT_DEFINE_KEY(iSurfaceViewObject,AlternativeIsoSurfaceReductionMethod,rda,Bool,1);
IOBJECT_DEFINE_KEY(iSurfaceViewObject,IsoSurfaceSmoothing,sm,Int,1);
IOBJECT_DEFINE_KEY(iSurfaceViewObject,IsoSurfaceVar,v,OffsetInt,1);

IOBJECT_DEFINE_DISTANCE_KEY(iSurfaceViewObject,Size,s);
IOBJECT_DEFINE_POSITION_KEY(iSurfaceViewObject,Position,x);
//
//  Inherited keys
//
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COMMON(iSurfaceViewObject);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iSurfaceViewObject);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COLOR(iSurfaceViewObject,2);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_OPACITY(iSurfaceViewObject,2);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_PALETTE(iSurfaceViewObject,2);


//
// iSurfaceViewObject class
//
iSurfaceViewObject* iSurfaceViewObject::New(iViewModule *vm)
{
	iSurfaceViewObject *tmp = iPointerCast<iSurfaceViewObject,iObject>(iExtensionFactory::CreateObject(Type(),vm));
	IERROR_ASSERT_NULL_POINTER(tmp);
	//
	//  Create pipeline (must be created after the ibject is fully created)
	//	
	tmp->mPipeline = tmp->CreateParallelPipeline(); IERROR_ASSERT_NULL_POINTER(tmp->mPipeline);
	tmp->mActors[0]->SetInput(tmp->mPipeline->GetOutput(0));
	tmp->mActors[1]->SetInput(tmp->mPipeline->GetOutput(1));

	tmp->Reset();

	return tmp;
}


iSurfaceViewObject::iSurfaceViewObject(iViewModule *vm, const iDataInfo &info) : iSolidViewObject(vm,info,2,0U), mLev(0.0f), mVar(0), mPaintVarExt(0), mPaintVarInt(0), mSize(vm,true)
{
	int i;

	mName = "iSurfaceViewObject";
	mObjectType = _ObjectTypeSurface;

	if(mPrimaryDataInfo->Index(iDataType::UniformScalars()) == -1)
	{
		IERROR_REPORT_FATAL_ERROR("iSurfaceViewObject is configured incorrectly.");
	}
	for(i=0; i<mPrimaryDataInfo->Count(); i++)
	{
		mLev.Add(0.0f);
		mVar.Add(0);
		mPaintVarExt.Add(-1);
		mPaintVarInt.Add(-1);
	}

	int nvar = this->GetLimits()->GetNumVars();
//
//  mPaintVar[0,1] = -1 means no painting on the exterior,interior.
//
	float lmin = this->GetLimits()->GetMin(this->GetIsoSurfaceVar());
	float lmax = this->GetLimits()->GetMax(this->GetIsoSurfaceVar());

	if(this->GetIsoSurfaceVar()==0 && (this->GetLimits()->GetName(0)=="Neutral fraction" || this->GetLimits()->GetName(0)=="HI fraction"))
	{
		mLev[0] = 0.5; 
		mLev[0] = (mLev[0] < lmin) ? lmin : mLev[0];
		mLev[0] = (mLev[0] > lmax) ? lmax : mLev[0];
		if(nvar > 1)
		{
			mPaintVarExt[0] = 1;
			mActors[0]->SyncWithLimits(this->GetLimits(),1);
			mActors[0]->SetScalarVisibility(true);
		}
	} 
	else mLev[0] = lmax; 
	
	mReduction = 1;
	mSmoothing = 0;
	mNormalsFlipped = false;
	mOptimization = false;
	mAlternativeReductionMethod = false;

	mShading = true;
	this->UpdateMaterialProperties();
	mMethod = _SurfaceMethodIsoSurface;
	mSize = 0.5;
	mDir[0] = 0.0;
	mDir[1] = 0.0;
	mDir[2] = 1.0;

	mAttachedMarkerIndex = -1;
	mAttachedMarker = 0;

	//
	//  Add observer to keep information about this object
	//
	iViewObjectObserver::IsoSurfaceData d;
	d.Var = mVar[0];
	d.Level = mLev[0];
	mObjectObserver->SetIsoSurfaceData(d);
}


iSurfaceViewObject::~iSurfaceViewObject()
{
	mPipeline->Delete();
}


void iSurfaceViewObject::FinishInitialization()
{
}


iSurfaceParallelPipeline* iSurfaceViewObject::CreateParallelPipeline()
{
	iSurfaceParallelPipeline *tmp = new iSurfaceParallelPipeline(this); IERROR_ASSERT_NULL_POINTER(tmp);
	tmp->FinishInitialization();
	return tmp;
}


iSurfaceSerialPipeline* iSurfaceViewObject::CreateSerialPipeline()
{
	return new iSurfaceSerialPipeline(this);
}


void iSurfaceViewObject::ResetInput()
{
	mPipeline->SetInput(iPointerCast<vtkImageData,vtkDataSet>(this->GetData()));
}


void iSurfaceViewObject::ResetPipeline()
{
}


void iSurfaceViewObject::ShowColorBarsBody(bool show)
{
	if(this->GetPaintVar(0) >= 0) this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPrioritySurfaceExterior,this->GetPaintVar(0),this->GetActiveDataType(),mPalettes[0],show);
	if(this->GetPaintVar(1) >= 0) this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPrioritySurfaceInterior,this->GetPaintVar(1),this->GetActiveDataType(),mPalettes[1],show);
}


void iSurfaceViewObject::SetIsoSurfaceVar(int v)
{ 
	if(v>=0 && v<this->GetLimits()->GetNumVars() && v!=this->GetIsoSurfaceVar())
	{
		mVar[mActiveDataTypeIndex] = v;
		mPipeline->UpdateIsoSurfaceVar();
		this->SetIsoSurfaceLevel(this->GetLimits()->GetMax(this->GetIsoSurfaceVar()));
		this->SetNormalsFlipped(mNormalsFlipped);
		this->SyncWithData(this->GetIsoSurfaceVar()+1,0);
		this->ClearCache();
	}
}


void iSurfaceViewObject::SetIsoSurfaceLevel(float l)
{
	if(this->GetIsoSurfaceVar() < this->GetLimits()->GetNumVars())
	{
		float lmin = this->GetLimits()->GetMin(this->GetIsoSurfaceVar());
		float lmax = this->GetLimits()->GetMax(this->GetIsoSurfaceVar());
		l = (l < lmin) ? lmin : l;
		l = (l > lmax) ? lmax : l;
		mLev[mActiveDataTypeIndex] = l;
		mPipeline->UpdateIsoSurfaceLevel();
		this->ClearCache();

		//
		//  Update observer
		//
		iViewObjectObserver::IsoSurfaceData d;
		d.Var = mVar[0];
		d.Level = mLev[0];
		mObjectObserver->SetIsoSurfaceData(d);
	}
}


float iSurfaceViewObject::GetIsoSurfaceLevel() const
{
	if(this->GetIsoSurfaceVar() < this->GetLimits()->GetNumVars())
	{
		return mLev[mActiveDataTypeIndex];
	}
	else return 0.0f;
}


void iSurfaceViewObject::SetPaintVar(int s, int v)
{
	if(s<0 || s>1) return;

	if(v<-1 || v>=this->GetLimits()->GetNumVars()) v = -1;

	this->ShowColorBars(false);
	if(s == 0) mPaintVarExt[mActiveDataTypeIndex] = v; else mPaintVarInt[mActiveDataTypeIndex] = v;
	this->ShowColorBars(true);
	mPipeline->UpdateVarPaint(s);
	if(s == 0)
	{
		if(v>=0 && v<this->GetLimits()->GetNumVars()) 
		{
			mActors[0]->SyncWithLimits(this->GetLimits(),v);
			mActors[0]->SetScalarVisibility(true);
		} 
		else 
		{
			mActors[0]->SetScalarVisibility(false);
		}
	} 
	else 
	{
		if(v>=0 && v<this->GetLimits()->GetNumVars()) 
		{
			mActors[0]->SyncWithLimits(this->GetLimits(),v);
			mActors[1]->SetScalarVisibility(true);
			mActors[1]->VisibilityOn();
		} 
		else 
		{
			mActors[1]->SetScalarVisibility(false);
			mActors[1]->VisibilityOff();
		}
	}
	this->ClearCache();
}


void iSurfaceViewObject::SetIsoSurfaceReduction(int s)
{
	if(s != mReduction)
	{
		mReduction = s; 
		mPipeline->UpdateIsoSurfacePipeline();
		this->ClearCache();
	}
} 


void iSurfaceViewObject::SetAlternativeIsoSurfaceReductionMethod(bool s)
{
	if(s != mAlternativeReductionMethod)
	{
		mAlternativeReductionMethod = s; 
		mPipeline->UpdateIsoSurfacePipeline();
		this->ClearCache();
	}
} 


void iSurfaceViewObject::SetIsoSurfaceOptimization(bool s)
{
	if(s != mOptimization)
	{
		mOptimization = s; 
		mPipeline->UpdateIsoSurfacePipeline();
		this->ClearCache();
	}
} 


void iSurfaceViewObject::SetIsoSurfaceSmoothing(int p)
{
	if(p>=0 && p<10 && p!=mSmoothing)
	{
		mSmoothing = p;
		mPipeline->UpdateIsoSurfacePipeline();
		this->ClearCache();
	}
}


void iSurfaceViewObject::SetNormalsFlipped(bool s)
{ 
	if(mNormalsFlipped != s)
	{
		mNormalsFlipped = s;
		mPipeline->UpdateNormalsFlipped();
		this->ClearCache();
	}
}


void iSurfaceViewObject::SetOpacity(int l, float o)
{
	if(l == 0)
	{
		iSolidViewObject::SetOpacity(0,o);
		iSolidViewObject::SetOpacity(1,o);
	}
}


void iSurfaceViewObject::ShowBody(bool show)
{
	if(show)
	{
		if(this->GetPaintVar(0) >= 0) 
		{
			mActors[0]->SyncWithLimits(this->GetLimits(),this->GetPaintVar(0));
		}
		if(this->GetPaintVar(1) >= 0) 
		{
			mActors[0]->SyncWithLimits(this->GetLimits(),this->GetPaintVar(1));
			mActors[1]->VisibilityOn();
		}
		else mActors[1]->VisibilityOff();
		mActors[0]->VisibilityOn();
		this->ShowColorBars(true);
	} 
	else 
	{
		this->ShowColorBars(false);
		mActors[0]->VisibilityOff();
		mActors[1]->VisibilityOff();
	}
}


float iSurfaceViewObject::GetMemorySize() const
{
	return mPipeline->GetMemorySize();
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iSurfaceViewObject::SolidViewObjectPackStateBody(iString &s) const
{
	int iv[2];

	this->PackValue(s,KeyNormalsFlipped(),mNormalsFlipped);
	this->PackValue(s,KeyIsoSurfaceReduction(),mReduction);
	this->PackValue(s,KeyAlternativeIsoSurfaceReductionMethod(),mAlternativeReductionMethod);
	this->PackValue(s,KeyIsoSurfaceOptimization(),mOptimization);

	this->PackValue(s,KeyIsoSurfaceVar(),this->GetIsoSurfaceVar());
	iv[0] = this->GetPaintVar(0);
	iv[1] = this->GetPaintVar(1);
	this->PackValue(s,KeyPaintVar(),iv,2);
	this->PackValue(s,KeyIsoSurfaceSmoothing(),mSmoothing);

	this->PackValue(s,KeyIsoSurfaceLevel(),this->GetIsoSurfaceLevel());

	this->PackValue(s,KeyMethod(),mMethod);
	this->PackValue(s,KeyPlaneDirection(),mDir,3);
	this->PackValue(s,KeyAttachProbeToMarker(),mAttachedMarkerIndex);  //  for sphere and plane

	this->PackValueDistance(s,KeySize(),mSize);
	this->PackValuePosition(s,KeyPosition(),mPosition);
}


void iSurfaceViewObject::SolidViewObjectUnPackStateBody(const iString &s)
{
	int i, iv[2]; bool b; float f; float fv[3];

	if(this->UnPackValue(s,KeyNormalsFlipped(),b)) this->SetNormalsFlipped(b);
	if(this->UnPackValue(s,KeyIsoSurfaceReduction(),i)) this->SetIsoSurfaceReduction(i);
	if(this->UnPackValue(s,KeyAlternativeIsoSurfaceReductionMethod(),b)) this->SetAlternativeIsoSurfaceReductionMethod(b);
	if(this->UnPackValue(s,KeyIsoSurfaceOptimization(),b)) this->SetIsoSurfaceOptimization(b);

	if(this->UnPackValue(s,KeyIsoSurfaceVar(),i)) this->SetIsoSurfaceVar(i);
	
	iv[0] = this->GetPaintVar(0);
	iv[1] = this->GetPaintVar(1);
	if(this->UnPackValue(s,KeyPaintVar(),iv,2))
	{
		this->SetPaintVar(0,iv[0]);
		this->SetPaintVar(1,iv[1]);
	}

	if(this->UnPackValue(s,KeyIsoSurfaceSmoothing(),i)) this->SetIsoSurfaceSmoothing(i);
	if(this->UnPackValue(s,KeyMethod(),i)) this->SetMethod(i);

	if(this->UnPackValue(s,KeyIsoSurfaceLevel(),f)) this->SetIsoSurfaceLevel(f);

	for(i=0; i<3; i++) fv[i] = mDir[i];
	if(this->UnPackValue(s,KeyPlaneDirection(),fv,3)) this->SetPlaneDirection(fv);

	if(this->UnPackValue(s,KeyAttachProbeToMarker(),i)) this->SetAttachProbeToMarker(i);  //  for sphere and plane

	if(this->UnPackValueDistance(s,KeySize(),mSize)) this->SetSize(mSize);
	if(this->UnPackValuePosition(s,KeyPosition(),mPosition)) this->SetPosition(mPosition);
	
	//
	//  Special "action" keys
	//
#ifdef I_CHECK1
	bool saveReportMissingKeys = iObject::ReportMissingKeys;
	iObject::ReportMissingKeys = false; //action keys are not part of the states
#endif

	if(this->UnPackValue(s,KeyProbePositionToPickedPoint(),b) && b) this->SetProbePositionToPickedPoint(); 

#ifdef I_CHECK1
	iObject::ReportMissingKeys = saveReportMissingKeys;
#endif
}


void iSurfaceViewObject::SetMethod(int m)
{
	if(m>=0 && m<3 && m!=mMethod)
	{
		mMethod = m;
		this->SetShading(mMethod != _SurfaceMethodPlane);
		mPipeline->UpdateMethod();
		this->ClearCache();
	}
}


void iSurfaceViewObject::SetSize(const iDistance &s)
{
	if(s < 1.0)
	{
		mSize = s;
		mPipeline->UpdateProbeSize();
		this->ClearCache();
	}
}
	

void iSurfaceViewObject::DetachFromMarker()
{
	if(mAttachedMarker != 0)
	{
		mAttachedMarker->DetachViewObject(this);
		mAttachedMarker = 0;
		mAttachedMarkerIndex = -1;
	}
}


void iSurfaceViewObject::SetPosition(const iPosition &pos)
{
	this->DetachFromMarker();

	mPosition = pos;
	mPosition.PeriodicWrap();
	mPipeline->UpdateProbePosition();
	this->ClearCache();
}


void iSurfaceViewObject::SetPlaneDirection(const float n[3])
{
	int i;
	for(i=0; i<3; i++) mDir[i] = n[i];
	mPipeline->UpdatePlaneDirection();
	this->ClearCache();
}


void iSurfaceViewObject::SetAttachProbeToMarker(int m)
{
	this->DetachFromMarker();

	if(m>=0 && m<=this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex())
	{
		mAttachedMarkerIndex = m;
		mAttachedMarker = this->GetViewModule()->GetMarkerFamily()->GetMember(mAttachedMarkerIndex);
		mAttachedMarker->AttachViewObject(this);

		mPosition = mAttachedMarker->GetPosition();
		mPipeline->UpdateProbePosition();
		this->ClearCache();
	}
}


void iSurfaceViewObject::SetProbePositionToPickedPoint()
{
	double *x = this->GetViewModule()->GetPicker()->GetPickPosition();

	this->DetachFromMarker();

	mPosition = x;
	mPipeline->UpdateProbePosition();
	this->ClearCache();
}


void iSurfaceViewObject::UpdateAfterMarkerMove()
{
	if(mAttachedMarker != 0)
	{
		mPosition = mAttachedMarker->GetPosition();
		mPipeline->UpdateProbePosition();
		this->ClearCache();
	}
}


bool iSurfaceViewObject::CanBeShown() const
{
	return (this->GetIsoSurfaceVar() < this->GetLimits()->GetNumVars());
}


void iSurfaceViewObject::ViewObjectSyncWithData(int info, void *)
{
	int i;

	if(info==0 || info==1+this->GetIsoSurfaceVar()) this->SetIsoSurfaceLevel(this->GetIsoSurfaceLevel());

	for(i=0; i<2; i++)
	{
		if(info==0 || info==1+this->GetPaintVar(i)) this->SetPaintVar(i,this->GetPaintVar(i));
	}
}

