/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "igenericfilter.h"


#include "ierror.h"

#include <vtkDataSetAttributes.h>
#include <vtkFloatArray.h>
#include <vtkImageData.h>
#include <vtkPolyData.h>


template<class Source, class OutputType>
void iGenericSource<Source,OutputType>::ExecuteData(vtkDataObject *)
{
	OutputType *output = this->GetOutput();

	IERROR_ASSERT_NULL_POINTER(output);

	this->ProduceOutput(output);
}

	
template<class Filter, class InputType, class OutputType>
iGenericFilter<Filter,InputType,OutputType>::iGenericFilter(int numInputs)
{
	if(numInputs > 1)
	{
#ifdef IVTK_4
		this->SetNumberOfInputs(numInputs);
#else
		this->SetNumberOfInputConnections(0,numInputs);
#endif
	}
}


#ifdef IVTK_4

template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::ExecuteInformation()
{
	InputType *input = dynamic_cast<InputType*>(this->GetInput());
	OutputType *output = this->GetOutput();

	IERROR_ASSERT_NULL_POINTER(output);
	if(input != 0)
	{
		this->InitExecution(input,output);
	}

	this->Filter::ExecuteInformation();
}

#else

#include <vtkDemandDrivenPipeline.h>
#include <vtkInformation.h>
#include <vtkInformationVector.h>


template<class Filter, class InputType, class OutputType>
int iGenericFilter<Filter,InputType,OutputType>::ProcessRequest(vtkInformation *request, vtkInformationVector **inInfo, vtkInformationVector *outInfo)
{
	//
	//  This replaces ExecuteInformation call
	//
	if(request->Has(vtkDemandDrivenPipeline::REQUEST_INFORMATION()))
	{
		InputType *input = dynamic_cast<InputType*>(this->GetInput());
		OutputType *output = this->GetOutput();

		IERROR_ASSERT_NULL_POINTER(output);
		
		if(input != 0)
		{
			this->InitExecution(input,output);
		}

		if(this->IsA("vtkSource"))
		{
			//
			//  PipelineInformation may contain wrong values of Origin and Spacing. These values get written into the Origin and Spacing fields
			//  of vtkImageData in vtkSource::ProcessRequest. We need to fix this bug by putting correct values into PipelineInformation.
			//
			int i;
			vtkInformation *info;
			for(i=0; i<inInfo[0]->GetNumberOfInformationObjects(); i++)
			{
				info = inInfo[0]->GetInformationObject(i);
				if(info != 0)
				{
					vtkImageData *data = vtkImageData::SafeDownCast(info->Get(vtkDataObject::DATA_OBJECT()));
					if(data != 0)
					{
						double v[3];
						vtkInformation *info = data->GetPipelineInformation();
						if(info != 0)
						{
							if(info->Has(vtkDataObject::ORIGIN()))
							{
								data->GetOrigin(v);
								info->Set(vtkDataObject::ORIGIN(),v,3);
							}
							if(info->Has(vtkDataObject::SPACING()))
							{
								data->GetSpacing(v);
								info->Set(vtkDataObject::SPACING(),v,3);
							}
						}
					}
				}
			}
		}
	}

	return this->Filter::ProcessRequest(request,inInfo,outInfo);
}

#endif


template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::ExecuteData(vtkDataObject *)
{
	InputType *input = dynamic_cast<InputType*>(this->GetInput());
	OutputType *output = this->GetOutput();

	IERROR_ASSERT_NULL_POINTER(output);
	if(input == 0)
	{
		output->Initialize();
		return;
	}

	this->ProduceOutput(input,output);
	this->VerifyResults(input,output);
}

	
template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::InitExecution(InputType *, OutputType *)
{
	//
	//  Default implementation is to do nothing
	//
}


template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::VerifyResults(InputType *, OutputType *)
{
	//
	//  Default implementation is to do nothing
	//
}


template<class Filter, class InputType, class OutputType>
bool iGenericFilter<Filter,InputType,OutputType>::ScalarsInit(vtkDataSetAttributes *data, vtkIdType nSizeOut, int nDimOut)
{
	if(data==0 || data->GetScalars()==0)
	{
		wScalarArrIn = wScalarArrOut = 0;
		wScalarPtrIn = wScalarPtrOut = 0;
		wScalarDimIn = wScalarDimOut = 0;
		return true;
	}

	wScalarArrIn = vtkFloatArray::SafeDownCast(data->GetScalars()); 
	if(wScalarArrIn == 0)
    {
		vtkErrorMacro("Input scalars are not float.");
		return false;
    }
	wScalarDimIn = wScalarArrIn->GetNumberOfComponents();
	wScalarPtrIn = wScalarArrIn->GetPointer(0);

	if(nSizeOut < 1) nSizeOut = wScalarArrIn->GetNumberOfTuples();
	if(nDimOut < 1) nDimOut = wScalarDimIn;

	wScalarArrOut = vtkFloatArray::New();
	if(wScalarArrOut == 0)
    {
		vtkErrorMacro("Not enough memory to create output scalars.");
		return false;
    }
	wScalarArrOut->SetNumberOfComponents(nDimOut);
	wScalarArrOut->SetNumberOfTuples(nSizeOut);

	wScalarDimOut = nDimOut;
	wScalarPtrOut = wScalarArrOut->GetPointer(0);
	if(wScalarPtrOut == 0)
    {
		vtkErrorMacro("Not enough memory to create output scalars.");
		return false;
    }
	return true;
}


template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::ScalarsDone(vtkDataSetAttributes *data)
{
	if(data!=0 && wScalarArrOut!=0)
	{
		data->SetScalars(wScalarArrOut);
		wScalarArrOut->Delete();
	}
}


template<class Filter>
void iGenericPolyDataToPolyDataFilter<Filter>::VerifyResults(vtkPolyData *input, vtkPolyData *output)
{
	if(input->GetPoints()!=0 && output->GetPoints()!=0 && (input->GetPoints()->GetDataType()!=output->GetPoints()->GetDataType()))
	{
#ifdef I_DEBUG
		int it = input->GetPoints()->GetDataType();
		int ot = output->GetPoints()->GetDataType();
#endif
		vtkErrorMacro("Incompatible point types of input and output");
	}
}

