/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ierrorstatus.h"


#include "ierrorstatusholder.h"


const iString iErrorStatus::mAbortMessage("Aborted.");


iErrorStatus::iErrorStatus(iErrorStatusHolder*)
{
	mLastSupervisor = mSupervisor = 0;

	this->Clear();
}

	
void iErrorStatus::Monitor(iErrorStatusHolder *parent, const iString &prefix)
{
	if(parent != 0) this->Monitor(parent->GetErrorStatus(),prefix);
}


void iErrorStatus::Monitor(iErrorStatus *object, const iString &prefix)
{
	//
	//  this->Clear() is configured so that this function needs to be called again before
	//  every execution
	//
	if(object != 0)
	{
		object->mSupervisor = this;
		object->mSupervisorPrefix = prefix;
	}
}


void iErrorStatus::Clear()
{
	//
	//  The proper use of this class is to call this function before any activity starts.
	//  This function checks the current and the last supervisor -  if they are the same, the
	//  supervisor is removed. This way no action is needed after monitoring is set - at the
	//  next execution, old supervisor will be automatically removed.
	//
	mData.Message.Clear();
	mData.Level = iMath::_IntMax;
	mInUpdate = false;

	if(mSupervisor!=0 && mSupervisor==mLastSupervisor)
	{
		//
		//  Supervision expired
		//
		mSupervisor = mLastSupervisor = 0;
	}
	mLastSupervisor = mSupervisor;
}


void iErrorStatus::SetAbort(int lev)
{
	this->Set(mAbortMessage,lev);
}


void iErrorStatus::Set(const iString &msg, int lev)
{
	if(mInUpdate) return; // avoid recursion

	mData.Message = msg;
	mData.Level = lev;
	//
	//  If we are monitored, report to supervisor
	//
	if(mSupervisor != 0)
	{
		//
		//  Reset supervisor's error info if it has no error, or its error is of lower level
		//
		if(mSupervisor->GetMessage().IsEmpty() || mSupervisor->GetLevel()<lev)
		{
			//
			//  Avoid recursion
			//
			mInUpdate = true;
			mSupervisor->Set(mSupervisorPrefix+msg,lev);
			mInUpdate = false;
		}
	}
}

