/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idataexplorer.h"


#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "iextensionfactory.h"
#include "igriddataconverter.h"
#include "ihistogram.h"
#include "ihistogrammaker.h"
#include "isurfaceviewobject.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"

#include <vtkPolyData.h>
#include <vtkStructuredPoints.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


//
//  DataExplorer must be an object so that it can be created by the ObjectFactory
//
IOBJECT_DEFINE_TYPE(iDataExplorer,DataExplorer,-de,iObjectType::_Helper);  // helper type


iDataExplorer* iDataExplorer::New(iViewModule *vm)
{
	return iPointerCast<iDataExplorer,iObject>(iExtensionFactory::CreateObject(Type(),vm));
}


iDataExplorer::iDataExplorer(iViewModule *vm, const iDataInfo &info) : iDataConsumer(vm,info)
{
	mHistogramMaker = 0;
	mMTime = 0;
	mComputeRangeOnly = false;
	
	mComponent = -1;
	mNumComponents = 0;

	mDataConverter = iGridDataConverter::New(vm->GetSurfaceFamily()->GetMember(0)); IERROR_ASSERT_NULL_POINTER(mDataConverter);
}


iDataExplorer::~iDataExplorer()
{
	if(mHistogramMaker != 0) mHistogramMaker->Delete();
	mDataConverter->Delete();
}


void iDataExplorer::SyncWithDataBody(int, void *)
{
	if(mComponent >= mNumComponents)
	{
		mComponent = 0;
		this->SetInputComponent(mComponent);
		mMTime = 0;
	}
}


void iDataExplorer::UpdateDataType()
{
	const iDataType &type = this->GetActiveDataType();

	if(type == iDataType::UniformScalars())
	{
		mNumComponents = this->GetViewModule()->GetReader()->GetLimits(type)->GetNumVars();
		mDataConverter->SetDataRank(0U);
		mDataConverter->SetScalarInput(iPointerCast<vtkImageData,vtkDataSet>(this->GetData()));
		if(mHistogramMaker != 0)
		{
			mHistogramMaker->SetDataRank(0U);
			mHistogramMaker->SetInput(mDataConverter->GetOutput(),0);
		}
		return;
	}

	if(type == iDataType::UniformVectors())
	{
		mNumComponents = this->GetViewModule()->GetReader()->GetLimits(type)->GetNumVars();
		mDataConverter->SetDataRank(1U);
		mDataConverter->SetVectorInput(iPointerCast<vtkImageData,vtkDataSet>(this->GetData()));
		if(mHistogramMaker != 0)
		{
			mHistogramMaker->SetDataRank(1U);
			mHistogramMaker->SetInput(mDataConverter->GetOutput(),0);
		}
		return;
	}

	if(type == iDataType::UniformTensors())
	{
		mNumComponents = 3*this->GetViewModule()->GetReader()->GetLimits(type)->GetNumVars();
		mDataConverter->SetDataRank(2U);
		mDataConverter->SetTensorInput(iPointerCast<vtkImageData,vtkDataSet>(this->GetData()));
		if(mHistogramMaker != 0)
		{
			mHistogramMaker->SetDataRank(2U);
			mHistogramMaker->SetInput(mDataConverter->GetOutput(),0);
		}
		return;
	}

	if(type == iDataType::BasicParticles())
	{
		mNumComponents = this->GetViewModule()->GetReader()->GetLimits(type)->GetNumVars();
		if(mHistogramMaker != 0)
		{
			mHistogramMaker->SetDataRank(0U);
			mHistogramMaker->SetInput(this->GetData(),0);
		}
		return;
	}

	IERROR_REPORT_FATAL_ERROR("Invalid data type.");
}


iHistogramMaker* iDataExplorer::CreateHistogramMaker() const
{
	return iHistogramMaker::New(this->GetViewModule());
}


const iDataExplorer::Info& iDataExplorer::GetInfo(bool rangeonly)
{
	if(mHistogramMaker == 0)
	{
		vtkErrorMacro("iDataExplorer has no input.");
		return mInfo;
	}

	mHistogramMaker->GetInput()->Update();
	if(mHistogramMaker->GetInput()->GetMTime() > mMTime)
	{
		mComputeRangeOnly = rangeonly;
		this->Execute();
	}
	return mInfo;
}

void iDataExplorer::SetStretch(int v)
{
	if(mHistogramMaker == 0)
	{
		mHistogramMaker = this->CreateHistogramMaker();
		IERROR_ASSERT_NULL_POINTER(mHistogramMaker);
		this->UpdateDataType();
	}

	if(v != mHistogramMaker->GetStretch())
	{
		mHistogramMaker->SetStretch(v);
		mMTime = 0;
	}
}

void iDataExplorer::SetInputComponent(int c)
{
	if(mHistogramMaker == 0)
	{
		mHistogramMaker = this->CreateHistogramMaker();
		IERROR_ASSERT_NULL_POINTER(mHistogramMaker);
		this->UpdateDataType();
	}

	if(c < 0) c = 0;
	if(c >= mNumComponents) c = mNumComponents - 1;
	mComponent = c;
	this->SetInputComponentBody(c);
	mMTime = 0;
}


void iDataExplorer::SetInputComponentBody(int c)
{
	const iDataType &type = this->GetActiveDataType();

	if(type == iDataType::UniformScalars())
	{
		mDataConverter->SetCurrentVar(c);
		mHistogramMaker->SetInputComponent(0);
		return;
	}

	if(type == iDataType::UniformVectors())
	{
		mDataConverter->SetCurrentVar(0);
		mHistogramMaker->SetInputComponent(0);
		return;
	}

	if(type == iDataType::UniformTensors())
	{
		mDataConverter->SetCurrentVar(0);
		mHistogramMaker->SetInputComponent(c);
		return;
	}

	//
	//  Default action
	//
	mHistogramMaker->SetInputComponent(c);
}


void iDataExplorer::Execute()
{
	int j;

	if(mHistogramMaker == 0)
	{
		vtkErrorMacro("iDataExplorer has no input.");
		return;
	}

	if(mComputeRangeOnly)
	{
		mHistogramMaker->GetHistogramRange(mInfo.Minimum,mInfo.CellMin,mInfo.PosMin,mInfo.Maximum,mInfo.CellMax,mInfo.PosMax);
		mInfo.Histogram = 0;
		mInfo.Median = 0.0;
		mInfo.Average = 0.0;
		mInfo.Dispersion = 0.0;
		return;
	}

	const iHistogram *h = mHistogramMaker->GetHistogram(); // dereference for better performance

	if(h == 0)
	{
		vtkErrorMacro("iDataExplorer has null histogram.");
#ifdef I_DEBUG
		h = mHistogramMaker->GetHistogram();
#endif
		mInfo.Histogram = 0;
		mInfo.Maximum = iMath::_LargeFloat;
		mInfo.Minimum = -iMath::_LargeFloat;
		mInfo.Median = 0.0;
		mInfo.Average = 0.0;
		mInfo.Dispersion = 0.0;
		return;
	}

	mMTime = mHistogramMaker->GetInput()->GetMTime();

	float max = h->GetMaxValue();
	float min = h->GetMinValue();

	//
	//  make sure Min is less than Max
	//
	float s = iMath::_FloatTolerance;
	while(max<min+s && s<iMath::_LargeFloat) 
	{
		max += s; 
		s *= 2;
	}

	double sum0 = 0.0, sum1 = 0.0, sum2 = 0.0;
	
	for(j=0; j<h->N(); j++)
	{
		sum0 += h->Y(j);
		sum1 += h->Y(j)*h->X(j);
		sum2 += h->Y(j)*h->X(j)*h->X(j);
	}

	float avg, dis, med;
	if(sum0 > 0.0)
	{
		avg = min + sum1/sum0*(max-min);
		dis = sqrt(sum2/sum0-pow(sum1/sum0,2.0))*(max-min);
		med = 0.0;
		sum1 = 0.0;
		for(j=0; j<h->N(); j++)
		{
			if(sum1<0.5*sum0 && (sum1+=h->Y(j))>=0.5*sum0)
			{
				med = min + h->X(j)*(max-min);
				break;
			}
		}
	}
	else
	{
		avg = dis = med = 0.5*(max+min);
	}

	mInfo.Histogram = h;
	mInfo.Maximum = max;
	mInfo.Minimum = min;
	mInfo.Median = med;
	mInfo.Average = avg;
	mInfo.Dispersion = dis;
}
