/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idata.h"

#include <limits.h>

//
//  Templates
//
#include "iarraytemplate.h"


namespace iData_Private
{
	bool Initialized = false;
	iSearchableArray<iString> TypeQualifiers("");
	iSearchableArray<iString> TypeSpecifiers("");
	iSearchableArray<const iDataType*> TypeList(0);

	void InitPrivateNamespace()
	{
		if(!Initialized)
		{
			Initialized = true;
			TypeSpecifiers.Add("ScalarField");
			TypeSpecifiers.Add("VectorField");
			TypeSpecifiers.Add("TensorField");
			TypeSpecifiers.Add("ParticleSet");
		}
	}

	bool IsValidKeyword(const iString &kw, iString &qual, iString &spec)
	{
		if(kw.Contains(",") != 0) return false;

		iString q, s;
		switch(kw.Contains(":"))
		{
		case 0:
			{
				q = qual;
				s = kw;  // remember previous qualifier
				break;
			}
		case 1:
			{
				q = kw.Section(":",0,0) + ":";
				s = kw.Section(":",1,1);
				break;
			}
		default:
			{
				return false;
			}
		}
	
		if((!q.IsEmpty() && TypeQualifiers.Find(q)==-1) || TypeSpecifiers.Find(s)==-1) return false;

		qual = q;
		spec = s;
		return true;
	}
};


using namespace iData_Private;


//
//  DataType class
//
iDataType::iDataType(int id, const iString& tname, const iString& sname, int priority, const iString &keywords, bool primary) : mId(id)
{
	int i, n;

	InitPrivateNamespace();

	mTextName = tname;
	mShortName = sname;
	mPriority = priority;
	mKeywords = keywords;
	mIsPrimary = primary;
	mName = tname.Substitute(" ","");

	for(i=0; i<TypeList.Size(); i++)
	{
		if(mId==TypeList[i]->GetId() || mName==TypeList[i]->GetName() || mShortName==TypeList[i]->GetShortName() || (primary && mKeywords==TypeList[i]->mKeywords))
		{
			IERROR_REPORT_FATAL_ERROR("Incorrectly configured DataType object.");
		}
	}

	TypeList.Add(this);

	//
	//  Analyse keywords
	//
	if(!mKeywords.IsEmpty())
	{
		iString q = mKeywords.Section(":",0,0);
		iString s = mKeywords.Section(":",1,1);
		if(q.IsEmpty() || s.IsEmpty())
		{
			IERROR_REPORT_FATAL_ERROR("Incorrectly configured DataType object.");
		}

		q += ":";
		if(TypeQualifiers.Find(q) == -1) // this qualifier has not been registered yet
		{
			TypeQualifiers.Add(q);
		}

		n = s.Contains(",");
		for(i=0; i<=n; i++) if(TypeSpecifiers.Find(s.Section(",",i,i)) == -1)
		{
			IERROR_REPORT_FATAL_ERROR("Incorrectly configured DataType object.");
		}
	}
}


iDataType::~iDataType()
{
	TypeList.Remove(this);
}


const iDataType& iDataType::FindTypeById(int id)
{
	int i;
	for(i=0; i<TypeList.Size(); i++) if(TypeList[i]->GetId() == id) return *(TypeList[i]);
	return iDataType::Null();
}


const iDataType& iDataType::FindTypeByName(const iString &name)
{
	int i;
	for(i=0; i<TypeList.Size(); i++) if(TypeList[i]->IsA(name)) return *(TypeList[i]);
	return iDataType::Null();
}


const iDataType& iDataType::ReSpecifyType(const iDataType &type, const iString &specifier)
{
	iString q, s;
	
	if(!IsValidKeyword(type.mKeywords.Section(",",0,0),q,s))
	{
		IERROR_REPORT_ERROR("Inorrectly configured data types.");
		return iDataType::Null();
	}

	return FindTypeByKeyword(q+specifier);
}


const iDataType& iDataType::FindTypeByKeyword(const iString &keyword)
{
	int i;
	iString q, s;
	
	if(!IsValidKeyword(keyword,q,s)) return iDataType::Null();

	for(i=0; i<TypeList.Size(); i++) if(TypeList[i]->mKeywords.Find(q)!=-1 && TypeList[i]->mKeywords.Find(s)!=-1) return *(TypeList[i]);
	
	IERROR_REPORT_ERROR("Inorrectly configured data types.");
	return iDataType::Null();
}


void iDataType::FindAllTypesByKeywords(iDataInfo &info, const iString &keywords)
{
	int i, j, n = keywords.Contains(",");
	iString k, s, q;
	for(i=0; i<=n; i++)
	{
		k = keywords.Section(",",i,i);
		if(IsValidKeyword(k,q,s))
		{
			for(j=0; j<TypeList.Size(); j++) if(TypeList[j]->mKeywords.Find(s)!=-1 && (q.IsEmpty() || TypeList[j]->mKeywords.Find(s)!=-1)) info += *TypeList[j];
		}
	}
}


const iDataType& iDataType::Null()
{
	static const iDataType tmp(0,"Null","",iMath::_IntMin,"");
	return tmp;
}


const iDataType& iDataType::UniformScalars()
{
	static const iDataType tmp(1,"Uniform Scalars","us",0,"basic:ScalarField");
	return tmp;
}


const iDataType& iDataType::UniformVectors()
{
	static const iDataType tmp(2,"Uniform Vectors","uv",100,"basic:VectorField");
	return tmp;
}


const iDataType& iDataType::UniformTensors()
{
	static const iDataType tmp(3,"Uniform Tensors","ut",200,"basic:TensorField");
	return tmp;
}


const iDataType& iDataType::BasicParticles()
{
	static const iDataType tmp(4,"Basic Particles","bp",300,"basic:ParticleSet");
	return tmp;
}


//
//  DataType class
//
iDataInfo::iDataInfo() : mArr(&iDataType::Null())
{
}


iDataInfo::iDataInfo(const iDataInfo &info) : mArr(&iDataType::Null())
{
	int i;
	for(i=0; i<info.Count(); i++) mArr.AddUnique(&(info.Type(i)));
}


iDataInfo::iDataInfo(const iDataType &type) : mArr(&iDataType::Null())
{
	mArr.Add(&type);
}


iDataInfo::~iDataInfo()
{
}


void iDataInfo::Erase()
{
	mArr.Clear();
}


iDataInfo& iDataInfo::operator=(const iDataInfo &info)
{
	int i;

	mArr.Clear();
	for(i=0; i<info.Count(); i++) mArr.Add(&(info.Type(i)));

	return *this;
}


iDataInfo& iDataInfo::operator+=(const iDataType &type)
{
	mArr.AddUnique(&type);
	return *this;
}


iDataInfo& iDataInfo::operator-=(const iDataType &type)
{
	mArr.Remove(&type);
	return *this;
}


bool iDataInfo::Includes(const iDataType &type) const
{
	return mArr.Find(&type) >= 0;
}


int iDataInfo::Index(const iDataType &type) const
{
	int i;
	for(i=0; i<mArr.Size(); i++) if(*mArr[i] == type) return i;
	return -1;
}


const iDataType& iDataInfo::Type(int i) const
{
	if(i>=0 && i<mArr.Size()) return *(mArr[i]); else return iDataType::Null();
}


iDataInfo& iDataInfo::Null()
{
	static iDataInfo tmp;
	return tmp;
}
