/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconsole.h"


#include "ifile.h"
#include "imath.h"
#include "ishell.h"
#include "ishellfactory.h"
#include "istring.h"
#include "ivtk.h"

#include <vtkIOStream.h>
#include <vtkTimerLog.h>

#include <stdlib.h>


namespace iConsoleDateTime
{
	//
	//  Converts a Gregorian date to a Julian day.
	//  This algorithm is taken from Communications of the ACM, Vol 6, No 8.
	//
	unsigned int DateToJd(int y, int m, int d)
	{
		unsigned int c, ya;
		if ( y <= 99 )
			y += 1900;
		if ( m > 2 ) {
			m -= 3;
		} else {
			m += 9;
			y--;
		}
		c = y;
		c /= 100;
		ya = y - 100*c;
		return 1721119 + d + (146097*c)/4 + (1461*ya)/4 + (153*m+2)/5;
	}
	
	//
	//  Converts a Julian day to a Gregorian date.
	//	This algorithm is taken from Communications of the ACM, Vol 6, No 8.
	//
	void JdToDate(unsigned int jd, int &y, int &m, int &d)
	{
		unsigned int x;
		unsigned int j = jd - 1721119;
		y = (j*4 - 1)/146097;
		j = j*4 - 146097*y - 1;
		x = j/4;
		j = (x*4 + 3) / 1461;
		y = 100*y + j;
		x = (x*4) + 3 - 1461*j;
		x = (x + 4)/4;
		m = (5*x - 3)/153;
		x = 5*x - 3 - 153*m;
		d = (x + 5)/5;
		if ( m < 10 ) {
			m += 3;
		} else {
			m -= 9;
			y++;
		}
	}

	//
	//  Converts Universal Coordinated time to a Julian day
	//
	unsigned int UctToJd(double uct)
	{
		return 2440588 + int(uct/86400.0);
	}

	//
	//  Converts a Julian day to Universal Coordinated time
	//
	double JdToUct(unsigned int jd)
	{
		return 86400.0*(jd-2440588);
	}

	//
	//  Converts Universal Coordinated time to Gregorian date/time
	//
	void UctToDateTime(double uct, int &year, int &month, int &day, int &hour, int &minute, int &second)
	{
		unsigned int jd = UctToJd(uct);
		JdToDate(jd,year,month,day);
		unsigned int s = round(uct-JdToUct(jd));
		hour = s/3600;
		minute = (s-3600*hour)/60;
		second = s - 3600*hour - 60*minute;
	}

};
	

using namespace iConsoleDateTime;


iConsole* iConsole::mInstance = 0;


void iConsole::CreateInstance(iShell *s)
{
	if(mInstance == 0) mInstance = iShellFactory::CreateConsole(s);
}


iConsole::iConsole(iShell *s) : iShellComponent(s)
{
	mLoggingBlocked = mLogCreated = mInUpdateLog = false;

#ifdef I_DEBUG
	int i;
	for(i=0; i<10; i++) mTimer[i] = vtkTimerLog::New();
#endif
}


iConsole::~iConsole()
{
#ifdef I_DEBUG
	int i;
	for(i=0; i<10; i++) mTimer[i]->Delete();
#endif
}


void iConsole::BlockLogging(bool s)
{
	mLoggingBlocked = s;
}


iString iConsole::GetLogFileName() const
{
	iString fn;
    char *e = getenv("IFRIT_DIR");
	if(e != 0)
	{
		fn = iString(e) + "/";
	}
	fn += "ifrit.err";
	return fn;
}


void iConsole::UpdateLog(const iString &message, const iString &file, int line)
{
	if(mLoggingBlocked || mInUpdateLog) return;

	mInUpdateLog = true;

	iFile F(this->GetLogFileName());
	if(F.Open(iFile::_Append,iFile::_Text) || F.Open(iFile::_Write,iFile::_Text))
	{
		iString ws = message + "  File: " + file + "  Line: " + iString::FromNumber(line);
	
		if(!mLogCreated)
		{
#ifdef IVTK_4
			double s = vtkTimerLog::GetCurrentTime();
#else
			double s = vtkTimerLog::GetUniversalTime();
#endif
			int year, month, day, hour, minute, second;
			UctToDateTime(s,year,month,day,hour,minute,second);

			F.WriteLine(" ");
			F.WriteLine("New run: "+iString::FromNumber(year)+"/"+iString::FromNumber(month)+"/"+iString::FromNumber(day)+"/"+iString::FromNumber(hour)+"/"+iString::FromNumber(minute)+"/"+iString::FromNumber(second));
			F.WriteLine("---------------------------");
		}

		if(F.WriteLine(ws)) mLogCreated = true;
		F.Close();
	}

	mInUpdateLog = false;
}


void iConsole::DisplayTextMessage(const char *message)
{
	this->OutputText(message);
}


void iConsole::DisplayErrorMessage(const char *message, const char *file, int line)
{
	this->OutputError(message,file,line);
	this->UpdateLog(message,file,line);
}


void iConsole::DisplayFatalErrorMessage(const char *message, const char *file, int line)
{
	this->OutputFatalError(message,file,line);
	this->UpdateLog(message,file,line);
	exit(1);
}


void iConsole::DisplayCriticalInformationMessage(const char *message)
{
	this->OutputCriticalInformation(message);
}


//
//  Inheritable members
//
void iConsole::OutputText(const iString &message) const
{
	//
	//  Default implementation: send everything to cout
	//
	cout << message.ToCharPointer() << endl;
}


void iConsole::OutputError(const iString &message, const iString &file, int line) const
{
	//
	//  Default implementation: send everything to cerr
	//
	cerr << "! NON-FATAL ERROR: " << message.ToCharPointer() << endl;
	cerr << "             FILE: " << file.ToCharPointer() << endl;
	cerr << "             LINE: " << line << endl;
}


void iConsole::OutputFatalError(const iString &message, const iString &file, int line) const
{
	//
	//  Default implementation: send everything to cerr
	//
	cerr << "!!! FATAL ERROR: " << message.ToCharPointer() << endl;
	cerr << "           FILE: " << file.ToCharPointer() << endl;
	cerr << "           LINE: " << line << endl;
}


void iConsole::OutputCriticalInformation(const iString &message) const
{
	//
	//  Default implementation: send everything to cerr
	//
	cerr << "!IMPORTANT: " << message.ToCharPointer() << endl;
}


#ifdef I_DEBUG
//
//  Debugging helpers
//
void iConsole::BreakPoint(const char *file, int line)
{
	//
	//  Set a break point here
	//
	int o = 0;

	//
	//  Use the variables to avoid compiler warnings
	//
	file++;
	line++;
	o++;
}


void iConsole::StartTimer(int n)
{
	if(n>=0 && n<10) mTimer[n]->StartTimer();
}


void iConsole::StopTimer(const iString &message, int n)
{
	if(n>=0 && n<10)
	{
		mTimer[n]->StopTimer();
		double v = mTimer[n]->GetElapsedTime();
		if(!message.IsEmpty()) this->OutputDebugMessage(message+": "+iString::FromNumber(v)); else this->OutputDebugMessage("Time used: "+iString::FromNumber(v));
	}
}


void iConsole::OutputDebugMessage(const iString &message) const
{
	this->GetShell()->OutputDebugMessage(message);
}

#endif
