/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icamera.h"


#include "ierror.h"
#include "irendertool.h"

#include <vtkMath.h>
#include <vtkRenderer.h>

IOBJECT_DEFINE_TYPE(iCamera,Camera,c,iObjectType::_Helper);


IOBJECT_DEFINE_KEY(iCamera,Azimuth,a,Float,1);
IOBJECT_DEFINE_KEY(iCamera,ClippingRange,cr,Double,2);
IOBJECT_DEFINE_KEY(iCamera,ClippingRangeAuto,cra,Bool,1);
IOBJECT_DEFINE_KEY(iCamera,Elevation,e,Float,1);
IOBJECT_DEFINE_KEY(iCamera,EyeAngle,ea,Float,1);
IOBJECT_DEFINE_KEY(iCamera,Pitch,p,Float,1);
IOBJECT_DEFINE_KEY(iCamera,ParallelProjection,pp,Bool,1);
IOBJECT_DEFINE_KEY(iCamera,ParallelScale,ps,Float,1);
IOBJECT_DEFINE_KEY(iCamera,Roll,r,Float,1);
IOBJECT_DEFINE_KEY(iCamera,Reset,rs,Bool,1);
IOBJECT_DEFINE_KEY(iCamera,ViewAngle,va,Float,1);
IOBJECT_DEFINE_KEY(iCamera,ViewAngleVertical,vav,Bool,1);
IOBJECT_DEFINE_KEY(iCamera,ViewUp,u,Double,3);
IOBJECT_DEFINE_KEY(iCamera,Yaw,y,Float,1);
IOBJECT_DEFINE_KEY(iCamera,Zoom,z,Float,1);

IOBJECT_DEFINE_POSITION_KEY(iCamera,FocalPoint,f);
IOBJECT_DEFINE_POSITION_KEY(iCamera,Position,x);


iCamera* iCamera::New(iRenderTool *rt)
{
	return new iCamera(rt);
}


iCamera::iCamera(iRenderTool *rt) : iViewModuleComponent(rt==0?0:rt->GetViewModule()), mPosition(rt==0?0:rt->GetViewModule()), mFocalPoint(rt==0?0:rt->GetViewModule())
{
	mRenderTool = rt; IERROR_ASSERT_NULL_POINTER(rt);
	mDevice = vtkCamera::New(); IERROR_ASSERT_NULL_POINTER(mDevice);

	this->ParallelProjectionOn();
	mNeedsSceneReset = false;

	mDevice->Modified();
	this->Modified();  // this way we will update device on the first render
}


iCamera::~iCamera()
{
	mDevice->Delete();
}


void iCamera::Modified()
{
	//
	//  Update position and focal point
	//
	mPosition = this->Position;
	mFocalPoint = this->FocalPoint;

	this->ClearCache();
	vtkCamera::Modified();
}


void iCamera::Render(vtkRenderer *ren)
{
	this->UpdateDevice();
	mDevice->Render(ren);
}


void iCamera::UpdateViewport(vtkRenderer *ren)
{
	this->UpdateDevice();
	mDevice->UpdateViewport(ren);
}


void iCamera::UpdateDevice()
{
	if(this->GetMTime() > mDevice->GetMTime())
	{
		if(mNeedsSceneReset)
		{
			mNeedsSceneReset = false;
			mRenderTool->GetRenderer()->ResetCamera();
		}

		mDevice->SetPosition(this->GetPosition());
		mDevice->SetFocalPoint(this->GetFocalPoint());
		mDevice->SetViewUp(this->GetViewUp());
		mDevice->OrthogonalizeViewUp();

		mDevice->SetEyeAngle(this->GetEyeAngle());
		mDevice->SetParallelProjection(this->GetParallelProjection());
		mDevice->SetParallelScale(this->GetParallelScale());
		mDevice->SetUseHorizontalViewAngle(this->GetUseHorizontalViewAngle());
		mDevice->SetViewAngle(this->GetViewAngle());

		mDevice->SetUserTransform(this->GetUserTransform()); // needed for magnified image creation

		//
		// This is tricky - avoid the limits
		//
		double *dcr = mDevice->GetClippingRange();
		dcr[0] = this->ClippingRange[0];
		dcr[1] = this->ClippingRange[1];

		mDevice->SetWindowCenter(WindowCenter[0],WindowCenter[1]);
	}
}


void iCamera::Reset()
{
	mRenderTool->ResetCamera();
	this->ClearCache();
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iCamera::PackStateBody(iString &s) const
{
	double dv[2];

	this->PackValuePosition(s,KeyPosition(),mPosition);
	this->PackValuePosition(s,KeyFocalPoint(),mFocalPoint);
	this->PackValue(s,KeyViewUp(),this->ViewUp,3);

	this->PackValue(s,KeyParallelProjection(),this->ParallelProjection!=0);
	this->PackValue(s,KeyParallelScale(),float(this->ParallelScale));
	this->PackValue(s,KeyViewAngle(),float(this->ViewAngle));
	this->PackValue(s,KeyViewAngleVertical(),this->UseHorizontalViewAngle==0);
	this->PackValue(s,KeyEyeAngle(),float(this->EyeAngle));
	this->PackValue(s,KeyClippingRangeAuto(),mRenderTool->GetAdjustCameraClippingRangeAutomatically());

	dv[0] = this->ClippingRange[0]/this->Distance;
	dv[1] = this->ClippingRange[1]/this->Distance;
	this->PackValue(s,KeyClippingRange(),dv,2);
}


void iCamera::UnPackStateBody(const iString &s)
{
	bool b; float f; double dv[2];

	//
	//  Camera orientation
	//
	if(this->UnPackValuePosition(s,KeyPosition(),mPosition)) this->SetPosition(mPosition);
	if(this->UnPackValuePosition(s,KeyFocalPoint(),mFocalPoint)) this->SetFocalPoint(mFocalPoint);
	if(this->UnPackValue(s,KeyViewUp(),this->ViewUp,3)) this->SetViewUp(this->ViewUp);

	if(this->UnPackValue(s,KeyParallelProjection(),b))
	{
		if(b && this->ParallelProjection==0) mNeedsSceneReset = true;
		this->SetParallelProjection(b?1:0);
	}
	if(this->UnPackValue(s,KeyParallelScale(),f)) this->SetParallelScale(f);
	if(this->UnPackValue(s,KeyViewAngle(),f)) this->SetViewAngle(f);
	if(this->UnPackValue(s,KeyViewAngleVertical(),b)) this->SetUseHorizontalViewAngle(b?0:1);
	if(this->UnPackValue(s,KeyEyeAngle(),f)) this->SetEyeAngle(f);

	if(this->UnPackValue(s,KeyClippingRangeAuto(),b))
	{
		mRenderTool->SetAdjustCameraClippingRangeAutomatically(b);
		this->ClearCache();
	}

	mRenderTool->GetCameraClippingRange(dv);
	dv[0] /= this->Distance;
	dv[1] /= this->Distance;
	if(this->UnPackValue(s,KeyClippingRange(),dv,2))
	{
		dv[0] *= this->Distance;
		dv[1] *= this->Distance;
		mRenderTool->SetCameraClippingRange(dv);
		this->ClearCache();
	}

	//
	//  "Action" keys
	//
#ifdef I_CHECK1
	bool saveReportMissingKeys = iObject::ReportMissingKeys;
	iObject::ReportMissingKeys = false; //action keys are not part of the states
#endif

	if(this->UnPackValue(s,KeyReset(),b) && b) this->Reset();

	if(this->UnPackValue(s,KeyAzimuth(),f)) this->Azimuth(f);
	if(this->UnPackValue(s,KeyElevation(),f)) this->Elevation(f);
	if(this->UnPackValue(s,KeyYaw(),f)) this->Yaw(f);
	if(this->UnPackValue(s,KeyPitch(),f)) this->Pitch(f);
	if(this->UnPackValue(s,KeyRoll(),f)) this->Roll(f);
	if(this->UnPackValue(s,KeyZoom(),f) && f>1.0e-30)
	{
		if(this->GetParallelProjection())
		{
			this->SetParallelScale(this->GetParallelScale()/f);
		}
		else
		{
			this->Dolly(f);
		}
	}

#ifdef I_CHECK1
	iObject::ReportMissingKeys = saveReportMissingKeys;
#endif

	this->OrthogonalizeViewUp();
}


void iCamera::SetPosition(const iPosition &p)
{
	int i;

	mPosition = p;
	
	for(i=0; i<3; i++) this->Position[i] = mPosition[i];
	this->UpdatePositions();
}


void iCamera::SetFocalPoint(const iPosition &p)
{
	int i;

	mFocalPoint = p;
	
	for(i=0; i<3; i++) this->FocalPoint[i] = mFocalPoint[i];
	this->UpdatePositions();
}


void iCamera::UpdatePositions()
{
	this->ComputeViewTransform();
	this->ComputeDistance();
	this->ComputeCameraLightTransform();
	this->Modified();
}

