/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
Implementation of ixsection.h
*/


#include "iglobals.h"
#include "ixsection.h"

#include "ivtk.h"
#include "ipalette.h"
#include "ilimits.h"
#include "iobjectfactory.h"

#include "idatareader.h"
#include "idatagateway.h"
#include "ivtkcallback.h"
#include "iorthoslicer.h"
#include "iorthoplane.h"
#include "iuniformmeshdata.h"
#include "iuniformmeshdataconverter.h"
#include "ivisualobjectobserver.h"
#include "iorthotextureplane.h"
#include "icolorbars.h"

#include <vtkCommand.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkLODActor.h>
#include <vtkLookupTable.h>
#include <vtkProperty.h>
#include <vtkProbeFilter.h>
#include <vtkPlaneSource.h>
#include <vtkStructuredPoints.h>
#include <vtkTexture.h>
#include <vtkLookupTable.h>
#include <vtkCellArray.h>
#include <vtkPointData.h>
#include <vtkFloatArray.h>
#include <vtkCleanPolyData.h>


void reportNullPointer(int ec);


//
// iXsection class
//

iXsection* iXsection::New(iVTK *mVTK)
{
	return (iXsection *)iObjectFactory::createObject("Xsection",mVTK);
}


iXsection::iXsection(iVTK *mVTK) : iVisualObject(mVTK)
{
	
	var = 1;
	dir = 3;
	pos = -1.0;
	posmax = 1.0;
	
	nvec[2] = 1.0; nvec[0] = nvec[1] = 0.0;
		
	pal = 0;

	mode = 1;
	interpolateData = true;

	ambient = 1.0;
	diffuse = 0.0;
	specular = 0.0;
	specularPower = 0.0;
	//
	//  Do VTK stuff
	//	
	xsectionActor = vtkLODActor::New();
	if(xsectionActor == 0) reportNullPointer(6501);
	xsectionMapper = vtkPolyDataMapper::New();
	if(xsectionMapper == 0) reportNullPointer(6502);
	xsectionProbe = vtkProbeFilter::New();
	if(xsectionProbe == 0) reportNullPointer(6503);
	xsectionSource = vtkPlaneSource::New();
	if(xsectionSource == 0) reportNullPointer(6504);
	xsectionDataConverter = iUniformMeshDataConverter::New(this);
	if(xsectionDataConverter == 0) reportNullPointer(6505);
	xsectionSlicer = iOrthoSlicer::New(this);
	if(xsectionSlicer == 0) reportNullPointer(6506);
	xsectionTexture = vtkTexture::New();
	if(xsectionTexture == 0) reportNullPointer(6507);
	colorLookupTable = vtkLookupTable::New();
	if(colorLookupTable == 0) reportNullPointer(6608);
	xsectionOrthoPlane = iOrthoPlane::New();
	if(xsectionOrthoPlane == 0) reportNullPointer(6509);
	xsectionTextureConverter = iOrthoTexturePlane::New();
	if(xsectionTextureConverter == 0) reportNullPointer(6514);

	iVTKAbortRenderEventObserver *obsAbortRender = myVTK->getAbortRenderEventObserver();
	xsectionMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	xsectionProbe->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	xsectionDataConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	xsectionSlicer->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	xsectionOrthoPlane->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	xsectionTextureConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	
	xsectionDataConverter->setCurrentVar(var);
	xsectionDataConverter->SetInput(myVTK->getGateway()->GetOutput());
//  
//	Plane + Probe approach: arbitrary orientation plane (does not quite work yet)
//
	if(dir == 0) 
	{
		
		xsectionSource->SetResolution(100,100);
		xsectionSource->SetNormal(nvec);
		
		vtkFloat f0[3], f1[3], f2[3];
		xsectionSource->GetOrigin(f0);
		xsectionSource->GetPoint1(f1);
		xsectionSource->GetPoint2(f2);
		for(int i=0; i<3; i++)
		{
			f1[i] = 2*(f1[i]-f0[i]);
			f2[i] = 2*(f2[i]-f0[i]);
		}
		xsectionSource->SetOrigin(0.0,0.0,0.0);
		xsectionSource->SetPoint1(f1);
		xsectionSource->SetPoint2(f2);
		
		xsectionProbe->SetInput(xsectionSource->GetOutput());
		xsectionProbe->SetSource(xsectionDataConverter->GetOutput());
		xsectionProbe->SpatialMatchOn();
		xsectionMapper->SetInput(xsectionProbe->GetPolyDataOutput());

	} 
	else 
	{
		//
		//  OrthoSlicer approach, 3 orthogonal planes
		//
		xsectionSlicer->SetInput(xsectionDataConverter->GetOutput());
		xsectionSlicer->SetAxis(dir-1);
		xsectionOrthoPlane->SetInput(xsectionSlicer->GetOutput());

	}
	
	xsectionMapper->ScalarVisibilityOn();
	xsectionMapper->ColorByArrayComponent(0,0);
	xsectionMapper->SetLookupTable(iPaletteList::getInstance()->getPalette(pal)->getTable());
	xsectionMapper->SetScalarRange(this->getLimits()->getVarLow(var),this->getLimits()->getVarHigh(var));
	xsectionMapper->SetScalarModeToDefault();
	xsectionMapper->UseLookupTableScalarRangeOff();
	
	xsectionActor->SetMapper(xsectionMapper);
	xsectionActor->SetPosition(0.0,0.0,0.0);

	xsectionTextureConverter->SetInput(xsectionSlicer->GetOutput());

	xsectionTexture->SetLookupTable(colorLookupTable);
	xsectionTexture->SetInput(xsectionTextureConverter->getTexture());
	xsectionTexture->SetQualityTo32Bit();
	iPaletteList::getInstance()->getPalette(pal)->fillLookupTable(colorLookupTable);
	colorLookupTable->SetTableRange(this->getLimits()->getVarLow(var),this->getLimits()->getVarHigh(var));

	xsectionProperty = xsectionActor->GetProperty();
	xsectionProperty->SetAmbient(ambient);
	xsectionProperty->SetDiffuse(diffuse);
	xsectionProperty->SetSpecular(specular);
	xsectionProperty->SetSpecularPower(specularPower);
	xsectionProperty->SetColor(1.0,1.0,1.0);
	xsectionProperty->SetOpacity(1.0);
	//
	//  Add observer to keep information about this object
	//
	xsectionActor->AddObserver(vtkCommand::UserEvent,objectObserver);
	objectObserver->setTypes(IOBJECTTYPE_XSEC,IDATATYPE_MESH);

	xsectionActor->VisibilityOff();

}


iXsection::~iXsection()
{
	
	if(!init) myVTK->removeProp(xsectionActor);
	xsectionActor->Delete();	
	xsectionMapper->Delete();
	xsectionProbe->Delete();
	xsectionSlicer->Delete();
	xsectionDataConverter->Delete();
	xsectionSource->Delete();
	xsectionTexture->Delete();
	xsectionOrthoPlane->Delete();
	xsectionTextureConverter->Delete();

}


void iXsection::reset()
{
	this->updateDataChannel();
	//
	//  Un-initialized, if needed
	//
	if(!init)
	{
		myVTK->removeProp(xsectionActor);
		init = true;
	}
}


void iXsection::setMode(int m)
{
	
	switch(m) 
	{
	case 0:
		{
			xsectionActor->SetTexture(0); 
			xsectionMapper->SetInput(xsectionOrthoPlane->GetOutput());
			xsectionOrthoPlane->Modified();
			break;
		}
	case 1: 
		{
			xsectionActor->SetTexture(xsectionTexture); 
			xsectionMapper->SetInput(xsectionTextureConverter->GetOutput());
			xsectionTextureConverter->Modified();
			break;
		}
	default: return;
	}
	mode = m;
	
}


void iXsection::forcePolygonalMode(bool s)
{
	if(s)
	{
		int sm = mode;
		this->setMode(0);
		mode = sm;
	}
	else this->setMode(mode);
}


void iXsection::showColorBars(bool show)
{
	if(xsectionActor->GetVisibility() == 0) return;
	myVTK->getColorBars()->showBar(COLORBARS_PRIORITY_XSECTION,var,dataChannel,pal,show);
}


void iXsection::setDir(int d)
{ 

	if(d>=1 && d<=3)
	{
		dir = d;
		xsectionSlicer->SetAxis(d-1);
		xsectionSlicer->SetAxis((float)pos);
	}
	
}


void iXsection::setVar(int v)
{ 
	
	if(v>0 && v<=this->getLimits()->getNvar())
	{
		this->showColorBars(false);
		var = v;
		this->showColorBars(true);
		xsectionMapper->SetScalarRange(this->getLimits()->getVarLow(var),this->getLimits()->getVarHigh(var));
		colorLookupTable->SetTableRange(this->getLimits()->getVarLow(var),this->getLimits()->getVarHigh(var));
		xsectionDataConverter->setCurrentVar(var);
	}
	
}


bool iXsection::setPos(double p, bool checkPosMax)
{ 
	bool ret = false;
	
	vtkFloat spa[3], org[3];
	org[0] = org[1] = org[2] = 0.0;
	if(dir == 0) posmax = 1.0; else 
	{
		int dims[3];
		xsectionDataConverter->GetOutput()->GetDimensions(dims);
		if(dims[0]>0 && dims[1]>0 && dims[2]>0)
		{
			xsectionDataConverter->GetOutput()->GetSpacing(spa);
			xsectionDataConverter->GetOutput()->GetOrigin(org);
			posmax = org[dir-1] + spa[dir-1]*(dims[dir-1]-1);
		}
		else posmax = 1.0;
	}
	
	p = (p < -1.0) ? -1.0 : p;
	if(p>posmax && checkPosMax)
	{
		p = posmax;
		ret = true;
	}
	
	xsectionSource->Push(p-pos);
	pos = p;
	xsectionSlicer->SetAxis((float)pos);
	
	return ret;
	
}


void iXsection::setPal(int p)
{ 
	p = ( p < 0 ) ? 0 : p;
	p = ( p > iPaletteList::getInstance()->getNumPalettes() ) ? iPaletteList::getInstance()->getNumPalettes() : p;
	this->showColorBars(false);
	pal = p;
	this->showColorBars(true);
	xsectionMapper->SetLookupTable(iPaletteList::getInstance()->getPalette(pal)->getTable());
	iPaletteList::getInstance()->getPalette(pal)->fillLookupTable(colorLookupTable);
	colorLookupTable->SetTableRange(this->getLimits()->getVarLow(var),this->getLimits()->getVarHigh(var));
}


void iXsection::setAmbient(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	ambient = v;
	xsectionProperty->SetAmbient(v);
}


void iXsection::setDiffuse(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	diffuse = v;
	xsectionProperty->SetDiffuse(v);
}


void iXsection::setSpecular(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	specular = v;
	xsectionProperty->SetSpecular(v);
}


void iXsection::setSpecularPower(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>100.0) ? 100.0 : v;
	specularPower = v;
	xsectionProperty->SetSpecularPower(v);
}


void iXsection::setInterpolateData(bool s)
{
	interpolateData = s;
	if(interpolateData) 
	{
		xsectionTexture->InterpolateOn(); 
		xsectionProperty->SetInterpolationToGouraud();
	}
	else 
	{
		xsectionTexture->InterpolateOff(); 
		xsectionProperty->SetInterpolationToFlat();
	}
	xsectionOrthoPlane->setCorrectForFlatShading(!interpolateData);
}


void iXsection::setDirVector(float n1, float n2, float n3)
{
	nvec[0] = n1;
	nvec[1] = n2;
	nvec[2] = n3;
}


void iXsection::show(bool show)
{
	  
	if(show)
	{
		if(init)
		{
			this->setMode(mode);
			this->setInterpolateData(interpolateData);
			this->setDir(dir);

			myVTK->addProp(xsectionActor);
			init = false;
		}
		if(!init)
		{
			xsectionMapper->SetScalarRange(this->getLimits()->getVarLow(var),this->getLimits()->getVarHigh(var));
			colorLookupTable->SetTableRange(this->getLimits()->getVarLow(var),this->getLimits()->getVarHigh(var));
			xsectionActor->VisibilityOn();
			this->showColorBars(true);
		}
	} 
	else 
	{
		if(!init) 
		{
			this->showColorBars(false);
			xsectionActor->VisibilityOff();
		}
	}
	
}


float iXsection::getMemorySize()
{
	float s = 0.0;
	
	if(dir == 0) 
	{
		s += xsectionSource->GetOutput()->GetActualMemorySize();
		float s1 = (float)xsectionProbe->GetOutput()->GetActualMemorySize() - (float)xsectionProbe->GetInput()->GetActualMemorySize();
		if(s1 < 0.0) s1 = 0.0;
		s += s1;
	} 
	else 
	{
		s = xsectionSlicer->GetOutput()->GetActualMemorySize();
	}

	return s;

}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iXsection::packState(iString &s)
{

	iVisualObject::packState(s);

	this->packValue(s,"iXsection::interpolateData",interpolateData);

	this->packValue(s,"iXsection::var",var);
	this->packValue(s,"iXsection::dir",dir);
	this->packValue(s,"iXsection::pal",pal);
	this->packValue(s,"iXsection::mode",mode);

	this->packValue(s,"iXsection::pos",pos);
	this->packValue(s,"iXsection::nvec",nvec,3);

}


void iXsection::unpackState(iString s)
{
	int i; bool b; double f; float v[3];

	iVisualObject::unpackState(s);

	if(this->unpackValue(s,"iXsection::interpolateData",b)) this->setInterpolateData(b);

	if(this->unpackValue(s,"iXsection::var",i)) this->setVar(i);
	if(this->unpackValue(s,"iXsection::dir",i)) this->setDir(i);
	if(this->unpackValue(s,"iXsection::pal",i)) this->setPal(i);
	if(this->unpackValue(s,"iXsection::mode",i)) this->setMode(i);
	
	if(this->unpackValue(s,"iXsection::pos",f)) this->setPos(f,false);
	if(this->unpackValue(s,"iXsection::nvec",v,3)) this->setDirVector(v);

}


