/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

  Implementation of istreamline.h
  
=========================================================================*/


#include "iglobals.h"
#include "istreamline.h"

#include "imath.h"

#include <vtkFloatArray.h>
#include <vtkCellArray.h>
#include <vtkPolyData.h>
#include <vtkMath.h>
#include <vtkStructuredPoints.h>
#include <vtkPoints.h>
#include <vtkPointData.h>


void reportNullPointer(int);


//------------------------------------------------------------------------------

iStreamLine* iStreamLine::New()
{
	return new iStreamLine;
}


iStreamLine::iStreamLine()
{
	length = 2.0;
	source = 0;
	scalars = 0;
	vmin = 1.0e-0;
	dir = 2;
	axisDir = 0;
	splitLines = 0;
	computeVort = computeDiv = true;
	quality = 0;
	this->setQuality(quality);
	dmin = 1.0e-3;
}


iStreamLine::~iStreamLine()
{
	source = 0;
	scalars = 0;
}


void iStreamLine::setQuality(int q)
{ 
	const float eps0 = 1.0e-3;
	const int itMax0 = 10000;

	if(quality>=0 && quality <=7)
	{ 
		quality = q; 
		eps = eps0*pow(0.1,(double)quality);
		itMax = itMax0*round(pow(10.0,0.25*q));
		this->Modified(); 
	} 
}

void iStreamLine::setSource(vtkPoints *s)
{ 
	source = s; 
	this->Modified(); 
}


void iStreamLine::ExecuteData(vtkDataObject *)
{
	int i,line;

	if(source == 0) return;
	
	vtkStructuredPoints *input = (vtkStructuredPoints *)this->GetInput();
	vtkPolyData *output = (vtkPolyData *)this->GetOutput();

	if(input==0 || !this->definePointers(input)) return;

	output->Initialize();
	
	int nlines = source->GetNumberOfPoints();
	
	vtkPoints *poi = vtkPoints::New();
	if(poi == 0) reportNullPointer(7501);
	vtkCellArray *lin = vtkCellArray::New();
	if(lin == 0) reportNullPointer(7502);
	vtkFloatArray *sca = vtkFloatArray::New();
	if(sca == 0) reportNullPointer(7503);

	input->GetDimensions(dims);
	input->GetOrigin(org);
	input->GetSpacing(spa);

	vtkFloat cell = spa[0];
	if(cell > spa[1]) cell = spa[1];
	if(cell > spa[2]) cell = spa[2];
	//
	//  Determine how many scalar components we need to carry with us
	//
	int ns = 2;
	if(computeVort) ns++;
	if(computeDiv) ns++;
	ns += nadd;
	sca->SetNumberOfComponents(ns);
	//
	//  Loop over all lines
	//
	int npoi;
	float v1[3];
	double x1[3];
	double d;
	float vv1, sdir, h;
	bool done;
	vtkIdType l;

	float vort0[3], div0[1];
	float *vort = 0, *div = 0;
	if(computeVort) vort = vort0;
	if(computeDiv) div = div0;

	float *ss = new float[ns];
	if(ss == 0) reportNullPointer(7504);

	int idir, idirMin = 0, idirMax = -1;
	switch (dir) 
	{
	case 0:	{ idirMin = 0; idirMax = 0; break; }
	case 1:	{ idirMin = 1; idirMax = 1; break; }
	case 2:	
	case 3:	
	case 4:	{ idirMin = 0; idirMax = 1; break; }
	}

	int isplit, isplitMax = 1;
	if(splitLines) isplitMax = 2;

	int idir1, it, ret, stopIndex;
	bool out;

	vtkIdType vertex = 0;
	for(line=0; line<nlines; line++) 
	{
		
		this->UpdateProgress((float)line/nlines);
		if(this->GetAbortExecute()) break;
		
		for(idir=idirMin; idir<=idirMax; idir++)
		{
			for(isplit=0; isplit<isplitMax; isplit++)
			{
				
				stopIndex = 10;
				d = 0.0;
				npoi = 0;
				done = false;
				sdir = 1.0 - 2.0*idir;
				
				//
				//  Locate the starting point
				//
				source->GetPoint(line,x1);
				
				if(splitLines)
				{
					idir1 = (axisDir+1)%3;
					x1[idir1] += 0.25*spa[idir1]*(2*isplit-1);
				}
				
				l = getVector(x1,v1);

				if(dir==3 && v1[axisDir]<0.0) continue;
				if(dir==4 && v1[axisDir]>0.0) continue;
				
				h = sdir*spa[axisDir]/(1.0e-30+sqrt(v1[0]*v1[0]+v1[1]*v1[1]+v1[2]*v1[2]));

				it = 0;
				ret = 1;

				do
				{
					
					if(ret >= 0)
					{
						//
						//  Save the point
						//
						l = getVector(x1,v1,vort,div);
						
						out = false;
						for(i=0; i<3; i++)
						{
							if(x1[i]<-0.999999 && sdir*v1[i]<0.0) out = true;
							if(x1[i]>0.999999 && sdir*v1[i]>0.0) out = true;
						}
						if(out) break;

						vv1 = sqrt(v1[0]*v1[0]+v1[1]*v1[1]+v1[2]*v1[2]);
						poi->InsertNextPoint(x1); npoi++;
						i = 0;
						ss[i++] = 1.0/(vmin+vv1);
						ss[i++] = vv1;
						if(computeVort)
						{
							//
							//  vorticity calculation will be here
							//
							ss[i++] = vtkMath::Dot(v1,vort);
						}
						if(computeDiv)
						{
							//
							//  divergence calculation will be here
							//
							ss[i++] = div[0];
						}

						if(nadd > 0) 
						{
							this->assignScalars(ss,i,l);
						}

						sca->InsertNextTuple(ss);
					}
					//
					//  Compute the distance to the nearest edge
					//
					it++;

					ret = followLine(x1,d,h,eps,l);
					if(ret == 0) stopIndex--;
				} 
				while(stopIndex>0 && d<length && it<itMax);
				
				if(npoi > 0)
				{
					lin->InsertNextCell(npoi);
					for(i=0; i<npoi; i++) lin->InsertCellPoint(vertex+i);
					vertex += npoi;
				}
			}
		}			
	}
	
	delete [] ss;

	output->SetPoints(poi);
	poi->Delete();
	output->SetLines(lin);
	lin->Delete();
	output->GetPointData()->SetScalars(sca);
	sca->Delete();

	this->Modified();

}


float iStreamLine::getMemorySize()
{
	return this->GetOutput()->GetActualMemorySize();
}


int iStreamLine::followLine(double x[3], double &d, float &h, float eps, vtkIdType l)
{
	const float b21 = 2.0/9.0;
	const float b31 = 1.0/12.0;
	const float b32 = 1.0/4.0;
	const float b41 = 69.0/128.0;
	const float b42 = -243.0/128.0;
	const float b43 = 135.0/64.0;
	const float b51 = -17.0/12.0;
	const float b52 = 27.0/4.0;
	const float b53 = -54.0/10.0;
	const float b54 = 16.0/15.0;
	const float b61 = 65.0/432.0;
	const float b62 = -5.0/16.0;
	const float b63 = 13.0/16.0;
	const float b64 = 4.0/27.0;
	const float b65 = 5.0/144.0;
	const float c1 = 47.0/450.0;
	const float c2 = 0.0;
	const float c3 = 12.0/25.0;
	const float c4 = 32.0/225.0;
	const float c5 = 1.0/30.0;
	const float c6 = 6.0/25.0;
	const float d1 = 1.0/150.0;
	const float d2 = 0.0;
	const float d3 = -3.0/100.0;
	const float d4 = 16.0/75.0;
	const float d5 = 1.0/20.0;
	const float d6 = -6.0/25.0;

	int i, ret;
	float k1[3], k2[3], k3[3], k4[3], k5[3], k6[3], vtmp[3], err, errmax;
	double xtmp[3];
	float ddmin, dx;

	this->getDmin(l,ddmin,dx);

	getVector(x,vtmp);
	for(i=0; i<3; i++)
	{
		k1[i] = h*vtmp[i];
		xtmp[i] = x[i] + b21*k1[i];
	}

	getVector(xtmp,vtmp);
	for(i=0; i<3; i++)
	{
		k2[i] = h*vtmp[i];
		xtmp[i] = x[i] + b31*k1[i] + b32*k2[i];
	}

	getVector(xtmp,vtmp);
	for(i=0; i<3; i++)
	{
		k3[i] = h*vtmp[i];
		xtmp[i] = x[i] + b41*k1[i] + b42*k2[i] + b43*k3[i];
	}

	getVector(xtmp,vtmp);
	for(i=0; i<3; i++)
	{
		k4[i] = h*vtmp[i];
		xtmp[i] = x[i] + b51*k1[i] + b52*k2[i] + b53*k3[i] + b54*k4[i];
	}

	getVector(xtmp,vtmp);
	for(i=0; i<3; i++)
	{
		k5[i] = h*vtmp[i];
		xtmp[i] = x[i] + b61*k1[i] + b62*k2[i] + b63*k3[i] + b64*k4[i] + b65*k5[i];
	}

	errmax = 0.0;
	getVector(xtmp,vtmp);
	for(i=0; i<3; i++)
	{
		k6[i] = h*vtmp[i];
		xtmp[i] = c1*k1[i] + c2*k2[i] + c3*k3[i] + c4*k4[i] + c5*k5[i] + c6*k6[i];
		err     = d1*k1[i] + d2*k2[i] + d3*k3[i] + d4*k4[i] + d5*k5[i] + d6*k6[i];
		err = fabs(err)/dx;
		if(err > errmax) errmax = err;
	}

	float dd = sqrt(xtmp[0]*xtmp[0]+xtmp[1]*xtmp[1]+xtmp[2]*xtmp[2]);
	errmax /= eps;

	if(errmax > 1.0)
	{
		h = 0.9*h*pow((double)errmax,-0.2);
		ret = -1;
	}
	else
	{
		d += dd;
		for(i=0; i<3; i++) x[i] += xtmp[i];
		h = 0.9*h*pow(1.0e-4+(double)errmax,-0.25);
		ret = 1;
	}

	if(dd < ddmin) 
	{
		if(ret == -1)
		{
			for(i=0; i<3; i++) x[i] += xtmp[i];
		}
		return 0;
	}
	else
	{
		return ret;
	}

}


void iStreamLine::getDmin(vtkIdType vtkNotUsed(l), float &ddmin, float &dx)
{
	ddmin = dmin*spa[0];
	dx = spa[0];
}


bool iStreamLine::definePointers(vtkStructuredPoints *input)
{
	vtkFloatArray *vec = (vtkFloatArray *)input->GetPointData()->GetVectors();
	if(vec==0 || vec->GetNumberOfComponents()!=3) return false;
	ptrvec = (float *)vec->GetVoidPointer(0);

	nadd = 0;
	scalarSize = 0;
	ptrsca = 0;
	if(scalars != 0) 
	{
		int d[3];
		scalars->GetDimensions(d);
		if(d[0]==dims[0] && d[1]==dims[1] && d[2]==dims[2]) 
		{
			nadd = scalars->GetNumberOfScalarComponents();
			ptrsca = (float *)scalars->GetScalarPointer();
			scalarSize = ((vtkIdType)d[0])*d[1]*d[2];
		}
	}

	return true;
}


void iStreamLine::assignScalars(float *ss, int i, vtkIdType l)
{
	int j;
	//
	//  assuming scalars are placed horizontally, as in iDateReader::getMeshOutput() 
	//
	for(j=0; j<nadd; j++) ss[i++] = *(ptrsca+l+scalarSize*j);
}


vtkIdType iStreamLine::getVector(double x[3], float v[3], float *vort, float *div)
{
	return iStreamLine::getVector(x,v,this->ptrvec,this->dims,this->org,this->spa,vort,div);
}


vtkIdType iStreamLine::getVector(double x[3], float v[3], float *ptrvec, int dims[3], vtkFloat org[3], vtkFloat spa[3], float *vort, float *div)
{
	int ijk1[3], ijk2[3];
	int i;
	float w1[3], w2[3];
	float *v111, *v112, *v121, *v122, *v211, *v212, *v221, *v222;

	for(i=0; i<3; i++)
	{
		w2[i] = (x[i]-org[i])/spa[i];
		ijk1[i] = (int)floor(w2[i]);
		w2[i] -= ijk1[i];
		w1[i] = 1.0 - w2[i];
		if(ijk1[i] < 0) ijk1[i] = 0;
		if(ijk1[i] >= dims[i]) ijk1[i] = dims[i] - 1;
		ijk2[i] = ijk1[i] + 1;
		if(ijk2[i] >= dims[i]) ijk2[i] = dims[i] - 1;
	}

	//
	//  Vectors of the CIC neigbors
	//
	v111 = ptrvec + 3*(ijk1[0]+dims[0]*(ijk1[1]+dims[1]*ijk1[2]));
	v112 = ptrvec + 3*(ijk1[0]+dims[0]*(ijk1[1]+dims[1]*ijk2[2]));
	v121 = ptrvec + 3*(ijk1[0]+dims[0]*(ijk2[1]+dims[1]*ijk1[2]));
	v122 = ptrvec + 3*(ijk1[0]+dims[0]*(ijk2[1]+dims[1]*ijk2[2]));
	v211 = ptrvec + 3*(ijk2[0]+dims[0]*(ijk1[1]+dims[1]*ijk1[2]));
	v212 = ptrvec + 3*(ijk2[0]+dims[0]*(ijk1[1]+dims[1]*ijk2[2]));
	v221 = ptrvec + 3*(ijk2[0]+dims[0]*(ijk2[1]+dims[1]*ijk1[2]));
	v222 = ptrvec + 3*(ijk2[0]+dims[0]*(ijk2[1]+dims[1]*ijk2[2]));

	for(i=0; i<3; i++)
	{
		v[i] =
			v111[i]*w1[0]*w1[1]*w1[2] +
			v211[i]*w2[0]*w1[1]*w1[2] +
			v121[i]*w1[0]*w2[1]*w1[2] +
			v221[i]*w2[0]*w2[1]*w1[2] +
			v112[i]*w1[0]*w1[1]*w2[2] +
			v212[i]*w2[0]*w1[1]*w2[2] +
			v122[i]*w1[0]*w2[1]*w2[2] +
			v222[i]*w2[0]*w2[1]*w2[2];
	}

	if(vort != NULL)
	{
		vort[0] = (
			(v112[1]-v111[1])*w1[0]*w1[1] +
			(v122[1]-v121[1])*w1[0]*w2[1] +
			(v212[1]-v211[1])*w2[0]*w1[1] +
			(v222[1]-v221[1])*w2[0]*w2[1])/spa[2] - (
			(v121[2]-v111[2])*w1[0]*w1[2] +
			(v122[2]-v112[2])*w1[0]*w2[2] +
			(v221[2]-v211[2])*w2[0]*w1[2] +
			(v222[2]-v212[2])*w2[0]*w2[2])/spa[1];
		
		vort[1] = (
			(v211[2]-v111[2])*w1[1]*w1[2] +
			(v212[2]-v112[2])*w1[1]*w2[2] +
			(v221[2]-v121[2])*w2[1]*w1[2] +
			(v222[2]-v122[2])*w2[1]*w2[2])/spa[0] - (
			(v112[0]-v111[0])*w1[0]*w1[1] +
			(v122[0]-v121[0])*w1[0]*w2[1] +
			(v212[0]-v211[0])*w2[0]*w1[1] +
			(v222[0]-v221[0])*w2[0]*w2[1])/spa[2];
		
		vort[2] = (
			(v121[0]-v111[0])*w1[0]*w1[2] +
			(v122[0]-v112[0])*w1[0]*w2[2] +
			(v221[0]-v211[0])*w2[0]*w1[2] +
			(v222[0]-v212[0])*w2[0]*w2[2])/spa[1] - (
			(v211[1]-v111[1])*w1[1]*w1[2] +
			(v212[1]-v112[1])*w1[1]*w2[2] +
			(v221[1]-v121[1])*w2[1]*w1[2] +
			(v222[1]-v122[1])*w2[1]*w2[2])/spa[0];
	}

	if(div != NULL)
	{
		*div = (
			(v211[0]-v111[0])*w1[1]*w1[2] +
			(v212[0]-v112[0])*w1[1]*w2[2] +
			(v221[0]-v121[0])*w2[1]*w1[2] +
			(v222[0]-v122[0])*w2[1]*w2[2])/spa[0] + (
			(v121[1]-v111[1])*w1[0]*w1[2] +
			(v122[1]-v112[1])*w1[0]*w2[2] +
			(v221[1]-v211[1])*w2[0]*w1[2] +
			(v222[1]-v212[1])*w2[0]*w2[2])/spa[1] + (
			(v112[2]-v111[2])*w1[0]*w1[1] +
			(v122[2]-v121[2])*w1[0]*w2[1] +
			(v212[2]-v211[2])*w2[0]*w1[1] +
			(v222[2]-v221[2])*w2[0]*w2[1])/spa[2];
	}

	return ijk1[0] + dims[0]*(ijk1[1]+(vtkIdType)dims[1]*ijk1[2]);

}

