/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ihistogrammaker.h"
#include "imath.h"
#include "ipiecewisefunction.h"
#include "iglobals.h"

#include <vtkDataSetSource.h>
#include <vtkObjectFactory.h>
#include <vtkFloatArray.h>
#include <vtkIdTypeArray.h>
#include <vtkStructuredPoints.h>
#include <vtkPointData.h>


void reportNullPointer(int ec);


//------------------------------------------------------------------------------
iHistogramMaker* iHistogramMaker::New()
{
	// First try to create the object from the vtkObjectFactory
	vtkObject* ret = vtkObjectFactory::CreateInstance("iHistogramMaker");
	if(ret)
    {
		return (iHistogramMaker*)ret;
    }
	// If the factory was unable to create the object, then create it here.
	return new iHistogramMaker;
}


// Construct object to extract all of the input data.
iHistogramMaker::iHistogramMaker()
{

	histo = new iPiecewiseFunction;
	if(histo == 0) reportNullPointer(5801);

	nBinsPrev = 0;
	inLogPrev = 0;

	vMinPrev = vMaxPrev = 0.0;
	component = -1;

	myMTime = 0;

}


iHistogramMaker::~iHistogramMaker()
{
	delete histo;
}

	
//----------------------------------------------------------------------------
// Specify the input data or filter.
void iHistogramMaker::SetInput(vtkDataSet *input, int c)
{
	this->vtkProcessObject::SetNthInput(0, input);
	component = c;
	myMTime = 0;
}

//----------------------------------------------------------------------------
// Specify the input data or filter.
void iHistogramMaker::SetInputComponent(int c)
{
	component = c;
	myMTime = 0;
}

//----------------------------------------------------------------------------
// Specify the input data or filter.
vtkDataSet* iHistogramMaker::GetInput()
{
	if (this->NumberOfInputs < 1)
    {
		return NULL;
    }
	
	return (vtkDataSet *)(this->Inputs[0]);
}


int iHistogramMaker::getEstimatedNumberOfBins()
{ 
	if(this->GetInput() == 0) return 0;

	vtkFloatArray *input = (vtkFloatArray *)this->GetInput()->GetPointData()->GetScalars();
	if(input == 0) return 0;
	vtkIdType ntup = input->GetNumberOfTuples();
	return round(1.0+sqrt((float)ntup)); 
}


iPiecewiseFunction* iHistogramMaker::getHistogram(int nBins, bool inLog)
{
	//
	//  Get the data limits
	//
	vtkDataSet *input = this->GetInput();
	if(input==0 || input->GetPointData()==0 || input->GetPointData()->GetScalars()==0) return 0;
	//
	//  Access the data for both vtkPolyData and vtkStructuredPoints
	//
	float *data = ((vtkFloatArray *)input->GetPointData()->GetScalars())->GetPointer(0);
	if(data == 0) return 0;

	vtkIdType ncom = input->GetPointData()->GetScalars()->GetNumberOfComponents();
	vtkIdType ntup = input->GetPointData()->GetScalars()->GetNumberOfTuples();

	if(component<0 || component>=ncom) return 0;
	//
	//  No need to recompute
	//
	if(input->GetMTime() > myMTime) 
	{
		//	
		//  Recompute the limits
		//
		int i;
		vtkIdType l, lmin, lmax;
		float v, vmin, vmax;
		vmin = vmax = *(data+component);
		lmin = lmax = 0;
		for(l=1; l<ntup; l++)
		{
			v = *(data+component+l*ncom);
			if(v < vmin) 
			{
				vmin = v;
				lmin = l;
			}
			if(v > vmax) 
			{
				vmax = v;
				lmax = l;
			}
		}
		vMinPrev = vmin;
		vMaxPrev = vmax;

		lMinPrev = lmin;
		lMaxPrev = lmax;

		if(input->IsA("vtkImageData"))
		{
			int dims[3], ijk[3];
			vtkFloat spa[3], org[3];
			vtkImageData *id = (vtkImageData *)input;
			id->GetDimensions(dims);
			id->GetSpacing(spa);
			id->GetOrigin(org);

			ijk[2] = lmin/(dims[0]*dims[1]);
			ijk[1] = (lmin-dims[0]*dims[1]*ijk[2])/dims[0];
			ijk[0] = lmin % dims[0];
			for(i=0; i<3; i++) xMinPrev[i] = org[i] + spa[i]*ijk[i];

			ijk[2] = lmax/(dims[0]*dims[1]);
			ijk[1] = (lmax-dims[0]*dims[1]*ijk[2])/dims[0];
			ijk[0] = lmax % dims[0];
			for(i=0; i<3; i++) xMaxPrev[i] = org[i] + spa[i]*ijk[i];

		}
		else
		{
			for(i=0; i<3; i++) xMinPrev[i] = xMaxPrev[i] = 0.0;
		}


	}
	
	return this->getHistogram(nBins,inLog,vMinPrev,vMaxPrev);

}


iPiecewiseFunction* iHistogramMaker::getHistogram(int nBins, bool inLog, float vMin, float vMax)
{
	if(nBins<0 || nBins>1000000 || vMin>=vMax || this->GetInput()==0) return 0;
	if(this->GetInput()->GetPointData() == 0) return 0;
	if(this->GetInput()->GetPointData()->GetScalars() == 0) return 0;
	if(component<0 || component>=this->GetInput()->GetPointData()->GetScalars()->GetNumberOfComponents()) return 0;
	//
	//  No need to recompute
	//
	if(nBins==nBinsPrev && inLog==inLogPrev && vMin==vMinPrev && vMax==vMaxPrev &&
		this->GetInput()->GetMTime()<myMTime) return histo;
	//
	//  Must recompute
	//
	vMinPrev = vMin;
	vMaxPrev = vMax;
	nBinsPrev = nBins;
	inLogPrev = inLog;
	this->ExecuteData(0);
	return histo;

}


void iHistogramMaker::ExecuteData(vtkDataObject *vtkNotUsed(out))
{
	//
	//  Must recompute
	//
	vtkFloatArray *input = (vtkFloatArray *)this->GetInput()->GetPointData()->GetScalars();
	float *data = input->GetPointer(0);
	int ncom = input->GetNumberOfComponents();
	vtkIdType ntup = input->GetNumberOfTuples();

	this->Modified();
	myMTime = this->GetMTime();

	int nBins;
	int i, iv;
	vtkIdType l;
	float v, vmin, vmax, vstp;

	if(nBinsPrev <= 0) nBins = round(1.0+sqrt((float)ntup)); else nBins = nBinsPrev;

	float *pbin = new float[nBins];
	if(pbin == 0) reportNullPointer(5802);
	float *vbin = new float[nBins];
	if(vbin == 0) reportNullPointer(5803);

	if(inLogPrev)
	{
		vmin = log10(1.0e-30+fabs(vMinPrev));
		vmax = log10(1.0e-30+fabs(vMaxPrev));
	} 
	else 
	{
		vmin = vMinPrev;
		vmax = vMaxPrev;
	}
	vstp = 1.0001*(vmax-vmin)/nBins;

	if(vstp <= 0.0)
	{
		delete [] pbin;
		delete [] vbin;
		delete histo;
		histo = 0;
		return;
	}

	for(i=0; i<nBins; i++)
	{
		vbin[i] = vmin + vstp*(i+0.5);
		pbin[i] = 0.0;
	}
	
	for(l=0; l<ntup; l++)
	{

		if(this->GetAbortExecute()) break;

		v = *(data+component+l*ncom);
		if(inLogPrev) v = log10(1.0e-30+fabs(v));
		iv = round((v-vmin)/vstp-0.4999);
		if(iv>=0 && iv<nBins)
		{
			pbin[iv] += 1.0;
		} 
	}

	delete histo;
	histo = new iPiecewiseFunction;
	if(histo == 0) reportNullPointer(5904);

	histo->movFunctionPoint(0,0.0,0.0);
	histo->movFunctionPoint(1,1.0,0.0);
	for(i=0; i<nBins; i++) histo->addFunctionPoint((i+0.5)/nBins,pbin[i]);

	delete [] pbin;
	delete [] vbin;

}
