/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

This class is based on:

  Program:   Visualization Toolkit
  Module:    $RCSfile: iexpressionparser.h,v $
  Language:  C++
  Date:      $Date: 2003/12/08 03:36:58 $
  Version:   $Revision: 1.2 $

  Copyright (c) 1993-2002 Ken Martin, Will Schroeder, Bill Lorensen 
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef IEXPRESSIONPARSER_H
#define IEXPRESSIONPARSER_H

enum ParserTypes
{
	VTK_PARSER_IMMEDIATE = 1,
	VTK_PARSER_UNARY_MINUS,
	VTK_PARSER_NOT,
	// supported math functions
	VTK_PARSER_OR,
	VTK_PARSER_AND,
	VTK_PARSER_LT,
	VTK_PARSER_GT,
	VTK_PARSER_LE,
	VTK_PARSER_GE,
	VTK_PARSER_EQ,
	VTK_PARSER_NE,
	VTK_PARSER_ADD,
	VTK_PARSER_SUBTRACT,
	VTK_PARSER_MULTIPLY,
	VTK_PARSER_DIVIDE,
	VTK_PARSER_POWER,
	VTK_PARSER_ABSOLUTE_VALUE,
	VTK_PARSER_EXPONENT,
	VTK_PARSER_CEILING,
	VTK_PARSER_FLOOR,
	VTK_PARSER_LOGARITHM,
	VTK_PARSER_SQUARE_ROOT,
	VTK_PARSER_SINE,
	VTK_PARSER_COSINE,
	VTK_PARSER_TANGENT,
	VTK_PARSER_ARCSINE,
	VTK_PARSER_ARCCOSINE,
	VTK_PARSER_ARCTANGENT,
	VTK_PARSER_HYPERBOLIC_SINE,
	VTK_PARSER_HYPERBOLIC_COSINE,
	VTK_PARSER_HYPERBOLIC_TANGENT,
	// functions involving vectors
	VTK_PARSER_VECTOR_UNARY_MINUS,
	VTK_PARSER_DOT_PRODUCT,
	VTK_PARSER_VECTOR_ADD,
	VTK_PARSER_VECTOR_SUBTRACT,
	VTK_PARSER_SCALAR_MULTIPLE,
	VTK_PARSER_MAGNITUDE,
	VTK_PARSER_NORMALIZE,
	// codes for scalar variables come before those for vectors
	VTK_PARSER_BEGIN_VARIABLES
};


// the value that is retuned as a result if there is an error
#define VTK_PARSER_ERROR_RESULT 1.0e300

#include "istring.h"

class iExpressionParser
{
	
public:
	
	iExpressionParser();
	virtual ~iExpressionParser();
	
	// Decription:
	// Set/Get input string to evaluate. 
	void SetFunction(const char *function);
	
	// Description:
	// Get a scalar result from evaluating the input function.
	bool GetScalarResult(double &result);
	
	// Description:
	// Get a vector result from evaluating the input function.
	bool GetVectorResult(double result[3]);
	
	// Description:
	// Set the value of a scalar variable.  If a variable with this name
	// exists, then its value will be set to the new value.  If there is not
	// already a variable with this name, variableName will be added to the
	// list of variables, and its value will be set to the new value.
	void SetScalarVariableValue(const char* variableName, double value);
	void SetScalarVariableValue(int i, double value);
	
	// Description:
	// Get the value of a scalar variable.
	double GetScalarVariableValue(const char* variableName);
	double GetScalarVariableValue(int i);
	
	// Description:
	// Set the value of a vector variable.  If a variable with this name
	// exists, then its value will be set to the new value.  If there is not
	// already a variable with this name, variableName will be added to the
	// list of variables, and its value will be set to the new value.
	void SetVectorVariableValue(const char* variableName, double xValue, double yValue, double zValue);
	void SetVectorVariableValue(const char* variableName, const double values[3]) { this->SetVectorVariableValue(variableName,values[0],values[1],values[2]); }
	void SetVectorVariableValue(int i, double xValue, double yValue, double zValue);
	void SetVectorVariableValue(int i, const double values[3]) { this->SetVectorVariableValue(i,values[0],values[1],values[2]); }
	
	// Description:
	// Get the value of a vector variable.
	double* GetVectorVariableValue(const char* variableName);
	double* GetVectorVariableValue(int i);
	void GetVectorVariableValue(const char* variableName, double value[3]) 
	{
		double *r = this->GetVectorVariableValue(variableName);
		value[0] = r[0]; value[1] = r[1]; value[2] = r[2]; 
	}
	void GetVectorVariableValue(int i, double value[3]) 
	{
		double *r = this->GetVectorVariableValue(i);
		value[0] = r[0]; value[1] = r[1]; value[2] = r[2]; 
	}
	
	// Description:
	// Get the number of scalar variables.
	int GetNumberOfScalarVariables(){ return NumberOfScalarVariables; }
	
	// Description:
	// Get the number of vector variables.
	int GetNumberOfVectorVariables(){ return NumberOfVectorVariables; }
	
	// Description:
	// Get the ith scalar variable name.
	char* GetScalarVariableName(int i);
	
	// Description:
	// Get the ith vector variable name.
	char* GetVectorVariableName(int i);
	
	// Description:
	// Remove all the current variables.
	void RemoveAllVariables();

	//  get the error string
	iString GetErrorMessage(){ return ErrorMessage; }

	//  block/unblock division by zero errors
	bool GetIgnoreDivisionByZero(){ return ignoreDivZero; }
	void SetIgnoreDivisionByZero(bool s){ ignoreDivZero = s; modified = true; }
	
protected:
	
	int Parse();
	void Evaluate();
	
	int CheckSyntax();
	void RemoveSpaces();
	
	int BuildInternalFunctionStructure();
	void BuildInternalSubstringStructure(int beginIndex, int endIndex);
	void AddInternalByte(unsigned char newByte);
	
	int IsSubstringCompletelyEnclosed(int beginIndex, int endIndex);
	int FindEndOfMathFunction(int beginIndex);
	
	int IsVariableName(int currentIndex);
	int IsElementaryOperator(int op);
	
	int GetMathFunctionNumber(int currentIndex);
	int GetMathFunctionStringLength(int mathFunctionNumber);
	int GetElementaryOperatorNumber(char op);
	int GetOperandNumber(int currentIndex);
	int GetVariableNameLength(int variableNumber);
	
	int DisambiguateOperators();
	
	char* Function;
	int FunctionLength;
	int NumberOfScalarVariables;
	int NumberOfVectorVariables;
	char** ScalarVariableNames;
	char** VectorVariableNames;
	double* ScalarVariableValues;
	double** VectorVariableValues;
	unsigned char *ByteCode;
	int ByteCodeSize;
	double *Immediates;
	int ImmediatesSize;
	double *Stack;
	int StackSize;
	int StackPointer;

	iString ErrorMessage;
	bool modified, ignoreDivZero;
	
};

#endif  // IEXPRESSIONPARSER_H
