/*
 * SpanDSP - a series of DSP components for telephony
 *
 * fax_tests.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2003 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: fax_tests.c,v 1.37 2006/05/13 02:21:19 steveu Exp $
 */

/*! \page fax_tests_page FAX tests
\section fax_tests_page_sec_1 What does it do?
\section fax_tests_page_sec_2 How does it work?
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <inttypes.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <audiofile.h>
#include <tiffio.h>

#include "spandsp.h"

#define SAMPLES_PER_CHUNK   160

#define MACHINES            2
struct machine_s
{
    int chan;
    AFfilehandle handle;
    int16_t amp[SAMPLES_PER_CHUNK];
    int len;
    fax_state_t fax;
    int done;
    char tag[50];
} machines[MACHINES];

static void phase_b_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    
    i = (intptr_t) user_data;
    printf("%d: Phase B handler on channel %d - (0x%X) %s\n", i, i, result, t30_frametype(result));
}
/*- End of function --------------------------------------------------------*/

static void phase_d_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    t30_stats_t t;
    char ident[21];

    i = (intptr_t) user_data;
    printf("%d: Phase D handler on channel %d - (0x%X) %s\n", i, i, result, t30_frametype(result));
    t30_get_transfer_statistics(s, &t);
    printf("%d: Phase D: bit rate %d\n", i, t.bit_rate);
    printf("%d: Phase D: pages transferred %d\n", i, t.pages_transferred);
    printf("%d: Phase D: image size %d x %d\n", i, t.width, t.length);
    printf("%d: Phase D: image resolution %d x %d\n", i, t.x_resolution, t.y_resolution);
    printf("%d: Phase D: bad rows %d\n", i, t.bad_rows);
    printf("%d: Phase D: longest bad row run %d\n", i, t.longest_bad_row_run);
    printf("%d: Phase D: compression type %d\n", i, t.encoding);
    printf("%d: Phase D: image size %d\n", i, t.image_size);
    t30_get_local_ident(s, ident);
    printf("%d: Phase D: local ident '%s'\n", i, ident);
    t30_get_far_ident(s, ident);
    printf("%d: Phase D: remote ident '%s'\n", i, ident);
}
/*- End of function --------------------------------------------------------*/

static void phase_e_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    const char *t;
    
    i = (intptr_t) user_data;
    printf("%d: Phase E handler on channel %d - (%d) %s\n", i, i, result, t30_completion_code_to_str(result));
    if ((t = t30_get_far_country(s)))
        printf("%d: Phase E: Remote was made in '%s'\n", i, t);
    if ((t = t30_get_far_vendor(s)))
        printf("%d: Phase E: Remote was made by '%s'\n", i, t);
    if ((t = t30_get_far_model(s)))
        printf("%d: Phase E: Remote is model '%s'\n", i, t);
    machines[i].done = TRUE;
}
/*- End of function --------------------------------------------------------*/

static int document_handler(t30_state_t *s, void *user_data, int event)
{
    int i;
    
    i = (intptr_t) user_data;
    printf("Document handler on channel %d, event %d\n", i, event);
    return FALSE;
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    AFfilesetup filesetup;
    int i;
    int j;
    struct machine_s *mc;
    int outframes;
    char buf[128 + 1];
    int16_t silence[SAMPLES_PER_CHUNK];
    int alldone;

    filesetup = afNewFileSetup();
    if (filesetup == AF_NULL_FILESETUP)
    {
        fprintf(stderr, "    Failed to create file setup\n");
        exit(2);
    }
    afInitSampleFormat(filesetup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, 16);
    afInitRate(filesetup, AF_DEFAULT_TRACK, (float) SAMPLE_RATE);
    afInitFileFormat(filesetup, AF_FILE_WAVE);
    afInitChannels(filesetup, AF_DEFAULT_TRACK, 1);

    memset(silence, 0, sizeof(silence));
    for (j = 0;  j < MACHINES;  j++)
    {
        machines[j].chan = j;
        mc = &machines[j];

        sprintf(buf, "/tmp/fax%d.wav", mc->chan + 1);
        mc->handle = afOpenFile(buf, "w", filesetup);
        if (mc->handle == AF_NULL_FILEHANDLE)
        {
            fprintf(stderr, "    Cannot create wave file '%s'\n", buf);
            exit(2);
        }
        i = mc->chan + 1;
        sprintf(buf, "%d%d%d%d%d%d%d%d", i, i, i, i, i, i, i, i);
        fax_init(&mc->fax, (mc->chan & 1)  ?  FALSE  :  TRUE, (void *) (intptr_t) j);
        t30_set_local_ident(&mc->fax.t30_state, buf);
        t30_set_local_nsf(&mc->fax.t30_state, (const uint8_t *) "\x50\x00\x00\x00Spandsp\x00", 12);
        if ((mc->chan & 1))
        {
            sprintf(buf, "fax_tests_%d.tif", (mc->chan + 1)/2);
            t30_set_rx_file(&mc->fax.t30_state, buf, -1);
        }
        else
        {
            t30_set_tx_file(&mc->fax.t30_state, "itutests.tif", -1, -1);
        }
        t30_set_phase_b_handler(&mc->fax.t30_state, phase_b_handler, (void *) (intptr_t) mc->chan);
        t30_set_phase_d_handler(&mc->fax.t30_state, phase_d_handler, (void *) (intptr_t) mc->chan);
        t30_set_phase_e_handler(&mc->fax.t30_state, phase_e_handler, (void *) (intptr_t) mc->chan);
        t30_set_document_handler(&mc->fax.t30_state, document_handler, (void *) (intptr_t) mc->chan);
        sprintf(mc->tag, "FAX-%d", j + 1);
        mc->fax.t30_state.logging.level = SPAN_LOG_SHOW_SEVERITY | SPAN_LOG_SHOW_PROTOCOL | SPAN_LOG_SHOW_TAG | SPAN_LOG_FLOW;
        mc->fax.t30_state.logging.tag = mc->tag;
        mc->fax.v29rx.logging.level = SPAN_LOG_SHOW_SEVERITY | SPAN_LOG_SHOW_PROTOCOL | SPAN_LOG_SHOW_TAG | SPAN_LOG_FLOW;
        mc->fax.v29rx.logging.tag = mc->tag;
        mc->fax.logging.level = SPAN_LOG_SHOW_SEVERITY | SPAN_LOG_SHOW_PROTOCOL | SPAN_LOG_SHOW_TAG | SPAN_LOG_FLOW;
        mc->fax.logging.tag = mc->tag;
        memset(mc->amp, 0, sizeof(mc->amp));
        mc->done = FALSE;
    }
    for (;;)
    {
        alldone = TRUE;
        for (j = 0;  j < MACHINES;  j++)
        {
            mc = &machines[j];

            mc->len = fax_tx(&mc->fax, mc->amp, SAMPLES_PER_CHUNK);
            /* The receive side always expects a full block of samples, but the
               transmit side may not be sending any when it doesn't need to. We
               may need to pad with some silence. */
            if (mc->len < SAMPLES_PER_CHUNK)
            {
                memset(mc->amp + mc->len, 0, sizeof(int16_t)*(SAMPLES_PER_CHUNK - mc->len));
                mc->len = SAMPLES_PER_CHUNK;
            }
#if 1
            outframes = afWriteFrames(mc->handle, AF_DEFAULT_TRACK, mc->amp, mc->len);
            if (outframes != mc->len)
                break;
#endif
            if (machines[j ^ 1].len < SAMPLES_PER_CHUNK)
                memset(machines[j ^ 1].amp + machines[j ^ 1].len, 0, sizeof(int16_t)*(SAMPLES_PER_CHUNK - machines[j ^ 1].len));
            if (fax_rx(&mc->fax, machines[j ^ 1].amp, SAMPLES_PER_CHUNK))
                break;
            if (!mc->done)
                alldone = FALSE;
        }
        if (alldone  ||  j < MACHINES)
            break;
    }
    for (j = 0;  j < MACHINES;  j++)
    {
        mc = &machines[j];
        fax_release(&mc->fax);
        if (afCloseFile(mc->handle) != 0)
        {
            fprintf(stderr, "    Cannot close wave file for channel %d\n", mc->chan);
            exit(2);
        }
    }
    afFreeFileSetup(filesetup);
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
