#define ENABLE_V17
/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t38.c - An implementation of a T.38 terminal, less the packet exchange part
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2005 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t38_terminal.c,v 1.17 2006/05/18 14:15:36 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <inttypes.h>
#include <fcntl.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <tiffio.h>

#include "spandsp/telephony.h"
#include "spandsp/logging.h"
#include "spandsp/bit_operations.h"
#include "spandsp/queue.h"
#include "spandsp/power_meter.h"
#include "spandsp/complex.h"
#include "spandsp/tone_generate.h"
#include "spandsp/async.h"
#include "spandsp/hdlc.h"
#include "spandsp/fsk.h"
#include "spandsp/v29rx.h"
#include "spandsp/v29tx.h"
#include "spandsp/v27ter_rx.h"
#include "spandsp/v27ter_tx.h"
#if defined(ENABLE_V17)
#include "spandsp/v17rx.h"
#include "spandsp/v17tx.h"
#endif
#include "spandsp/t4.h"

#include "spandsp/t30_fcf.h"
#include "spandsp/t35.h"
#include "spandsp/t30.h"

#include "spandsp/t38.h"

#define ms_to_samples(t)            (((t)*SAMPLE_RATE)/1000)

#define MS_PER_TX_CHUNK             30

#define INDICATOR_TX_COUNT          3

enum
{
    T38_CED_TONE,
    T38_CNG_TONE,
    T38_V21_RX,
    T38_V27TER_RX,
    T38_V29_RX,
    T38_V17_RX
};

enum
{
    T38_TIMED_STEP_NONE = 0,
    T38_TIMED_STEP_NON_ECM_MODEM,
    T38_TIMED_STEP_NON_ECM_MODEM_2,
    T38_TIMED_STEP_NON_ECM_MODEM_3,
    T38_TIMED_STEP_HDLC_MODEM,
    T38_TIMED_STEP_HDLC_MODEM_2,
    T38_TIMED_STEP_HDLC_MODEM_3,
    T38_TIMED_STEP_HDLC_MODEM_4,
    T38_TIMED_STEP_PAUSE
};

static int get_non_ecm_image_chunk(t38_state_t *s, uint8_t *buf, int len)
{
    int i;
    int j;
    int bit;
    int byte;

    for (i = 0;  i < len;  i++)
    {
        byte = 0;
        for (j = 0;  j < 8;  j++)
        {
            bit = t30_non_ecm_getbit(&s->t30_state);
            byte = (byte << 1) | (bit & 0x01);
            if (bit & 2)
            {
                byte <<= (7 - j);
                buf[i++] = (uint8_t) byte;
                return -i;
            }
        }
        buf[i] = (uint8_t) byte;
    }
    return i;
}
/*- End of function --------------------------------------------------------*/

static void put_non_ecm_image_chunk(t38_state_t *s, const uint8_t *buf, int len)
{
    int i;
    int j;
    int byte;

    for (i = 0;  i < len;  i++)
    {
        byte = buf[i];
        for (j = 7;  j >= 0;  j--)
            t30_non_ecm_putbit(&(s->t30_state), (byte >> j) & 1);
    }
}
/*- End of function --------------------------------------------------------*/

int t38_terminal_process_rx_missing(t38_state_t *s, int seq_no)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "IFP seq no %d missing\n", seq_no);
    return 0;
}
/*- End of function --------------------------------------------------------*/

int t38_terminal_process_rx_indicator(t38_state_t *s, int indicator)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "Rx indicator %s\n", t38_indicator(indicator));
    switch (indicator)
    {
    case T38_IND_NO_SIGNAL:
        if (s->current_rx_indicator == T38_IND_V21_PREAMBLE
            &&
            (s->current_rx_type == T30_MODEM_V21  ||  s->current_rx_type == T30_MODEM_CNG))
        {
            t30_hdlc_accept(&(s->t30_state), TRUE, NULL, PUTBIT_CARRIER_DOWN);
        }
        break;
    case T38_IND_CNG:
        break;
    case T38_IND_CED:
        break;
    case T38_IND_V21_PREAMBLE:
        if (s->current_rx_type == T30_MODEM_V21)
        {
            t30_hdlc_accept(&(s->t30_state), TRUE, NULL, PUTBIT_CARRIER_UP);
            t30_hdlc_accept(&(s->t30_state), TRUE, NULL, PUTBIT_FRAMING_OK);
        }
        s->data_final = FALSE;
        s->hdlc_len = 0;
        s->tx_out_bytes = 0;
        break;
    case T38_IND_V27TER_2400_TRAINING:
        break;
    case T38_IND_V27TER_4800_TRAINING:
        break;
    case T38_IND_V29_7200_TRAINING:
        break;
    case T38_IND_V29_9600_TRAINING:
        break;
    case T38_IND_V17_7200_SHORT_TRAINING:
        break;
    case T38_IND_V17_7200_LONG_TRAINING:
        break;
    case T38_IND_V17_9600_SHORT_TRAINING:
        break;
    case T38_IND_V17_9600_LONG_TRAINING:
        break;
    case T38_IND_V17_12000_SHORT_TRAINING:
        break;
    case T38_IND_V17_12000_LONG_TRAINING:
        break;
    case T38_IND_V17_14400_SHORT_TRAINING:
        break;
    case T38_IND_V17_14400_LONG_TRAINING:
        break;
    case T38_IND_V8_ANSAM:
        break;
    case T38_IND_V8_SIGNAL:
        break;
    case T38_IND_V34_CNTL_CHANNEL_1200:
        break;
    case T38_IND_V34_PRI_CHANNEL:
        break;
    case T38_IND_V34_CC_RETRAIN:
        break;
    case T38_IND_V33_12000_TRAINING:
        break;
    case T38_IND_V33_14400_TRAINING:
        break;
    default:
        break;
    }
    s->current_rx_indicator = indicator;
    s->tx_out_bytes = 0;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int t38_terminal_process_rx_data(t38_state_t *s, int data_type, int field_type, const uint8_t *buf, int len)
{
    int i;

    span_log(&s->logging, SPAN_LOG_FLOW, "Rx data type %s/%s\n", t38_data_type(data_type), t38_field_type(field_type));
    switch (data_type)
    {
    case T38_DATA_V21:
    case T38_DATA_V27TER_2400:
    case T38_DATA_V27TER_4800:
    case T38_DATA_V29_7200:
    case T38_DATA_V29_9600:
    case T38_DATA_V17_7200:
    case T38_DATA_V17_9600:
    case T38_DATA_V17_12000:
    case T38_DATA_V17_14400:
    case T38_DATA_V8:
    case T38_DATA_V34_PRI_RATE:
    case T38_DATA_V34_CC_1200:
    case T38_DATA_V34_PRI_CH:
    case T38_DATA_V33_12000:
    case T38_DATA_V33_14400:
    default:
        break;
    }

    switch (field_type)
    {
    case T38_FIELD_HDLC_DATA:
        for (i = 0;  i < len;  i++)
            s->tx_data[s->tx_out_bytes++] = bit_reverse8(buf[i]);
        break;
    case T38_FIELD_HDLC_SIG_END:
        /* There should be no data */
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_SIG_END!\n");
        /* This is only expected when the HDLC signal drops unexpectedly.
           If it drops normally, following the end of a final frame, this
           message is not expected. */
        if (s->current_rx_type == T30_MODEM_V21)
            t30_hdlc_accept(&(s->t30_state), TRUE, NULL, PUTBIT_CARRIER_DOWN);
         s->tx_out_bytes = 0;
       break;
    case T38_FIELD_HDLC_FCS_OK:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_FCS_OK!\n");
        span_log(&s->logging, SPAN_LOG_FLOW, "Type %s - CRC OK\n", t30_frametype(s->tx_data[2]));
        if (s->current_rx_type == T30_MODEM_V21)
            t30_hdlc_accept(&(s->t30_state), TRUE, s->tx_data, s->tx_out_bytes);
        s->tx_out_bytes = 0;
        break;
    case T38_FIELD_HDLC_FCS_BAD:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_FCS_BAD!\n");
        span_log(&s->logging, SPAN_LOG_FLOW, "Type %s - CRC bad\n", t30_frametype(s->tx_data[2]));
        if (s->current_rx_type == T30_MODEM_V21)
            t30_hdlc_accept(&(s->t30_state), FALSE, s->tx_data, s->tx_out_bytes);
        s->tx_out_bytes = 0;
        break;
    case T38_FIELD_HDLC_FCS_OK_SIG_END:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_FCS_OK_SIG_END!\n");
        span_log(&s->logging, SPAN_LOG_FLOW, "Type %s - CRC OK, sig end\n", t30_frametype(s->tx_data[2]));
        if (s->current_rx_type == T30_MODEM_V21)
        {
            t30_hdlc_accept(&(s->t30_state), TRUE, s->tx_data, s->tx_out_bytes);
            t30_hdlc_accept(&(s->t30_state), TRUE, NULL, PUTBIT_CARRIER_DOWN);
        }
        s->tx_out_bytes = 0;
        break;
    case T38_FIELD_HDLC_FCS_BAD_SIG_END:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_FCS_BAD_SIG_END!\n");
        span_log(&s->logging, SPAN_LOG_FLOW, "Type %s - CRC bad, sig end\n", t30_frametype(s->tx_data[2]));
        if (s->current_rx_type == T30_MODEM_V21)
        {
            t30_hdlc_accept(&(s->t30_state), FALSE, s->tx_data, s->tx_out_bytes);
            t30_hdlc_accept(&(s->t30_state), TRUE, NULL, PUTBIT_CARRIER_DOWN);
        }
        s->tx_out_bytes = 0;
        break;
    case T38_FIELD_T4_NON_ECM_DATA:
        if (!s->rx_signal_present)
        {
            t30_non_ecm_putbit(&(s->t30_state), PUTBIT_TRAINING_SUCCEEDED);
            s->rx_signal_present = TRUE;
        }
        put_non_ecm_image_chunk(s, buf, len);
        break;
    case T38_FIELD_T4_NON_ECM_SIG_END:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_NON_ECM_SIG_END!\n");
        t30_non_ecm_putbit(&(s->t30_state), PUTBIT_CARRIER_DOWN);
        s->rx_signal_present = FALSE;
        break;
    case T38_FIELD_CM_MESSAGE:
    case T38_FIELD_JM_MESSAGE:
    case T38_FIELD_CI_MESSAGE:
    case T38_FIELD_V34RATE:
    default:
        break;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void t38_send_hdlc(void *user_data, const uint8_t *msg, int len)
{
    t38_state_t *s;

    s = (t38_state_t *) user_data;

    if (s->hdlc_len == 0)
    {
        memcpy(s->hdlc_buf, msg, len);
        s->hdlc_len = len;
        s->hdlc_ptr = 0;
    }
    else if (s->hdlc_len2 == 0)
    {
        memcpy(s->hdlc_buf2, msg, len);
        s->hdlc_len2 = len;
    }
    else if (s->hdlc_len3 == 0)
    {
        memcpy(s->hdlc_buf3, msg, len);
        s->hdlc_len3 = len;
    }
}
/*- End of function --------------------------------------------------------*/

int t38_send_timeout(t38_state_t *s, int samples)
{
    int len;
    uint8_t buf[100];
    /* Training times for all the modem options, with and without TEP */
    static const int training_time[] =
    {
           0,      0,   /* T38_IND_NO_SIGNAL */
           0,      0,   /* T38_IND_CNG */
           0,      0,   /* T38_IND_CED */
        1000,   1000,   /* T38_IND_V21_PREAMBLE */
         943,   1158,   /* T38_IND_V27TER_2400_TRAINING */
         708,    923,   /* T38_IND_V27TER_4800_TRAINING */
         234,    454,   /* T38_IND_V29_7200_TRAINING */
         234,    454,   /* T38_IND_V29_9600_TRAINING */
         142,    367,   /* T38_IND_V17_7200_SHORT_TRAINING */
        1393,   1618,   /* T38_IND_V17_7200_LONG_TRAINING */
         142,    367,   /* T38_IND_V17_9600_SHORT_TRAINING */
        1393,   1618,   /* T38_IND_V17_9600_LONG_TRAINING */
         142,    367,   /* T38_IND_V17_12000_SHORT_TRAINING */
        1393,   1618,   /* T38_IND_V17_12000_LONG_TRAINING */
         142,    367,   /* T38_IND_V17_14400_SHORT_TRAINING */
        1393,   1618,   /* T38_IND_V17_14400_LONG_TRAINING */
           0,      0,   /* T38_IND_V8_ANSAM */
           0,      0,   /* T38_IND_V8_SIGNAL */
           0,      0,   /* T38_IND_V34_CNTL_CHANNEL_1200 */
           0,      0,   /* T38_IND_V34_PRI_CHANNEL */
           0,      0,   /* T38_IND_V34_CC_RETRAIN */
           0,      0,   /* T38_IND_V33_12000_TRAINING */
           0,      0,   /* T38_IND_V33_14400_TRAINING */
    };

    s->samples += samples;
    
    t30_timer_update(&s->t30_state, samples);
    
    if (s->timed_step == T38_TIMED_STEP_NONE)
        return 0;
    if (s->samples < s->next_send_samples)
        return 0;
    
    switch (s->timed_step)
    {
    case T38_TIMED_STEP_NON_ECM_MODEM:
        /* Create a 75ms silence */
        t38_send_indicator(s, T38_IND_NO_SIGNAL, INDICATOR_TX_COUNT);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM_2;
        s->next_send_samples += ms_to_samples(75);
        break;
    case T38_TIMED_STEP_NON_ECM_MODEM_2:
        /* Switch on a fast modem, and give the training time to complete */
        t38_send_indicator(s, s->next_tx_indicator, INDICATOR_TX_COUNT);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM_3;
        s->next_send_samples += ms_to_samples(training_time[s->next_tx_indicator << 1]);
        break;
    case T38_TIMED_STEP_NON_ECM_MODEM_3:
        /* Send a chunk of non-ECM image data */
        if ((len = get_non_ecm_image_chunk(s, buf, s->octets_per_non_ecm_packet)))
            t38_send_data(s, s->current_tx_data, T38_FIELD_T4_NON_ECM_DATA, buf, abs(len));
        if (len > 0)
        {
            s->next_send_samples += ms_to_samples(MS_PER_TX_CHUNK);
        }
        else
        {
            t38_send_data(s, s->current_tx_data, T38_FIELD_T4_NON_ECM_SIG_END, NULL, 0);
            t38_send_indicator(s, T38_IND_NO_SIGNAL, INDICATOR_TX_COUNT);
            s->timed_step = T38_TIMED_STEP_NONE;
            t30_send_complete(&(s->t30_state));
        }
        break;
    case T38_TIMED_STEP_HDLC_MODEM:
        /* Send HDLC preambling */
        t38_send_indicator(s, s->next_tx_indicator, INDICATOR_TX_COUNT);
        s->next_send_samples += ms_to_samples(training_time[s->next_tx_indicator << 1]);
        s->timed_step = T38_TIMED_STEP_HDLC_MODEM_2;
        break;
    case T38_TIMED_STEP_HDLC_MODEM_2:
        /* Send HDLC octet */
        buf[0] = bit_reverse8(s->hdlc_buf[s->hdlc_ptr++]);
        t38_send_data(s, s->current_tx_data, T38_FIELD_HDLC_DATA, buf, 1);
        if (s->hdlc_ptr >= s->hdlc_len)
            s->timed_step = T38_TIMED_STEP_HDLC_MODEM_3;
        s->next_send_samples += ms_to_samples(MS_PER_TX_CHUNK);
        break;
    case T38_TIMED_STEP_HDLC_MODEM_3:
        /* End of HDLC frame */
        s->hdlc_ptr = 0;
        if (s->hdlc_len2)
        {
            /* Another frame follows */
            s->hdlc_len = s->hdlc_len2;
            memcpy(s->hdlc_buf, s->hdlc_buf2, s->hdlc_len);
            s->hdlc_len2 = s->hdlc_len3;
            if (s->hdlc_len2)
            {
                memcpy(s->hdlc_buf2, s->hdlc_buf3, s->hdlc_len2);
                s->hdlc_len3 = 0;
            }
            t38_send_data(s, s->current_tx_data, T38_FIELD_HDLC_FCS_OK, NULL, 0);
            s->timed_step = T38_TIMED_STEP_HDLC_MODEM_2;
        }
        else
        {
            s->hdlc_len = 0;
            t38_send_data(s, s->current_tx_data, T38_FIELD_HDLC_FCS_OK_SIG_END, NULL, 0);
            s->timed_step = T38_TIMED_STEP_HDLC_MODEM_4;
            t30_send_complete(&s->t30_state);
        }
        s->next_send_samples += ms_to_samples(MS_PER_TX_CHUNK);
        break;
    case T38_TIMED_STEP_HDLC_MODEM_4:
        /* End of HDLC transmission */
        t38_send_data(s, s->current_tx_data, T38_FIELD_HDLC_SIG_END, NULL, 0);
        s->timed_step = T38_TIMED_STEP_NONE;
        break;
    case T38_TIMED_STEP_PAUSE:
        /* End of timed pause */
        s->timed_step = T38_TIMED_STEP_NONE;
        t30_send_complete(&s->t30_state);
        break;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void t38_set_rx_type(void *user_data, int type, int short_train)
{
    t38_state_t *s;

    s = (t38_state_t *) user_data;
    span_log(&s->logging, SPAN_LOG_FLOW, "Set rx type %d\n", type);
    s->current_rx_type = type;
}
/*- End of function --------------------------------------------------------*/

static void t38_set_tx_type(void *user_data, int type, int short_train)
{
    t38_state_t *s;

    s = (t38_state_t *) user_data;
    span_log(&s->logging, SPAN_LOG_FLOW, "Set tx type %d\n", type);
    if (s->current_tx_type == type)
        return;

    switch (type)
    {
    case T30_MODEM_PAUSE:
        s->next_send_samples = s->samples + ms_to_samples(short_train);
        s->timed_step = T38_TIMED_STEP_PAUSE;
        t38_send_indicator(s, T38_IND_NO_SIGNAL, INDICATOR_TX_COUNT);
        break;
    case T30_MODEM_CED:
        s->next_send_samples = s->samples + ms_to_samples(3000);
        s->timed_step = T38_TIMED_STEP_PAUSE;
        t38_send_indicator(s, T38_IND_CED, INDICATOR_TX_COUNT);
        break;
    case T30_MODEM_CNG:
        t38_send_indicator(s, T38_IND_CNG, INDICATOR_TX_COUNT);
        break;
    case T30_MODEM_V21:
        if (s->current_tx_type > T30_MODEM_V21)
        {
            /* Pause before switching from phase C, as per T.30. If we omit this, the receiver
               might not see the carrier fall between the high speed and low speed sections. */
            t38_send_indicator(s, T38_IND_NO_SIGNAL, INDICATOR_TX_COUNT);
            s->next_send_samples = s->samples + ms_to_samples(75);
        }
        else
        {
            s->next_send_samples = s->samples;
        }
        s->next_tx_indicator = T38_IND_V21_PREAMBLE;
        s->current_tx_data = T38_DATA_V21;
        s->timed_step = T38_TIMED_STEP_HDLC_MODEM;
        break;
    case T30_MODEM_V27TER_2400:
        s->octets_per_non_ecm_packet = MS_PER_TX_CHUNK*2400/8000;
        s->next_tx_indicator = T38_IND_V27TER_2400_TRAINING;
        s->current_tx_data = T38_DATA_V27TER_2400;
        s->next_send_samples = s->samples + ms_to_samples(MS_PER_TX_CHUNK);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM;
        break;
    case T30_MODEM_V27TER_4800:
        s->octets_per_non_ecm_packet = MS_PER_TX_CHUNK*4800/8000;
        s->next_tx_indicator = T38_IND_V27TER_4800_TRAINING;
        s->current_tx_data = T38_DATA_V27TER_4800;
        s->next_send_samples = s->samples + ms_to_samples(MS_PER_TX_CHUNK);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM;
        break;
    case T30_MODEM_V29_7200:
        s->octets_per_non_ecm_packet = MS_PER_TX_CHUNK*7200/8000;
        s->next_tx_indicator = T38_IND_V29_7200_TRAINING;
        s->current_tx_data = T38_DATA_V29_7200;
        s->next_send_samples = s->samples + ms_to_samples(MS_PER_TX_CHUNK);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM;
        break;
    case T30_MODEM_V29_9600:
        s->octets_per_non_ecm_packet = MS_PER_TX_CHUNK*9600/8000;
        s->next_tx_indicator = T38_IND_V29_9600_TRAINING;
        s->current_tx_data = T38_DATA_V29_9600;
        s->next_send_samples = s->samples + ms_to_samples(MS_PER_TX_CHUNK);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM;
        break;
    case T30_MODEM_V17_7200:
        s->octets_per_non_ecm_packet = MS_PER_TX_CHUNK*7200/8000;
        s->next_tx_indicator = (short_train)  ?  T38_IND_V17_7200_SHORT_TRAINING  :  T38_IND_V17_7200_LONG_TRAINING;
        s->current_tx_data = T38_DATA_V17_7200;
        s->next_send_samples = s->samples + ms_to_samples(MS_PER_TX_CHUNK);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM;
        break;
    case T30_MODEM_V17_9600:
        s->octets_per_non_ecm_packet = MS_PER_TX_CHUNK*9600/8000;
        s->next_tx_indicator = (short_train)  ?  T38_IND_V17_9600_SHORT_TRAINING  :  T38_IND_V17_9600_LONG_TRAINING;
        s->current_tx_data = T38_DATA_V17_9600;
        s->next_send_samples = s->samples + ms_to_samples(MS_PER_TX_CHUNK);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM;
        break;
    case T30_MODEM_V17_12000:
        s->octets_per_non_ecm_packet = MS_PER_TX_CHUNK*12000/8000;
        s->next_tx_indicator = (short_train)  ?  T38_IND_V17_12000_SHORT_TRAINING  :  T38_IND_V17_12000_LONG_TRAINING;
        s->current_tx_data = T38_DATA_V17_12000;
        s->next_send_samples = s->samples + ms_to_samples(MS_PER_TX_CHUNK);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM;
        break;
    case T30_MODEM_V17_14400:
        s->octets_per_non_ecm_packet = MS_PER_TX_CHUNK*14400/8000;
        s->next_tx_indicator = (short_train)  ?  T38_IND_V17_14400_SHORT_TRAINING  :  T38_IND_V17_14400_LONG_TRAINING;
        s->current_tx_data = T38_DATA_V17_14400;
        s->next_send_samples = s->samples + ms_to_samples(MS_PER_TX_CHUNK);
        s->timed_step = T38_TIMED_STEP_NON_ECM_MODEM;
        break;
    case T30_MODEM_DONE:
        span_log(&s->logging, SPAN_LOG_FLOW, "FAX exchange complete\n");
        break;
    }
    s->current_tx_type = type;
}
/*- End of function --------------------------------------------------------*/

t38_state_t *t38_terminal_init(t38_state_t *s,
                               int calling_party,
                               t38_tx_packet_handler_t *tx_packet_handler,
                               void *tx_packet_user_data)
{
    memset(s, 0, sizeof(*s));
    if (tx_packet_handler == NULL)
        return NULL;
    s->rx_signal_present = FALSE;

    s->gateway = FALSE;
    s->timed_step = T38_TIMED_STEP_NONE;
    s->hdlc_ptr = 0;
    s->tx_packet_handler = tx_packet_handler;
    s->tx_packet_user_data = tx_packet_user_data;

    s->data_rate_management_method = 2;
    s->data_transport_protocol = T38_TRANSPORT_UDPTL;
    s->fill_bit_removal = FALSE;
    s->mmr_transcoding = FALSE;
    s->jbig_transcoding = FALSE;
    s->max_buffer_size = 400;
    s->max_datagram_size = 100;
    s->t38_version = 0;
    s->iaf = TRUE;

    t30_init(&(s->t30_state), calling_party, NULL);
    s->t30_state.set_rx_type_handler = t38_set_rx_type;
    s->t30_state.set_rx_type_user_data = (void *) s;
    s->t30_state.set_tx_type_handler = t38_set_tx_type;
    s->t30_state.set_tx_type_user_data = (void *) s;
    s->t30_state.send_hdlc_handler = t38_send_hdlc;
    s->t30_state.send_hdlc_user_data = (void *) s;

    s->samples = 0;
    s->next_send_samples = 0;

    t30_restart(&s->t30_state);
    return s;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
