#define ENABLE_V17
/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t30.c - ITU T.30 FAX transfer processing
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2003 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t30.c,v 1.96 2006/05/24 09:19:11 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <inttypes.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <tgmath.h>
#include <tiffio.h>

#include "spandsp/telephony.h"
#include "spandsp/logging.h"
#include "spandsp/queue.h"
#include "spandsp/power_meter.h"
#include "spandsp/complex.h"
#include "spandsp/tone_generate.h"
#include "spandsp/async.h"
#include "spandsp/hdlc.h"
#include "spandsp/fsk.h"
#include "spandsp/v29rx.h"
#include "spandsp/v29tx.h"
#include "spandsp/v27ter_rx.h"
#include "spandsp/v27ter_tx.h"
#if defined(ENABLE_V17)
#include "spandsp/v17rx.h"
#include "spandsp/v17tx.h"
#endif
#include "spandsp/t4.h"

#include "spandsp/t30_fcf.h"
#include "spandsp/t35.h"
#include "spandsp/t30.h"

#define MAXMESSAGE          (MAXFRAME + 4)  /* HDLC frame, including address, control, and CRC */
#define MAX_MESSAGE_TRIES   3

#define ms_to_samples(t)    (((t)*SAMPLE_RATE)/1000)

/* T.30 defines the following call phases:
   Phase A: Call set-up.
       Exchange of CNG, CED and the called terminal identification.
   Phase B: Pre-message procedure for identifying and selecting the required facilities.
       Capabilities negotiation, and training, up the the confirmation to receive.
   Phase C: Message transmission (includes phasing and synchronization where appropriate).
       Transfer of the message at high speed.
   Phase D: Post-message procedure, including end-of-message and confirmation and multi-document procedures.
       End of message and acknowledgement.
   Phase E: Call release
       Final call disconnect. */
enum
{
    T30_PHASE_IDLE = 0,     /* Freshly initialised */
    T30_PHASE_A_CED,        /* Doing the CED (answer) sequence */
    T30_PHASE_A_CNG,        /* Doing the CNG (caller) sequence */
    T30_PHASE_B_RX,         /* Receiving pre-message control messages */
    T30_PHASE_B_TX,         /* Transmitting pre-message control messages */
    T30_PHASE_C_RX,         /* Receiving a document message */
    T30_PHASE_C_TX,         /* Transmitting a document message */
    T30_PHASE_D_RX,         /* Receiving post-message control messages */
    T30_PHASE_D_TX,         /* Transmitting post-message control messages */
    T30_PHASE_E,            /* In phase E */
    T30_PHASE_CALL_FINISHED /* Call completely finished */
};

static const char *phase_names[] =
{
    "T30_PHASE_IDLE",
    "T30_PHASE_A_CED",
    "T30_PHASE_A_CNG",
    "T30_PHASE_B_RX",
    "T30_PHASE_B_TX",
    "T30_PHASE_C_RX",
    "T30_PHASE_C_TX",
    "T30_PHASE_D_RX",
    "T30_PHASE_D_TX",
    "T30_PHASE_E",
    "T30_PHASE_CALL_FINISHED"
};

/* These state names are modelled after places in the T.30 flow charts. */
enum
{
    T30_STATE_ANSWERING = 1,
    T30_STATE_B,
    T30_STATE_C,
    T30_STATE_D,
    T30_STATE_D_TCF,
    T30_STATE_D_POST_TCF,
    T30_STATE_F_TCF,
    T30_STATE_F_CFR,
    T30_STATE_F_DOC,
    T30_STATE_F_POST_DOC,
    T30_STATE_R,
    T30_STATE_T,
    T30_STATE_I,
    T30_STATE_II,
    T30_STATE_II_MPS,
    T30_STATE_II_EOP,
    T30_STATE_II_EOM,
    T30_STATE_III_MPS_MCF,
    T30_STATE_III_MPS_RTP,
    T30_STATE_III_MPS_RTN,
    T30_STATE_III_EOP_MCF,
    T30_STATE_III_EOP_RTP,
    T30_STATE_III_EOP_RTN,
    T30_STATE_IV,
    T30_STATE_VA,
    T30_STATE_VB,
    T30_STATE_VC,
    T30_STATE_VD,
    T30_STATE_VIA,
    T30_STATE_VIB,
    T30_STATE_VIC,
    T30_STATE_VID,
    T30_STATE_VII,
    T30_STATE_IX
};

enum
{
    T30_MODE_SEND_DOC = 1,
    T30_MODE_RECEIVE_DOC
};

enum
{
    T30_COPY_QUALITY_GOOD = 0,
    T30_COPY_QUALITY_POOR,
    T30_COPY_QUALITY_BAD
};

#define DISBIT1     0x01
#define DISBIT2     0x02
#define DISBIT3     0x04
#define DISBIT4     0x08
#define DISBIT5     0x10
#define DISBIT6     0x20
#define DISBIT7     0x40
#define DISBIT8     0x80

/* All timers specified in milliseconds */

/* Time-out T0 defines the amount of time an automatic calling terminal waits for the called terminal
to answer the call.
T0 begins after the dialling of the number is completed and is reset:
a)       when T0 times out; or
b)       when timer T1 is started; or
c)       if the terminal is capable of detecting any condition which indicates that the call will not be
         successful, when such a condition is detected.
The recommended value of T0 is 60+-5s; however, when it is anticipated that a long call set-up
time may be encountered, an alternative value of up to 120s may be used.
NOTE - National regulations may require the use of other values for T0. */
#define DEFAULT_TIMER_T0            60000

/* Time-out T1 defines the amount of time two terminals will continue to attempt to identify each
other. T1 is 35+-5s, begins upon entering phase B, and is reset upon detecting a valid signal or
when T1 times out.
For operating methods 3 and 4 (see 3.1), the calling terminal starts time-out T1 upon reception of
the V.21 modulation scheme.
For operating method 4 bis a (see 3.1), the calling terminal starts time-out T1 upon starting
transmission using the V.21 modulation scheme. */
#define DEFAULT_TIMER_T1            35000

/* Time-out T2 makes use of the tight control between commands and responses to detect the loss of
command/response synchronization. T2 is 6+-1s and begins when initiating a command search
(e.g., the first entrance into the "command received" subroutine, reference flow diagram in 5.2).
T2 is reset when an HDLC flag is received or when T2 times out. */
#define DEFAULT_TIMER_T2            7000

/* Time-out T3 defines the amount of time a terminal will attempt to alert the local operator in
response to a procedural interrupt. Failing to achieve operator intervention, the terminal will
discontinue this attempt and shall issue other commands or responses. T3 is 10+-5s, begins on the
first detection of a procedural interrupt command/response signal (i.e., PIN/PIP or PRI-Q) and is
reset when T3 times out or when the operator initiates a line request. */
#define DEFAULT_TIMER_T3            15000

/* NOTE - For manual FAX units, the value of timer T4 may be either 3.0s +-15% or 4.5s +-15%.
If the value of 4.5s is used, then after detection of a valid response to the first DIS, it may
be reduced to 3.0s +-15%. T4 = 3.0s +-15% for automatic units. */
#define DEFAULT_TIMER_T4            3450

/* Time-out T5 is defined for the optional T.4 error correction mode. Time-out T5 defines the amount
of time waiting for clearance of the busy condition of the receiving terminal. T5 is 60+-5s and
begins on the first detection of the RNR response. T5 is reset when T5 times out or the MCF or PIP
response is received or when the ERR or PIN response is received in the flow control process after
transmitting the EOR command. If the timer T5 has expired, the DCN command is transmitted for
call release. */
#define DEFAULT_TIMER_T5            65000

#define DEFAULT_TIMER_T6            5000

#define DEFAULT_TIMER_T7            6000

#define DEFAULT_TIMER_T8            10000

/* Exact widths in PELs for the difference resolutions, and page widths:
    R4    864 pels/215mm for ISO A4, North American Letter and Legal
    R4   1024 pels/255mm for ISO B4
    R4   1216 pels/303mm for ISO A3
    R8   1728 pels/215mm for ISO A4, North American Letter and Legal
    R8   2048 pels/255mm for ISO B4
    R8   2432 pels/303mm for ISO A3
    R16  3456 pels/215mm for ISO A4, North American Letter and Legal
    R16  4096 pels/255mm for ISO B4
    R16  4864 pels/303mm for ISO A3
*/

#if defined(ENABLE_V17)
#define T30_V17_FALLBACK_START          0
#define T30_V29_FALLBACK_START          3
#define T30_V27TER_FALLBACK_START       6
#else
#define T30_V29_FALLBACK_START          0
#define T30_V27TER_FALLBACK_START       2
#endif

static const struct
{
    int bit_rate;
    int modem_type;
    int min_scan_class;
    uint8_t dcs_code;
} fallback_sequence[] =
{
#if defined(ENABLE_V17)
    {14400, T30_MODEM_V17_14400,    0, DISBIT6},
    {12000, T30_MODEM_V17_12000,    0, (DISBIT6 | DISBIT4)},
    { 9600, T30_MODEM_V17_9600,     0, (DISBIT6 | DISBIT3)},
#endif
    { 9600, T30_MODEM_V29_9600,     0, DISBIT3},
#if defined(ENABLE_V17)
    { 7200, T30_MODEM_V17_7200,     1, (DISBIT6 | DISBIT4 | DISBIT3)},
#endif
    { 7200, T30_MODEM_V29_7200,     1, (DISBIT4 | DISBIT3)},
    { 4800, T30_MODEM_V27TER_4800,  2, DISBIT4},
    { 2400, T30_MODEM_V27TER_2400,  3, 0},
    {    0, 0, 0, 0}
};

static void queue_phase(t30_state_t *s, int phase);
static void set_phase(t30_state_t *s, int phase);
static void set_state(t30_state_t *s, int state);
static void send_simple_frame(t30_state_t *s, int type);
static void send_frame(t30_state_t *s, const uint8_t *fr, int frlen);
static void send_dcn(t30_state_t *s);
static void disconnect(t30_state_t *s);
static void decode_password(t30_state_t *s, char *msg, const uint8_t *pkt, int len);
static void decode_20digit_msg(t30_state_t *s, char *msg, const uint8_t *pkt, int len);
static void decode_url_msg(t30_state_t *s, char *msg, const uint8_t *pkt, int len);

static void rx_start_page(t30_state_t *s)
{
    t4_rx_set_image_width(&(s->t4), s->image_width);
    t4_rx_set_sub_address(&(s->t4), s->sub_address);
    t4_rx_set_far_ident(&(s->t4), s->far_ident);
    t4_rx_set_vendor(&(s->t4), s->vendor);
    t4_rx_set_model(&(s->t4), s->model);

    t4_rx_set_rx_encoding(&(s->t4), s->line_encoding);
    t4_rx_set_y_resolution(&(s->t4), s->y_resolution);

    t4_rx_start_page(&(s->t4));
}
/*- End of function --------------------------------------------------------*/

static int copy_quality(t30_state_t *s)
{
    t4_stats_t stats;

    /* There is no specification for judging copy quality. However, we need to classify
       it at three levels, to control what we do next: OK; tolerable, but retrain;
       intolerable, so retrain. */
    t4_get_transfer_statistics(&(s->t4), &stats);
    span_log(&s->logging, SPAN_LOG_FLOW, "rows %d, bad %d\n", stats.length, stats.bad_rows);
    if (stats.bad_rows*50 < stats.length)
        return T30_COPY_QUALITY_GOOD;
    if (stats.bad_rows*20 < stats.length)
        return T30_COPY_QUALITY_POOR;
    return T30_COPY_QUALITY_BAD;
}
/*- End of function --------------------------------------------------------*/

const char *t30_completion_code_to_str(int result)
{
    switch (result)
    {
    case T30_ERR_OK:
        return "OK";
    case T30_ERR_CEDTONE:
        return "The CED tone exceeded 5s";
    case T30_ERR_T0EXPIRED:
        return "Timed out waiting for initial communication";
    case T30_ERR_T1EXPIRED:
        return "Timed out waiting for the first message";
    case T30_ERR_T3EXPIRED:
        return "Timed out waiting for procedural interrupt";
    case T30_ERR_HDLCCARR:
        return "The HDLC carrier did not stop in a timely manner";
    case T30_ERR_CANNOTTRAIN:
        return "Failed to train with any of the compatible modems";
    case T30_ERR_OPERINTFAIL:
        return "Operator intervention failed";
    case T30_ERR_INCOMPATIBLE:
        return "Far end is not compatible";
    case T30_ERR_NOTRXCAPABLE:
        return "Far end is not receive capable";
    case T30_ERR_NOTTXCAPABLE:
        return "Far end is not transmit capable";
    case T30_ERR_UNEXPECTED:
        return "Unexpected message received";
    case T30_ERR_NORESSUPPORT:
        return "Far end cannot receive at the resolution of the image";
    case T30_ERR_NOSIZESUPPORT:
        return "Far end cannot receive at the size of image";
    case T30_ERR_FILEERROR:
        return "TIFF/F file cannot be opened";
    case T30_ERR_NOPAGE:
        return "TIFF/F page not found";
    case T30_ERR_BADTIFF:
        return "TIFF/F format is not compatible";
    case T30_ERR_UNSUPPORTED:
        return "Unsupported feature";
    case T30_ERR_BADDCSTX:
        return "Received bad response to DCS or training";
    case T30_ERR_BADPGTX:
        return "Received a DCN from remote after sending a page";
    case T30_ERR_ECMPHDTX:
        return "Invalid ECM response received from receiver";
    case T30_ERR_ECMRNRTX:
        return "Timer T5 expired, receiver not ready";
    case T30_ERR_GOTDCNTX:
        return "Received a DCN while waiting for a DIS";
    case T30_ERR_INVALRSPTX:
        return "Invalid response after sending a page";
    case T30_ERR_NODISTX:
        return "Received other than DIS while waiting for DIS";
    case T30_ERR_NXTCMDTX:
        return "Timed out waiting for next send_page command from driver";
    case T30_ERR_PHBDEADTX:
        return "Received no response to DCS, training or TCF";
    case T30_ERR_PHDDEADTX:
        return "No response after sending a page";
    case T30_ERR_ECMPHDRX:
        return "Invalid ECM response received from transmitter";
    case T30_ERR_GOTDCSRX:
        return "DCS received while waiting for DTC";
    case T30_ERR_INVALCMDRX:
        return "Unexpected command after page received";
    case T30_ERR_NOCARRIERRX:
        return "Carrier lost during fax receive";
    case T30_ERR_NOEOLRX:
        return "Timed out while waiting for EOL (end Of line)";
    case T30_ERR_NOFAXRX:
        return "Timed out while waiting for first line";
    case T30_ERR_NXTCMDRX:
        return "Timed out waiting for next receive page command";
    case T30_ERR_T2EXPDCNRX:
        return "Timer T2 expired while waiting for DCN";
    case T30_ERR_T2EXPDRX:
        return "Timer T2 expired while waiting for phase D";
    case T30_ERR_T2EXPFAXRX:
        return "Timer T2 expired while waiting for fax page";
    case T30_ERR_T2EXPMPSRX:
        return "Timer T2 expired while waiting for next fax page";
    case T30_ERR_T2EXPRRRX:
        return "Timer T2 expired while waiting for RR command";
    case T30_ERR_T2EXPRX:
        return "Timer T2 expired while waiting for NSS, DCS or MCF";
    case T30_ERR_DCNWHYRX:
        return "Unexpected DCN while waiting for DCS or DIS";
    case T30_ERR_DCNDATARX:
        return "Unexpected DCN while waiting for image data";
    case T30_ERR_DCNFAXRX:
        return "Unexpected DCN while waiting for EOM, EOP or MPS";
    case T30_ERR_DCNPHDRX:
        return "Unexpected DCN after EOM or MPS sequence";
    case T30_ERR_DCNRRDRX:
        return "Unexpected DCN after RR/RNR sequence";
    case T30_ERR_DCNNORTNRX:
        return "Unexpected DCN after requested retransmission";
    case T30_ERR_BADPAGE:
        return "TIFF/F page number tag missing";
    case T30_ERR_BADTAG:
        return "Incorrect values for TIFF/F tags";
    case T30_ERR_BADTIFFHDR:
        return "Bad TIFF/F header - incorrect values in fields";
    case T30_ERR_BADPARM:
        return "Invalid value for fax parameter";
    case T30_ERR_BADSTATE:
        return "Invalid initial state value specified";
    case T30_ERR_CMDDATA:
        return "Last command contained invalid data";
    case T30_ERR_DISCONNECT:
        return "Fax call disconnected by the other station";
    case T30_ERR_INVALARG:
        return "Illegal argument to function";
    case T30_ERR_INVALFUNC:
        return "Illegal call to function";
    case T30_ERR_NODATA:
        return "Data requested is not available (NSF, DIS, DCS)";
    case T30_ERR_NOMEM:
        return "Cannot allocate memory for more pages";
    case T30_ERR_NOPOLL:
        return "Poll not accepted";
    case T30_ERR_NOSTATE:
        return "Initial state value not set";
    case T30_ERR_RETRYDCN:
        return "Disconnected after permitted retries";
    };
    return "???";
}
/*- End of function --------------------------------------------------------*/

void t30_non_ecm_putbit(void *user_data, int bit)
{
    t30_state_t *s;
    int signal_was_present;

    s = (t30_state_t *) user_data;
    if (bit < 0)
    {
        /* Special conditions */
        switch (bit)
        {
        case PUTBIT_TRAINING_FAILED:
            span_log(&s->logging, SPAN_LOG_FLOW, "Non-ECM carrier training failed\n");
            break;
        case PUTBIT_TRAINING_SUCCEEDED:
            /* The modem is now trained */
            span_log(&s->logging, SPAN_LOG_FLOW, "Non-ECM carrier trained\n");
            /* In case we are in trainability test mode... */
            /* A FAX machine is supposed to send 1.5s of training test
               data, but some send a little bit less. Lets just check
               the first 1s, and be safe. */
            s->training_current_zeros = 0;
            s->training_most_zeros = 0;
            s->rx_signal_present = TRUE;
            s->timer_t2_t4 = 0;
            break;
        case PUTBIT_CARRIER_UP:
            span_log(&s->logging, SPAN_LOG_FLOW, "Non-ECM carrier up\n");
            break;
        case PUTBIT_CARRIER_DOWN:
            span_log(&s->logging, SPAN_LOG_FLOW, "Non-ECM carrier down\n");
            signal_was_present = s->rx_signal_present;
            s->rx_signal_present = FALSE;
            switch (s->state)
            {
            case T30_STATE_F_TCF:
                /* Only respond if we managed to actually sync up with the source. We don't
                   want to respond just because we saw a click. These often occur just
                   before the real signal, with many modems. Presumably this is due to switching
                   within the far end modem. We also want to avoid the possibility of responding
                   to the tail end of any slow modem signal. If there was a genuine data signal
                   which we failed to train on it should not matter. If things are that bad, we
                   do not stand much chance of good quality communications. */
                if (signal_was_present)
                {
                    /* Although T.30 says the training test should be 1.5s of all 0's, some FAX
                       machines send a burst of all 1's before the all 0's. Tolerate this. */
                    if (s->training_current_zeros > s->training_most_zeros)
                        s->training_most_zeros = s->training_current_zeros;
                    if (s->training_most_zeros < fallback_sequence[s->current_fallback].bit_rate)
                    {
                        span_log(&s->logging, SPAN_LOG_FLOW, "Trainability test failed - longest run of zeros was %d\n", s->training_most_zeros);
                        set_phase(s, T30_PHASE_B_TX);
                        send_simple_frame(s, T30_FTT);
                    }
                    else
                    {
                        /* The training went OK */
                        if (!s->in_message  &&  t4_rx_init(&(s->t4), s->rx_file, T4_COMPRESSION_ITU_T4_2D))
                        {
                            span_log(&s->logging, SPAN_LOG_WARNING, "Cannot open target TIFF file '%s'\n", s->rx_file);
                            s->current_status = T30_ERR_FILEERROR;
                            send_dcn(s);
                        }
                        else
                        {
                            /* The document is OK */
                            s->short_train = TRUE;
                            s->in_message = TRUE;
                            rx_start_page(s);
                            set_phase(s, T30_PHASE_B_TX);
                            set_state(s, T30_STATE_F_CFR);
                            send_simple_frame(s, T30_CFR);
                        }
                    }
                }
                break;
            case T30_STATE_F_POST_DOC:
                /* Page ended cleanly */
                break;
            default:
                /* We should be receiving a document right now, but it did not end cleanly. */
                if (signal_was_present)
                {
                    span_log(&s->logging, SPAN_LOG_WARNING, "Page did not end cleanly\n");
                    /* We trained OK, so we should have some kind of received page, even though
                       it did not end cleanly. */
                    set_state(s, T30_STATE_F_POST_DOC);
                    set_phase(s, T30_PHASE_D_RX);
                    s->timer_t2_t4 = ms_to_samples(DEFAULT_TIMER_T2);
                    s->timer_is_t4 = FALSE;
                    if (s->current_status == T30_ERR_NOCARRIERRX)
                        s->current_status = T30_ERR_OK;
                }
                else
                {
                    span_log(&s->logging, SPAN_LOG_WARNING, "Non-ECM carrier not found\n");
                    s->current_status = T30_ERR_NOCARRIERRX;
                }
                break;
            }
            if (s->next_phase != T30_PHASE_IDLE)
            {
                set_phase(s, s->next_phase);
                s->next_phase = T30_PHASE_IDLE;
            }
            break;
        default:
            span_log(&s->logging, SPAN_LOG_FLOW, "Eh!\n");
            break;
        }
        return;
    }
    switch (s->state)
    {
    case T30_STATE_F_TCF:
        /* Trainability test */
        if (bit)
        {
            if (s->training_current_zeros > s->training_most_zeros)
                s->training_most_zeros = s->training_current_zeros;
            s->training_current_zeros = 0;
        }
        else
        {
            s->training_current_zeros++;
        }
        break;
    case T30_STATE_F_DOC:
        /* Document transfer */
        if (t4_rx_putbit(&(s->t4), bit))
        {
            /* That is the end of the document */
            set_state(s, T30_STATE_F_POST_DOC);
            queue_phase(s, T30_PHASE_D_RX);
            s->timer_t2_t4 = ms_to_samples(DEFAULT_TIMER_T2);
            s->timer_is_t4 = FALSE;
        }
        break;
    }
}
/*- End of function --------------------------------------------------------*/

int t30_non_ecm_getbit(void *user_data)
{
    int bit;
    t30_state_t *s;

    s = (t30_state_t *) user_data;
    switch (s->state)
    {
    case T30_STATE_D_TCF:
        /* Trainability test. */
        bit = 0;
        if (s->training_test_bits-- < 0)
        {
            /* Finished sending training test. */
            bit = 2;
        }
        break;
    case T30_STATE_I:
        /* Transferring real data. */
        bit = t4_tx_getbit(&(s->t4));
        break;
    case T30_STATE_D_POST_TCF:
    case T30_STATE_II_MPS:
    case T30_STATE_II_EOM:
    case T30_STATE_II_EOP:
        /* We should be padding out a block of samples if we are here */
        bit = 0;
        break;
    default:
        span_log(&s->logging, SPAN_LOG_WARNING, "t30_non_ecm_getbit in bad state %d\n", s->state);
        bit = 2;
        break;
    }
    return bit;
}
/*- End of function --------------------------------------------------------*/

static void print_frame(t30_state_t *s, const char *io, const uint8_t *fr, int frlen)
{
    span_log(&s->logging,
             SPAN_LOG_FLOW,
             "%s %s with%s final frame tag\n",
             io,
             t30_frametype(fr[2]),
             (fr[1] & 0x10)  ?  ""  :  "out");
    span_log_buf(&s->logging, SPAN_LOG_FLOW, io, fr, frlen);
}
/*- End of function --------------------------------------------------------*/

static void send_simple_frame(t30_state_t *s, int type)
{
    uint8_t frame[3];

    /* The simple command/response frames are always final frames */
    frame[0] = 0xFF;
    frame[1] = 0x13;
    frame[2] = (uint8_t) (type | s->dis_received);
    send_frame(s, frame, 3);
}
/*- End of function --------------------------------------------------------*/

static void send_frame(t30_state_t *s, const uint8_t *fr, int frlen)
{
    print_frame(s, ">>>", fr, frlen);

    if (s->send_hdlc_handler)
        s->send_hdlc_handler(s->send_hdlc_user_data, fr, frlen);
}
/*- End of function --------------------------------------------------------*/

static void send_ident_frame(t30_state_t *s, uint8_t cmd)
{
    size_t len;
    int p;
    uint8_t frame[23];

    /* Only send if there is an ident to send. */
    if (s->local_ident[0])
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Sending ident '%s'\n", s->local_ident);
        len = strlen(s->local_ident);
        p = 0;
        frame[p++] = 0xFF;
        frame[p++] = 0x03;
        frame[p++] = (uint8_t) (cmd | s->dis_received); /* T30_TSI, T30_CIG or T30_CSI */
        while (len > 0)
            frame[p++] = s->local_ident[--len];
        while (p < 23)
            frame[p++] = ' ';
        send_frame(s, frame, 23);
    }
}
/*- End of function --------------------------------------------------------*/

static void send_nsf_frame(t30_state_t *s)
{
    int p;
    uint8_t frame[100 + 3];

    /* Only send if there is an NSF message to send. */
    if (s->local_nsf_len)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Sending user supplied NSF - %d octets\n", s->local_nsf_len);
        p = 0;
        frame[p++] = 0xFF;
        frame[p++] = 0x03;
        frame[p++] = (uint8_t) (T30_NSF | s->dis_received);
        for (  ;  p < s->local_nsf_len + 3;  p++)
            frame[p] = s->local_nsf[p - 3];
        send_frame(s, frame, s->local_nsf_len + 3);
    }
}
/*- End of function --------------------------------------------------------*/

static int set_dis_or_dtc(t30_state_t *s)
{
    /* Whether we use a DIS or a DTC is determined by whether we have received a DIS.
       We just need to edit the prebuilt message. */
    s->dis_dtc_frame[2] = (uint8_t) (T30_DIS | s->dis_received);
    /* If we have a file name to receive into, then we are receive capable */
    if (s->rx_file[0])
        s->dis_dtc_frame[4] |= DISBIT2;
    else
        s->dis_dtc_frame[4] &= ~DISBIT2;
    /* If we have a file name to transmit, then we are ready to transmit (polling) */
    if (s->tx_file[0])
        s->dis_dtc_frame[4] |= DISBIT1;
    else
        s->dis_dtc_frame[4] &= ~DISBIT1;
    t30_decode_dis_dtc_dcs(s, s->dis_dtc_frame, s->dis_dtc_len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int build_dis_or_dtc(t30_state_t *s)
{
    /* Build a skeleton for the DIS and DTC messages. This will be edited for
       the dynamically changing capabilities (e.g. can receive) just before
       it is sent. It might also be edited if the application changes our
       capabilities (e.g. disabling fine mode). Right now we set up all the
       unchanging stuff about what we are capable of doing. */
    s->dis_dtc_frame[0] = 0xFF;
    s->dis_dtc_frame[1] = 0x13;
    s->dis_dtc_frame[2] = (uint8_t) (T30_DIS | s->dis_received);
    /* No T.37; no T.38; no 3G mobile; no V.8; 256 octets per ECM frame preferred */
    s->dis_dtc_frame[3] = 0x00;
#if defined(ENABLE_V17)
    /* V.17, V.27ter and V.29 capable */
    s->dis_dtc_frame[4] = DISBIT6 | DISBIT4 | DISBIT3;
#else
    /* V.27ter and V.29 capable */
    s->dis_dtc_frame[4] = DISBIT4 | DISBIT3;
#endif
    /* 2D; fine capable */
    s->dis_dtc_frame[4] |= DISBIT8 | DISBIT7;
    /* 215mm wide; unlimited length; no scan-line padding required. */
    s->dis_dtc_frame[5] = DISBIT8 | DISBIT7 | DISBIT6 | DISBIT5 | DISBIT3;
    /* No uncompressed; no ECM; no T.6 */
    s->dis_dtc_frame[6] = DISBIT8;
    /* No FNV; no multiple selective polling; no polled subaddress; no T.43; no plane
       interleave; no G.726; no extended voice coding */
    s->dis_dtc_frame[7] = DISBIT8;
    /* Super-fine; no 300x300; no 400x400; metric; */ 
    s->dis_dtc_frame[8] = DISBIT8 | DISBIT1;
    /* No sub-addressing; no password; no data file (polling); no BFT; no DTM; no EDI */
    s->dis_dtc_frame[9] = DISBIT8;
    /* No BTM; no mixed mode (polling); no character mode; no mixed mode */
    s->dis_dtc_frame[10] = DISBIT8;
    /* No mode 26; no digital network capable; no JPEG; no full colour; no 12bits/pel */
    s->dis_dtc_frame[11] = DISBIT8;
    /* No sub-sampling;
       North American Letter (215.9mm x 279.4mm);
       North American Legal (215.9mm x 355.6mm);
       No T.85 basic;
       No T.85 optional. */
    s->dis_dtc_frame[12] = DISBIT4 | DISBIT5;
    s->dis_dtc_len = 13;
    t30_decode_dis_dtc_dcs(s, s->dis_dtc_frame, s->dis_dtc_len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int build_dcs(t30_state_t *s, const uint8_t *dis_dtc_frame, int len)
{
    static const uint8_t translate_min_scan_time[3][8] =
    {
        {0, 1, 2, 0, 4, 4, 2, 7}, /* normal */
        {0, 1, 2, 2, 4, 0, 2, 7}, /* fine */
        {0, 1, 2, 2, 4, 0, 2, 7}  /* super-fine */
    };
    static const int scanbitstab[4][8] =
    {
        /* Translate the minimum scan time to a minimum number of transmitted bits
           at the various bit rates. */
        {144,  36,  72, -1,  288, -1, -1, 0},
        {192,  48,  96, -1,  384, -1, -1, 0},
        {288,  72, 144, -1,  576, -1, -1, 0},
        {576, 144, 288, -1, 1152, -1, -1, 0}
    };
    uint8_t min_bits_field;
    
    /* Make a DCS frame from a received DIS frame. Negotiate the result
       based on what both parties can do. */
    s->dcs_frame[0] = 0xFF;
    s->dcs_frame[1] = 0x13;
    s->dcs_frame[2] = (uint8_t) (T30_DCS | s->dis_received);
    /* 256 octets per ECM frame */
    s->dcs_frame[3] = 0x00;
    switch (dis_dtc_frame[4] & (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3))
    {
    case 0:
        s->current_fallback = T30_V27TER_FALLBACK_START + 1;
        break;
    case DISBIT4:
        s->current_fallback = T30_V27TER_FALLBACK_START;
        break;
    case DISBIT3:
        /* TODO: this doesn't allow for skipping the V.27ter modes */
        s->current_fallback = T30_V29_FALLBACK_START;
        break;
    case (DISBIT4 | DISBIT3):
        s->current_fallback = T30_V29_FALLBACK_START;
        break;
    case (DISBIT6 | DISBIT4 | DISBIT3):
#if defined(ENABLE_V17)
        s->current_fallback = T30_V17_FALLBACK_START;
#else
        s->current_fallback = T30_V29_FALLBACK_START;
#endif
        break;
    default:
        span_log(&s->logging, SPAN_LOG_FLOW, "Remote does not support a compatible modem\n");
        /* We cannot talk to this machine! */
        return -1;
    }

    /* Set to required modem rate; standard resolution */
    s->dcs_frame[4] = fallback_sequence[s->current_fallback].dcs_code;
    /* Set the minimum scan time, in bits at the chosen bit rate */
    min_bits_field = (dis_dtc_frame[5] >> 4) & 7;
    s->min_row_bits = scanbitstab[fallback_sequence[s->current_fallback].min_scan_class][min_bits_field];
    /* If remote supports 2D compression, use it. */
    if ((dis_dtc_frame[4] & DISBIT8))
    {
        s->line_encoding = T4_COMPRESSION_ITU_T4_2D;
        s->dcs_frame[4] |= DISBIT8;
    }
    else
    {
        s->line_encoding = T4_COMPRESSION_ITU_T4_1D;
    }
    /* If we have a file to send, tell the far end to go into receive mode. */
    if (s->tx_file[0])
        s->dcs_frame[4] |= DISBIT2;
    /* Set the minimum scan time bits */
    switch (s->y_resolution)
    {
    case T4_Y_RESOLUTION_SUPERFINE:
        if ((dis_dtc_frame[8] & DISBIT1))
        {
            s->dcs_frame[8] |= DISBIT1;
            s->dcs_frame[5] = (translate_min_scan_time[2][min_bits_field] + 8) << 4;
            break;
        }
        /* Fall back */
        s->y_resolution = T4_Y_RESOLUTION_FINE;
        span_log(&s->logging, SPAN_LOG_FLOW, "Remote FAX does not support super-fine resolution.\n");
        /* Fall through */
    case T4_Y_RESOLUTION_FINE:
        if ((dis_dtc_frame[4] & DISBIT7))
        {
            s->dcs_frame[4] |= DISBIT7;
            s->dcs_frame[5] = (translate_min_scan_time[1][min_bits_field] + 8) << 4;
            break;
        }
        /* Fall back */
        s->y_resolution = T4_Y_RESOLUTION_STANDARD;
        span_log(&s->logging, SPAN_LOG_FLOW, "Remote FAX does not support fine resolution.\n");
        /* Fall through */
    case T4_Y_RESOLUTION_STANDARD:
        s->dcs_frame[5] = translate_min_scan_time[0][min_bits_field] << 4;
        break;
    }
    switch (s->image_width)
    {
    case 2432:
        s->dcs_frame[5] |= DISBIT2;
        break;
    case 2048:
        s->dcs_frame[5] |= DISBIT1;
        break;
    case 1728:
    default:
        break;
    }

    s->dcs_frame[5] |= DISBIT8;
    s->dcs_frame[6] = DISBIT8;
    s->dcs_frame[7] = DISBIT8;
    s->dcs_frame[8] |= DISBIT8;
    s->dcs_frame[9] = 0;
    s->dcs_len = 10;
    t30_decode_dis_dtc_dcs(s, s->dcs_frame, s->dcs_len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int check_dcs(t30_state_t *s, const uint8_t *dcs_frame, int len)
{
    static const int widths[3][4] =
    {
        { 864, 1024, 1216, -1},
        {1728, 2048, 2432, -1},
        {3456, 4096, 4864, -1}
    };
    int speed;
    int i;

    /* Check DCS frame from remote */
    t30_decode_dis_dtc_dcs(s, dcs_frame, len);
    if (len < 6)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Short DCS frame\n");
        return -1;
    }
    s->octets_per_ecm_frame = (dcs_frame[6] & DISBIT4)  ?  256  :  64;
    if (len >= 9  &&  (dcs_frame[8] & DISBIT1))
        s->y_resolution = T4_Y_RESOLUTION_SUPERFINE;
    else if (dcs_frame[4] & DISBIT7)
        s->y_resolution = T4_Y_RESOLUTION_FINE;
    else
        s->y_resolution = T4_Y_RESOLUTION_STANDARD;
    s->image_width = widths[1][dcs_frame[5] & (DISBIT2 | DISBIT1)];
    s->line_encoding = (dcs_frame[4] & DISBIT8)  ?  T4_COMPRESSION_ITU_T4_2D  :  T4_COMPRESSION_ITU_T4_1D;
    if (!(dcs_frame[4] & DISBIT2))
        span_log(&s->logging, SPAN_LOG_FLOW, "Remote cannot receive\n");

    s->error_correcting = (dcs_frame[6] & DISBIT3);
    speed = dcs_frame[4] & (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3);
    for (i = 0;  fallback_sequence[i].bit_rate;  i++)
    {
        if (fallback_sequence[i].dcs_code == speed)
            break;
    }
    if (fallback_sequence[i].bit_rate == 0)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Remote asked for a modem standard we do not support\n");
        return -1;
    }
    s->current_fallback = i;
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void send_dcn(t30_state_t *s)
{
    queue_phase(s, T30_PHASE_D_TX);
    set_state(s, T30_STATE_C);
    send_simple_frame(s, T30_DCN);
}
/*- End of function --------------------------------------------------------*/

static void disconnect(t30_state_t *s)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "Disconnecting\n");
    /* Make sure any FAX in progress is tidied up. If the tidying up has
       already happened, repeating it here is harmless. */
    t4_rx_end(&(s->t4));
    t4_tx_end(&(s->t4));
    s->timer_t0_t1 = 0;
    s->timer_t2_t4 = 0;
    s->timer_t3 = 0;
    s->timer_t5 = 0;
    set_phase(s, T30_PHASE_E);
    set_state(s, T30_STATE_B);
}
/*- End of function --------------------------------------------------------*/

static int start_sending_document(t30_state_t *s)
{
    if (s->tx_file[0] == '\0')
    {
        /* There is nothing to send */
        span_log(&s->logging, SPAN_LOG_FLOW, "No document to send\n");
        return  FALSE;
    }
    span_log(&s->logging, SPAN_LOG_FLOW, "Start sending document\n");
    if (t4_tx_init(&(s->t4), s->tx_file, s->tx_start_page, s->tx_stop_page))
    {
        span_log(&s->logging, SPAN_LOG_WARNING, "Cannot open source TIFF file '%s'\n", s->tx_file);
        s->current_status = T30_ERR_FILEERROR;
        return  FALSE;
    }
    t4_tx_set_tx_encoding(&(s->t4), s->line_encoding);
    t4_tx_set_min_row_bits(&(s->t4), s->min_row_bits);
    s->y_resolution = t4_tx_get_y_resolution(&(s->t4));
    switch (s->y_resolution)
    {
    case T4_Y_RESOLUTION_STANDARD:
        s->dcs_frame[4] &= ~DISBIT7;
        s->dcs_frame[8] &= ~DISBIT1;
        break;
    case T4_Y_RESOLUTION_FINE:
        s->dcs_frame[4] |= DISBIT7;
        s->dcs_frame[8] &= ~DISBIT1;
        break;
    case T4_Y_RESOLUTION_SUPERFINE:
        s->dcs_frame[4] &= ~DISBIT7;
        s->dcs_frame[8] |= DISBIT1;
        break;
    }
    /* Schedule training after the messages */
    set_state(s, T30_STATE_D);
    send_ident_frame(s, T30_TSI);
    send_frame(s, s->dcs_frame, s->dcs_len);
    s->retries = 0;
    return  TRUE;
}
/*- End of function --------------------------------------------------------*/

static int start_receiving_document(t30_state_t *s)
{
    if (s->rx_file[0] == '\0')
    {
        /* There is nothing to receive to */
        span_log(&s->logging, SPAN_LOG_FLOW, "No document to receive\n");
        return  FALSE;
    }
    span_log(&s->logging, SPAN_LOG_FLOW, "Start receiving document\n");
    queue_phase(s, T30_PHASE_B_TX);
    set_state(s, T30_STATE_R);
    s->dis_received = FALSE;
    send_nsf_frame(s);
    send_ident_frame(s, T30_CSI);
    set_dis_or_dtc(s);
    send_frame(s, s->dis_dtc_frame, s->dis_dtc_len);
    return  TRUE;
}
/*- End of function --------------------------------------------------------*/

static void unexpected_frame(t30_state_t *s, const uint8_t *msg, int len)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "Unexpected %s received in state %d\n", t30_frametype(msg[2]), s->state);
    if (s->state == T30_STATE_F_DOC)
        s->current_status = T30_ERR_INVALCMDRX;
}
/*- End of function --------------------------------------------------------*/

static void unexpected_final_frame(t30_state_t *s, const uint8_t *msg, int len)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "Unexpected %s received in state %d\n", t30_frametype(msg[2]), s->state);
    s->current_status = T30_ERR_UNEXPECTED;
    send_dcn(s);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_dis_or_dtc(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Digital identification signal or digital transmit command */
    s->dis_received = TRUE;
    switch (s->state)
    {
    case T30_STATE_D_POST_TCF:
        /* It appears they didn't see what we sent - retry the TCF */
        if (++s->retries > MAX_MESSAGE_TRIES)
        {
            s->current_status = T30_ERR_RETRYDCN;
            send_dcn(s);
            break;
        }
        queue_phase(s, T30_PHASE_B_TX);
        /* Schedule training after the messages */
        set_state(s, T30_STATE_D);
        send_ident_frame(s, T30_TSI);
        send_frame(s, s->dcs_frame, s->dcs_len);
        break;
    case T30_STATE_R:
    case T30_STATE_T:
    case T30_STATE_F_DOC:
        t30_decode_dis_dtc_dcs(s, msg, len);
        if (s->phase_b_handler)
            s->phase_b_handler(s, s->phase_d_user_data, msg[2]);
        queue_phase(s, T30_PHASE_B_TX);
        /* Try to send something */
        if (s->tx_file[0])
        {
            if (!(msg[4] & DISBIT2))
            {
                span_log(&s->logging, SPAN_LOG_FLOW, "%s far end cannot receive\n", t30_frametype(msg[2]));
                s->current_status = T30_ERR_NOTRXCAPABLE;
                send_dcn(s);
                break;
            }
            if (build_dcs(s, msg, len))
            {
                s->current_status = T30_ERR_INCOMPATIBLE;
                send_dcn(s);
                break;
            }
            if (!start_sending_document(s))
            {
                send_dcn(s);
                break;
            }
            break;
        }
        span_log(&s->logging, SPAN_LOG_FLOW, "%s nothing to send\n", "", t30_frametype(msg[2]));
        /* ... then try to receive something */
        if (s->rx_file[0])
        {
            if (!(msg[4] & DISBIT1))
            {
                span_log(&s->logging, SPAN_LOG_FLOW, "%s far end cannot transmit\n", t30_frametype(msg[2]));
                s->current_status = T30_ERR_NOTTXCAPABLE;
                send_dcn(s);
                break;
            }
            if (set_dis_or_dtc(s))
            {
                s->current_status = T30_ERR_INCOMPATIBLE;
                send_dcn(s);
                break;
            }
            if (!start_receiving_document(s))
            {
                send_dcn(s);
                break;
            }
            break;
        }
        span_log(&s->logging, SPAN_LOG_FLOW, "%s nothing to receive\n", t30_frametype(msg[2]));
        /* There is nothing to do, or nothing we are able to do. */
        send_dcn(s);
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_dcs(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Digital command signal */
    switch (s->state)
    {
    case T30_STATE_R:
    case T30_STATE_F_DOC:
        /* (TSI) DCS */
        /* (PWD) (SUB) (TSI) DCS */
        check_dcs(s, msg, len);
        if (s->phase_b_handler)
            s->phase_b_handler(s, s->phase_d_user_data, T30_DCS);
        span_log(&s->logging,
                SPAN_LOG_FLOW, 
                "Get document at %dbps, modem %d\n",
                fallback_sequence[s->current_fallback].bit_rate,
                fallback_sequence[s->current_fallback].modem_type);
        set_state(s, T30_STATE_F_TCF);
        set_phase(s, T30_PHASE_C_RX);
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_cfr(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Confirmation to receive */
    switch (s->state)
    {
    case T30_STATE_D_POST_TCF:
        /* Trainability test succeeded. Send the document. */
        span_log(&s->logging, SPAN_LOG_FLOW, "Trainability test succeeded\n");
        s->retries = 0;
        /* Send the next page. This is usually the first page, but after a retrain it
           might not be. */
        t4_tx_set_local_ident(&(s->t4), s->local_ident);

        t4_tx_start_page(&(s->t4));
        if (s->error_correcting_mode)
            set_state(s, T30_STATE_IV);
        else
            set_state(s, T30_STATE_I);
        s->short_train = TRUE;
        queue_phase(s, T30_PHASE_C_TX);
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_ftt(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Failure to train */
    switch (s->state)
    {
    case T30_STATE_D_POST_TCF:
        /* Trainability test failed. Try again. */
        span_log(&s->logging, SPAN_LOG_FLOW, "Trainability test failed\n");
        s->retries = 0;
        s->short_train = FALSE;
        if (fallback_sequence[++s->current_fallback].bit_rate == 0)
        {
            /* Give up */
            s->current_fallback = 0;
            s->current_status = T30_ERR_CANNOTTRAIN;
            send_dcn(s);
            break;
        }
        /* TODO: Renegotiate for a different speed */
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_eom(t30_state_t *s, const uint8_t *msg, int len)
{
    /* End of message */
    switch (s->state)
    {
    case T30_STATE_F_POST_DOC:
        /* Return to phase B */
        queue_phase(s, T30_PHASE_B_TX);
        switch (copy_quality(s))
        {
        case T30_COPY_QUALITY_GOOD:
            t4_rx_end_page(&(s->t4));
            rx_start_page(s);
            set_state(s, T30_STATE_R);
            send_simple_frame(s, T30_MCF);
            break;
        case T30_COPY_QUALITY_POOR:
            t4_rx_end_page(&(s->t4));
            rx_start_page(s);
            set_state(s, T30_STATE_R);
            send_simple_frame(s, T30_RTP);
            break;
        case T30_COPY_QUALITY_BAD:
            rx_start_page(s);
            set_state(s, T30_STATE_R);
            send_simple_frame(s, T30_RTN);
            break;
        }
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_EOM);
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_mps(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Multi-page signal */
    switch (s->state)
    {
    case T30_STATE_F_POST_DOC:
        /* Return to phase C */
        queue_phase(s, T30_PHASE_D_TX);
        switch (copy_quality(s))
        {
        case T30_COPY_QUALITY_GOOD:
            t4_rx_end_page(&(s->t4));
            rx_start_page(s);
            set_state(s, T30_STATE_III_MPS_MCF);
            send_simple_frame(s, T30_MCF);
            break;
        case T30_COPY_QUALITY_POOR:
            t4_rx_end_page(&(s->t4));
            rx_start_page(s);
            set_state(s, T30_STATE_III_MPS_RTP);
            send_simple_frame(s, T30_RTP);
            break;
        case T30_COPY_QUALITY_BAD:
            rx_start_page(s);
            set_state(s, T30_STATE_III_MPS_RTN);
            send_simple_frame(s, T30_RTN);
            break;
        }
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_MPS);
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_eop(t30_state_t *s, const uint8_t *msg, int len)
{
    /* End of procedure */
    switch (s->state)
    {
    case T30_STATE_F_POST_DOC:
        queue_phase(s, T30_PHASE_D_TX);
        switch (copy_quality(s))
        {
        case T30_COPY_QUALITY_GOOD:
            t4_rx_end_page(&(s->t4));
            t4_rx_end(&(s->t4));
            s->in_message = FALSE;
            set_state(s, T30_STATE_III_EOP_MCF);
            send_simple_frame(s, T30_MCF);
            break;
        case T30_COPY_QUALITY_POOR:
            t4_rx_end_page(&(s->t4));
            t4_rx_end(&(s->t4));
            s->in_message = FALSE;
            set_state(s, T30_STATE_III_EOP_RTP);
            send_simple_frame(s, T30_RTP);
            break;
        case T30_COPY_QUALITY_BAD:
            set_state(s, T30_STATE_III_EOP_RTN);
            send_simple_frame(s, T30_RTN);
            break;
        }
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_EOP);
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_pri_eom(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Procedure interrupt - end of procedure */
    switch (s->state)
    {
    case T30_STATE_F_POST_DOC:
        switch (copy_quality(s))
        {
        case T30_COPY_QUALITY_GOOD:
            t4_rx_end_page(&(s->t4));
            t4_rx_end(&(s->t4));
            s->in_message = FALSE;
            set_state(s, T30_STATE_III_EOP_MCF);
            break;
        case T30_COPY_QUALITY_POOR:
            t4_rx_end_page(&(s->t4));
            t4_rx_end(&(s->t4));
            s->in_message = FALSE;
            set_state(s, T30_STATE_III_EOP_RTP);
            break;
        case T30_COPY_QUALITY_BAD:
            set_state(s, T30_STATE_III_EOP_RTN);
            break;
        }
        if (s->phase_d_handler)
        {
            s->phase_d_handler(s, s->phase_d_user_data, T30_PRI_EOM);
            s->timer_t3 = ms_to_samples(DEFAULT_TIMER_T3);
        }
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_pri_mps(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Procedure interrupt - multipage signal */
    switch (s->state)
    {
    case T30_STATE_F_POST_DOC:
        switch (copy_quality(s))
        {
        case T30_COPY_QUALITY_GOOD:
            t4_rx_end_page(&(s->t4));
            t4_rx_end(&(s->t4));
            s->in_message = FALSE;
            set_state(s, T30_STATE_III_EOP_MCF);
            break;
        case T30_COPY_QUALITY_POOR:
            t4_rx_end_page(&(s->t4));
            t4_rx_end(&(s->t4));
            s->in_message = FALSE;
            set_state(s, T30_STATE_III_EOP_RTP);
            break;
        case T30_COPY_QUALITY_BAD:
            set_state(s, T30_STATE_III_EOP_RTN);
            break;
        }
        if (s->phase_d_handler)
        {
            s->phase_d_handler(s, s->phase_d_user_data, T30_PRI_MPS);
            s->timer_t3 = ms_to_samples(DEFAULT_TIMER_T3);
        }
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_pri_eop(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Procedure interrupt - end of procedure */
    switch (s->state)
    {
    case T30_STATE_F_POST_DOC:
    case T30_STATE_VII:
        switch (copy_quality(s))
        {
        case T30_COPY_QUALITY_GOOD:
            t4_rx_end_page(&(s->t4));
            t4_rx_end(&(s->t4));
            s->in_message = FALSE;
            set_state(s, T30_STATE_III_EOP_MCF);
            break;
        case T30_COPY_QUALITY_POOR:
            t4_rx_end_page(&(s->t4));
            t4_rx_end(&(s->t4));
            s->in_message = FALSE;
            set_state(s, T30_STATE_III_EOP_RTP);
            break;
        case T30_COPY_QUALITY_BAD:
            set_state(s, T30_STATE_III_EOP_RTN);
            break;
        }
        if (s->phase_d_handler)
        {
            s->phase_d_handler(s, s->phase_d_user_data, T30_PRI_EOP);
            s->timer_t3 = ms_to_samples(DEFAULT_TIMER_T3);
        }
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_nss(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Non-standard facilities set-up */
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_ctc(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Continue to correct */
    switch (s->state)
    {
    case T30_STATE_VII:
        /* TODO: */
        set_state(s, T30_STATE_F_DOC);
        send_simple_frame(s, T30_CTR);
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_ctr(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Response for continue to correct */
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_err(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Response for end of retransmission */
    switch (s->state)
    {
    case T30_STATE_VIA:
        /* TODO: */
        break;
    case T30_STATE_VIB:
        /* TODO: */
        break;
    case T30_STATE_VIC:
        /* TODO: */
        break;
    case T30_STATE_VID:
        /* TODO: */
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_rr(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Receiver ready */
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_ppr(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Partial page request */
    switch (s->state)
    {
    case T30_STATE_VA:
        /* TODO: */
        break;
    case T30_STATE_VB:
        /* TODO: */
        break;
    case T30_STATE_VC:
        if (++s->ppr_count >= 4)
        {
            /* Continue to correct? */
            /* TODO: */
        }
        else
        {
            /* TODO: */
            /* Check which frames are OK */
        }
        break;
    case T30_STATE_VD:
        /* TODO: */
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_rnr(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Receive not ready */
    s->ppr_count = 0;
    switch (s->state)
    {
    case T30_STATE_VA:
        /* TODO: */
        break;
    case T30_STATE_VB:
        /* TODO: */
        break;
    case T30_STATE_VC:
        /* TODO: */
        break;
    case T30_STATE_VD:
        /* TODO: */
        break;
    case T30_STATE_VIA:
        /* TODO: */
        break;
    case T30_STATE_VIB:
        /* TODO: */
        break;
    case T30_STATE_VIC:
        /* TODO: */
        break;
    case T30_STATE_VID:
        /* TODO: */
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_eos(t30_state_t *s, const uint8_t *msg, int len)
{
    /* End of selection */
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_tr(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Transmit ready */
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_fcd(t30_state_t *s, const uint8_t *msg, int len)
{
    int frame_no;

    /* Facsimile coded data */
    if (len == 4 + 256  ||  len == 4 + 64)
    {
        frame_no = msg[3];
        memcpy(s->ecm_data[frame_no], msg + 4, len - 4);
        s->ecm_status[frame_no] = (uint8_t) (len - 4);
    }
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_rcp(t30_state_t *s, const uint8_t *msg, int len)
{
    int i;
    int j;
    int frame_no;
    uint8_t frame[3 + 32];

    /* Return to control for partial page */
    /* Check which frames we now have stored OK */
    frame[0] = 0xFF;
    frame[1] = 0x13;
    frame[2] = T30_PPR;
    frame_no = 0;
    for (i = 3;  i < 3 + 32;  i++)
    {
        frame[i] = 0;
        for (j = 0;  j < 8;  j++)
        {
            if (s->ecm_status[frame_no++] == 0)
                frame[i] |= (1 << j);
        }
    }
    send_frame(s, frame, 3 + 32);
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_fnv(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Field not valid */
    /* TODO: analyse the message, as per 5.3.6.2.13 */
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_pps(t30_state_t *s, const uint8_t *msg, int len)
{
    int fcf2;
    int page;
    int block;
    int frames;
    
    /* Partial page signal */
    if (len >= 7)
    {
        fcf2 = msg[3];
        page = msg[4];
        block = msg[5];
        frames = msg[6];
    }
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_tnr(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Transmit not ready */
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_eor(t30_state_t *s, const uint8_t *msg, int len)
{
    /* End of retransmission */
    /* Partial page signal */
    switch (s->state)
    {
    case T30_STATE_IX:
        if (len > 3)
        {
            switch (msg[3])
            {
            case T30_NULL:
                break;
            case T30_EOM:
                break;
            case T30_MPS:
                break;
            case T30_EOP:
                break;
            case T30_PRI_EOM:
                break;
            case T30_PRI_MPS:
                break;
            case T30_PRI_EOP:
                break;
            default:
                unexpected_final_frame(s, msg, len);
                break;
            }
        }
        else
        {
            unexpected_final_frame(s, msg, len);
        }
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_fdm(t30_state_t *s, const uint8_t *msg, int len)
{
    /* File diagnostics message */
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_mcf(t30_state_t *s, const uint8_t *msg, int len)
{
    t4_stats_t stats;

    /* Message confirmation */
    switch (s->state)
    {
    case T30_STATE_II_MPS:
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_MCF);
        set_state(s, T30_STATE_I);
        queue_phase(s, T30_PHASE_C_TX);
        break;
    case T30_STATE_II_EOM:
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_MCF);
        set_state(s, T30_STATE_R);
        if (span_log_test(&s->logging, SPAN_LOG_FLOW))
        {
            t4_get_transfer_statistics(&(s->t4), &stats);
            span_log(&s->logging, SPAN_LOG_FLOW, "Success - delivered %d pages\n", stats.pages_transferred);
        }
        break;
    case T30_STATE_II_EOP:
        t4_tx_end(&(s->t4));
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_MCF);
        send_dcn(s);
        if (span_log_test(&s->logging, SPAN_LOG_FLOW))
        {
            t4_get_transfer_statistics(&(s->t4), &stats);
            span_log(&s->logging, SPAN_LOG_FLOW, "Success - delivered %d pages\n", stats.pages_transferred);
        }
        break;
    case T30_STATE_VA:
        /* TODO: */
        break;
    case T30_STATE_VB:
        /* TODO: */
        break;
    case T30_STATE_VC:
        send_dcn(s);
        break;
    case T30_STATE_VD:
        /* TODO: */
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_rtp(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Retrain positive */
    s->short_train = FALSE;
    switch (s->state)
    {
    case T30_STATE_II_MPS:
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_RTP);
        /* Send fresh training, and then the next page */
        set_state(s, T30_STATE_I);
        queue_phase(s, T30_PHASE_C_TX);
        break;
    case T30_STATE_II_EOM:
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_RTP);
        /* TODO: should go back to T, and resend */
        set_state(s, T30_STATE_R);
        break;
    case T30_STATE_II_EOP:
        t4_tx_end(&(s->t4));
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_RTP);
        /* Send fresh training, and then the next page */
        set_state(s, T30_STATE_I);
        queue_phase(s, T30_PHASE_C_TX);
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_rtn(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Retrain negative */
    s->short_train = FALSE;
    switch (s->state)
    {
    case T30_STATE_II_MPS:
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_RTN);
        /* Send fresh training, and then repeat the last page */
        set_state(s, T30_STATE_I);
        queue_phase(s, T30_PHASE_C_TX);
        break;
    case T30_STATE_II_EOM:
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_RTN);
        /* TODO: */
        send_dcn(s);
        break;
    case T30_STATE_II_EOP:
        t4_tx_end(&(s->t4));
        if (s->phase_d_handler)
            s->phase_d_handler(s, s->phase_d_user_data, T30_RTN);
        /* Send fresh training, and then repeat the last page */
        set_state(s, T30_STATE_I);
        queue_phase(s, T30_PHASE_C_TX);
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_pip(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Procedure interrupt positive */
    switch (s->state)
    {
    case T30_STATE_II_MPS:
    case T30_STATE_II_EOM:
    case T30_STATE_II_EOP:
        if (s->phase_d_handler)
        {
            s->phase_d_handler(s, s->phase_d_user_data, T30_PIP);
            s->timer_t3 = ms_to_samples(DEFAULT_TIMER_T3);
        }
        break;
    case T30_STATE_VA:
        /* TODO: */
        break;
    case T30_STATE_VB:
        /* TODO: */
        break;
    case T30_STATE_VC:
        /* TODO: */
        break;
    case T30_STATE_VD:
        /* TODO: */
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_pin(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Procedure interrupt negative */
    switch (s->state)
    {
    case T30_STATE_II_MPS:
    case T30_STATE_II_EOM:
    case T30_STATE_II_EOP:
        if (s->phase_d_handler)
        {
            s->phase_d_handler(s, s->phase_d_user_data, T30_PIN);
            s->timer_t3 = ms_to_samples(DEFAULT_TIMER_T3);
        }
        break;
    case T30_STATE_VA:
        /* TODO: */
        break;
    case T30_STATE_VB:
        /* TODO: */
        break;
    case T30_STATE_VC:
        /* TODO: */
        break;
    case T30_STATE_VD:
        /* TODO: */
        break;
    case T30_STATE_VIB:
        /* TODO: */
        break;
    case T30_STATE_VIC:
        /* TODO: */
        break;
    case T30_STATE_VID:
        /* TODO: */
        break;
    default:
        unexpected_final_frame(s, msg, len);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void process_rx_dcn(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Disconnect */
#if 0
    /* TODO: test if this is expected or unexpected */
    switch (s->state)
    {
    case ??????:
        /* Unexpected DCN while waiting for image data */
        s->current_status = T30_ERR_DCNDATARX;
        break;
    case ??????:
        /* Unexpected DCN while waiting for EOM, EOP or MPS */
        s->current_status = T30_ERR_DCNFAXRX;
        break;
    case ??????:
        /* Unexpected DCN after EOM or MPS sequence */
        s->current_status = T30_ERR_DCNPHDRX;
        break;
    case ??????:
        /* Unexpected DCN after RR/RNR sequence */
        s->current_status = T30_ERR_DCNRRDRX;
        break;
    case ??????:
        /* Unexpected DCN after requested retransmission */
        s->current_status = T30_ERR_DCNNORTNRX;
        break;
    }
#endif
    /* Time to disconnect */
    disconnect(s);
}
/*- End of function --------------------------------------------------------*/

static void process_rx_crp(t30_state_t *s, const uint8_t *msg, int len)
{
    /* Command repeat */
    if (s->crp_enabled)
    {
        /* TODO: */
        return;
    }
    unexpected_final_frame(s, msg, len);
}
/*- End of function --------------------------------------------------------*/

void t30_hdlc_accept(void *user_data, int ok, const uint8_t *msg, int len)
{
    t30_state_t *s;
    int final_frame;
    
    s = (t30_state_t *) user_data;

    if (len < 0)
    {
        /* Special conditions */
        switch (len)
        {
        case PUTBIT_TRAINING_FAILED:
            span_log(&s->logging, SPAN_LOG_FLOW, "HDLC carrier training failed\n");
            break;
        case PUTBIT_TRAINING_SUCCEEDED:
            /* The modem is now trained */
            span_log(&s->logging, SPAN_LOG_FLOW, "HDLC carrier trained\n");
            s->rx_signal_present = TRUE;
            break;
        case PUTBIT_CARRIER_UP:
            span_log(&s->logging, SPAN_LOG_FLOW, "HDLC carrier up\n");
            s->rx_signal_present = TRUE;
            break;
        case PUTBIT_CARRIER_DOWN:
            span_log(&s->logging, SPAN_LOG_FLOW, "HDLC carrier down\n");
            /* If a phase change has been queued to occur after the receive signal drops,
               its time to change. */
            if (s->next_phase != T30_PHASE_IDLE)
            {
                set_phase(s, s->next_phase);
                s->next_phase = T30_PHASE_IDLE;
            }
            s->rx_signal_present = FALSE;
            break;
        case PUTBIT_FRAMING_OK:
            span_log(&s->logging, SPAN_LOG_FLOW, "HDLC framing OK\n");
            if (!s->far_end_detected  &&  s->timer_t0_t1 > 0)
            {
                s->timer_t0_t1 = ms_to_samples(DEFAULT_TIMER_T1);
                s->far_end_detected = TRUE;
                if (s->phase == T30_PHASE_A_CED  ||  s->phase == T30_PHASE_A_CNG)
                    set_phase(s, T30_PHASE_B_RX);
            }
            /* 5.4.3.1 Timer T2 is reset is flag is received */
            if (!s->timer_is_t4  &&  s->timer_t2_t4 > 0)
                s->timer_t2_t4 = 0;
            break;
        case PUTBIT_ABORT:
            /* Just ignore these */
            break;
        default:
            span_log(&s->logging, SPAN_LOG_FLOW, "Unexpected HDLC special length - %d!\n", len);
            break;
        }
        return;
    }

    /* The spec. says a command or response is not valid if:
        - any of the frames, optional or mandatory, have an FCS error.
        - any single frame exceeds 3s +- 15% (i.e. no frame should exceed 2.55s)
        - the final frame is not tagged as a final frame
        - the final frame is not a recognised one.
       The first point seems benign. If we accept an optional frame, and a later
       frame is bad, having accepted the optional frame should be harmless.
       The 2.55s maximum seems to limit signalling frames to no more than 95 octets,
       including FCS, and flag octets (assuming the use of V.21).
    */
    if (!ok)
    {
        if (s->crp_enabled)
            send_simple_frame(s, T30_CRP);
        return;
    }

    /* Cancel the command or response timer */
    s->timer_t2_t4 = 0;
    if (msg[0] != 0xFF  ||  !(msg[1] == 0x03  ||  msg[1] == 0x13))
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Bad HDLC frame header - %02x %02x\n", msg[0], msg[1]);
        return;
    }
    print_frame(s, "<<<", msg, len);

    final_frame = msg[1] & 0x10;
    switch (s->phase)
    {
    case T30_PHASE_A_CED:
    case T30_PHASE_A_CNG:
    case T30_PHASE_B_RX:
    case T30_PHASE_D_RX:
        break;
    default:
        span_log(&s->logging, SPAN_LOG_FLOW, "Unexpected HDLC frame received in phase %s, state %d\n", phase_names[s->phase], s->state);
        break;
    }

    if (!final_frame)
    {
        /* Restart the command or response timer, T2 or T4 */
        s->timer_t2_t4 = ms_to_samples((s->timer_is_t4)  ?  DEFAULT_TIMER_T4  :  DEFAULT_TIMER_T2);

        /* The following handles all the message types we expect to get without
           a final frame tag. If we get one that T.30 says we should not expect
           in a particular context, its pretty harmless, so don't worry. */
        switch (msg[2] & 0xFE)
        {
        case T30_CSI:
            if (msg[2] == T30_CSI)
            {
                /* OK in (NSF) (CSI) DIS */
                decode_20digit_msg(s, s->far_ident, &msg[2], len - 2);
            }
            else
            {
                /* OK in (NSC) (CIG) DTC */
                /* OK in (PWD) (SEP) (CIG) DTC */
                decode_20digit_msg(s, s->far_ident, &msg[2], len - 2);
            }
            break;
        case T30_NSF:
            if (msg[2] == T30_NSF)
            {
                /* OK in (NSF) (CSI) DIS */
                if (t35_decode(&msg[3], len - 3, &s->country, &s->vendor, &s->model))
                {
                    if (s->country)
                        span_log(&s->logging, SPAN_LOG_FLOW, "The remote was made in '%s'\n", s->country);
                    if (s->vendor)
                        span_log(&s->logging, SPAN_LOG_FLOW, "The remote was made by '%s'\n", s->vendor);
                    if (s->model)
                        span_log(&s->logging, SPAN_LOG_FLOW, "The remote is a '%s'\n", s->model);
                }
            }
            else
            {
                /* OK in (NSC) (CIG) DTC */
            }
            break;
        case T30_PWD:
            if (msg[2] == T30_PWD)
            {
                /* OK in (PWD) (SUB) (TSI) DCS */
                /* OK in (PWD) (SEP) (CIG) DTC */
                decode_password(s, s->password, &msg[2], len - 2);
            }
            else
            {
                unexpected_frame(s, msg, len);
            }
            break;
        case T30_SEP:
            if (msg[2] == T30_SEP)
            {
                /* OK in (PWD) (SEP) (CIG) DTC */
                decode_20digit_msg(s, NULL, &msg[2], len - 2);
            }
            else
            {
                unexpected_frame(s, msg, len);
            }
            break;
        case T30_PSA:
            if (msg[2] == T30_PSA)
                decode_20digit_msg(s, NULL, &msg[2], len - 2);
            else
                unexpected_frame(s, msg, len);
            break;
        case T30_CIA:
            if (msg[2] == T30_CIA)
                decode_url_msg(s, NULL, &msg[2], len - 2);
            else
                unexpected_frame(s, msg, len);
            break;
        case T30_ISP:
            if (msg[2] == T30_ISP)
                decode_url_msg(s, NULL, &msg[2], len - 2);
            else
                unexpected_frame(s, msg, len);
            break;
        case T30_TSI:
            /* OK in (TSI) DCS */
            /* OK in (PWD) (SUB) (TSI) DCS */
            decode_20digit_msg(s, s->far_ident, &msg[2], len - 2);
            break;
        case T30_SUB:
            /* OK in (PWD) (SUB) (TSI) DCS */
            decode_20digit_msg(s, s->sub_address, &msg[2], len - 2);
            break;
        case T30_SID:
            /* T.30 does not say where this is OK */
            decode_20digit_msg(s, NULL, &msg[2], len - 2);
            break;
        case T30_CSA:
            decode_url_msg(s, NULL, &msg[2], len - 2);
            break;
        case T30_TSA:
            decode_url_msg(s, NULL, &msg[2], len - 2);
            break;
        case T30_IRA:
            decode_url_msg(s, NULL, &msg[2], len - 2);
            break;
        default:
            span_log(&s->logging, SPAN_LOG_FLOW, "Unexpected %s frame\n", t30_frametype(msg[2]));
            break;
        }
    }
    else
    {
        /* Once we have any successful message from the far end, we
           cancel timer T1 */
        s->timer_t0_t1 = 0;

        /* The following handles context sensitive message types, which should
           occur at the end of message sequences. They should, therefore have
           the final frame flag set. */
        span_log(&s->logging, SPAN_LOG_FLOW, "In state %d\n", s->state);
        switch (msg[2] & 0xFE)
        {
        case T30_DIS:
            process_rx_dis_or_dtc(s, msg, len);
            break;
        case T30_DCS:
            process_rx_dcs(s, msg, len);
            break;
        case T30_NSS:
            process_rx_nss(s, msg, len);
            break;
        case T30_CTC:
            process_rx_ctc(s, msg, len);
            break;
        case T30_CFR:
            process_rx_cfr(s, msg, len);
            break;
        case T30_FTT:
            process_rx_ftt(s, msg, len);
            break;
        case T30_CTR:
            process_rx_ctr(s, msg, len);
            break;
        case T30_EOM:
            process_rx_eom(s, msg, len);
            break;
        case T30_MPS:
            process_rx_mps(s, msg, len);
            break;
        case T30_EOP:
            process_rx_eop(s, msg, len);
            break;
        case T30_PRI_EOM:
            process_rx_pri_eom(s, msg, len);
            break;
        case T30_PRI_MPS:
            process_rx_pri_mps(s, msg, len);
            break;
        case T30_PRI_EOP:
            process_rx_pri_eop(s, msg, len);
            break;
        case T30_EOS:
            process_rx_eos(s, msg, len);
            break;
        case T30_PPS:
            process_rx_pps(s, msg, len);
            break;
        case T30_EOR:
            process_rx_eor(s, msg, len);
            break;
        case T30_RR:
            process_rx_rr(s, msg, len);
            break;
        case T30_MCF:
            process_rx_mcf(s, msg, len);
            break;
        case T30_RTP:
            process_rx_rtp(s, msg, len);
            break;
        case T30_RTN:
            process_rx_rtn(s, msg, len);
            break;
        case T30_PIP:
            process_rx_pip(s, msg, len);
            break;
        case T30_PIN:
            process_rx_pin(s, msg, len);
            break;
        case T30_PPR:
            process_rx_ppr(s, msg, len);
            break;
        case T30_RNR:
            process_rx_rnr(s, msg, len);
            break;
        case T30_ERR:
            process_rx_err(s, msg, len);
            break;
        case T30_FDM:
            process_rx_fdm(s, msg, len);
            break;
        case T30_DCN:
            process_rx_dcn(s, msg, len);
            break;
        case T30_CRP:
            process_rx_crp(s, msg, len);
            break;
        case T30_FNV:
            process_rx_fnv(s, msg, len);
            break;
        case T30_TNR:
            process_rx_tnr(s, msg, len);
            break;
        case T30_TR:
            process_rx_tr(s, msg, len);
            break;
        case T4_FCD:
            process_rx_fcd(s, msg, len);
            break;
        case T4_RCP:
            process_rx_rcp(s, msg, len);
            break;
        default:
            /* We don't know what to do with this. */
            unexpected_final_frame(s, msg, len);
            break;
        }
    }
}
/*- End of function --------------------------------------------------------*/

static void queue_phase(t30_state_t *s, int phase)
{
    if (s->rx_signal_present)
    {
        /* We need to wait for that signal to go away */
        s->next_phase = phase;
    }
    else
    {
        set_phase(s, phase);
        s->next_phase = T30_PHASE_IDLE;
    }
}
/*- End of function --------------------------------------------------------*/

static void set_phase(t30_state_t *s, int phase)
{
    if (phase != s->phase)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Changing from phase %s to %s\n", phase_names[s->phase], phase_names[phase]);
        /* We may be killing a receiver before it has declared the end of the
           signal. Force the signal present indicator to off, because the
           receiver will never be able to. */
        if (s->phase != T30_PHASE_A_CED  &&  s->phase != T30_PHASE_A_CNG)
            s->rx_signal_present = FALSE;
        s->phase = phase;
        switch (phase)
        {
        case T30_PHASE_A_CED:
            if (s->set_rx_type_handler)
                s->set_rx_type_handler(s->set_rx_type_user_data, T30_MODEM_V21, FALSE);
            if (s->set_tx_type_handler)
                s->set_tx_type_handler(s->set_tx_type_user_data, T30_MODEM_CED, FALSE);
            break;
        case T30_PHASE_A_CNG:
            if (s->set_rx_type_handler)
                s->set_rx_type_handler(s->set_rx_type_user_data, T30_MODEM_V21, FALSE);
            if (s->set_tx_type_handler)
                s->set_tx_type_handler(s->set_tx_type_user_data, T30_MODEM_CNG, FALSE);
            break;
        case T30_PHASE_B_RX:
        case T30_PHASE_D_RX:
            if (s->set_rx_type_handler)
                s->set_rx_type_handler(s->set_rx_type_user_data, T30_MODEM_V21, FALSE);
            if (s->set_tx_type_handler)
                s->set_tx_type_handler(s->set_tx_type_user_data, T30_MODEM_NONE, FALSE);
            break;
        case T30_PHASE_B_TX:
        case T30_PHASE_D_TX:
            if (!s->far_end_detected  &&  s->timer_t0_t1 > 0)
            {
                s->timer_t0_t1 = ms_to_samples(DEFAULT_TIMER_T1);
                s->far_end_detected = TRUE;
            }
            if (s->set_rx_type_handler)
                s->set_rx_type_handler(s->set_rx_type_user_data, T30_MODEM_NONE, FALSE);
            if (s->set_tx_type_handler)
                s->set_tx_type_handler(s->set_tx_type_user_data, T30_MODEM_V21, FALSE);
            break;
        case T30_PHASE_C_RX:
            s->timer_t2_t4 = ms_to_samples(DEFAULT_TIMER_T2);
            s->timer_is_t4 = FALSE;
            if (s->set_rx_type_handler)
                s->set_rx_type_handler(s->set_rx_type_user_data, fallback_sequence[s->current_fallback].modem_type, s->short_train);
            if (s->set_tx_type_handler)
                s->set_tx_type_handler(s->set_tx_type_user_data, T30_MODEM_NONE, FALSE);
            break;
        case T30_PHASE_C_TX:
            /* Pause before switching from anything to phase C */
            s->training_test_bits = (3*fallback_sequence[s->current_fallback].bit_rate)/2;
            if (s->set_rx_type_handler)
                s->set_rx_type_handler(s->set_rx_type_user_data, T30_MODEM_NONE, FALSE);
            if (s->set_tx_type_handler)
                s->set_tx_type_handler(s->set_tx_type_user_data, fallback_sequence[s->current_fallback].modem_type, s->short_train);
            break;
        case T30_PHASE_E:
            /* Send a little silence before ending things, to ensure the
               buffers are all flushed through, and the far end has seen
               the last message we sent. */
            s->training_current_zeros = 0;
            s->training_most_zeros = 0;
            if (s->set_rx_type_handler)
                s->set_rx_type_handler(s->set_rx_type_user_data, T30_MODEM_NONE, FALSE);
            if (s->set_tx_type_handler)
                s->set_tx_type_handler(s->set_tx_type_user_data, T30_MODEM_PAUSE, 200);
            break;
        case T30_PHASE_CALL_FINISHED:
            if (s->set_rx_type_handler)
                s->set_rx_type_handler(s->set_rx_type_user_data, T30_MODEM_DONE, FALSE);
            if (s->set_tx_type_handler)
                s->set_tx_type_handler(s->set_tx_type_user_data, T30_MODEM_DONE, FALSE);
            break;
        }
    }
}
/*- End of function --------------------------------------------------------*/

static void set_state(t30_state_t *s, int state)
{
    if (s->state != state)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Changing from state %d to %d\n", s->state, state);
        s->state = state;
    }
}
/*- End of function --------------------------------------------------------*/

void t30_send_complete(void *user_data)
{
    t30_state_t *s;
    int more;
    
    s = (t30_state_t *) user_data;

    span_log(&s->logging, SPAN_LOG_FLOW, "Send complete in phase %s, state %d\n", phase_names[s->phase], s->state);
    /* We have finished sending our messages, so move on to the next operation. */
    switch (s->state)
    {
    case T30_STATE_ANSWERING:
        span_log(&s->logging, SPAN_LOG_FLOW, "Starting answer mode\n");
        set_phase(s, T30_PHASE_B_TX);
        s->timer_t2_t4 = ms_to_samples(DEFAULT_TIMER_T2);
        s->timer_is_t4 = FALSE;
        set_state(s, T30_STATE_R);
        s->dis_received = FALSE;
        send_nsf_frame(s);
        send_ident_frame(s, T30_CSI);
        set_dis_or_dtc(s);
        send_frame(s, s->dis_dtc_frame, s->dis_dtc_len);
        break;
    case T30_STATE_R:
        /* Wait for an acknowledgement. */
        set_phase(s, T30_PHASE_B_RX);
        s->timer_t2_t4 = ms_to_samples(DEFAULT_TIMER_T4);
        s->timer_is_t4 = TRUE;
        break;
    case T30_STATE_III_MPS_MCF:
    case T30_STATE_III_MPS_RTP:
    case T30_STATE_III_MPS_RTN:
    case T30_STATE_F_CFR:
        set_state(s, T30_STATE_F_DOC);
        set_phase(s, T30_PHASE_C_RX);
        break;
    case T30_STATE_II_MPS:
    case T30_STATE_II_EOM:
    case T30_STATE_II_EOP:
    case T30_STATE_III_EOP_MCF:
    case T30_STATE_III_EOP_RTP:
    case T30_STATE_III_EOP_RTN:
        /* We have finished sending the post image message. Wait for an
           acknowledgement. */
        set_phase(s, T30_PHASE_D_RX);
        s->timer_t2_t4 = ms_to_samples(DEFAULT_TIMER_T4);
        s->timer_is_t4 = TRUE;
        break;
    case T30_STATE_B:
        /* We have now allowed time for the last message to flush
           through the system, so it is safe to report the end of the
           call. */
        if (s->phase_e_handler)
            s->phase_e_handler(s, s->phase_e_user_data, s->current_status);
        set_phase(s, T30_PHASE_CALL_FINISHED);
        break;
    case T30_STATE_C:
        /* We just sent the disconnect message. Now it is time to disconnect */
        disconnect(s);
        break;
    case T30_STATE_D:
        /* Do the trainability test */
        set_state(s, T30_STATE_D_TCF);
        set_phase(s, T30_PHASE_C_TX);
        break;
    case T30_STATE_D_TCF:
        /* Finished sending training test. Listen for the response. */
        set_phase(s, T30_PHASE_B_RX);
        s->timer_t2_t4 = ms_to_samples(DEFAULT_TIMER_T4);
        s->timer_is_t4 = TRUE;
        set_state(s, T30_STATE_D_POST_TCF);
        break;
    case T30_STATE_I:
        /* Send the end of page message */
        set_phase(s, T30_PHASE_D_TX);
        t4_tx_end_page(&(s->t4));
        t4_tx_set_local_ident(&(s->t4), s->local_ident);
        /* We might need to resend the page we are on, but we need to check if there
           are any more pages to send, so we can send the correct signal */
        if (t4_tx_start_page(&(s->t4)) == 0)
        {
            set_state(s, T30_STATE_II_MPS);
            send_simple_frame(s, (s->local_interrupt_pending)  ?  T30_PRI_MPS  :  T30_MPS);
        }
        else
        {
            /* Call a user handler, if one is set, to check if another document is to be sent.
               If so, we send an EOM, rather than an EOP. Then we will renegotiate, and the new
               document will begin. */
            if (s->document_handler)
                more = s->document_handler(s, s->document_user_data, 0);
            else
                more = FALSE;
            if (more)
            {
                set_state(s, T30_STATE_II_EOM);
                send_simple_frame(s, (s->local_interrupt_pending)  ?  T30_PRI_EOM  :  T30_EOM);
            }
            else
            {
                set_state(s, T30_STATE_II_EOP);
                send_simple_frame(s, (s->local_interrupt_pending)  ?  T30_PRI_EOP  :  T30_EOP);
            }
        }
        break;
    default:
        span_log(&s->logging, SPAN_LOG_FLOW, "Bad state in t30_send_complete - %d\n", s->state);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void timer_t0_expired(t30_state_t *s)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "T0 timeout in state %d\n", s->state);
    s->current_status = T30_ERR_T0EXPIRED;
    /* Just end the call */
    disconnect(s);
}
/*- End of function --------------------------------------------------------*/

static void timer_t1_expired(t30_state_t *s)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "T1 timeout in state %d\n", s->state);
    /* The initial connection establishment has timeout out. In other words, we
       have been unable to communicate successfully with a remote machine.
       It is time to abandon the call. */
    s->current_status = T30_ERR_T1EXPIRED;
    switch (s->state)
    {
    case T30_STATE_T:
        /* Just end the call */
        disconnect(s);
        break;
    case T30_STATE_R:
        /* Send disconnect, and then end the call. Since we have not
           successfully contacted the far end, it is unclear why we should
           send a disconnect message at this point. However, it is what T.30
           says we should do. */
        send_dcn(s);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void timer_t2_expired(t30_state_t *s)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "T2 timeout in phase %s, state %d\n", phase_names[s->phase], s->state);
    switch (s->state)
    {
    case T30_STATE_F_DOC:
        /* Timer T2 expired while waiting for FAX page */
        s->current_status = T30_ERR_T2EXPFAXRX;
        break;
    case T30_STATE_F_POST_DOC:
        /* Timer T2 expired while waiting for next FAX page */
        s->current_status = T30_ERR_T2EXPMPSRX;
        break;
#if 0
    case ??????:
        /* Timer T2 expired while waiting for DCN */
        s->current_status = T30_ERR_T2EXPDCNRX;
        break;
    case ??????:
        /* Timer T2 expired while waiting for phase D */
        s->current_status = T30_ERR_T2EXPDRX;
        break;
    case ??????:
        /* Timer T2 expired while waiting for RR command */
        s->current_status = T30_ERR_T2EXPRRRX;
        break;
    case ??????:
        /* Timer T2 expired while waiting for NSS, DCS or MCF */
        s->current_status = T30_ERR_T2EXPRX;
        break;
#endif
    }
    set_phase(s, T30_PHASE_B_TX);
    start_receiving_document(s);
}
/*- End of function --------------------------------------------------------*/

static void timer_t3_expired(t30_state_t *s)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "T3 timeout in phase %s, state %d\n", phase_names[s->phase], s->state);
    s->current_status = T30_ERR_T3EXPIRED;
    disconnect(s);
}
/*- End of function --------------------------------------------------------*/

static void timer_t4_expired(t30_state_t *s)
{
    /* There was no response (or only a corrupt response) to a command */
    span_log(&s->logging, SPAN_LOG_FLOW, "T4 timeout in phase %s, state %d\n", phase_names[s->phase], s->state);
    switch (s->state)
    {
    case T30_STATE_R:
        set_phase(s, T30_PHASE_B_TX);
        s->dis_received = FALSE;
        send_nsf_frame(s);
        send_ident_frame(s, T30_CSI);
        set_dis_or_dtc(s);
        send_frame(s, s->dis_dtc_frame, s->dis_dtc_len);
        break;
    case T30_STATE_III_EOP_MCF:
        set_phase(s, T30_PHASE_D_TX);
        send_simple_frame(s, T30_MCF);
        break;
    case T30_STATE_III_EOP_RTP:
        set_phase(s, T30_PHASE_D_TX);
        send_simple_frame(s, T30_RTP);
        break;
    case T30_STATE_III_EOP_RTN:
        set_phase(s, T30_PHASE_D_TX);
        send_simple_frame(s, T30_RTN);
        break;
    case T30_STATE_II_MPS:
        set_phase(s, T30_PHASE_D_TX);
        send_simple_frame(s, (s->local_interrupt_pending)  ?  T30_PRI_MPS  :  T30_MPS);
        break;
    case T30_STATE_II_EOM:
        set_phase(s, T30_PHASE_D_TX);
        send_simple_frame(s, (s->local_interrupt_pending)  ?  T30_PRI_EOM  :  T30_EOM);
        break;
    case T30_STATE_II_EOP:
        set_phase(s, T30_PHASE_D_TX);
        send_simple_frame(s, (s->local_interrupt_pending)  ?  T30_PRI_EOP  :  T30_EOP);
        break;
    case T30_STATE_D:
        break;
    }
}
/*- End of function --------------------------------------------------------*/

void t30_timer_update(t30_state_t *s, int samples)
{
    if (s->timer_t0_t1 > 0)
    {
        s->timer_t0_t1 -= samples;
        if (s->timer_t0_t1 <= 0)
        {
            if (s->far_end_detected)
                timer_t1_expired(s);
            else
                timer_t0_expired(s);
        }
    }
    if (s->timer_t3 > 0)
    {
        s->timer_t3 -= samples;
        if (s->timer_t3 <= 0)
            timer_t3_expired(s);
    }
    if (s->timer_t2_t4 > 0)
    {
        s->timer_t2_t4 -= samples;
        if (s->timer_t2_t4 <= 0)
        {
            if (s->timer_is_t4)
                timer_t4_expired(s);
            else
                timer_t2_expired(s);
        }
    }
}
/*- End of function --------------------------------------------------------*/

static void decode_20digit_msg(t30_state_t *s, char *msg, const uint8_t *pkt, int len)
{
    int p;
    int k;
    char text[20 + 1];

    if (msg == NULL)
        msg = text;
    if (len > 21)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Bad %s frame length - %d\n", t30_frametype(pkt[0]), len);
        msg[0] = '\0';
        return;
    }
    p = len;
    /* Strip trailing spaces */
    while (p > 1  &&  pkt[p - 1] == ' ')
        p--;
    /* The string is actually backwards in the message */
    k = 0;
    while (p > 1)
        msg[k++] = pkt[--p];
    msg[k] = '\0';
    span_log(&s->logging, SPAN_LOG_FLOW, "Remote fax gave %s as: \"%s\"\n", t30_frametype(pkt[0]), msg);
}
/*- End of function --------------------------------------------------------*/

static void decode_password(t30_state_t *s, char *msg, const uint8_t *pkt, int len)
{
    int p;
    int k;
    char text[20 + 1];

    if (msg == NULL)
        msg = text;
    if (len > 21)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Bad password frame length - %d\n", len);
        msg[0] = '\0';
        return;
    }
    p = len;
    /* Strip trailing spaces */
    while (p > 1  &&  pkt[p - 1] == ' ')
        p--;
    /* The string is actually backwards in the message */
    k = 0;
    while (p > 1)
        msg[k++] = pkt[--p];
    msg[k] = '\0';
    span_log(&s->logging, SPAN_LOG_FLOW, "Remote fax gave the password as: \"%s\"\n", msg);
}
/*- End of function --------------------------------------------------------*/

static void decode_url_msg(t30_state_t *s, char *msg, const uint8_t *pkt, int len)
{
    char text[77 + 1];

    /* TODO: decode properly, as per T.30 5.3.6.2.12 */
    if (msg == NULL)
        msg = text;
    if (len < 3  ||  len > 77 + 3  ||  len != pkt[2] + 3)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Bad %s frame length - %d\n", t30_frametype(pkt[0]), len);
        msg[0] = '\0';
        return;
    }
    memcpy(msg, &pkt[3], len - 3);
    msg[len - 3] = '\0';
    span_log(&s->logging, SPAN_LOG_FLOW, "Remote fax gave %s as: %d, %d, \"%s\"\n", t30_frametype(pkt[0]), pkt[0], pkt[1], msg);
}
/*- End of function --------------------------------------------------------*/

const char *t30_frametype(uint8_t x)
{
    switch (x & 0xFE)
    {
    case T30_DIS:
        if (x == T30_DTC)
            return "DTC";
        return "DIS";
    case T30_CSI:
        if (x == T30_CIG)
            return "CIG";
        return "CSI";
    case T30_NSF:
        if (x == T30_NSC)
            return "NSC";
        return "NSF";
    case T30_PWD & 0xFE:
        if (x == T30_PWD)
            return "PWD";
        break;
    case T30_SEP & 0xFE:
        if (x == T30_SEP)
            return "SEP";
        break;
    case T30_PSA & 0xFE:
        if (x == T30_PSA)
            return "PSA";
        break;
    case T30_CIA & 0xFE:
        if (x == T30_CIA)
            return "CIA";
        break;
    case T30_ISP & 0xFE:
        if (x == T30_ISP)
            return "ISP";
        break;
    case T30_DCS:
        return "DCS";
    case T30_TSI:
        return "TSI";
    case T30_NSS:
        return "NSS";
    case T30_SUB:
        return "SUB";
    case T30_SID:
        return "SID";
    case T30_CTC:
        return "CTC";
    case T30_TSA:
        return "TSA";
    case T30_IRA:
        return "IRA";
    case T30_CFR:
        return "CFR";
    case T30_FTT:
        return "FTT";
    case T30_CTR:
        return "CTR";
    case T30_CSA:
        return "CSA";
    case T30_EOM:
        return "EOM";
    case T30_MPS:
        return "MPS";
    case T30_EOP:
        return "EOP";
    case T30_PRI_EOM:
        return "PRI_EOM";
    case T30_PRI_MPS:
        return "PRI_MPS";
    case T30_PRI_EOP:
        return "PRI_EOP";
    case T30_EOS:
        return "EOS";
    case T30_PPS:
        return "PPS";
    case T30_EOR:
        return "EOR";
    case T30_RR:
        return "RR";
    case T30_MCF:
        return "MCF";
    case T30_RTP:
        return "RTP";
    case T30_RTN:
        return "RTN";
    case T30_PIP:
        return "PIP";
    case T30_PIN:
        return "PIN";
    case T30_PPR:
        return "PPR";
    case T30_RNR:
        return "RNR";
    case T30_ERR:
        return "ERR";
    case T30_FDM:
        return "FDM";
    case T30_DCN:
        return "DCN";
    case T30_CRP:
        return "CRP";
    case T30_FNV:
        return "FNV";
    case T30_TNR:
        return "TNR";
    case T30_TR:
        return "TR";
    case T4_FCD:
        return "FCD";
    case T4_RCP:
        return "RCP";
    }
    return "???";
}
/*- End of function --------------------------------------------------------*/

void t30_decode_dis_dtc_dcs(t30_state_t *s, const uint8_t *pkt, int len)
{
    logging_state_t *log;

    if (!span_log_test(&s->logging, SPAN_LOG_FLOW))
        return;
    log = &s->logging;
    if (len <= 2)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }
    
    span_log(log, SPAN_LOG_FLOW, "%s:\n", t30_frametype(pkt[2]));
    if (len <= 3)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }
    if ((pkt[3] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  Store and forward Internet fax (T.37)\n");
    if ((pkt[3] & DISBIT3))
        span_log(log, SPAN_LOG_FLOW, "  Real-time Internet fax (T.38)\n");
    if ((pkt[3] & DISBIT4))
        span_log(log, SPAN_LOG_FLOW, "  3G mobile network\n");

    if (pkt[2] == T30_DCS)
    {
        if ((pkt[3] & DISBIT6))
            span_log(log, SPAN_LOG_FLOW, "  Invalid: 1\n");
        if ((pkt[3] & DISBIT7))
            span_log(log, SPAN_LOG_FLOW, "  Invalid: 1\n");
    }
    else
    {
        if ((pkt[3] & DISBIT6))
            span_log(log, SPAN_LOG_FLOW, "  V.8 capable\n");
        span_log(log, SPAN_LOG_FLOW, "  Prefer %d octet blocks\n", (pkt[3] & DISBIT7)  ?  64  :  256);
    }
    if ((pkt[3] & (DISBIT2 | DISBIT5 | DISBIT8)))
        span_log(log, SPAN_LOG_FLOW, "  Reserved: 0x%X\n", (pkt[3] & (DISBIT2 | DISBIT4 | DISBIT5 | DISBIT8)));
    if (len <= 4)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }
    
    if (pkt[2] == T30_DCS)
    {
        if ((pkt[4] & DISBIT1))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
    }
    else
    {
        if ((pkt[4] & DISBIT1))
            span_log(log, SPAN_LOG_FLOW, "  Ready to transmit a fax document (polling)\n");
    }
    if ((pkt[4] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  Can receive fax\n");
    if (pkt[2] == T30_DCS)
    {
        span_log(log, SPAN_LOG_FLOW, "  Selected data signalling rate: ");
        switch (pkt[4] & (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3))
        {
        case 0:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.27ter, 2400bps\n");
            break;
        case DISBIT4:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.27ter, 4800bps\n");
            break;
        case DISBIT3:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.29, 9600bps\n");
            break;
        case (DISBIT4 | DISBIT3):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.29, 7200bps\n");
            break;
        case DISBIT6:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.17, 14400bps\n");
            break;
        case (DISBIT6 | DISBIT4):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.17, 12000bps\n");
            break;
        case (DISBIT6 | DISBIT3):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.17, 9600bps\n");
            break;
        case (DISBIT6 | DISBIT4 | DISBIT3):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.17, 7200bps\n");
            break;
        case (DISBIT5 | DISBIT3):
        case (DISBIT5 | DISBIT4 | DISBIT3):
        case (DISBIT6 | DISBIT5):
        case (DISBIT6 | DISBIT5 | DISBIT3):
        case (DISBIT6 | DISBIT5 | DISBIT4):
        case (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Reserved\n");
            break;
        default:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Not used\n");
            break;
        }
    }
    else
    {
        span_log(log, SPAN_LOG_FLOW, "  Supported data signalling rates: ");
        switch (pkt[4] & (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3))
        {
        case 0:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.27ter fallback mode\n");
            break;
        case DISBIT4:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.27ter\n");
            break;
        case DISBIT3:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.29\n");
            break;
        case (DISBIT4 | DISBIT3):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.27ter and V.29\n");
            break;
        case (DISBIT6 | DISBIT4 | DISBIT3):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "V.27ter, V.29 and V.17\n");
            break;
        case (DISBIT5 | DISBIT4):
        case (DISBIT6 | DISBIT4):
        case (DISBIT6 | DISBIT5 | DISBIT4):
        case (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Reserved\n");
            break;
        default:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Not used\n");
            break;
        }
    }
    if ((pkt[4] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  R8x7.7lines/mm and/or 200x200pels/25.4mm\n");
    if ((pkt[4] & DISBIT8))
        span_log(log, SPAN_LOG_FLOW, "  2D coding\n");
    if (len <= 5)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if (pkt[2] == T30_DCS)
    {
        span_log(log, SPAN_LOG_FLOW, "  Scan line length: ");
        switch (pkt[5] & (DISBIT2 | DISBIT1))
        {
        case 0:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "215mm\n");
            break;
        case DISBIT2:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "303mm\n");
            break;
        case DISBIT1:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "255mm\n");
            break;
        default:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Invalid\n");
            break;
        }
        span_log(log, SPAN_LOG_FLOW, "  Recording length: ");
        switch (pkt[5] & (DISBIT4 | DISBIT3))
        {
        case 0:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "A4 (297mm)\n");
            break;
        case DISBIT3:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Unlimited\n");
            break;
        case DISBIT4:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "B4 (364mm)\n");
            break;
        case (DISBIT4 | DISBIT3):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Invalid\n");
            break;
        }
        span_log(log, SPAN_LOG_FLOW, "  Minimum scan line time: ");
        switch (pkt[5] & (DISBIT7 | DISBIT6 | DISBIT5))
        {
        case 0:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "20ms\n");
            break;
        case DISBIT7:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "40ms\n");
            break;
        case DISBIT6:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "10ms\n");
            break;
        case DISBIT5:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "5ms\n");
            break;
        case (DISBIT7 | DISBIT6 | DISBIT5):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "0ms\n");
            break;
        default:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Invalid\n");
            break;
        }
    }
    else
    {
        span_log(log, SPAN_LOG_FLOW, "  Scan line length: ");
        switch (pkt[5] & (DISBIT2 | DISBIT1))
        {
        case 0:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "215mm\n");
            break;
        case DISBIT2:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "215mm, 255mm or 303mm\n");
            break;
        case DISBIT1:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "215mm or 255mm\n");
            break;
        default:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Invalid\n");
            break;
        }
        span_log(log, SPAN_LOG_FLOW, "  Recording length: ");
        switch (pkt[5] & (DISBIT4 | DISBIT3))
        {
        case 0:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "A4 (297mm)\n");
            break;
        case DISBIT3:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Unlimited\n");
            break;
        case DISBIT4:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "A4 (297mm) and B4 (364mm)\n");
            break;
        case (DISBIT4 | DISBIT3):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "Invalid\n");
            break;
        }
        span_log(log, SPAN_LOG_FLOW, "  Receiver's minimum scan line time: ");
        switch (pkt[5] & (DISBIT7 | DISBIT6 | DISBIT5))
        {
        case 0:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "20ms at 3.85 l/mm: T7.7 = T3.85\n");
            break;
        case DISBIT7:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "40ms at 3.85 l/mm: T7.7 = T3.85\n");
            break;
        case DISBIT6:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "10ms at 3.85 l/mm: T7.7 = T3.85\n");
            break;
        case DISBIT5:
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "5ms at 3.85 l/mm: T7.7 = T3.85\n");
            break;
        case (DISBIT7 | DISBIT6):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "10ms at 3.85 l/mm: T7.7 = 1/2 T3.85\n");
            break;
        case (DISBIT6 | DISBIT5):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "20ms at 3.85 l/mm: T7.7 = 1/2 T3.85\n");
            break;
        case (DISBIT7 | DISBIT5):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "40ms at 3.85 l/mm: T7.7 = 1/2 T3.85\n");
            break;
        case (DISBIT7 | DISBIT6 | DISBIT5):
            span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "0ms at 3.85 l/mm: T7.7 = T3.85\n");
            break;
        }
    }
    if (!(pkt[5] & DISBIT8))
        return;
    if (len <= 6)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[6] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  Uncompressed mode\n");
    if ((pkt[6] & DISBIT3))
        span_log(log, SPAN_LOG_FLOW, "  Error correction mode\n");
    if (pkt[2] == T30_DCS)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame size: %s\n", (pkt[6] & DISBIT4)  ?  "64 octets"  :  "256 octets");
    }
    else
    {
        if ((pkt[6] & DISBIT4))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 0x%X\n", (pkt[6] & DISBIT4));
    }
    if ((pkt[6] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  T.6 coding\n");
    if ((pkt[6] & (DISBIT1 | DISBIT5 | DISBIT6)))
        span_log(log, SPAN_LOG_FLOW, "  Reserved: 0x%X\n", (pkt[6] & (DISBIT1 | DISBIT5 | DISBIT6)));
    if (!(pkt[6] & DISBIT8))
        return;
    if (len <= 7)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[7] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  \"Field not valid\" supported\n");
    if (pkt[2] == T30_DCS)
    {
        if ((pkt[7] & DISBIT2))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
        if ((pkt[7] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
    }
    else
    {
        if ((pkt[7] & DISBIT2))
            span_log(log, SPAN_LOG_FLOW, "  Multiple selective polling\n");
        if ((pkt[7] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Polled Subaddress\n");
    }
    if ((pkt[7] & DISBIT4))
        span_log(log, SPAN_LOG_FLOW, "  T.43 coding\n");
    if ((pkt[7] & DISBIT5))
        span_log(log, SPAN_LOG_FLOW, "  Plane interleave\n");
    if ((pkt[7] & DISBIT6))
        span_log(log, SPAN_LOG_FLOW, "  Voice coding with 32kbit/s ADPCM (Rec. G.726)\n");
    if ((pkt[7] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  Reserved for the use of extended voice coding set\n");
    if (!(pkt[7] & DISBIT8))
        return;
    if (len <= 8)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[8] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  R8x15.4lines/mm\n");
    if ((pkt[8] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  300x300pels/25.4mm\n");
    if ((pkt[8] & DISBIT3))
        span_log(log, SPAN_LOG_FLOW, "  R16x15.4lines/mm and/or 400x400pels/25.4 mm\n");
    if (pkt[2] == T30_DCS)
    {
        span_log(log, SPAN_LOG_FLOW, "  Resolution type selection: %s\n", (pkt[8] & DISBIT4)  ?  "inch-based"  :  "metric-based");
        if ((pkt[8] & DISBIT5))
            span_log(log, SPAN_LOG_FLOW, "  Don't care: 1\n");
        if ((pkt[8] & DISBIT6))
            span_log(log, SPAN_LOG_FLOW, "  Don't care: 1\n");
    }
    else
    {
        if ((pkt[8] & DISBIT4))
            span_log(log, SPAN_LOG_FLOW, "  Inch-based resolution preferred\n");
        if ((pkt[8] & DISBIT5))
            span_log(log, SPAN_LOG_FLOW, "  Metric-based resolution preferred\n");
        span_log(log, SPAN_LOG_FLOW, "  Minimum scan line time for higher resolutions: %s\n", (pkt[8] & DISBIT6)  ?  "T15.4 = 1/2 T7.7"  :  "T15.4 = T7.7");
    }
    if (pkt[2] == T30_DCS)
    {
        if ((pkt[8] & DISBIT7))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
    }
    else
    {
        if ((pkt[8] & DISBIT7))
            span_log(log, SPAN_LOG_FLOW, "  Selective polling OK\n");
    }
    if (!(pkt[8] & DISBIT8))
        return;
    if (len <= 9)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[9] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  Subaddressing\n");
    if ((pkt[9] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  Password\n");
    if (pkt[2] == T30_DCS)
    {
        if ((pkt[9] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
    }
    else
    {
        if ((pkt[9] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Ready to transmit a data file (polling)\n");
    }
    if ((pkt[9] & DISBIT5))
        span_log(log, SPAN_LOG_FLOW, "  Binary file transfer (BFT)\n");
    if ((pkt[9] & DISBIT6))
        span_log(log, SPAN_LOG_FLOW, "  Document transfer mode (DTM)\n");
    if ((pkt[9] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  Electronic data interchange (EDI)\n");
    if ((pkt[9] & DISBIT4))
        span_log(log, SPAN_LOG_FLOW, "  Reserved: 1\n");
    if (!(pkt[9] & DISBIT8))
        return;
    if (len <= 10)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[10] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  Basic transfer mode (BTM)\n");
    if (pkt[2] == T30_DCS)
    {
        if ((pkt[10] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
    }
    else
    {
        if ((pkt[10] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Ready to transfer a character or mixed mode document (polling)\n");
    }
    if ((pkt[10] & DISBIT4))
        span_log(log, SPAN_LOG_FLOW, "  Character mode\n");
    if ((pkt[10] & DISBIT6))
        span_log(log, SPAN_LOG_FLOW, "  Mixed mode (Annex E/T.4)\n");
    if ((pkt[10] & (DISBIT2 | DISBIT5 | DISBIT7)))
        span_log(log, SPAN_LOG_FLOW, "  Reserved: 0x%X\n", (pkt[10] & (DISBIT2 | DISBIT5 | DISBIT7)));
    if (!(pkt[10] & DISBIT8))
        return;
    if (len <= 11)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[11] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  Processable mode 26 (Rec. T.505)\n");
    if ((pkt[11] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  Digital network\n");
    if (pkt[2] == T30_DCS)
    {
        if ((pkt[11] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Duplex or half-duplex\n");
        if ((pkt[11] & DISBIT4))
            span_log(log, SPAN_LOG_FLOW, "  Full colour mode\n");
    }
    else
    {
        if ((pkt[11] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Duplex\n");
        if ((pkt[11] & DISBIT4))
            span_log(log, SPAN_LOG_FLOW, "  JPEG coding\n");
    }
    if ((pkt[11] & DISBIT5))
        span_log(log, SPAN_LOG_FLOW, "  Full colour mode\n");
    if (pkt[2] == T30_DCS)
    {
        if ((pkt[11] & DISBIT6))
            span_log(log, SPAN_LOG_FLOW, "  Preferred Huffman tables\n");
    }
    else
    {
        if ((pkt[11] & DISBIT6))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
    }
    if ((pkt[11] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  12bits/pel component\n");
    if (!(pkt[11] & DISBIT8))
        return;
    if (len <= 12)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[12] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  No subsampling (1:1:1)\n");
    if ((pkt[12] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  Custom illuminant\n");
    if ((pkt[12] & DISBIT3))
        span_log(log, SPAN_LOG_FLOW, "  Custom gamut range\n");
    if ((pkt[12] & DISBIT4))
        span_log(log, SPAN_LOG_FLOW, "  North American Letter (215.9mm x 279.4mm)\n");
    if ((pkt[12] & DISBIT5))
        span_log(log, SPAN_LOG_FLOW, "  North American Legal (215.9mm x 355.6mm)\n");
    if ((pkt[12] & DISBIT6))
        span_log(log, SPAN_LOG_FLOW, "  Single-progression sequential coding (Rec. T.85) basic\n");
    if ((pkt[12] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  Single-progression sequential coding (Rec. T.85) optional L0\n");
    if (!(pkt[12] & DISBIT8))
        return;
    if (len <= 13)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[13] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  HKM key management\n");
    if ((pkt[13] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  RSA key management\n");
    if ((pkt[13] & DISBIT3))
        span_log(log, SPAN_LOG_FLOW, "  Override\n");
    if ((pkt[13] & DISBIT4))
        span_log(log, SPAN_LOG_FLOW, "  HFX40 cipher\n");
    if ((pkt[13] & DISBIT5))
        span_log(log, SPAN_LOG_FLOW, "  Alternative cipher number 2\n");
    if ((pkt[13] & DISBIT6))
        span_log(log, SPAN_LOG_FLOW, "  Alternative cipher number 3\n");
    if ((pkt[13] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  HFX40-I hashing\n");
    if (!(pkt[13] & DISBIT8))
        return;
    if (len <= 14)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[14] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  Alternative hashing system 2\n");
    if ((pkt[14] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  Alternative hashing system 3\n");
    if ((pkt[14] & DISBIT3))
        span_log(log, SPAN_LOG_FLOW, "  Reserved for future security features\n");
    if ((pkt[14] & (DISBIT4 | DISBIT5 | DISBIT6)))
        span_log(log, SPAN_LOG_FLOW, "  T.44 (Mixed Raster Content): 0x%X\n", (pkt[14] & (DISBIT4 | DISBIT5 | DISBIT6)));
    if ((pkt[14] & DISBIT6))
        span_log(log, SPAN_LOG_FLOW, "  Page length maximum stripe size for T.44 (Mixed Raster Content)\n");
    if (!(pkt[14] & DISBIT8))
        return;
    if (len <= 15)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[15] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  Colour/gray-scale 300pels/25.4mm x 300lines/25.4mm or 400pels/25.4mm x 400lines/25.4mm resolution\n");
    if ((pkt[15] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  100pels/25.4mm x 100lines/25.4mm for colour/gray scale\n");
    if ((pkt[15] & DISBIT3))
        span_log(log, SPAN_LOG_FLOW, "  Simple phase C BFT negotiations\n");
    if (pkt[2] == T30_DCS)
    {
        if ((pkt[15] & DISBIT4))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
        if ((pkt[15] & DISBIT5))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
    }
    else
    {
        if ((pkt[15] & DISBIT4))
            span_log(log, SPAN_LOG_FLOW, "  Reserved for Extended BFT Negotiations capable\n");
        if ((pkt[15] & DISBIT5))
            span_log(log, SPAN_LOG_FLOW, "  Internet Selective Polling address (ISP)\n");
    }
    if ((pkt[15] & DISBIT6))
        span_log(log, SPAN_LOG_FLOW, "  Internet Routing Address (IRA)\n");
    if ((pkt[15] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  Reserved: 1\n");
    if (!(pkt[15] & DISBIT8))
        return;
    if (len <= 16)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[16] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  600pels/25.4mm x 600lines/25.4mm\n");
    if ((pkt[16] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  1200pels/25.4mm x 1200lines/25.4mm\n");
    if ((pkt[16] & DISBIT3))
        span_log(log, SPAN_LOG_FLOW, "  300pels/25.4mm x 600lines/25.4mm\n");
    if ((pkt[16] & DISBIT4))
        span_log(log, SPAN_LOG_FLOW, "  400pels/25.4mm x 800lines/25.4mm\n");
    if ((pkt[16] & DISBIT5))
        span_log(log, SPAN_LOG_FLOW, "  600pels/25.4mm x 1200lines/25.4mm\n");
    if ((pkt[16] & DISBIT6))
        span_log(log, SPAN_LOG_FLOW, "  Colour/gray scale 600pels/25.4mm x 600lines/25.4mm\n");
    if ((pkt[16] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  Colour/gray scale 1200pels/25.4mm x 1200lines/25.4mm\n");
    if (!(pkt[16] & DISBIT8))
        return;
    if (len <= 17)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[17] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  Double sided printing capability (alternate mode)\n");
    if ((pkt[17] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  Double sided printing capability (continuous mode)\n");
    if (pkt[2] == T30_DCS)
    {
        if ((pkt[17] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Black and white mixed raster content profile (MRCbw)\n");
    }
    else
    {
        if ((pkt[17] & DISBIT3))
            span_log(log, SPAN_LOG_FLOW, "  Set to \"0\": 1\n");
    }
    if ((pkt[17] & DISBIT4))
        span_log(log, SPAN_LOG_FLOW, "  T.45 run length colour encoded\n");
    span_log(log, SPAN_LOG_FLOW, "  Shared memory ");
    switch (pkt[17] & (DISBIT5 | DISBIT6))
    {
    case 0:
        span_log(log, SPAN_LOG_FLOW, "not available\n");
        break;
    case DISBIT5:
        span_log(log, SPAN_LOG_FLOW, "level 1 = 1.0M bytes\n");
        break;
    case DISBIT6:
        span_log(log, SPAN_LOG_FLOW, "level 2 = 2.0M bytes\n");
        break;
    case DISBIT5 | DISBIT6:
        span_log(log, SPAN_LOG_FLOW, "level 3 = unlimited (i.e. >=32M bytes)\n");
        break;
    }
    if ((pkt[17] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  Reserved: 1\n");
    if (!(pkt[17] & DISBIT8))
        return;
    if (len <= 18)
    {
        span_log(log, SPAN_LOG_FLOW, "  Frame is short\n");
        return;
    }

    if ((pkt[18] & DISBIT1))
        span_log(log, SPAN_LOG_FLOW, "  Flow control capability for T.38 communication\n");
    if ((pkt[18] & DISBIT2))
        span_log(log, SPAN_LOG_FLOW, "  K>4\n");
    if ((pkt[18] & DISBIT3))
        span_log(log, SPAN_LOG_FLOW, "  Internet aware T.38 mode fax device capability\n");
    span_log(log, SPAN_LOG_FLOW, "  T.89 (Application profiles for ITU-T Rec T.8)");
    switch (pkt[18] & (DISBIT4 | DISBIT5 | DISBIT6))
    {
    case 0:
        span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "not used\n");
        break;
    case DISBIT6:
        span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "profile 1\n");
        break;
    case DISBIT5:
        span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "profile 2\n");
        break;
    case DISBIT5 | DISBIT6:
        span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "profile 3\n");
        break;
    case DISBIT4:
        span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "profiles 2 and 3\n");
        break;
    case DISBIT4 | DISBIT6:
        span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "reserved\n");
        break;
    case DISBIT4 | DISBIT5:
        span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "reserved\n");
        break;
    case DISBIT4 | DISBIT5 | DISBIT6:
        span_log(log, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "reserved\n");
        break;
    }
    if ((pkt[18] & DISBIT7))
        span_log(log, SPAN_LOG_FLOW, "  sYCC-JPEG coding\n");
    if (!(pkt[18] & DISBIT8))
        return;

    span_log(log, SPAN_LOG_FLOW, "  Extended beyond the current T.30 specification!\n");
}
/*- End of function --------------------------------------------------------*/

int t30_restart(t30_state_t *s)
{
    s->phase = T30_PHASE_IDLE;
    s->next_phase = T30_PHASE_IDLE;
    s->current_fallback = 0;
    s->rx_signal_present = FALSE;
    s->current_status = T30_ERR_OK;
    build_dis_or_dtc(s);
    if (s->calling_party)
    {
        set_state(s, T30_STATE_T);
        set_phase(s, T30_PHASE_A_CNG);
    }
    else
    {
        set_state(s, T30_STATE_ANSWERING);
        set_phase(s, T30_PHASE_A_CED);
    }
    s->far_end_detected = FALSE;
    s->timer_t0_t1 = ms_to_samples(DEFAULT_TIMER_T0);
    return 0;
}
/*- End of function --------------------------------------------------------*/

int t30_init(t30_state_t *s, int calling_party, void *user_data)
{
    memset(s, 0, sizeof(*s));
    s->calling_party = calling_party;
    span_log_init(&s->logging, SPAN_LOG_NONE, NULL);
    span_log_set_protocol(&s->logging, "T.30");
    t30_restart(s);
    return 0;
}
/*- End of function --------------------------------------------------------*/

void t30_release(t30_state_t *s)
{
    /* Make sure any FAX in progress is tidied up. If the tidying up has
       already happened, repeating it here is harmless. */
    t4_rx_end(&(s->t4));
    t4_tx_end(&(s->t4));
}
/*- End of function --------------------------------------------------------*/

t30_state_t *t30_create(int calling_party, void *user_data)
{
    t30_state_t *s;
    
    if ((s = (t30_state_t *) malloc(sizeof(t30_state_t))) == NULL)
        return NULL;
    if (t30_init(s, calling_party, user_data))
    {
        free(s);
        return NULL;
    }
    return s;
}
/*- End of function --------------------------------------------------------*/

void t30_free(t30_state_t *s)
{
    t30_release(s);
    free(s);
}
/*- End of function --------------------------------------------------------*/

void t30_set_iaf_mode(t30_state_t *s, int iaf)
{
    s->iaf = iaf;
}
/*- End of function --------------------------------------------------------*/

int t30_set_header_info(t30_state_t *s, const char *info)
{
    return t4_tx_set_header_info(&(s->t4), info);
}
/*- End of function --------------------------------------------------------*/

int t30_set_local_ident(t30_state_t *s, const char *id)
{
    if (id == NULL)
    {
        s->local_ident[0] = '\0';
        return 0;
    }
    if (strlen(id) > 20)
        return -1;
    strcpy(s->local_ident, id);
    return 0;
}
/*- End of function --------------------------------------------------------*/

int t30_set_local_nsf(t30_state_t *s, const uint8_t *nsf, int len)
{
    if (len > 100)
        return -1;
    memcpy(s->local_nsf, nsf, len);
    s->local_nsf_len = len;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int t30_set_sub_address(t30_state_t *s, const char *sub_address)
{
    if (sub_address == NULL)
    {
        s->sub_address[0] = '\0';
        return 0;
    }
    if (strlen(sub_address) > 20)
        return -1;
    strcpy(s->sub_address, sub_address);
    return 0;
}
/*- End of function --------------------------------------------------------*/

size_t t30_get_sub_address(t30_state_t *s, char *sub_address)
{
    if (sub_address)
        strcpy(sub_address, s->sub_address);
    return strlen(s->sub_address);
}
/*- End of function --------------------------------------------------------*/

size_t t30_get_header_info(t30_state_t *s, char *info)
{
    if (info)
        strcpy(info, t4_tx_get_header_info(&(s->t4)));
    return strlen(info);
}
/*- End of function --------------------------------------------------------*/

size_t t30_get_local_ident(t30_state_t *s, char *id)
{
    if (id)
        strcpy(id, s->local_ident);
    return strlen(s->local_ident);
}
/*- End of function --------------------------------------------------------*/

size_t t30_get_far_ident(t30_state_t *s, char *id)
{
    if (id)
        strcpy(id, s->far_ident);
    return strlen(s->far_ident);
}
/*- End of function --------------------------------------------------------*/

const char *t30_get_far_country(t30_state_t *s)
{
    return s->country;
}
/*- End of function --------------------------------------------------------*/

const char *t30_get_far_vendor(t30_state_t *s)
{
    return s->vendor;
}
/*- End of function --------------------------------------------------------*/

const char *t30_get_far_model(t30_state_t *s)
{
    return s->model;
}
/*- End of function --------------------------------------------------------*/

void t30_get_transfer_statistics(t30_state_t *s, t30_stats_t *t)
{
    t4_stats_t stats;

    t->bit_rate = fallback_sequence[s->current_fallback].bit_rate;
    t4_get_transfer_statistics(&(s->t4), &stats);
    t->pages_transferred = stats.pages_transferred;
    t->width = stats.width;
    t->length = stats.length;
    t->bad_rows = stats.bad_rows;
    t->longest_bad_row_run = stats.longest_bad_row_run;
    t->x_resolution = stats.x_resolution;
    t->y_resolution = stats.y_resolution;
    t->encoding = stats.encoding;
    t->image_size = stats.image_size;
    t->current_status = s->current_status;
}
/*- End of function --------------------------------------------------------*/

void t30_set_phase_b_handler(t30_state_t *s, t30_phase_b_handler_t *handler, void *user_data)
{
    s->phase_b_handler = handler;
    s->phase_b_user_data = user_data;
}
/*- End of function --------------------------------------------------------*/

void t30_set_phase_d_handler(t30_state_t *s, t30_phase_d_handler_t *handler, void *user_data)
{
    s->phase_d_handler = handler;
    s->phase_d_user_data = user_data;
}
/*- End of function --------------------------------------------------------*/

void t30_set_phase_e_handler(t30_state_t *s, t30_phase_e_handler_t *handler, void *user_data)
{
    s->phase_e_handler = handler;
    s->phase_e_user_data = user_data;
}
/*- End of function --------------------------------------------------------*/

void t30_set_document_handler(t30_state_t *s, t30_document_handler_t *handler, void *user_data)
{
    s->document_handler = handler;
    s->document_user_data = user_data;
}
/*- End of function --------------------------------------------------------*/

void t30_set_rx_file(t30_state_t *s, const char *file, int stop_page)
{
    strncpy(s->rx_file, file, sizeof(s->rx_file));
    s->rx_file[sizeof(s->rx_file) - 1] = '\0';
    s->rx_stop_page = stop_page;
}
/*- End of function --------------------------------------------------------*/

void t30_set_tx_file(t30_state_t *s, const char *file, int start_page, int stop_page)
{
    strncpy(s->tx_file, file, sizeof(s->tx_file));
    s->tx_file[sizeof(s->tx_file) - 1] = '\0';
    s->tx_start_page = start_page;
    s->tx_stop_page = stop_page;
}
/*- End of function --------------------------------------------------------*/

void t30_local_interrupt_request(t30_state_t *s, int state)
{
    if (s->timer_t3 > 0  &&  state)
    {
        /* Accept the far end's outstanding request for interrupt. */
        /* TODO: */
        send_simple_frame(s, T30_PIP);
    }
    s->local_interrupt_pending = state;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
