/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: MixerLine.cpp,v 1.70 2004/02/27 13:17:56 comix Exp $
 *
 */


#include "MixerLine.h"
#include <stdio.h>

#include "HydrogenApp.h"
#include "lib/Hydrogen.h"
#include "lib/PreferencesMng.h"


#define MIXERLINE_WIDTH		42
#define MIXERLINE_HEIGHT	240
#define MIXERLINE_FADER_H	135
#define MIXERLINE_LABEL_H	115

QPixmap* MixerLine::m_pMixerLineBackground = NULL;
QPixmap* MixerLine::m_pMixerLineBackground_selected = NULL;

/**
 * Constructor
 */
MixerLine::MixerLine(QWidget* parent) : QWidget(parent), Object( "MixerLine" )
{
//	cout << "mixerLine init" << endl;

	m_nWidth = MIXERLINE_WIDTH;
	m_nHeight = MIXERLINE_HEIGHT;
	maxPeak = 0.0;
	activity = 0;
	m_bIsSelected = false;

	setMinimumSize( m_nWidth, m_nHeight );
	setMaximumSize( m_nWidth, m_nHeight );
	resize( m_nWidth, m_nHeight );


	// MixerLine Background image
	if (m_pMixerLineBackground == NULL ) {
		//infoLog( "loading background pixmap" );
		string mixerLineBackground_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_background.png");
		m_pMixerLineBackground = new QPixmap();
		bool ok = m_pMixerLineBackground->load(mixerLineBackground_path.c_str());
		if( ok == false ){
			errorLog( string("Error loading pixmap ") + mixerLineBackground_path );
		}
	}
	if (m_pMixerLineBackground_selected == NULL ) {
		//infoLog( "loading background pixmap" );
		string mixerLineBackground_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_background_on.png");
		m_pMixerLineBackground_selected = new QPixmap();
		bool ok = m_pMixerLineBackground_selected->load(mixerLineBackground_path.c_str());
		if( ok == false ){
			errorLog( string("Error loading pixmap ") + mixerLineBackground_path );
		}
	}
	setBackgroundPixmap( *m_pMixerLineBackground );


	// Play sample button
	string playSample_on_path = string(IMG_PATH).append("/img/mixerPanel/btn_play_on.png");
	string playSample_off_path = string(IMG_PATH).append("/img/mixerPanel/btn_play_off.png");
	string playSample_over_path = string(IMG_PATH).append("/img/mixerPanel/btn_play_over.png");
	playSampleBtn = new Button(this, QSize( 18, 12 ), playSample_on_path, playSample_off_path, playSample_over_path);
	playSampleBtn->move( 2, 5 );
	QToolTip::add( playSampleBtn, trUtf8( "Play sample" ) );
	connect(playSampleBtn, SIGNAL(clicked(Button*)), this, SLOT(click(Button*)));
	connect(playSampleBtn, SIGNAL(rightClicked(Button*)), this, SLOT(rightClick(Button*)));

	// load button
	string load_off_path = string(IMG_PATH).append("/img/mixerPanel/btn_load_off.png");
	string load_over_path = string(IMG_PATH).append("/img/mixerPanel/btn_load_over.png");
	loadBtn = new Button(this, QSize( 18, 12 ), load_over_path, load_off_path, load_over_path);
	loadBtn->move( 22, 5 );
	QToolTip::add( loadBtn, trUtf8( "Load external sample" ) );
	connect(loadBtn, SIGNAL(clicked(Button*)), this, SLOT(click(Button*)));

	// Mute button
	string mute_on_path = string(IMG_PATH).append("/img/mixerPanel/btn_mute_on.png");
	string mute_off_path = string(IMG_PATH).append("/img/mixerPanel/btn_mute_off.png");
	string mute_over_path = string(IMG_PATH).append("/img/mixerPanel/btn_mute_over.png");
	muteBtn = new ToggleButton(this, QSize( 18, 12 ), mute_on_path, mute_off_path, mute_over_path);
	muteBtn->move( 2, 20 );
	QToolTip::add( muteBtn, trUtf8( "Mute" ) );
	connect(muteBtn, SIGNAL(clicked(Button*)), this, SLOT(click(Button*)));

	// Solo button
	string solo_on_path = string(IMG_PATH).append("/img/mixerPanel/btn_solo_on.png");
	string solo_off_path = string(IMG_PATH).append("/img/mixerPanel/btn_solo_off.png");
	string solo_over_path = string(IMG_PATH).append("/img/mixerPanel/btn_solo_over.png");
	soloBtn = new ToggleButton(this, QSize( 18, 12 ), solo_on_path, solo_off_path, solo_over_path);
	soloBtn->move( 22, 20);
	QToolTip::add( soloBtn, trUtf8( "Solo" ) );
	connect(soloBtn, SIGNAL(clicked(Button*)), this, SLOT(click(Button*)));

	// pan fader
	panFader = new PanFader(this);
	panFader->move( 2, 35 );
	QToolTip::add( panFader, trUtf8( "Pan" ) );
	connect( panFader, SIGNAL( valueChanged(PanFader*) ), this, SLOT( panFaderChanged(PanFader*) ) );

	for (uint i = 0; i < MAX_FX; i++) {
		m_pKnob[i] = new Knob(this);
		if ( (i % 2) == 0 ) {
			m_pKnob[i]->move( 2, 50 + (15 * i) );
		}
		else {
			m_pKnob[i]->move( 22, 50 + (15 * i) );
		}
		connect( m_pKnob[i], SIGNAL( valueChanged(Knob*) ), this, SLOT( knobChanged(Knob*) ) );
	}

	PreferencesMng *pref = PreferencesMng::getInstance();
	QString family = pref->getMixerFontFamily().c_str();
	int size = pref->getMixerFontPointSize();
	QFont mixerFont( family, size );


	string mixerline_text_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_text_background.png");
	QPixmap textBackground;
	bool ok = textBackground.load( mixerline_text_path .c_str() );
	if( ok == false ){
		qWarning( "MixerLine: Error loading pixmap" );
	}

	// peak volume label
	QFont font1( family, size );
	volumeLbl = new QLabel( this );
	volumeLbl->move( 2, MIXERLINE_LABEL_H );
	volumeLbl->resize( 38, 20 );
	volumeLbl->setText( "0.0" );
	volumeLbl->setAlignment( AlignVCenter | AlignHCenter );
	volumeLbl->setBackgroundPixmap( textBackground );
	volumeLbl->setFont( font1 );
	QToolTip::add( volumeLbl, trUtf8( "Peak volume" ) );

	// instrument name widget
	nameWidget = new InstrumentNameWidget( this );
	nameWidget->move( 2, MIXERLINE_FADER_H );
	QToolTip::add( nameWidget, trUtf8( "Instrument name (double click to edit)" ) );
	connect( nameWidget, SIGNAL( doubleClicked () ), this, SLOT( nameClicked() ) );
	connect( nameWidget, SIGNAL( clicked () ), this, SLOT( nameSelected() ) );

	// fader
	fader = new Fader(this);
	fader->move( 22, MIXERLINE_FADER_H );
	connect( fader, SIGNAL( valueChanged(Fader*) ), this, SLOT( faderChanged(Fader*) ) );
}




/**
 *
 */
MixerLine::~MixerLine() {
//	cout << "MixerLine destroy" << endl;
	delete fader;
	delete m_pMixerLineBackground;
	m_pMixerLineBackground = NULL;
}




/**
 *
 */
void MixerLine::updateMixerLine() {
	fader->updateFader();
	panFader->updateFader();

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		m_pKnob[nFX]->updateKnob();
	}
}





/**
 *
 */
void MixerLine::click(Button *ref) {
	Song *song = (Hydrogen::getInstance())->getSong();

	if (ref == muteBtn) {
		song->setModified(true);
		emit muteBtnClicked(this);
	}
	else if (ref == soloBtn) {
		song->setModified(true);
		emit soloBtnClicked(this);
	}
	else if (ref == loadBtn) {
		song->setModified(true);
		emit loadBtnClicked(this);
	}
	else if (ref == playSampleBtn) {
		emit noteOnClicked(this);
	}
}





/**
 *
 */
void MixerLine::rightClick(Button *ref)
{
	if (ref == playSampleBtn) {
		emit noteOffClicked(this);
	}

}



/**
 *
 */
void MixerLine::faderChanged(Fader *ref) {
	maxPeak = 0.0f;
	char tmp[20];
	sprintf(tmp, "%#.1f", maxPeak);
	volumeLbl->setText(tmp);

	Song *song = (Hydrogen::getInstance())->getSong();
	song->setModified(true);
	emit volumeChanged(this);

	char faderPos[100];
	float value = ref->getValue() / 100.0f;
	sprintf( faderPos, "%#.2f",  value);
	( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Set instrument volume [%1]" ).arg( faderPos ), 2000 );
}





/**
 *
 */
bool MixerLine::isMuteClicked() {
	return muteBtn->isPressed();
}





/**
 *
 */
void MixerLine::setMuteClicked(bool isClicked) {
	muteBtn->setPressed(isClicked);
}




/**
 *
 */
bool MixerLine::isSoloClicked() {
	return soloBtn->isPressed();
}




/**
 *
 */
void MixerLine::setSoloClicked(bool isClicked) {
	soloBtn->setPressed(isClicked);
}




/**
 *
 */
float MixerLine::getVolume() {
	return (fader->getValue() / 100.0);
}



/**
 *
 */
void MixerLine::setVolume(float value) {
	fader->setValue((int)(value * 100.0));
}



void MixerLine::setPeak_L( float peak ) {
	if (peak != getPeak_L() ) {
		fader->setPeak_L( peak );
		if (peak > maxPeak) {
			char tmp[20];
			sprintf(tmp, "%#.1f", peak);
			volumeLbl->setText(tmp);
			maxPeak = peak;
		}
	}
}



/**
 *
 */
float MixerLine::getPeak_L() {
	return fader->getPeak_L();
}




/**
 *
 */
void MixerLine::setPeak_R( float peak ) {
	if (peak != getPeak_R() ) {
		fader->setPeak_R( peak );
		if (peak > maxPeak) {
			char tmp[20];
			sprintf(tmp, "%#.1f", peak);
			volumeLbl->setText(tmp);
			maxPeak = peak;
		}
	}
}



/**
 *
 */
float MixerLine::getPeak_R() {
	return fader->getPeak_R();
}





void MixerLine::nameClicked() {
	emit instrumentNameClicked(this);
}


void MixerLine::nameSelected() {
	emit instrumentNameSelected(this);
}





/**
 *
 */
void MixerLine::panFaderChanged(PanFader *ref) {
	Song *song = ( Hydrogen::getInstance() )->getSong();
	song->setModified( true );
	emit panChanged( this );

	float panValue = ref->getValue();
	float pan_L = (100.0 - panValue) / 100.0;
	float pan_R = panValue / 100.0;

	char faderPos[100];
	sprintf( faderPos, "%#.2fL, %#.2fR",  pan_L, pan_R);
	( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Set instrument pan [%1]" ).arg( faderPos ), 2000 );

	QToolTip::remove( panFader );
	QToolTip::add( panFader, QString("Pan ") + QString( faderPos ) );

}





/**
 *
 */
int MixerLine::getPan() {
	return (int)( panFader->getValue() );
}



/**
 *
 */
void MixerLine::setPan(int value) {
	if ( value != panFader->getValue() ) {
		panFader->setValue( value );

		float pan_L = (100.0 - value) / 100.0;
		float pan_R = value / 100.0;
		char faderPos[100];
		sprintf( faderPos, "Pan %#.2fL, %#.2fR",  pan_L, pan_R);
		QToolTip::remove( panFader );
		QToolTip::add( panFader, QString( faderPos ) );
	}
}


void MixerLine::setPlayClicked( bool clicked ) {
	playSampleBtn->setPressed( clicked );
}


void MixerLine::knobChanged(Knob* pRef)
{
//	infoLog( "knobChanged" );
	for (uint i = 0; i < MAX_FX; i++) {
		if (m_pKnob[i] == pRef) {
			emit knobChanged( this, i );
			break;
		}
	}
}


void MixerLine::setFXLevel( uint nFX, float fValue )
{
	if (nFX > MAX_FX) {
		errorLog( "[setFXLevel] nFX > MAX_FX (nFX=" + toString(nFX) + ")" );
	}
	m_pKnob[nFX]->setValue( fValue );
}

float MixerLine::getFXLevel(uint nFX)
{
	if (nFX > MAX_FX) {
		errorLog( "[setFXLevel] nFX > MAX_FX (nFX=" + toString(nFX) + ")" );
	}
	return m_pKnob[nFX]->getValue();
}


void MixerLine::setSelected( bool bIsSelected )
{
	if (m_bIsSelected == bIsSelected )	return;

	m_bIsSelected = bIsSelected;
	if (m_bIsSelected) {
		setBackgroundPixmap( *m_pMixerLineBackground_selected );
	}
	else {
		setBackgroundPixmap( *m_pMixerLineBackground );
	}

}



/////////////


/**
 * Constructor
 */
MasterMixerLine::MasterMixerLine(QWidget* parent) : QWidget(parent), Object( "MasterMixerLine" )
{
//	cout << "mixerLine init" << endl;
	m_nWidth = MIXERLINE_WIDTH;
	m_nHeight = MIXERLINE_HEIGHT;

	setMinimumSize( m_nWidth, m_nHeight );
	setMaximumSize( m_nWidth, m_nHeight );
	resize( m_nWidth, m_nHeight );

	// Background image
	string background_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_background.png");
	bool ok = background.load(background_path.c_str());
	if( ok == false ){
		errorLog( string("Error loading pixmap ") + background_path );
	}
	setBackgroundPixmap(background);

	// MixerLine LABEL Background image
	QPixmap mixerLineLabelBackground;
	string mixerLineLabelBackground_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_label_background.png");
	ok = mixerLineLabelBackground.load(mixerLineLabelBackground_path.c_str());
	if( ok == false ){
		errorLog( string( "Error loading pixmap " ) + mixerLineLabelBackground_path );
	}

	// Text background
	string mixerline_text_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_text_background.png");
	QPixmap textBackground;
	ok = textBackground.load( mixerline_text_path .c_str() );
	if( ok == false ){
		errorLog( string("Error loading pixmap " ) + mixerline_text_path );
	}


	PreferencesMng *pref = PreferencesMng::getInstance();
	int size = pref->getMixerFontPointSize();
	QString family = pref->getMixerFontFamily().c_str();
	QFont font1( family, size );
	// peak volume label
	leftVolumeLbl = new QLabel(this);
	leftVolumeLbl->move( 2, MIXERLINE_LABEL_H );
	leftVolumeLbl->resize( 38, 20 );
	leftVolumeLbl->setText( "0.0" );
	leftVolumeLbl->setAlignment( AlignVCenter | AlignHCenter );
	leftVolumeLbl->setBackgroundPixmap( textBackground );
	leftVolumeLbl->setFont( font1 );
	QToolTip::add( leftVolumeLbl, trUtf8( "Peak volume (left channel)" ) );

	fader = new Fader( this );
	fader->move( 22, MIXERLINE_FADER_H );
	connect( fader, SIGNAL( valueChanged(Fader*) ), this, SLOT( faderChanged(Fader*) ) );

	QFont mixerFont( family, size );

	InstrumentNameWidget *nameWidget;
	// instrument name widget
	nameWidget = new InstrumentNameWidget( this );
	nameWidget->move( 2, MIXERLINE_FADER_H );
	nameWidget->setText( trUtf8( "Master output" ) );
}




/**
 *
 */
MasterMixerLine::~MasterMixerLine() {
//	cout << "MixerLine destroy" << endl;
	maxPeak = 0.0;
}





/**
 *
 */
void MasterMixerLine::faderChanged(Fader *ref) {
	maxPeak = 0.0;

	char tmp[20];
	sprintf(tmp, "%#.1f", maxPeak);
	leftVolumeLbl->setText(tmp);
	fader->setValue(ref->getValue());

	emit volumeChanged(this);

	Song *song = (Hydrogen::getInstance())->getSong();
	song->setModified(true);

	char faderPos[100];
	float value = ref->getValue() / 100.0;
	sprintf( faderPos, "%#.2f",  value);
	( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Set master volume [%1]" ).arg( faderPos ), 2000 );

}




/**
 *
 */
float MasterMixerLine::getVolume() {
	return (fader->getValue() / 100.0);
}



/**
 *
 */
void MasterMixerLine::setVolume(float value) {
	fader->setValue((int)(value * 100.0));
//	rightFader->setValue((int)(value * 100.0));
}




/**
 *
 */
void MasterMixerLine::setPeak_L(float peak) {
	if ( peak != getPeak_L() ) {
		fader->setPeak_L(peak);
		if (peak > maxPeak) {
			char tmp[20];
			sprintf(tmp, "%#.1f", peak);
			leftVolumeLbl->setText(tmp);
			maxPeak = peak;
		}
	}
}




/**
 *
 */
float MasterMixerLine::getPeak_L() {
	return fader->getPeak_L();
}




/**
 *
 */
void MasterMixerLine::setPeak_R(float peak) {
	if ( peak != getPeak_R() ) {
		fader->setPeak_R(peak);
		if (peak > maxPeak) {
			char tmp[20];
			sprintf(tmp, "%#.1f", peak);
			leftVolumeLbl->setText(tmp);
			maxPeak = peak;
		}
	}
}





/**
 *
 */
float MasterMixerLine::getPeak_R() {
	return fader->getPeak_R();
}




/**
 *
 */
void MasterMixerLine::updateMixerLine() {
	fader->updateFader();
}




/////////////////////////////////////////



/**
 * Constructor
 */
FxMixerLine::FxMixerLine(QWidget* parent) : QWidget(parent), Object( "FxMixerLine" )
{
//	cout << "mixerLine init" << endl;

	m_nWidth = MIXERLINE_WIDTH;
	m_nHeight = MIXERLINE_HEIGHT;

	setMinimumSize( m_nWidth, m_nHeight );
	setMaximumSize( m_nWidth, m_nHeight );
	resize( m_nWidth, m_nHeight );

	maxPeak = 0.0;

	// MixerLine Background image
	string mixerLineBackground_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_background.png");
	bool ok = mixerLineBackground.load(mixerLineBackground_path.c_str());
	if( ok == false ){
		errorLog( string("Error loading pixmap ") + mixerLineBackground_path );
	}
	setBackgroundPixmap(mixerLineBackground);

	// MixerLine LABEL Background image
	QPixmap mixerLineLabelBackground;
	string mixerLineLabelBackground_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_label_background.png");
	ok = mixerLineLabelBackground.load(mixerLineLabelBackground_path.c_str());
	if( ok == false ){
		errorLog( string("Error loading pixmap ") + mixerLineLabelBackground_path );
	}

	string mixerline_text_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_text_background.png");
	QPixmap textBackground;
	ok = textBackground.load( mixerline_text_path.c_str() );
	if( ok == false ){
		errorLog( string("Error loading pixmap ") + mixerline_text_path );
	}


	// active button
	string activeBtn_on_path = string(IMG_PATH).append( "/img/mixerPanel/btn_on_on.png" );
	string activeBtn_off_path = string(IMG_PATH).append( "/img/mixerPanel/btn_on_off.png" );
	string activeBtn_over_path = string(IMG_PATH).append( "/img/mixerPanel/btn_on_over.png" );
	activeBtn = new ToggleButton( this, QSize( 18, 12 ), activeBtn_on_path, activeBtn_off_path, activeBtn_over_path);
	activeBtn->move( 2, 5 );
	QToolTip::add( activeBtn, trUtf8( "FX on/off") );
	connect( activeBtn, SIGNAL( clicked(Button*) ), this, SLOT( click(Button*) ) );

	PreferencesMng *pref = PreferencesMng::getInstance();

	// peak volume label
	QString family = pref->getMixerFontFamily().c_str();
	int size = pref->getMixerFontPointSize();
	QFont font1( family, size);
	volumeLbl = new QLabel( this );
	volumeLbl->move( 2, MIXERLINE_LABEL_H );
	volumeLbl->resize( 38, 18 );
	volumeLbl->setText( "0.0" );
	volumeLbl->setAlignment( AlignVCenter | AlignHCenter );
	volumeLbl->setBackgroundPixmap( textBackground );
	volumeLbl->setFont( font1 );
//	QToolTip::add( volumeLbl, "Peak volume" );

	// fader
	fader = new Fader( this );
	fader->move( 22, MIXERLINE_FADER_H );
	connect( fader, SIGNAL( valueChanged(Fader*) ), this, SLOT( faderChanged(Fader*) ) );

	QFont mixerFont( family, size );


	nameWidget = new InstrumentNameWidget( this );
	nameWidget->move( 2, MIXERLINE_FADER_H );
	nameWidget->setText( trUtf8( "Master output" ) );
}




/**
 *
 */
FxMixerLine::~FxMixerLine() {
//	cout << "MixerLine destroy" << endl;
	delete fader;
}





/**
 *
 */
void FxMixerLine::click(Button *ref) {
	Song *song = (Hydrogen::getInstance())->getSong();

	if (ref == activeBtn ) {
		song->setModified( true );
		emit activeBtnClicked( this );
	}
}




/**
 *
 */
void FxMixerLine::faderChanged(Fader *ref) {
	maxPeak = 0.0;
	char tmp[20];
	sprintf( tmp, "%#.1f", maxPeak );
	volumeLbl->setText( tmp );

	Song *song = ( Hydrogen::getInstance() )->getSong();
	song->setModified( true );
	emit volumeChanged( this );

}




/**
 *
 */
float FxMixerLine::getVolume() {
	return (fader->getValue() / 100.0);
}





/**
 *
 */
void FxMixerLine::setVolume( float value ) {
	fader->setValue( (int)(value * 100.0) );
}




/**
 *
 */
void FxMixerLine::setPeak_L( float peak ) {
	if (peak != getPeak_L() ) {
		fader->setPeak_L( peak );
		if (peak > maxPeak) {
			char tmp[20];
			sprintf(tmp, "%#.1f", peak);
			volumeLbl->setText(tmp);
			maxPeak = peak;
		}
	}
}



/**
 *
 */
float FxMixerLine::getPeak_L() {
	return fader->getPeak_L();
}





/**
 *
 */
void FxMixerLine::setPeak_R(float peak) {
	if (peak != getPeak_R() ) {
		fader->setPeak_R( peak );
		if (peak > maxPeak) {
			char tmp[20];
			sprintf(tmp, "%#.1f", peak);
			volumeLbl->setText(tmp);
			maxPeak = peak;
		}
	}
}





/**
 *
 */
float FxMixerLine::getPeak_R() {
	return fader->getPeak_R();
}




bool FxMixerLine::isFxActive() {
	return activeBtn->isPressed();
}




void FxMixerLine::setFxActive( bool active ) {
	activeBtn->setPressed( active );
}



/**
 *
 */
void FxMixerLine::updateMixerLine() {
	fader->updateFader();
}






////////////////////////////////

QPixmap* InstrumentNameWidget::m_pBackground = NULL;


InstrumentNameWidget::InstrumentNameWidget(QWidget* parent)  : QWidget(parent), Object( "InstrumentNameWidget" )
{
//	infoLog( "INIT" );

	m_bChanged = true;
	m_nWidgetWidth = 18;
	m_nWidgetHeight = 100;

	PreferencesMng *pref = PreferencesMng::getInstance();
	QString family = pref->getMixerFontFamily().c_str();
	int size = pref->getMixerFontPointSize();
	m_mixerFont.setFamily( family );
	m_mixerFont.setPointSize( size );
//	m_mixerFont.setBold( true );
//	m_mixerFont.setItalic( true );

	// MixerLine LABEL Background image
	if (m_pBackground == NULL) {
//		infoLog( "loading background pixmap" );
		string mixerLineLabelBackground_path = string(IMG_PATH).append( "/img/mixerPanel/mixerline_label_background.png" );
		m_pBackground = new QPixmap();
		bool ok = m_pBackground->load( mixerLineLabelBackground_path.c_str() );
		if( ok == false ){
			errorLog( string("Error loading pixmap ") + mixerLineLabelBackground_path );
		}
	}

	m_temp.resize( m_nWidgetWidth, m_nWidgetHeight );
	this->resize( m_nWidgetWidth, m_nWidgetHeight );
}


InstrumentNameWidget::~InstrumentNameWidget()
{
//	infoLog( "DESTROY" );
//	delete m_pBackground;
//	m_pBackground = NULL;
}



void InstrumentNameWidget::paintEvent( QPaintEvent* ev ) {
	if (!isVisible()) {
		return;
	}

	if (m_bChanged) {
		m_bChanged = false;
		//cout << "----------- InstrumentNameWidget PAINT --------------" << endl;

		bitBlt(&m_temp, 0, 0, m_pBackground, 0, 0, m_nWidgetWidth, m_nWidgetHeight, CopyROP);
		QPainter p( &m_temp );
		p.setPen( QColor(0, 0, 0) );
		p.setFont( m_mixerFont );
		p.rotate( -90 );
		p.drawText( -95, 15, m_sInstrName );

		setErasePixmap( m_temp );

		// copio l'immagine definitiva
		bitBlt( this, 0, 0, &m_temp, 0, 0, m_nWidgetWidth, m_nWidgetHeight, CopyROP );
	}
}




void InstrumentNameWidget::setText( QString text )
{
	if (m_sInstrName != text ) {
		m_sInstrName = text;
		m_bChanged = true;
		update();
	}
}

QString InstrumentNameWidget::text()
{
	return m_sInstrName;
}


void InstrumentNameWidget::mousePressEvent( QMouseEvent * e ) {
	emit clicked();
}

void InstrumentNameWidget::mouseDoubleClickEvent( QMouseEvent * e ) {
	emit doubleClicked();
}



/////////////






////////////////

QPixmap* LadspaFXMixerLine::m_pMixerLineBackground = NULL;

LadspaFXMixerLine::LadspaFXMixerLine(QWidget* parent) : QWidget( parent ), Object( "LadspaFXMixerLine" )
{
//	infoLog( "INIT" );

	resize( MIXERLINE_WIDTH, MIXERLINE_HEIGHT );
	setMinimumSize( width(), height() );
	setMaximumSize( width(), height() );

	// MixerLine Background image
	if (m_pMixerLineBackground == NULL ) {
//		infoLog( "loading background pixmap" );
		string mixerLineBackground_path = string(IMG_PATH).append("/img/mixerPanel/mixerline_background.png");
		m_pMixerLineBackground = new QPixmap();
		bool ok = m_pMixerLineBackground->load(mixerLineBackground_path.c_str());
		if( ok == false ){
			errorLog( string("Error loading pixmap ") + mixerLineBackground_path );
		}
	}
	setBackgroundPixmap( *m_pMixerLineBackground );

	// active button
	string activeBtn_on_path = string(IMG_PATH).append( "/img/mixerPanel/btn_on_on.png" );
	string activeBtn_off_path = string(IMG_PATH).append( "/img/mixerPanel/btn_on_off.png" );
	string activeBtn_over_path = string(IMG_PATH).append( "/img/mixerPanel/btn_on_over.png" );
	m_pActiveBtn = new ToggleButton( this, QSize( 18, 12 ), activeBtn_on_path, activeBtn_off_path, activeBtn_over_path);
	m_pActiveBtn->move( 2, 5 );
	QToolTip::add( m_pActiveBtn, trUtf8( "FX on/off") );
	connect( m_pActiveBtn, SIGNAL( clicked(Button*) ), this, SLOT( click(Button*) ) );

	// edit button
	string editBtn_on_path = string(IMG_PATH).append( "/img/mixerPanel/btn_edit_on.png" );
	string editBtn_off_path = string(IMG_PATH).append( "/img/mixerPanel/btn_edit_off.png" );
	string editBtn_over_path = string(IMG_PATH).append( "/img/mixerPanel/btn_edit_over.png" );
	m_pEditBtn = new ToggleButton( this, QSize( 18, 12 ), editBtn_on_path, editBtn_off_path, editBtn_over_path);
	m_pEditBtn->move( 22, 5 );
	QToolTip::add( m_pEditBtn, trUtf8( "Edit FX parameters") );
	connect( m_pEditBtn, SIGNAL( clicked(Button*) ), this, SLOT( click(Button*) ) );

	// instrument name widget
	m_pNameWidget = new InstrumentNameWidget( this );
	m_pNameWidget->move( 2, MIXERLINE_FADER_H );
	m_pNameWidget->setText( "No name" );
	QToolTip::add( m_pNameWidget, trUtf8( "Ladspa FX name (double click to change FX)" ) );
	connect( m_pNameWidget, SIGNAL( doubleClicked () ), this, SLOT( nameClicked() ) );

	// fader
	m_pFader = new Fader(this);
	m_pFader->move( 22, MIXERLINE_FADER_H );
	connect( m_pFader, SIGNAL( valueChanged(Fader*) ), this, SLOT( faderChanged(Fader*) ) );

}


LadspaFXMixerLine::~LadspaFXMixerLine()
{
//	infoLog( "DESTROY" );
	delete m_pMixerLineBackground;
	m_pMixerLineBackground  = NULL;
}




void LadspaFXMixerLine::nameClicked()
{
	emit( ladspaNameClicked( this ) );
}


/**
 *
 */
void LadspaFXMixerLine::click(Button *ref) {
	if ( ref == m_pActiveBtn ) {
		emit activeBtnClicked( this );
	}
	else if( ref == m_pEditBtn ) {
		emit editBtnClicked( this );
	}
}



bool LadspaFXMixerLine::isFxActive() {
	return m_pActiveBtn->isPressed();
}




void LadspaFXMixerLine::setFxActive( bool active ) {
	m_pActiveBtn->setPressed( active );
}



void LadspaFXMixerLine::faderChanged(Fader *ref) {
	m_fMaxPeak = 0.0;
// FIXME: aggiungere il volumeLabel
//	char tmp[20];
//	sprintf(tmp, "%#.1f", fMaxPeak);
//	volumeLbl->setText(tmp);

	Song *song = (Hydrogen::getInstance())->getSong();
	song->setModified(true);
	emit volumeChanged(this);
}


void LadspaFXMixerLine::setPeaks( float fPeak_L, float fPeak_R )
{
	m_pFader->setPeak_L( fPeak_L );
	m_pFader->setPeak_R( fPeak_R );
	m_pFader->updateFader();
}


void LadspaFXMixerLine::getPeaks( float *fPeak_L, float *fPeak_R )
{
	(*fPeak_L) = m_pFader->getPeak_L();
	(*fPeak_R) = m_pFader->getPeak_R();
}


/**
 *
 */
float LadspaFXMixerLine::getVolume() {
	return (m_pFader->getValue() / 100.0);
}



/**
 *
 */
void LadspaFXMixerLine::setVolume(float value) {
	m_pFader->setValue((int)(value * 100.0));
	m_pFader->updateFader();
}




