/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <errno.h>
#include <hulautil.h>
#include <connio.h>

#include <mdb.h>
#include <msgapi.h>
#include <management.h>

#include "hulastats.h"

HSGlobals HulaStats;

void 
StatPrintNumber(unsigned long number, unsigned char *buffer)
{
    register long x;
    register long len;

    if (number < 1000) {
        len = sprintf(buffer, "        %3lu", number);
    } else if (number < 1000000) {
        x = number % 1000;
        len = sprintf(buffer, "    %3lu,%03lu", (number - x) / 1000, x);
    } else {
        x = number / 1000000;
        len = sprintf(buffer, "%3lu,%03lu,%03lu",  number / 1000000, (number - x * 1000000) / 1000, number % 1000);
    }

    return;
}

void 
StatPrintFloat(unsigned long number, unsigned long divideBy, unsigned char *buffer)
{
    register long x;
    register long y;

    if (divideBy == 0) {
        x = 0;
        y = 0;
    } else {
        x = (number * 100) / divideBy;
        y = x % 100;
    }

    sprintf(buffer, "%3lu.%02lu", x / 100, y);

    return;
}


void 
StatPrintTime(unsigned long seconds, unsigned char *buffer)
{
    long d;
    long h;
    long m;
    long s;
    long a;
    long b;

    d = seconds / 86400;
    a = d * 86400;
    h = (seconds - a) / 3600;
    b = h * 3600;
    m = (seconds - (a + b)) / 60;
    s = seconds - (a + b + m * 60);

    sprintf(buffer, "%3lud %02luh %02lum %02lus", d, h, m, s);

    return;
}

int 
LoadStatistics(StatisticsStruct *stats)
{
    int ccode;
    unsigned long i;
    unsigned long *statistic;

    if (HulaStats.selected.conn) {
        if (((ccode = ConnWrite(HulaStats.selected.conn, "STAT0\r\n", 7)) != -1) 
                && ((ccode = ConnFlush(HulaStats.selected.conn)) != -1) 
                && ((ccode = ConnReadLine(HulaStats.selected.conn, HulaStats.selected.line, CONN_BUFSIZE)) != -1) 
                && (XplStrNCaseCmp(HulaStats.selected.line, DMCMSG1001RESPONSES_COMMING, 5) == 0) 
                && (atol(HulaStats.selected.line + 5) == (sizeof(StatisticsStruct) / sizeof(unsigned long)))) {
            statistic = &(stats->ModulesLoaded);

            for (i = 0; (ccode != -1) && (i < sizeof(StatisticsStruct) / sizeof(unsigned long)); i++, statistic++) {
                if ((ccode = ConnReadLine(HulaStats.selected.conn, HulaStats.selected.line, CONN_BUFSIZE)) != -1) {
                    *statistic = atol(HulaStats.selected.line);
                }
            }

            if (ccode != -1) {
                if (i == sizeof(StatisticsStruct) / sizeof(unsigned long)) {
                    ccode = ConnRead(HulaStats.selected.conn, HulaStats.selected.line, CONN_BUFSIZE);
                } else {
                    ccode = -1;
                }
            }
        } else if (ccode != -1) {
            ccode = -1;
        }
    } else {
        ccode = -1;
    }

    return(ccode);
}

int 
LoadSpamStatistics(AntispamStatisticsStruct *spamStats)
{
    int ccode;
    unsigned long i;
    unsigned long *statistic;

    if (HulaStats.selected.conn) {
        if (((ccode = ConnWrite(HulaStats.selected.conn, "STAT1\r\n", 7)) != -1) 
                && ((ccode = ConnFlush(HulaStats.selected.conn)) != -1) 
                && ((ccode = ConnReadLine(HulaStats.selected.conn, HulaStats.selected.line, CONN_BUFSIZE)) != -1) 
                && (XplStrNCaseCmp(HulaStats.selected.line, DMCMSG1001RESPONSES_COMMING, 5) == 0) 
                && (atol(HulaStats.selected.line + 5) == (sizeof(AntispamStatisticsStruct) / sizeof(unsigned long)))) {
            statistic = &(spamStats->QueueSpamBlocked);

            for (i = 0; (ccode != -1) && (i < sizeof(AntispamStatisticsStruct) / sizeof(unsigned long)); i++, statistic++) {
                if ((ccode = ConnReadLine(HulaStats.selected.conn, HulaStats.selected.line, CONN_BUFSIZE)) != -1) {
                    *statistic = atol(HulaStats.selected.line);
                }
            }

            if (ccode != -1) {
                if (i == sizeof(AntispamStatisticsStruct) / sizeof(unsigned long)) {
                    ccode = ConnRead(HulaStats.selected.conn, HulaStats.selected.line, CONN_BUFSIZE);
                } else {
                    ccode = -1;
                }
            }
        } else if (ccode != -1) {
            ccode = -1;
        }
    } else {
        ccode = -1;
    }

    return(ccode);
}

int 
ReleaseMessagingServer(void)
{
    int ccode = 0;

    if (HulaStats.selected.conn) {
        if ((ccode = ConnWrite(HulaStats.selected.conn, "QUIT\r\n", 6)) != -1) {
            ccode = ConnFlush(HulaStats.selected.conn);
        }

        ConnClose(HulaStats.selected.conn, 0);
        ConnFree(HulaStats.selected.conn);

        HulaStats.selected.conn = NULL;
    }

    return(ccode);
}

int 
ConnectToMessagingServer(unsigned char *dn)
{
    int ccode;
    unsigned long used;
    unsigned long port[2];
    MDBValueStruct *config;

    if (HulaStats.selected.conn) {
        ReleaseMessagingServer();
    }

    port[0] = DMC_MANAGER_SSL_PORT;
    port[1] = DMC_MANAGER_PORT;

    HulaStats.selected.conn = ConnAlloc(TRUE);

    config = MDBCreateValueStruct(HulaStats.directory, NULL);
    if (config) {
        MDBRead(dn, MSGSRV_A_CONFIGURATION, config);
    } else {
        return(FALSE);
    }

    for (used = 0; used < config->Used; used++) {
        if (XplStrNCaseCmp(config->Value[used], "DMC: ManagerPort=", 17) == 0) {
            if (isdigit(config->Value[used][17])) {
                port[1] = atol(config->Value[used] + 17);
            } else {
                port[1] = -1;
            }
        } else if (XplStrNCaseCmp(config->Value[used], "DMC: ManagerSSLPort=", 20) == 0) {
            if (isdigit(config->Value[used][20])) {
                port[0] = atol(config->Value[used] + 20);
            } else {
                port[0] = -1;
            }
        }
    }

    MDBFreeValues(config);

    if (((port[0] != (unsigned long)-1) || (port[1] != (unsigned long)-1)) && (MDBRead(dn, MSGSRV_A_IP_ADDRESS, config))) {
        ccode = 0;

        HulaStats.selected.conn->socketAddress.sin_family = AF_INET;
        HulaStats.selected.conn->socketAddress.sin_addr.s_addr = inet_addr(config->Value[0]);
    } else {
        ccode = -1;
    }

    MDBDestroyValueStruct(config);
    config = NULL;

    if (!ccode) {
        if (HulaStats.ssl.enable && ((HulaStats.selected.conn->socketAddress.sin_port = htons((unsigned short)port[0])) != (unsigned short)-1)) {
            ccode = ConnConnect(HulaStats.selected.conn, NULL, 0, HulaStats.ssl.context);
        } else if ((HulaStats.selected.conn->socketAddress.sin_port = htons((unsigned short)port[1])) != (unsigned short)-1) {
            ccode = ConnConnect(HulaStats.selected.conn, NULL, 0, NULL);
        } else {
            ccode = -1;
        }

        if (ccode != -1) {
            ccode = ConnRead(HulaStats.selected.conn, HulaStats.selected.line, CONN_BUFSIZE);
        }
    }

    return(ccode);
}

static BOOL 
ReadConfiguration(int argc, char *argv[])
{
    BOOL result = FALSE;
    MDBValueStruct *config;

    MDBGetServerInfo(HulaStats.dn, NULL, NULL);

    config = MDBCreateValueStruct(HulaStats.directory, NULL);
    if (config) {
        if (MDBReadDN(HulaStats.dn, MSGSRV_A_HULA_MESSAGING_SERVER, config)) {
            strcpy(HulaStats.serverDN, config->Value[0]);

            result = TRUE;
        }

        MDBDestroyValueStruct(config);
    }

    return(result);
}

#if defined(NETWARE) || defined(LIBC) || defined(WIN32)
static int 
_NonAppCheckUnload(void)
{
    static BOOL checked = FALSE;

    if (!checked) {
        checked = TRUE;
        HulaStats.state = HULASTATS_STATE_UNLOADING;

        XplWaitOnLocalSemaphore(HulaStats.sem.shutdown);

        XplWaitOnLocalSemaphore(HulaStats.sem.main);
    }

    return(0);
}
#endif

static void 
SignalHandler(int sigtype)
{
    switch(sigtype) {
        case SIGHUP: {
            if (HulaStats.state < HULASTATS_STATE_UNLOADING) {
                HulaStats.state = HULASTATS_STATE_UNLOADING;
            }

            break;
        }

        case SIGINT:
        case SIGTERM: {
            if (HulaStats.state == HULASTATS_STATE_STOPPING) {
                XplUnloadApp(getpid());
            } else if (HulaStats.state < HULASTATS_STATE_STOPPING) {
                HulaStats.state = HULASTATS_STATE_STOPPING;
            }

            break;
        }

        default: {
            break;
        }
    }

    return;
}

int 
HulaStatsInitialize(int argc, char *argv[])
{
    XplSignalHandler(SignalHandler);
	XplRenameThread(GetThreadID(), "Hula Statistics Utility");

    HulaStats.id.main = XplGetThreadID();
    HulaStats.id.group = XplGetThreadGroupID();

    HulaStats.state = HULASTATS_STATE_INITIALIZING;

    HulaStats.selected.conn = NULL;
    HulaStats.selected.dn[0] = '\0';
    HulaStats.selected.rdn[0] = '\0';

    HulaStats.ssl.conn = NULL;
    HulaStats.ssl.enable = FALSE;
    HulaStats.ssl.context = NULL;
    HulaStats.ssl.config.options = 0;

    HulaStats.directory = NULL;

    HulaStats.start = time(NULL);

    if (MemoryManagerOpen("HulaStats") == TRUE) {
        XplOpenLocalSemaphore(HulaStats.sem.main, 0);
        XplOpenLocalSemaphore(HulaStats.sem.shutdown, 1);
    } else {
        XplConsolePrintf("hulastats: Unable to initialize memory manager; shutting down.\r\n");
        return(-1);
    }

    ConnStartup(3 * 60, TRUE);

    MDBInit();
    HulaStats.directory = (MDBHandle)MsgInit();
    if (HulaStats.directory == NULL) {
        XplBell();
        XplConsolePrintf("hulastats: Invalid directory credentials; exiting!\r\n");
        XplBell();

        MemoryManagerClose("HulaStats");

        return(-1);
    }

    SetCurrentNameSpace(NWOS2_NAME_SPACE);
    SetTargetNameSpace(NWOS2_NAME_SPACE);

    if (ReadConfiguration(argc, argv)) {
        HulaStats.ssl.enable = FALSE;
        HulaStats.ssl.config.method = SSLv23_client_method;
        HulaStats.ssl.config.options = SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS;
        HulaStats.ssl.config.mode = SSL_MODE_AUTO_RETRY;
        HulaStats.ssl.config.cipherList = NULL;
        HulaStats.ssl.config.certificate.type = SSL_FILETYPE_PEM;
        HulaStats.ssl.config.certificate.file = MsgGetTLSCertPath(NULL);
        HulaStats.ssl.config.key.type = SSL_FILETYPE_PEM;
        HulaStats.ssl.config.key.file = MsgGetTLSKeyPath(NULL);

        HulaStats.ssl.context = ConnSSLContextAlloc(&(HulaStats.ssl.config));
        if (HulaStats.ssl.context) {
            HulaStats.ssl.enable = TRUE;
        }

        HulaStats.state = HULASTATS_STATE_RUNNING;

        if ((HulaStatsUIStart() == 0) && (ConnectToMessagingServer(HulaStats.serverDN) != -1)) {
            HulaStatsUIRun();
            HulaStatsUIStop();
        }

        ReleaseMessagingServer();
    } else {
        HulaStats.state = HULASTATS_STATE_STOPPING;
    }

    return(0);
}
