
/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>

#include <xpl.h>
#include <connio.h>

#include <logger.h>

#include <mdb.h>
#include <msgapi.h>
#include <nmap.h>

/* Management Client Header Include */
#include <management.h>
#include <hulautil.h>
#include "imapd.h"


HulaKeywordIndex *FetchFlagsAttIndex = NULL;
HulaKeywordIndex *FetchFlagsSoloIndex = NULL;
HulaKeywordIndex *FetchFlagsSectionIndex = NULL;

unsigned char TopLevelMime[] = "Content-Type: message/rfc822\r\nContent-Tranfer-Encoding: 7bit\r\n";
unsigned long TopLevelMimeLen = sizeof(TopLevelMime) - 1;

static BOOL
FindRFC822Header(unsigned char *Header, unsigned char *SearchFor, unsigned char **Buffer, int BufSize)
{
    unsigned char *NL, *ptr, *ptr2, *start=Header, *ptr3;
    BOOL    KeepGoing=TRUE;
    int    len;

    ptr=Header;
    while (KeepGoing) {
        NL=strchr(ptr,0x0a);
        if (NL) {
            /* Check for line continuation, if yes, whack CR/LF and keep going */
            if (isspace(*(NL+1))) {
                *NL=' ';
                NL[1]=' ';
                if (*(NL-1)==0x0d) {
                    *(NL-1)=' ';
                }
                continue;
            } else {
                *NL='\0';
            }
        }
        ptr2=strchr(ptr, ':');
        if (ptr2) {
            *ptr2='\0';
    
            if (XplStrCaseCmp(ptr, SearchFor)==0) {
                ptr3=ptr2+1;
                while (isspace(*ptr3))
                    ptr3++;
                len=strlen(ptr3);
                if (!BufSize) {
                    *Buffer=MemMalloc(len+1);
                    if (!*Buffer)
                        return(FALSE);
                } else {
                    if (len>BufSize)
                        return(FALSE);
                }
                strncpy(*Buffer, ptr3, len);
                (*Buffer)[len]='\0';
                if ((ptr3=strrchr(*Buffer,0x0d))!=NULL)
                    *ptr3='\0';
                if ((ptr3=strrchr(*Buffer,0x0a))!=NULL)
                    *ptr3='\0';
                *ptr2=':';
                if (NL)
                    *NL=0x0a;
//    while ((ptr=strchr(*Buffer, '"'))!=0)
//     memmove(ptr, ptr+1, strlen(ptr+1)+1);
                return(TRUE);
            }
            *ptr2=':';
        }
        if (NL) {
            *NL=0x0a;
            ptr=NL+1;
        } else {
            KeepGoing=FALSE;
        }

        start=ptr;
    }
    return(FALSE);
}

static void
SendRFC822Address(ImapClient *client, unsigned char *Address, unsigned char *Answer)
{
    RFC822AddressStruct *Current;
    unsigned long   Len;

    RFC822ParseAddressList(&client->RFC822AddressList, Address, ".MissingHostName.");

    if (client->RFC822AddressList) {
        SendClient(client, "(", 1);
        Current=client->RFC822AddressList;
        while (Current) {
            if (Current->Personal) {
                Len=strlen(Current->Personal);
                SendClient(client, Answer, sprintf(Answer, "({%lu}\r\n", Len));
                SendClient(client, Current->Personal, Len);
                SendClient(client, " ", 1);
            } else {
                SendClient(client, "(NIL ", 5);
            }

            if (Current->ADL) {
                SendClient(client, "\"", 1);
                SendClient(client, Current->ADL, strlen(Current->ADL));
                SendClient(client, "\" ", 2);
            } else {
                SendClient(client, "NIL ", 4);
            }

            if (Current->Mailbox) {
                SendClient(client, "\"", 1);
                SendClient(client, Current->Mailbox, strlen(Current->Mailbox));
                SendClient(client, "\" ", 2);
            } else {
                SendClient(client, "NIL ", 4);
            }

            if (Current->Host) {
                SendClient(client, "\"", 1);
                SendClient(client, Current->Host, strlen(Current->Host));
                SendClient(client, "\")", 2);
            } else {
                SendClient(client, "NIL)", 4);
            }
            Current=Current->Next;
        }
        RFC822FreeAddressList(client->RFC822AddressList);
        client->RFC822AddressList=NULL;
        SendClient(client, ") ", 2);
    } else {
        SendClient(client, "NIL ", 4);
    }
}

static void
MakeRFC822Header(unsigned char *Header, unsigned long *HSize)
{
    unsigned char *NL, *ptr, *ptr2, *start=Header;
    BOOL    StripLine, KeepGoing=TRUE;

    ptr=Header;
    Header[*HSize]='\0';
    while (KeepGoing) {
        StripLine=FALSE;
        NL=strchr(ptr,0x0a);
        if (NL)
            *NL='\0';
        if ((*ptr!='\t') && (*ptr!=' ')) {
            ptr2=strchr(ptr, ':');
            if (ptr2) {
                *ptr2='\0';
                if (strchr(ptr,' '))
                    StripLine=TRUE;
                *ptr2=':';
            } else {
                if (NL)
                    StripLine=TRUE;
            }
        }
        if (NL) {
            *NL=0x0a;
            ptr=NL+1;
        } else {
            KeepGoing=FALSE;
        }

        if (StripLine) {
            if (NL) {
                memmove(start, ptr, strlen(ptr)+1);
                start=NL-ptr+start+1;
                ptr=start;
            } else {
                *ptr='\0';
            }
        } else {
            start=ptr;
            if ((*ptr==0x0d) || (*ptr==0x0a))
                KeepGoing=FALSE;
        }
    }
    *HSize=strlen(Header);
}

__inline static unsigned char *
GetSubHeader(unsigned char *header, unsigned long headerSize, unsigned char **fields, unsigned long fieldCount, BOOL not, unsigned long *newHeaderLen)
{
    unsigned char *newHeader;
    unsigned char *start;
    unsigned char *colon;
    unsigned char *newLine;
    unsigned char *dest;
    unsigned long i;
    unsigned long lineLen;
    BOOL keepLine;
    
    newHeader = MemMalloc(headerSize);
    if (newHeader) {
        start = header;
        dest = newHeader;
        *newHeaderLen = 0;
        do {
            colon = strchr(start, ':');
            if (colon) {
                *colon = '\0';
                i = 0;
                do {
                    if (XplStrCaseCmp(start, fields[i]) != 0) {
                        i++;
                        if (i < fieldCount) {
                            continue;
                        }

                        /* this is not a match */
                        keepLine = not;
                        break;
                    }
                    
                    /* this line is a match */
                    keepLine = !not;
                    break;
                } while (TRUE);
                *colon = ':';

                newLine = start;
                do {
                    newLine = strchr(newLine, '\n');
                    if (newLine) {
                        if (isspace(*(newLine + 1))) {
                            newLine +=2;
                            continue;
                        }
                    }
                    break;
                } while (TRUE);

                if (newLine) {
                    newLine++;
                    if (keepLine) {
                        lineLen = newLine - start;
                        memcpy(dest + *newHeaderLen, start, lineLen);
                        *newHeaderLen += lineLen;
                    }
                    start = newLine;
                    continue;
                }
            }

            newLine = strchr(start, '\n');
            if (newLine) {
                start = newLine + 1;
                continue;
            }

            break;
        } while(TRUE);

        if (*newHeaderLen > 0) {
            return(newHeader);
        }

        MemFree(newHeader);
    }

    return(NULL);
}

__inline static void
ParseMIMELine(unsigned char *MIME, unsigned char *Type, unsigned long TypeSize, unsigned char *SubType, unsigned long SubTypeSize, unsigned char *Charset, unsigned long CharsetSize, unsigned char *Encoding, unsigned long EncodingSize, unsigned char *Name, unsigned long NameSize, unsigned long *StartPos, unsigned long *Size, unsigned long *HeaderSize, unsigned long *Lines)
{
    unsigned char *p, *p2;
    /* Type */
    p2 = strchr(MIME, ' ');
    if (p2) {
        *p2 = '\0';
        if ((unsigned long)(p2 - MIME) < TypeSize) {
            strcpy(Type, MIME);
        } else {
            strncpy(Type, MIME, TypeSize);
            Type[TypeSize] = '\0';
        }
        p = strchr(++p2, ' ');
        if (p) {
            *p = '\0';
            if ((unsigned long)(p - p2) < SubTypeSize) {
                strcpy(SubType, p2);
            } else {
                strncpy(SubType, p2, SubTypeSize);
                SubType[SubTypeSize] = '\0';
            }
            p2 = strchr(++p, ' ');
            if (p2) {
                *p2 = '\0';
                if ((unsigned long)(p2 - p) < CharsetSize) {
                    strcpy(Charset, p);
                } else {
                    strncpy(Charset, p, CharsetSize);
                    Charset[CharsetSize] = '\0';
                }
                p = strchr(++p2, '"');
                if (p) {
                    *(p-1) = '\0';
                    if ((unsigned long)(p - p2) < EncodingSize) {
                        strcpy(Encoding, p2);
                    } else {
                        strncpy(Encoding, p2, EncodingSize);
                        Encoding[EncodingSize] = '\0';
                    }
                    p2 = strchr(++p, '"');
                    if (p2) {
                        *(p2++) = '\0';

                        if ((unsigned long)(p2 - p) < NameSize) {
                            strcpy(Name, p);
                        } else {
                            strncpy(Name, p, NameSize);
                            Name[NameSize] = '\0';
                        }
                        p = strchr(++p2, ' ');
                        if (p) {
                            *p = '\0';
                            *StartPos = atol(p2);
                            p2 = strchr(++p, ' ');
                            if (p2) {
                                *p2 = '\0';
                                *Size = atol(p);
                                p = strchr(++p2, ' ');
                                if (p) {
                                    *p = '\0';
                                    *HeaderSize = atol(p2);
                                    *Lines = atol(p+1);
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return;
}

__inline static void
ParseMIMEDLine(unsigned char *MIME, unsigned char *Type, unsigned long TypeSize, unsigned char *SubType, unsigned long SubTypeSize, unsigned char *Charset, unsigned long CharsetSize, unsigned char *Encoding, unsigned long EncodingSize, unsigned char *Name, unsigned long NameSize, long *headStart, unsigned long *headSize, unsigned long *StartPos, unsigned long *Size, unsigned long *HeaderSize, unsigned long *Lines)
{
    unsigned char *p, *p2;
    /* Type */
    p2 = strchr(MIME, ' ');
    if (p2) {
        *p2 = '\0';
        if ((unsigned long)(p2 - MIME) < TypeSize) {
            strcpy(Type, MIME);
        } else {
            strncpy(Type, MIME, TypeSize);
            Type[TypeSize] = '\0';
        }
        p = strchr(++p2, ' ');
        if (p) {
            *p = '\0';
            if ((unsigned long)(p - p2) < SubTypeSize) {
                strcpy(SubType, p2);
            } else {
                strncpy(SubType, p2, SubTypeSize);
                SubType[SubTypeSize] = '\0';
            }
            p2 = strchr(++p, ' ');
            if (p2) {
                *p2 = '\0';
                if ((unsigned long)(p2 - p) < CharsetSize) {
                    strcpy(Charset, p);
                } else {
                    strncpy(Charset, p, CharsetSize);
                    Charset[CharsetSize] = '\0';
                }
                p = strchr(++p2, '"');
                if (p) {
                    *(p-1) = '\0';
                    if ((unsigned long)(p - p2) < EncodingSize) {
                        strcpy(Encoding, p2);
                    } else {
                        strncpy(Encoding, p2, EncodingSize);
                        Encoding[EncodingSize] = '\0';
                    }
                    p2 = strchr(++p, '"');
                    if (p2) {
                        *(p2++) = '\0';

                        if ((unsigned long)(p2 - p) < NameSize) {
                            strcpy(Name, p);
                        } else {
                            strncpy(Name, p, NameSize);
                            Name[NameSize] = '\0';
                        }
                        p = strchr(++p2, ' ');
                        if (p) {
                            *headStart = atol(p2);
                            p++;
                            p2 = strchr(p, ' ');
                            if (p2) {
                                p2++;
                                *headSize = atol(p);
                                p = strchr(p2, ' ');
                                if (p) {
                                    *StartPos = atol(p2);
                                    p2 = strchr(++p, ' ');
                                    if (p2) {
                                        *Size = atol(p);
                                        p = strchr(++p2, ' ');
                                        if (p) {
                                            *HeaderSize = atol(p2);
                                            *Lines = atol(p+1);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return;
}

__inline static long
GetDataFromNmap(ImapClient *client, unsigned char **bufferAddress, long count, unsigned char *workBuff, unsigned long workBuffLen)
{
    long Got;
    long len;
    long ccode;
    unsigned char *memoryBuffer;
    
    memoryBuffer = MemMalloc(count + 1);
    if (memoryBuffer) {
        Got = 0;
        if (client->NBufferPtr > 0) {
            if (client->NBufferPtr > count) {
                memcpy(memoryBuffer + Got, client->NBuffer, count);
                Got += count;
                memoryBuffer[Got] = '\0';
                memmove(client->NBuffer, client->NBuffer + count, client->NBufferPtr - count + 1);
                client->NBufferPtr -= count;
                ccode = GetNMAPAnswer(client, workBuff, workBuffLen, TRUE);
                if (ccode == 1000) {
                    *bufferAddress = memoryBuffer;
                    return(0);
                }
                MemFree(memoryBuffer);
                return(ccode);
            }

            memcpy(memoryBuffer + Got, client->NBuffer, client->NBufferPtr);
            Got += client->NBufferPtr;
            memoryBuffer[Got] = '\0';
            count -= client->NBufferPtr;
            client->NBufferPtr = 0;
            client->NBuffer[0] = '\0';
        }

        while (count > 0) {
            if ((unsigned long)count < workBuffLen) {
                len = DoNMAPRead(client, workBuff, count, 0);
            } else {
                len = DoNMAPRead(client, workBuff, workBuffLen, 0);
            }
            if (len > 0) {
                memcpy(memoryBuffer + Got, workBuff, len);
                Got += len;
                memoryBuffer[Got] = '\0';
                count -= len;
                continue;
            }
            MemFree(memoryBuffer);
            return(-1);
        }

        ccode = GetNMAPAnswer(client, workBuff, workBuffLen, TRUE);
        if (ccode == 1000) {
            *bufferAddress = memoryBuffer;
            return(0);
        }
        MemFree(memoryBuffer);
        return(ccode);
    }
    return(-1);
}

__inline static void
FreeFetchMessageInfo(MessageInfo *messageInfo)
{
    if (messageInfo->header) {
        MemFree(messageInfo->header);
        messageInfo->header = NULL;
    }

    if (messageInfo->mimeInfo) {
        MDBDestroyValueStruct(messageInfo->mimeInfo);
        messageInfo->mimeInfo = NULL;
    }
}

__inline static void
FreeFetchFlags(FetchStruct *FetchRequest)
{
    unsigned long i;
    FetchFlagStruct *flag;

    for (i = 0; i < FetchRequest->flagCount; i++) {
        flag = &(FetchRequest->flag[i]);
        if (!(flag->hasAllocated)) {
            continue;
        }

        if (flag->hasAllocated & ALLOCATED_FIELDS) {
            MemFree(flag->fieldList.fieldRequest);
        }

        if (!(flag->hasAllocated & ALLOCATED_FIELD_POINTERS)) {
            ;
        } else {
            MemFree(flag->fieldList.field);
        }

        if (!(flag->hasAllocated & ALLOCATED_PART_NUMBERS)) {
            ;
        } else {
            MemFree(flag->bodyPart.part);
        }
    }
    return;
}

__inline static void
FreeFetchResourcesSuccess(FetchStruct *FetchRequest)
{
    if (FetchRequest->messageSet) {
        MemFree(FetchRequest->messageSet);
    }

    if (!(FetchRequest->hasAllocated)) {
        return;
    }

    FreeFetchFlags(FetchRequest);

    if (!(FetchRequest->hasAllocated & ALLOCATED_FLAGS)) {
        return;
    }
    
    MemFree(FetchRequest->flag);
    return;
}

__inline static void
FreeFetchResourcesFailure(FetchStruct *FetchRequest)
{
    FreeFetchMessageInfo(&(FetchRequest->currentMessage));
    FreeFetchResourcesSuccess(FetchRequest);
    return;
}

__inline static BOOL
ReadBodyPart(ImapClient *client, unsigned char **Part, unsigned long ID, unsigned long Start, unsigned long Size)
{
    long    count, Got, ReplyInt, len;
    unsigned char Answer[BUFSIZE+1];
    unsigned char Reply[BUFSIZE+1];
    unsigned char *Header;

    if (!Size) {
        return(FALSE);
    }

    *Part=MemMalloc(Size+1);
    if (!*Part) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO IMAP Out of memory\r\n" , client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(FALSE);
    }

    Header=*Part;

    /* Request the body part*/
    ReplyInt = sprintf(Answer, "BRAW %lu %lu %lu\r\n", client->IDList[ID], Start, Size);
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if ((ReplyInt!=2021) && (ReplyInt!=2025)) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP error:%ld\r\n", client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        MemFree(Header);
        return(FALSE);
    }

    count=atoi(Reply);
    Got=0;
    if (client->NBufferPtr>0) {
        if (client->NBufferPtr>count) {
            memcpy(Header+Got, client->NBuffer,count);
            Got+=count;
            Header[Got]='\0';
            memmove(client->NBuffer, client->NBuffer+count, client->NBufferPtr-count+1);
            client->NBufferPtr-=count;
            count=0;
        } else {
            memcpy(Header+Got, client->NBuffer, client->NBufferPtr);
            Got+=client->NBufferPtr;
            Header[Got]='\0';
            count-=client->NBufferPtr;
            client->NBufferPtr=0;
            client->NBuffer[0]='\0';
        }
    }
    while (count>0) {
        if ((unsigned long)count < sizeof(Answer)) {
            len=DoNMAPRead(client, Answer, count, 0);
        } else {
            len=DoNMAPRead(client, Answer, sizeof(Answer), 0);
        }
        if (len>0) {
            memcpy(Header+Got, Answer, len);
            Got+=len;
            Header[Got]='\0';
            count-=len;
        } else {
            break;
        }
    }

    /* Grab confirmation of BRAW */
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt!=1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO error:%ld\r\n",client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        MemFree(Header);
        return(FALSE);
    }
    return(TRUE);
}

__inline static void
SendEnvelope(ImapClient *client, unsigned char *Header, unsigned char *Answer)
{
    unsigned char *Date=NULL, *Subject=NULL, *From=NULL, *Sender=NULL, *Reply=NULL, *To=NULL, *CC=NULL, *BCC=NULL, *InRepTo=NULL, *ID=NULL;
    unsigned long len, len2;

    /* this function is memory intensive */

    SendClient(client, "(", 1);

    /* First comes the date, just as text */
    if (FindRFC822Header(Header, "Date", &Date, 0)) {
        len = strlen(Date);
        len2 = sprintf(Answer, "{%d}\r\n", (int)len);
        SendClient(client, Answer, len2);
        SendClient(client, Date, len);
        SendClient(client, " ", 1);

#if 0
        SendClient(client, "\"", 1);
        SendClient(client, Date, strlen(Date));
        SendClient(client, "\" ", 2);
#endif
    } else {
        SendClient(client, "NIL ", 4);
    }

    /* now env_subject */
    if (FindRFC822Header(Header, "Subject", &Subject, 0)) {
        len=strlen(Subject);
        len2 = sprintf(Answer, "{%d}\r\n", (int)len);
        SendClient(client, Answer, len2);
        SendClient(client, Subject, len);
        SendClient(client, " ", 1);
    } else {
        SendClient(client, "NIL ", 4);
    }

    /* now env_from */
    if (FindRFC822Header(Header, "From", &From, 0)) {
        SendRFC822Address(client, From, Answer);
    } else {
        SendClient(client, "NIL ", 4);
    }

    /* now env_sender */
    if (FindRFC822Header(Header, "From", &Sender, 0)) {
        SendRFC822Address(client, Sender, Answer);
    } else {
        SendClient(client, "NIL ", 4);
    }

    /* now env_in_reply_to */
    if (FindRFC822Header(Header, "Reply-To", &Reply, 0)) {
        SendRFC822Address(client, Reply, Answer);
    } else {
        if (FindRFC822Header(Header, "From", &Reply, 0)) {
            SendRFC822Address(client, Reply, Answer);
        } else {
            SendClient(client, "NIL ", 4);
        }
    }

    /* now env_to */
    if (FindRFC822Header(Header, "To", &To, 0)) {
        SendRFC822Address(client, To, Answer);
    } else {
        SendClient(client, "NIL ", 4);
    }

    /* now env_cc */
    if (FindRFC822Header(Header, "Cc", &CC, 0)) {
        SendRFC822Address(client, CC, Answer);
    } else {
        SendClient(client, "NIL ", 4);
    }

    /* now env_bcc */
    if (FindRFC822Header(Header, "Bcc", &BCC, 0)) {
        SendRFC822Address(client, BCC, Answer);
    } else {
        SendClient(client, "NIL ", 4);
    }

    /* now env_in_reply_to */
    if (FindRFC822Header(Header, "In-Reply-To", &InRepTo, 0)) {
        int inRepToLen = strlen(InRepTo);

        /* Break up reply to client,  Answer may not be large enough */
        /* to hold all of the recipients in the reply to field. */

        do {
            if (*InRepTo == '"' || *InRepTo < 0x20 || *InRepTo > 0x7f) {
                len = sprintf(Answer, "{%lu}\r\n", (long unsigned int)inRepToLen);
                SendClient(client, Answer, len);
                SendClient(client, InRepTo, inRepToLen);
                SendClient(client, " ", 1);
                break;
            }
            
            if (*(++InRepTo)) {
                continue;
            }

            SendClient(client, "\"", 1);
            SendClient(client, InRepTo, inRepToLen);
            SendClient(client, "\" ", 2);
            break;
        } while (TRUE);
    } else {
        SendClient(client, "NIL ", 4);
    }

    /* now env_message_id */
    if (FindRFC822Header(Header, "Message-Id", &ID, 0)) {
        SendClient(client, "\"", 1);
        SendClient(client, ID, strlen(ID));
        SendClient(client, "\"", 1);
    } else {
        SendClient(client, "NIL", 3);
    }
    SendClient(client, ")", 1);

    /* Clean up the mess */
    if (Date)
        MemFree(Date);
    if (Subject)
        MemFree(Subject);
    if (From)
        MemFree(From);
    if (Sender)
        MemFree(Sender);
    if (Reply)
        MemFree(Reply);
    if (To)
        MemFree(To);
    if (CC)
        MemFree(CC);
    if (BCC)
        MemFree(BCC);
    if (InRepTo)
        MemFree(InRepTo);
    if (ID)
        MemFree(ID);
}

__inline static long
ReadAndSendNmapResponse(ImapClient *client, long count, unsigned char *reply, unsigned long replySize, unsigned char *answer, unsigned long answerSize)
{
    int replyInt;
    long len;

    if (client->NBufferPtr > 0) {
        if (client->NBufferPtr > count) {
            SendClient(client, client->NBuffer, count);
            memmove(client->NBuffer, client->NBuffer + count, client->NBufferPtr - count + 1);
            client->NBufferPtr -= count;
            count = 0;
            FlushClient(client);
            replyInt = GetNMAPAnswer(client, reply, replySize, TRUE);
            return(replyInt);
        }

        SendClient(client, client->NBuffer, client->NBufferPtr);
        count -= client->NBufferPtr;
        client->NBufferPtr = 0;
        client->NBuffer[0] = '\0';

    }

    while (count > 0) {
        if ((unsigned long)count < answerSize) {
            len = DoNMAPRead(client, answer, count, 0);
        } else {
            len = DoNMAPRead(client, answer, answerSize, 0);
        }

        if (len > 0) {
            SendClient(client, answer, len);
            count -= len;
            continue;
        }

        return(-1);
    }

    FlushClient(client);
    replyInt = GetNMAPAnswer(client, reply, replySize, TRUE);
    return(replyInt);
}

static long 
FetchFlagResponderUid(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "UID %lu", client->UIDList[FetchRequest->currentMessage.imapId]));
    return(0);
}

static long 
FetchFlagResponderFlags(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    unsigned char addSpace[2];

    SendClient(client, "FLAGS (", strlen("FLAGS ("));
    addSpace[0]='\0';
    addSpace[1]='\0';
    if (FetchRequest->currentMessage.flags & MSG_STATE_ANSWERED) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "\\Answered"));
        addSpace[0]=' ';
    } 
    if (FetchRequest->currentMessage.flags & MSG_STATE_DELETED) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "%s\\Deleted",addSpace));
        addSpace[0]=' ';
    } 
    if (FetchRequest->currentMessage.flags & MSG_STATE_RECENT) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "%s\\Recent", addSpace));
        addSpace[0]=' ';
    } 
    if (FetchRequest->currentMessage.flags & MSG_STATE_READ) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf,"%s\\Seen", addSpace));
        addSpace[0]=' ';
    }
    if (FetchRequest->currentMessage.flags & MSG_STATE_DRAFT) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf,"%s\\Draft", addSpace));
        addSpace[0]=' ';
    }
    if (FetchRequest->currentMessage.flags & MSG_STATE_PRIOHIGH) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf,"%s\\Flagged", addSpace));
        addSpace[0]=' ';
    }
    SendClient(client, ")", strlen(")"));
    return(0);
}

static long 
FetchFlagResponderEnvelope(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;

    SendClient(client, "ENVELOPE ", strlen("ENVELOPE "));
    SendEnvelope(client, FetchRequest->currentMessage.header, FetchRequest->sendBuf);
    return(0);
}

static long 
FetchFlagResponderXSender(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    long ccode;
    unsigned char *pp;

    SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "AINFO %lu\r\n", FetchRequest->currentMessage.nmapId));
    ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
    if (ccode != 2001) {
        SendClient(client, "XSENDER {0}\r\n", 13);
    } else {
        pp=strrchr(FetchRequest->responseBuf, ' ');
        if (!pp) {
            SendClient(client, "XSENDER {0}\r\n", 13);
        } else {
            pp++;
            if (*pp!='-') {
                unsigned long authLen;
       
                authLen = strlen(pp);
                if (!strchr(pp, '@')) {
                    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "XSENDER {%lu}\r\n", authLen + 1 + HostnameLen));
                    SendClient(client, pp, authLen);
                    SendClient(client, "@", 1);
                    SendClient(client, Hostname, HostnameLen);
                } else {
                    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "XSENDER {%lu}\r\n", authLen));
                    SendClient(client, pp, authLen);
                }
            } else {
                SendClient(client, "XSENDER {0}\r\n", strlen("XSENDER {0}\r\n"));
            }
        }
    }
    return(0);
}

static long
FetchFlagResponderInternalDate(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    struct tm  ltm;

    gmtime_r(&FetchRequest->currentMessage.internalDate, &ltm);

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "INTERNALDATE \"%02d-%3s-%04d %02d:%02d:%02d +0000\"", ltm.tm_mday, MonthNames[ltm.tm_mon], 1900+ltm.tm_year, ltm.tm_hour, ltm.tm_min, ltm.tm_sec));
    return(0);
}

static long 
FetchFlagResponderBodyStructure(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    long ccode;
    unsigned char Type[MIME_TYPE_LEN+1], Subtype[MIME_SUBTYPE_LEN+1], Charset[MIME_CHARSET_LEN+1], Encoding[MIME_ENCODING_LEN+1], Name[MIME_NAME_LEN+1];
    unsigned long SPos, Lines, Size, len, MPCount, RFCCount, partHeaderSize;
    unsigned long MultiPartTypeListLen = 20;
    unsigned long RFCSizeListLen   = 20;
    unsigned char *ptr, *lineend;
    unsigned long MIMERAlloc = 0;
    unsigned char **tmpTypeList;
    unsigned long *tmpSizeList;
    unsigned char *tmpResponse;

    tmpTypeList = MemMalloc(sizeof(unsigned char *)*MultiPartTypeListLen);
    if (tmpTypeList) {
        client->MultiPartTypeList = tmpTypeList;
        MPCount=0;

        tmpSizeList = MemMalloc(sizeof(long)*RFCSizeListLen);
        if (tmpSizeList) {
            client->RFCSizeList = tmpSizeList;
            RFCCount=0;

            SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "MIME %lu\r\n", FetchRequest->currentMessage.nmapId));
            MIMERAlloc = 2048;

            tmpResponse = MemMalloc(MIMERAlloc);
            if (tmpResponse) {
                client->MIMEResponse = tmpResponse;
                client->MIMEResponse[0]='\0';
                len=0;
                do {
                    ccode=GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FALSE);
                    len += strlen(FetchRequest->responseBuf)+2;
                    if (len >= MIMERAlloc) {
                        unsigned char *tmp;
         
                        tmp = MemRealloc(client->MIMEResponse, MIMERAlloc * 2);
                        if (tmp) {
                            client->MIMEResponse = tmp;
                        } else {
                            return(-1);
                        }
                        MIMERAlloc *= 2;
                    }
                    FetchRequest->responseBuf[4]=' ';    
                    strcat(client->MIMEResponse, FetchRequest->responseBuf);
                    strcat(client->MIMEResponse, "\n");
                } while (ccode!=1000);
                if (FetchRequest->flags & F_BODY) {
                    SendClient(client, "BODY ", strlen("BODY "));
                } else {
                    SendClient(client, "BODYSTRUCTURE ", strlen("BODYSTRUCTURE "));
                }

                ptr=client->MIMEResponse;
                lineend=client->MIMEResponse;
                do {
                    ccode=atoi(ptr);
                    lineend=strchr(lineend+1, 0x0a);
                    if (lineend) {
                        lineend++;
                    }
                    switch(ccode) {
                    case 2002: {

                        ParseMIMELine(ptr + 5, 
                                      Type, MIME_TYPE_LEN, 
                                      Subtype, MIME_SUBTYPE_LEN, 
                                      Charset, MIME_CHARSET_LEN, 
                                      Encoding, MIME_ENCODING_LEN, 
                                      Name, MIME_NAME_LEN, 
                                      &SPos, &Size, &partHeaderSize, &Lines);  

                        /* First, set the default values if nothing is specified by the message */
                        if (Type[0]=='-') {
                            strcpy(Type, "TEXT");
                            strcpy(Subtype, "PLAIN");
                        }
   
                        if (Encoding[0]=='-') {
                            strcpy(Encoding, "7BIT");
                        }

                        /************
                         **MULTIPART**
                         ************/
                        if (XplStrCaseCmp(Type, "Multipart")==0) {
                            if (MPCount < MultiPartTypeListLen) {  
                                client->MultiPartTypeList[MPCount++] = MemStrdup(Subtype);
                            } else {
                                unsigned char *tmp;

                                MultiPartTypeListLen <<= 1;  /* Double the Size */
                                tmp = MemRealloc(client->MultiPartTypeList, sizeof(unsigned char *) * MultiPartTypeListLen);
                                if (tmp) {
                                    client->MultiPartTypeList = (unsigned char **)tmp; 
                                    client->MultiPartTypeList[MPCount++] = MemStrdup(Subtype);
                                } else {
                                    while (MPCount > 0) {
                                        MPCount--;           
                                        MemFree(client->MultiPartTypeList[MPCount]);
                                    }
                                    MemFree(client->MultiPartTypeList);
                                    client->MultiPartTypeList = NULL;
                                    MemFree(client->RFCSizeList);
                                    client->RFCSizeList = NULL;
                                    return(-1);
                                } 
                            }
                            SendClient(client, "(", 1);
                            /************
                             **   TEXT  **
                             ************/
                        } else if (XplStrCaseCmp(Type, "Text")==0) {
                            if (Charset[0]=='-') {
                                strcpy(Charset, "US-ASCII");
                            }

                            len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"%s\" \"%s\" (\"CHARSET\" \"%s\") NIL NIL \"%s\" %lu %lu)", Type, Subtype, Charset, Encoding, Size, Lines);
                            SendClient(client, FetchRequest->sendBuf, len);
                            /************
                             ** MESSAGE **
                             ************/
                        } else if ((XplStrCaseCmp(Type, "Message")==0) && (XplStrCaseCmp(Subtype, "RFC822")==0)) {
                            unsigned char *PartHeader;

                            /* we have to print the basic stuff, then get the headers, print the envelope */
                            /* and then continue in the loop */
                            if (RFCCount < RFCSizeListLen) {
                                client->RFCSizeList[RFCCount++] = Lines;
                            } else {
                                unsigned char *tmp;

                                RFCSizeListLen <<= 1;  /* Double the Size */
                                tmp = MemRealloc(client->RFCSizeList, sizeof(long) * RFCSizeListLen);
                                if (tmp) {
                                    client->RFCSizeList = (long *)tmp; 
                                    client->RFCSizeList[RFCCount++] = Lines;
                                } else {
                                    while (MPCount > 0) {
                                        MPCount--;           
                                        MemFree(client->MultiPartTypeList[MPCount]);
                                    }
                                    MemFree(client->MultiPartTypeList);
                                    client->MultiPartTypeList = NULL;
                                    MemFree(client->RFCSizeList);
                                    client->RFCSizeList = NULL;
                                    return(-1);
                                } 
                            }

                            /* Body Type & Subtype */
                            len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"%s\" \"%s\" ", Type, Subtype);
                            SendClient(client, FetchRequest->sendBuf, len);

                            /* Body parameter parentesized list */
                            if ((Charset[0]!='-') && (Name[0]!='\0')) {
                                unsigned char *langPtr;
                                if ((Name[0] != '*') || (Name[1] != '=') || ((langPtr = strchr(Name + 2, '\'')) == NULL) || ((langPtr = strchr(langPtr + 1, '\'')) == NULL)) {
                                    len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"CHARSET\" \"%s\" \"NAME\" \"%s\") ", Charset, Name);
                                    SendClient(client, FetchRequest->sendBuf, len);
                                } else {
                                    /* this is an rfc2231 name */
                                    len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"CHARSET\" \"%s\" \"NAME*\" \"%s\") ", Charset, Name + 2);
                                    SendClient(client, FetchRequest->sendBuf, len);
                                }
                            } else if (Charset[0]!='-') {
                                len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"CHARSET\" \"%s\") ", Charset);
                                SendClient(client, FetchRequest->sendBuf, len);
                            } else if (Name[0]!='\0') {
                                unsigned char *langPtr;
                                if ((Name[0] != '*') || (Name[1] != '=') || ((langPtr = strchr(Name + 2, '\'')) == NULL) || ((langPtr = strchr(langPtr + 1, '\'')) == NULL)) {
                                    len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"NAME\" \"%s\") ", Name);
                                    SendClient(client, FetchRequest->sendBuf, len);
                                } else {
                                    /* this is an rfc2231 name */
                                    len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"NAME*\" \"%s\") ", Name + 2);
                                    SendClient(client, FetchRequest->sendBuf, len);
                                }
                            } else {
                                SendClient(client, "NIL ", 4);
                            }

                            /* Body ID & Body description */
                            SendClient(client, "NIL NIL ", 8);

                            /* Body encoding && Size */
                            len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "\"%s\" %lu ", Encoding, Size);
                            SendClient(client, FetchRequest->sendBuf, len);

                            /* Grab the envelope  */
                            if (!ReadBodyPart(client, &PartHeader, FetchRequest->currentMessage.imapId, SPos, partHeaderSize)) {
                                return(-1);
                            }
                            SendEnvelope(client, PartHeader, FetchRequest->sendBuf);
                            MemFree(PartHeader);

                            /************
                             ** DEFAULT **
                             ************/
                        } else {
                            /* Body Type & Subtype */
                            len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"%s\" \"%s\" ", Type, Subtype);
                            SendClient(client, FetchRequest->sendBuf, len);

                            /* Body parameter parentesized list */
                            if ((Charset[0]!='-') && (Name[0]!='\0')) {
                                unsigned char *langPtr;
                                if ((Name[0] != '*') || (Name[1] != '=') || ((langPtr = strchr(Name + 2, '\'')) == NULL) || ((langPtr = strchr(langPtr + 1, '\'')) == NULL)) {
                                    len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"CHARSET\" \"%s\" \"NAME\" \"%s\") ", Charset, Name);
                                    SendClient(client, FetchRequest->sendBuf, len);
                                } else {
                                    /* This is an rfc2231 encoded name */
                                    len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"CHARSET\" \"%s\" \"NAME*\" \"%s\") ", Charset, Name + 2);
                                    SendClient(client, FetchRequest->sendBuf, len);
                                }
                            } else if (Charset[0]!='-') {
                                len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"CHARSET\" \"%s\") ", Charset);
                                SendClient(client, FetchRequest->sendBuf, len);
                            } else if (Name[0]!='\0') {
                                unsigned char *langPtr;
                                if ((Name[0] != '*') || (Name[1] != '=') || ((langPtr = strchr(Name + 2, '\'')) == NULL) || ((langPtr = strchr(langPtr + 1, '\'')) == NULL)) {
                                    len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"NAME\" \"%s\") ", Name);
                                    SendClient(client, FetchRequest->sendBuf, len);
                                } else {
                                    /* This is an rfc2231 encoded name */
                                    len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "(\"NAME*\" \"%s\") ", Name + 2);
                                    SendClient(client, FetchRequest->sendBuf, len);
                                }
                            } else {
                                SendClient(client, "NIL ", 4);
                            }

                            /* Body ID & Body description */
                            SendClient(client, "NIL NIL ", 8);

                            /* Body encoding && Size */
                            len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "\"%s\" %lu)", Encoding, Size);
                            SendClient(client, FetchRequest->sendBuf, len);
                        }
                        break;
                    }

                    case 2003: {
                        if (MPCount > 0) {
                            MPCount--;
                            len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), " \"%s\")", client->MultiPartTypeList[MPCount]);
                            SendClient(client, FetchRequest->sendBuf, len);
                            MemFree(client->MultiPartTypeList[MPCount]);
                            client->MultiPartTypeList[MPCount]=NULL;
                        }
                        break;
                    }

                    case 2004: {
                        RFCCount--;
                        len = sprintf(FetchRequest->sendBuf, " %lu)", client->RFCSizeList[RFCCount]);
                        SendClient(client, FetchRequest->sendBuf, len);
                        break;
                    }

                    case 6000: {
                        unsigned long Event;

                        Event = atol(FetchRequest->responseBuf + 5);
                        if (Event & EVENT_MBOX_FLAGS_CHANGED) {
                            unsigned char *ptr;
                            unsigned long MsgNum;
                            unsigned long MsgFlags;

                            Event &= ~EVENT_MBOX_FLAGS_CHANGED;

                            /* 6000 Event MsgNum MsgFlags HumanReadableText */
                            ptr = strchr(FetchRequest->responseBuf + 5, ' ');
                            if (ptr) {
                                ptr++;
                                MsgNum = atol(ptr);
                                ptr = strchr(ptr, ' ');
                                if (ptr) {
                                    ptr++;
                                    MsgFlags = atol(ptr);
                                    /* We need to hold onto these flags until we have a chance to tell the client about them. */
                                    RememberFlags(client->UnseenFlags, MsgNum, MsgFlags);
                                }
                            }
                        }

                        client->NMAPAsyncEvent |= Event;
                        break;
                    }
                    }
                    ptr=lineend;
                } while ((ccode!=1000) && lineend);
                MemFree(client->MultiPartTypeList);
                client->MultiPartTypeList = NULL;
                MemFree(client->RFCSizeList);
                client->RFCSizeList = NULL;
                MemFree(client->MIMEResponse);
                client->MIMEResponse = NULL;
                return(0);
            }
        }
    }

    return(-1);
}

static long 
FetchFlagResponderRfc822Text(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    long ccode;
    long count;

    SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY %lu\r\n", FetchRequest->currentMessage.nmapId));

    ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
    if ((ccode == 2021) || (ccode == 2025)) {
        count = atol(FetchRequest->responseBuf);
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "RFC822.TEXT {%lu}\r\n", count));
    } else {
        return(ccode);
    }

    ccode = ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf));
    if (ccode == 1000) {
        return(0);
    }

    return(ccode);
}

static long 
FetchFlagResponderRfc822Size(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "RFC822.SIZE %lu", FetchRequest->currentMessage.totalSize));
    return(0);
}

static long 
FetchFlagResponderRfc822Header(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "RFC822.HEADER {%lu}\r\n", FetchRequest->currentMessage.headerSize));
    SendClient(client, FetchRequest->currentMessage.header, FetchRequest->currentMessage.headerSize);
    return(0);
}

static long
FetchFlagResponderRfc822(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    long ccode;
    long count;

    SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY %lu\r\n", FetchRequest->currentMessage.nmapId));

    ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
    if ((ccode == 2021) || (ccode == 2025)) {
        count = atol(FetchRequest->responseBuf);
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "RFC822 {%lu}\r\n", count + FetchRequest->currentMessage.headerSize));
        SendClient(client, FetchRequest->currentMessage.header, FetchRequest->currentMessage.headerSize);
        ccode = ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf));
        if (ccode == 1000) {
            return(0);
        }
    }

    return(ccode);
}

static long
FetchFlagResponderBody(void *param1, void *param2, void *param3)
{
    return(FetchFlagResponderBodyStructure(param1, param2, param3));
}

static long 
FetchFlagResponderBodyHeader(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[HEADER] {%lu}\r\n", FetchRequest->currentMessage.headerSize));
    SendClient(client, FetchRequest->currentMessage.header, FetchRequest->currentMessage.headerSize);
    return(0);
}

static long
FetchFlagResponderBodyHeaderPartial(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;

    if (flag->partial.start < FetchRequest->currentMessage.headerSize) {
        if ((flag->partial.start + flag->partial.len) < FetchRequest->currentMessage.headerSize) {
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[HEADER]<%lu> {%lu}\r\n", flag->partial.start, flag->partial.len));
            SendClient(client, FetchRequest->currentMessage.header + flag->partial.start, flag->partial.len);
            return(0);
        }

        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[HEADER]<%lu> {%lu}\r\n", flag->partial.start, FetchRequest->currentMessage.headerSize - flag->partial.start));
        SendClient(client, FetchRequest->currentMessage.header + flag->partial.start, FetchRequest->currentMessage.headerSize - flag->partial.start);
        return(0);
    }

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[HEADER]<%lu> {0}\r\n", flag->partial.start));
    return(0);
}

static long
FetchFlagResponderBodyText(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    long ccode;

    long count;

    SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY %lu\r\n", FetchRequest->currentMessage.nmapId));

    ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
    if ((ccode == 2021) || (ccode == 2025)) {
        count = atol(FetchRequest->responseBuf);
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[TEXT] {%lu}\r\n", count));
        ccode = ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf));
        if (ccode == 1000) {
            return(0);
        }
    }
    
    return(ccode);
}

static long
FetchFlagResponderBodyTextPartial(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long count;
    long ccode;

    if (flag->partial.start < FetchRequest->currentMessage.bodySize) {
        if ((flag->partial.start + flag->partial.len) < FetchRequest->currentMessage.bodySize) {
            SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->partial.start, flag->partial.len));
        } else {
            SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->partial.start, FetchRequest->currentMessage.bodySize - flag->partial.start));
        }

        ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
        if ((ccode == 2021) || (ccode == 2025)) {
            count = atol(FetchRequest->responseBuf);
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[TEXT]<%lu> {%lu}\r\n", flag->partial.start, count));

            ccode = ReadAndSendNmapResponse(client, 
                                            count, 
                                            FetchRequest->responseBuf, 
                                            sizeof(FetchRequest->responseBuf), 
                                            FetchRequest->sendBuf, 
                                            sizeof(FetchRequest->sendBuf));
            if (ccode == 1000) {
                return(0);
            }
        }
        return(ccode);
    }

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[TEXT]<%lu> {0}\r\n", flag->partial.start));
    return(0);
}

static long
FetchFlagResponderBodyBoth(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    long ccode;
    long count;

    SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %d %lu\r\n", FetchRequest->currentMessage.nmapId, (int)(0 - FetchRequest->currentMessage.headerSize), FetchRequest->currentMessage.headerSize + FetchRequest->currentMessage.bodySize));

    ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
    if ((ccode == 2021) || (ccode == 2025)) {
        count = atol(FetchRequest->responseBuf);
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[] {%lu}\r\n", count));
        ccode = ReadAndSendNmapResponse(client, 
                                        count, 
                                        FetchRequest->responseBuf, 
                                        sizeof(FetchRequest->responseBuf), 
                                        FetchRequest->sendBuf, 
                                        sizeof(FetchRequest->sendBuf));
        if (ccode == 1000) {
            return(0);
        }
    }

    return(ccode);
}

static long
FetchFlagResponderBodyBothPartial(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long ccode;
    long count;

    if (flag->partial.start < FetchRequest->currentMessage.totalSize) {
        SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %d %lu\r\n", FetchRequest->currentMessage.nmapId, (int)(flag->partial.start - FetchRequest->currentMessage.headerSize), flag->partial.len));

        ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
        if ((ccode == 2021) || (ccode == 2025)) {
            count = atol(FetchRequest->responseBuf);
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[]<%lu> {%lu}\r\n", flag->partial.start, count));
            if (ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf)) == 1000) {
                return(0);
            }
        }

        return(ccode);
    }

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[]<%lu> {0}\r\n", flag->partial.start));
    return(0);
}

static long 
FetchFlagResponderBodyMime(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[MIME] {%lu}\r\n", TopLevelMimeLen));
    SendClient(client, TopLevelMime, TopLevelMimeLen);
    return(0);
}

static long
FetchFlagResponderBodyMimePartial(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;

    if (flag->partial.start < TopLevelMimeLen) {
        if ((flag->partial.start + flag->partial.len) < TopLevelMimeLen) {
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[MIME]<%lu> {%lu}\r\n", flag->partial.start, flag->partial.len));
            SendClient(client, TopLevelMime + flag->partial.start, flag->partial.len);
            return(0);
        }

        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[MIME]<%lu> {%lu}\r\n", flag->partial.start, TopLevelMimeLen - flag->partial.start));
        SendClient(client, TopLevelMime + flag->partial.start, TopLevelMimeLen - flag->partial.start);
        return(0);
    }

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[MIME]<%lu> {0}\r\n", flag->partial.start));
    return(0);
}

static long
FetchFlagResponderBodyHeaderFields(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    unsigned long j;
    unsigned long len;
    unsigned char *resultHeader;
    unsigned long resultHeaderLen;

    if (!flag->fieldList.skip) {
        SendClient(client, "BODY[HEADER.FIELDS (", strlen("BODY[HEADER.FIELDS ("));
    } else {
        SendClient(client, "BODY[HEADER.FIELDS.NOT (", strlen("BODY[HEADER.FIELDS.NOT ("));
    }

    /* We go one less to have an easy way of knowing if to put a space */
    for (j = 0; j < flag->fieldList.count - 1; j++) {
        len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "\"%s\" ", flag->fieldList.field[j]);
        SendClient(client, FetchRequest->sendBuf, len);
    } 
    len = snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "\"%s\")]", flag->fieldList.field[j]);
    SendClient(client, FetchRequest->sendBuf, len);

    resultHeader = GetSubHeader(FetchRequest->currentMessage.header, FetchRequest->currentMessage.headerSize, flag->fieldList.field, flag->fieldList.count, flag->fieldList.skip, &resultHeaderLen);
    if (resultHeader) {
        if (!flag->isPartial) {
            len = sprintf(FetchRequest->sendBuf, " {%lu}\r\n", resultHeaderLen + 2);
            SendClient(client, FetchRequest->sendBuf, len);
            SendClient(client, resultHeader, resultHeaderLen);
            SendClient(client, "\r\n", 2);
        } else {
            if (flag->partial.start < resultHeaderLen) {
                if ((flag->partial.start + flag->partial.len) < resultHeaderLen) {
                    ;
                } else {
                    flag->partial.len = resultHeaderLen - flag->partial.start;
                }

            } else {
                flag->partial.start = 0;
                flag->partial.len = 0;
            }

            if (flag->partial.len > 0) {
                len = sprintf(FetchRequest->sendBuf, "<%lu> {%lu}\r\n", flag->partial.start, flag->partial.len);
                SendClient(client, FetchRequest->sendBuf, len);
                SendClient(client, resultHeader + flag->partial.start, flag->partial.len);
            } else {
                len = sprintf(FetchRequest->sendBuf, "<%lu> {0}\r\n", flag->partial.start);
                SendClient(client, FetchRequest->sendBuf, len);
            }
        }

        MemFree(resultHeader);
    } else {
        if (!flag->isPartial) {
            SendClient(client, " {0}\r\n", 6); 
        } else {
            len = sprintf(FetchRequest->sendBuf, "<%lu> {0}\r\n", flag->partial.start);
            SendClient(client, FetchRequest->sendBuf, len);
        }
    }

    return(0);
}

__inline static BOOL
HasPart(MDBValueStruct *mimeInfo, BodyPartRequestStruct *request)
{
    unsigned char type[MIME_TYPE_LEN+1];
    unsigned char dummy[MIME_NAME_LEN+1];
    unsigned long dummyLong;

    long depthChange;

    BOOL haveRfc822 = TRUE;
    unsigned long mimeResponseLine = 0;
    unsigned long currentDepth = 0;

    unsigned long matchDepth = 0;
    unsigned long matchDepthPart = 0;

    do {
        switch (atol(mimeInfo->Value[mimeResponseLine])) {
        case 2002: {
            ParseMIMEDLine(mimeInfo->Value[mimeResponseLine] + 5, 
                           type, MIME_TYPE_LEN, 
                           dummy, MIME_NAME_LEN, 
                           dummy, MIME_NAME_LEN, 
                           dummy, MIME_NAME_LEN, 
                           dummy, MIME_NAME_LEN, 
                           &(request->mimeOffset), 
                           &(request->mimeLen), 
                           &(request->partOffset), 
                           &(request->partLen), 
                           &(request->messageHeaderLen), 
                           &dummyLong);  

            if (toupper(type[0]) == 'M') {
                if (XplStrCaseCmp(type, "MULTIPART") == 0) {
                    if (!haveRfc822) {
                        if (currentDepth == matchDepth) {
                            matchDepthPart++;
                        }
                        depthChange = 1;
                    } else {
                        depthChange = 0;
                    }
                    haveRfc822 = FALSE;
                } else if (XplStrCaseCmp(type, "MESSAGE") == 0) {
                    if (!haveRfc822) {
                        if (currentDepth == matchDepth) {
                            matchDepthPart++;
                        }
                        depthChange = 1;
                        haveRfc822 = TRUE;
                    } else {
                        depthChange = 0;
                    }
                } else {
                    if (currentDepth == matchDepth) {
                        matchDepthPart++;
                    }
                    haveRfc822 = FALSE;
                    depthChange = 0;
                }
            } else {
                if (currentDepth == matchDepth) {
                    matchDepthPart++;
                }
                depthChange = 0;
                haveRfc822 = FALSE;
            }

            /* Check to see if this is the part we are looking for */
            if (currentDepth == matchDepth) {
                if (matchDepthPart == request->part[matchDepth]) {
                    matchDepth++;
                    if (matchDepth == request->depth) {
                        /* We found the part!!!! */
                        request->isMessage = haveRfc822;
                        return(TRUE);

                    }
                
                    matchDepthPart = 0;
                } else if (matchDepthPart > request->part[matchDepth]) {
                    /* Not Found! Send Nothing */
                    return(FALSE);
                }
            } else if (currentDepth < matchDepth) {
                /* Not Found! Send Nothing */
                return(FALSE);
            }

            break;
        }

        case 2003:
        case 2004: {
            if (currentDepth > 0) {
                depthChange = -1;
            } else {
                depthChange = 0;
            }
            break;
        }
        }

        mimeResponseLine++;
        if (mimeResponseLine < mimeInfo->Used) {
            if (depthChange == 0) {
                continue;
            }

            if (depthChange > 0) {
                currentDepth++;
                if (currentDepth == matchDepth) {
                    matchDepthPart = 0;
                }
                continue;
            }
            
            currentDepth--;
            continue;
            
        }

        /* Not Found; Send Nothing */
        return(FALSE);
    } while(TRUE);
}

static long 
FetchFlagResponderBodyPartNumber(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long ccode;
    long count;
    unsigned long i;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[%lu", flag->bodyPart.part[0]));
    for (i = 1; i < flag->bodyPart.depth; i++) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".%lu", flag->bodyPart.part[i]));
    }

    if (HasPart(FetchRequest->currentMessage.mimeInfo, &(flag->bodyPart))) {
        SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->bodyPart.partOffset, flag->bodyPart.partLen));

        ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
        if ((ccode == 2021) || (ccode == 2025)) {
            count = atol(FetchRequest->responseBuf);
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "] {%lu}\r\n", count));
            if (ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf)) == 1000) {

                return(0);
            }
        }
    
        return(ccode);
    }

    SendClient(client, "] {0}\r\n", strlen("] {0}\r\n"));
    return(0);
}

static long 
FetchFlagResponderBodyPartNumberPartial(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long ccode;
    long count;
    unsigned long i;
    long resultLen;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[%lu", flag->bodyPart.part[0]));
    for (i = 1; i < flag->bodyPart.depth; i++) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".%lu", flag->bodyPart.part[i]));
    }

    if (HasPart(FetchRequest->currentMessage.mimeInfo, &(flag->bodyPart))) {
        if (flag->partial.start < flag->bodyPart.partLen) {
            if ((flag->partial.start + flag->partial.len) < flag->bodyPart.partLen) {
                /* all in range */
                resultLen = flag->partial.len;
            } else {
                /* len out of range */
                resultLen = flag->bodyPart.partLen - flag->partial.start;
            }
            SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->bodyPart.partOffset + flag->partial.start, resultLen));
            ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
            if ((ccode == 2021) || (ccode == 2025)) {
                count = atol(FetchRequest->responseBuf);
                SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "]<%lu> {%lu}\r\n", flag->partial.start, count));
                if (ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf)) == 1000) {
                    return(0);
                }
            }
            /* nmap error */
            return(ccode);
        }

        /* start out of range */
    }

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "]<%lu> {0}\r\n", flag->partial.start));
    return(0);
}

static long 
FetchFlagResponderBodyPartText(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long ccode;
    long count;
    unsigned long i;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[%lu", flag->bodyPart.part[0]));
    for (i = 1; i < flag->bodyPart.depth; i++) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".%lu", flag->bodyPart.part[i]));
    }

    if (HasPart(FetchRequest->currentMessage.mimeInfo, &(flag->bodyPart)) && flag->bodyPart.isMessage) {
        SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->bodyPart.partOffset + flag->bodyPart.messageHeaderLen, flag->bodyPart.partLen - flag->bodyPart.messageHeaderLen));

        ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
        if ((ccode == 2021) || (ccode == 2025)) {
            count = atol(FetchRequest->responseBuf);
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".TEXT] {%lu}\r\n", count));
            if (ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf)) == 1000) {

                return(0);
            }
        }
    
        return(ccode);
    }

    SendClient(client, ".TEXT] {0}\r\n", strlen(".TEXT] {0}\r\n"));
    return(0);
}

static long 
FetchFlagResponderBodyPartTextPartial(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long ccode;
    long count;
    unsigned long i;
    long resultLen;
    unsigned long textStart;
    unsigned long textLen;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[%lu", flag->bodyPart.part[0]));
    for (i = 1; i < flag->bodyPart.depth; i++) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".%lu", flag->bodyPart.part[i]));
    }

    if (HasPart(FetchRequest->currentMessage.mimeInfo, &(flag->bodyPart)) && flag->bodyPart.isMessage) {
        textStart = flag->bodyPart.partOffset + flag->bodyPart.messageHeaderLen;
        textLen = flag->bodyPart.partLen - flag->bodyPart.messageHeaderLen;
        if (flag->partial.start < textLen) {
            if ((flag->partial.start + flag->partial.len) < textLen) {
                /* all in range */
                resultLen = flag->partial.len;
            } else {
                /* len out of range */
                resultLen = textLen - flag->partial.start;
            }
            SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, textStart + flag->partial.start, resultLen));
            ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
            if ((ccode == 2021) || (ccode == 2025)) {
                count = atol(FetchRequest->responseBuf);
                SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".TEXT]<%lu> {%lu}\r\n", flag->partial.start, count));
                if (ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf)) == 1000) {
                    return(0);
                }
            }
            /* nmap error */
            return(ccode);
        }

        /* start out of range */
    }

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".TEXT]<%lu> {0}\r\n", flag->partial.start));
    return(0);
}

static long 
FetchFlagResponderBodyPartHeader(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long ccode;
    long count;
    unsigned long i;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[%lu", flag->bodyPart.part[0]));
    for (i = 1; i < flag->bodyPart.depth; i++) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".%lu", flag->bodyPart.part[i]));
    }

    if (HasPart(FetchRequest->currentMessage.mimeInfo, &(flag->bodyPart)) && flag->bodyPart.isMessage ) {
        SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->bodyPart.partOffset, flag->bodyPart.messageHeaderLen));

        ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
        if ((ccode == 2021) || (ccode == 2025)) {
            count = atol(FetchRequest->responseBuf);
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".HEADER] {%lu}\r\n", count));
            if (ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf)) == 1000) {

                return(0);
            }
        }
        return(ccode);
    }
    SendClient(client, ".HEADER] {0}\r\n", strlen(".HEADER] {0}\r\n"));
    return(0);
}

static long 
FetchFlagResponderBodyPartHeaderPartial(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long ccode;
    long count;
    unsigned long i;
    long resultLen;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[%lu", flag->bodyPart.part[0]));
    for (i = 1; i < flag->bodyPart.depth; i++) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".%lu", flag->bodyPart.part[i]));
    }
    if (HasPart(FetchRequest->currentMessage.mimeInfo, &(flag->bodyPart)) && flag->bodyPart.isMessage) {
        if (flag->partial.start < flag->bodyPart.messageHeaderLen) {
            if ((flag->partial.start + flag->partial.len) < flag->bodyPart.messageHeaderLen) {
                /* all in range */
                resultLen = flag->partial.len;
            } else {
                /* len out of range */
                resultLen = flag->bodyPart.messageHeaderLen - flag->partial.start;
            }
            SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->bodyPart.partOffset + flag->partial.start, resultLen));
            ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
            if ((ccode == 2021) || (ccode == 2025)) {
                count = atol(FetchRequest->responseBuf);
                SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".HEADER]<%lu> {%lu}\r\n", flag->partial.start, count));
                if (ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf)) == 1000) {
                    return(0);
                }
            }
            /* nmap error */
            return(ccode);
        }
        /* start out of range */
    }
    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".HEADER]<%lu> {0}\r\n", flag->partial.start));
    return(0);
}

static long 
FetchFlagResponderBodyPartHeaderFields(void *param1, void *param2, void *param3)
{
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
  
    long ccode;
    unsigned long j;
    unsigned char *fullHeader;
    unsigned long fullHeaderLen;
    unsigned char *resultHeader;
    unsigned long resultHeaderLen;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[%lu", flag->bodyPart.part[0]));
    for (j = 1; j < flag->bodyPart.depth; j++) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".%lu", flag->bodyPart.part[j]));
    }

    if (!(flag->fieldList.skip)) {
        SendClient(client, ".HEADER.FIELDS (", strlen(".HEADER.FIELDS ("));
    } else {
        SendClient(client, ".HEADER.FIELDS.NOT (", strlen(".HEADER.FIELDS.NOT ("));
    }
    for (j = 0; j < flag->fieldList.count - 1; j++) {
        SendClient(client, FetchRequest->sendBuf, snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "\"%s\" ", flag->fieldList.field[j]));
    } 
    SendClient(client, FetchRequest->sendBuf, snprintf(FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf), "\"%s\")]", flag->fieldList.field[j]));

    if (HasPart(FetchRequest->currentMessage.mimeInfo, &(flag->bodyPart)) && flag->bodyPart.isMessage) {
        SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->bodyPart.partOffset, flag->bodyPart.messageHeaderLen));
        ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
        if ((ccode == 2021) || (ccode == 2025)) {
            fullHeaderLen = atoi(FetchRequest->responseBuf);
            ccode = GetDataFromNmap(client, &fullHeader, fullHeaderLen, FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf));
            if (ccode == 0) {
                resultHeader = GetSubHeader(fullHeader, fullHeaderLen, flag->fieldList.field, flag->fieldList.count, flag->fieldList.skip, &resultHeaderLen);
                MemFree(fullHeader);
                if (resultHeader) {
                    if (!flag->isPartial) {
                        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, " {%lu}\r\n", resultHeaderLen));
                        SendClient(client, resultHeader, resultHeaderLen);
                        MemFree(resultHeader);
                        return(0);
                    }

                    if (flag->partial.start < resultHeaderLen) {
                        if ((flag->partial.start + flag->partial.len) < resultHeaderLen) {
                            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "<%lu> {%lu}\r\n", flag->partial.start, flag->partial.len));
                            SendClient(client, resultHeader + flag->partial.start, flag->partial.len);
                            MemFree(resultHeader);
                            return(0);
                        }

                        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "<%lu> {%lu}\r\n", flag->partial.start, resultHeaderLen - flag->partial.start));
                        SendClient(client, resultHeader + flag->partial.start, resultHeaderLen - flag->partial.start);
                        MemFree(resultHeader);
                        return(0);
                    }

                    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "<%lu> {0}\r\n", flag->partial.start));
                    MemFree(resultHeader);
                    return(0);
                }
                    
                MemFree(fullHeader);
                return(-1);
            }
        }

        return(ccode);
    }

    if (!flag->isPartial) {
        SendClient(client, " {0}\r\n", strlen(" {0}\r\n"));
        return(0);
    }

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "<%lu> {0}\r\n", flag->partial.start));
    return(0);
}

static long 
FetchFlagResponderBodyPartMime(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long ccode;
    long count;
    unsigned long i;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[%lu", flag->bodyPart.part[0]));
    for (i = 1; i < flag->bodyPart.depth; i++) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".%lu", flag->bodyPart.part[i]));
    }
    if (HasPart(FetchRequest->currentMessage.mimeInfo, &(flag->bodyPart))) {
        SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->bodyPart.mimeOffset, flag->bodyPart.mimeLen));

        ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
        if ((ccode == 2021) || (ccode == 2025)) {
            count = atol(FetchRequest->responseBuf);
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".MIME] {%lu}\r\n", count));
            if (ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf)) == 1000) {

                return(0);
            }
        }
        return(ccode);
    }
    SendClient(client, ".MIME] {0}\r\n", strlen(".MIME] {0}\r\n"));
    return(0);
}

static long 
FetchFlagResponderBodyPartMimePartial(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1;
    FetchStruct *FetchRequest = (FetchStruct *)param2;
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    long ccode;
    long count;
    unsigned long i;

    SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BODY[%lu", flag->bodyPart.part[0]));
    for (i = 1; i < flag->bodyPart.depth; i++) {
        SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".%lu", flag->bodyPart.part[i]));
    }
    if (HasPart(FetchRequest->currentMessage.mimeInfo, &(flag->bodyPart)) && (flag->partial.start < flag->bodyPart.mimeLen)) {
        if ((flag->partial.start + flag->partial.len) < flag->bodyPart.mimeLen) {
            count = flag->partial.len;
        } else {
            count = flag->bodyPart.mimeLen - flag->partial.start;
        }
        SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "BRAW %lu %lu %lu\r\n", FetchRequest->currentMessage.nmapId, flag->bodyPart.mimeOffset + flag->partial.start, count));
        ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
        if ((ccode == 2021) || (ccode == 2025)) {
            count = atol(FetchRequest->responseBuf);
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, ".MIME] {%lu}\r\n", count));
            if (ReadAndSendNmapResponse(client, count, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf)) == 1000) {
                return(0);
            }
        }
        return(ccode);
    }
    SendClient(client, ".MIME] {0}\r\n", strlen(".MIME] {0}\r\n"));
    return(0);
}

static long 
FetchFlagResponderBodyPart(void *param1, void *param2, void *param3)
{
    FetchFlagStruct *flag = (FetchFlagStruct *)param3;
    return(flag->bodyPart.responder(param1, param2, param3));
}

static long
FetchFlagResponderFast(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1; 
    long ccode;

    ccode = FetchFlagResponderFlags(param1, param2, param3);
    if (ccode == 0) {
        SendClient(client, " ", 1);
        ccode = FetchFlagResponderInternalDate(param1, param2, param3);
        if (ccode == 0) {
            SendClient(client, " ", 1);
            return(FetchFlagResponderRfc822Size(param1, param2, param3));
        }
    }
    
    return(ccode);
}

static long 
FetchFlagResponderFull(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1; 
    long ccode;

    ccode = FetchFlagResponderFlags(param1, param2, param3);
    if (ccode == 0) {
        SendClient(client, " ", 1);
        ccode = FetchFlagResponderInternalDate(param1, param2, param3);
        if (ccode == 0) {
            SendClient(client, " ", 1);
            ccode = FetchFlagResponderRfc822Size(param1, param2, param3);
            if (ccode == 0) {
                SendClient(client, " ", 1);
                ccode = FetchFlagResponderEnvelope(param1, param2, param3);
                if (ccode == 0) {
                    SendClient(client, " ", 1);
                    return(FetchFlagResponderBodyStructure(param1, param2, param3));
                }
            }
        }
    }
    return(ccode);
}

static long 
FetchFlagResponderAll(void *param1, void *param2, void *param3)
{
    ImapClient *client = (ImapClient *)param1; 
    long ccode;

    ccode = FetchFlagResponderFlags(param1, param2, param3);
    if (ccode == 0) {
        SendClient(client, " ", 1);
        ccode = FetchFlagResponderInternalDate(param1, param2, param3);
        if (ccode == 0) {
            SendClient(client, " ", 1);
            ccode = FetchFlagResponderRfc822Size(param1, param2, param3);
            if (ccode == 0) {
                SendClient(client, " ", 1);
                return(FetchFlagResponderEnvelope(param1, param2, param3));
            }
        }
    }
    return(ccode);
}

static unsigned long
ParseFetchFlagPartial(unsigned char **parentPtr, FetchFlagStruct *flag)
{
    unsigned char *ptr;

    ptr = *parentPtr;
    if (isdigit(*ptr)) {
        flag->partial.start = atol(ptr);
        do {
            ptr++;
            if (isdigit(*ptr)) {
                continue;
            }
            break;
        } while(TRUE);

        if (*ptr == '.') {
            ptr++;
            if (isdigit(*ptr)) {
                flag->partial.len = atol(ptr);
                do {
                    ptr++;
                } while(isdigit(*ptr));
                if (*ptr == '>') {
                    flag->isPartial = TRUE;
                    *parentPtr = ptr + 1;
                    return(0);
                }
            }
        }
    }

    return(F_PARSE_ERROR);
}

__inline static unsigned long
ParseHeaderFields(unsigned char **parentPtr, FetchFlagStruct *flag)
{
    unsigned char *start;
    unsigned char *end;
    unsigned char *current;
    unsigned char *fieldBuffer;
    unsigned long fieldBufferLen;
    unsigned long spaceCount;
    unsigned char len;
    unsigned long fieldCount;
    unsigned long ccode;

    end = strchr(*parentPtr, ')');
    if (end) {
        fieldBufferLen = end - *parentPtr;
        fieldBuffer = MemMalloc(fieldBufferLen + 1);
        if (fieldBuffer) {
            flag->hasAllocated |= ALLOCATED_FIELDS;
            memcpy(fieldBuffer, *parentPtr, fieldBufferLen);
            fieldBuffer[fieldBufferLen] = '\0';

            start = fieldBuffer;
            end = fieldBuffer + fieldBufferLen;
            current = start;

            spaceCount = 0;
            do {
                if (*current != ' ') {
                    current++;
                    continue;
                }
                
                spaceCount++;
                current++;
            } while (current < end);

            flag->fieldList.fieldRequest = fieldBuffer;
            if (spaceCount < FETCH_FIELD_ALLOC_THRESHOLD) {
                flag->fieldList.field = &(flag->fieldList.fieldEmbedded[0]);
            } else {
                flag->fieldList.field = MemCalloc((spaceCount + 1), sizeof(unsigned char *));
                if (flag->fieldList.field) {
                    flag->hasAllocated |= ALLOCATED_FIELD_POINTERS;
                } else {
                    return(F_SYSTEM_ERROR);
                }
            }

            fieldCount = 0;
            current = start;

            do {
                if (*current == '"') {
                    current++;
                    start = current;
                    do {
                        if (current < end) {
                            if (*current != '"') {
                                current++;
                                continue;
                            }

                            if (*(current - 1) != '\\') {
                                len = current - start;
                                *current = '\0';
                                current++;
                                break;
                            }

                            current++;
                            continue;
                        }

                        return(F_PARSE_ERROR);
                    } while (TRUE);
                } else {
                    start = current;
                    do {
                        if (current < end) {
                            if (*current != ' ') {
                                current++;
                                continue;
                            }
                        }

                        len = current - start;
                            
                        break;
                    } while (TRUE);
                }

                /* we've got a header name */
                if (len > 0) {
                    flag->fieldList.field[fieldCount] = start;
                    fieldCount++;
                    flag->fieldList.count = fieldCount;

                    if (current < end) {
                        if (*current == ' ') {
                            *current = '\0';
                            current++;
                            continue;
                        }

                        return(F_PARSE_ERROR);
                    }
                    
                    break;
                }
                
                return(F_PARSE_ERROR);
            } while (current < end);

            /* we have fields; lets go back to the original buffer and see what comes after the fields */
            current = *parentPtr + fieldBufferLen + 1;
            if (*current == ']') {
                current++;
                if (*current != '<') {
                    flag->isPartial = FALSE;
                    *parentPtr = current;
                    return(0);
                }

                current++;
                ccode = ParseFetchFlagPartial(&current, flag);
                if (ccode == 0) {
                    *parentPtr = current;
                    return(0);
                }

                return(ccode);
            }
            
            return(F_PARSE_ERROR);
        }
        return(F_SYSTEM_ERROR);
    }

    return(F_PARSE_ERROR);
}

__inline static unsigned long
ParseFetchFlagHeaderFields(unsigned char **parentPtr, FetchFlagStruct *flag)
{
    return(ParseHeaderFields(parentPtr, flag));
}

__inline static unsigned long
ParseFetchFlagHeaderFieldsNot(unsigned char **parentPtr, FetchFlagStruct *flag)
{
    flag->fieldList.skip = TRUE;
    return(ParseHeaderFields(parentPtr, flag));
}

static FetchFlag FetchFlagsSection[] = {
    {"TEXT]", sizeof("TEXT]") - 1, F_BODY_TEXT, NULL, FetchFlagResponderBodyPartText},
    {"TEXT]<", sizeof("TEXT]<") - 1, F_BODY_TEXT, ParseFetchFlagPartial, FetchFlagResponderBodyPartTextPartial},
    {"MIME]", sizeof("MIME]") - 1, F_BODY_MIME, NULL, FetchFlagResponderBodyPartMime},
    {"MIME]<", sizeof("MIME]<") - 1, F_BODY_MIME, ParseFetchFlagPartial, FetchFlagResponderBodyPartMimePartial},
    {"HEADER]", sizeof("HEADER]") - 1, F_BODY_HEADER, NULL, FetchFlagResponderBodyPartHeader},
    {"HEADER]<", sizeof("HEADER]<") - 1, F_BODY_HEADER, ParseFetchFlagPartial, FetchFlagResponderBodyPartHeaderPartial},
    {"HEADER.FIELDS (", sizeof("HEADER.FIELDS (") - 1, F_BODY_HEADER_FIELDS, ParseFetchFlagHeaderFields, FetchFlagResponderBodyPartHeaderFields},
    {"HEADER.FIELDS.NOT (", sizeof("HEADER.FIELDS.NOT (") - 1, F_BODY_HEADER_FIELDS_NOT, ParseFetchFlagHeaderFieldsNot, FetchFlagResponderBodyPartHeaderFields},
    {NULL, 0, 0, NULL, NULL}
};

static unsigned long
ParseFetchFlagBodyPart(unsigned char **parentPtr, FetchFlagStruct *flag)
{
    unsigned char *ptr;
    unsigned long *tmpPart;
    long sectionId;
    unsigned long ccode;

    ptr = *parentPtr;

    flag->bodyPart.part = (unsigned long *)&(flag->bodyPart.partEmbedded);
    do {
        if (isdigit(*ptr)) {
            if (flag->bodyPart.depth < PART_DEPTH_ALLOC_THRESHOLD) {
                /* No allocation needed; use space in the structure */
                ;
            } else if (flag->bodyPart.depth > PART_DEPTH_ALLOC_THRESHOLD) {
                /* Grow the existing allocated structure */
                tmpPart = MemRealloc(flag->bodyPart.part, sizeof(unsigned long) * (flag->bodyPart.depth + 1));
                if (tmpPart) {
                    flag->bodyPart.part = tmpPart;
                } else {
                    return(F_SYSTEM_ERROR);
                }
            } else {
                /* switch from embedded space to allocated space */
                tmpPart = MemMalloc(sizeof(unsigned long) * (PART_DEPTH_ALLOC_THRESHOLD + 1));
                if (tmpPart) {
                    flag->hasAllocated |= ALLOCATED_PART_NUMBERS;
                    /* copy the data from embedded array to allocated array */
                    memcpy(tmpPart, flag->bodyPart.part, sizeof(unsigned long) * PART_DEPTH_ALLOC_THRESHOLD);
                    flag->bodyPart.part = tmpPart;
                } else {
                    return(F_SYSTEM_ERROR);
                }
            }

            flag->bodyPart.part[flag->bodyPart.depth] = atoi(ptr);
            flag->bodyPart.depth++;

            do {
                ptr++;
            } while (isdigit(*ptr));

            if (*ptr == '.') {
                ptr++;
                continue;
            }
            
            if (*ptr == ']') {
                ptr++;
                if (*ptr != '<') {
                    flag->bodyPart.responder = FetchFlagResponderBodyPartNumber;
                    *parentPtr = ptr;
                    return(0);
                }

                ptr++;
                ccode = ParseFetchFlagPartial(&ptr, flag);
                if (ccode == 0) {
                    flag->bodyPart.responder = FetchFlagResponderBodyPartNumberPartial;
                    *parentPtr = ptr;
                    return(0);
                }

                return(ccode);
            }
            
            return(F_PARSE_ERROR);
        }

        sectionId = HulaKeywordFind(FetchFlagsSectionIndex, ptr);
        if (sectionId != -1) {
            ptr += FetchFlagsSection[sectionId].nameLen;
            flag->bodyPart.responder = FetchFlagsSection[sectionId].responder;
            if (FetchFlagsSection[sectionId].parser == NULL) {
                *parentPtr = ptr;
                return(0);
            }

            ccode = FetchFlagsSection[sectionId].parser(&ptr, flag);
            if (ccode == 0) {
                *parentPtr = ptr;
                return(0);
            }
            
            return(ccode);
        }

        return(F_PARSE_ERROR);
    } while(TRUE);
}


#define FETCH_ATT_FLAGS \
    {"UID", sizeof("UID") - 1, F_UID, NULL, FetchFlagResponderUid}, \
    {"FLAGS", sizeof("FLAGS") - 1, F_FLAGS, NULL, FetchFlagResponderFlags}, \
    {"ENVELOPE", sizeof("ENVELOPE") - 1, F_ENVELOPE, NULL, FetchFlagResponderEnvelope}, \
    {"XSENDER", sizeof("XSENDER") - 1, F_XSENDER, NULL, FetchFlagResponderXSender}, \
    {"INTERNALDATE", sizeof("INTERNALDATE") - 1, F_INTERNALDATE, NULL, FetchFlagResponderInternalDate}, \
    {"BODYSTRUCTURE", sizeof("BODYSTRUCTURE") - 1, F_BODYSTRUCTURE, NULL, FetchFlagResponderBodyStructure}, \
    {"RFC822.TEXT", sizeof("RFC822.TEXT") - 1, F_RFC822_TEXT | F_BODY_SEEN, NULL, FetchFlagResponderRfc822Text}, \
    {"RFC822.SIZE", sizeof("RFC822.SIZE") - 1, F_RFC822_SIZE, NULL, FetchFlagResponderRfc822Size}, \
    {"RFC822.HEADER", sizeof("RFC822.HEADER") - 1, F_RFC822_HEADER, NULL, FetchFlagResponderRfc822Header}, \
    {"RFC822", sizeof("RFC822") - 1, F_RFC822_BOTH | F_BODY_SEEN, NULL, FetchFlagResponderRfc822}, \
    {"BODY", sizeof("BODY") - 1, F_BODY, NULL, FetchFlagResponderBody}, \
    {"BODY[HEADER]", sizeof("BODY[HEADER]") - 1, F_BODY_HEADER | F_BODY_SEEN, NULL, FetchFlagResponderBodyHeader}, \
    {"BODY[HEADER]<", sizeof("BODY[HEADER]<") - 1, F_BODY_HEADER_PARTIAL | F_BODY_SEEN, ParseFetchFlagPartial, FetchFlagResponderBodyHeaderPartial}, \
    {"BODY[TEXT]", sizeof("BODY[TEXT]") - 1, F_BODY_TEXT | F_BODY_SEEN, NULL, FetchFlagResponderBodyText}, \
    {"BODY[TEXT]<", sizeof("BODY[TEXT]<") - 1, F_BODY_TEXT_PARTIAL | F_BODY_SEEN, ParseFetchFlagPartial, FetchFlagResponderBodyTextPartial}, \
    {"BODY[]", sizeof("BODY[]") - 1, F_BODY_BOTH | F_BODY_SEEN, NULL, FetchFlagResponderBodyBoth}, \
    {"BODY[]<", sizeof("BODY[]<") - 1, F_BODY_BOTH_PARTIAL | F_BODY_SEEN, ParseFetchFlagPartial, FetchFlagResponderBodyBothPartial}, \
    {"BODY[MIME]", sizeof("BODY[MIME]") - 1, F_BODY_MIME | F_BODY_SEEN, NULL, FetchFlagResponderBodyMime}, \
    {"BODY[MIME]<", sizeof("BODY[MIME]<") - 1, F_BODY_MIME_PARTIAL | F_BODY_SEEN, ParseFetchFlagPartial, FetchFlagResponderBodyMimePartial}, \
    {"BODY[HEADER.FIELDS (", sizeof("BODY[HEADER.FIELDS (") - 1, F_BODY_HEADER_FIELDS | F_BODY_SEEN, ParseFetchFlagHeaderFields, FetchFlagResponderBodyHeaderFields}, \
    {"BODY[HEADER.FIELDS.NOT (", sizeof("BODY[HEADER.FIELDS.NOT (") - 1, F_BODY_HEADER_FIELDS_NOT | F_BODY_SEEN, ParseFetchFlagHeaderFieldsNot, FetchFlagResponderBodyHeaderFields}, \
    {"BODY[", sizeof("BODY[") - 1, F_BODY_PART | F_BODY_SEEN, ParseFetchFlagBodyPart, FetchFlagResponderBodyPart}, \
    {"BODY.PEEK[TEXT]", sizeof("BODY.PEEK[TEXT]") - 1, F_BODY_TEXT, NULL, FetchFlagResponderBodyText}, \
    {"BODY.PEEK[TEXT]<", sizeof("BODY.PEEK[TEXT]<") - 1, F_BODY_TEXT_PARTIAL, ParseFetchFlagPartial, FetchFlagResponderBodyTextPartial}, \
    {"BODY.PEEK[HEADER]", sizeof("BODY.PEEK[HEADER]") - 1, F_BODY_HEADER, NULL, FetchFlagResponderBodyHeader}, \
    {"BODY.PEEK[HEADER]<", sizeof("BODY.PEEK[HEADER]<") - 1, F_BODY_HEADER | F_BODY_HEADER_PARTIAL, ParseFetchFlagPartial, FetchFlagResponderBodyHeaderPartial}, \
    {"BODY.PEEK[]", sizeof("BODY.PEEK[]") - 1, F_BODY_BOTH, NULL, FetchFlagResponderBodyBoth}, \
    {"BODY.PEEK[]<", sizeof("BODY.PEEK[]<") - 1, F_BODY_BOTH | F_BODY_BOTH_PARTIAL, ParseFetchFlagPartial, FetchFlagResponderBodyBothPartial}, \
    {"BODY.PEEK[MIME]", sizeof("BODY.PEEK[MIME]") - 1, F_BODY_MIME, NULL, FetchFlagResponderBodyMime}, \
    {"BODY.PEEK[MIME]<", sizeof("BODY.PEEK[MIME]<") - 1, F_BODY_MIME | F_BODY_MIME_PARTIAL, ParseFetchFlagPartial, FetchFlagResponderBodyMimePartial}, \
    {"BODY.PEEK[HEADER.FIELDS (", sizeof("BODY.PEEK[HEADER.FIELDS (") - 1, F_BODY_HEADER_FIELDS, ParseFetchFlagHeaderFields, FetchFlagResponderBodyHeaderFields}, \
    {"BODY.PEEK[HEADER.FIELDS.NOT (", sizeof("BODY.PEEK[HEADER.FIELDS.NOT (") - 1, F_BODY_HEADER_FIELDS_NOT, ParseFetchFlagHeaderFieldsNot, FetchFlagResponderBodyHeaderFields}, \
    {"BODY.PEEK[", sizeof("BODY.PEEK[") - 1, F_BODY_PART, ParseFetchFlagBodyPart, FetchFlagResponderBodyPart},

static FetchFlag FetchFlagsSolo[] = {
    {"FAST", sizeof("FAST") - 1, F_FLAGS | F_INTERNALDATE | F_RFC822_SIZE, NULL, FetchFlagResponderFast},
    {"FULL", sizeof("FULL") - 1, F_FLAGS | F_INTERNALDATE | F_RFC822_SIZE | F_ENVELOPE | F_BODYSTRUCTURE, NULL, FetchFlagResponderFull},
    {"ALL", sizeof("ALL") - 1, F_FLAGS | F_INTERNALDATE | F_RFC822_SIZE | F_ENVELOPE, NULL, FetchFlagResponderAll},
    FETCH_ATT_FLAGS
    {NULL, 0, 0, NULL, NULL}
};

static FetchFlag FetchFlagsAtt[] = {
    FETCH_ATT_FLAGS
    {NULL, 0, 0, NULL, NULL}
};


__inline static HulaKeywordIndex *
InitializeIndexFetchFlag(FetchFlag *FetchFlagList)
{
    HulaKeywordIndex *index;
    unsigned char **fetchFlags;
    unsigned long count;
    unsigned long i;

    count = 0;

    while(FetchFlagList[count].name != NULL) {
        count++;
    }
    
    fetchFlags = MemMalloc(sizeof(unsigned char *) * (count + 1));
    if (!fetchFlags) {
        return(NULL);
    }

    for (i = 0; i < count; i++) {
        fetchFlags[i] = FetchFlagList[i].name;
    }

    fetchFlags[count] = NULL;

    index = HulaKeywordIndexCreate(fetchFlags, TRUE);
    MemFree(fetchFlags);
    return(index);
}

__inline static void
RemoveLastFlag(FetchStruct *FetchRequest)
{
    FetchRequest->flagCount--;
}

__inline static FetchFlagStruct *
AddNewFlag(FetchStruct *FetchRequest, FetchResponder responder)
{
    FetchFlagStruct *newFlag;
    FetchFlagStruct *tmpFlag;

    if (FetchRequest->flagCount < FETCH_FLAG_ALLOC_THRESHOLD) {
        FetchRequest->flag = FetchRequest->flagEmbedded;
        newFlag = &(FetchRequest->flag[FetchRequest->flagCount]);
        newFlag->responder = responder;
        FetchRequest->flagCount++;
        return(newFlag);
    }

    if (FetchRequest->flagCount > FETCH_FLAG_ALLOC_THRESHOLD) {
        tmpFlag = MemRealloc(FetchRequest->flag, sizeof(FetchFlagStruct) * (FetchRequest->flagCount + 1));
        if (tmpFlag) {
            FetchRequest->flag = tmpFlag;
            newFlag = &(FetchRequest->flag[FetchRequest->flagCount]);
            memset(newFlag, 0, sizeof(FetchFlagStruct));
            newFlag->responder = responder;
            FetchRequest->flagCount++;
            return(newFlag);
        }

        return(NULL);
    }

    tmpFlag = MemMalloc(sizeof(FetchFlagStruct) * (FETCH_FLAG_ALLOC_THRESHOLD + 1));
    if (tmpFlag) {
        FetchRequest->hasAllocated |= ALLOCATED_FLAGS;
        memcpy(tmpFlag, FetchRequest->flagEmbedded, sizeof(FetchFlagStruct) *  FETCH_FLAG_ALLOC_THRESHOLD);
        FetchRequest->flag = tmpFlag;
        newFlag = &(FetchRequest->flag[FetchRequest->flagCount]);
        memset(newFlag, 0, sizeof(FetchFlagStruct));
        newFlag->responder = responder;
        FetchRequest->flagCount++;
        return(newFlag);
    }

    return(NULL);
}

__inline static long
ParseFetchArguments(ImapClient *client, unsigned char *ptr, FetchStruct *FetchRequest)
{
    unsigned char *Items;
    long allFlags = FetchRequest->flags;
    long flagID;
    FetchFlag flagInfo;
    FetchFlagStruct *currentFlag;
    long ccode;

    /* Grab the range */
    ptr = GrabArgument(client, ptr, &FetchRequest->messageSet);

    if (FetchRequest->messageSet) {
        if (*ptr != '\0') {
            /* parse the fetch flags */
            if (*ptr == '(') {
                /* we have parens so we can have one or more 'fetch-att' flags */
                ptr = GrabArgument(client, ptr, &Items);
                if (Items) {
                    ptr = Items - 1;
                    do {
                        ptr++;
                        flagID = HulaKeywordFind(FetchFlagsAttIndex, ptr);
                        if (flagID != -1) {
                            flagInfo = FetchFlagsAtt[flagID];
                            ptr += flagInfo.nameLen;
                            currentFlag = AddNewFlag(FetchRequest, flagInfo.responder);
                            if (currentFlag) {
                                if (flagInfo.parser == NULL) {
                                    /* make sure uid only gets added once; it gets added automatically for uid fetch requests */
                                    if ((!(flagInfo.value & F_UID)) || (!(allFlags & F_UID))) {
                                        allFlags |= flagInfo.value;
                                    } else {
                                        RemoveLastFlag(FetchRequest);
                                    }
                                    continue;
                                }
                
                                allFlags |= flagInfo.value;

                                ccode = flagInfo.parser(&ptr, currentFlag);
                                FetchRequest->hasAllocated |= currentFlag->hasAllocated;
                                if (ccode == 0) {
                                    continue;
                                }
                                return(ccode);
                            }
                            return(F_SYSTEM_ERROR);
                        }
                        MemFree(Items);
                        return(F_PARSE_ERROR);
                    } while (*ptr == ' ');

                    if (*ptr == '\0') {
                        MemFree(Items);
                        return(allFlags);
                    }

                    MemFree(Items);
                }

                return(F_PARSE_ERROR);
            }

            /* there are no parens so there can only be one macro or one 'fetch-att' flag */
            flagID = HulaKeywordFind(FetchFlagsSoloIndex, ptr);
            if (flagID != -1) {
                flagInfo = FetchFlagsSolo[flagID];
                            
                ptr += flagInfo.nameLen;
                currentFlag = AddNewFlag(FetchRequest, flagInfo.responder);
                if (currentFlag) {
                    if (flagInfo.parser == NULL) {
                        if (*ptr == '\0') {
                            if ((!(flagInfo.value & F_UID)) || (!(allFlags & F_UID))) {
                                allFlags |= flagInfo.value;
                            } else {
                                RemoveLastFlag(FetchRequest);
                            }

                            return(allFlags);
                        }

                        return(F_PARSE_ERROR);
                    }
                
                    allFlags |= flagInfo.value;

                    ccode = flagInfo.parser(&ptr, currentFlag);
                    FetchRequest->hasAllocated |= currentFlag->hasAllocated;
                    if (ccode == 0) {
                        if (*ptr == '\0') {
                            return(allFlags);
                        }
                        return(F_PARSE_ERROR);
                    }

                    return(ccode);

                }
                return(F_SYSTEM_ERROR);
            }
        }
    }

    return(F_PARSE_ERROR);
}

__inline static BOOL 
GetMessageRangeByUid(ImapClient *client, FetchStruct *FetchRequest)
{
    unsigned char *colonPtr;
    unsigned char *currentRange;
    long retValue;

    currentRange = FetchRequest->nextRange;

    /* Prepare the command list */
    if ((FetchRequest->nextRange = strchr(currentRange, ',')) != NULL) {
        *(FetchRequest->nextRange) = '\0';
        FetchRequest->nextRange++;
    }

    if ((colonPtr = strchr(currentRange, ':')) != NULL) {
        /* We've got a set ! */
        *colonPtr = '\0';         

        if (*(colonPtr + 1) == '*') {
            FetchRequest->rangeEnd = UID_HIGHEST;
        } else {
            FetchRequest->rangeEnd = atol(colonPtr + 1);
        }

        if (UIDtoNUMRange(client, atol(currentRange), FetchRequest->rangeEnd, &FetchRequest->rangeStart, &FetchRequest->rangeEnd)) {
            *colonPtr=':';
            return(TRUE);
        }

        *colonPtr=':';
        return(FALSE);  /* nothing found */
    }

    /* Just a single number */
        
    if (currentRange[0] == '*') {
        /* Special: Last available item */
        FetchRequest->rangeStart = client->Messages - 1;
        FetchRequest->rangeEnd = FetchRequest->rangeStart;
        return(TRUE);
    }

    retValue = UIDtoNUM(client, atol(currentRange));
    if (retValue > -1) {
        FetchRequest->rangeStart = retValue;
        FetchRequest->rangeEnd = FetchRequest->rangeStart;
        return(TRUE);
    }

    LoadUIDList(client);

    retValue = UIDtoNUM(client, atol(currentRange));
    if (retValue > -1) {
        FetchRequest->rangeStart = retValue;
        FetchRequest->rangeEnd = FetchRequest->rangeStart;
        return(TRUE);
    }

    return(FALSE);  /* nothing found */
}

__inline static unsigned long 
GetMessageRange(ImapClient *client, FetchStruct *FetchRequest)
{
    unsigned char *colonPtr;
    unsigned char *currentRange;

    currentRange = FetchRequest->nextRange;

    /* Prepare the command list */
    if ((FetchRequest->nextRange = strchr(currentRange, ',')) != NULL) {
        *(FetchRequest->nextRange) = '\0';
        FetchRequest->nextRange++;
    }

    if ((colonPtr = strchr(currentRange, ':')) != NULL) {
        /* We've got a set ! */
        *colonPtr = '\0';         

        FetchRequest->rangeStart = atol(currentRange) - 1;     
        *colonPtr=':';

        if (client->Messages > 0 && FetchRequest->rangeStart < client->Messages) {
            if (*(colonPtr + 1) == '*') {     
                /* Special: Last available item */
                FetchRequest->rangeEnd = client->Messages - 1;
                return(TRUE);
            }

            /* Make sure we got a message there */
            FetchRequest->rangeEnd = atol(colonPtr + 1) - 1;     
            if (FetchRequest->rangeEnd >= client->Messages) {
                FetchRequest->rangeEnd = client->Messages - 1;
            }

            return(TRUE);
        }

        return(FALSE);  /* nothing found */
    }

    /* Just a single number */
    if (currentRange[0] == '*') {
        /* Special: Last available item */
        FetchRequest->rangeStart = client->Messages - 1;
        FetchRequest->rangeEnd = FetchRequest->rangeStart;
        return(TRUE);
    }

    FetchRequest->rangeStart = atol(currentRange) - 1;
    if (FetchRequest->rangeStart < client->Messages) {
        FetchRequest->rangeEnd = FetchRequest->rangeStart;
        return(TRUE);
    }

    return(FALSE);  /* nothing found */
}

__inline static void
SetMessageSeenFlag(ImapClient *client, FetchStruct *FetchRequest)
{
    long ccode;

    SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "AFLG %lu %lu\r\n", FetchRequest->currentMessage.nmapId, (unsigned long)MSG_STATE_READ));
    ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
    if ((ccode == 1000) && !(FetchRequest->flags & F_FLAGS)) {  
        unsigned long Flags = atol(FetchRequest->responseBuf);
        /* We have to hold onto these flags until we have a chance to tell the client about them. */
        RememberFlags(client->UnseenFlags, FetchRequest->currentMessage.nmapId, Flags);
    }
}

__inline static BOOL
GetMessageDetails(ImapClient *client, FetchStruct *FetchRequest)
{
    long ccode;
    MessageInfo *message;

    /* Grab the details about this message */
    SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "INFO %lu\r\n", FetchRequest->currentMessage.nmapId));
    ccode=GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);
    if (ccode == 2001) {
        message = &(FetchRequest->currentMessage);
        sscanf(FetchRequest->responseBuf, "%lu %lu %lu %lu %lu %*s %*u %*u %lu",
               &message->nmapId, 
               &message->totalSize, 
               &message->headerSize, 
               &message->bodySize, 
               &message->flags,
               &message->internalDate);
        return(TRUE);
    }

    return(FALSE);
}

__inline static long
GetMessageHeader(ImapClient *client, FetchStruct *FetchRequest)
{
    int ccode;

    /* Request the header */
    SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "HEAD %lu\r\n", FetchRequest->currentMessage.nmapId));
    ccode=GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), TRUE);

    if (ccode == 2020) {
        ccode = GetDataFromNmap(client, &(FetchRequest->currentMessage.header), atoi(FetchRequest->responseBuf), FetchRequest->sendBuf, sizeof(FetchRequest->sendBuf));
        if (ccode == 0) {
            /* Skip non-rfc822 From<sp> line */
            FetchRequest->currentMessage.headerSize=strlen(FetchRequest->currentMessage.header);
            MakeRFC822Header(FetchRequest->currentMessage.header, &FetchRequest->currentMessage.headerSize);
            return(0);
        }
    }
    return(ccode);
}

__inline static long
GetMimeInfo(ImapClient *client, FetchStruct *FetchRequest)
{
    long ccode;

    if (FetchRequest->currentMessage.mimeInfo == NULL) {
        FetchRequest->currentMessage.mimeInfo = MDBCreateValueStruct(IMAPDirectoryHandle, NULL);
        if (FetchRequest->currentMessage.mimeInfo) {
            ;
        } else {
            return(-1);
        }
    } else {
        MDBFreeValues(FetchRequest->currentMessage.mimeInfo);
    }

    SendNMAPServer(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "MIMED %lu\r\n", FetchRequest->currentMessage.nmapId));
    do {
        ccode = GetNMAPAnswer(client, FetchRequest->responseBuf, sizeof(FetchRequest->responseBuf), FALSE);
        if ((ccode > 2001) && (ccode < 2005)) {
            MDBAddValue(FetchRequest->responseBuf, FetchRequest->currentMessage.mimeInfo);
            continue;
        }

        if (ccode == 1000) {
            return(0);
        }

        return(ccode);

    } while (TRUE);
}

__inline static long
RespondForMessageRange(ImapClient *client, FetchStruct *FetchRequest)
{
    long ccode;
    unsigned long i;
        
        FetchRequest->currentMessage.imapId = FetchRequest->rangeStart;

        do {
            /* Do the dirty work */
            FetchRequest->currentMessage.nmapId = client->IDList[FetchRequest->currentMessage.imapId];

            /* Set the flags - got to do this here so the flags include the change - see the rfc */
            if (FetchRequest->flags & F_BODY_SEEN) {
                SetMessageSeenFlag(client, FetchRequest);
            }

            GetMessageDetails(client, FetchRequest);

            if (FetchRequest->flags & F_NEED_HEADER) {
                ccode = GetMessageHeader(client, FetchRequest);
                if (ccode == 0) {
                    ;
                } else {
                    FreeFetchMessageInfo(&(FetchRequest->currentMessage));
                    return(ccode);
                }
            }

            if (FetchRequest->flags & F_NEED_MIME) {
                ccode = GetMimeInfo(client, FetchRequest);
                if (ccode == 0) {
                    ;
                } else {
                    FreeFetchMessageInfo(&(FetchRequest->currentMessage));
                    return(ccode);
                }
            }

            /* Now send the response */
            SendClient(client, FetchRequest->sendBuf, sprintf(FetchRequest->sendBuf, "* %lu FETCH (", FetchRequest->currentMessage.imapId + 1));
            if (FetchRequest->flagCount > 0) {
                i = 0;

                do {
                    ccode = FetchRequest->flag[i].responder(client, FetchRequest, &(FetchRequest->flag[i]));
                    if (ccode == 0) {
                        i++;
                        if (i < FetchRequest->flagCount) {
                            SendClient(client, " ", 1);
                            continue;
                        }
                        break;
                    }
                    FreeFetchMessageInfo(&(FetchRequest->currentMessage));
                    return(ccode);
                } while(TRUE);
            }

            SendClient(client, ")\r\n", 3);
            FetchRequest->currentMessage.imapId++;
            if (FetchRequest->currentMessage.imapId > FetchRequest->rangeEnd) {
                FreeFetchMessageInfo(&(FetchRequest->currentMessage));
                break;
            }
        
        } while (TRUE);

        return(0);
}

int
ImapCommandFetch(void *param)
{
    ImapClient *client = (ImapClient *)param;
    FetchStruct FetchRequest;
    int ccode;

    if (client->State == STATE_SELECTED) {
        if (SendAsyncNewMessages(client) != CLIENT_TERMINATE) {

            memset(&FetchRequest, 0, sizeof(FetchRequest));
      
            FetchRequest.flags = ParseFetchArguments(client, client->Command + 6, &FetchRequest); 
            if (!(FetchRequest.flags & F_ERROR)) {
                FetchRequest.nextRange = FetchRequest.messageSet;

                do {
                    if (GetMessageRange(client, &FetchRequest)) {
                        ccode = RespondForMessageRange(client, &FetchRequest);
                        if (ccode == 0) {
                            continue;
                        }
                        
                        if (ccode > 0) {
                            SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s NO FETCH NMAP error:%d\r\n", client->Ident, ccode));
                        } else {
                            SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s NO FETCH Out of memory\r\n", client->Ident));
                        }
                        
                        FreeFetchResourcesFailure(&FetchRequest);
                        return(CLIENT_TERMINATE);
                    }

                    break;
                } while (FetchRequest.nextRange);

                SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s OK FETCH completed\r\n", client->Ident));
                FreeFetchResourcesSuccess(&FetchRequest);
                return(CLIENT_CONTINUE);
            }

            if (!(FetchRequest.flags & F_SYSTEM_ERROR)) {
                SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s BAD FETCH Invalid argument\r\n", client->Ident));
                FreeFetchResourcesFailure(&FetchRequest);
                return(CLIENT_CONTINUE);
            }

            SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s NO FETCH Out of memory\r\n", client->Ident));
            FreeFetchResourcesFailure(&FetchRequest);
        }

        return(CLIENT_TERMINATE);
    }

    SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s BAD Wrong state\r\n", client->Ident));
    return(CLIENT_CONTINUE);
}

int
ImapCommandUidFetch(void *param)
{
    ImapClient *client = (ImapClient *)param;
    FetchStruct FetchRequest;
    int ccode;
    FetchFlagStruct *firstFlag;

    if (client->State == STATE_SELECTED) {
        if (SendAsyncNewMessages(client) != CLIENT_TERMINATE) {
            memmove(client->Command, client->Command + 4, strlen(client->Command + 4) + 1);
            memset(&FetchRequest, 0, sizeof(FetchRequest));
      
            /* Respond with uid even if it has not been explicitly requested */ 
            FetchRequest.flags = F_UID;
            firstFlag = AddNewFlag(&FetchRequest, FetchFlagResponderUid);
            if (firstFlag) {
                FetchRequest.flags = ParseFetchArguments(client, client->Command + 6, &FetchRequest); 
                if (!(FetchRequest.flags & F_ERROR)) {

                    FetchRequest.nextRange = FetchRequest.messageSet;

                    do {
                        if (GetMessageRangeByUid(client, &FetchRequest)) {
                            ccode = RespondForMessageRange(client, &FetchRequest);
                            if (ccode == 0) {
                                continue;
                            }

                            if (ccode > 0) {
                                SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s NO UID FETCH NMAP error:%d\r\n", client->Ident, ccode));
                            } else {
                                SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s NO UID FETCH Out of memory\r\n", client->Ident));
                            }
                        
                            FreeFetchResourcesFailure(&FetchRequest);
                            return(CLIENT_TERMINATE);
                        }

                        break;
                    } while (FetchRequest.nextRange);

                    SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s OK UID FETCH completed\r\n", client->Ident));

                    FreeFetchResourcesSuccess(&FetchRequest);
                    return(CLIENT_CONTINUE);
                }

                if (!(FetchRequest.flags & F_SYSTEM_ERROR)) {
                    SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s BAD UID FETCH Invalid argument\r\n", client->Ident));
                    FreeFetchResourcesFailure(&FetchRequest);
                    return(CLIENT_CONTINUE);
                }

                SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s NO UID FETCH Out of memory\r\n", client->Ident));
                FreeFetchResourcesFailure(&FetchRequest);
                return(CLIENT_TERMINATE);
            }

            SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s NO UID FETCH Out of memory\r\n", client->Ident));
        }

        return(CLIENT_TERMINATE);
    }

    SendClient(client, FetchRequest.sendBuf, snprintf(FetchRequest.sendBuf, sizeof(FetchRequest.sendBuf), "%s BAD Wrong state\r\n",client->Ident));
    return(CLIENT_CONTINUE);
}

void
FetchCleanup(void)
{
    HulaKeywordIndexFree(FetchFlagsAttIndex);
    HulaKeywordIndexFree(FetchFlagsSoloIndex);
    HulaKeywordIndexFree(FetchFlagsSectionIndex);
}

BOOL
FetchInit(void)
{
    FetchFlagsAttIndex = InitializeIndexFetchFlag(FetchFlagsAtt);
    if (FetchFlagsAttIndex != NULL) {
        FetchFlagsSoloIndex = InitializeIndexFetchFlag(FetchFlagsSolo);
        if (FetchFlagsSoloIndex != NULL) {
            FetchFlagsSectionIndex = InitializeIndexFetchFlag(FetchFlagsSection);
            if (FetchFlagsSectionIndex) {
                return(TRUE);
            }
            HulaKeywordIndexFree(FetchFlagsSoloIndex);
        }
        HulaKeywordIndexFree(FetchFlagsAttIndex);
    } 

    return(FALSE);
}
