
/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>

#define N_PLAT_NLM
#define N_IAPX386

/* Product defines */
#define PRODUCT_SHORT_NAME "imapd.nlm"
#define PRODUCT_NAME   "Hula IMAP4 Agent"
#define PRODUCT_DESCRIPTION "Provides IMAP4 access to the Hula mail-store. (IMAP4 = Internet Mail Access Protocol, Rev4, RFC2060)"
#define PRODUCT_VERSION  "$Revision: 1.8 $"

#define MAX_THREAD_LOAD  50
#define MAX_FAILED_LOGINS  3
#undef DEBUG_FETCH
#define NMLOGID_H_NEED_LOGGING_KEY
#define NMLOGID_H_NEED_LOGGING_CERT
#include <xpl.h>
#include <connio.h>

#include <logger.h>

#include <mdb.h>
#include <msgapi.h>
#include <cmlib.h>
#include <nmap.h>

/* Management Client Header Include */
#include <management.h>
#include <hulautil.h>
#include "imapd.h"

/* Globals */
BOOL           Exiting = FALSE;
MDBHandle      IMAPDirectoryHandle = NULL;
XplSemaphore   IMAPServerSemaphore;
XplSemaphore   IMAPShutdownSemaphore;
unsigned long  IMAPMaxThreadLoad = 100000;
XplAtomic      IMAPServerThreads;
XplAtomic      IMAPConnThreads;
XplAtomic      IMAPIdleConnThreads;
BOOL           IMAPACLEnabled = TRUE;
BOOL           IMAPReceiverStopped = FALSE;
int            IMAPServerSocket = -1;
int            IMAPServerSocketSSL  = -1;
unsigned short IMAPServerPort = IMAP4_PORT;
unsigned short IMAPServerPortSSL = IMAP4_PORT_SSL;
unsigned char  Hostname[256] = "";
unsigned long  HostnameLen = 0;
unsigned long  MaximumCommandLen = 4 * 1048576;
int            TGid;
unsigned char  NMAP_ADDR[80] = "127.0.0.1";
int            UTCOffset    = 0;
int            ClientSSLOptions  = 0;
BOOL           UseNMAPSSL    = FALSE;
unsigned char  Postmaster[MAXEMAILNAMESIZE + 1] = "admin";
unsigned char  ManagementURL[256] = "";
SSL_CTX        *SSLContext    = NULL;
void           *LogHandle    = NULL;
static unsigned char NMAPHash[NMAP_HASH_SIZE];
const unsigned char *WSpecials = " ()<>@,;:\\\"[]\1\2\3\4\5\6\7\10\11\12\13\14\15\16\17\20\21\22\23\24\25\26\27\30\31\32\33\34\35\36\37\177";

IOFuncs FuncTbl = 
{ XplIPWrite,XplIPWriteSSL,XplIPRead,XplIPReadSSL };

ImapGlobal Imap;

static ProtocolCommand ImapProtocolCommands[] = {
    { IMAP_COMMAND_CAPABILITY, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_CAPABILITY) - 1, ImapCommandCapability, NULL, NULL },
    { IMAP_COMMAND_NOOP, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_NOOP) - 1, ImapCommandNoop, NULL, NULL },
    { IMAP_COMMAND_LOGOUT, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_LOGOUT) - 1, ImapCommandLogout, NULL, NULL },
    { IMAP_COMMAND_STARTTLS, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_STARTTLS) - 1, ImapCommandStartTls, NULL, NULL },
    { IMAP_COMMAND_AUTHENTICATE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_AUTHENTICATE) - 1, ImapCommandAuthenticate, NULL, NULL },
    { IMAP_COMMAND_LOGIN, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_LOGIN) - 1, ImapCommandLogin, NULL, NULL },
    { IMAP_COMMAND_SELECT, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_SELECT) - 1, ImapCommandSelect, NULL, NULL },
    { IMAP_COMMAND_EXAMINE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_EXAMINE) - 1, ImapCommandExamine, NULL, NULL },
    { IMAP_COMMAND_CREATE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_CREATE) - 1, ImapCommandCreate, NULL, NULL },
    { IMAP_COMMAND_DELETE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_DELETE) - 1, ImapCommandDelete, NULL, NULL },
    { IMAP_COMMAND_RENAME, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_RENAME) - 1, ImapCommandRename, NULL, NULL },
    { IMAP_COMMAND_SUBSCRIBE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_SUBSCRIBE) - 1, ImapCommandSubscribe, NULL, NULL },
    { IMAP_COMMAND_UNSUBSCRIBE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_UNSUBSCRIBE) - 1, ImapCommandUnsubscribe, NULL, NULL },
    { IMAP_COMMAND_LIST, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_LIST) - 1, ImapCommandList, NULL, NULL },
    { IMAP_COMMAND_LSUB, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_LSUB) - 1, ImapCommandLsub, NULL, NULL },
    { IMAP_COMMAND_STATUS, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_STATUS) - 1, ImapCommandStatus, NULL, NULL },
    { IMAP_COMMAND_APPEND, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_APPEND) - 1, ImapCommandAppend, NULL, NULL },
    { IMAP_COMMAND_CHECK, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_CHECK) - 1, ImapCommandCheck, NULL, NULL },
    { IMAP_COMMAND_CLOSE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_CLOSE) - 1, ImapCommandClose, NULL, NULL },
    { IMAP_COMMAND_EXPUNGE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_EXPUNGE) - 1, ImapCommandExpunge, NULL, NULL },
    { IMAP_COMMAND_SEARCH, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_SEARCH) - 1, ImapCommandSearch, NULL, NULL },
    { IMAP_COMMAND_UID_SEARCH, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_UID_SEARCH) - 1, ImapCommandSearch, NULL, NULL },
    { IMAP_COMMAND_FETCH, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_FETCH) - 1, ImapCommandFetch, NULL, NULL },
    { IMAP_COMMAND_UID_FETCH, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_UID_FETCH) - 1, ImapCommandUidFetch, NULL, NULL },
    { IMAP_COMMAND_STORE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_STORE) - 1, ImapCommandStore, NULL, NULL },
    { IMAP_COMMAND_UID_STORE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_UID_STORE) - 1, ImapCommandUidStore, NULL, NULL },
    { IMAP_COMMAND_COPY, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_COPY) - 1, ImapCommandCopy, NULL, NULL },
    { IMAP_COMMAND_UID_COPY, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_UID_COPY) - 1, ImapCommandUidCopy, NULL, NULL },
    { IMAP_COMMAND_SETACL, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_SETACL) - 1, ImapCommandSetAcl, NULL, NULL },
    { IMAP_COMMAND_DELETEACL, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_DELETEACL) - 1, ImapCommandDeleteAcl, NULL, NULL },
    { IMAP_COMMAND_GETACL, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_GETACL) - 1, ImapCommandGetAcl, NULL, NULL },
    { IMAP_COMMAND_LISTRIGHTS, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_LISTRIGHTS) - 1, ImapCommandListRights, NULL, NULL },
    { IMAP_COMMAND_MYRIGHTS, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_MYRIGHTS) - 1, ImapCommandMyRights, NULL, NULL },
    { IMAP_COMMAND_SETQUOTA, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_SETQUOTA) - 1, ImapCommandSetQuota, NULL, NULL },
    { IMAP_COMMAND_GETQUOTA, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_GETQUOTA) - 1, ImapCommandGetQuota, NULL, NULL },
    { IMAP_COMMAND_GETQUOTAROOT, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_GETQUOTAROOT) - 1, ImapCommandGetQuotaRoot, NULL, NULL },
    { IMAP_COMMAND_NAMESPACE, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_NAMESPACE) - 1, ImapCommandNameSpace, NULL, NULL },
    { IMAP_COMMAND_PROXYAUTH, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_PROXYAUTH) - 1, ImapCommandProxyAuth, NULL, NULL },
    { IMAP_COMMAND_QUIT, IMAP_HELP_NOT_DEFINED, sizeof(IMAP_COMMAND_QUIT) - 1, ImapCommandQuit, NULL, NULL },
    
    { NULL, NULL, 0, NULL, NULL, NULL }
};


struct {
    int    len;
    unsigned char message[128];

    struct {
        int    len;
        unsigned char message[128];
    } SSL;
} Capability;

unsigned char *MonthNames[] = /* all months */
{
    "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

struct {
    struct {
        XplAtomic Serviced;
    } Clients;

    XplAtomic WrongPassword;
} IMAPStats;




int ClientRead(ImapClient *client, unsigned char *Buf, int Len, int Flags);
int ClientReadSSL(ImapClient *client, unsigned char *Buf, int Len, int Flags);
int ClientWrite(ImapClient *client, unsigned char *Buf, int Len, int Flags);
int ClientWriteSSL(ImapClient *client, unsigned char *Buf, int Len, int Flags);
int NMAPRead(ImapClient *client, unsigned char *Buf, int Len, int Flags);
int NMAPReadSSL(ImapClient *client, unsigned char *Buf, int Len, int Flags);
int NMAPWrite(ImapClient *client, unsigned char *Buf, int Len, int Flags);
int NMAPWriteSSL(ImapClient *client, unsigned char *Buf, int Len, int Flags);
void IMAPServer(void *unused);
void IMAPSSLServer(void *unused);





#undef DEBUG

#if !defined(DEBUG)
#define GetIMAPConnection()   MemPrivatePoolGetEntry(IMAPConnectionPool)
#else
#define GetIMAPConnection()   MemPrivatePoolGetEntryDebug(IMAPConnectionPool, __FILE__, __LINE__)
#endif

void    *IMAPConnectionPool  = NULL;

/* Management Client Declarations */
BOOL ReadIMAPVariable(unsigned int Variable, unsigned char *Data, size_t *DataLength);
BOOL WriteIMAPVariable(unsigned int Variable, unsigned char *Data, size_t DataLength);

ManagementVariables IMAPManagementVariables[] = {
    /* 0 #define    PRODUCT_VERSION       */ { DMCMV_REVISIONS,     DMCMV_REVISIONS_HELP,    ReadIMAPVariable, NULL }, 
    /* 1 unsigned long  IMAPMaxThreadLoad       */ { DMCMV_MAX_CONNECTION_COUNT, DMCMV_MAX_CONNECTION_COUNT_HELP, ReadIMAPVariable, WriteIMAPVariable }, 
    /* 2 XplAtomic   IMAPServerThreads       */ { DMCMV_SERVER_THREAD_COUNT,  DMCMV_SERVER_THREAD_COUNT_HELP, ReadIMAPVariable, NULL }, 
    /* 3 XplAtomic   IMAPConnThreads       */ { DMCMV_CONNECTION_COUNT,   DMCMV_CONNECTION_COUNT_HELP,  ReadIMAPVariable, NULL }, 
    /* 4 XplAtomic   IMAPIdleConnThreads       */ { DMCMV_IDLE_CONNECTION_COUNT,   DMCMV_IDLE_CONNECTION_COUNT_HELP,  ReadIMAPVariable, NULL }, 
    /* 5 BOOL     IMAPACLEnabled        */ { IMAPMV_ACL_ENABLED,    IMAPMV_ACL_ENABLED_HELP,   ReadIMAPVariable, NULL }, 
    /* 6 BOOL     IMAPReceiverStopped      */ { DMCMV_RECEIVER_DISABLED,  DMCMV_RECEIVER_DISABLED_HELP,  ReadIMAPVariable, WriteIMAPVariable }, 
    /* 7 unsigned char  Hostname[256]        */ { DMCMV_HOSTNAME,     DMCMV_HOSTNAME_HELP,     ReadIMAPVariable, NULL }, 
    /* 8 unsigned char  NMAP_ADDR[80]        */ { DMCMV_NMAP_ADDRESS,    DMCMV_NMAP_ADDRESS_HELP,   ReadIMAPVariable, NULL }, 
    /* 9 int     UTCOffset         */ { DMCMV_UTC_OFFSET,     DMCMV_UTC_OFFSET_HELP,    ReadIMAPVariable, NULL }, 
    /* 10 unsigned char  Postmaster[MAXEMAILNAMESIZE + 1]  */ { DMCMV_POSTMASTER,     DMCMV_POSTMASTER_HELP,    ReadIMAPVariable, NULL }, 
    /* 11 unsigned char  ManagementURL[256]      */ { IMAPMV_MANAGEMENT_URL,   IMAPMV_MANAGEMENT_URL_HELP,  ReadIMAPVariable, NULL }, 
    /* 12 XplAtomic   IMAPStats.Clients.Serviced    */ { DMCMV_TOTAL_CONNECTIONS,  DMCMV_TOTAL_CONNECTIONS_HELP,  ReadIMAPVariable, WriteIMAPVariable }, 
    /* 13 XplAtomic   IMAPStats.WrongPassword     */ { DMCMV_BAD_PASSWORD_COUNT,  DMCMV_BAD_PASSWORD_COUNT_HELP, ReadIMAPVariable, WriteIMAPVariable }, 
    /* 14 unsigned char              */ { DMCMV_VERSION,      DMCMV_VERSION_HELP,     ReadIMAPVariable, NULL }, 
};



static BOOL 
IMAPShutdown(unsigned char *Arguments, unsigned char **Response, BOOL *CloseConnection)
{
    int   s;
    XplThreadID id;

    if (Response) {
        if (!Arguments) {
            if (IMAPServerSocket != -1) {
                *Response = MemStrdup("Shutting down.\r\n");
                if (*Response) {
                    id = XplSetThreadGroupID(TGid);

                    Exiting = TRUE;

                    s = IMAPServerSocket;
                    IMAPServerSocket = -1;

                    if (s != -1) {
                        IPclose(s);
                    }

                    if (CloseConnection) {
                        *CloseConnection = TRUE;
                    }

                    XplSetThreadGroupID(id);
                }
            } else if (Exiting) {
                *Response = MemStrdup("Shutdown in progress.\r\n");
            } else {
                *Response = MemStrdup("Unknown shutdown state.\r\n");
            }

            if (*Response) {
                return(TRUE);
            }

            return(FALSE);
        }

        *Response = MemStrdup("Arguments not allowed.\r\n");
        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
IMAPDMCCommandHelp(unsigned char *Arguments, unsigned char **Response, BOOL *CloseConnection)
{
    BOOL responded = FALSE;

    if (Response) {
        if (Arguments) {
            switch(toupper(Arguments[0])) {
            case 'M': {
                if (XplStrCaseCmp(Arguments, DMCMC_DUMP_MEMORY_USAGE) == 0) {
                    if ((*Response = MemStrdup(DMCMC_DUMP_MEMORY_USAGE_HELP)) != NULL) {
                        responded = TRUE;
                    }

                    break;
                }
            }

            case 'S': {
                if (XplStrCaseCmp(Arguments, DMCMC_SHUTDOWN) == 0) {
                    if ((*Response = MemStrdup(DMCMC_SHUTDOWN_HELP)) != NULL) {
                        responded = TRUE;
                    }

                    break;
                } else if (XplStrCaseCmp(Arguments, DMCMC_STATS) == 0) {
                    if ((*Response = MemStrdup(DMCMC_STATS_HELP)) != NULL) {
                        responded = TRUE;
                    }

                    break;
                }
            }

            default: {
                break;
            }
            }
        } else if ((*Response = MemStrdup(DMCMC_HELP_HELP)) != NULL) {
            responded = TRUE;
        }

        if (responded || ((*Response = MemStrdup(DMCMC_UNKOWN_COMMAND)) != NULL)) {
            return(TRUE);
        }
    }

    return(FALSE);
}

static BOOL 
SendIMAPStatistics(unsigned char *Arguments, unsigned char **Response, BOOL *CloseConnection)
{
    MemStatistics poolStats;

    if (!Arguments && Response) {
        memset(&poolStats, 0, sizeof(MemStatistics));

        *Response = MemMalloc(sizeof(PRODUCT_NAME)                             /* Long Name */
                              + sizeof(PRODUCT_SHORT_NAME) /* Short Name */
                              + 10    /* PRODUCT_MAJOR_VERSION */
                              + 10    /* PRODUCT_MINOR_VERSION */
                              + 10    /* PRODUCT_LETTER_VERSION */
                              + 10    /* Connection Pool Alloc Count */
                              + 10    /* Connection Pool Memory Usage */
                              + 10    /* Connection Pool Pitches */
                              + 10    /* Connection Pool Strikes */
                              + 10    /* DMCMV_SERVER_THREAD_COUNT */
                              + 10     /* DMCMV_CONNECTION_COUNT */
                              + 10     /* DMCMV_IDLE_CONNECTION_COUNT */
                              + 10    /* DMCMV_MAX_CONNECTION_COUNT */
                              + 10    /* DMCMV_TOTAL_CONNECTIONS */
                              + 10    /* DMCMV_BAD_PASSWORD_COUNT */
                              + 28);    /* Formatting */

        MemPrivatePoolStatistics(IMAPConnectionPool, &poolStats);

        if (*Response) {
            sprintf(*Response, "%s (%s: v%d.%d.%d)\r\n%lu:%lu:%lu:%lu:%d:%d:%d:%lu:%d:%d\r\n", 
                    PRODUCT_NAME, 
                    PRODUCT_SHORT_NAME, 
                    PRODUCT_MAJOR_VERSION, 
                    PRODUCT_MINOR_VERSION, 
                    PRODUCT_LETTER_VERSION, 
                    poolStats.totalAlloc.count, 
                    poolStats.totalAlloc.size, 
                    poolStats.pitches, 
                    poolStats.strikes, 
                    XplSafeRead(IMAPServerThreads), 
                    XplSafeRead(IMAPConnThreads), 
                    XplSafeRead(IMAPIdleConnThreads), 
                    IMAPMaxThreadLoad, 
                    XplSafeRead(IMAPStats.Clients.Serviced), 
                    XplSafeRead(IMAPStats.WrongPassword));

            return(TRUE);
        }

        if ((*Response = MemStrdup("Out of memory.\r\n")) != NULL) {
            return(TRUE);
        }
    } else if ((Arguments) && ((*Response = MemStrdup("Arguments not allowed.\r\n")) != NULL)) {
        return(TRUE);
    }

    return(FALSE);
}

ManagementCommands IMAPManagementCommands[] = {
    /* 0 HELP[ <IMAPCommand>]     */ { DMCMC_HELP,        IMAPDMCCommandHelp }, 
    /* 1 SHUTDOWN         */ { DMCMC_SHUTDOWN,      IMAPShutdown },
    /* 2 STATS          */ { DMCMC_STATS,       SendIMAPStatistics }, 
    /* 3 MEMORY         */ { DMCMC_DUMP_MEMORY_USAGE,   ManagementMemoryStats  }, 
};

/* Management Client Read Function */
BOOL ReadIMAPVariable(unsigned int Variable, unsigned char *Data, size_t *DataLength)
{
    unsigned long count;
    unsigned char *ptr;

    switch (Variable) {
    case 0: { /* #define    PRODUCT_VERSION       */
        unsigned char version[30];

        PVCSRevisionToVersion(PRODUCT_VERSION, version);
        count = strlen(version) + 11;

        PVCSRevisionToVersion(IMSearchVersion, version);
        count += strlen(version) + 14;

        if (Data && (*DataLength > count)) {
            ptr = Data;

            PVCSRevisionToVersion(PRODUCT_VERSION, version);
            ptr += sprintf(ptr, "imapd.c: %s\r\n", version);

            PVCSRevisionToVersion(IMSearchVersion, version);
            ptr += sprintf(ptr, "imsearch.c: %s\r\n", version);

            *DataLength = ptr - Data;
        } else {
            *DataLength = count;
        }

        break;
    }

    case 1: { /* unsigned long  IMAPMaxThreadLoad       */
        if (Data && (*DataLength > 12)) {
            sprintf(Data, "%010lu\r\n", IMAPMaxThreadLoad);
        }

        *DataLength = 12;
        break;
    }

    case 2: { /* XplAtomic   IMAPServerThreads       */
        if (Data && (*DataLength > 12)) {
            sprintf(Data, "%010lu\r\n", (long unsigned int)XplSafeRead(IMAPServerThreads));
        }

        *DataLength = 12;
        break;
    }

    case 3: { /* XplAtomic   IMAPConnThreads       */
        if (Data && (*DataLength > 12)) {
            sprintf(Data, "%010lu\r\n", (long unsigned int)XplSafeRead(IMAPConnThreads));
        }

        *DataLength = 12;
        break;
    }

    case 4: { /* XplAtomic   IMAPIdleConnThreads       */
        if (Data && (*DataLength > 12)) {
            sprintf(Data, "%010lu\r\n", (long unsigned int)XplSafeRead(IMAPIdleConnThreads));
        }

        *DataLength = 12;
        break;
    }

    case 5: { /* BOOL     IMAPACLEnabled        */
        if (IMAPACLEnabled == FALSE) {
            ptr = "FALSE\r\n";
            count = 7;
        } else {
            ptr = "TRUE\r\n";
            count = 6;
        }

        if (Data && (*DataLength > count)) {
            strcpy(Data, ptr);
        }

        *DataLength = count;
        break;
    }

    case 6: { /* BOOL     IMAPReceiverStopped      */
        if (IMAPReceiverStopped == FALSE) {
            ptr = "FALSE\r\n";
            count = 7;
        } else {
            ptr = "TRUE\r\n";
            count = 6;
        }

        if (Data && (*DataLength > count)) {
            strcpy(Data, ptr);
        }

        *DataLength = count;
        break;
    }

    case 7: { /* unsigned char  Hostname[256]        */
        count = strlen(Hostname) + 2;
        if (Data && (*DataLength > count)) {
            sprintf(Data, "%s\r\n", Hostname);
        }

        *DataLength = count;
        break;
    }

    case 8: { /* unsigned char  NMAP_ADDR[80]        */
        count = strlen(NMAP_ADDR) + 2;
        if (Data && (*DataLength > count)) {
            sprintf(Data, "%s\r\n", NMAP_ADDR);
        }

        *DataLength = count;
        break;
    }

    case 9: { /*  int  UTCOffset                                        */
        if (Data && (*DataLength > 13)) {
            sprintf(Data, "%011li\r\n", (long int)UTCOffset);
        }
            
        *DataLength = 13;
        break;
    }

    case 10: { /* unsigned char  Postmaster[MAXEMAILNAMESIZE + 1]  */
        count = strlen(Postmaster) + 2;
        if (Data && (*DataLength > count)) {
            sprintf(Data, "%s\r\n", Postmaster);
        }

        *DataLength = count;
        break;
    }

    case 11: { /* unsigned char  ManagementURL[256]      */
        count = strlen(ManagementURL) + 2;
        if (Data && (*DataLength > count)) {
            sprintf(Data, "%s\r\n", ManagementURL);
        }

        *DataLength = count;
        break;
    }

    case 12: { /* XplAtomic   IMAPStats.Clients.Serviced    */
        if (Data && (*DataLength > 12)) {
            sprintf(Data, "%010lu\r\n", (long unsigned int)XplSafeRead(IMAPStats.Clients.Serviced));
        }

        *DataLength = 12;
        break;
    }

    case 13: { /* XplAtomic   IMAPStats.WrongPassword     */
        if (Data && (*DataLength > 12)) {
            sprintf(Data, "%010lu\r\n", (long unsigned int)XplSafeRead(IMAPStats.WrongPassword));
        }

        *DataLength = 12;
        break;
    }

    case 14: { /* unsigned char */
        DMC_REPORT_PRODUCT_VERSION(Data, *DataLength);
        break;
    }
    }

    return(TRUE);
}

/* Management Client Write Function */
BOOL WriteIMAPVariable(unsigned int Variable, unsigned char *Data, size_t DataLength)
{
    unsigned char *ptr;
    unsigned char *ptr2;
    BOOL    result = TRUE;

    if (!Data || !DataLength) {
        return(FALSE);
    }

    switch (Variable) {
    case 1: { /* unsigned long  IMAPMaxThreadLoad       */
        ptr = strchr(Data, '\n');
        if (ptr) {
            *ptr = '\0';
        }

        ptr2 = strchr(Data, '\r');
        if (ptr2) {
            ptr2 = '\0';
        }

        IMAPMaxThreadLoad = atol(Data);

        if (ptr) {
            *ptr = '\n';
        }

        if (ptr2) {
            *ptr2 = 'r';
        }

        break;
    }

    case 6: { /* BOOL     IMAPReceiverStopped      */
        if ((toupper(Data[0]) == 'T') || (atol(Data) != 0)) {
            IMAPReceiverStopped = TRUE;
        } else if ((toupper(Data[0] == 'F')) || (atol(Data) == 0)) {
            IMAPReceiverStopped = FALSE;
        } else {
            result = FALSE;
        }

        break;
    }

    case 12: { /* XplAtomic   IMAPStats.Clients.Serviced    */
        ptr = strchr(Data, '\n');
        if (ptr) {
            *ptr = '\0';
        }

        ptr2 = strchr(Data, '\r');
        if (ptr2) {
            ptr2 = '\0';
        }

        XplSafeWrite(IMAPStats.Clients.Serviced, atol(Data));

        if (ptr) {
            *ptr = '\n';
        }

        if (ptr2) {
            *ptr2 = 'r';
        }

        break;
    }

    case 13: { /* XplAtomic   IMAPStats.WrongPassword     */
        ptr = strchr(Data, '\n');
        if (ptr) {
            *ptr = '\0';
        }

        ptr2 = strchr(Data, '\r');
        if (ptr2) {
            ptr2 = '\0';
        }

        XplSafeWrite(IMAPStats.WrongPassword, atol(Data));

        if (ptr) {
            *ptr = '\n';
        }

        if (ptr2) {
            *ptr2 = 'r';
        }

        break;
    }

    case 0: /* #define    PRODUCT_VERSION       */
    case 2: /* XplAtomic   IMAPServerThreads       */
    case 3: /* XplAtomic   IMAPConnThreads       */
    case 4: /* XplAtomic   IMAPIdleConnThreads       */
    case 5: /* BOOL     IMAPACLEnabled        */
    case 7: /* unsigned char  Hostname[256]        */
    case 8: /* unsigned char  NMAP_ADDR[80]        */
    case 9: /* int     UTCOffset         */
    case 10: /* unsigned char  Postmaster[MAXEMAILNAMESIZE + 1]  */
    case 11: /* unsigned char  ManagementURL[256]      */
    default: {
        result = FALSE;
        break;
    }
    }

    return(result);
}

static BOOL 
IMAPConnectionAllocCB(void *Buffer, void *ClientData)
{
    register ImapClient *c = (ImapClient *)Buffer;

    memset(c, 0, sizeof(ImapClient));

#if !defined(DEBUG)
    c->Command = (unsigned char *)MemMalloc(BUFSIZE + 1);
#else
    c->Command = (unsigned char *)MemDebugMalloc(BUFSIZE + 1, __FILE__, __LINE__);
#endif

    if (c->Command != NULL) {
        c->Command[0] = '\0';

        c->State = STATE_FRESH;
        c->s = -1;
        c->NMAPs = -1;
        c->CommandLen = BUFSIZE;

        c->ClientWrite = FuncTbl.write;
        c->ClientRead = FuncTbl.read;
        c->ClientSSL = FALSE;

        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
IMAPConnectionFreeCB(void *Buffer, void *ClientData)
{
    register ImapClient *c = (ImapClient *)Buffer;

    MemFree(c->Command);

    return(TRUE);
}

static void 
ReturnIMAPConnection(ImapClient *client)
{
    register ImapClient *c = client;
    register unsigned char  *command;

    if (c->CommandLen == BUFSIZE) {
        command = c->Command;
    } else {
        if (c->Command != NULL) {
            MemFree(c->Command);
        }

        command = (unsigned char *)MemMalloc(BUFSIZE + 1);
    }

    if (c->Command != NULL) {
        memset(c, 0, sizeof(ImapClient));

        c->State = STATE_FRESH;
        c->s = -1;
        c->NMAPs = -1;
        c->CommandLen = BUFSIZE;
        c->Command = command;

        c->ClientWrite = FuncTbl.write;
        c->ClientRead = FuncTbl.read;
        c->ClientSSL = FALSE;

        c->Command[0] = '\0';

        MemPrivatePoolReturnEntry(c);
    } else {
        MemPrivatePoolDiscardEntry(c);
    }

    return;
}

BOOL
LoadUIDList(ImapClient *client)
{
    unsigned long NMAPCount;
    unsigned long State;
    unsigned long UID;
    unsigned long ID;
    int    ReplyInt;
    unsigned char Answer[1024];
    unsigned char Reply[1024];

    SendNMAPServer(client, "INFO\r\n", 6);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt!=2002) {
        ReplyInt=snprintf(Answer, sizeof(Answer), "%s NO NMAP Failure:%d\r\n",client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        FlushClient(client);
        return(FALSE);
    }

    NMAPCount=atol(Reply);

    if (client->UIDList!=NULL) {
        MemFree(client->UIDList);
    }
    if (client->IDList!=NULL) {
        MemFree(client->IDList);
    }


    client->Messages=0;
    client->UIDList=MemMalloc(sizeof(unsigned long)*(NMAPCount+1));
    client->IDList=MemMalloc(sizeof(unsigned long)*(NMAPCount+1));
// memset(client->UIDList, 0, sizeof(unsigned long)*(NMAPCount+1));

    for (ID=1; ID<=NMAPCount; ID++) {
        GetNMAPAnswer(client, Reply, sizeof(Reply), FALSE);
        sscanf(Reply, "%*u %*u %*u %*u %lu %*s %lu", &State, &UID);
        if (!(State & MSG_STATE_PURGED)) {
            client->IDList[client->Messages]=ID;
            client->UIDList[client->Messages]=UID;
//XplConsolePrintf("IMAP %d -> NMAP %d -> UID: %d\n", client->Messages, client->IDList[client->Messages], client->UIDList[client->Messages]);
            client->Messages++;
        }
    }

    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt!=1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP Failure:%d\r\n",client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        FlushClient(client);
        return(FALSE);
    }
    return(TRUE);
}

static long
NMAPtoNUM(ImapClient *client, unsigned long NMAP)
{
    register unsigned long Start;
    register unsigned long End;
    register unsigned long Middle;

    if (client->Messages==0)
        return(-1);

    Start=0;
    End=client->Messages;
    Middle=End/2;

    /* Won't happen often enough, and we handle it after the search, too */
    /* if ((client->UIDList[End-1]<UID) || (client->UIDList[0]>UID)) { */
    /*  return(-1); */
    /* } */

    do {
        if (client->IDList[Middle]>NMAP) {
            End=Middle;
        } else {
            Start=Middle;
        }
        Middle=Start+(End-Start)/2;
    } while ((End-Start)>1);

    if (NMAP==client->IDList[Start]) {
        return(Start+1);
    }

    return(-1);
}

long
UIDtoNUM(ImapClient *client, unsigned long UID)
{
    register unsigned long Start;
    register unsigned long End;
    register unsigned long Middle;

    if (client->Messages==0)
        return(-1);

    Start=0;
    End=client->Messages;
    Middle=End/2;

    /* Won't happen often enough, and we handle it after the search, too */
    /* if ((client->UIDList[End-1]<UID) || (client->UIDList[0]>UID)) { */
    /*  return(-1); */
    /* } */

    do {
        if (client->UIDList[Middle]>UID) {
            End=Middle;
        } else {
            Start=Middle;
        }
        Middle=Start+(End-Start)/2;
    } while ((End-Start)>1);

    if (UID==client->UIDList[Start]) {
        return(Start);
    }

    return(-1);
}

BOOL
UIDtoNUMRange(ImapClient *client, unsigned long StartUID, unsigned long EndUID, long *StartIndex, long *EndIndex)
{
    register unsigned long Start;
    register unsigned long End;
    register unsigned long Middle;

    if (client->Messages==0)
        return(FALSE);

    if (StartUID>EndUID) {
        Start=EndUID;
        EndUID=StartUID;
        StartUID=Start;  
    }

    Start=0;
    End=client->Messages;
    Middle=End/2;

    do {
        if (client->UIDList[Middle]>StartUID) {
            End=Middle;
        } else {
            Start=Middle;
        }
        Middle=Start+(End-Start)/2;
    } while ((End-Start)>1);

    if (client->UIDList[Start]>=StartUID) {
        *StartIndex=Start;
    } else {
        if (Start==(client->Messages-1)) {
            if (EndUID==UID_HIGHEST) {
                *StartIndex=client->Messages-1;
                *EndIndex=*StartIndex;
                /*printf("returning special splat Start:%d End:%d\n", *StartIndex, *EndIndex); */
                return(TRUE);
            }
            return(FALSE);
        } else {
            *StartIndex=++Start;
        }
    }

    if (EndUID==UID_HIGHEST) {
        *EndIndex=client->Messages-1;
        return(TRUE);
    }

    /* Now check end of range */
    End=client->Messages;
    Middle=((End-Start)/2)+Start;

    do {
        if (client->UIDList[Middle]>EndUID) {
            End=Middle;
        } else {
            Start=Middle;
        }
        Middle=Start+(End-Start)/2;
    } while ((End-Start)>1);

    if (client->UIDList[Start]<=EndUID) {
        *EndIndex=Start;
    } else {
        if (Start==0) {
            return(FALSE);
        } else {
            *EndIndex=Start-1;
        }
    }

    return(TRUE);
}

BOOL
EndClientConnection(ImapClient *client)
{
    if (client) {
        if (client->State == STATE_EXITING)
            return(FALSE);
        client->State=STATE_EXITING;

        FreeUnseenFlags(client);

        if (client->NMAPs!=-1) {
            SendNMAPServer(client, "QUIT\r\n", 6);
            IPclose(client->NMAPs);
            client->NMAPs=-1;
        }
        if (client->s!=-1) {
            FlushClient(client);
            IPclose(client->s);
        }

        if (client->CSSL) {
            SSL_shutdown(client->CSSL);
            SSL_free(client->CSSL);
            client->CSSL = NULL;
        }

        if (client->NSSL) {
            SSL_shutdown(client->NSSL);
            SSL_free(client->NSSL);
            client->NSSL = NULL;
        }

        if (client->UIDList) {
            MemFree(client->UIDList);
        }

        if (client->IDList) {
            MemFree(client->IDList);
        }

        if (client->MultiPartTypeList) {
            MemFree(client->MultiPartTypeList);
            client->MultiPartTypeList=NULL;
        }
        if (client->RFCSizeList) {
            MemFree(client->RFCSizeList);
            client->RFCSizeList=NULL;
        }
        if (client->MIMEResponse) {
            MemFree(client->MIMEResponse);
            client->MIMEResponse=NULL;
        }
        if (client->Header) {
            MemFree(client->Header);
            client->Header=NULL;
        }
        if (client->RFC822AddressList) {
            RFC822FreeAddressList(client->RFC822AddressList);
            client->RFC822AddressList=NULL;
        }
        if (client->MailSet) {
            MemFree(client->MailSet);
            client->MailSet=NULL;
        }

        ReturnIMAPConnection(client);
    }
 
    XplSafeDecrement(IMAPConnThreads);
    XplSafeIncrement(IMAPStats.Clients.Serviced);

    XplExitThread(EXIT_THREAD, 0);
    return(TRUE);
}

BOOL 
FlushClient(ImapClient *client)
{
    int count;
    int sent=0;
    while (sent<client->SBufferPtr) {
        count=DoClientWrite(client, client->SBuffer+sent, client->SBufferPtr-sent, 0);
        if ((count<1) || (Exiting)) {
            return(EndClientConnection(client));
        }
        sent+=count;
    }
    client->SBufferPtr=0;
    client->SBuffer[0]='\0';
    return(TRUE);
}

BOOL
SendClient(ImapClient *client, unsigned char *Data, int Len)
{
    int sent=0;

    if (!Len)
        return(TRUE);

    if(!Data)
        return(FALSE);

    while((sent<Len) && !Exiting) {
        if (Len-sent+client->SBufferPtr>=MTU) {
            memcpy(client->SBuffer+client->SBufferPtr, Data+sent, (MTU-client->SBufferPtr));
            sent+=(MTU-client->SBufferPtr);
            client->SBufferPtr+=(MTU-client->SBufferPtr);
            if (!FlushClient(client))
                return(FALSE);
        } else {
            memcpy(client->SBuffer+client->SBufferPtr, Data+sent, Len-sent);
            client->SBufferPtr+=Len-sent;
            sent=Len;
        }
    }
    client->SBuffer[client->SBufferPtr]='\0';
    return(TRUE);
}

BOOL
SendNMAPServer(ImapClient *client, unsigned char *Data, int Len)
{
    int count;
    int sent=0;

    while (sent<Len) {
        count=DoNMAPWrite(client, Data+sent, Len-sent, 0);
        if ((count<1) || (Exiting)) {
            if (!Exiting) {
                IPclose(client->NMAPs);
                client->NMAPs=-1;
            }
            if (!(client->State == STATE_EXITING))
                return(EndClientConnection(client));
            else
                return(FALSE);
        }
        sent+=count;
    }

    return(TRUE);
}

int
GetNMAPAnswer(ImapClient *client, unsigned char *Reply, int ReplyLen, BOOL CheckForResult)
{
    int    count;
    int    Result;
    unsigned char *ptr;

    if (client->NMAPs == -1)
        return(-1);

 GetNextLine:
    if ((client->NBufferPtr > 0) && ((ptr = strchr(client->NBuffer, 0x0a)) != NULL)) {
        *ptr = '\0';
        if ((unsigned long)ReplyLen < strlen(client->NBuffer)) {
            strcpy(Reply, client->NBuffer);
        } else {
            strncpy(Reply, client->NBuffer, ReplyLen - 1);
            Reply[ReplyLen - 1] = '\0';
        }
        client->NBufferPtr = strlen(ptr + 1);
        memmove(client->NBuffer, ptr + 1, client->NBufferPtr + 1);
        if ((ptr = strrchr(Reply, 0x0d)) != NULL) {
            *ptr='\0';
        }
    } else {
        do {
            if ((!Exiting) && ((count = DoNMAPRead(client, client->NBuffer + client->NBufferPtr, BUFSIZE - client->NBufferPtr, 0)) > 0)) {
                client->NBufferPtr += count;
                client->NBuffer[client->NBufferPtr] = '\0';
                if ((ptr = strchr(client->NBuffer,0x0a)) != NULL) {
                    *ptr = '\0';
                    count = min((ptr - client->NBuffer) + 1, ReplyLen);
                    memcpy(Reply, client->NBuffer, count);

                    client->NBufferPtr = strlen(ptr + 1);
                    memmove(client->NBuffer, ptr + 1, client->NBufferPtr + 1);
                    if ((ptr = strrchr(Reply, 0x0d)) != NULL) {
                        *ptr = '\0';
                    }
                    break;
                } 
            } else {
                if (!Exiting) {
                    if (count < 0) {
                        XplConsolePrintf("IMAPD: Error %d getting answer from NMAP for user %s\n", count, client->User);
                    } else {
                        XplConsolePrintf("IMAPD: Buffer had %d bytes available to get answer from NMAP for user %s\n", BUFSIZE - client->NBufferPtr, client->User);
                    }
                }
                return(EndClientConnection(client));
            }    
        } while (TRUE);
    }

    if (CheckForResult) {
        if (Reply[0] != '6' || Reply[1] != '0' || Reply[2] != '0' || Reply[3] != '0') {
            if ((Reply[4] == '-') || (Reply[4] == ' ')) {
                /* Expected Case */

                Result = atoi(Reply);
                memmove(Reply, Reply+5, strlen(Reply+5)+1);
                return(Result);
            } else {
                /* NMAP is not supposed to do this */
                return(-1);
            }
        } else {
            /* 6000 NMAP Notification of an asyncronous event */
            unsigned long Event;
            Event = atol(Reply + 5);

            if (Event & EVENT_MBOX_FLAGS_CHANGED) {
                unsigned char *ptr;
                unsigned long MsgNum;
                unsigned long MsgFlags;

                Event &= ~EVENT_MBOX_FLAGS_CHANGED;

                /* 6000 Event MsgNum MsgFlags HumanReadableText */
                ptr = strchr(Reply + 5, ' ');
                if (ptr) {
                    ptr++;
                    MsgNum = atol(ptr);
                    ptr = strchr(ptr, ' ');
                    if (ptr) {
                        ptr++;
                        MsgFlags = atol(ptr);
                        /* We need to hold onto these flags until we have a chance to tell the client about them. */

                        RememberFlags(client->UnseenFlags, MsgNum, MsgFlags);
                    }
                }
            }
   
            client->NMAPAsyncEvent |= Event;
            goto GetNextLine;
        }

    } else {
        Result = atoi(Reply);
        return(Result);
    }
}

static BOOL
ReadCommandLine(ImapClient *client)
{
    unsigned char *ptr;
    int    count;
    BOOL    Ready;
    BOOL    tooLong;

    tooLong = FALSE;
    Ready=FALSE;
    client->Command[0] = '\0';
    if ((client->BufferPtr>0) && ((ptr=strchr(client->Buffer, 0x0a))!=NULL)) {
        *ptr='\0';
        memcpy(client->Command, client->Buffer, ptr-client->Buffer+1);
        client->BufferPtr=strlen(ptr+1);
        memmove(client->Buffer, ptr+1, client->BufferPtr+1);
        if ((ptr=strrchr(client->Command, 0x0d))!=NULL)
            *ptr='\0';
        Ready=TRUE;
    } else {
        while (!Ready) {
            if (!Exiting) {
                count = DoClientRead(client, client->Buffer+client->BufferPtr, BUFSIZE-client->BufferPtr, 0);

                if (!Exiting) {
                    if (count > 0) {
                        client->BufferPtr += count;
                        client->Buffer[client->BufferPtr] = '\0';
                        if ((ptr = strchr(client->Buffer, 0x0a)) != NULL) {
                            *ptr = '\0';
                            strncat(client->Command, client->Buffer, ptr-client->Buffer+1);
                            client->BufferPtr=strlen(ptr+1);
                            memmove(client->Buffer, ptr+1, client->BufferPtr+1);
                            if ((ptr=strrchr(client->Command, 0x0d))!=NULL) 
                                *ptr='\0';
                            Ready=TRUE;
                        }
                        continue;
                    }

                    if (BUFSIZE == client->BufferPtr) {
                        /* the buffer is full */
                        if ((client->CommandLen + BUFSIZE) < MaximumCommandLen) {
                            ptr = MemRealloc(client->Command, client->CommandLen + (BUFSIZE * 2));
                            if (ptr) {
                                client->Command = ptr;

                                client->CommandLen += BUFSIZE * 2;
                                strncat(client->Command, client->Buffer, BUFSIZE);
                                client->BufferPtr = 0;
                                continue;
                            } 

                            return(EndClientConnection(client));
                        }

                        /* The command length is larger than the supported limit; dump the buffer */
                        client->BufferPtr = 0;
                        tooLong = TRUE;
                        continue;
                    }
                }
            }
   
            return(EndClientConnection(client));
        }
    }

    if (!tooLong) {
        return(TRUE);
    }

    return(FALSE);
}


static unsigned char
*GrabOctet(ImapClient *client, unsigned char *Input, unsigned char **Destination)
{
    unsigned char *ptr;
    unsigned char *start;
    unsigned char *end;
    long Size; 
    long count;

    if (Input[0] == '{') {
        start = Input + 1;
        end = strchr(start, '}');

        if (end != NULL) {
            *end = '\0';
            if ((end - start) < 7) {
                Size = atol(start);
                if (Size) {
                    SendClient(client, "+ Ready for more data\r\n", 23);
                    FlushClient(client);
                    *Destination = MemMalloc(Size + 1);
                    if (*Destination) {
                        ptr = *Destination;

                        while (Size > 0) {
                            if (client->BufferPtr > 0) {
                                if (Size < client->BufferPtr) {
                                    memcpy(ptr, client->Buffer, Size);
                                    ptr += Size;
                                    client->BufferPtr -= Size;
                                    memmove(client->Buffer, client->Buffer + Size, client->BufferPtr);
                                    client->Buffer[client->BufferPtr]='\0';
                                    Size=0;
                                } else {
                                    memcpy(ptr, client->Buffer, client->BufferPtr);
                                    ptr+=client->BufferPtr;
                                    Size-=client->BufferPtr;
                                    client->BufferPtr=0;
                                    client->Buffer[0]='\0';
                                }
                            } else {
                                if (Exiting) {
                                    return((unsigned char *)EndClientConnection(client));
                                }
                                count=DoClientRead(client, client->Buffer+client->BufferPtr, BUFSIZE-client->BufferPtr, 0);
                                if ((count<1) || (Exiting)) {
                                    return((unsigned char *)EndClientConnection(client));
                                }
                                if (Size>count) {
                                    memcpy(ptr, client->Buffer, count);
                                    Size-=count;
                                    ptr+=count;
                                } else {
                                    memcpy(ptr, client->Buffer, Size);
                                    memmove(client->Buffer, client->Buffer+Size, count-Size);
                                    ptr+=Size;
                                    client->BufferPtr=count-Size;
                                    client->Buffer[client->BufferPtr]='\0';
                                    Size=0;
                                }
                            }
                        }

                        *ptr = '\0';
                        if (ReadCommandLine(client)) {
                            return(client->Command);
                        }

                        EndClientConnection(client);
                        return(client->Command);
                    }

                    *Destination = NULL;
                    return(NULL);
                }

                *Destination = MemStrdup("");
                if (ReadCommandLine(client)) {
                    return(client->Command);
                }

                EndClientConnection(client);
                return(client->Command);
            }
        }
    }

    *Destination = NULL;
    return(NULL);
}

static unsigned char
*GrabLiteral(ImapClient *client, unsigned char *Input, unsigned char **Destination)
{
    unsigned char *ptr, *ptr2;

    if (*Input != '"') {
        *Destination = NULL;
        return(NULL);
    }

    ptr = Input + 1;

    if ((ptr2 = strchr(ptr, '"')) == NULL) {
        *Destination = NULL;
        return(NULL);
    }

    *ptr2 = '\0';
    *Destination = MemMalloc(strlen(ptr) + 1);
    if (*Destination) {
        strcpy(*Destination, ptr);
        *ptr2 = '"';
        return(ptr2 + 1);
    } else {
        return(NULL);
    }
}

static unsigned char
*GrabGroup(ImapClient *client, unsigned char *Input, unsigned char **Destination)
{
    unsigned char *ptr, *ptr2;
    int    Nesting=1;

    if (*Input!='(') {
        *Destination=NULL;
        return(NULL);
    }

    ptr=Input+1;

    /* We have to make sure, we grab the proper closing paren, since they can be nested */

    ptr2=ptr;

    do {
        ptr2++;
        if (*ptr2=='(')
            Nesting++;
        if (*ptr2==')')
            Nesting--;
    } while (*ptr2 && Nesting!=0);


    if (!*ptr2) {
        *Destination=MemStrdup("");
        return(ptr2);
    }


    *ptr2='\0';
    *Destination=MemMalloc(strlen(ptr)+1);
    if (*Destination) {
        strcpy(*Destination,ptr);
        *ptr2=')';
        return(ptr2+1);
    } else {
        return(NULL);
    }
}

unsigned char
*GrabArgument(ImapClient *client, unsigned char *Input, unsigned char **Destination)
{
    unsigned char *ptr=Input, *ptr2;

    if (!ptr) {
        *Destination = MemStrdup("");
        return(NULL);
    }

    while (isspace(*ptr)) {
        ptr++;
    }

    if (*ptr == '{') {           /* octets */
        return(GrabOctet(client, ptr, Destination));
    } else if (*ptr == '"') {         /*literal */
        return(GrabLiteral(client, ptr, Destination));
    } else if (*ptr == '(') {
        return(GrabGroup(client, ptr, Destination));
    } else {
        ptr2 = strchr(ptr,' ');
        if (!ptr2 && (*ptr == '\0')) {
            *Destination = NULL;
            return(NULL);
        }

        if (ptr2) {
            *ptr2 = '\0';
        }

        *Destination = MemMalloc(strlen(ptr)+1);
        if (*Destination) {
            strcpy(*Destination, ptr);
            if (ptr2) {
                *ptr2 = ' ';
                ptr = ptr2 + 1;
            } else {
                ptr = ptr + strlen(ptr);
            }
        } else {
            return(NULL);
        }
    }
    return(ptr);
}


static unsigned long
ParseStoreFlags(unsigned char *command)
{
    unsigned long retval=0;
    unsigned char *ptr;

    ptr=command;

    while (ptr && *ptr) {
        if (XplStrNCaseCmp(ptr, "\\Deleted", 8)==0)
            retval |= MSG_STATE_DELETED;

        if (XplStrNCaseCmp(ptr, "\\Seen", 5)==0)
            retval |= MSG_STATE_READ;

        if (XplStrNCaseCmp(ptr, "\\Recent", 7)==0)
            retval |= MSG_STATE_RECENT;

        if (XplStrNCaseCmp(ptr, "\\Draft", 6)==0)
            retval |= MSG_STATE_DRAFT;

        if (XplStrNCaseCmp(ptr, "\\Answered", 9)==0)
            retval |= MSG_STATE_ANSWERED;

        if (XplStrNCaseCmp(ptr, "\\Flagged", 8)==0)
            retval |= MSG_STATE_PRIOHIGH;

        if (XplStrNCaseCmp(ptr, "\\Draft", 6)==0)
            retval |= MSG_STATE_DRAFT;

        ptr=strchr(ptr,' ');
        if (ptr)
            ptr++;
    }
    return(retval);
}




unsigned char
*RFC822SkipComment(unsigned char **String, unsigned long Trim)
{
    unsigned char *RetVal;
    unsigned char *ptr = *String;
    unsigned char *t = NULL;

    /* Skip past whitespace */
    for (RetVal = ++ptr; (*RetVal==' ' || *RetVal==0x09); RetVal++) {
        ;
    }

    do {
        switch(*ptr) {
        case '(': {
            if (!RFC822SkipComment(&ptr, 0)) {
                return(NULL);
            }
            t = --ptr;
            break;
        }

        case ')': {
            *String = ++ptr;
            if (Trim) {
                if (t) {
                    t[1] = '\0';
                } else {
                    *RetVal = '\0';
                }
            }
            return(RetVal);
        }

        case '\\': {
            if (*++ptr) {
                t = ptr;
                break;
            }
        } /* Fall-through */

        case '\0': {
            **String = '\0';
            return(NULL);
        }

        case 0x09:
        case ' ': {
            break;
        }

        default: {
            t = ptr;
            break;
        }
        }
    } while (ptr++);

    return(NULL);
}


void
RFC822SkipWhitespace(unsigned char **String)
{
    while (TRUE) {
        if ((**String == ' ') || (**String == 0x09)) {
            ++*String;
        } else if ((**String != '(') || !RFC822SkipComment(String,0)) {
            return;
        }
    }
}


unsigned char
*RFC822Quote(unsigned char *src)
{
    unsigned char *ret = src;

    if (strpbrk(src,"\\\"")) {
        unsigned char *dst = ret;

        while (*src) {
            if (*src == '\"') {
                src++;
            } else {
                if (*src == '\\') {
                    src++;
                }
                *dst++ = *src++;
            }
        }
        *dst = '\0';
    }
    return(ret);
}


unsigned char
*RFC822Copy(unsigned char *Source)
{
    return(RFC822Quote(MemStrdup(Source)));
}


unsigned char
*RFC822ParseWord(unsigned char *s, const unsigned char *Delimiters)
{
    unsigned char *st;
    unsigned char *str;

    if (!s) {
        return(NULL);
    }

    RFC822SkipWhitespace(&s);

    if (!*s) {
        return(NULL);
    }

    str = s;

    while (TRUE) {
        if (!(st = strpbrk (str, Delimiters ? Delimiters : WSpecials))) {
            return(str + strlen (str));
        }

        /* ESC in phrase */
        if (!Delimiters && (*st == I2C_ESC)) {
            str = ++st;
            switch (*st) {
            case I2C_MULTI: {
                switch (*++st) {
                case I2CS_94x94_JIS_OLD: /* old JIS (1978) */
                case I2CS_94x94_JIS_NEW: { /* new JIS (1983) */
                    str = ++st;
                    while ((st = strchr (st, I2C_ESC))!=NULL) {
                        if ((*++st == I2C_G0_94) && ((st[1] == I2CS_94_ASCII) || (st[1] == I2CS_94_JIS_ROMAN) || (st[1] == I2CS_94_JIS_BUGROM))) {
                            str = st += 2;
                            break;
                        }
                    }

                    /* eats entire text if no shift back */
                    if (!st || !*st) {
                        return(str + strlen (str));
                    }
                }
                }
                break;
            }

            case I2C_G0_94:
                switch (st[1]) {
                case I2CS_94_ASCII:
                case I2CS_94_JIS_ROMAN:
                case I2CS_94_JIS_BUGROM:
                    str = st + 2;
                    break;
                }
            }
        } else {
            switch (*st) {
            case '"': {
                while (*++st != '"') {
                    switch (*st) {
                    case '\0': {
                        return(NULL);
                    }

                    case '\\': {
                        if (!*++st) {
                            return(NULL);
                        }
                    }

                    default: {
                        break;
                    }
                    }
                }
                str = ++st;
                break;
            }

            case '\\': {
                if (st[1]) {
                    str = st + 2;
                    break;
                }
            }

            default: {
                return((st == s) ? NULL : st);
            }
            }
        }
    }
}


unsigned char
*RFC822ParsePhrase(unsigned char *s)
{
    unsigned char *curpos;

    if (!s) {
        return(NULL);
    }

    /* find first word of phrase */
    curpos=RFC822ParseWord(s, NULL);
    if (!curpos) {
        return(NULL);
    }

    if (!*curpos) {
        return(curpos);
    }
    s = curpos;
    RFC822SkipWhitespace(&s);

    return((s = RFC822ParsePhrase(s)) ? s : curpos);
}


RFC822AddressStruct
*RFC822ParseAddrSpec(unsigned char *Address, unsigned char **ret, unsigned char *DefaultHost)
{
    RFC822AddressStruct *Adr;
    unsigned char   c;
    unsigned char   *s;
    unsigned char   *t;
    unsigned char   *v;
    unsigned char   *end;

    if (!Address) {
        return(NULL);
    }

    RFC822SkipWhitespace(&Address);

    if (!*Address) {
        return(NULL);
    }

    if ((t=RFC822ParseWord(Address, WSpecials))==NULL) {
        return(NULL);
    }

    Adr=MemMalloc(sizeof(RFC822AddressStruct));
    memset(Adr, 0, sizeof(RFC822AddressStruct));
 
    c=*t;
    *t='\0';

    Adr->Mailbox=RFC822Copy(Address);
    *t=c;
    end=t;
    RFC822SkipWhitespace(&t);
    while (*t == '.') {
        Address = ++t;

        RFC822SkipWhitespace(&Address);

        /* get next word of mailbox */
        if ((t=RFC822ParseWord(Address, WSpecials))!=NULL) {
            end=t;
            c=*t;
            *t='\0';
            s=RFC822Copy(Address);
            *t=c;

            v = MemMalloc(strlen(Adr->Mailbox) + strlen(s) + 2);
            if (v) {
                sprintf(v, "%s.%s", Adr->Mailbox, s);
                MemFree(Adr->Mailbox);
                Adr->Mailbox = v;
            }    
            RFC822SkipWhitespace(&t);
        } else {
            break;
        }
    }

    t=end;

    RFC822SkipWhitespace(&end);

    if (*end!='@') {
        end=t;
    } else if (!(Adr->Host=RFC822ParseDomain(++end,&end))) {
        Adr->Host=MemStrdup(BROKEN_ADDRESS_HOST);
    }

    if (!Adr->Host) {
        Adr->Host=MemStrdup(DefaultHost);
    }

    if (end && !Adr->Personal) {
        while (*end==' ') {
            ++end;
        }
        if ((*end == '(') && ((s=RFC822SkipComment(&end, 1))!=NULL) && strlen (s)) {
            Adr->Personal=RFC822Copy(s);
        }
        RFC822SkipWhitespace(&end);
    }

    if (end && *end) {
        *ret=end;
    } else {
        *ret=NULL;
    }
    return(Adr);
}


unsigned char
*RFC822ParseDomain(unsigned char *Address, unsigned char **end)
{
    unsigned char *ret = NULL;
    unsigned char c;
    unsigned char *s;
    unsigned char *t;
    unsigned char *v;

    RFC822SkipWhitespace(&Address);
    if (*Address == '[') {
        if ((*end=RFC822ParseWord(Address + 1,"]\\"))!=NULL) {
            size_t len = ++*end - Address;

            ret=(unsigned char *)MemMalloc(len + 1);
            strncpy(ret, Address, len);
            ret[len]='\0';
        }
    } else if ((t=RFC822ParseWord(Address, WSpecials))!=NULL) {
        c=*t;
        *t='\0';
        ret=RFC822Copy(Address);
        *t=c;
        *end=t;
        RFC822SkipWhitespace(&t);
        while (*t=='.') {
            Address = ++t;
            RFC822SkipWhitespace(&Address);
            if ((Address=RFC822ParseDomain(Address, &t))!=NULL) {
                *end=t;
                c=*t;
                *t='\0';
                s=RFC822Copy(Address);
                *t=c;

                v=(unsigned char *)MemMalloc(strlen(ret)+strlen(s)+2);
                if (v) {
                    sprintf (v, "%s.%s", ret, s);
                    MemFree(ret);
                    ret = v;
                }
                RFC822SkipWhitespace(&t);
            } else {
                break;
            }
        }
    }

    return(ret);
}


RFC822AddressStruct
*RFC822ParseRouteAddress(unsigned char *Address, unsigned char **ret, unsigned char *DefaultHost)
{
    RFC822AddressStruct *Adr;
    unsigned char *s;
    unsigned char *t;
    unsigned char *adl;
    size_t adllen, i;

    if (!Address) {
        return(NULL);
    }

    RFC822SkipWhitespace(&Address);

    /* must start with open bracket */
    if (*Address != '<') {
        return(NULL);
    }

    t = ++Address;

    RFC822SkipWhitespace(&t);

    for (adl = NULL, adllen = 0; (*t == '@') && ((s = RFC822ParseDomain(t + 1, &t)) != NULL); ) { /* parse possible A-D-L */
        i = strlen (s) + 2;
        if (adl) {
            unsigned char *tmp;

            tmp = MemRealloc(adl, adllen + i);
            if (tmp) {
                adl = tmp;
                sprintf (adl + adllen - 1,",@%s",s);
                adllen += i;
            }
        } else {
            adl = MemMalloc(i);
            if (adl) {
                sprintf(adl, "@%s",s);
            }
            adllen += i;
        }
  
        MemFree(s);
        RFC822SkipWhitespace(&t);
        if (*t != ',') {
            break;
        }
        t++;
        RFC822SkipWhitespace(&t);
    }

    if (adl) {
        if (*t == ':') {
            Address=++t;
        }
    }

    /* parse address spec */
    if (!(Adr=RFC822ParseAddrSpec(Address, ret, DefaultHost))) {
        if (adl) {
            MemFree(adl);
        }
        return(NULL);
    }

    if (adl) {
        Adr->ADL = adl;
    }

    if (*ret) {
        if (**ret == '>') {
            ++*ret;
            RFC822SkipWhitespace(ret);
            if (!**ret) {
                *ret=NULL;
            }
            return(Adr);
        }
    }

    /* Unterminated mailbox */
    Adr->Next=MemMalloc(sizeof(RFC822AddressStruct));
    memset(Adr->Next, 0, sizeof(RFC822AddressStruct));
    Adr->Next->Mailbox = MemStrdup(BROKEN_ADDRESS_MAILBOX);
    Adr->Next->Host=MemStrdup(BROKEN_ADDRESS_HOST);
    return(Adr);
}


RFC822AddressStruct
*RFC822ParseMailbox(unsigned char **Address, unsigned char *DefaultHost)
{
    RFC822AddressStruct *Adr = NULL;
    unsigned char   *s;
    unsigned char   *end;

    if (!*Address) {
        return(NULL);
    }

    RFC822SkipWhitespace(Address);

    if (!**Address) {
        return(NULL);
    }

    if (*(s = *Address) == '<') {
        Adr = RFC822ParseRouteAddress(s, Address, DefaultHost);
    } else {
        if ((end = RFC822ParsePhrase(s))!=NULL) {
            if ((Adr = RFC822ParseRouteAddress(end, Address, DefaultHost))!=NULL) {
                if (Adr->Personal) {
                    MemFree(Adr->Personal);
                }
                *end = '\0';
                Adr->Personal = RFC822Copy(s);
            } else {
                Adr = RFC822ParseAddrSpec(s, Address, DefaultHost);
            }
        }
    }
    return(Adr);
}


RFC822AddressStruct
*RFC822ParseGroup(RFC822AddressStruct **List, RFC822AddressStruct *Last, unsigned char **Address, char *DefaultHost, unsigned long Depth)
{
    unsigned char   *p;
    unsigned char   *s;
    RFC822AddressStruct *Adr;

    if (Depth > 50) {
        return(NULL);
    }
    if (!*Address) {
        return(NULL);
    }

    RFC822SkipWhitespace(Address);

    if (!**Address || ((*(p = *Address) != ':') && !(p = RFC822ParsePhrase(*Address)))) {
        return(NULL);
    }

    s = p;
    RFC822SkipWhitespace(&s);

    if (*s != ':') {
        return NULL;
    }

    *p = '\0';
    p = ++s;

    RFC822SkipWhitespace(&p);

    /* write as address */
    Adr=MemMalloc(sizeof(RFC822AddressStruct));
    memset(Adr, 0, sizeof(RFC822AddressStruct));
    Adr->Mailbox=RFC822Copy(*Address);

    if (!*List) {
        *List = Adr;
    } else {
        Last->Next = Adr;
    }

    Last = Adr;
    *Address = p;

    while (*Address && **Address && (**Address != ';')) {
        if ((Adr=RFC822ParseAddress(List, Last, Address, DefaultHost, Depth+1))!=NULL) {
            Last=Adr;
            if (*Address) {
                RFC822SkipWhitespace(Address);
                switch (**Address) {
                case ',': {
                    ++*Address;
                }

                case ';':
                case '\0': {
                    break;
                }

                default: {
                    *Address = NULL;
                    Last->Next=MemMalloc(sizeof(RFC822AddressStruct));
                    Last=Last->Next;
                    memset(Last, 0, sizeof(RFC822AddressStruct));
                    Last->Mailbox=MemStrdup(BROKEN_ADDRESS_MAILBOX);
                    Last->Host=MemStrdup(BROKEN_ADDRESS_HOST);
                }
                }
            }
        } else {
            *Address = NULL;
            Adr=MemMalloc(sizeof(RFC822AddressStruct));
            memset(Adr, 0, sizeof(RFC822AddressStruct));
            Adr->Mailbox=MemStrdup(BROKEN_ADDRESS_MAILBOX);
            Adr->Host=MemStrdup(BROKEN_ADDRESS_HOST);
            Last = Last->Next = Adr;
        }
    }

    if (*Address) {
        if (**Address == ';') {
            ++*Address;
        }
        RFC822SkipWhitespace(Address);
    }

    Adr=MemMalloc(sizeof(RFC822AddressStruct));
    memset(Adr, 0, sizeof(RFC822AddressStruct));
    Last->Next=Adr;
    Last=Adr;
    return(Last);
}


RFC822AddressStruct
*RFC822ParseAddress(RFC822AddressStruct **List, RFC822AddressStruct *Last, unsigned char **Address, unsigned char *DefaultHost, unsigned long Depth)
{
    RFC822AddressStruct *Adr;

    if (!*Address) {
        return(NULL);
    }

    RFC822SkipWhitespace(Address);

    if (!**Address) {
        return(NULL);
    }

    if ((Adr=RFC822ParseGroup(List, Last, Address, DefaultHost, Depth))!=NULL) {
        Last=Adr;
    } else if ((Adr=RFC822ParseMailbox(Address, DefaultHost))!=NULL) {
        if (!*List) {
            *List=Adr;
        } else {
            Last->Next=Adr;
        }
        for (Last=Adr; Last->Next; Last=Last->Next) {
            ;
        }
    } else if (*Address) {
        return(NULL);
    }
    return(Last);
}


BOOL
RFC822ParseAddressList(RFC822AddressStruct **List, unsigned char *Address, unsigned char *DefaultHost)
{
    RFC822AddressStruct *Last=*List;
    RFC822AddressStruct *Adr;
    unsigned char   c;
 
    if (!Address) {
        return(FALSE);
    }

    RFC822SkipWhitespace(&Address);

    if (Address[0]=='\0') {
        return(FALSE);
    }

    while (Address) {
        if ((Adr=RFC822ParseAddress(List, Last, &Address, DefaultHost, 0))!=NULL) {
            Last=Adr;
            if (Address) {
                RFC822SkipWhitespace(&Address);
                switch(c=Address[0]) {
                case ',': {
                    ++Address;
                    break;
                }

                default: {
                    Last->Next=MemMalloc(sizeof(RFC822AddressStruct));
                    Last=Last->Next;
                    memset(Last, 0, sizeof(RFC822AddressStruct));
                    Last->Mailbox=MemStrdup(BROKEN_ADDRESS_MAILBOX);
                    Last->Host=MemStrdup(BROKEN_ADDRESS_HOST);
                } /* Fall-through */

                case '\0': {
                    Address=NULL;
                    break;
                }
                }
            }
        } else if (Address) {
            RFC822SkipWhitespace(&Address);
            Address=NULL;
            Adr=MemMalloc(sizeof(RFC822AddressStruct));
            memset(Adr, 0, sizeof(RFC822AddressStruct));
            Adr->Mailbox=MemStrdup(BROKEN_ADDRESS_MAILBOX);
            Adr->Host=MemStrdup(BROKEN_ADDRESS_HOST);
            if (Last) {
                Last=Last->Next=Adr;
            } else {
                *List=Last=Adr;
            }
            break;
        }
    }
    return(TRUE);
}

void
RFC822FreeAddressList(RFC822AddressStruct *List)
{
    RFC822AddressStruct *ToFree;

    while (List) {
        if (List->Personal) {
            MemFree(List->Personal);
        }
        if (List->Mailbox) {
            MemFree(List->Mailbox);
        }
        if (List->Host) {
            MemFree(List->Host);
        }
        if (List->ADL) {
            MemFree(List->ADL);
        }
        ToFree=List;
        List=List->Next;
        MemFree(ToFree);
    }
}

__inline static BOOL
NeedLiteral(const unsigned char *Text)
{
    do {
        if (*Text=='"' || *Text<0x20 || *Text>0x7f) {
            return(TRUE);
        }
    } while (*(++Text));
    return(FALSE);
}

static void
HandleStore(ImapClient *client, BOOL ByUID)
{
    int ReplyInt;
    unsigned char Reply[1024], Answer[1024];
    unsigned char *ptr, *ptr2, *MailSet, *Type, *Value, *Command, *NMAPCommand, *SetPtr;
    int WorkType;
    BOOL KeepGoing, Silent=FALSE, MoreSetElements;
    unsigned long StartNum;
    unsigned long EndNum;
    unsigned long LoopCntr;
    long Got;
    unsigned char DoSpace[2];
 
    /* Message attributes/values*/
    unsigned long Flags;


    if (ByUID)
        Command="UID STORE";
    else
        Command="STORE";

    ptr=client->Command+5;

    /* Grab the range */
    ptr=GrabArgument(client, ptr, &MailSet);
    if (!MailSet) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD %s Invalid argument\r\n", client->Ident, Command);
        SendClient(client, Answer, ReplyInt);
        return;
    }
    if (client->MailSet) {
        MemFree(client->MailSet);
    }
    client->MailSet=MailSet;

    /* Grab the way the flags are to be set */
    ptr=GrabArgument(client, ptr, &Type);
    if (!Type) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD %s Invalid argument\r\n", client->Ident, Command);
        SendClient(client, Answer, ReplyInt);
        MemFree(MailSet);
        client->MailSet=NULL;
        return;
    }

    /* Grab the flag values */
    ptr=GrabArgument(client, ptr, &Value);
    if (!Value) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD %s Invalid argument\r\n", client->Ident, Command);
        SendClient(client, Answer, ReplyInt);
        MemFree(MailSet);
        client->MailSet=NULL;
        MemFree(Type);
        return;
    }

    Flags=ParseStoreFlags(Value);
    MemFree(Value);

    NMAPCommand="NOOP";
    if (XplStrCaseCmp(Type, "FLAGS.SILENT")==0) {
        NMAPCommand="SFLG";
        Silent=TRUE;
    } else if (XplStrCaseCmp(Type, "+FLAGS.SILENT")==0) {
        NMAPCommand="AFLG";
        Silent=TRUE;
    } else if (XplStrCaseCmp(Type, "-FLAGS.SILENT")==0) {
        NMAPCommand="DFLG";
        Silent=TRUE;
    } else if (XplStrCaseCmp(Type, "FLAGS")==0) {
        NMAPCommand="SFLG";
    } else if (XplStrCaseCmp(Type, "+FLAGS")==0) {
        NMAPCommand="AFLG";
    } else if (XplStrCaseCmp(Type, "-FLAGS")==0) {
        NMAPCommand="DFLG";
    }
    MemFree(Type);

    /* Prepare the command list */
    SetPtr=MailSet;

    do {
        KeepGoing=TRUE;
        /* Prepare the command list */
        if ((ptr2=strchr(SetPtr, ','))!=NULL) {
            *ptr2='\0';
            MoreSetElements=TRUE;
        } else {
            MoreSetElements=FALSE;
        }

        if ((ptr2=strchr(SetPtr, ':'))!=NULL) {    /* We've got a set !      */
            WorkType=WORK_RANGE;

            /* Are we working by number or UID? */
            if (ByUID) {
                *ptr2='\0';         /* Separate the two items    */
                if (*(ptr2+1)=='*') {
                    EndNum = UID_HIGHEST;
                } else {
                    EndNum = atol(ptr2+1);
                }
                if (!UIDtoNUMRange(client, atol(SetPtr), EndNum, &StartNum, &EndNum)) {
                    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                    SendClient(client, Answer, ReplyInt);
                    MemFree(MailSet);
                    client->MailSet=NULL;
                    return;
                }
                *ptr2=':';
            } else {
                *ptr2='\0';
                StartNum = atol(SetPtr)-1;     /* Make sure we got a message there */

                if (client->Messages==0 || StartNum >= client->Messages) {
                    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                    SendClient(client, Answer, ReplyInt);
                    MemFree(MailSet);
                    client->MailSet=NULL;
                    return;
                }

                if (*(ptr2+1)=='*') {     /* Special: Last available item  */
                    EndNum = client->Messages-1;
                } else {
                    EndNum = atol(ptr2+1)-1;     /* Make sure we got a message there */
                    if (EndNum >= client->Messages) {
                        if (client->Messages==0) {
                            ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                            SendClient(client, Answer, ReplyInt);
                            MemFree(MailSet);
                            client->MailSet=NULL;
                            return;
                        } else {
                            EndNum = client->Messages-1;
                        }
                    }
                }
                *ptr2=':';
            }
            /* Now we have StartNum and EndNum filled out! */
        } else {            /* Just a single number     */
            WorkType=WORK_SINGLE;
            if (ByUID) {
                long retValue;

                retValue = UIDtoNUM(client, atol(SetPtr));
                if (retValue > -1) {
                    StartNum = retValue;
                } else {
                    LoadUIDList(client);
                    retValue = UIDtoNUM(client, atol(SetPtr));
                    if (retValue > -1) {
                        StartNum = retValue;
                    } else {
                        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                        SendClient(client, Answer, ReplyInt);
                        MemFree(MailSet);
                        client->MailSet=NULL;
                        return;
                    }
                }
                EndNum = StartNum;
            } else {
                StartNum = atol(SetPtr)-1;
                EndNum = StartNum;
                if (StartNum >= client->Messages) {
                    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                    SendClient(client, Answer, ReplyInt);
                    MemFree(MailSet);
                    client->MailSet=NULL;
                    return;
                }
            }
        }

        LoopCntr = StartNum;

        while(KeepGoing) {
            /* Do the dirty work */

            ReplyInt=sprintf(Answer, "%s %lu %lu\r\n", NMAPCommand, client->IDList[LoopCntr], Flags);
            SendNMAPServer(client, Answer, ReplyInt);
            ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
            switch(ReplyInt) {
            case 1000:
                if (!Silent) {
                    Got=atol(Reply);

                    ReplyInt=sprintf(Answer, "* %lu FETCH (FLAGS (", LoopCntr+1);
                    SendClient(client, Answer, ReplyInt);
                    DoSpace[0]='\0';
                    DoSpace[1]='\0';
                    if (Got & MSG_STATE_ANSWERED) {
                        ReplyInt=sprintf(Answer, "\\Answered");
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    } 
                    if (Got & MSG_STATE_DELETED) {
                        ReplyInt=sprintf(Answer, "%s\\Deleted",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    } 
                    if (Got & MSG_STATE_RECENT) {
                        ReplyInt=sprintf(Answer, "%s\\Recent",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    } 
                    if (Got & MSG_STATE_READ) {
                        ReplyInt=sprintf(Answer,"%s\\Seen",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    }
                    if (Got & MSG_STATE_DRAFT) {
                        ReplyInt=sprintf(Answer,"%s\\Draft",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    }
                    if (Got & MSG_STATE_PRIOHIGH) {
                        ReplyInt=sprintf(Answer,"%s\\Flagged",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    }
                    if (ByUID) {
                        ReplyInt=sprintf(Answer,") UID %lu)\r\n", client->UIDList[LoopCntr]);
                    } else {
                        ReplyInt=sprintf(Answer,"))\r\n");
                    }
                    SendClient(client, Answer, ReplyInt);
                }
                break;

            case NMAP_FAILED_READONLY: {
                if (Silent) {
                    ReplyInt=sprintf(Answer, "GFLG %lu\r\n", client->IDList[LoopCntr]);
                    SendNMAPServer(client, Answer, ReplyInt);
                    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
                    Got=atol(Reply);

                    ReplyInt=sprintf(Answer, "* %lu FETCH (FLAGS (", LoopCntr + 1);
                    SendClient(client, Answer, ReplyInt);
                    DoSpace[0]='\0';
                    DoSpace[1]='\0';
                    if (Got & MSG_STATE_ANSWERED) {
                        ReplyInt=sprintf(Answer, "\\Answered");
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    } 
                    if (Got & MSG_STATE_DELETED) {
                        ReplyInt=sprintf(Answer, "%s\\Deleted",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    } 
                    if (Got & MSG_STATE_RECENT) {
                        ReplyInt=sprintf(Answer, "%s\\Recent",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    } 
                    if (Got & MSG_STATE_READ) {
                        ReplyInt=sprintf(Answer,"%s\\Seen",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    }
                    if (Got & MSG_STATE_DRAFT) {
                        ReplyInt=sprintf(Answer,"%s\\Draft",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    }
                    if (Got & MSG_STATE_PRIOHIGH) {
                        ReplyInt=sprintf(Answer,"%s\\Flagged",DoSpace);
                        SendClient(client, Answer, ReplyInt);
                        DoSpace[0]=' ';
                    }
                    if (ByUID) {
                        ReplyInt=sprintf(Answer,") UID %lu)\r\n", client->UIDList[LoopCntr]);
                    } else {
                        ReplyInt=sprintf(Answer,"))\r\n");
                    }
                    SendClient(client, Answer, ReplyInt);
                }
                break;
            }

            default:
                ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO %s command failed\r\n", client->Ident, Command);
                SendClient(client, Answer, ReplyInt);
                MemFree(MailSet);
                client->MailSet=NULL;
                return;
            }
   

            /* Should we terminate the while loop ? */
            switch (WorkType) {
            case WORK_SINGLE:
                KeepGoing=FALSE;
                break;

            case WORK_RANGE:
                if (LoopCntr >= EndNum)
                    KeepGoing=FALSE;
                else
                    LoopCntr++;
                break;
            }
        }
        SetPtr+=strlen(SetPtr)+1;
    } while (MoreSetElements);

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s completed\r\n",client->Ident, Command);
    SendClient(client, Answer, ReplyInt);

    MemFree(MailSet);
    client->MailSet=NULL;

    return;
}

#define CHECK_PATH_LEN(path)  if (strlen(path)>=XPL_MAX_PATH) {  \
            path[XPL_MAX_PATH-1]='\0';   \
           }

#define REMOVE_MBOX_SPACES(mbox)    \
{              \
 unsigned char *PTR = mbox;    \
              \
 while((PTR = strchr(PTR, ' '))!=NULL) {\
  *PTR++ = 127;        \
 }             \
}              \

#define ADD_MBOX_SPACES(mbox)     \
{              \
 unsigned char *PTR = mbox;    \
              \
 while((PTR = strchr(PTR, 127))!=NULL) {\
  *PTR++ = ' ';        \
 }             \
}              \

static void
HandleCopy(ImapClient *client, BOOL ByUID)
{
    int    ReplyInt;
    unsigned char Reply[1024], Answer[1024];
    unsigned char *ptr, *ptr2, *MailSet, *Target, *Command, *SetPtr;
    int    WorkType;
    BOOL    KeepGoing, MoreSetElements;
    unsigned long StartNum;
    unsigned long EndNum;
    unsigned long LoopCntr;
 
    if (ByUID)
        Command="UID COPY";
    else
        Command="COPY";

    ptr=client->Command+5;

    /* Grab the range */
    ptr=GrabArgument(client, ptr, &MailSet);
    if (!MailSet) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD %s Invalid argument\r\n", client->Ident, Command);
        SendClient(client, Answer, ReplyInt);
        return;
    }

    if (client->MailSet) {
        MemFree(client->MailSet);
    }
    client->MailSet=MailSet;

    /* Grab the target mailbox */
    ptr=GrabArgument(client, ptr, &Target);
    if (!Target) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD %s Invalid argument\r\n", client->Ident, Command);
        SendClient(client, Answer, ReplyInt);
        MemFree(MailSet);
        client->MailSet=NULL;
        return;
    }
    REMOVE_MBOX_SPACES(Target);
    CHECK_PATH_LEN(Target);

    /* Prepare the command list */
    SetPtr=MailSet;

    do {
        KeepGoing=TRUE;
        /* Prepare the command list */
        if ((ptr2=strchr(SetPtr, ','))!=NULL) {
            *ptr2='\0';
            MoreSetElements=TRUE;
        } else {
            MoreSetElements=FALSE;
        }

        if ((ptr2=strchr(SetPtr, ':'))!=NULL) {    /* We've got a set !      */
            WorkType=WORK_RANGE;

            /* Are we working by number or UID? */
            if (ByUID) {
                *ptr2='\0';         /* Separate the two items    */
                if (*(ptr2+1)=='*') {
                    EndNum = UID_HIGHEST;
                } else {
                    EndNum = atol(ptr2+1);
                }
                if (!UIDtoNUMRange(client, atol(SetPtr), EndNum, &StartNum, &EndNum)) {
                    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                    SendClient(client, Answer, ReplyInt);
                    MemFree(Target);
                    MemFree(MailSet);
                    client->MailSet=NULL;
                    return;
                }
                *ptr2=':';
            } else {
                *ptr2='\0';
                StartNum = atol(SetPtr)-1;     /* Make sure we got a message there */

                if (client->Messages==0 || StartNum >= client->Messages) {
                    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                    SendClient(client, Answer, ReplyInt);
                    MemFree(Target);
                    MemFree(MailSet);
                    client->MailSet=NULL;
                    return;
                }

                if (*(ptr2+1)=='*') {     /* Special: Last available item  */
                    EndNum = client->Messages-1;
                } else {
                    EndNum = atol(ptr2+1)-1;     /* Make sure we got a message there */
                    if (EndNum >= client->Messages) {
                        if (client->Messages==0) {
                            ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                            SendClient(client, Answer, ReplyInt);
                            MemFree(Target);
                            MemFree(MailSet);
                            client->MailSet=NULL;
                            return;
                        } else {
                            EndNum = client->Messages-1;
                        }
                    }
                }
                *ptr2=':';
            }
            /* Now we have StartNum and EndNum filled out! */
        } else {            /* Just a single number     */
            WorkType=WORK_SINGLE;
            if (ByUID) {
                long retValue;

                retValue = UIDtoNUM(client, atol(SetPtr));
                if (retValue > -1) {
                    StartNum = retValue;
                } else {
                    LoadUIDList(client);
                    retValue = UIDtoNUM(client, atol(SetPtr));
                    if (retValue > -1) {
                        StartNum = retValue;
                    } else {
                        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                        SendClient(client, Answer, ReplyInt);
                        MemFree(Target);
                        MemFree(MailSet);
                        client->MailSet=NULL;
                        return;
                    }
                }
                EndNum = StartNum;
            } else {
                StartNum = atol(SetPtr)-1;
                EndNum = StartNum;
                if (StartNum >= client->Messages) {
                    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s Completed\r\n", client->Ident, Command);
                    SendClient(client, Answer, ReplyInt);
                    MemFree(Target);
                    MemFree(MailSet);
                    client->MailSet=NULL;
                    return;
                }
            }
        }

        LoopCntr = StartNum;

        while(KeepGoing) {
            /* Do the dirty work */

            ReplyInt=sprintf(Answer, "COPY %lu %s\r\n", client->IDList[LoopCntr], Target);
            SendNMAPServer(client, Answer, ReplyInt);

            ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
            if (ReplyInt!=1000) {
                if (ReplyInt==4224) {
                    ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO [TRYCREATE] %s Could not copy message\r\n",client->Ident, Command);
                    SendClient(client, Answer, ReplyInt);
                    MemFree(MailSet);
                    MemFree(Target);
                    client->MailSet=NULL;
                    return;
                } else {
                    ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO %s Could not copy message\r\n", client->Ident, Command);
                    SendClient(client, Answer, ReplyInt);
                    MemFree(MailSet);
                    client->MailSet=NULL;
                    MemFree(Target);
                    return;
                }
            } 

            /* Should we terminate the while loop ? */
            switch (WorkType) {
            case WORK_SINGLE:
                KeepGoing=FALSE;
                break;

            case WORK_RANGE:
                if (LoopCntr >= EndNum)
                    KeepGoing=FALSE;
                else
                    LoopCntr++;
                break;
            }
        }
        SetPtr+=strlen(SetPtr)+1;
    } while (MoreSetElements);

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK %s completed\r\n",client->Ident, Command);
    SendClient(client, Answer, ReplyInt);

    MemFree(MailSet);
    client->MailSet=NULL;
    MemFree(Target);

    return;
}


static int
ConnectUserToNMAPServer(ImapClient *client, unsigned char *Username, unsigned long UsernameBufferSize, unsigned char *Password)
{
    struct sockaddr_in soc_address;
    struct sockaddr_in *sin=&soc_address;
    unsigned char Answer[BUFSIZE+1];
    unsigned char Reply[BUFSIZE+1];
    int ReplyInt;
    unsigned long len;
    MDBValueStruct *User;

    User = MDBCreateValueStruct(IMAPDirectoryHandle, NULL);
    if (Password) {
        if (!MsgFindObject(Username, Answer, NULL, &sin->sin_addr, User)) {
            MDBDestroyValueStruct(User);
            LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_UNKNOWN_USER, LOG_NOTICE, 0, Username, Password, XplHostToLittle(client->cs.sin_addr.s_addr), 0, NULL, 0);
            return(FALSE);
        }
  
        if (!MDBVerifyPassword(Answer, Password, User)) {
            MDBDestroyValueStruct(User);
            XplSafeIncrement(IMAPStats.WrongPassword);
            LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_WRONG_PASSWORD, LOG_NOTICE, 0, Username, Password, XplHostToLittle(client->cs.sin_addr.s_addr), 0, NULL, 0);
            return(FALSE);
        }
    } else {
        if (!MsgFindObject(Username, Answer, NULL, &sin->sin_addr, User)) {
            MDBDestroyValueStruct(User);
            LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_UNKNOWN_USER, LOG_NOTICE, 0, Username, Password, XplHostToLittle(client->cs.sin_addr.s_addr), 0, NULL, 0);
            return(FALSE);
        }
    }

    len = strlen(User->Value[0]);
    if (len < UsernameBufferSize) {
        strcpy(Username, User->Value[0]);
    } else {
        return(FALSE);
    }
 
    MDBDestroyValueStruct(User);
    if (!MsgGetUserFeature(Answer, FEATURE_IMAP, NULL, NULL)) {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_DISABLED_FEATURE, LOG_NOTICE, 0, Username, NULL, XplHostToLittle(client->cs.sin_addr.s_addr), 0, NULL, 0);
        return(2);
    }

    /* Connect to NMAP server */
    sin->sin_family=AF_INET;
    sin->sin_port=htons(NMAP_PORT);
    client->NMAPs=socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (client->NMAPs==client->s) {
#if defined(DEBUG)
#if defined(NETWARE) || defined(LIBC)
        EnterDebugger();
#endif
#endif
        XplConsolePrintf("IMAPD.NLM: INFO: NMAP socket == client socket\n");
        client->NMAPs=socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
        if(client->NMAPs==client->s) {
            IPclose(client->NMAPs);
            client->NMAPs=-1;
            client->s=-1;
            ReplyInt=snprintf(Reply, sizeof(Reply), "* NO %s IMAP4 server not ready (cannot contact NMAPd)\r\n",Hostname);
            SendClient(client, Reply, ReplyInt);
            FlushClient(client);
            return(EndClientConnection(client));
        }
    }

    ReplyInt=connect(client->NMAPs, (struct sockaddr *)&soc_address, sizeof(soc_address));
    if (ReplyInt) {
        IPclose(client->NMAPs);
        client->NMAPs=-1;
        ReplyInt = snprintf(Reply, sizeof(Reply), "* NO %s IMAP4 server not ready (cannot contact NMAPd)\r\n",Hostname);
        SendClient(client, Reply, ReplyInt);
        FlushClient(client);
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_UNAVAILABLE, LOG_ERROR, 0, NULL, NULL, sin->sin_addr.s_addr, 0, NULL, 0);
        return(EndClientConnection(client));
    }

    ReplyInt=GetNMAPAnswer(client, Answer, sizeof(Answer),TRUE);

    switch (ReplyInt) {
    case NMAP_READY: {
        break;
    }

    case 4242: {
        unsigned char *ptr, *salt;
        MD5_CTX   mdContext;
        unsigned char digest[16];
        unsigned char HexDigest[33];
        int    i;

        ptr=strchr(Answer, '<');
        if (ptr) {
            ptr++;
            salt=ptr;
            if ((ptr=strchr(ptr, '>'))!=NULL) {
                *ptr='\0';
            }

            MD5_Init(&mdContext);
            MD5_Update(&mdContext, salt, strlen(salt));
            MD5_Update(&mdContext, NMAPHash, NMAP_HASH_SIZE);
            MD5_Final(digest, &mdContext);
            for (i=0; i<16; i++) {
                sprintf(HexDigest+(i*2),"%02x",digest[i]);
            }
            ReplyInt=sprintf(Answer, "AUTH %s\r\n", HexDigest);

            SendNMAPServer(client, Answer, ReplyInt);
            if (GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE)==1000) {
                break;
            }
        }
        /* Fall-through */
    }

    default: {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_UNAVAILABLE, LOG_ERROR, 0, NULL, NULL, sin->sin_addr.s_addr, ReplyInt, NULL, 0);
        ReplyInt = snprintf(Reply, sizeof(Reply), "* NO %s IMAP4 server not ready (cannot contact NMAPd %d)\r\n",Hostname, ReplyInt);
        SendClient(client, Reply, ReplyInt);
        FlushClient(client);
        return(EndClientConnection(client));
    }
    }

    ReplyInt=sprintf(Answer,"FLAG %lu\r\n",(unsigned long)(NMAP_SHOWDELETED | NMAP_OOBMESSAGES));
    if (SendNMAPServer(client, Answer, ReplyInt)) {
        ReplyInt=GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE);
    } else {
        ReplyInt=5000;
    }
    if (ReplyInt != NMAP_FLAGS_SET) {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_UNAVAILABLE, LOG_ERROR, 0, NULL, NULL, sin->sin_addr.s_addr, ReplyInt, NULL, 0);
        ReplyInt = snprintf(Reply, sizeof(Reply), "* NO %s IMAP4 server not ready (NMAP error:%d)\r\n",Hostname, ReplyInt);
        SendClient(client, Reply, ReplyInt);
        FlushClient(client);
        return(EndClientConnection(client));
    }

    /* Set TCP non blocking io */
    ReplyInt=1;
    setsockopt(client->NMAPs, IPPROTO_TCP, 1, (unsigned char *)&ReplyInt, sizeof(ReplyInt));

    LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_LOGIN, LOG_INFO, 0, Username, NULL, XplHostToLittle(client->cs.sin_addr.s_addr), 0, NULL, 0);

    return(TRUE);
}

__inline static long
NegotiateSSL(ImapClient *client)
{
    int ReplyInt;

    if (client->ClientSSL) {

        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_SSL_CONNECTION, LOG_INFO, 0, NULL, NULL, XplHostToLittle(client->cs.sin_addr.s_addr), 0, NULL, 0);

        ReplyInt = SSL_accept(client->CSSL);
        if (ReplyInt != 1) {
            return(CLIENT_TERMINATE);
        }
    } else {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_CONNECTION, LOG_INFO, 0, NULL, NULL, XplHostToLittle(client->cs.sin_addr.s_addr), 0, NULL, 0);
    }
    return(CLIENT_CONTINUE);
}

__inline static long
SendAsyncFlags(ImapClient *client)
{
    int ReplyInt;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned long Exists;
    unsigned long Recent;
    unsigned long Purged;
    unsigned long ExpungeCount=0;
    long retValue;

    if ((client->NMAPAsyncEvent & EVENT_MBOX_UNSEEN_PURGES) == 0) {
        return(CLIENT_CONTINUE);
    }

    /* If we have unseen purges we want to tell the client as soon as we possibly can */
    /* Unfortunately, RFC 2180 does not permit us to report this information in response */
    /* to the STORE, FETCH or SEARCH commands.  It also does not make sense to report it */
    /* if we are already doing and Expunge */

    SendNMAPServer(client, "UPDA\r\n", 6);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);

    /* Any last flags should get picked up in the GetNMAPAnswer() */
    SendUnseenFlags(client, Answer, ReplyInt);

    while (ReplyInt!=1000) {
        sscanf(Reply, "%lu %lu", &Exists, &Recent);
        /*
          Expunge messages are relative to the previous expunge, so we need to account for that 
          in the EXPUNGE and FETCH response by reducing the msg id accordingly; this assumes
          that NMAP returns the information in ascending order
        */
        retValue = NMAPtoNUM(client, Exists);
        if (retValue > -1) {
            Purged = retValue;

            if (Recent & MSG_STATE_PURGED) {
                ReplyInt=sprintf(Answer, "* %lu EXPUNGE\r\n", Purged-ExpungeCount);
                SendClient(client, Answer, ReplyInt);
                ExpungeCount++;
            } else {
                ReplyInt=sprintf(Answer, "* %lu FETCH (FLAGS (%s%s%s%s%s%s", Purged-ExpungeCount,
                                 (Recent & MSG_STATE_ANSWERED) ? "\\Answered " : "",
                                 (Recent & MSG_STATE_DELETED) ? "\\Deleted " : "",
                                 (Recent & MSG_STATE_RECENT) ? "\\Recent " : "",
                                 (Recent & MSG_STATE_READ) ? "\\Seen " : "",
                                 (Recent & MSG_STATE_DRAFT) ? "\\Draft " : "",
                                 (Recent & MSG_STATE_PRIOHIGH) ? "\\Flagged " : "");

                if (Answer[ReplyInt-1]==' ') {
                    ReplyInt--;
                }

                ReplyInt+=sprintf(Answer+ReplyInt, "))\r\n");  /* This way we overwrite the last space :-) */
                SendClient(client, Answer, ReplyInt);
            }
        }
        ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    }

    if ((Exists != client->NMAPCount) || (ExpungeCount!=0)) {
        sscanf(Reply, "%*s %lu %*u %lu %*u %lu", &Exists, &Recent, &Purged);

        ReplyInt=sprintf(Answer, "* %lu EXISTS\r\n",Exists-Purged);
        SendClient(client, Answer, ReplyInt);

        ReplyInt=sprintf(Answer, "* %lu RECENT\r\n",Recent);
        SendClient(client, Answer, ReplyInt);

        LoadUIDList(client);
    }
    client->NMAPAsyncEvent &= ~(EVENT_MBOX_NEW_MESSAGES | EVENT_MBOX_UNSEEN_PURGES);

    return(CLIENT_CONTINUE);
}

long
SendAsyncNewMessages(ImapClient *client)
{
    int ReplyInt;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned long Exists;
    unsigned long Recent;
    unsigned long Purged;


    if ((client->NMAPAsyncEvent & EVENT_MBOX_NEW_MESSAGES) == 0) {
        return(CLIENT_CONTINUE);
    }

    /* If we have new messages, we want to tell the client as soon as we possibly can.  */
    /* Untagged EXIST and RECENT responses is how this is done.  If the current command    */
    /* changes the mailbox state, (Close, Logout, Select) we don't want to confuse the client with these responses.*/
    /* They are redundant on the EXPUNGE command since it sends EXIST and RECENT already. */


    SendNMAPServer(client, "STAT\r\n", 6);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);

    sscanf(Reply, "%*s %lu %*u %lu %*u %lu", &Exists, &Recent, &Purged);

    if (Exists > client->NMAPCount) {
        LoadUIDList(client);
    }

    ReplyInt=sprintf(Answer, "* %lu EXISTS\r\n", Exists-Purged);
    SendClient(client, Answer, ReplyInt);
    ReplyInt=sprintf(Answer, "* %lu RECENT\r\n", Recent);
    SendClient(client, Answer, ReplyInt);
    client->NMAPAsyncEvent &= ~EVENT_MBOX_NEW_MESSAGES;

    return(CLIENT_CONTINUE);
}

static int
ImapCommandUnknown(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    }

    LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_UNHANDLED, LOGGER_EVENT_UNHANDLED_REQUEST, LOG_INFO, 0, 
                client->User, 
                client->Command, 
                XplHostToLittle(client->cs.sin_addr.s_addr), 
                0, 
                NULL, 
                0);
    ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Command unrecognized,\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

/********** IMAP client commands - any state (Capability, Noop, Logout) **********/
int
ImapCommandCapability(void *param)
{
    ImapClient *client = (ImapClient *)param;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    }

    if (!ClientSSLOptions || client->ClientSSL) {
        SendClient(client, Capability.message, Capability.len);
    } else {
        SendClient(client, Capability.SSL.message, Capability.SSL.len);
    }

    SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s OK CAPABILITY completed,\r\n", client->Ident));
    return(CLIENT_CONTINUE);
}

int
ImapCommandNoop(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];
    unsigned char Reply[1024];

    if (client->State == STATE_SELECTED) {
        SendNMAPServer(client, "NOOP\r\n", 6);
        ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);

        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK NOOP completed\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandLogout(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    SendClient(client, "* BYE IMAP4rev1 Server signing off\r\n", 36);
    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK LOGOUT completed\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_TERMINATE);
}


/********** IMAP client commands - not authenticated state (STARTTLS, AUTHENTICATE, LOGIN) **********/

int
ImapCommandStartTls(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    if (client->State >= STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    if (!ClientSSLOptions) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Command unrecognized,\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK Begin TLS negotiation\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    FlushClient(client);

    client->ClientSSL=TRUE;
    client->CSSL = SSL_new(SSLContext);
    if (client->CSSL) {
        client->ClientWrite = FuncTbl.writeSSL;
        client->ClientRead = FuncTbl.readSSL;
        client->ClientSktCtx = client->CSSL;        
        if ((SSL_set_bsdfd(client->CSSL, client->s) == 1) && (SSL_accept(client->CSSL) == 1)) {
            return(CLIENT_CONTINUE);
        }
        client->ClientWrite = FuncTbl.write;
        client->ClientRead = FuncTbl.read;
        client->ClientSktCtx = (void *)client->s;
        SSL_free(client->CSSL);
        client->CSSL=NULL;
        client->ClientSSL=FALSE;
    }
    return(CLIENT_CONTINUE);
}

int
ImapCommandAuthenticate(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];
    unsigned char Reply[1024];

    unsigned char *PW;
    unsigned long len;

    if (client->State != STATE_FRESH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    if (XplStrNCaseCmp(client->Command+13,"LOGIN", 5)!=0) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Authentication mechanism unknown\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    SendClient(client, "+ VXNlcm5hbWU6AA==\r\n", 20);
    FlushClient(client);
    if (ReadCommandLine(client)) {
        len = strlen(client->Command);
        if (len < sizeof(Reply)) {
            memcpy(Reply, client->Command, len);
            Reply[len] = '\0';
        } else {
            memcpy(Reply, client->Command, sizeof(Reply) - 1);
            Reply[sizeof(Reply) - 1] = '\0';
        }
       
        SendClient(client, "+ UGFzc3dvcmQ6AA==\r\n", 20);
        FlushClient(client);
        if (ReadCommandLine(client)) {
            PW=DecodeBase64(client->Command);
            DecodeBase64(Reply);

            len = strlen(Reply);
            if (len < sizeof(client->User)) {
                strcpy(client->User, Reply);
            } else {
                SendClient(client, "* BYE IMAP4rev1 Server signing off\r\n", 36);
                ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO permission denied\r\n", client->Ident);
                SendClient(client, Answer, ReplyInt);
                return(CLIENT_TERMINATE);
            }

            ReplyInt=ConnectUserToNMAPServer(client, client->User, sizeof(client->User) - 1,  PW);
            if (ReplyInt!=1) {
                if (ReplyInt==0) {
                    XplDelay(2000);
                    client->LoginCount++;
                    if (client->LoginCount>=MAX_FAILED_LOGINS) {
                        SendClient(client, "* BYE IMAP4rev1 Server signing off\r\n", 36);
                        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO permission denied\r\n",client->Ident);
                        SendClient(client, Answer, ReplyInt);
                        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_MAX_FAILED_LOGINS, LOG_NOTICE, 0, client->User, NULL, XplHostToLittle(client->cs.sin_addr.s_addr), MAX_FAILED_LOGINS, NULL, 0);
                        return(CLIENT_TERMINATE);
                    } else {
                        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO permission denied\r\n",client->Ident);
                        SendClient(client, Answer, ReplyInt);
                    }
                    return(CLIENT_CONTINUE);
                } else {
                    SendClient(client, "* BYE IMAP4rev1 Server signing off\r\n", 36);
                    ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO IMAP Access for account disabled\r\n",client->Ident);
                    SendClient(client, Answer, ReplyInt);
                    return(CLIENT_TERMINATE);
                }
            }
            client->State=STATE_AUTH;

            CMAuthenticated((unsigned long)client->cs.sin_addr.s_addr, client->User);

            ReplyInt=snprintf(Answer, sizeof(Answer), "USER %s\r\n",client->User);
            SendNMAPServer(client, Answer, ReplyInt);
            ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
            switch(ReplyInt) {
            case 1000:
                ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK AUTHENTICATE completed\r\n",client->Ident);
                SendClient(client, Answer, ReplyInt);
                snprintf(Answer, sizeof(Answer), "IMAP:%s", client->User);
                XplRenameThread(XplGetThreadID(), Answer);
                break;

            case 4221:
                ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO User already connected\r\n",client->Ident);
                SendClient(client, Answer, ReplyInt);
                break;

            case 4224:
                ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO User does not exist\r\n",client->Ident);
                SendClient(client, Answer, ReplyInt);
                break;

            default:
                ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP returned: %d\r\n",client->Ident, ReplyInt);
                SendClient(client, Answer, ReplyInt);
                break;
            }
            return(CLIENT_CONTINUE);
        }
    }

    return(CLIENT_TERMINATE);
}

int
ImapCommandLogin(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned char *Username=NULL, *Password=NULL;

    if (client->State!=STATE_FRESH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=client->Command+6;
    while (isspace(*ptr))
        ptr++;

    /* ptr points to username */

    if (!*ptr) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
       
    ptr=GrabArgument(client, ptr, &Username);
    if (!ptr || !Username) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    /* We're abusing password as a placeholder, save a stack variable */
    while ((Password=strchr(Username, ' '))!=NULL) {
        *Password='_';
    }
    Password=NULL;
 
    if (strlen(Username) < sizeof(client->User)) {
        strcpy(client->User, Username);
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO permission denied\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=GrabArgument(client, ptr, &Password);
    if (!ptr || !Password) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        MemFree(Username);
        return(CLIENT_CONTINUE);
    }

    ReplyInt = ConnectUserToNMAPServer(client, client->User, sizeof(client->User) - 1, Password);
    if (ReplyInt!=1) {
        if (ReplyInt==0) {
            XplDelay(2000);
            client->LoginCount++;
            if (client->LoginCount>=MAX_FAILED_LOGINS) {
                SendClient(client, "* BYE IMAP4rev1 Server signing off\r\n", 36);
                ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO permission denied\r\n",client->Ident);
                SendClient(client, Answer, ReplyInt);
                LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_MAX_FAILED_LOGINS, LOG_NOTICE, 0, client->User, NULL, XplHostToLittle(client->cs.sin_addr.s_addr), MAX_FAILED_LOGINS, NULL, 0);
                if (Username) {
                    MemFree(Username);
                }
                if (Password) {
                    MemFree(Password);
                }
                return(CLIENT_TERMINATE);
            } else {
                ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO permission denied\r\n",client->Ident);
                SendClient(client, Answer, ReplyInt);
                if (Username) {
                    MemFree(Username);
                }
                if (Password) {
                    MemFree(Password);
                }
            }
            return(CLIENT_CONTINUE);
        } else {
            SendClient(client, "* BYE IMAP4rev1 Server signing off\r\n", 36);
            ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO IMAP Access for account disabled\r\n",client->Ident);
            SendClient(client, Answer, ReplyInt);
            if (Username) {
                MemFree(Username);
            }
            if (Password) {
                MemFree(Password);
            }
            return(CLIENT_TERMINATE);
        }
    }

    CMAuthenticated((unsigned long)client->cs.sin_addr.s_addr, client->User);

    if (Username) {
        MemFree(Username);
    }

    if (Password) {
        MemFree(Password);
    }

    client->State=STATE_AUTH;

    ReplyInt = snprintf(Answer, sizeof(Answer), "USER %s\r\n",client->User);
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    switch(ReplyInt) {
    case 1000:
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK LOGIN completed\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        snprintf(Answer, sizeof(Answer), "IMAP:%s", client->User);
        XplRenameThread(XplGetThreadID(), Answer);
        break;

    case 4221:
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO User already connected\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        break;

    case 4224:
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO User does not exist\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        break;

    default:
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP returned: %d\r\n",client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        break;
    }
    return(CLIENT_CONTINUE);
}

/********** IMAP client commands - authenticated state **********/

int
ImapCommandSelect(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];

    unsigned char *mbox;
    unsigned long Exists;
    unsigned long Recent;
    unsigned long Purged;
    unsigned long BoxID;
    BOOL IsExamine = FALSE;

    if (client->State == STATE_SELECTED) {

        SendNMAPServer(client, "RSET MBOX\r\n", 11);
        ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);

        /* Any last flags should get picked up in the GetNMAPAnswer() */
        SendUnseenFlags(client, Answer, ReplyInt);

        /* Any events are for the old mailbox, so forget about them. */
        client->NMAPAsyncEvent = 0;

        client->State = STATE_AUTH;

        if (ReplyInt!=1000) {
            ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Could not select box\r\n",client->Ident);
            SendClient(client, Answer, ReplyInt);
            return(CLIENT_CONTINUE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=client->Command+7;
    if (toupper(*(ptr-1))=='E') {
        IsExamine=TRUE;
    } else if (*(ptr-1)!=' ') {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    GrabArgument(client, ptr, &mbox);
    if (!mbox) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    REMOVE_MBOX_SPACES(mbox);
    CHECK_PATH_LEN(mbox);

    if (strlen(mbox) < sizeof(client->MBox)) {
        strcpy(client->MBox, mbox);
        MemFree(mbox);
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Could not select box\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }

    if((client->MBox[0] == '/') || (client->MBox[0] == '\\')) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "MBOX %s\r\n", client->MBox + 1);
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "MBOX %s\r\n", client->MBox);
    }

    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt!=NMAP_MBOX_RW_SELECTED) {
        if (ReplyInt!=NMAP_MBOX_RO_SELECTED) {
            ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Could not select box\r\n", client->Ident);
            SendClient(client, Answer, ReplyInt);
            return(CLIENT_CONTINUE);
        } else {
            client->ReadOnly=TRUE;
        }
    }
    BoxID=atol(Reply);

    SendNMAPServer(client, "STAT\r\n", 6);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);

    sscanf(Reply, "%*s %lu %*u %lu %*u %lu", &Exists, &Recent, &Purged);
    ReplyInt=sprintf(Answer, "* %lu EXISTS\r\n", Exists-Purged);
    SendClient(client, Answer, ReplyInt);
    ReplyInt=sprintf(Answer, "* %lu RECENT\r\n", Recent);
    SendClient(client, Answer, ReplyInt);

    LoadUIDList(client);

    SendClient(client, "* FLAGS (\\Answered \\Flagged \\Deleted \\Draft \\Seen)\r\n", 52);
    ReplyInt=sprintf(Answer, "* OK [UIDVALIDITY %lu] UID validity status\r\n",BoxID);
    SendClient(client, Answer, ReplyInt);
    if (IsExamine || client->ReadOnly) {
        SendClient(client, "* OK [PERMANENTFLAGS ()] Permanent flags\r\n", 42);
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK [READ-ONLY] %s completed\r\n",client->Ident, IsExamine ? "EXAMINE" : "SELECT");
    } else {
        SendClient(client, "* OK [PERMANENTFLAGS (\\Answered \\Flagged \\Deleted \\Draft \\Seen)] Permanent flags\r\n", 82);
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK [READ-WRITE] SELECT completed\r\n",client->Ident);
    }
    SendClient(client, Answer, ReplyInt);

    client->State=STATE_SELECTED;
    return(CLIENT_CONTINUE);
}

int
ImapCommandExamine(void *param)
{
    return(ImapCommandSelect(param));
}

int
ImapCommandCreate(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned char *mbox;
    int len;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=client->Command+2;
    while(!isspace(*ptr) && *ptr != '\0')
        ++ptr;
    if(*ptr == '\0') {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    GrabArgument(client, ptr, &mbox);
      
    if (!mbox || mbox[0]=='\0') {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    REMOVE_MBOX_SPACES(mbox);
    CHECK_PATH_LEN(mbox);
    len = strlen(mbox);

    if (XplStrCaseCmp(mbox,client->User)==0) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Cannot create box with user's name\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }
    while(len > 0 && (mbox[len - 1] == '/' || mbox[len - 1] == '\\')) {
        mbox[--len] = '\0';
    }
    snprintf(Answer, sizeof(Answer), "CREA %s\r\n",mbox);
    SendNMAPServer(client, Answer, len + 7);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt!=1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Could not create box\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK Mailbox created\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
    }
    MemFree(mbox);
    return(CLIENT_CONTINUE);
}

int
ImapCommandDelete(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    BOOL repeat = FALSE;
    unsigned char *mbox;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=client->Command+7;
    ptr=GrabArgument(client, ptr, &mbox);
    if (!mbox) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    REMOVE_MBOX_SPACES(mbox);
    CHECK_PATH_LEN(mbox);
    if(!XplStrCaseCmp(mbox, "INBOX")) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Can't delete inbox\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    do {
        ReplyInt=snprintf(Answer, sizeof(Answer), "RMOV %s\r\n",mbox);
        SendNMAPServer(client, Answer, ReplyInt);
        ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
        if(ReplyInt == 4227 && client->State == STATE_SELECTED) {
            repeat = repeat ? FALSE : TRUE; /* Only try once */
            SendNMAPServer(client, "RSET MBOX\r\n", 11);
            if(GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE) != 1000) {
                repeat = FALSE;
            } else {
                client->State=STATE_AUTH;
                client->MBox[0]='\0';
            }
        } else {
            repeat = FALSE;
        }
    } while(repeat);

    MemFree(mbox);

    if (ReplyInt!=1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO DELETE failed: %d\r\n",client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK DELETE completed\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandRename(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned char *newbox;
    unsigned char *mbox;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=client->Command+7;
    ptr=GrabArgument(client, ptr, &mbox);
    if (!mbox) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    REMOVE_MBOX_SPACES(mbox);
    ptr=GrabArgument(client, ptr, &newbox);
    if (!newbox) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }
    REMOVE_MBOX_SPACES(newbox);
    if (client->State==STATE_SELECTED) { 
        if (XplStrCaseCmp(client->MBox, mbox)==0) {
            SendNMAPServer(client, "RSET MBOX\r\n", 11);
            ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
        }
    }

    CHECK_PATH_LEN(mbox);
    CHECK_PATH_LEN(newbox);

    ReplyInt=snprintf(Answer, sizeof(Answer), "RNAM %s %s\r\n",mbox, newbox);
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);

    if (ReplyInt!=1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO RENAME failed: %d\r\n",client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        MemFree(mbox);
        MemFree(newbox);
        return(CLIENT_CONTINUE);
    }

    if (XplStrCaseCmp(mbox, "INBOX")==0) {
        SendNMAPServer(client, "CREA INBOX\r\n", 12);
        ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    }
    if (XplStrCaseCmp(client->MBox, mbox)==0) {
        ReplyInt=snprintf(Answer, sizeof(Answer), "MBOX %s\r\n",client->MBox);
        SendNMAPServer(client, Answer, ReplyInt);
        ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
        LoadUIDList(client);
    }

    MemFree(mbox);
    MemFree(newbox);

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK RENAME completed\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandSubscribe(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned char *mbox;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr = client->Command + 9;
    while(isspace(*ptr)) {
        ptr++;
    }

    if (!*ptr) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
       
    ptr = GrabArgument(client, ptr, &mbox);
    if ((ptr == NULL) || (mbox == NULL)) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    REMOVE_MBOX_SPACES(mbox);

    ReplyInt = snprintf(Answer, sizeof(Answer), "SUBS %s\r\n", mbox);
    SendNMAPServer(client, Answer, ReplyInt);

    ReplyInt = GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt == 1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK SUBSCRIBE completed\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error: %d\r\n", client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
    }

    MemFree(mbox);
    return(CLIENT_CONTINUE);
}

int
ImapCommandUnsubscribe(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned char *mbox;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr = client->Command + 11;
    while(isspace(*ptr)) {
        ptr++;
    }

    if (!*ptr) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
       
    ptr = GrabArgument(client, ptr, &mbox);
    if ((ptr == NULL) || (mbox == NULL)) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    REMOVE_MBOX_SPACES(mbox);

    ReplyInt = snprintf(Answer, sizeof(Answer), "UNSUBS %s\r\n", mbox);
    SendNMAPServer(client, Answer, ReplyInt);

    ReplyInt = GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt == 1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK UNSUBSCRIBE completed\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error: %d\r\n", client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
    }

    MemFree(mbox);
    return(CLIENT_CONTINUE);
}

int
ImapCommandList(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];

    unsigned char *mbox, *pattern;
    int    len;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=client->Command+5;
    while(isspace(*ptr))
        ptr++;

    if (!*ptr) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
         
    ptr=GrabArgument(client, ptr, &mbox);
    if (!ptr || !mbox) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=GrabArgument(client, ptr, &pattern);
    if (!ptr || !pattern) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }

    REMOVE_MBOX_SPACES(pattern);
    len = strlen(pattern);
    while(len > 0 && (pattern[len - 1] == '/' || pattern[len - 1] == '\\')) {
        pattern[--len] = '\0';
    }

    /* Handle RFC special command "" on the pattern */
    if (pattern[0]=='\0') {
        len=sprintf(Answer, "* LIST (\\Noselect) \"/\" \"\"\r\n");
        SendClient(client, Answer, len);
        len=snprintf(Answer, sizeof(Answer), "%s OK LIST completed\r\n",client->Ident);
        SendClient(client, Answer, len);
        MemFree(pattern);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }


    ReplyInt=snprintf(Answer, sizeof(Answer), "SHOW %s\r\n",pattern);
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt!=2002) {
        if (ReplyInt==1000) {
            ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Pattern doesn't match mailbox\r\n",client->Ident);
            SendClient(client, Answer, ReplyInt);
        } else {
            ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error:%d\r\n",client->Ident, ReplyInt);
            SendClient(client, Answer, ReplyInt);
        }
        MemFree(pattern);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }

    while ((ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE))!=1000 && !Exiting) {
        ADD_MBOX_SPACES(Reply);
        if (Reply[strlen(Reply)-1]!='/') { 
            ReplyInt=snprintf(Answer, sizeof(Answer), "* LIST () \"/\" \"%s\"\r\n",Reply);
        } else {
            Reply[strlen(Reply)-1]='\0';
            ReplyInt=snprintf(Answer, sizeof(Answer), "* LIST (\\NoSelect) \"/\" \"%s\"\r\n",Reply);
        }
        SendClient(client, Answer, ReplyInt);
    }
    if (ReplyInt!=1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error:%d\r\n",client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        MemFree(pattern);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK LIST completed\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);

    MemFree(mbox);
    MemFree(pattern);
    return(CLIENT_CONTINUE);
}

int
ImapCommandLsub(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned char *mbox, *pattern;
    int    len;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=client->Command+5;
    while(isspace(*ptr))
        ptr++;

    if (!*ptr) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
       
    ptr=GrabArgument(client, ptr, &mbox);
    if (!ptr || !mbox) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=GrabArgument(client, ptr, &pattern);
    if (!ptr || !pattern) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }

    REMOVE_MBOX_SPACES(pattern);
    len = strlen(pattern);
    while(len > 0 && (pattern[len - 1] == '/' || pattern[len - 1] == '\\')) {
        pattern[--len] = '\0';
    }

    /* Handle RFC special command "" on the pattern */
    if (pattern[0]=='\0') {
        len=sprintf(Answer, "* LSUB (\\Noselect) \"/\" \"\"\r\n");
        SendClient(client, Answer, len);
        len=snprintf(Answer, sizeof(Answer), "%s OK LSUB completed\r\n",client->Ident);
        SendClient(client, Answer, len);
        MemFree(pattern);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }


    ReplyInt=snprintf(Answer, sizeof(Answer), "SHOWSUB %s\r\n",pattern);
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt!=2002) {
        if (ReplyInt==1000) {
            ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Pattern doesn't match mailbox\r\n",client->Ident);
            SendClient(client, Answer, ReplyInt);
        } else {
            ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error:%d\r\n",client->Ident, ReplyInt);
            SendClient(client, Answer, ReplyInt);
        }
        MemFree(pattern);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }

    while ((ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE))!=1000 && !Exiting) {
        ADD_MBOX_SPACES(Reply);
        if (Reply[strlen(Reply)-1]!='/') { 
            ReplyInt=snprintf(Answer, sizeof(Answer), "* LSUB () \"/\" \"%s\"\r\n",Reply);
        } else {
            Reply[strlen(Reply)-1]='\0';
            ReplyInt=snprintf(Answer, sizeof(Answer), "* LSUB (\\NoSelect) \"/\" \"%s\"\r\n",Reply);
        }
        SendClient(client, Answer, ReplyInt);
    }
    if (ReplyInt!=1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error:%d\r\n",client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        MemFree(mbox);
        MemFree(pattern);
        return(CLIENT_CONTINUE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK LSUB completed\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);

    MemFree(mbox);
    MemFree(pattern);
    return(CLIENT_CONTINUE);
}

int
ImapCommandStatus(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned char *mbox;
    unsigned char *items;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=client->Command+6;
    while(isspace(*ptr))
        ptr++;

    if (!*ptr) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
         
    ptr=GrabArgument(client, ptr, &mbox);
    if (!ptr || !mbox) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    REMOVE_MBOX_SPACES(mbox);

    ptr=GrabArgument(client, ptr, &items);
    if (!ptr || !items) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }
    ReplyInt=snprintf(Answer, sizeof(Answer), "CHECK %s\r\n", mbox);
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt!=1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO STATUS NMAP failed:%d\r\n",client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
    } else {
        unsigned char *ptr,*DoSpace="";
        unsigned long TotalCount;
        unsigned long RecentCount;
        unsigned long ReadCount;
        unsigned long PurgedCount;
        unsigned long UIDNext;
        unsigned long MBoxUID;

        ptr=items;
        if (*ptr=='(')
            ptr++;

        // Total, Recent, Purged, Read, IDs
        sscanf(Reply, "%lu %*u %lu %*u %lu %*u %lu %*u %lu %lu", &TotalCount, &RecentCount, &PurgedCount, &ReadCount, &UIDNext, &MBoxUID);

        ADD_MBOX_SPACES(mbox);
        ReplyInt=snprintf(Answer, sizeof(Answer), "* STATUS \"%s\" (", mbox);
        SendClient(client, Answer, ReplyInt);

        while (ptr && *ptr) {
            if (XplStrNCaseCmp(ptr, "MESSAGES", 8)==0) {
                ReplyInt=sprintf(Answer, "%sMESSAGES %lu", DoSpace, TotalCount-PurgedCount);
                SendClient(client, Answer, ReplyInt);
                DoSpace=" ";
            }

            if (XplStrNCaseCmp(ptr, "RECENT", 6)==0) {
                ReplyInt=sprintf(Answer, "%sRECENT %lu", DoSpace, RecentCount);
                SendClient(client, Answer, ReplyInt);
                DoSpace=" ";
            }

            if (XplStrNCaseCmp(ptr, "UIDNEXT", 7)==0) {
                ReplyInt=sprintf(Answer, "%sUIDNEXT %lu", DoSpace, UIDNext);
                SendClient(client, Answer, ReplyInt);
                DoSpace=" ";
            }

            if (XplStrNCaseCmp(ptr, "UIDVALIDITY", 11)==0) {
                ReplyInt=sprintf(Answer, "%sUIDVALIDITY %lu", DoSpace, MBoxUID);
                SendClient(client, Answer, ReplyInt);
                DoSpace=" ";
            }

            if (XplStrNCaseCmp(ptr, "UNSEEN", 6)==0) {
                ReplyInt=sprintf(Answer, "%sUNSEEN %lu", DoSpace, TotalCount-ReadCount-PurgedCount);
                SendClient(client, Answer, ReplyInt);
                DoSpace=" ";
            }

            ptr=strchr(ptr,' ');
            if (ptr)
                ptr++;
        }
        SendClient(client, ")\r\n", 3);
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK STATUS completed\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
    }
    if (mbox)
        MemFree(mbox);
    if (items)
        MemFree(items);

    return(CLIENT_CONTINUE);
}

int
ImapCommandAppend(void *param)
{
    ImapClient *client = (ImapClient *)param;
    unsigned char *mbox=NULL, *tmp, *ptr2;
    unsigned long Flags=0;
    long    count;
    long Size;
    unsigned long Date=0;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];

    if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr=client->Command+6;
    if (*ptr!=' ') {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    while (isspace(*ptr))
        ptr++;

    GrabArgument(client, ptr, &mbox);
    if (!mbox) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Out of memory\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    REMOVE_MBOX_SPACES(mbox);
    CHECK_PATH_LEN(mbox);

    tmp=strchr(ptr, '(');
    if (tmp) {
        ptr=tmp;
        tmp=strchr(ptr, ')');
        if (tmp) {
            *tmp='\0';
            Flags=ParseStoreFlags(ptr+1);
            *tmp=')';
        }
        ptr=tmp;
    }

    tmp=ptr;  /* To remember where the date might be */

    ptr=strchr(ptr, '{'); /* fix me } */
    if (!ptr) {
        MemFree(mbox);
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    /* Parse any date... */
    ptr2=strchr(tmp, ' ');
    if (!ptr2) {
        MemFree(mbox);
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    /* Format is: day-month-year HH:mm:ss UTFoffset */
    while (isspace(*ptr2)) {
        ptr2++;
    }

    /* Did we get a date-time? */
    if (ptr2[0]=='"') {
        unsigned long day, month, year, hour, min, sec, offset;
        unsigned char months[20];
        unsigned long i=0;

        ptr2++;
        while (isspace(*ptr2)) {
            ptr2++;
        }
        tmp=ptr2;
        while (tmp!=ptr) {
            if (i<2 && tmp[0]=='-') {
                tmp[0]=' ';
                i++;;
            } else if (tmp[0]==':') {
                tmp[0]=' ';
            }
            tmp++;
        }
 
        sscanf(ptr2, "%d %s %d %d %d %d %d", (int*)&day, months, (int *)&year, (int *)&hour, (int *)&min, (int *)&sec, (int *)&offset);

        month=1;
        for (i=0; i<12; i++) {
            if (XplStrCaseCmp(months, MonthNames[i])==0) {
                month=i+1;
                break;
            }
        }
        Date=MsgGetUTC(day, month, year, hour, min, sec);
        Date-=offset*36;
    }

    Size = atol(ptr+1);

    ReplyInt=sprintf(Answer, "+ Ready for more data\r\n");
    SendClient(client, Answer, ReplyInt);
    FlushClient(client);
    ReplyInt=snprintf(Answer, sizeof(Answer), "STOR %s D%lu %lu %lu\r\n", mbox, Date, Flags, Size);
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt!=2002) {
        MemFree(mbox);
        SendClient(client, client->Ident, strlen(client->Ident));
        SendClient(client, " NO [TRYCREATE] ", 16);
        SendClient(client, client->Command, strlen(client->Command));
        ReplyInt=sprintf(Answer, " NMAP error:%d\r\n", ReplyInt);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    while (Size > 0) {
        if (client->BufferPtr > 0) {
            if (Size < client->BufferPtr) {
                SendNMAPServer(client, client->Buffer, Size);
                client->BufferPtr -= Size;
                memmove(client->Buffer, client->Buffer + Size, client->BufferPtr);
                client->Buffer[client->BufferPtr]='\0';
                Size = 0;
            } else {
                SendNMAPServer(client, client->Buffer, client->BufferPtr);
                Size -= client->BufferPtr;
                client->BufferPtr=0;
                client->Buffer[0]='\0';
            }
        } else {
            if (Exiting) {
                MemFree(mbox);
                return(CLIENT_TERMINATE);
            }
            count=DoClientRead(client, client->Buffer+client->BufferPtr, BUFSIZE-client->BufferPtr, 0);
            if ((count<1) || (Exiting)) {
                MemFree(mbox);
                return(CLIENT_TERMINATE);
            }
            if (Size > count) {
                SendNMAPServer(client, client->Buffer, count);
                Size -= count;
            } else {
                SendNMAPServer(client, client->Buffer, Size);
                memmove(client->Buffer, client->Buffer + Size, count - Size);
                client->BufferPtr = count - Size;
                client->Buffer[client->BufferPtr]='\0';
                Size = 0;
            }
        }
    }

    if (ReadCommandLine(client)) {
        ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
        if (ReplyInt!=1000) {
            SendClient(client, client->Ident, strlen(client->Ident));
            SendClient(client, " NO ", 4);
            SendClient(client, client->Command, strlen(client->Command));
            ReplyInt=sprintf(Answer, " NMAP error:%d\r\n", ReplyInt);
            SendClient(client, Answer, ReplyInt);
        } else {
            /* Check Flags */
            if (XplStrCaseCmp(client->MBox, mbox)==0) {
                client->NMAPAsyncEvent |= EVENT_MBOX_NEW_MESSAGES;
            }

            if (client->State == STATE_SELECTED) {
                if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
                    ;
                } else {
                    return(CLIENT_TERMINATE);
                }
            }

            ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK APPEND completed:\r\n",client->Ident);
            SendClient(client, Answer, ReplyInt);
        }
        MemFree(mbox);
        return(CLIENT_CONTINUE);
    }

    return(CLIENT_TERMINATE);
}

/********** IMAP client commands - selected state **********/

int
ImapCommandCheck(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK CHECK completed\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandClose(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];
    unsigned char Reply[1024];

    if (client->State == STATE_SELECTED) {
        if (!client->ReadOnly) {
            SendNMAPServer(client, "PURG\r\n", 6);
            ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
        }

        SendNMAPServer(client, "RSET MBOX\r\n", 11);
        ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
       
        /* Any last flags should get picked up in the GetNMAPAnswer() */
        SendUnseenFlags(client, Answer, ReplyInt);

        /* Any events are for the old mailbox, so forget about them. */
        client->NMAPAsyncEvent = 0;

        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK CLOSE completed\r\n",client->Ident);
        SendClient(client, Answer, ReplyInt);
        client->State=STATE_AUTH;
        client->MBox[0]='\0';
 
        return(CLIENT_CONTINUE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Wrong state\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandExpunge(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    int SoFar=0;

    if (client->State == STATE_SELECTED) {
        if (!(client->ReadOnly)) {
            SendNMAPServer(client, "PRGV\r\n", 6);
            ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);

            while (ReplyInt!=1000) {
                ReplyInt=sprintf(Answer, "* %lu EXPUNGE\r\n",NMAPtoNUM(client, atol(Reply))-SoFar);
                SendClient(client, Answer, ReplyInt);
                SoFar++;
                ReplyInt=GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
            }

            ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK Messages removed\r\n",client->Ident);
            SendClient(client, Answer, ReplyInt);
            LoadUIDList(client);
            return(CLIENT_CONTINUE);
        }

        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Read-only Mailbox! Your browser is not RFC compliant since it ignores the mailbox state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);

    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Wrong state\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandSearch(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if (SendAsyncNewMessages(client) != CLIENT_TERMINATE) {
            IMAPSearch(client, FALSE);
            return(CLIENT_CONTINUE);
        }

        return(CLIENT_TERMINATE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandUidSearch(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if (SendAsyncNewMessages(client) != CLIENT_TERMINATE) {
            IMAPSearch(client, TRUE);
            return(CLIENT_CONTINUE);
        }

        return(CLIENT_TERMINATE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}


int
ImapCommandStore(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if (SendAsyncNewMessages(client) != CLIENT_TERMINATE) {
            HandleStore(client, FALSE);
            return(CLIENT_CONTINUE);
        }

        return(CLIENT_TERMINATE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandUidStore(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if (SendAsyncNewMessages(client) != CLIENT_TERMINATE) {
            ptr = client->Command + 4;
            memmove(client->Command, ptr, strlen(ptr) + 1);
            HandleStore(client,TRUE);
            return(CLIENT_CONTINUE);
        }

        return(CLIENT_TERMINATE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandCopy(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            HandleCopy(client,FALSE);
            return(CLIENT_CONTINUE);
        }

        return(CLIENT_TERMINATE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandUidCopy(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ptr = client->Command + 4;
            memmove(client->Command, ptr, strlen(ptr) + 1);
            HandleCopy(client,TRUE);
            return(CLIENT_CONTINUE);
        }

        return(CLIENT_TERMINATE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n",client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

/********** IMAP ACL commands rfc2086 **********/

int
ImapCommandSetAcl(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned long permissions;
    unsigned long newPermissions;
    unsigned long action;
    unsigned char *ptr2;
    unsigned char *mbox;
    unsigned char *mailbox;
    unsigned char *identifier;
    BOOL    ready;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) { 
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr = client->Command + 6;
    if (isspace(*ptr)) {
        do { ptr++; } while (isspace(*ptr));
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    /* Grab the mailbox name. */
    if ((ptr = GrabArgument(client, ptr, &mbox)) != NULL) {
        /* Grab the authentication identifier */
        if ((ptr = GrabArgument(client, ptr, &identifier)) != NULL) {
            /* Advance as needed. */
            if (isspace(*ptr)) {
                do { ptr++; } while (isspace(*ptr));
            }
        } else {
            MemFree(mbox);

            ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
            SendClient(client, Answer, ReplyInt);
            return(CLIENT_CONTINUE);
        }
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    switch (*ptr) {
    case '\0': { action = 0;         break; }
    case '-': {  action = -1; ptr++;     break; }
    case '+': {  action = 1;  ptr++;     break; }
    default: {  action = 0;         break; }
    }

    /* Advance as needed. */
    if (isspace(*ptr)) {
        do { ptr++; } while (isspace(*ptr));
    }

    ready = TRUE;
    newPermissions = 0;
    while ((ready == TRUE) && (*ptr != '\0')) {
        switch(toupper(*ptr)) {
        case 'A': { /* Administer (perform SETACL) */
            newPermissions |= NMAP_SHARE_ADMINISTER; break; }

        case 'C': { /* Create (CREATE new sub-mailboxes in any implementation-defined hierarchy. */
            newPermissions |= NMAP_SHARE_CREATE;  break; }

        case 'D': { /* Delete (STORE DELETED flag, perform EXPUNGE). */
            newPermissions |= NMAP_SHARE_DELETE;  break; }

        case 'I': { /* Insert (perform APPEND, COPY into mailbox). */
            newPermissions |= NMAP_SHARE_INSERT;  break; }

        case 'L': { /* Lookup (mailbox is visible to LIST/LSUB commands). */
            newPermissions |= NMAP_SHARE_VISIBLE;  break; }

        case 'P': { /* Post (send mail to submission address for mailbox, not enforced by IMAP4 itself. */
            newPermissions |= NMAP_SHARE_POST;   break; }

        case 'R': { /* Read (SELECT the mailbox, perform CHECK, FETCH, PARTIAL, SEARCH, COPY from mailbox. */
            newPermissions |= NMAP_SHARE_READ;   break; }

        case 'S': { /* Keep seen/unseen information across sessions (STORE SEEN flag) */
            newPermissions |= NMAP_SHARE_SEEN;   break; }

        case 'W': { /* Write (STORE flags other than SEEN and DELETED). */
            newPermissions |= NMAP_SHARE_WRITE;   break; }

        default: {
            MemFree(mbox);
            MemFree(identifier);

            ready = FALSE;
            break;
        }
        }

        do { ptr++; } while(isspace(*ptr));
    }

    if (ready == TRUE) {
        mailbox = mbox;
        if ((*mailbox == '/') || (*mailbox == '\\')) {
            mailbox++;
        }

        REMOVE_MBOX_SPACES(mailbox);
        CHECK_PATH_LEN(mailbox);
    } else {
        /* Invalid permissions! */
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    if (action != 0) {
        permissions = 0;

        /* Load the NMAP permissions for this resource. */
        SendNMAPServer(client, "SHARE MBOX\r\n", 12);
        ReplyInt = GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE);

        if (ReplyInt == 2002) {
            ready = FALSE;
            while ((ReplyInt = GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE)) == 2002) {
                if (ready == FALSE) {
                    ptr = strchr(Answer, ' ');
                    if (ptr != NULL) {
                        *(ptr++) = '\0';

                        if (XplStrCaseCmp(mailbox, Answer) != 0) {
                            continue;
                        }
                    } else {
                        continue;
                    }

                    ptr2 = strchr(ptr, ' ');
                    if (ptr2 != NULL) {
                        *(ptr2++) = '\0';           

                        if (XplStrCaseCmp(identifier, ptr) != 0) {
                            continue;
                        }
                    } else {
                        continue;
                    }

                    permissions = atol(ptr2);
                    ready = TRUE;

                    /* Keep looping to absorb the remaining NMAP Server responses. */
                }
            }
        } else {
            MemFree(mbox);
            MemFree(identifier);

            ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error: %d\r\n", client->Ident, ReplyInt);
            SendClient(client, Answer, ReplyInt);
            return(CLIENT_CONTINUE);
        }

        if (action == 1) {
            ReplyInt = snprintf(Answer, sizeof(Answer), "SHARE MBOX %s %s %lu%s\r\n", mailbox, identifier, permissions | newPermissions, (ready == TRUE)? " NOMESSAGE": "");
        } else {
            ReplyInt = snprintf(Answer, sizeof(Answer), "SHARE MBOX %s %s %lu%s\r\n", mailbox, identifier, permissions & ~newPermissions, (ready == TRUE)? " NOMESSAGE": "");
        }
    } else {
        /* Override existing permissions. */
        ReplyInt = snprintf(Answer, sizeof(Answer), "SHARE MBOX %s %s %lu NOMESSAGE\r\n", mailbox, identifier, newPermissions);
    }

    MemFree(mbox);
    MemFree(identifier);

    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt = GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE);
    if (ReplyInt == 1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK SETACL completed\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error: %d\r\n", client->Ident, ReplyInt);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandDeleteAcl(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char *mbox;
    unsigned char *mailbox;
    unsigned char *identifier;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ptr = client->Command + 9;
    if (isspace(*ptr)) {
        do { ptr++; } while (isspace(*ptr));
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    /* Grab the mailbox name. */
    if ((ptr = GrabArgument(client, ptr, &mbox)) != NULL) {
        /* Grab the authentication identifier */
        if ((ptr = GrabArgument(client, ptr, &identifier)) != NULL) {
            mailbox = mbox;
            if ((*mailbox == '/') || (*mailbox == '\\')) {
                mailbox++;
            }

            REMOVE_MBOX_SPACES(mailbox);
            CHECK_PATH_LEN(mailbox);
        } else {
            MemFree(mbox);

            ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
            SendClient(client, Answer, ReplyInt);
            return(CLIENT_CONTINUE);
        }
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "SHARE MBOX %s %s 0 NOMESSAGE\r\n", mailbox, identifier);

    MemFree(mbox);
    MemFree(identifier);

    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt = GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE);
    if (ReplyInt == 1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK DELETEACL completed\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error: %d\r\n", client->Ident, ReplyInt);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandGetAcl(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned long permissions;
    unsigned char *ptr2;
    unsigned char *mbox;
    unsigned char *mailbox;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident));
        return(CLIENT_CONTINUE);
    }

    ptr = client->Command + 6;
    if (isspace(*ptr)) {
        do { ptr++; } while (isspace(*ptr));
    } else {
        SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident));
        return(CLIENT_CONTINUE);
    }

    /* Grab the mailbox name. */
    if ((ptr = GrabArgument(client, ptr, &mbox)) != NULL) {
        mailbox = mbox;
        if ((*mailbox == '/') || (*mailbox == '\\')) {
            mailbox++;
        }

        ptr2 = Reply + snprintf(Reply, sizeof(Reply), "* ACL \"%s\"", mailbox);

        REMOVE_MBOX_SPACES(mailbox);
        CHECK_PATH_LEN(mailbox);
    } else {
        SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident));
        return(CLIENT_CONTINUE);
    }

    SendNMAPServer(client, Answer, snprintf(Answer, sizeof(Answer), "RIGHTS %s\r\n", mailbox));
    MemFree(mbox);

    ReplyInt = GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE);
    if (ReplyInt == 2002) {
        while ((ReplyInt = GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE)) == 2002) {
            ptr = strchr(Answer, ' ');
            if (ptr != NULL) {
                *(ptr++) = '\0';

                permissions = atol(ptr);

                /* Add the grantee name to the response. */
                ptr2 += snprintf(ptr2, sizeof(Reply) - (ptr2 - Reply), " \"%s\" ", Answer);
                if (permissions) {
                    if (permissions & NMAP_SHARE_VISIBLE) {
                        *(ptr2++) = 'l';
                    }

                    if (permissions & NMAP_SHARE_READ) {
                        *(ptr2++) = 'r';
                    }

                    if (permissions & NMAP_SHARE_SEEN) {
                        *(ptr2++) = 's';
                    }

                    if (permissions & NMAP_SHARE_WRITE) {
                        *(ptr2++) = 'w';
                    }

                    if (permissions & NMAP_SHARE_INSERT) {
                        *(ptr2++) = 'i';
                    }

                    if (permissions & NMAP_SHARE_POST) {
                        *(ptr2++) = 'p';
                    }

                    if (permissions & NMAP_SHARE_CREATE) {
                        *(ptr2++) = 'c';
                    }

                    if (permissions & NMAP_SHARE_DELETE) {
                        *(ptr2++) = 'd';
                    }

                    if (permissions & NMAP_SHARE_ADMINISTER) {
                        *(ptr2++) = 'a';
                    }

                    continue;
                } else {
                    *(ptr2++) = '"';
                    *(ptr2++) = '"';
                }
            }

            continue;
        }

        if (ReplyInt == 1000) {
            *(ptr2++) = '\r';
            *(ptr2++) = '\n';
            *(ptr2) = '\0';

            SendClient(client, Reply, ptr2 - Reply);
            SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s OK GETACL completed\r\n", client->Ident));
            return(CLIENT_CONTINUE);
        }
    }

    SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s NO NMAP response error: %d\r\n", client->Ident, ReplyInt));
    return(CLIENT_CONTINUE);
}

int
ImapCommandListRights(void *param)
{
    ImapClient *client = (ImapClient *)param;
    unsigned char *mbox;
    unsigned char *identifier;
    unsigned char *ptr;
    unsigned char Answer[1024];
    int ReplyInt;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    }

    if (client->State >= STATE_AUTH) {
        ptr = client->Command + 10;
        if (isspace(*ptr)) {
            do { ptr++; } while (isspace(*ptr));
        } else {
            ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
            SendClient(client, Answer, ReplyInt);
            return(CLIENT_CONTINUE);
        }
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    /* Grab the mailbox name. */
    if ((ptr = GrabArgument(client, ptr, &mbox)) != NULL) {
        identifier = NULL;

        /* Grab the authentication identifier */
        if ((ptr = GrabArgument(client, ptr, &identifier)) != NULL) {
            SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "* LISTRIGHTS \"%s\" %s \"\" l r s w i p c d a\r\n", mbox, identifier));
            SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s OK LISTRIGHTS completed\r\n", client->Ident));

            MemFree(mbox);
            MemFree(identifier);
            return(CLIENT_CONTINUE);
        }

        MemFree(mbox);
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandMyRights(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned long permissions;
    unsigned char *ptr2;
    unsigned char *mbox;
    unsigned char *mailbox;
    BOOL found;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident));
        return(CLIENT_CONTINUE);
    }

    ptr = client->Command + 8;
    if (isspace(*ptr)) {
        do { ptr++; } while (isspace(*ptr));
    } else {
        SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident));
        return(CLIENT_CONTINUE);
    }

    /* Grab the mailbox name. */
    if ((ptr = GrabArgument(client, ptr, &mbox)) != NULL) {
        mailbox = mbox;
        if ((*mailbox == '/') || (*mailbox == '\\')) {
            mailbox++;
        }

        ptr2 = Reply + snprintf(Reply, sizeof(Answer), "* MYRIGHTS \"%s\" \"\"", mailbox);

        REMOVE_MBOX_SPACES(mailbox);
        CHECK_PATH_LEN(mailbox);
    } else {
        SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident));
        return(CLIENT_CONTINUE);
    }

    SendNMAPServer(client, Answer, snprintf(Answer, sizeof(Answer), "RIGHTS %s\r\n", mailbox));
    MemFree(mbox);

    ReplyInt = GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE);
    if (ReplyInt == 2002) {
        found = FALSE;

        while ((ReplyInt = GetNMAPAnswer(client, Answer, sizeof(Answer), TRUE)) == 2002) {
            ptr = strchr(Answer, ' ');
            if (ptr != NULL) {
                *(ptr++) = '\0';

                /* We are only interested in this user's permissions.
                   Keep looping to absorb all NMAP responses. */
                if (XplStrCaseCmp(client->User, Answer) == 0) {
                    permissions = atol(ptr);
                    if (permissions) {
                        ptr2 -= 2; /* Back over the default empty permissions. */

                        if (permissions & NMAP_SHARE_VISIBLE) {
                            *(ptr2++) = 'l';
                        }

                        if (permissions & NMAP_SHARE_READ) {
                            *(ptr2++) = 'r';
                        }

                        if (permissions & NMAP_SHARE_SEEN) {
                            *(ptr2++) = 's';
                        }

                        if (permissions & NMAP_SHARE_WRITE) {
                            *(ptr2++) = 'w';
                        }

                        if (permissions & NMAP_SHARE_INSERT) {
                            *(ptr2++) = 'i';
                        }

                        if (permissions & NMAP_SHARE_POST) {
                            *(ptr2++) = 'p';
                        }

                        if (permissions & NMAP_SHARE_CREATE) {
                            *(ptr2++) = 'c';
                        }

                        if (permissions & NMAP_SHARE_DELETE) {
                            *(ptr2++) = 'd';
                        }

                        if (permissions & NMAP_SHARE_ADMINISTER) {
                            *(ptr2++) = 'a';
                        }

                        continue;
                    }
                }

                continue;
            }
        }

        if (ReplyInt == 1000) {
            *(ptr2++) = '\r';
            *(ptr2++) = '\n';
            *(ptr2) = '\0';
            SendClient(client, Reply, ptr2 - Reply);
            SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s OK MYRIGHTS completed\r\n", client->Ident));
            return(CLIENT_CONTINUE);
        }
    }

    SendClient(client, Answer, snprintf(Answer, sizeof(Answer), "%s NO NMAP response error: %d\r\n", client->Ident, ReplyInt));
    return(CLIENT_CONTINUE);
}

/********** IMAP Quota commands rfc2087 **********/

int
ImapCommandSetQuota(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO Only administrator can set quota\r\n", client->Ident);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

int
ImapCommandGetQuota(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned long quota = 0;
    unsigned long used = 0;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ReplyInt=sprintf(Answer, "SPACE\r\n");
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt = GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if(ReplyInt != 1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO couldn't determine quota\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    ptr = strchr(Reply, ' ');
    if(!ptr) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO couldn't determine quota\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
 
    quota = atol(ptr);
    used = atol(Reply);

    if(client->Command[9] == '\"' && client->Command[10] == '\"') {
        ReplyInt=sprintf(Answer, "* QUOTA \"\" (STORAGE %lu %lu)\r\n", used, quota);
        SendClient(client, Answer, ReplyInt);
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK GETQUOTA completed\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO No such quota root\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
    }

    return(CLIENT_CONTINUE);
}

int
ImapCommandGetQuotaRoot(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned long quota = 0;
    unsigned long used = 0;
    unsigned char *mbox;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    ReplyInt=sprintf(Answer, "SPACE\r\n");
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt = GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if(ReplyInt != 1000) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO couldn't determine quota\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    ptr = strchr(Reply, ' ');
    if(!ptr) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO couldn't determine quota\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    quota = atol(ptr);
    used = atol(Reply);


    ptr = client->Command + 13;
    if(*(ptr - 1) != ' ') {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    GrabArgument(client, ptr, &mbox);
    if(!mbox) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    ReplyInt=snprintf(Answer, sizeof(Answer), "* QUOTAROOT %s \"\"\r\n", mbox);
    SendClient(client, Answer, ReplyInt);
    ReplyInt=sprintf(Answer, "* QUOTA \"\" (STORAGE %lu %lu)\r\n", used, quota);
    SendClient(client, Answer, ReplyInt);
    ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK GETQUOTAROOT completed\r\n", client->Ident);
    SendClient(client, Answer, ReplyInt);
    MemFree(mbox);
    return(CLIENT_CONTINUE);
}

/********** IMAP Namespace command rfc2342 **********/

int
ImapCommandNameSpace(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    SendNMAPServer(client, "NAMESPACE\r\n", 11);
    ReplyInt = GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    if (ReplyInt == 1000) {
        unsigned char delim;
        unsigned char *user;
        unsigned char *userShare;
        unsigned char *publicShare;

        /* "/" "" "Shares/" "Public Shares/" */
        /* (("" "/")) (("Shares/" "/")) (("Public Shares/" "/")) */
        ptr = Reply;
        while (isspace(*ptr)) {
            ptr++;
        }

        delim = *(++ptr);
        ptr += 2;

        while (*ptr != '"') {
            ptr++;
        }

        user = ++ptr;
        while (*ptr != '"') {
            ptr++;
        }
        *ptr = '\0';
        ptr += 2;
        while (*ptr != '"') {
            ptr++;
        }

        userShare = ++ptr;
        while (*ptr != '"') {
            ptr++;
        }
        *ptr = '\0';
        ptr += 2;
        while (*ptr != '"') {
            ptr++;
        }

        publicShare = ++ptr;
        while (*ptr != '"') {
            ptr++;
        }
        *ptr = '\0';

        ReplyInt = snprintf(Answer, sizeof(Answer), "* NAMESPACE ((\"%s\" \"%c\")) ((\"%s\" \"%c\")) ((\"%s\" \"%c\"))\r\n", user, delim, userShare, delim, publicShare, delim);
        SendClient(client, Answer, ReplyInt);

        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK NAMESPACE command completed\r\n", client->Ident);
    } else {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP response error: %d\r\n", client->Ident, ReplyInt);
    }

    SendClient(client, Answer, ReplyInt);
    return(CLIENT_CONTINUE);
}

/********** non-standard commands **********/

int
ImapCommandProxyAuth(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    unsigned char  *user = NULL;
    MDBValueStruct  *User;

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    } else if (client->State < STATE_AUTH) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Wrong state\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    /* One of the consequences of switching users the way we do is that proxyauth
     * will only succeed the first time. After that, the user has changed and can
     * no longer proxyauth. This is the way Netscape does it, so supposedly its
     * correct, although inefficient for the client. */
    if(client->State < STATE_AUTH || XplStrCaseCmp(client->User, Postmaster)) {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_UNHANDLED, LOGGER_EVENT_UNHANDLED_REQUEST, LOG_INFO, 0, client->User, client->Command, XplHostToLittle(client->cs.sin_addr.s_addr), 0, NULL, 0);
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Command unrecognized,\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    ptr = client->Command + 10;
    while(isspace(*ptr)) {
        ++ptr;
    }

    if(!*ptr) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }
    ptr = GrabArgument(client, ptr, &user);
    if(!ptr || !user) {
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s BAD Invalid arguments\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        return(CLIENT_CONTINUE);
    }

    /* Fixme - this doesn't connect the user to *his/her* nmap server */

    User = MDBCreateValueStruct(IMAPDirectoryHandle, NULL);
    if (!MsgFindObject(user, NULL, NULL, NULL, User)) { 
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO User not found\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        MDBDestroyValueStruct(User);
        MemFree(user);
        return(CLIENT_CONTINUE);
    }

    if(XplStrCaseCmp(client->User, User->Value[0]) != 0) {
        if (strlen(User->Value[0]) < sizeof(client->User)) {
            strcpy(client->User, User->Value[0]);
        }
    }
    MemFree(user);
    MDBDestroyValueStruct(User);

    client->State = STATE_AUTH;
    ReplyInt = snprintf(Answer, sizeof(Answer), "USER %s\r\n", client->User);
    SendNMAPServer(client, Answer, ReplyInt);
    ReplyInt = GetNMAPAnswer(client, Reply, sizeof(Reply), TRUE);
    switch(ReplyInt) {
    case 1000:
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s OK Logged in as %s\r\n", client->Ident, client->User);
        SendClient(client, Answer, ReplyInt);
        break;
    case 4221:
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO User already connected\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        break;

    case 4224:
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO User does not exist\r\n", client->Ident);
        SendClient(client, Answer, ReplyInt);
        break;

    default:
        ReplyInt = snprintf(Answer, sizeof(Answer), "%s NO NMAP returned: %d\r\n", client->Ident, ReplyInt);
        SendClient(client, Answer, ReplyInt);
        break;
    }

    return(CLIENT_CONTINUE);
}

int
ImapCommandQuit(void *param)
{
    ImapClient *client = (ImapClient *)param;
    int ReplyInt;
    unsigned char Answer[1024];

    if (client->State == STATE_SELECTED) {
        if ((SendAsyncFlags(client) != CLIENT_TERMINATE) && (SendAsyncNewMessages(client) != CLIENT_TERMINATE)) {
            ;
        } else {
            return(CLIENT_TERMINATE);
        }
    }

    ReplyInt = snprintf(Answer, sizeof(Answer), "+OK %s signing off\r\n",Hostname);
    SendClient(client, Answer, ReplyInt);
    return(CLIENT_TERMINATE);
}

static BOOL
HandleConnection(void *param)
{
    long retValue;
    ImapClient  *client=(ImapClient *)param;
    int ReplyInt;
    unsigned char *ptr;
    unsigned char Answer[1024];
    unsigned char Reply[1024];
    ProtocolCommand *localCommand;

    if (NegotiateSSL(client) == CLIENT_CONTINUE) {
        ;
    } else {
        EndClientConnection(client);
    }

    ReplyInt = snprintf(Reply, sizeof(Answer), "* OK %s %s server ready <%ud.%lu@%s>\r\n",Hostname, PRODUCT_NAME, XplGetThreadID(),time(NULL),Hostname);
    SendClient(client, Reply, ReplyInt);
    FlushClient(client);

    while (TRUE) {
        if (ReadCommandLine(client)) {
            /* Grab ident from command */
            ptr = strchr(client->Command,' ');
            if (ptr) {
                if ((unsigned long)(ptr - client->Command) < sizeof(client->Ident)) {
                    ptr[0]='\0';
                    memcpy(client->Ident, client->Command, ptr - client->Command + 1);
                } else {
                    memcpy(client->Ident, client->Command, sizeof(client->Ident));
                    client->Ident[sizeof(client->Ident)-1]='\0';
                }
                memmove(client->Command, ptr+1, strlen(ptr+1)+1);
            } else {
                SendClient(client, "* BAD Missing command\r\n", 23);
                FlushClient(client);
                continue;
            }
        } else {
            SendClient(client, "* BAD command line too long\r\n", 29);
            FlushClient(client);
            continue;
        }

        localCommand = ProtocolCommandTreeSearch(&Imap.commands, client->Command);
        if (localCommand) {
            retValue = localCommand->handler(client);
        } else {
            retValue = ImapCommandUnknown(client);
        }

        if (retValue != CLIENT_TERMINATE) {
            ;
        } else {
            EndClientConnection(client);
        }

        /* Post-Command Processing */
        if (client->State == STATE_SELECTED) {
            SendUnseenFlags(client, Answer, ReplyInt);
        }
        FlushClient(client);
    }
    return(TRUE);
}

#if defined(NETWARE) || defined(LIBC) || defined(WIN32)
static int 
_NonAppCheckUnload(void)
{
    int   s;
    static BOOL checked = FALSE;
    XplThreadID id;

    if (!checked) {
        checked = Exiting = TRUE;

        XplWaitOnLocalSemaphore(IMAPShutdownSemaphore);

        id = XplSetThreadGroupID(TGid);
        if (IMAPServerSocket != -1) {
            s = IMAPServerSocket;
            IMAPServerSocket = -1;

            IPclose(s);
        }
        XplSetThreadGroupID(id);

        XplWaitOnLocalSemaphore(IMAPServerSemaphore);
    }

    return(0);
}
#endif

static void 
IMAPShutdownSigHandler(int sigtype)
{
    int   s;
    static BOOL signaled = FALSE;

    if (!signaled && ((sigtype == SIGTERM) || (sigtype == SIGINT))) {
        signaled = Exiting = TRUE;

        if (IMAPServerSocket != -1) {
            s = IMAPServerSocket;
            IMAPServerSocket = -1;

            IPclose(s);
        }

        XplSignalLocalSemaphore(IMAPShutdownSemaphore);

        /* Allow the main thread to close the final semaphores and exit. */
        XplDelay(1000);
    }

    return;
}

static int
ServerSocketInit(void)
{
    struct sockaddr_in server_sockaddr;
    int ccode;

    IMAPServerSocket = IPsocket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (IMAPServerSocket < 0) {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_CREATE_SOCKET_FAILED, LOG_ERROR, 0, "", NULL, __LINE__, 0, NULL, 0);
        XplConsolePrintf("hulaimap: Could not allocate server socket\n");
        return -1;
    }

    ccode = 1;
    setsockopt(IMAPServerSocket, SOL_SOCKET, SO_REUSEADDR, (unsigned char *)&ccode, sizeof(ccode));
    memset(&server_sockaddr, 0, sizeof(struct sockaddr));
    
    server_sockaddr.sin_family = AF_INET;
    server_sockaddr.sin_port = htons(IMAPServerPort);
    server_sockaddr.sin_addr.s_addr = MsgGetAgentBindIPAddress();
    
    /* Get root privs back for the bind.  It's ok if this fails - 
     * the user might not need to be root to bind to the port */
    XplSetEffectiveUserId(0);

    ccode = IPbind(IMAPServerSocket, (struct sockaddr *)&server_sockaddr, sizeof(server_sockaddr));

    if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_PRIV_FAILURE, LOG_ERROR, 0, MsgGetUnprivilegedUser(), NULL, 0, 0, NULL, 0);
        XplConsolePrintf("hulaimap: Could not drop to unprivileged user '%s'\n", MsgGetUnprivilegedUser());
        return -1;
    }

    if (ccode < 0) {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_BIND_FAILURE, LOG_ERROR, 0, "IMAP ", NULL, IMAPServerPort, 0, NULL, 0);
        XplConsolePrintf("hulaimap: Could not bind to port %hu\n", IMAPServerPort);

        IPclose(IMAPServerSocket);
        return ccode;
    }

    return 0;
}

static int
ServerSocketSSLInit(void)
{
    struct sockaddr_in server_sockaddr;
    int ccode;

    IMAPServerSocketSSL = IPsocket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (IMAPServerSocketSSL < 0) {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_CREATE_SOCKET_FAILED, LOG_ERROR, 0, "", NULL, __LINE__, 0, NULL, 0);
        XplConsolePrintf("hulaimap: Could not allocate ssl server socket\n");
        return IMAPServerSocketSSL;
    }
    
    ccode = 1;
    setsockopt(IMAPServerSocketSSL, SOL_SOCKET, SO_REUSEADDR, (unsigned char *)&ccode, sizeof(ccode));
    memset(&server_sockaddr, 0, sizeof(struct sockaddr));
    
    server_sockaddr.sin_family = AF_INET;
    server_sockaddr.sin_port = htons(IMAPServerPortSSL);
    server_sockaddr.sin_addr.s_addr = MsgGetAgentBindIPAddress();
    
    /* Get root privs back for the bind.  It's ok if this fails - 
     * the user might not need to be root to bind to the port */
    XplSetEffectiveUserId(0);

    ccode = IPbind(IMAPServerSocketSSL, (struct sockaddr *)&server_sockaddr, sizeof(server_sockaddr));

    if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_PRIV_FAILURE, LOG_ERROR, 0, MsgGetUnprivilegedUser(), NULL, 0, 0, NULL, 0);
        XplConsolePrintf("hulaimap: Could not drop to unprivileged user '%s'\n", MsgGetUnprivilegedUser());
        return -1;
    }

    if (ccode < 0) {
        LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_BIND_FAILURE, LOG_ERROR, 0, "IMAP-SSL ", NULL, IMAPServerPortSSL, 0, NULL, 0);
        XplConsolePrintf("hulaimap: Could not bind to ssl port %hu\n", IMAPServerPortSSL);

        IPclose(IMAPServerSocketSSL);
        return ccode;
    }
    
    return 0;
}

void IMAPServer(void *unused)
{
    int      ccode;
    int      arg;
    XplThreadID oldTGID;
    IPSOCKET     ds;
    struct sockaddr_in client_sockaddr;
    ImapClient  *client;
    XplThreadID    id = 0;

    XplRenameThread(XplGetThreadID(), "IMAP Server");

    XplSafeIncrement(IMAPServerThreads);

    ccode = IPlisten(IMAPServerSocket, 4096);
    if (ccode != -1) {
        while (!Exiting) {
            arg = sizeof(client_sockaddr);
            ds = IPaccept(IMAPServerSocket, (struct sockaddr *)&client_sockaddr, &arg);
            if (!Exiting) {
                if (ds != -1) {
                    if (!IMAPReceiverStopped) {
                        if (XplSafeRead(IMAPConnThreads) < (long)IMAPMaxThreadLoad) {
                            client = GetIMAPConnection();
                            if (client) {
                                client->s = ds;
                                client->ClientSktCtx = (void *)ds;
                                client->cs = client_sockaddr;
        
                                /* Set TCP non blocking io */
                                ccode = 1;
                                setsockopt(ds, IPPROTO_TCP, 1, (unsigned char *)&ccode, sizeof(ccode));
        
                                XplBeginCountedThread(&id, HandleConnection, IMAP_STACK_SIZE, client, ccode, IMAPConnThreads);
                                if (ccode == 0) {
                                    continue;
                                }
        
                                ReturnIMAPConnection(client);
                            }
    
                            IPsend(ds, "* BYE IMAP server out of memory\r\n", 33, 0);
                        } else {
                            IPsend(ds, "* BYE IMAP connection limit for this server reached\r\n", 53, 0);
                        }
                    } else {
                        IPsend(ds, "* BYE IMAP access to this server currently disabled\r\n", 53, 0);
                    }
   
                    IPclose(ds);
                    continue;
                }
      
                switch (errno) {
                case ECONNABORTED:
#ifdef EPROTO
                case EPROTO: 
#endif
                case EINTR: {
                    LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, NULL, NULL, errno, 0, NULL, 0);
                    continue;
                }

                case EIO: {
                    LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, NULL, NULL, errno, 2, NULL, 0);
                    XplConsolePrintf("IMAPD: Server exiting after an accept() failure with an errno: %d\n", arg);

                    break;
                }
                }

                break;
            }

            /* Shutdown signaled */
            break;
        }
    }

/* Shutting down */
    Exiting = TRUE;

    oldTGID = XplSetThreadGroupID(TGid);

    if (IMAPServerSocket != -1) {
        IPclose(IMAPServerSocket);
        IMAPServerSocket = -1;
    }

    if(IMAPServerSocketSSL != -1) {
        IPclose(IMAPServerSocketSSL);
        IMAPServerSocketSSL = -1;
    }

    /* Management Client Shutdown */
    if (ManagementState() == MANAGEMENT_RUNNING) {
        ManagementShutdown();
    }

    for (arg = 0; (ManagementState() != MANAGEMENT_STOPPED) && (arg < 60); arg++) {
        XplDelay(1000);
    }

    /* Wake up the children and set them free! */
    /* fixme - SocketShutdown; */

    /* Wait for the our siblings to leave quietly. */
    for (arg = 0; (XplSafeRead(IMAPServerThreads) > 1) && (arg < 60); arg++) {
        XplDelay(1000);
    }

    if (XplSafeRead(IMAPServerThreads) > 1) {
        XplConsolePrintf("\rIMAPD: %d server threads outstanding; attempting forceful unload.\r\n", XplSafeRead(IMAPServerThreads) - 1);
    }

    XplConsolePrintf("\rIMAPD: Shutting down %d client threads\r\n", XplSafeRead(IMAPConnThreads));

    /* Make sure the kids have flown the coop. */
    for (arg = 0; XplSafeRead(IMAPConnThreads) && (arg < 3 * 60); arg++) {
        XplDelay(1000);
    }

    if (XplSafeRead(IMAPConnThreads)) {
        XplConsolePrintf("IMAPD: %d threads outstanding; attempting forceful unload.\r\n", XplSafeRead(IMAPConnThreads));
    }

    /* Cleanup SSL */
    XplConsolePrintf("IMAPD: Removing SSL data\r\n");
    if (SSLContext) {
        SSL_CTX_free(SSLContext);
    }
    ERR_free_strings();
    ERR_remove_state(0);
    EVP_cleanup();
    XPLCryptoLockDestroy();

    LoggerClose(LogHandle);
    LogHandle = NULL;

    MsgShutdown();
// MDBShutdown();

    ConnShutdown();
        
    FetchCleanup();
    MemPrivatePoolFree(IMAPConnectionPool);
    MemoryManagerClose(MSGSRV_AGENT_IMAP);

    XplConsolePrintf("IMAPD: Shutdown complete\r\n");

    XplSignalLocalSemaphore(IMAPServerSemaphore);
    XplWaitOnLocalSemaphore(IMAPShutdownSemaphore);

    XplCloseLocalSemaphore(IMAPShutdownSemaphore);
    XplCloseLocalSemaphore(IMAPServerSemaphore);

    XplSetThreadGroupID(oldTGID);

    return;
}

void IMAPSSLServer(void *unused)
{
    int      ccode;
    int      arg;
    unsigned char   *message;
    IPSOCKET     ds;
    struct sockaddr_in client_sockaddr;
    ImapClient  *client;
    SSL      *cSSL = NULL;
    XplThreadID    id = 0;
    
    XplRenameThread(XplGetThreadID(), "IMAP SSL Server");

    ccode = IPlisten(IMAPServerSocketSSL, 2048);
    if (ccode != -1) {
        while (!Exiting) {
            arg = sizeof(client_sockaddr);
            ds = IPaccept(IMAPServerSocketSSL, (struct sockaddr *)&client_sockaddr, &arg);
            if (!Exiting) {
                if (ds != -1) {
                    if (!IMAPReceiverStopped) {
                        if (XplSafeRead(IMAPConnThreads) < (long)IMAPMaxThreadLoad) {
                            client = GetIMAPConnection();
                            if (client) {
                                client->s = ds;
                                client->cs = client_sockaddr;

                                client->CSSL = SSL_new(SSLContext);
                                if (client->CSSL) {
                                    ccode = SSL_set_bsdfd(client->CSSL, client->s);
                                    if (ccode == 1) {
                                        /* Set TCP non blocking io */
                                        ccode = 1;
                                        setsockopt(ds, IPPROTO_TCP, 1, (unsigned char *)&ccode, sizeof(ccode));

                                        client->ClientSSL = TRUE;
                                        client->ClientWrite = FuncTbl.writeSSL;
                                        client->ClientRead = FuncTbl.readSSL;
                                        client->ClientSktCtx = client->CSSL;

                                        XplBeginCountedThread(&id, HandleConnection, IMAP_STACK_SIZE, client, ccode, IMAPConnThreads);
                                        if (ccode == 0) {
                                            continue;
                                        }

                                        if (SSL_accept(client->CSSL) == 1) {
                                            XplIPWriteSSL(client->CSSL, "* BYE server error\r\n", 20);
                                        }
                                    }

                                    SSL_free(client->CSSL);
                                    client->CSSL = NULL;
                                }
                                /* If we get here, we have already sent an error or SSL is not functional and we won't be able to */
                                ReturnIMAPConnection(client);
                                IPclose(ds);
                                continue;
                            }
                            message = "* BYE IMAP server out of memory\r\n";
                            arg = 33;
                        } else {
                            message = "* BYE IMAP connection limit for this server reached\r\n";
                            arg = 53;
                        }
                    } else {
                        message = "* BYE IMAP access to this server currently disabled\r\n";
                        arg = 53;
                    }

                    cSSL = SSL_new(SSLContext);
                    if (cSSL) {
                        ccode = SSL_set_bsdfd(cSSL, ds);
                        if (ccode == 1) {
                            if (SSL_accept(cSSL) == 1) {
                                XplIPWriteSSL(cSSL, message, arg);
                            }
                        }
                        SSL_free(cSSL);
                        cSSL = NULL;
                    }

                    IPclose(ds);
                    continue;
                }

                switch (errno) {
                case ECONNABORTED:
#ifdef EPROTO
                case EPROTO:
#endif
                case EINTR: {
                    LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "SSL Server", NULL, errno, 1, NULL, 0);
                    continue;
                }

                case EIO: {
                    Exiting = TRUE;
                    XplConsolePrintf("IMAPD: SSL Server exiting after an accept() failure with an errno: %d\n", errno);
                    LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "SSL Server", NULL, errno, 3, NULL, 0);

                    break;
                }

                default: {
                    XplConsolePrintf("IMAPD: SSL Server exiting after an accept() failure with an errno: %d\n", errno);
                    LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ALERT, 0, "SSL Server", NULL, errno, 1, NULL, 0);

                    break;
                }
                }

                break;
            }

            /* Shutdown signaled */
            break;
        }
    }

    XplConsolePrintf("IMAPD: Server SSL listener done.\r\n");
 
    XplSafeDecrement(IMAPServerThreads);
 
    if (!Exiting) {
        raise(SIGTERM);
    }
 
    return;
}

static BOOL
ReadConfiguration(void)
{
    MDBValueStruct *Config;
    unsigned long i;
    unsigned char *ptr;
    time_t   gm_time_of_day, time_of_day;
    struct tm  gm, ltm;

    tzset();
    time(&time_of_day);
    time(&gm_time_of_day);
    gmtime_r(&gm_time_of_day, &gm);
    localtime_r(&time_of_day, &ltm);
    gm_time_of_day=mktime(&gm);
    UTCOffset=((signed int)(time_of_day-gm_time_of_day))/36;

    Config = MDBCreateValueStruct(IMAPDirectoryHandle, MsgGetServerDN(NULL));

    if (MDBRead(MSGSRV_AGENT_IMAP, MSGSRV_A_CONFIGURATION, Config)>0) {
        for (i=0; i<Config->Used; i++) {
            if (XplStrNCaseCmp(Config->Value[i], "MaxLoad:", 8)==0) {
                IMAPMaxThreadLoad=atol(Config->Value[i]+8);
            } else if (XplStrNCaseCmp(Config->Value[i], "ACLEnabled:", 11) == 0) {
                if (atol(Config->Value[i] + 11)) {
                    IMAPACLEnabled = TRUE;
                } else {
                    IMAPACLEnabled = FALSE;
                }
            } else if (XplStrNCaseCmp(Config->Value[i], "Port=", 5) == 0) {
                IMAPServerPort = (unsigned short)atol(Config->Value[i] +5);
            } else if (XplStrNCaseCmp(Config->Value[i], "SecurePort=", 5) == 0) {
                IMAPServerPortSSL = (unsigned short)atol(Config->Value[i] +11);
            }
        }
    }
    MDBFreeValues(Config);

    if (MDBRead(MSGSRV_SELECTED_CONTEXT, MSGSRV_A_SSL_OPTIONS, Config)) {
        ClientSSLOptions=atoi(Config->Value[0]);
        ClientSSLOptions &= ~SSL_USE_CLIENT_CERT;
    }
    MDBFreeValues(Config);

    if (MDBRead(MSGSRV_SELECTED_CONTEXT, MSGSRV_A_URL, Config)) {
        if (strlen(Config->Value[0]) < sizeof(ManagementURL))
            strcpy(ManagementURL, Config->Value[0]);
    }
    MDBFreeValues(Config);

    if (MDBRead(MSGSRV_SELECTED_CONTEXT, MSGSRV_A_OFFICIAL_NAME, Config)) {
        if ((Config->Value[0]) && (strlen(Config->Value[0]) < sizeof(Hostname))) {
      
            strcpy(Hostname, Config->Value[0]);
        } else {
            gethostname(Hostname, sizeof(Hostname));
        }
  
    } else {
        gethostname(Hostname, sizeof(Hostname));
    }
    HostnameLen = strlen(Hostname);
    MDBFreeValues(Config);

    if (MDBRead(MSGSRV_SELECTED_CONTEXT, MSGSRV_A_POSTMASTER, Config)) {
        ptr=strchr(Config->Value[0], '.');
        if (ptr) {
            *ptr='\0';
        }
        if (strlen(Config->Value[0]) < sizeof(Postmaster)) {
            strcpy(Postmaster, Config->Value[0]);
        }
    }
    MDBFreeValues(Config);

    MDBSetValueStructContext(NULL, Config);
    if (MDBRead(MSGSRV_ROOT, MSGSRV_A_ACL, Config)>0) { 
        HashCredential(MsgGetServerDN(NULL), Config->Value[0], NMAPHash);
    }

    MDBDestroyValueStruct(Config);

    return(TRUE);
}

XplServiceCode(IMAPShutdownSigHandler)

int
XplServiceMain(int argc, char *argv[])
{
    int   ccode;
    XplThreadID id=0;

    if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
        XplConsolePrintf("hulaimap: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
        return 1;
    }

    XplSignalHandler(IMAPShutdownSigHandler);

    XplSafeWrite(IMAPServerThreads, 0);
    XplSafeWrite(IMAPConnThreads, 0);
    XplSafeWrite(IMAPIdleConnThreads, 0);
    XplSafeWrite(IMAPStats.Clients.Serviced, 0);
    XplSafeWrite(IMAPStats.WrongPassword, 0);

    TGid = XplGetThreadGroupID();

    if (MemoryManagerOpen(MSGSRV_AGENT_IMAP) == TRUE) {
        IMAPConnectionPool = MemPrivatePoolAlloc("IMAP Connections", sizeof(ImapClient), 0, 3072, TRUE, FALSE, IMAPConnectionAllocCB, IMAPConnectionFreeCB, NULL);
        if (IMAPConnectionPool != NULL) {
            XplOpenLocalSemaphore(IMAPServerSemaphore, 0);
            XplOpenLocalSemaphore(IMAPShutdownSemaphore, 1);
        } else {
            MemoryManagerClose(MSGSRV_AGENT_IMAP);

            XplConsolePrintf("IMAPD: Unable to create command buffer pool; shutting down.\r\n");
            return(-1);
        }
    } else {
        XplConsolePrintf("IMAPD: Unable to initialize memory manager; shutting down.\r\n");
        return(-1);
    }
        
    if (!FetchInit()) {
        XplBell();
        XplConsolePrintf("\rIMAPD: Failed to initialize fetch flag index\n");
        MemoryManagerClose(MSGSRV_AGENT_IMAP);
        return(-1);
    }

    LoadProtocolCommandTree(&Imap.commands, ImapProtocolCommands);

    ConnStartup(CONNECTION_TIMEOUT, TRUE);

    MDBInit();
    IMAPDirectoryHandle = (MDBHandle)MsgInit();
    if (IMAPDirectoryHandle == NULL) {
        XplBell();
        XplConsolePrintf("\rIMAPD: Invalid directory credentials; exiting!\n");
        XplBell();

        MemoryManagerClose(MSGSRV_AGENT_IMAP);

        return(-1);
    }

    CMInitialize(IMAPDirectoryHandle, "IMAP");
    InitSearchModule();

    LogHandle = LoggerOpen("hulaimap");
    if (LogHandle != NULL) {
        ;
    } else {
        XplConsolePrintf("IMAPD: Unable to initialize Nsure Audit.  Logging disabled.\r\n");
    }

    Capability.len = sprintf(Capability.message, "%s\r\n", "* CAPABILITY IMAP4 IMAP4rev1 ACL AUTH=LOGIN NAMESPACE XSENDER");

    ReadConfiguration();

    if (ServerSocketInit() < 0) {
        XplConsolePrintf("hulaimap: Exiting\n");
        return -1;
    }

    if (ClientSSLOptions) {
        Capability.SSL.len = sprintf(Capability.SSL.message, "%s\r\n", "* CAPABILITY IMAP4 IMAP4rev1 ACL AUTH=LOGIN NAMESPACE STARTTLS XSENDER LOGINDISABLED");

        SSL_load_error_strings();
        SSL_library_init();
        XPLCryptoLockInit();
        SSLContext=SSL_CTX_new(SSLv23_server_method());
        SSL_CTX_set_mode(SSLContext, SSL_MODE_AUTO_RETRY);
        /* SSL_CTX_set_options(SSLContext, SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS); */

        if (SSLContext) {
            int result;

            if (ClientSSLOptions & SSL_DONT_INSERT_EMPTY_FRAGMENTS) {
                SSL_CTX_set_options(SSLContext, SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS);
            }

            if (ClientSSLOptions & SSL_ALLOW_CHAIN) {
                result = SSL_CTX_use_certificate_chain_file(SSLContext, MsgGetTLSCertPath(NULL));
            } else {
                result = SSL_CTX_use_certificate_file(SSLContext, MsgGetTLSCertPath(NULL), SSL_FILETYPE_PEM);
            }

            if (result>0) {
                if (SSL_CTX_use_PrivateKey_file(SSLContext, MsgGetTLSKeyPath(NULL), SSL_FILETYPE_PEM)>0) {
                    if (SSL_CTX_check_private_key(SSLContext)) {
                        if (ServerSocketSSLInit() >= 0) {
                            /* Done binding to ports, drop privs permanently */
                            if (XplSetRealUser(MsgGetUnprivilegedUser()) < 0) {
                                XplConsolePrintf("hulaimap: Could not drop to unprivileged user '%s', exiting\n", MsgGetUnprivilegedUser());
                                return 1;
                            }
                            XplBeginCountedThread(&id, IMAPSSLServer, IMAP_STACK_SIZE, NULL, ccode, IMAPServerThreads);
                        } else {
                            ClientSSLOptions = 0;
                        }
         
                    } else {
                        XplConsolePrintf("\rIMAPD: PrivateKey check failed\n");
                        ClientSSLOptions=0;
                    }
                } else {
                    XplConsolePrintf("\rIMAPD: Could not load private key\n");
                    ClientSSLOptions=0;
                }
            } else {
                XplConsolePrintf("\rIMAPD: Could not load public key\n");
                ClientSSLOptions=0;
            }
        } else {
            XplConsolePrintf("\rIMAPD: Could not generate SSL context\n");
            ClientSSLOptions=0;
        }
    }
    /* Done binding to ports, drop privs permanently */
    if (XplSetRealUser(MsgGetUnprivilegedUser()) < 0) {
        XplConsolePrintf("hulaimap: Could not drop to unprivileged user '%s', exiting\n", MsgGetUnprivilegedUser());
        return 1;
    }

    /* Management Client Startup */
    if ((ManagementInit(MSGSRV_AGENT_IMAP, IMAPDirectoryHandle)) 
        && (ManagementSetVariables(IMAPManagementVariables, sizeof(IMAPManagementVariables) / sizeof(ManagementVariables))) 
        && (ManagementSetCommands(IMAPManagementCommands, sizeof(IMAPManagementCommands) / sizeof(ManagementCommands)))) {
        XplBeginThread(&id, ManagementServer, DMC_MANAGEMENT_STACKSIZE, NULL, ccode);
    }


    XplStartMainThread(PRODUCT_SHORT_NAME, &id, IMAPServer, 8192, NULL, ccode);

    XplUnloadApp(XplGetThreadID());
    return(0);
}
