/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/
#include <config.h>
#include <stdlib.h>
#include <xpl.h>

#include <hulautil.h>
#include <mwtempl.h>
#include <webadmin.h>
#include <hulautil.h>
#include <wacert.tok>
#include <wacert.ary>

#include <openssl/ssl.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

WAAPIDefine;

/* General stuff */
struct {
    BOOL unload;

    struct {
        XplThreadID main;
        XplThreadID group;
    } id;

    struct {
        XplSemaphore shutdown;
    } sem;
} WACert;

typedef struct {
    unsigned char requestor[256];
    unsigned char eMail[256];
    unsigned char commonName[256];
    unsigned char organization[256];
    unsigned char department[256];
    unsigned char city[256];
    unsigned char state[256];
    unsigned char country[256];

    unsigned char *privKey;
    unsigned char *certificate;
    unsigned char *csr;
} WACertSessionStruct;

/* Requests */
#define REQUEST_NEW_CERT            16000
#define REQUEST_NEW_SS_CERT         16001
#define REQUEST_INSTALL_ROOT_CERT   16002

#define REQUEST_PRIV_KEY            16020
#define REQUEST_CERTIFICATE         16021
#define REQUEST_CSR                 16022

#define TEMP_DIR XPL_DEFAULT_WORK_DIR

/*
    Prototypes for registration functions
*/
BOOL WACertInitSession(SessionStruct *Session, void **ModuleData);
BOOL WACertDestroySession(SessionStruct *Session, void *ModuleData);
BOOL WACertHandleTemplate(ConnectionStruct *Client, SessionStruct *Session, unsigned long Page, TokenOverlayStruct *Token, unsigned long *GotoToken, void *ModuleData);
BOOL WACertHandleURL(ConnectionStruct *Client, SessionStruct *Session, URLStruct *URL, void *ModuleData);

static void
PrintOpenSSLError(void)
{
    BIO *bio_err;
    bio_err=BIO_new(BIO_s_file());
    BIO_set_fp(bio_err, stdout, BIO_NOCLOSE|BIO_FP_TEXT);
    ERR_print_errors(bio_err);
    BIO_free(bio_err);
}

static void
GenerateCertificateCallback(int p, int n, void *arg)
{
#if 0
    char    c='B';

    switch(p) {
        case 0: c='.'; break;
        case 1: c='+'; break;
        case 2: c='*'; break;
        case 3: c='\n'; break;
    }
    fputc(c, stderr);
#endif

    XplDelay(55);

    return;
}

static BOOL
AddV3Extension(X509 *Certificate, int nid, char *Value)
{
    X509_EXTENSION        *Extension;
    X509V3_CTX            Context;

//    X509V3_set_ctx_nodb(&Context);

    /* The issuer and subject cert is the same since we're self-signed */
    X509V3_set_ctx(&Context, Certificate, Certificate, NULL, NULL, 0);
    Extension = X509V3_EXT_conf_nid(NULL, &Context, nid, Value);
    if (!Extension) {
        return(FALSE);
    }

    X509_add_ext(Certificate, Extension, -1);
    X509_EXTENSION_free(Extension);
    return(TRUE);
}

static BOOL
AddV3ReqExtension(STACK_OF(X509_REQUEST) *x509Object, int nid, char *Value)
{
    X509_EXTENSION        *Extension;
//    X509V3_CTX            Context;

//    X509V3_set_ctx_nodb(&Context);

    /* The issuer and subject cert is the same since we're self-signed */
//    X509V3_set_ctx(&Context, Request, Request, NULL, NULL, 0);
    Extension = X509V3_EXT_conf_nid(NULL, NULL, nid, Value);
    if (!Extension) {
        return(FALSE);
    }

    sk_X509_EXTENSION_push(x509Object, Extension);

    return(TRUE);
}

static BOOL
GenerateCACertificate(WACertSessionStruct *CertSession, X509 **Certificate, EVP_PKEY **PrivateKey, int Bits, int SerialNo, int DaysValid, unsigned char *random)
{
    X509                *Cert;
    X509_NAME        *Name = NULL;
    EVP_PKEY            *PrivKey;
    RSA                *RSAData;
    unsigned char    buffer[512];

    /* We seed with the random data from the client.  If this isn't enough we just use good old rand for the rest */
    if (random) {
        RAND_seed(random, strlen(random));
    }

    while (RAND_status() != 1) {
        unsigned long        len;

        srand((unsigned) time(NULL));

        len = snprintf(buffer, sizeof(buffer), "%d", rand());
        RAND_seed(buffer, len);
    }

    /* Did we get a private key? If not, generate one */
    if ((PrivateKey == NULL) || (*PrivateKey == NULL)) {
        if ((PrivKey = EVP_PKEY_new()) == NULL) {
            return(FALSE);
        }
    } else {
        PrivKey = *PrivateKey;
    }

    if ((Certificate == NULL) || (*Certificate == NULL)) {
        if ((Cert = X509_new()) == NULL) {
            EVP_PKEY_free(PrivKey);

            return(FALSE);
        }
    } else {
        Cert = *Certificate;
    }

    RSAData = RSA_generate_key(Bits, RSA_F4, GenerateCertificateCallback, NULL);
    if (!EVP_PKEY_assign_RSA(PrivKey, RSAData)) {
        X509_free(Cert);
        EVP_PKEY_free(PrivKey);

        return(FALSE);
    }

    RSAData = NULL;

    X509_set_version(Cert, 2);
    ASN1_INTEGER_set(X509_get_serialNumber(Cert), SerialNo);
    X509_gmtime_adj(X509_get_notBefore(Cert), 0);
    X509_gmtime_adj(X509_get_notAfter(Cert), (long)DaysValid);
    X509_set_pubkey(Cert, PrivKey);

    Name = X509_get_subject_name(Cert);
    X509_NAME_add_entry_by_txt(Name, "C", MBSTRING_ASC, CertSession->country, -1, -1, 0);
    X509_NAME_add_entry_by_txt(Name, "CN", MBSTRING_ASC, CertSession->commonName, -1, -1, 0);

    /* This makes it "self-signed" */
    X509_set_issuer_name(Cert, Name);

    /* Add extensions */
    AddV3Extension(Cert, NID_basic_constraints, "critical,CA:TRUE");
    AddV3Extension(Cert, NID_key_usage, "critical,digitalSignature,keyEncipherment,keyCertSign");

    AddV3Extension(Cert, NID_subject_key_identifier, "hash");
    snprintf(buffer, sizeof(buffer), "email: %s", CertSession->eMail);
    AddV3Extension(Cert, NID_subject_alt_name, buffer);

    /* Add the fields we use to verify it's "us" */
    AddV3Extension(Cert, NID_issuer_alt_name, "WACert");
    AddV3Extension(Cert, NID_organizationalUnitName, CertSession->organization);

    if (!X509_sign(Cert, PrivKey, EVP_md5())) {
        X509_free(Cert);
        EVP_PKEY_free(PrivKey);

        return(FALSE);
    }

    if (Certificate) {
        *Certificate = Cert;
    }

    if (PrivateKey) {
        *PrivateKey = PrivKey;
    }

    return(TRUE);
}

#if 0
#define            d                                            \
{                                                                \
    printf("[%d] %s\r\n", __LINE__, __FILE__);    \
}
#endif

static BOOL
GenerateCSR(WACertSessionStruct *CertSession, X509_REQ **Request, EVP_PKEY **PrivateKey, int Bits, unsigned char *random)
{
    X509_REQ                            *req = NULL;
    X509_NAME                        *Name = NULL;
    EVP_PKEY                            *PrivKey;
    RSA                                *RSAData;
    STACK_OF(X509_EXTENSION)    *exts = NULL;
    unsigned char                    buffer[512];


    /* We seed with the random data from the client.  If this isn't enough we just use good old rand for the rest */
    if (random) {
        RAND_seed(random, strlen(random));
    }

    while (RAND_status() != 1) {
        unsigned long        len;

        srand((unsigned) time(NULL));

        len = snprintf(buffer, sizeof(buffer), "%d", rand());
        RAND_seed(buffer, len);
    }

    /* Did we get a private key? If not, generate one */
    if ((PrivateKey == NULL) || (*PrivateKey == NULL)) {
        if ((PrivKey = EVP_PKEY_new()) == NULL) {
            return(FALSE);
        }
    } else {
        PrivKey = *PrivateKey;
    }

    if ((Request == NULL) || (*Request == NULL)) {
        if ((req = X509_REQ_new()) == NULL) {
            EVP_PKEY_free(PrivKey);

            return(FALSE);
        }
    } else {
        req = *Request;
    }

    RSAData = RSA_generate_key(Bits, RSA_F4, GenerateCertificateCallback, NULL);
    if (!EVP_PKEY_assign_RSA(PrivKey, RSAData)) {
        X509_REQ_free(req);
        EVP_PKEY_free(PrivKey);

        return(FALSE);
    }

    RSAData = NULL;

    X509_REQ_set_pubkey(req, PrivKey);
    X509_REQ_set_version(req, 0L);

    Name = X509_NAME_new();
    if (Name) {
        X509_NAME_add_entry_by_txt(Name, "C", MBSTRING_ASC, CertSession->country, -1, -1, 0);
        X509_NAME_add_entry_by_txt(Name, "CN", MBSTRING_ASC, CertSession->commonName, -1, -1, 0);

        X509_REQ_set_subject_name(req, Name);
    }

    /* Add extensions */
    exts = sk_X509_EXTENSION_new_null();

    AddV3ReqExtension(exts, NID_basic_constraints, "critical,CA:TRUE");
    //AddV3ReqExtension(exts, NID_key_usage, "critical,keyCertSign,cRLSign");
    AddV3ReqExtension(exts, NID_key_usage, "critical,digitalSignature,keyEncipherment");

    AddV3ReqExtension(exts, NID_subject_key_identifier, "hash");
    snprintf(buffer, sizeof(buffer), "email: %s", CertSession->eMail);
    AddV3ReqExtension(exts, NID_subject_alt_name, buffer);

    /* Add the fields we use to verify it's "us" */
    AddV3ReqExtension(exts, NID_issuer_alt_name, "WACert");
    AddV3ReqExtension(exts, NID_organizationalUnitName, CertSession->organization);

    X509_REQ_add_extensions(req, exts);
    sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);

    if (!X509_REQ_sign(req, PrivKey, EVP_md5())) {
        X509_REQ_free(req);
        EVP_PKEY_free(PrivKey);

        return(FALSE);
    }

    if (Request) {
        *Request = req;
    }

    if (PrivateKey) {
        *PrivateKey = PrivKey;
    }

    return(TRUE);
}

BOOL
WACertInitSession(SessionStruct *Session, void **ModuleData)
{
    WACertSessionStruct        *CertSession;

    CertSession = malloc(sizeof(WACertSessionStruct));
    if (!CertSession ) {
        XplConsolePrintf("WACert Module out of memory!\n");
        return(FALSE);
    }

    memset(CertSession, 0, sizeof(WACertSessionStruct));

    *ModuleData = (void *)CertSession;
    return(TRUE);
}

BOOL
WACertDestroySession(SessionStruct *Session, void *ModuleData)
{
    WACertSessionStruct        *CertSession = (WACertSessionStruct *)ModuleData;

    if (CertSession) {
        if (CertSession->privKey) {
            free(CertSession->privKey);
        }

        if (CertSession->certificate) {
            free(CertSession->certificate);
        }

        if (CertSession->csr) {
            free(CertSession->csr);
        }

        free(CertSession);
    }

    return(TRUE);
}

BOOL
WACertHandleTemplate(ConnectionStruct *Client, SessionStruct *Session, unsigned long Page, TokenOverlayStruct *Token, unsigned long *GotoToken, void *ModuleData)
{
    WACertSessionStruct        *CertSession = (WACertSessionStruct *)ModuleData;
    unsigned char                URL[256];

    switch(Token->TokenID) {
        case T_NEW_CERT_FORM: {
            WAEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_NEW_CERT, Token->ArgumentOffsetOrID[0], 0, 0, 0);
            WASendClient(Client, URL, strlen(URL));

            return(TRUE);
        }

        case T_NEW_SS_CERT_FORM: {
            WAEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_NEW_SS_CERT, Token->ArgumentOffsetOrID[0], 0, 0, 0);
            WASendClient(Client, URL, strlen(URL));

            return(TRUE);
        }

        case T_INSTALL_ROOT_FORM: {
            WAEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_INSTALL_ROOT_CERT, Token->ArgumentOffsetOrID[0], 0, 0, 0);
            WASendClient(Client, URL, strlen(URL));

            return(TRUE);
        }

        case T_CERT_DOWNLOAD: {
            switch (Token->ArgumentOffsetOrID[0]) {
                case AA_PRIVATEKEY: {
                    WAEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_PRIV_KEY, 0, 0, 0, 0);
                    WASendClient(Client, URL, strlen(URL));
                    return(TRUE);
                }

                case AA_CERTIFICATE: {
                    WAEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_CERTIFICATE, 0, 0, 0, 0);
                    WASendClient(Client, URL, strlen(URL));
                    return(TRUE);
                }

                case AA_CSR: {
                    WAEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_CSR, 0, 0, 0, 0);
                    WASendClient(Client, URL, strlen(URL));
                    return(TRUE);
                }
            }

            return(TRUE);
        }

        case T_CERT_VIEW: {
            switch (Token->ArgumentOffsetOrID[0]) {
                case AA_PRIVATEKEY: {
                    if (CertSession->privKey) {
                        WASendClient(Client, CertSession->privKey, strlen(CertSession->privKey));
                    }
                    return(TRUE);
                }

                case AA_CERTIFICATE: {
                    if (CertSession->certificate) {
                        WASendClient(Client, CertSession->certificate, strlen(CertSession->certificate));
                    }
                    return(TRUE);
                }

                case AA_CSR: {
                    if (CertSession->csr) {
                        WASendClient(Client, CertSession->csr, strlen(CertSession->csr));
                    }
                    return(TRUE);
                }
            }

            return(TRUE);
        }

        default: {
            break;
        }
    }

    return(FALSE);
}

BOOL
WACertHandleURL(ConnectionStruct *Client, SessionStruct *Session, URLStruct *URL, void *ModuleData)
{
    WACertSessionStruct        *CertSession = (WACertSessionStruct *)ModuleData;
    unsigned long                len;

    switch(URL->Request) {
        case REQUEST_NEW_CERT: {
            unsigned char            *data;
            unsigned char            *ptr;
            unsigned long            allocated        = 0;
            unsigned long            valueSize;
            unsigned char            random[4096];
            int                        bits = 512;

            random[0] = '\0';

            allocated = BUFSIZE;
            data = malloc(allocated * sizeof(unsigned char));

            WAProtectNMAP(Client, TRUE);

            while (WAGetFormNameEx(Client, Client->Temp, NULL, NULL, BUFSIZE)) {
                ptr = data;
                valueSize = allocated - (ptr - data);

                while (WAGetFormValue(Client, ptr, &valueSize) != FORMFIELD_NEXT) {
                    ptr += valueSize;

                    if ((ptr - data + 512) > allocated) {
                        allocated += BUFSIZE;
                        data = realloc(data, allocated * sizeof(unsigned char));
                        ptr = data + valueSize;
                    }
                    valueSize = allocated - (ptr - data);
                }

                /*
                    Right now we have both the form name (in Client->Temp) and the form
                    data (in data) so lets do sumfin.
                */

                if (strlen(data) < 256) {
                    if (WAQuickCmp(Client->Temp, "requestor")) {
                        HulaStrNCpy(CertSession->requestor, data, sizeof(CertSession->requestor));
                    } else if (WAQuickCmp(Client->Temp, "email")) {
                        HulaStrNCpy(CertSession->eMail, data, sizeof(CertSession->eMail));
                    } else if (WAQuickCmp(Client->Temp, "commonname")) {
                        HulaStrNCpy(CertSession->commonName, data, sizeof(CertSession->commonName));
                    } else if (WAQuickCmp(Client->Temp, "organization")) {
                        HulaStrNCpy(CertSession->organization, data, sizeof(CertSession->organization));
                    } else if (WAQuickCmp(Client->Temp, "department")) {
                        HulaStrNCpy(CertSession->department, data, sizeof(CertSession->department));
                    } else if (WAQuickCmp(Client->Temp, "city")) {
                        HulaStrNCpy(CertSession->city, data, sizeof(CertSession->city));
                    } else if (WAQuickCmp(Client->Temp, "state")) {
                        HulaStrNCpy(CertSession->state, data, sizeof(CertSession->state));
                    } else if (WAQuickCmp(Client->Temp, "countrycode")) {
                        HulaStrNCpy(CertSession->country, data, sizeof(CertSession->country));
                    } else if (WAQuickCmp(Client->Temp, "bits")) {
                        bits = atoi(data);
                        if (bits > 2048) {
                            bits = 2048;
                        }
                    }
                }

                if (WAQuickCmp(Client->Temp, "random")) {
		    HulaStrNCpy(random, data, sizeof(random));
                }
            }

            WAProtectNMAP(Client, FALSE);

            if (data) {
                free(data);
            }

            /* Should have everything now */
            if (CertSession->requestor[0] != '\0' && CertSession->eMail[0] != '\0' && CertSession->commonName[0] != '\0' &&
                CertSession->organization[0] != '\0' && CertSession->city[0] != '\0' && CertSession->state[0] != '\0' && 
                CertSession->country[0] != '\0' && strlen(CertSession->country) == 2)
            {
                X509_REQ            *Request = NULL;
                EVP_PKEY            *PrivateKey = NULL;
                FILE                *TempFile;
                unsigned long    size;

                if (!GenerateCSR(CertSession, &Request, &PrivateKey, bits, random)) {
                    /*
                        Oops we buggered that up
                    */

                    WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);
                    return(TRUE);
                }

                snprintf(Client->Temp, sizeof(Client->Temp), "%s/%lu.csr", TEMP_DIR, Session->SessionID);
                TempFile = fopen(Client->Temp, "wb");
                PEM_write_X509_REQ(TempFile, Request);

                size = ftell(TempFile);
                fclose(TempFile);

                if (CertSession->csr) {
                    free(CertSession->csr);
                }

                CertSession->csr = malloc(size);
                if (CertSession->csr) {
                    TempFile = fopen(Client->Temp, "rb");

                    if (TempFile) {
                        len = 0;

                        while ((size - len) > 0 && !feof(TempFile) && !ferror(TempFile)) {
                            len += fread(CertSession->csr + len, 1, size - len, TempFile);
                        }

                        fclose(TempFile);
                    }
                    CertSession->csr[size] = '\0';
                }

                unlink(Client->Temp);
                X509_REQ_free(Request);

                snprintf(Client->Temp, sizeof(Client->Temp), "%s/%lu.prv", TEMP_DIR, Session->SessionID);
                TempFile = fopen(Client->Temp, "wb");

                PEM_write_PrivateKey(TempFile, PrivateKey, NULL, NULL, 0, NULL, NULL);
                size = ftell(TempFile);
                fclose(TempFile);

                if (CertSession->privKey) {
                    free(CertSession->privKey);
                }

                CertSession->privKey = malloc(size);
                if (CertSession->privKey) {
                    snprintf(Client->Temp, sizeof(Client->Temp), "%s/%lu.prv", TEMP_DIR, Session->SessionID);
                    TempFile = fopen(Client->Temp, "rb");

                    if (TempFile) {
                        len = 0;

                        while ((size - len) > 0 && !feof(TempFile) && !ferror(TempFile)) {
                            len += fread(CertSession->privKey + len, 1, size - len, TempFile);
                        }

                        fclose(TempFile);
                    }
                    CertSession->privKey[size] = '\0';
                }

                unlink(Client->Temp);
                EVP_PKEY_free(PrivateKey);
            }

            WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);

            return(TRUE);
        }

        case REQUEST_NEW_SS_CERT: {
            unsigned char            *data;
            unsigned char            *ptr;
            unsigned long            allocated        = 0;
            unsigned long            valueSize;
            unsigned char            random[4096];
            int                        bits = 512;

            random[0] = '\0';

            allocated = BUFSIZE;
            data = malloc(allocated * sizeof(unsigned char));

            WAProtectNMAP(Client, TRUE);

            while (WAGetFormNameEx(Client, Client->Temp, NULL, NULL, BUFSIZE)) {
                ptr = data;
                valueSize = allocated - (ptr - data);

                while (WAGetFormValue(Client, ptr, &valueSize) != FORMFIELD_NEXT) {
                    ptr += valueSize;

                    if ((ptr - data + 512) > allocated) {
                        allocated += BUFSIZE;
                        data = realloc(data, allocated * sizeof(unsigned char));
                        ptr = data + valueSize;
                    }
                    valueSize = allocated - (ptr - data);
                }

                /*
                    Right now we have both the form name (in Client->Temp) and the form
                    data (in data) so lets do sumfin.
                */

                if (strlen(data) < 256) {
                    if (WAQuickCmp(Client->Temp, "requestor")) {
                        HulaStrNCpy(CertSession->requestor, data, sizeof(CertSession->requestor));
                    } else if (WAQuickCmp(Client->Temp, "email")) {
                        HulaStrNCpy(CertSession->eMail, data, sizeof(CertSession->eMail));
                    } else if (WAQuickCmp(Client->Temp, "commonname")) {
                        HulaStrNCpy(CertSession->commonName, data, sizeof(CertSession->commonName));
                    } else if (WAQuickCmp(Client->Temp, "organization")) {
                        HulaStrNCpy(CertSession->organization, data, sizeof(CertSession->organization));
                    } else if (WAQuickCmp(Client->Temp, "department")) {
                        HulaStrNCpy(CertSession->department, data, sizeof(CertSession->department));
                    } else if (WAQuickCmp(Client->Temp, "city")) {
                        HulaStrNCpy(CertSession->city, data, sizeof(CertSession->city));
                    } else if (WAQuickCmp(Client->Temp, "state")) {
                        HulaStrNCpy(CertSession->state, data, sizeof(CertSession->state));
                    } else if (WAQuickCmp(Client->Temp, "countrycode")) {
                        HulaStrNCpy(CertSession->country, data, sizeof(CertSession->country));
                    } else if (WAQuickCmp(Client->Temp, "bits")) {
                        bits = atoi(data);
                        if (bits > 2048) {
                            bits = 2048;
                        }
                    }
                }

                if (WAQuickCmp(Client->Temp, "random")) {
		    HulaStrNCpy(random, data, sizeof(random));
                }
            }

            WAProtectNMAP(Client, FALSE);

            if (data) {
                free(data);
            }

            /* Should have everything now */
            if (CertSession->requestor[0] != '\0' && CertSession->eMail[0] != '\0' && CertSession->commonName[0] != '\0' &&
                CertSession->organization[0] != '\0' && CertSession->city[0] != '\0' && CertSession->state[0] != '\0' && 
                CertSession->country[0] != '\0' && strlen(CertSession->country) == 2)
            {
                X509                *Certificate = NULL;
                EVP_PKEY            *PrivateKey = NULL;
                FILE                *TempFile;
                unsigned long    size;

                /* Creating a self signed cert */
                if (!GenerateCACertificate(CertSession, &Certificate, &PrivateKey, bits, 0x27051977, 86400*3650, random)) {
                    /*
                        Oops we buggered that up
                    */

                    WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);
                    return(TRUE);
                }

                //    RSA_print_fp(stdout, PrivateKey->pkey.rsa, 0);
                //    X509_print_fp(stdout, Certificate);

                snprintf(Client->Temp, sizeof(Client->Temp), "%s/%lu.crt", TEMP_DIR, Session->SessionID);
                TempFile = fopen(Client->Temp, "wb");
                PEM_write_X509(TempFile, Certificate);

                size = ftell(TempFile);
                fclose(TempFile);

                if (CertSession->certificate) {
                    free(CertSession->certificate);
                }

                CertSession->certificate = malloc(size);
                if (CertSession->certificate) {
                    TempFile = fopen(Client->Temp, "rb");

                    if (TempFile) {
                        len = 0;

                        while ((size - len) > 0 && !feof(TempFile) && !ferror(TempFile)) {
                            len += fread(CertSession->certificate + len, 1, size - len, TempFile);
                        }

                        fclose(TempFile);
                    }
                    CertSession->certificate[size] = '\0';
                }

                unlink(Client->Temp);
                X509_free(Certificate);

                snprintf(Client->Temp, sizeof(Client->Temp), "%s/%lu.prv", TEMP_DIR, Session->SessionID);
                TempFile = fopen(Client->Temp, "wb");

                PEM_write_PrivateKey(TempFile, PrivateKey, NULL, NULL, 0, NULL, NULL);
                size = ftell(TempFile);
                fclose(TempFile);

                if (CertSession->privKey) {
                    free(CertSession->privKey);
                }
                
                CertSession->privKey = malloc(size);
                if (CertSession->privKey) {
                    snprintf(Client->Temp, sizeof(Client->Temp), "%s/%lu.prv", TEMP_DIR, Session->SessionID);
                    TempFile = fopen(Client->Temp, "rb");

                    if (TempFile) {
                        len = 0;

                        while ((size - len) > 0 && !feof(TempFile) && !ferror(TempFile)) {
                            len += fread(CertSession->privKey + len, 1, size - len, TempFile);
                        }

                        fclose(TempFile);
                    }
                    CertSession->privKey[size] = '\0';
                }

                unlink(Client->Temp);
                EVP_PKEY_free(PrivateKey);
            }

            WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);

            return(TRUE);
        }

        case REQUEST_INSTALL_ROOT_CERT: {
            unsigned char            *data;
            unsigned char            *ptr;
            unsigned long            allocated        = 0;
            unsigned long            valueSize;
            unsigned char            *cert                = NULL;
            unsigned char            *rootCert        = NULL;

            allocated = BUFSIZE;
            data = malloc(allocated * sizeof(unsigned char));

            WAProtectNMAP(Client, TRUE);

            while (WAGetFormNameEx(Client, Client->Temp, NULL, NULL, BUFSIZE)) {
                ptr = data;
                valueSize = allocated - (ptr - data);

                while (WAGetFormValue(Client, ptr, &valueSize) != FORMFIELD_NEXT) {
                    ptr += valueSize;

                    if ((ptr - data + 512) > allocated) {
                        allocated += BUFSIZE;
                        data = realloc(data, allocated * sizeof(unsigned char));
                        ptr = data + valueSize;
                    }
                    valueSize = allocated - (ptr - data);
                }

                /*
                    Right now we have both the form name (in Client->Temp) and the form
                    data (in data) so lets do sumfin.
                */

                if (WAQuickCmp(Client->Temp, "certificate")) {
                    cert = strdup(data);
                } else if (WAQuickCmp(Client->Temp, "rootcert")) {
                    rootCert = strdup(data);
                }
            }

            WAProtectNMAP(Client, FALSE);

            if (data) {
                free(data);
            }

            if (cert && rootCert) {
		int size;
                /* Should have everything now */
                if (CertSession->certificate) {
                    free(CertSession->certificate);
                }
		
		size = strlen(cert) + strlen(rootCert) + 5;
                CertSession->certificate = malloc(size);
                if (CertSession->certificate) {
                    snprintf(CertSession->certificate, size, "%s\r\n\r\n%s", cert, rootCert);
                }
            }

            if (cert) {
                free(cert);
            }

            if (rootCert) {
                free(rootCert);
            }

            WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);

            return(TRUE);
        }

        case REQUEST_PRIV_KEY: {
            if (CertSession->privKey) {
                WASendHTTPHeader(Client, "application/octet-stream", strlen(CertSession->privKey), "osslpriv.pem", FALSE);
                WASendClient(Client, CertSession->privKey, strlen(CertSession->privKey));
            }

            return(TRUE);
        }

        case REQUEST_CERTIFICATE: {
            if (CertSession->certificate) {
                WASendHTTPHeader(Client, "application/octet-stream", strlen(CertSession->certificate), "osslcert.pem", FALSE);
                WASendClient(Client, CertSession->certificate, strlen(CertSession->certificate));
            }

            return(TRUE);
        }

        case REQUEST_CSR: {
            if (CertSession->csr) {
                WASendHTTPHeader(Client, "application/octet-stream", strlen(CertSession->csr), "osslcsr.pem", FALSE);
                WASendClient(Client, CertSession->csr, strlen(CertSession->csr));
            }

            return(TRUE);
        }

        default: {
            break;
        }
    }

    return(FALSE);
}

/************************************************************************
    Now comes the "stock" stuff any modweb module needs
*************************************************************************/

EXPORT BOOL
LIBWACERTInit(WAAPIArg)
{
    ModuleRegisterStruct    Register;

    WAAPISet;

    XplMakeDir(TEMP_DIR);

    SSL_load_error_strings();
    SSL_library_init();

    /* Register HandleURL and HandleTemplate */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType = MODULE_TEMPLATE;
    Register.Module.Template.InitSession = WACertInitSession;
    Register.Module.Template.DestroySession = WACertDestroySession;
    Register.Module.Template.HandleURL = WACertHandleURL;
    Register.Module.Template.HandleTemplate = WACertHandleTemplate;
    Register.Module.Template.TokenRangeStart = WACERT_TOKEN_START;
    Register.Module.Template.TokenRangeEnd = WACERT_TOKEN_END;

    WARegisterModule(&Register);

    WACert.unload = FALSE;
    return(TRUE);
}

EXPORT BOOL
LIBWACERTShutdown(void)
{
    ERR_free_strings();
    ERR_remove_state(0);
    EVP_cleanup();

    WACert.unload = TRUE;
    return(TRUE);
}



/************************************************************************
    Below are all the things that make loading and unloading possible :-)
*************************************************************************/

#if defined(NETWARE) || defined(LIBC)
int
RequestUnload(void)
{
    if (!WACert.unload) {
        int    OldTGid;

        OldTGid = XplSetThreadGroupID(WACert.id.group);

        XplConsolePrintf("\rThis NLM will automatically be unloaded by the thread that loaded it.\n");
        XplConsolePrintf("\rIt does not allow manual unloading.\n");
        SetCurrentScreen(CreateScreen("System Console", 0));
        XplUngetCh('n');

        XplSetThreadGroupID(OldTGid);

        return(1);
    }

    return(0);
}

void
WACertSigHandler(int Signal)
{
    int    OldTGid;

    OldTGid = XplSetThreadGroupID(WACert.id.group);

    XplSignalLocalSemaphore(WACert.sem.shutdown);    /* The signal will release main() */
    XplWaitOnLocalSemaphore(WACert.sem.shutdown);    /* The wait will wait until main() is gone */

    /* Do any required cleanup */
    XplCloseLocalSemaphore(WACert.sem.shutdown);
    XplSetThreadGroupID(OldTGid);
}


int
main(int argc, char *argv[])
{
    WACert.id.group = XplGetThreadGroupID();

    signal(SIGTERM, WACertSigHandler);

    XplOpenLocalSemaphore(WACert.sem.shutdown, 0);
    XplWaitOnLocalSemaphore(WACert.sem.shutdown);
    XplSignalLocalSemaphore(WACert.sem.shutdown);

    return(0);
}
#endif
